// utility functions
//
#include "Utility.h"
#include "Sum.h"
#include "Product.h"

#define PHRED_MAX 50000.0 // max Phred seems to be about 43015 (?), could be an underflow bug...

using namespace std;

short qualityChar2ShortInt(char c) {
    return static_cast<short>(c) - 33;
}

long double qualityChar2LongDouble(char c) {
    return static_cast<long double>(c) - 33;
}

long double lnqualityChar2ShortInt(char c) {
    return log(static_cast<short>(c) - 33);
}

char qualityInt2Char(short i) {
    return static_cast<char>(i + 33);
}

long double ln2log10(long double prob) {
    return M_LOG10E * prob;
}

long double log102ln(long double prob) {
    return M_LN10 * prob;
}

long double phred2ln(int qual) {
    return M_LN10 * qual * -.1;
}

long double ln2phred(long double prob) {
    return -10 * M_LOG10E * prob;
}

long double phred2float(int qual) {
    return pow(10, qual * -.1);
}

long double float2phred(long double prob) {
    if (prob == 1)
        return PHRED_MAX;  // guards against "-0"
    long double p = -10 * (long double) log10(prob);
    if (p < 0 || p > PHRED_MAX) // int overflow guard
        return PHRED_MAX;
    else
        return p;
}

long double big2phred(const BigFloat& prob) {
    return -10 * (long double) (ttmath::Log(prob, (BigFloat)10)).ToDouble();
}

long double nan2zero(long double x) {
    if (x != x) {
        return 0;
    } else {
        return x;
    }
}

long double powln(long double m, int n) {
    return m * n;
}

// the probability that we have a completely true vector of qualities
long double jointQuality(const std::vector<short>& quals) {
    std::vector<long double> probs;
    for (int i = 0; i<quals.size(); ++i) {
        probs.push_back(phred2float(quals[i]));
    }
    // product of probability we don't have a true event for each element
    long double prod = 1 - probs.front();
    for (int i = 1; i<probs.size(); ++i) {
        prod *= 1 - probs.at(i);
    }
    // and then invert it again to get probability of an event
    return 1 - prod;
}

long double jointQuality(const std::string& qualstr) {

    long double jq = 1;
    // product of probability we don't have a true event for each element
    for (string::const_iterator q = qualstr.begin(); q != qualstr.end(); ++q) {
        jq *= 1 - phred2float(qualityChar2ShortInt(*q));
    }

    // and then invert it again to get probability of an event
    return 1 - jq;

}

std::vector<short> qualities(const std::string& qualstr) {

    std::vector<short> quals;
    for (int i=0; i<qualstr.size(); i++)
        quals.push_back(qualityChar2ShortInt(qualstr.at(i)));

    return quals;

}

long double sumQuality(const std::string& qualstr) {
    long double qual = 0;
    for (string::const_iterator q = qualstr.begin(); q != qualstr.end(); ++q)
        qual += qualityChar2LongDouble(*q);
    return qual;
}

long double minQuality(const std::string& qualstr) {
    long double qual = 0;
    for (string::const_iterator q = qualstr.begin(); q != qualstr.end(); ++q) {
        long double nq = qualityChar2LongDouble(*q);
        if (qual == 0) {
            qual = nq;
        } else if (nq < qual) {
            qual = nq;
        }
    }
    return qual;
}

short minQuality(const std::vector<short>& qualities) {
    short m = 0;
    for (vector<short>::const_iterator q = qualities.begin(); q != qualities.end(); ++q) {
        if (*q < m) m = *q;
    }
    return m;
}

// crudely averages quality scores in phred space
long double averageQuality(const std::string& qualstr) {
    long double qual = 0; //(long double) *max_element(quals.begin(), quals.end());
    for (string::const_iterator q = qualstr.begin(); q != qualstr.end(); ++q)
        qual += qualityChar2LongDouble(*q);
    return qual / qualstr.size();
}

long double averageQuality(const vector<short>& qualities) {
    long double qual = 0;
    for (vector<short>::const_iterator q = qualities.begin(); q != qualities.end(); ++q) {
        qual += *q;
    }
    return qual / qualities.size();
}

bool stringInVector(string item, vector<string> items) {
    for (vector<string>::iterator i = items.begin(); i != items.end(); ++i) {
        if (item == *i) {
            return true;
        }
    }
    return false;
}

bool allATGC(string& s) {
    for (string::iterator c = s.begin(); c != s.end(); ++c) {
        char b = *c;
        if (b != 'A' && b != 'T' && b != 'G' && b != 'C') {
            return false;
        }
    }
    return true;
}

int upper(int c) {
    return toupper((unsigned char) c);
}

string uppercase(string s) {
    transform(s.begin(), s.end(), s.begin(), upper);
    return s;
}

string strip(string const& str, char const* separators) {
    string::size_type const first = str.find_first_not_of(separators);
    return (first == string::npos) ? string()
        : str.substr(first, str.find_last_not_of(separators) - first + 1);
}


int binomialCoefficient(int n, int k) {
    int i = 1;
    int result = n - k + i++;
    while (i <= k) {
        result *= (n - k + i) / i;
        ++i;
    }
    return result;
}

// k successes in n trials with prob of success p
long double binomialProb(int k, int n, long double p) {
    return factorial(n) / (factorial(k) * factorial(n - k)) * pow(p, k) * pow(1 - p, n - k);
}

long double impl_binomialProbln(int k, int n, long double p) {
    return factorialln(n) - (factorialln(k) + factorialln(n - k)) + powln(log(p), k) + powln(log(1 - p), n - k);
}

long double binomialCoefficientLn(int k, int n) {
    return factorialln(n) - (factorialln(k) + factorialln(n - k));
}

BinomialCache binomialCache;

long double binomialProbln(int k, int n, long double p) {
    return binomialCache.binomialProbln(k, n, p);
}

/*
long double probability(int k, int n, long double p) {
    int n = n - k;
    int m = k;
    long double q = 1 - p;
    long double temp = lgammal(m + n + 1.0);
    temp -= lgammal(n + 1.0) + lgammal(m + 1.0);
    temp += m*log(p) + n*log(q);
    return temp;
}
*/

long double poissonpln(int observed, int expected) {
    return ((log(expected) * observed) - expected) - factorialln(observed);
}

long double poissonp(int observed, int expected) {
    return (double) pow((double) expected, (double) observed) * (double) pow(M_E, (double) -expected) / factorial(observed);
}


// given the expected number of events is the max of a and b
// what is the probability that we might observe less than the observed?
long double poissonPvalLn(int a, int b) {

    int expected, observed;
    if (a > b) {
        expected = a; observed = b;
    } else {
        expected = b; observed = a;
    }

    vector<long double> probs;
    for (int i = 0; i < observed; ++i) {
        probs.push_back(poissonpln(i, expected));
    }

    return logsumexp_probs(probs);

}


long double gammaln(
    long double x
    ) {

    long double cofactors[] = { 76.18009173, 
                                -86.50532033,
                                24.01409822,
                                -1.231739516,
                                0.120858003E-2,
                                -0.536382E-5 };    

    long double x1 = x - 1.0;
    long double tmp = x1 + 5.5;
    tmp -= (x1 + 0.5) * log(tmp);
    long double ser = 1.0;
    for (int j=0; j<=5; j++) {
        x1 += 1.0;
        ser += cofactors[j]/x1;
    }
    long double y =  (-1.0 * tmp + log(2.50662827465 * ser));

    return y;
}

long double factorial(
    int n
    ) {
    if (n < 0) {
        return (long double)0.0;
    }
    else if (n == 0) {
        return (long double)1.0;
    }
    else {
        return exp(gammaln(n + 1.0));
    }
}

FactorialCache factorialCache;

/*
long double factorialln(int n) {
    return factorialCache.factorialln(n);
}
*/

long double impl_factorialln(
    int n
    ) {
    if (n < 0) {
        return (long double)-1.0;
    }
    else if (n == 0) {
        return (long double)0.0;
    }
    else {
        return gammaln(n + 1.0);
    }
}

long double cofactor(
    int n, 
    int i
    ) {
    if ((n < 0) || (i < 0) || (n < i)) {
        return (long double)0.0;
    }
    else if (n == i) {
        return (long double)1.0;
    }
    else {
        return exp(gammaln(n + 1.0) - gammaln(i + 1.0) - gammaln(n-i + 1.0));
    }
}

long double cofactorln(
    int n, 
    int i
    ) {
    if ((n < 0) || (i < 0) || (n < i)) {
        return (long double)-1.0;
    }
    else if (n == i) {
        return (long double)0.0;
    }
    else {
        return gammaln(n + 1.0) - gammaln(i + 1.0) - gammaln(n-i + 1.0);
    }
}

// prevent underflows by returning exp(LDBL_MIN_EXP) if exponentiation will produce an underflow
long double safe_exp(long double ln) {
    if (ln < LDBL_MIN_EXP) {  // -16381
        return LDBL_MIN;      // 3.3621e-4932
    } else {
        return exp(ln);
    }
}

BigFloat big_exp(long double ln) {
    BigFloat x, result;
    x.FromDouble(ln);
    result = ttmath::Exp(x);
    return result;
}

// 'safe' log summation for probabilities
long double logsumexp_probs(const vector<long double>& lnv) {
    vector<long double>::const_iterator i = lnv.begin();
    long double maxN = *i;
    ++i;
    for (; i != lnv.end(); ++i) {
        if (*i > maxN)
            maxN = *i;
    }
    BigFloat sum = 0;
    for (vector<long double>::const_iterator i = lnv.begin(); i != lnv.end(); ++i) {
        sum += big_exp(*i - maxN);
    }
    BigFloat maxNb; maxNb.FromDouble(maxN);
    BigFloat bigResult = maxNb + ttmath::Ln(sum);
    long double result;
    return bigResult.ToDouble();
}

// unsafe, kept for potential future use
long double logsumexp(const vector<long double>& lnv) {
    long double maxAbs, minN, maxN, c;
    vector<long double>::const_iterator i = lnv.begin();
    long double n = *i;
    maxAbs = n; maxN = n; minN = n;
    ++i;
    for (; i != lnv.end(); ++i) {
        n = *i;
        if (n > maxN)
            maxN = n;
        if (fabs(n) > maxAbs)
            maxAbs = fabs(n);
        if (n < minN)
            minN = n;
    }
    if (maxAbs > maxN) {
        c = minN;
    } else {
        c = maxN;
    }
    long double sum = 0;
    for (vector<long double>::const_iterator i = lnv.begin(); i != lnv.end(); ++i) {
        sum += exp(*i - c);
    }
    return c + log(sum);
}

long double betaln(const vector<long double>& alphas) {
    vector<long double> gammalnAlphas;
    gammalnAlphas.resize(alphas.size());
    transform(alphas.begin(), alphas.end(), gammalnAlphas.begin(), gammaln);
    return sum(gammalnAlphas) - gammaln(sum(alphas));
}

long double beta(const vector<long double>& alphas) {
    return exp(betaln(alphas));
}

long double hoeffding(double successes, double trials, double prob) {
    return 0.5 * exp(-2 * pow(trials * prob - successes, 2) / trials);
}

long double hoeffdingln(double successes, double trials, double prob) {
    return log(0.5) + (-2 * pow(trials * prob - successes, 2) / trials);
}

// the sum of the harmonic series 1, n
long double harmonicSum(int n) {
    long double r = 0;
    long double i = 1;
    while (i <= n) {
        r += 1 / i;
        ++i;
    }
    return r;
}

bool isTransition(string& ref, string& alt) {
    if (((ref == "A" && alt == "G") || (ref == "G" && alt == "A")) ||
        ((ref == "C" && alt == "T") || (ref == "T" && alt == "C"))) {
        return true;
    } else {
        return false;
    }
}

// Levenshtein Distance Algorithm: C++ Implementation
// by Anders Sewerin Johansen
// http://www.merriampark.com/ldcpp.htm

int levenshteinDistance(const std::string source, const std::string target) {

    // Step 1

    const int n = source.length();
    const int m = target.length();
    if (n == 0) {
        return m;
    }
    if (m == 0) {
        return n;
    }

    // Good form to declare a TYPEDEF

    typedef std::vector< std::vector<int> > Tmatrix; 

    Tmatrix matrix(n+1);

    // Size the vectors in the 2.nd dimension. Unfortunately C++ doesn't
    // allow for allocation on declaration of 2.nd dimension of vec of vec

    for (int i = 0; i <= n; i++) {
        matrix[i].resize(m+1);
    }

    // Step 2

    for (int i = 0; i <= n; i++) {
        matrix[i][0]=i;
    }

    for (int j = 0; j <= m; j++) {
        matrix[0][j]=j;
    }

    // Step 3

    for (int i = 1; i <= n; i++) {

        const char s_i = source[i-1];

        // Step 4

        for (int j = 1; j <= m; j++) {

            const char t_j = target[j-1];

            // Step 5

            int cost;
            if (s_i == t_j) {
                cost = 0;
            }
            else {
                cost = 1;
            }

            // Step 6

            const int above = matrix[i-1][j];
            const int left = matrix[i][j-1];
            const int diag = matrix[i-1][j-1];
            int cell = min( above + 1, min(left + 1, diag + cost));

            // Step 6A: Cover transposition, in addition to deletion,
            // insertion and substitution. This step is taken from:
            // Berghel, Hal ; Roach, David : "An Extension of Ukkonen's 
            // Enhanced Dynamic Programming ASM Algorithm"
            // (http://www.acm.org/~hlb/publications/asm/asm.html)

            if (i>2 && j>2) {
                int trans=matrix[i-2][j-2]+1;
                if (source[i-2]!=t_j) trans++;
                if (s_i!=target[j-2]) trans++;
                if (cell>trans) cell=trans;
            }

            matrix[i][j]=cell;
        }
    }

    // Step 7

    return matrix[n][m];
}

// current date string in YYYYMMDD format
string dateStr(void) {

    time_t rawtime;
    struct tm* timeinfo;
    char buffer[80];

    time(&rawtime);
    timeinfo = localtime(&rawtime);

    strftime(buffer, 80, "%Y%m%d", timeinfo);

    return string(buffer);

}

long double string2float(const string& s) {
    long double r;
    convert(s, r);
    return r;
}

long double log10string2ln(const string& s) {
    long double r;
    convert(s, r);
    return log102ln(r);
}

long double safedivide(long double a, long double b) {
    if (b == 0) {
        if (a == 0) {
            return 1;
        } else {
            return 0;
        }
    } else {
        return a / b;
    }
}

string mergeCigar(const string& c1, const string& c2) {
    vector<pair<int, string> > cigar1 = splitCigar(c1);
    vector<pair<int, string> > cigar2 = splitCigar(c2);
    // check if the middle elements are the same
    if (cigar1.back().second == cigar2.front().second) {
        cigar1.back().first += cigar2.front().first;
        cigar2.erase(cigar2.begin());
    }
    for (vector<pair<int, string> >::iterator c = cigar2.begin(); c != cigar2.end(); ++c) {
        cigar1.push_back(*c);
    }
    return joinCigar(cigar1);
}

vector<pair<int, string> > splitCigar(const string& cigarStr) {
    vector<pair<int, string> > cigar;
    string number;
    string type;
    // strings go [Number][Type] ...
    for (string::const_iterator s = cigarStr.begin(); s != cigarStr.end(); ++s) {
        char c = *s;
        if (isdigit(c)) {
            if (type.empty()) {
                number += c;
            } else {
                // signal for next token, push back the last pair, clean up
                cigar.push_back(make_pair(atoi(number.c_str()), type));
                number.clear();
                type.clear();
                number += c;
            }
        } else {
            type += c;
        }
    }
    if (!number.empty() && !type.empty()) {
        cigar.push_back(make_pair(atoi(number.c_str()), type));
    }
    return cigar;
}

list<pair<int, string> > splitCigarList(const string& cigarStr) {
    list<pair<int, string> > cigar;
    string number;
    string type;
    // strings go [Number][Type] ...
    for (string::const_iterator s = cigarStr.begin(); s != cigarStr.end(); ++s) {
        char c = *s;
        if (isdigit(c)) {
            if (type.empty()) {
                number += c;
            } else {
                // signal for next token, push back the last pair, clean up
                cigar.push_back(make_pair(atoi(number.c_str()), type));
                number.clear();
                type.clear();
                number += c;
            }
        } else {
            type += c;
        }
    }
    if (!number.empty() && !type.empty()) {
        cigar.push_back(make_pair(atoi(number.c_str()), type));
    }
    return cigar;
}

string joinCigar(const vector<pair<int, string> >& cigar) {
    string cigarStr;
    for (vector<pair<int, string> >::const_iterator c = cigar.begin(); c != cigar.end(); ++c) {
        if (c->first) {
            cigarStr += convert(c->first) + c->second;
        }
    }
    return cigarStr;
}

string joinCigarList(const list<pair<int, string> >& cigar) {
    string cigarStr;
    for (list<pair<int, string> >::const_iterator c = cigar.begin(); c != cigar.end(); ++c) {
        cigarStr += convert(c->first) + c->second;
    }
    return cigarStr;
}

bool isEmptyCigarElement(const pair<int, string>& elem) {
    return elem.first == 0;
}


// string * overload
// from http://stackoverflow.com/a/5145880
std::string operator*(std::string const &s, size_t n)
{
    std::string r;  // empty string
    r.reserve(n * s.size());
    for (size_t i=0; i<n; i++)
        r += s;
    return r;
}

// normalize vector sum to 1
void normalizeSumToOne(vector<long double>& v) {
    long double sum = 0;
    for (vector<long double>::iterator i = v.begin(); i != v.end(); ++i) {
        sum += *i;
    }
    for (vector<long double>::iterator i = v.begin(); i != v.end(); ++i) {
        *i /= sum;
    }
}

// splits the file on '\n', adds the resulting values to v
void addLinesFromFile(vector<string>& v, const string& f) {
    ifstream ifs;
    ifs.open(f.c_str(), ifstream::in);
    if (!ifs.is_open()) {
        cerr << "could not open " << f << endl;
        exit(1);
    }
    string line;
    while (std::getline(ifs, line)) {
        v.push_back(line);
    }
}


double lf[] = {
0.000000000000000,
0.000000000000000,
0.693147180559945,
1.791759469228055,
3.178053830347946,
4.787491742782046,
6.579251212010101,
8.525161361065415,
10.604602902745251,
12.801827480081469,
15.104412573075516,
17.502307845873887,
19.987214495661885,
22.552163853123421,
25.191221182738683,
27.899271383840894,
30.671860106080675,
33.505073450136891,
36.395445208033053,
39.339884187199495,
42.335616460753485,
45.380138898476908,
48.471181351835227,
51.606675567764377,
54.784729398112319,
58.003605222980518,
61.261701761002001,
64.557538627006323,
67.889743137181526,
71.257038967168000,
74.658236348830158,
78.092223553315307,
81.557959456115029,
85.054467017581516,
88.580827542197682,
92.136175603687079,
95.719694542143202,
99.330612454787428,
102.968198614513810,
106.631760260643446,
110.320639714757391,
114.034211781461693,
117.771881399745055,
121.533081515438639,
125.317271149356884,
129.123933639127245,
132.952575035616292,
136.802722637326355,
140.673923648234251,
144.565743946344895,
148.477766951773020,
152.409592584497346,
156.360836303078798,
160.331128216630930,
164.320112263195170,
168.327445448427653,
172.352797139162817,
176.395848406997374,
180.456291417543781,
184.533828861449507,
188.628173423671598,
192.739047287844897,
196.866181672889979,
201.009316399281573,
205.168199482641199,
209.342586752536818,
213.532241494563266,
217.736934113954248,
221.956441819130362,
226.190548323727569,
230.439043565776927,
234.701723442818263,
238.978389561834348,
243.268849002982734,
247.572914096186906,
251.890402209723192,
256.221135550009478,
260.564940971863223,
264.921649798552778,
269.291097651019811,
273.673124285693689,
278.067573440366118,
282.474292687630395,
286.893133295426992,
291.323950094270288,
295.766601350760595,
300.220948647014097,
304.686856765668722,
309.164193580146900,
313.652829949878992,
318.152639620209300,
322.663499126726208,
327.185287703775202,
331.717887196928473,
336.261181979198454,
340.815058870798964,
345.379407062266864,
349.954118040770254,
354.539085519440789,
359.134205369575341,
363.739375555563470,
368.354496072404686,
372.979468885689016,
377.614197873918670,
382.258588773060012,
386.912549123217559,
391.575988217329609,
396.248817051791491,
400.930948278915764,
405.622296161144902,
410.322776526937275,
415.032306728249580,
419.750805599544776,
424.478193418257092,
429.214391866651567,
433.959323995014870,
438.712914186121168,
443.475088120918940,
448.245772745384613,
453.024896238496126,
457.812387981278107,
462.608178526874894,
467.412199571608085,
472.224383926980522,
477.044665492585580,
481.872979229887903,
486.709261136839359,
491.553448223298005,
496.405478487217579,
501.265290891579241,
506.132825342034835,
511.008022665236069,
515.890824587822522,
520.781173716044236,
525.679013515995052,
530.584288294433577,
535.496943180169524,
540.416924105997737,
545.344177791154948,
550.278651724285623,
555.220294146894958,
560.169054037273099,
565.124881094874354,
570.087725725134192,
575.057539024710195,
580.034272767130801,
585.017879388839219,
590.008311975617858,
595.005524249382006,
600.009470555327425,
605.020105849423771,
610.037385686238736,
615.061266207084941,
620.091704128477431,
625.128656730891066,
630.172081847810205,
635.221937855059764,
640.278183660408104,
645.340778693435027,
650.409682895655237,
655.484856710889062,
660.566261075873513,
665.653857411105946,
670.747607611912713,
675.847474039736881,
680.953419513637527,
686.065407301994014,
691.183401114410799,
696.307365093814042,
701.437263808737157,
706.573062245787469,
711.714725802289990,
716.862220279103440,
722.015511873601326,
727.174567172815841,
732.339353146739313,
737.509837141777439,
742.685986874351215,
747.867770424643368,
753.055156230484158,
758.248113081374299,
763.446610112640201,
768.650616799717000,
773.860102952558464,
779.075038710167405,
784.295394535245691,
789.521141208958966,
794.752249825813465,
799.988691788643450,
805.230438803703123,
810.477462875863580,
815.729736303910158,
820.987231675937892,
826.249921864842804,
831.517780023906312,
836.790779582469895,
842.068894241700491,
847.352097970438422,
852.640365001133091,
857.933669825857464,
863.231987192405427,
868.535292100464630,
873.843559797865737,
879.156765776907605,
884.474885770751825,
889.797895749890245,
895.125771918679902,
900.458490711945274,
905.796028791646336,
911.138363043611207,
916.485470574328815,
921.837328707804886,
927.193914982476713,
932.555207148186241,
937.921183163208070,
943.291821191335657,
948.667099599019821,
954.046996952560448,
959.431492015349477,
964.820563745165941,
970.214191291518318,
975.612353993036209,
981.015031374908403,
986.422203146368588,
991.833849198223447,
997.249949600427840,
1002.670484599700330,
1008.095434617181695,
1013.524780246136174,
1018.958502249690241,
1024.396581558613434,
1029.838999269135456,
1035.285736640801588,
1040.736775094367431,
1046.192096209724923,
1051.651681723869160,
1057.115513528895008,
1062.583573670030091,
1068.055844343701438,
1073.532307895632812,
1079.012946818975024,
1084.497743752465567,
1089.986681478622359,
1095.479742921962725,
1100.976911147256033,
1106.478169357800880,
1111.983500893732980,
1117.492889230361016,
1123.006317976526134,
1128.523770872990781,
1134.045231790852995,
1139.570684729984805,
1145.100113817496094,
1150.633503306223702,
1156.170837573242352,
1161.712101118566125,
1167.257278563043883,
1172.806354647937269,
1178.359314232856832,
1183.916142294554675,
1189.476823925568624,
1195.041344332889139,
1200.609688836648729,
1206.181842868824560,
1211.757791971969255,
1217.337521797953741,
1222.921018106733982,
1228.508266765132475,
1234.099253745641590,
1239.693965125242130,
1245.292387084238726,
1250.894505905117057,
1256.500307971411530,
1262.109779766595011,
1267.722907872981750,
1273.339678970646673,
1278.960079836362411,
1284.584097342548375,
1290.211718456237577,
1295.842930238057534,
1301.477719841225507,
1307.116074510557837,
1312.757981581494732,
1318.403428479136664,
1324.052402717296673,
1329.704891897564039,
1335.360883708382744,
1341.020365924141288,
1346.683326404275704,
1352.349753092387118,
1358.019634015366364,
1363.692957282536781,
1369.369711084804067,
1375.049883693819766,
1380.733463461157498,
1386.420438817496006,
1392.110798271819021,
1397.804530410620600,
1403.501623897125000,
1409.202067470514521,
1414.905849945169621,
1420.612960209917674,
1426.323387227291505,
1432.037120032799976,
1437.754147734205389,
1443.474459510811812,
1449.198044612762942,
1454.924892360349304,
1460.654992143322261,
1466.388333420219169,
1472.124905717696947,
1477.864698629875647,
1483.607701817684074,
1489.353905008223364,
1495.103297994130799,
1500.855870632955657,
1506.611612846541448,
1512.370514620418362,
1518.132566003197553,
1523.897757105981555,
1529.666078101774701,
1535.437519224904008,
1541.212070770447326,
1546.989723093669227,
1552.770466609460982,
1558.554291791789865,
1564.341189173155954,
1570.131149344052574,
1575.924162952435836,
1581.720220703200539,
1587.519313357660394,
1593.321431733036661,
1599.126566701952470,
1604.934709191932598,
1610.745850184908250,
1616.559980716732753,
1622.377091876695204,
1628.197174807047077,
1634.020220702529286,
1639.846220809909255,
1645.675166427518889,
1651.507048904801650,
1657.341859641863721,
1663.179590089029034,
1669.020231746402033,
1674.863776163432703,
1680.710214938489798,
1686.559539718436099,
1692.411742198209822,
1698.266814120411709,
1704.124747274894844,
1709.985533498360155,
1715.849164673957603,
1721.715632730890093,
1727.584929644023759,
1733.457047433498474,
1739.331978164349721,
1745.209713946129114,
1751.090246932528999,
1756.973569321017294,
1762.859673352466871,
1768.748551310799485,
1774.640195522624481,
1780.534598356888637,
1786.431752224525098,
1792.331649578106862,
1798.234282911507989,
1804.139644759562316,
1810.047727697730807,
1815.958524341770499,
1821.872027347408448,
1827.788229410015674,
1833.707123264288384,
1839.628701683931922,
1845.552957481345629,
1851.479883507315890,
1857.409472650705084,
1863.341717838153045,
1869.276612033771926,
1875.214148238854250,
1881.154319491573915,
1887.097118866700384,
1893.042539475306512,
1898.990574464486826,
1904.941217017074223,
1910.894460351361658,
1916.850297720826120,
1922.808722413855321,
1928.769727753478264,
1934.733307097096713,
1940.699453836219618,
1946.668161396204823,
1952.639423235995082,
1958.613232847863628,
1964.589583757161563,
1970.568469522062060,
1976.549883733316392,
1982.533820014003140,
1988.520272019287404,
1994.509233436176601,
2000.500697983284454,
2006.494659410590884,
2012.491111499209183,
2018.490048061155903,
2024.491462939116445,
2030.495350006222907,
2036.501703165824438,
2042.510516351266688,
2048.521783525670799,
2054.535498681712852,
2060.551655841410593,
2066.570249055906970,
2072.591272405256404,
2078.614719998216970,
2084.640585972042118,
2090.668864492272405,
2096.699549752533585,
2102.732635974331515,
2108.768117406856618,
2114.805988326778333,
2120.846243038055036,
2126.888875871737582,
2132.933881185772861,
2138.981253364819167,
2145.030986820051567,
2151.083075988974997,
2157.137515335244188,
2163.194299348472668,
2169.253422544054501,
2175.314879462981935,
2181.378664671668957,
2187.444772761773038,
2193.513198350016410,
2199.583936078018723,
2205.656980612119241,
2211.732326643207671,
2217.809968886556362,
2223.889902081652053,
2229.972120992028067,
2236.056620405102876,
2242.143395132015030,
2248.232440007461719,
2254.323749889539613,
2260.417319659583882,
2266.513144222016308,
2272.611218504182489,
2278.711537456202223,
2284.814096050814896,
2290.918889283230328,
2297.025912170972333,
2303.135159753736843,
2309.246627093239113,
2315.360309273070925,
2321.476201398553712,
2327.594298596594854,
2333.714596015545794,
2339.837088825059709,
2345.961772215954170,
2352.088641400067900,
2358.217691610127986,
2364.348918099611183,
2370.482316142607488,
2376.617881033689173,
2382.755608087774817,
2388.895492640001066,
2395.037530045588028,
2401.181715679714216,
2407.328044937382856,
2413.476513233300011,
2419.627116001745890,
2425.779848696450244,
2431.934706790465953,
2438.091685776051690,
2444.250781164543696,
2450.411988486238442,
2456.575303290273041,
2462.740721144504732,
2468.908237635392652,
2475.077848367883689,
2481.249548965294707,
2487.423335069196128,
2493.599202339301883,
2499.777146453352088,
2505.957163107005272,
2512.139248013721044,
2518.323396904658694,
2524.509605528558950,
2530.697869651641213,
2536.888185057494411,
2543.080547546968774,
2549.274952938073511,
2555.471397065868132,
2561.669875782360123,
2567.870384956402631,
2574.072920473590329,
2580.277478236158913,
2586.484054162883695,
2592.692644188980012,
2598.903244266005004,
2605.115850361756202,
2611.330458460177852,
2617.547064561262687,
2623.765664680954615,
2629.986254851054127,
2636.208831119125534,
2642.433389548400555,
2648.659926217688280,
2654.888437221278764,
2661.118918668857532,
2667.351366685407356,
2673.585777411125946,
2679.822147001329540,
2686.060471626368781,
2692.300747471539580,
2698.542970736994903,
2704.787137637658361,
2711.033244403139634,
2717.281287277648062,
2723.531262519907614,
2729.783166403073210,
2736.036995214648414,
2742.292745256401759,
2748.550412844284892,
2754.809994308348905,
2761.071485992669750,
2767.334884255261841,
2773.600185467999381,
2779.867386016540422,
2786.136482300246371,
2792.407470732105139,
2798.680347738651108,
2804.955109759892821,
2811.231753249233861,
2817.510274673400090,
2823.790670512360066,
2830.072937259255923,
2836.357071420326974,
2842.643069514835588,
2848.930928074997610,
2855.220643645906421,
2861.512212785463817,
2867.805632064310885,
2874.100898065750243,
2880.398007385683741,
2886.696956632539695,
2892.997742427202866,
2899.300361402948056,
2905.604810205370086,
2911.911085492317852,
2918.219183933827026,
2924.529102212053658,
2930.840837021205971,
2937.154385067483418,
2943.469743069005290,
2949.786907755752964,
2956.105875869498959,
2962.426644163749643,
2968.749209403676559,
2975.073568366057771,
2981.399717839213281,
2987.727654622942282,
2994.057375528464490,
3000.388877378358757,
3006.722157006498492,
3013.057211257996187,
3019.394036989142478,
3025.732631067345210,
3032.072990371073047,
3038.415111789794082,
3044.758992223920814,
3051.104628584749207,
3057.452017794405037,
3063.801156785784769,
3070.152042502499171,
3076.504671898819197,
3082.859041939616418,
3089.215149600312088,
3095.572991866819848,
3101.932565735492517,
3108.293868213064798,
3114.656896316605525,
3121.021647073457643,
3127.388117521188633,
3133.756304707539130,
3140.126205690367442,
3146.497817537599076,
3152.871137327176257,
3159.246162147004270,
3165.622889094902803,
3172.001315278553648,
3178.381437815453864,
3184.763253832859846,
3191.146760467743661,
3197.531954866741671,
3203.918834186104050,
3210.307395591649765,
3216.697636258715193,
3223.089553372107275,
3229.483144126058050,
3235.878405724173717,
3242.275335379389162,
3248.673930313924757,
3255.074187759233610,
3261.476104955960182,
3267.879679153895268,
3274.284907611925519,
3280.691787597995244,
3287.100316389055024,
3293.510491271020783,
3299.922309538730588,
3306.335768495898265,
3312.750865455069288,
3319.167597737582128,
3325.585962673517770,
3332.005957601664704,
3338.427579869471174,
3344.850826833005158,
3351.275695856909806,
3357.702184314367969,
3364.130289587052630,
3370.560009065091890,
3376.991340147024857,
3383.424280239763903,
3389.858826758551913,
3396.294977126920458,
3402.732728776657041,
3409.172079147757358,
3415.613025688389826,
3422.055565854858287,
3428.499697111558817,
3434.945416930943793,
3441.392722793485518,
3447.841612187631654,
3454.292082609776116,
3460.744131564213149,
3467.197756563105941,
3473.652955126446159,
3480.109724782018475,
3486.568063065363276,
3493.027967519740287,
3499.489435696094461,
3505.952465153014600,
3512.417053456704707,
3518.883198180942145,
3525.350896907046263,
3531.820147223842014,
3538.290946727625396,
3544.763293022125708,
3551.237183718477809,
3557.712616435182099,
3564.189588798070872,
3570.668098440280573,
3577.148143002206325,
3583.629720131482827,
3590.112827482940247,
3596.597462718575116,
3603.083623507519405,
3609.571307526003693,
3616.060512457329423,
3622.551235991831618,
3629.043475826851591,
3635.537229666703752,
3642.032495222640136,
3648.529270212826304,
3655.027552362303140,
3661.527339402959115,
3668.028629073498905,
3674.531419119413840,
3681.035707292951884,
3687.541491353078982,
3694.048769065464967,
3700.557538202436263,
3707.067796542959059,
3713.579541872603841,
3720.092771983516741,
3726.607484674388616,
3733.123677750431398,
3739.641349023343537,
3746.160496311284078,
3752.681117438842193,
3759.203210237013081,
3765.726772543162951,
3772.251802201006285,
3778.778297060576733,
3785.306254978199377,
3791.835673816460712,
3798.366551444187280,
3804.898885736409738,
3811.432674574342855,
3817.967915845355947,
3824.504607442948327,
3831.042747266714741,
3837.582333222333091,
3844.123363221522595,
3850.665835182029241,
3857.209747027593949,
3863.755096687929381,
3870.301882098689475,
3876.850101201451707,
3883.399751943684805,
3889.950832278729195,
3896.503340165763348,
3903.057273569788777,
3909.612630461599110,
3916.169408817757812,
3922.727606620570441,
3929.287221858063276,
3935.848252523959218,
3942.410696617653684,
3948.974552144185964,
3955.539817114220114,
3962.106489544023589,
3968.674567455436318,
3975.244048875849785,
3981.814931838189750,
3988.387214380883506,
3994.960894547843509,
4001.535970388443275,
4008.112439957492370,
4014.690301315213674,
4021.269552527223823,
4027.850191664507292,
4034.432216803400934,
4041.015626025559868,
4047.600417417945209,
4054.186589072799507,
4060.774139087624008,
4067.363065565158195,
4073.953366613354774,
4080.545040345363304,
4087.138084879506096,
4093.732498339255471,
4100.328278853216943,
4106.925424555102836,
4113.523933583717735,
4120.123804082931201,
4126.725034201659582,
4133.327622093849641,
4139.931565918449451,
4146.536863839397483,
4153.143514025595323,
4159.751514650891295,
4166.360863894058639,
4172.971559938776409,
4179.583600973610373,
4186.196985191989370,
4192.811710792193480,
4199.427775977326746,
4206.045178955300798,
4212.663917938817576,
4219.283991145347500,
4225.905396797112189,
4232.528133121062638,
4239.152198348861930,
4245.777590716869781,
4252.404308466119801,
4259.032349842297663,
4265.661713095735649,
4272.292396481379001,
4278.924398258774090,
4285.557716692054782,
4292.192350049916058,
4298.828296605602191,
4305.465554636886736,
4312.104122426052527,
4318.743998259878936,
4325.385180429620050,
4332.027667230987390,
4338.671456964136269,
4345.316547933640322,
4351.962938448488785,
4358.610626822051927,
4365.259611372076506,
4371.909890420663942,
4378.561462294253943,
4385.214325323607227,
4391.868477843790060,
4398.523918194156977,
4405.180644718336225,
4411.838655764207033,
4418.497949683890511,
4425.158524833729643,
4431.820379574274739,
4438.483512270266147,
4445.147921290616978,
4451.813605008399463,
4458.480561800827672,
4465.148790049245690,
4471.818288139103061,
4478.489054459949330,
4485.161087405410399,
4491.834385373178520,
4498.508946764992288,
4505.184769986627543,
4511.861853447874637,
4518.540195562528424,
4525.219794748373715,
4531.900649427163444,
4538.582758024613213,
4545.266118970378557,
4551.950730698046755,
4558.636591645115914,
4565.323700252982235,
4572.012054966929099,
4578.701654236107061,
4585.392496513525657,
4592.084580256033405,
4598.777903924302336,
4605.472465982823451,
4612.168264899882161,
4618.865299147549194,
4625.563567201664227,
4632.263067541825876,
4638.963798651372599,
4645.665759017376331,
4652.368947130617016,
4659.073361485581700,
4665.779000580440879,
4672.485862917043960,
4679.193947000896515,
4685.903251341153918,
4692.613774450607707,
4699.325514845663747,
4706.038471046340419,
4712.752641576250426,
4719.468024962584423,
4726.184619736105560,
4732.902424431129475,
4739.621437585514286,
4746.341657740649680,
4753.063083441440540,
4759.785713236296942,
4766.509545677117785,
4773.234579319284421,
4779.960812721642469,
4786.688244446493627,
4793.416873059578393,
4800.146697130067878,
4806.877715230550166,
4813.609925937017579,
4820.343327828854854,
4827.077919488827320,
4833.813699503069984,
4840.550666461072069,
4847.288818955667011,
4854.028155583024272,
4860.768674942631151,
4867.510375637282777,
4874.253256273074840,
4880.997315459386300,
4887.742551808871212,
4894.488963937443259,
4901.236550464273023,
4907.985310011765250,
4914.735241205553393,
4921.486342674489606,
4928.238613050632011,
4934.992050969230149,
4941.746655068716791,
4948.502423990701573,
4955.259356379948258,
4962.017450884377467,
4968.776706155040301,
4975.537120846123798,
4982.298693614928197,
4989.061423121859661,
4995.825308030422093,
5002.590347007203491,
5009.356538721863217,
5016.123881847129269,
5022.892375058776452,
5029.662017035630015,
5036.432806459538369,
5043.204742015378542,
5049.977822391033442,
5056.752046277391855,
5063.527412368327532,
5070.303919360700093,
5077.081565954335019,
5083.860350852020019,
5090.640272759493200,
5097.421330385429428,
5104.203522441434870,
5110.986847642038811,
5117.771304704676368,
5124.556892349684858,
5131.343609300290154,
5138.131454282600316,
5144.920426025591951,
5151.710523261105664,
5158.501744723831507,
5165.294089151302614,
5172.087555283882466,
5178.882141864758523,
5185.677847639932224,
5192.474671358207161,
5199.272611771182710,
5206.071667633241304,
5212.871837701542972,
5219.673120736014425,
5226.475515499339053,
5233.279020756947830,
5240.083635277010217,
5246.889357830425979,
5253.696187190818819,
5260.504122134519093,
5267.313161440561998,
5274.123303890677562,
5280.934548269278821,
5287.746893363455456,
5294.560337962966514,
5301.374880860225858,
5308.190520850301255,
5315.007256730896188,
5321.825087302349857,
5328.644011367625353,
5335.464027732299655,
5342.285135204556354,
5349.107332595176558,
5355.930618717533434,
5362.754992387574930,
5369.580452423831048,
5376.406997647387470,
5383.234626881890108,
5390.063338953532366,
5396.893132691045139,
5403.724006925689537,
5410.555960491255973,
5417.388992224042340,
5424.223100962855824,
5431.058285549003813,
5437.894544826281162,
5444.731877640966559,
5451.570282841813423,
5458.409759280042636,
5465.250305809330712,
5472.091921285808894,
5478.934604568047689,
5485.778354517053231,
5492.623169996261822,
5499.469049871525385,
5506.315993011111459,
5513.163998285687740,
5520.013064568320260,
5526.863190734466116,
5533.714375661959821,
5540.566618231011489,
5547.419917324197741,
5554.274271826453514,
5561.129680625062974,
5567.986142609657691,
5574.843656672202997,
5581.702221706993441,
5588.561836610649152,
5595.422500282096735,
5602.284211622577459,
5609.146969535628159,
5616.010772927081234,
5622.875620705052825,
5629.741511779936445,
5636.608445064398438,
5643.476419473367969,
5650.345433924033387,
5657.215487335832222,
5664.086578630443000,
5670.958706731781604,
5677.831870565993995,
5684.706069061447124,
5691.581301148724378,
5698.457565760614671,
5705.334861832112438,
5712.213188300403090,
5719.092544104863919,
5725.972928187049547,
5732.854339490692837,
5739.736776961690339,
5746.620239548102290,
5753.504726200146251,
5760.390235870180732,
5767.276767512711558,
5774.164320084376413,
5781.052892543940288,
5787.942483852295481,
5794.833092972442500,
5801.724718869494609,
5808.617360510666913,
5815.511016865269994,
5822.405686904703543,
5829.301369602451814,
5836.198063934073616,
5843.095768877203227,
5849.994483411532201,
5856.894206518817555,
5863.794937182862668,
5870.696674389519103,
5877.599417126677508,
5884.503164384262163,
5891.407915154223701,
5898.313668430536381,
5905.220423209183537,
5912.128178488166668,
5919.036933267481800,
5925.946686549125843,
5932.857437337087504,
5939.769184637339094,
5946.681927457832899,
5953.595664808492984,
5960.510395701211564,
5967.426119149842634,
5974.342834170195601,
5981.260539780030740,
5988.179234999051914,
5995.098918848898393,
6002.019590353148487,
6008.941248537299543,
6015.863892428775216,
6022.787521056913647,
6029.712133452961098,
6036.637728650072859,
6043.564305683294151,
6050.491863589573768,
6057.420401407737700,
6064.349918178500957,
6071.280412944453019,
6078.211884750053287,
6085.144332641624715,
6092.077755667355632,
6099.012152877284279,
6105.947523323299720,
6112.883866059133652,
6119.821180140357683,
6126.759464624374232,
6133.698718570415622,
6140.638941039535894,
6147.580131094603530,
6154.522287800303275,
6161.465410223123399,
6168.409497431352065,
6175.354548495078234,
6182.300562486177114,
6189.247538478312890,
6196.195475546927810,
6203.144372769241272,
6210.094229224240735,
6217.045043992684441,
6223.996816157084140,
6230.949544801707816,
6237.903229012578777,
6244.857867877459285,
6251.813460485856012,
6258.770005929007311,
6265.727503299885029,
6272.685951693182687,
6279.645350205315481,
6286.605697934417549,
6293.566993980327425,
6300.529237444593491,
6307.492427430463977,
6314.456563042882408,
6321.421643388483062,
6328.387667575589148,
6335.354634714203712,
6342.322543916005088,
6349.291394294347811,
6356.261184964248969,
6363.231915042392757,
6370.203583647118649,
6377.176189898420489,
6384.149732917940128,
6391.124211828965599,
6398.099625756421119,
6405.075973826868903,
6412.053255168500073,
6419.031468911130105,
6426.010614186198836,
6432.990690126760455,
6439.971695867482595,
6446.953630544639054,
6453.936493296107074,
6460.920283261366421,
6467.904999581483935,
6474.890641399123524,
6481.877207858529800,
6488.864698105531716,
6495.853111287530737,
6502.842446553504487,
6509.832703053999467,
6516.823879941120140,
6523.815976368536212,
6530.808991491468987,
6537.802924466692275,
6544.797774452525118,
6551.793540608830881,
6558.790222097006335,
6565.787818079988938,
6572.786327722239548,
6579.785750189746977,
6586.786084650021621,
6593.787330272090912,
6600.789486226494773,
6607.792551685281069,
6614.796525822003787,
6621.801407811717581,
6628.807196830970497,
6635.813892057807607,
6642.821492671760097,
6649.829997853842542,
6656.839406786551081,
6663.849718653858872,
6670.860932641207910,
6677.873047935514478,
6684.886063725153690,
6691.899979199964946,
6698.914793551240109,
6705.930505971728053,
6712.947115655621928,
6719.964621798562803,
6726.983023597632382,
6734.002320251347555,
6741.022510959658575,
6748.043594923948149,
6755.065571347020523,
6762.088439433102394,
6769.112198387841090,
6776.136847418295474,
6783.162385732934126,
6790.188812541633524,
6797.216127055673496,
6804.244328487729945,
6811.273416051880304,
6818.303388963587167,
6825.334246439703747,
6832.365987698466597,
6839.398611959493792,
6846.432118443782201,
6853.466506373698394,
6860.501774972979547,
6867.537923466728898,
6874.574951081414838,
6881.612857044862722,
6888.651640586251233,
6895.691300936114203,
6902.731837326328787,
6909.773248990123648,
6916.815535162063497,
6923.858695078052733,
6930.902727975326343,
6937.947633092455362,
6944.993409669335051,
6952.040056947184894,
6959.087574168541323,
6966.135960577263177,
6973.185215418518965,
6980.235337938788689,
6987.286327385855657,
6994.338183008812848,
7001.390904058044725,
7008.444489785238147,
7015.498939443371455,
7022.554252286710835,
7029.610427570811225,
7036.667464552509955,
7043.725362489920371,
7050.784120642438211,
7057.843738270730682,
7064.904214636731012,
7071.965549003640263,
7079.027740635928239,
7086.090788799315305,
7093.154692760788748,
7100.219451788579136,
7107.285065152176685,
7114.351532122314893,
7121.418851970967808,
7128.487023971356393,
7135.556047397933980,
7142.625921526392631,
7149.696645633652224,
7156.768218997864096,
7163.840640898401034,
7170.913910615861823,
7177.988027432058516,
7185.062990630025524,
7192.138799494003251,
7199.215453309448094,
7206.292951363016982,
7213.371292942574655,
7220.450477337185475,
7227.530503837106153,
7234.611371733796659,
7241.693080319902947,
7248.775628889259679,
7255.859016736883859,
7262.943243158982114,
7270.028307452933404,
7277.114208917299038,
7284.200946851809931,
7291.288520557368429,
7298.376929336043759,
7305.466172491071120,
7312.556249326848047,
7319.647159148928040,
7326.738901264022388,
7333.831474979996528,
7340.924879605866408,
7348.019114451791211,
7355.114178829077900,
7362.210072050174858,
7369.306793428669152,
7376.404342279284720,
7383.502717917875088,
7390.601919661428838,
7397.701946828057771,
7404.802798737001467,
7411.904474708620910,
7419.006974064396672,
7426.110296126921639,
7433.214440219910102,
7440.319405668178661,
7447.425191797660773,
7454.531797935387658,
7461.639223409499209,
7468.747467549230350,
7475.856529684917405,
7482.966409147988998,
7490.077105270968786,
7497.188617387463637,
7504.300944832174537,
7511.414086940882044,
7518.528043050448105,
7525.642812498816056,
7532.758394624999710,
7539.874788769093357,
7546.991994272257216,
7554.110010476722891,
7561.228836725785186,
7568.348472363802102,
7575.468916736195752,
7582.590169189438711,
7589.712229071068577,
7596.835095729667955,
7603.958768514872645,
7611.083246777366185,
7618.208529868876212,
7625.334617142176285,
7632.461507951074054,
7639.589201650422183,
7646.717697596101061,
7653.846995145030633,
7660.977093655156750,
7668.107992485453906,
7675.239690995919773,
7682.372188547579754,
7689.505484502476065,
7696.639578223668650,
7703.774469075235174,
7710.910156422262844,
7718.046639630852951,
7725.183918068112689,
7732.321991102157881,
7739.460858102102975,
7746.600518438067411,
7753.740971481169254,
7760.882216603520646,
7768.024253178227809,
7775.167080579388312,
7782.310698182091073,
7789.455105362412723,
7796.600301497411238,
7803.746285965125935,
7810.893058144577481,
7818.040617415766974,
7825.188963159666855,
7832.338094758225452,
7839.488011594356976,
7846.638713051948798,
7853.790198515854172,
7860.942467371886778,
7868.095519006825270,
7875.249352808403273,
7882.403968165315746,
7889.559364467214436,
7896.715541104693330,
7903.872497469309565,
7911.030232953560699,
7918.188746950888344,
7925.348038855686355,
7932.508108063283544,
7939.668953969945505,
7946.830575972885526,
7953.992973470240941,
7961.156145861089499,
7968.320092545430271,
7975.484812924201833,
7982.650306399264082,
7989.816572373396411,
7996.983610250308629,
8004.151419434626405,
8011.319999331889449,
8018.489349348560609,
8025.659468892010409,
8032.830357370521597,
8040.002014193290051,
8047.174438770416600,
8054.347630512902469,
8061.521588832660200,
8068.696313142495455,
8075.871802856120667,
8083.048057388138659,
8090.225076154047201,
8097.402858570242643,
8104.581404054005361,
8111.760712023509768,
8118.940781897812485,
8126.121613096857800,
8133.303205041469482,
8140.485557153353511,
8147.668668855098986,
8154.852539570160843,
8162.037168722877141,
8169.222555738458141,
8176.408700042979035,
8183.595601063392678,
8190.783258227507758,
8197.971670964003351,
8205.160838702424371,
8212.350760873168838,
8219.541436907500611,
8226.732866237540293,
8233.925048296252498,
8241.117982517467681,
8248.311668335863942,
8255.506105186963396,
8262.701292507141261,
8269.897229733616768,
8277.093916304451341,
8284.291351658548592,
8291.489535235650692,
8298.688466476336544,
8305.888144822029062,
8313.088569714973346,
8320.289740598254866,
8327.491656915786734,
8334.694318112311521,
8341.897723633395799,
8349.101872925430143,
8356.306765435632769,
8363.512400612045894,
8370.718777903517548,
8377.925896759725219,
8385.133756631157667,
8392.342356969118555,
8399.551697225721000,
8406.761776853891206,
8413.972595307364827,
8421.184152040677873,
8428.396446509179441,
8435.609478169011709,
8442.823246477129942,
8450.037750891282485,
8457.252990870012582,
8464.468965872662011,
8471.685675359372908,
8478.903118791069573,
8486.121295629473025,
8493.340205337091902,
8500.559847377222468,
8507.780221213946788,
8515.001326312129095,
8522.223162137419422,
8529.445728156240875,
8536.669023835802363,
8543.893048644087685,
8551.117802049855527,
8558.343283522637648,
8565.569492532738877,
8572.796428551231656,
8580.024091049961498,
8587.252479501536072,
8594.481593379327023,
8601.711432157479067,
8608.941995310889979,
8616.173282315216056,
8623.405292646879388,
8630.638025783058765,
8637.871481201678762,
8645.105658381429748,
8652.340556801744242,
8659.576175942811460,
8666.812515285566406,
8674.049574311689867,
8681.287352503613874,
8688.525849344507151,
8695.765064318289660,
8703.004996909608963,
8710.245646603863861,
8717.487012887186211,
8724.729095246442739,
8731.971893169236864,
8739.215406143901419,
8746.459633659505926,
8753.704575205842048,
8760.950230273436318,
8768.196598353539230,
8775.443678938125231,
8782.691471519894549,
8789.939975592265910,
8797.189190649378361,
8804.439116186094907,
8811.689751697995234,
8818.941096681368435,
8826.193150633218465,
8833.445913051273237,
8840.699383433957337,
8847.953561280413851,
8855.208446090491634,
8862.464037364745309,
8869.720334604437085,
8876.977337311527663,
8884.235044988688969,
8891.493457139284146,
8898.752573267380285,
8906.012392877742968,
8913.272915475834452,
8920.534140567806389,
8927.796067660508925,
8935.058696261483419,
8942.322025878958812,
8949.586056021858894,
8956.850786199787763,
8964.116215923042546,
8971.382344702600676,
8978.649172050119887,
8985.916697477947309,
8993.184920499108557,
9000.453840627302270,
9007.723457376909209,
9014.993770262988619,
9022.264778801269131,
9029.536482508157860,
9036.808880900725853,
9044.081973496726278,
9051.355759814570774,
9058.630239373345830,
9065.905411692798225,
9073.181276293344126,
9080.457832696063633,
9087.735080422693500,
9095.013018995641687,
9102.291647937961898,
9109.570966773375403,
9116.850975026258311,
9124.131672221643385,
9131.413057885214585,
9138.695131543308889,
9145.977892722914476,
9153.261340951670718,
9160.545475757866370,
9167.830296670434109,
9175.115803218957808,
9182.401994933659807,
9189.688871345410007,
9196.976431985720410,
9204.264676386739666,
9211.553604081260346,
9218.843214602711669,
9226.133507485157679,
9233.424482263300888,
9240.716138472476814,
9248.008475648650347,
9255.301493328423021,
9262.595191049023924,
9269.889568348313333,
9277.184624764775435,
9284.480359837525612,
9291.776773106297696,
9299.073864111460352,
9306.371632393991604,
9313.670077495498845,
9320.969198958209745,
9328.268996324968612,
9335.569469139234570,
9342.870616945090660,
9350.172439287229281,
9357.474935710957652,
9364.778105762194173,
9372.081948987472060,
9379.386464933932075,
9386.691653149326157,
9393.997513182008333,
9401.304044580947448,
9408.611246895712611,
9415.919119676476839,
9423.227662474015233,
9430.536874839708616,
9437.846756325532624,
9445.157306484068613,
9452.468524868487293,
9459.780411032565098,
9467.092964530667814,
9474.406184917757855,
9481.720071749390627,
9489.034624581716344,
9496.349842971469116,
9503.665726475977863,
9510.982274653160857,
9518.299487061520267,
9525.617363260147613,
9532.935902808716492,
9540.255105267484396,
9547.574970197292714,
9554.895497159564911,
9562.216685716304710,
9569.538535430092452,
9576.861045864090556,
9584.184216582034423,
9591.508047148234255,
9598.832537127584146,
9606.157686085540263,
9613.483493588135389,
9620.809959201975289,
9628.137082494235074,
9635.464863032655558,
9642.793300385552357,
9650.122394121797697,
9657.452143810840425,
9664.782549022684179,
9672.113609327903760,
9679.445324297630577,
9686.777693503558112,
9694.110716517945548,
9701.444392913603224,
9708.778722263903546,
9716.113704142775532,
9723.449338124702990,
9730.785623784724521,
9738.122560698431698,
9745.460148441969068,
9752.798386592035968,
9760.137274725873795,
9767.476812421280556,
9774.816999256601775,
9782.157834810725035,
9789.499318663089070,
9796.841450393672858,
9804.184229583004708,
9811.527655812153171,
9818.871728662725218,
9826.216447716873517,
9833.561812557292797,
9840.907822767207108,
9848.254477930382563,
9855.601777631123696,
9862.949721454273458,
9870.298308985200492,
9877.647539809813679,
9884.997413514553045,
9892.347929686386124,
9899.699087912817049,
9907.050887781875645,
9914.403328882119240,
9921.756410802636310,
9929.110133133035561,
9936.464495463456842,
9943.819497384562055,
9951.175138487533331,
9958.531418364085766,
9965.888336606441953,
9973.245892807351993,
9980.604086560084397,
9987.962917458427910,
9995.322385096684229,
10002.682489069671647,
10010.043228972730503,
10017.404604401708639,
10024.766614952968666,
10032.129260223384335,
10039.492539810349626,
10046.856453311753285,
10054.221000326011563,
10061.586180452031840,
10068.951993289239908,
10076.318438437567238,
10083.685515497449160,
10091.053224069823045,
10098.421563756135583,
10105.790534158330047,
10113.160134878855388,
10120.530365520664418,
10127.901225687201077,
10135.272714982414982,
10142.644833010752336,
10150.017579377157745,
10157.390953687066940,
10164.764955546415877,
10172.139584561637093,
10179.514840339645161,
10186.890722487858511,
10194.267230614186701,
10201.644364327019503,
10209.022123235248728,
10216.400506948244583,
10223.779515075873860,
10231.159147228483562,
10238.539403016910001,
10245.920282052473340,
10253.301783946979413,
10260.683908312717904,
10268.066654762456892,
10275.450022909450126,
10282.834012367427931,
10290.218622750604482,
10297.603853673670528,
10304.989704751797035,
10312.376175600626084,
10319.763265836281789,
10327.150975075363021,
10334.539302934939769,
10341.928249032558597,
10349.317812986237186,
10356.707994414462519,
10364.098792936199970,
10371.490208170873302,
10378.882239738390126,
10386.274887259110983,
10393.668150353872988,
10401.062028643982558,
10408.456521751200853,
10415.851629297763793,
10423.247350906365682,
10430.643686200166485,
10438.040634802788190,
10445.438196338311172,
10452.836370431281466,
10460.235156706701673,
10467.634554790032780,
10475.034564307195978,
10482.435184884567207,
10489.836416148978969,
10497.238257727723976,
10504.640709248544226,
10512.043770339631010,
10519.447440629644916,
10526.851719747681273,
10534.256607323299249,
10541.662102986498212,
10549.068206367735911,
10556.474917097912112,
10563.882234808381327,
10571.290159130940083,
10578.698689697836016,
10586.107826141755140,
10593.517568095836396,
10600.927915193658919,
10608.338867069242042,
10615.750423357052568,
10623.162583691997497,
10630.575347709424022,
10637.988715045119534,
10645.402685335309798,
10652.817258216660775,
10660.232433326274986,
10667.648210301689687,
10675.064588780884151,
10682.481568402263292,
10689.899148804677679,
10697.317329627405343,
10704.736110510155413,
10712.155491093073579,
10719.575471016736628,
10726.996049922146995,
10734.417227450741848,
10741.839003244387641,
10749.261376945372831,
10756.684348196424253,
10764.107916640681651,
10771.532081921724966,
10778.956843683547049,
10786.382201570573670,
10793.808155227652605,
10801.234704300049998,
10808.661848433457635,
10816.089587273991128,
10823.517920468180819,
10830.946847662984510,
10838.376368505771097,
10845.806482644333300,
10853.237189726880388,
10860.668489402034538,
10868.100381318843574,
10875.532865126759134,
10882.965940475656680,
10890.399607015824586,
10897.833864397956859,
10905.268712273167694,
10912.704150292984195,
10920.140178109335466,
10927.576795374568974,
10935.014001741441461,
10942.451796863111667,
10949.890180393156697,
10957.329151985553835,
10964.768711294686000,
10972.208857975349019,
10979.649591682738901,
10987.090912072455467,
10994.532818800507812,
11001.975311523301571,
11009.418389897651650,
11016.862053580765860,
11024.306302230263100,
11031.751135504155172,
11039.196553060855877,
11046.642554559182827,
11054.089139658337444,
11061.536308017937699,
11068.984059297987187,
11076.432393158884224,
11083.881309261427305,
11091.330807266811462,
11098.780886836617356,
11106.231547632831280,
11113.682789317817878,
11121.134611554345611,
11128.587014005568562,
11136.039996335035539,
11143.493558206677335,
11150.947699284824921,
11158.402419234189438,
11165.857717719871289,
11173.313594407363780,
11180.770048962540386,
11188.227081051662026,
11195.684690341377063,
11203.142876498717669,
11210.601639191099821,
11218.060978086319665,
11225.520892852562611,
11232.981383158386961,
11240.442448672740284,
11247.904089064948494,
11255.366304004717676,
11262.829093162128629,
11270.292456207649593,
11277.756392812118065,
11285.220902646755349,
11292.685985383153820,
11300.151640693287845,
11307.617868249504681,
11315.084667724522660,
11322.552038791440282,
11330.019981123727121,
11337.488494395222006,
11344.957578280143935,
11352.427232453075703,
11359.897456588976638,
11367.368250363169864,
11374.839613451358673,
11382.311545529602881,
11389.784046274340653,
11397.257115362372133,
11404.730752470868538,
11412.204957277364883,
11419.679729459763621,
11427.155068696329181,
11434.630974665697067,
11442.107447046861125,
11449.584485519182635,
11457.062089762379401,
11464.540259456538479,
11472.018994282107087,
11479.498293919890784,
11486.978158051053470,
11494.458586357128297,
11501.939578519997667,
11509.421134221907778,
11516.903253145459530,
11524.385934973613985,
11531.869179389686906,
11539.352986077354217,
11546.837354720641088,
11554.322285003929210,
11561.807776611960435,
11569.293829229824041,
11576.780442542964010,
11584.267616237177208,
11591.755349998613383,
11599.243643513773350,
11606.732496469505350,
11614.221908553015965,
11621.711879451850109,
11629.202408853911038,
11636.693496447445796,
11644.185141921050672,
11651.677344963669384,
11659.170105264593076,
11666.663422513454861,
11674.157296400237101,
11681.651726615271400,
11689.146712849220421,
11696.642254793103348,
11704.138352138279515,
11711.635004576448409,
11719.132211799649667,
11726.629973500273991,
11734.128289371041319,
11741.627159105019018,
11749.126582395609148,
11756.626558936563015,
11764.127088421957524,
11771.628170546217007,
11779.129805004100490,
11786.631991490703513,
11794.134729701458127,
11801.638019332134718,
11809.141860078831087,
11816.646251637994283,
11824.151193706389677,
11831.656685981126429,
11839.162728159646576,
11846.669319939715933,
11854.176461019444105,
11861.684151097264476,
11869.192389871943305,
11876.701177042576091,
11884.210512308593024,
11891.720395369748076,
11899.230825926126272,
11906.741803678140059,
11914.253328326531118,
11921.765399572366732,
11929.278017117041600,
11936.791180662276020,
11944.304889910114071,
11951.819144562930887,
11959.333944323421747,
11966.849288894600249,
11974.365177979816508,
11981.881611282729864,
11989.398588507334352,
11996.916109357940513,
12004.434173539171752,
12011.952780755986169,
12019.471930713658367,
12026.991623117773997,
12034.511857674247949,
12042.032634089309795,
12049.553952069509251,
12057.075811321710717,
12064.598211553098736,
12072.121152471170717,
12079.644633783746031,
12087.168655198951456,
12094.693216425237551,
12102.218317171362287,
12109.743957146403773,
12117.270136059751167,
12124.796853621102855,
12132.324109540475547,
12139.851903528196999,
12147.380235294906015,
12154.909104551548808,
12162.438511009384456,
12169.968454379984905,
12177.498934375231329,
12185.029950707310491,
12192.561503088714744,
12200.093591232258404,
12207.626214851046825,
12215.159373658503682,
12222.693067368350967,
12230.227295694625354,
12237.762058351661835,
12245.297355054106447,
12252.833185516905360,
12260.369549455308515,
12267.906446584875084,
12275.443876621462550,
12282.981839281230350,
12290.520334280645329,
12298.059361336469010,
12305.598920165768504,
12313.139010485914696,
12320.679632014573144,
12328.220784469709542,
12335.762467569589717,
12343.304681032783265,
12350.847424578152641,
12358.390697924856795,
12365.934500792358449,
12373.478832900413181,
12381.023693969071246,
12388.569083718683032,
12396.115001869893604,
12403.661448143637244,
12411.208422261155647,
12418.755923943970629,
12426.303952913905960,
12433.852508893074628,
12441.401591603887937,
12448.951200769040952,
12456.501336111530691,
12464.051997354634295,
12471.603184221930860,
12479.154896437283242,
12486.707133724841697,
12494.259895809056616,
12501.813182414656694,
12509.366993266667123,
12516.921328090393217,
12524.476186611433150,
12532.031568555672493,
12539.587473649284220,
12547.143901618725067,
12554.700852190735532,
12562.258325092350788,
12569.816320050882496,
12577.374836793927898,
12584.933875049371636,
12592.493434545380296,
12600.053515010402407,
12607.614116173170260,
12615.175237762701727,
12622.736879508291167,
12630.299041139514884,
12637.861722386236579,
12645.424922978594623,
12652.988642647009328,
12660.552881122179315,
12668.117638135086963,
12675.682913416983865,
12683.248706699412651,
12690.815017714186979,
12698.381846193393358,
12705.949191869407514,
12713.517054474870747,
12721.085433742706300,
12728.654329406115721,
12736.223741198564312,
12743.793668853806594,
12751.364112105864479,
12758.935070689034546,
12766.506544337886226,
12774.078532787263612,
12781.651035772283649,
12789.224053028336130,
12796.797584291080057,
12804.371629296452738,
12811.946187780657056,
12819.521259480163280,
12827.096844131719990,
12834.672941472344974,
12842.249551239316133,
12849.826673170193317,
12857.404307002798305,
12864.982452475214814,
12872.561109325812140,
12880.140277293206964,
12887.719956116297908,
12895.300145534241892,
12902.880845286466865,
12910.462055112662711,
12918.043774752790341,
12925.626003947065328,
12933.208742435979730,
12940.791989960282081,
12948.375746260988308,
12955.960011079379910,
12963.544784156991227,
12971.130065235631264,
12978.715854057361867,
12986.302150364515910,
12993.888953899679109,
13001.476264405700931,
13009.064081625694598,
13016.652405303027990,
13024.241235181338197,
13031.830571004506055,
13039.420412516690703,
13047.010759462293208,
13054.601611585982027,
13062.192968632680277,
13069.784830347569368,
13077.377196476089011,
13084.970066763933573,
13092.563440957055718,
13100.157318801662768,
13107.751700044213067,
13115.346584431430529,
13122.941971710282814,
13130.537861627999519,
13138.134253932064894,
13145.731148370210576,
13153.328544690424678,
13160.926442640946334,
13168.524841970269335,
13176.123742427140314,
13183.723143760555104,
13191.323045719764195,
13198.923448054263645,
13206.524350513807804,
13214.125752848391130,
13221.727654808266379,
13229.330056143931870,
13236.932956606136941,
13244.536355945878313,
13252.140253914398272,
13259.744650263193762,
13267.349544744009108,
13274.954937108823287,
13282.560827109875390,
13290.167214499648253,
13297.774099030866637,
13305.381480456506324,
13312.989358529785022,
13320.597733004167821,
13328.206603633356281,
13335.815970171312074,
13343.425832372226068,
13351.036189990536514,
13358.647042780934498,
13366.258390498336666,
13373.870232897919777,
13381.482569735086145,
13389.095400765492741,
13396.708725745033007,
13404.322544429842310,
13411.936856576292485,
13419.551661941004568,
13427.166960280832427,
13434.782751352868218,
13442.399034914447839,
13450.015810723145478,
13457.633078536773610,
13465.250838113381178,
13472.869089211259052,
13480.487831588927293,
13488.107065005156983,
13495.726789218937483,
13503.347003989512814,
13510.967709076350729,
13518.588904239160911,
13526.210589237885870,
13533.832763832704586,
13541.455427784025233,
13549.078580852503364,
13556.702222799014635,
13564.326353384676622,
13571.950972370837917,
13579.576079519074483,
13587.201674591206029,
13594.827757349281455,
13602.454327555571581,
13610.081384972589149,
13617.708929363077914,
13625.336960490007186,
13632.965478116582744,
13640.594482006235921,
13648.223971922630881,
13655.853947629659160,
13663.484408891441490,
13671.115355472333249,
13678.746787136909916,
13686.378703649979798,
13694.011104776582215,
13701.643990281976585,
13709.277359931655155,
13716.911213491339367,
13724.545550726965303,
13732.180371404712787,
13739.815675290972649,
13747.451462152368549,
13755.087731755746063,
13762.724483868183597,
13770.361718256972381,
13777.999434689638292,
13785.637632933921850,
13793.276312757798223,
13800.915473929459040,
13808.555116217316026,
13816.195239390010101,
13823.835843216404101,
13831.476927465579138,
13839.118491906841882,
13846.760536309713643,
13854.403060443946742,
13862.046064079508142,
13869.689546986584901,
13877.333508935585996,
13884.977949697144140,
13892.622869042101229,
13900.268266741530169,
13907.914142566714872,
13915.560496289162984,
13923.207327680593153,
13930.854636512949583,
13938.502422558391117,
13946.150685589293062,
13953.799425378247179,
13961.448641698068968,
13969.098334321779475,
13976.748503022627119,
13984.399147574062226,
13992.050267749766135,
13999.701863323622092,
14007.353934069738898,
14015.006479762430899,
14022.659500176237998,
14030.312995085896546,
14037.966964266377545,
14045.621407492846629,
14053.276324540694986,
14060.931715185521170,
14068.587579203138375,
14076.243916369568979,
14083.900726461048180,
14091.558009254027638,
14099.215764525160921,
14106.873992051325331,
14114.532691609590984,
14122.191862977257188,
14129.851505931821521,
14137.511620250996202,
14145.172205712698997,
14152.833262095060491,
14160.494789176418635,
14168.156786735322385,
14175.819254550522601,
14183.482192400982967,
14191.145600065874532,
14198.809477324579348,
14206.473823956679553,
14214.138639741962834,
14221.803924460436974,
14229.469677892297113,
14237.135899817960308,
14244.802590018040064,
14252.469748273360892,
14260.137374364945572,
14267.805468074027885,
14275.474029182043523,
14283.143057470633721,
14290.812552721641623,
14298.482514717112281,
14306.152943239303568,
14313.823838070666170,
14321.495198993856320,
14329.167025791735796,
14336.839318247364645,
14344.512076144008461,
14352.185299265129288,
14359.858987394398355,
14367.533140315676064,
14375.207757813041098,
14382.882839670759495,
14390.558385673295561,
14398.234395605324607,
14405.910869251716576,
14413.587806397532404,
14421.265206828047667,
14428.943070328723479,
14436.621396685233776,
14444.300185683432574,
14451.979437109383070,
14459.659150749352193,
14467.339326389786947,
14475.019963817348980,
14482.701062818885475,
14490.382623181447343,
14498.064644692271031,
14505.747127138807627,
14513.430070308682843,
14521.113473989737031,
14528.797337969996079,
14536.481662037676870,
14544.166445981198194,
14551.851689589175294,
14559.537392650408947,
14567.223554953903658,
14574.910176288845832,
14582.597256444631057,
14590.284795210829543,
14597.972792377224323,
14605.661247733773052,
14613.350161070637114,
14621.039532178168884,
14628.729360846908094,
14636.419646867583651,
14644.110390031124552,
14651.801590128648968,
14659.493246951456968,
14667.185360291050529,
14674.877929939118985,
14682.570955687539026,
14690.264437328372878,
14697.958374653884675,
14705.652767456511356,
14713.347615528897222,
14721.042918663863020,
14728.738676654413212,
14736.434889293763263,
14744.131556375288710,
14751.828677692568817,
14759.526253039373842,
14767.224282209646844,
14774.922764997525519,
14782.621701197340371,
14790.321090603594712,
14798.020933010995577,
14805.721228214413713,
14813.421976008925412,
14821.123176189785227,
14828.824828552424151,
14836.526932892476907,
14844.229489005745563,
14851.932496688223182,
14859.635955736090182,
14867.339865945710699,
14875.044227113616216,
14882.749039036551039,
14890.454301511415906,
14898.160014335313463,
14905.866177305511883,
14913.572790219475792,
14921.279852874846256,
14928.987365069448060,
14936.695326601280613,
14944.403737268541590,
14952.112596869586923,
14959.821905202970811,
14967.531662067427533,
14975.241867261858715,
14982.952520585360617,
14990.663621837200481,
14998.375170816829268,
15006.087167323878020,
15013.799611158152402,
15021.512502119641795,
15029.225840008513842,
15036.939624625112629,
15044.653855769964139,
15052.368533243763522,
15060.083656847391467,
15067.799226381912376,
15075.515241648558003,
15083.231702448734723,
15090.948608584032627,
15098.665959856218251,
15106.383756067230934,
15114.101997019190094,
15121.820682514389773,
15129.539812355296817,
15137.259386344556333,
15144.979404284989869,
15152.699865979586320,
15160.420771231523759,
15168.142119844142144,
15175.863911620959698,
15183.586146365671084,
15191.308823882134675,
15199.031943974403475,
15206.755506446679647,
15214.479511103356344,
15222.203957748992252,
15229.928846188315219,
15237.654176226231357,
15245.379947667817760,
15253.106160318326147,
15260.832813983173764,
15268.559908467952482,
15276.287443578428793,
15284.015419120534716,
15291.743834900375077,
15299.472690724229324,
15307.201986398538793,
15314.931721729926721,
15322.661896525170960,
15330.392510591233076,
15338.123563735240168,
15345.855055764486679,
15353.586986486432579,
15361.319355708719741,
15369.052163239141009,
15376.785408885671131,
15384.519092456446742,
15392.253213759775463,
15399.987772604130441,
15407.722768798155812,
15415.458202150652141,
15423.194072470605533,
15430.930379567153068,
15438.667123249608267,
15446.404303327441085,
15454.141919610301557,
15461.879971907990694,
15469.618460030482311,
15477.357383787924846,
15485.096742990612256,
15492.836537449020398,
15500.576766973787016,
15508.317431375702654,
15516.058530465736112,
15523.800064055018083,
15531.542031954841150,
15539.284433976656146,
15547.027269932086710,
15554.770539632914733,
15562.514242891091271,
15570.258379518718357,
15578.002949328070827,
15585.747952131587226,
15593.493387741862534,
15601.239255971655439,
15608.985556633886517,
15616.732289541638238,
15624.479454508156778,
15632.227051346852022,
15639.975079871283015,
15647.723539895181602,
15655.472431232441522,
15663.221753697102031,
15670.971507103371550,
15678.721691265631307,
15686.472305998400770,
15694.223351116372214,
15701.974826434396164,
15709.726731767475940,
15717.479066930776753,
15725.231831739627523,
15732.985026009511785,
15740.738649556073142,
15748.492702195106176,
15756.247183742576453,
15764.002094014600516,
15771.757432827447701,
15779.513199997549236,
15787.269395341500058,
15795.026018676036983,
15802.783069818069634,
15810.540548584653152,
15818.298454793002747,
15826.056788260495523,
15833.815548804652281,
15841.574736243161169,
15849.334350393857676,
15857.094391074735540,
15864.854858103952211,
15872.615751299803378,
15880.377070480748444,
15888.138815465408697,
15895.900986072545493,
15903.663582121089348,
15911.426603430105388,
15919.190049818833359,
15926.953921106654889,
15934.718217113104402,
15942.482937657876391,
15950.248082560814510,
15958.013651641911565,
15965.779644721316799,
15973.546061619339525,
15981.312902156421842,
15989.080166153178652,
15996.847853430364921,
16004.615963808893866,
16012.384497109818767,
16020.153453154354793,
16027.922831763871727,
16035.692632759873959,
16043.462855964033224,
16051.233501198161321,
16059.004568284226480,
16066.776057044346089,
16074.547967300783057,
16082.320298875953085,
16090.093051592419215,
16097.866225272900920,
16105.639819740263192,
16113.413834817512907,
16121.188270327813370,
16128.963126094480685,
16136.738401940969197,
16144.514097690882409,
16152.290213167982074,
16160.066748196168191,
16167.843702599488097,
16175.621076202149197,
16183.398868828488048,
16191.177080302999457,
16198.955710450325569,
16206.734759095250411,
16214.514226062707166,
16222.294111177778177,
16230.074414265687665,
16237.855135151807190,
16245.636273661650193,
16253.417829620886550,
16261.199802855318922,
16268.982193190908220,
16276.765000453748144,
16284.548224470083369,
16292.331865066305909,
16300.115922068942382,
16307.900395304683116,
16315.685284600336672,
16323.470589782875322,
16331.256310679409580,
16339.042447117195479,
16346.828998923623658,
16354.615965926237550,
16362.403347952720651,
16370.191144830903795,
16377.979356388750602,
16385.767982454373850,
16393.557022856031836,
16401.346477422117459,
16409.136345981172781,
16416.926628361878102,
16424.717324393051967,
16432.508433903662080,
16440.299956722810748,
16448.091892679745797,
16455.884241603864211,
16463.677003324679390,
16471.470177671868441,
16479.263764475243079,
16487.057763564742345,
16494.852174770472629,
16502.646997922649462,
16510.442232851652079,
16518.237879387983412,
16526.033937362302822,
16533.830406605389726,
16541.627286948172696,
16549.424578221722186,
16557.222280257235980,
16565.020392886064656,
16572.818915939689759,
16580.617849249734718,
16588.417192647950287,
16596.216945966236381,
16604.017109036627517,
16611.817681691300095,
16619.618663762557844,
16627.420055082846375,
16635.221855484756816,
16643.024064801003988,
16650.826682864448230,
16658.629709508077212,
16666.433144565031398,
16674.236987868571305,
16682.041239252102969,
16689.845898549156118,
16697.650965593413275,
16705.456440218684293,
16713.262322258913628,
16721.068611548180343,
16728.875307920701744,
16736.682411210829741,
16744.489921253043576,
16752.297837881971645,
16760.106160932362400,
16767.914890239102533,
16775.724025637224258,
16783.533566961879842,
16791.343514048356155,
16799.153866732078313,
16806.964624848609674,
16814.775788233637286,
16822.587356722979166,
16830.399330152598850,
16838.211708358587202,
16846.024491177166055,
16853.837678444688208,
16861.651269997641066,
16869.465265672643000,
16877.279665306450624,
16885.094468735940609,
16892.909675798127864,
16900.725286330161907,
16908.541300169323222,
16916.357717153012345,
16924.174537118775334,
16931.991759904285573,
16939.809385347340140,
16947.627413285867078,
16955.445843557939952,
16963.264676001741464,
16971.083910455603473,
16978.903546757970616,
16986.723584747429413,
16994.544024262690073,
17002.364865142597409,
17010.186107226123568,
17018.007750352364383,
17025.829794360550295,
17033.652239090039075,
17041.475084380319458,
17049.298330071000237,
17057.121976001839357,
17064.946022012692993,
17072.770467943573749,
17080.595313634599734,
17088.420558926030935,
17096.246203658251034,
17104.072247671767400,
17111.898690807225648,
17119.725532905384171,
17127.552773807135964,
17135.380413353504991,
17143.208451385631633,
17151.036887744787236,
17158.865722272374114,
17166.694954809918272,
17174.524585199069406,
17182.354613281600905,
17190.185038899424399,
17198.015861894560658,
17205.847082109161420,
17213.678699385516666,
17221.510713566018239,
17229.343124493207142,
17237.175932009729877,
17245.009135958371189,
17252.842736182032240,
17260.676732523745159,
17268.511124826654850,
17276.345912934044463,
17284.181096689313563,
17292.016675935981766,
17299.852650517699658,
17307.689020278245152,
17315.525785061508941,
17323.362944711512682,
17331.200499072394450,
17339.038447988416010,
17346.876791303973732,
17354.715528863576765,
17362.554660511850670,
17370.394186093555618,
17378.234105453568191,
17386.074418436888664,
17393.915124888633727,
17401.756224654058315,
17409.597717578515585,
17417.439603507504216,
17425.281882286621112,
17433.124553761601419,
17440.967617778293061,
17448.811074182667653,
17456.654922820820502,
17464.499163538963330,
17472.343796183424274,
17480.188820600669715,
17488.034236637260619,
17495.880044139896199,
17503.726242955395719,
17511.572832930691220,
17519.419813912823884,
17527.267185748987686,
17535.114948286463914,
17542.963101372657547,
17550.811644855115446,
17558.660578581479058,
17566.509902399520797,
17574.359616157125856,
17582.209719702299481,
17590.060212883174245,
17597.911095547980949,
17605.762367545092275,
17613.614028722979128,
17621.466078930247022,
17629.318518015606969,
17637.171345827886398,
17645.024562216043705,
17652.878167029139149,
17660.732160116363957,
17668.586541327014857,
17676.441310510515905,
17684.296467516396660,
17692.152012194314011,
17700.007944394030346,
17707.864263965439022,
17715.720970758535259,
17723.578064623434329,
17731.435545410378836,
17739.293412969713245,
17747.151667151894799,
17755.010307807518984,
17762.869334787272237,
17770.728747941961046,
17778.588547122522868,
17786.448732179997023,
17794.309302965539246,
17802.170259330414410,
17810.031601126011083,
17817.893328203837882,
17825.755440415501653,
17833.617937612729293,
17841.480819647371391,
17849.344086371380399,
17857.207737636825186,
17865.071773295898311,
17872.936193200894195,
17880.800997204223677,
17888.666185158414009,
17896.531756916097947,
17904.397712330031936,
17912.264051253077923,
17920.130773538214271,
17927.997879038528481,
17935.865367607231747,
17943.733239097629848,
17951.601493363144982,
17959.470130257330311,
17967.339149633829948,
17975.208551346404420,
17983.078335248934309,
17990.948501195405697,
17998.819049039913807,
18006.689978636670276,
18014.561289839992241,
18022.432982504313259,
18030.305056484183297,
18038.177511634246912,
18046.050347809272353,
18053.923564864137006,
18061.797162653820124,
18069.671141033424647,
18077.545499858159019,
18085.420238983326271,
18093.295358264371316,
18101.170857556815463,
18109.046736716307350,
18116.922995598612033,
18124.799634059589152,
18132.676651955211128,
18140.554049141563155,
18148.431825474839570,
18156.309980811347486,
18164.188515007484966,
18172.067427919781039,
18179.946719404862961,
18187.826389319467125,
18195.706437520442705,
18203.586863864737097,
18211.467668209406838,
18219.348850411635794,
18227.230410328691505,
18235.112347817961563,
18242.994662736946339,
18250.877354943233513,
18258.760424294538097,
18266.643870648676966,
18274.527693863568857,
18282.411893797241646,
18290.296470307839627,
18298.181423253601679,
18306.066752492875821,
18313.952457884115574,
18321.838539285890874,
18329.724996556866245,
18337.611829555822624,
18345.499038141639176,
18353.386622173300566,
18361.274581509900599,
18369.162916010638583,
18377.051625534819323,
18384.940709941856767,
18392.830169091263087,
18400.720002842655958,
18408.610211055762193,
18416.500793590421381,
18424.391750306560425,
18432.283081064222642,
18440.174785723553214,
18448.066864144799183,
18455.959316188320372,
18463.852141714571189,
18471.745340584118821,
18479.638912657621404,
18487.532857795857126,
18495.427175859698764,
18503.321866710128234,
18511.216930208218400,
18519.112366215158545,
18527.008174592243449,
18534.904355200858845,
18542.800907902499603,
18550.697832558769733,
18558.595129031364195,
18566.492797182090726,
18574.390836872855289,
18582.289247965665709,
18590.188030322638951,
18598.087183805979294,
18605.986708278014703,
18613.886603601153183,
18621.786869637919153,
18629.687506250938895,
18637.588513302933279,
18645.489890656725038,
18653.391638175246044,
18661.293755721522757,
18669.196243158683501,
18677.099100349965738,
18685.002327158694243,
18692.905923448310205,
18700.809889082342124,
18708.714223924427642,
18716.618927838298987,
18724.524000687801163,
18732.429442336855573,
18740.335252649518225,
18748.241431489914248,
18756.147978722281550,
18764.054894210959901,
18771.962177820383658,
18779.869829415099957,
18787.777848859732330,
18795.686236019017088,
18803.594990757803316,
18811.504112941012863,
18819.413602433687629,
18827.323459100956825,
18835.233682808055164,
18843.144273420311947,
18851.055230803158338,
18858.966554822120088,
18866.878245342832088,
18874.790302231009264,
18882.702725352482958,
18890.615514573171822,
18898.528669759096374,
18906.442190776382631,
18914.356077491240285,
18922.270329769984528,
18930.184947479021503,
18938.099930484870129,
18946.015278654132999,
18953.930991853518208,
18961.847069949821162,
18969.763512809939130,
18977.680320300878520,
18985.597492289725778,
18993.515028643669211,
19001.432929229995352,
19009.351193916092598,
19017.269822569433018,
19025.188815057601460,
19033.108171248262806,
19041.027891009183804,
19048.947974208240339,
19056.868420713377418,
19064.789230392667378,
19072.710403114255314,
19080.631938746391825,
19088.553837157414819,
19096.476098215767706,
19104.398721789984847,
19112.321707748695189,
19120.245055960625905,
19128.168766294595116,
19136.092838619515533,
19144.017272804409004,
19151.942068718362862,
19159.867226230588130,
19167.792745210372232,
19175.718625527111726,
19183.644867050283210,
19191.571469649461505,
19199.498433194326935,
19207.425757554636220,
19215.353442600251583,
19223.281488201126194,
19231.209894227307814,
19239.138660548935150,
19247.067787036245136,
19254.997273559551104,
19262.927119989297353,
19270.857326195982751,
19278.787892050218943,
19286.718817422701250,
19294.650102184226853,
19302.581746205680247,
19310.513749358040513,
19318.446111512381322,
19326.378832539863652,
19334.311912311743072,
19342.245350699369737,
19350.179147574188391,
19358.113302807723812,
19366.047816271606280,
19373.982687837549747,
19381.917917377366393,
19389.853504762955708,
19397.789449866308132,
19405.725752559512330,
19413.662412714740640,
19421.599430204252712,
19429.536804900417337,
19437.474536675676063,
19445.412625402572303,
19453.351070953740418,
19461.289873201891169,
19469.229032019848091,
19477.168547280514758,
19485.108418856874778,
19493.048646622020897,
19500.989230449125898,
19508.930170211453515,
19516.871465782358428,
19524.813117035286268,
19532.755123843777255,
19540.697486081451643,
19548.640203622027911,
19556.583276339304575,
19564.526704107182013,
19572.470486799640639,
19580.414624290755455,
19588.359116454685136,
19596.303963165686582,
19604.249164298100368,
19612.194719726354379,
19620.140629324967449,
19628.086892968549364,
19636.033510531793581,
19643.980481889484508,
19651.927806916504778,
19659.875485487806145,
19667.823517478442227,
19675.771902763553953,
19683.720641218369565,
19691.669732718200976,
19699.619177138443774,
19707.568974354609963,
19715.519124242258840,
19723.469626677069755,
19731.420481534791179,
19739.371688691262534,
19747.323248022417829,
19755.275159404271108,
19763.227422712931002,
19771.180037824578903,
19779.133004615501704,
19787.086322962062695,
19795.039992740712478,
19802.994013827992603,
19810.948386100524658,
19818.903109435021179,
19826.858183708285651,
19834.813608797197958,
19842.769384578732570,
19850.725510929943994,
19858.681987727984961,
19866.638814850070048,
19874.595992173530249,
19882.553519575762039,
19890.511396934249206,
19898.469624126573763,
19906.428201030390483,
19914.387127523437812,
19922.346403483556060,
19930.306028788654658,
19938.266003316730348,
19946.226326945885376,
19954.186999554269278,
19962.148021020151646,
19970.109391221871192,
19978.071110037853941,
19986.033177346609591,
19993.995593026727875,
20001.958356956896750,
20009.921469015876937,
20017.884929082516464,
20025.848737035743397,
20033.812892754587665,
20041.777396118137403,
20049.742247005589888,
20057.707445296200603,
20065.672990869326895,
20073.638883604413422,
20081.605123380973964,
20089.571710078613251,
20097.538643577019684,
20105.505923755961703,
20113.473550495291420,
20121.441523674955533,
20129.409843174966227,
20137.378508875433909,
20145.347520656538109,
20153.316878398556582,
20161.286581981836207,
20169.256631286814809,
20177.227026194002974,
20185.197766584009514,
20193.168852337516000,
20201.140283335287677,
20209.112059458166186,
20217.084180587087758,
20225.056646603065019,
20233.029457387183356,
20241.002612820630020,
20248.976112784654106,
20256.949957160595659,
20264.924145829882036,
20272.898678674009716,
20280.873555574573402,
20288.848776413226005,
20296.824341071718663,
20304.800249431886186,
20312.776501375628868,
20320.753096784948866,
20328.730035541910183,
20336.707317528664134,
20344.684942627449345,
20352.662910720577202,
20360.641221690442762,
20368.619875419528398,
20376.598871790381963,
20384.578210685645900,
20392.557891988031770,
20400.537915580342087,
20408.518281345452124,
20416.498989166320825,
20424.480038925987174,
20432.461430507566547,
20440.443163794261636,
20448.425238669344253,
20456.407655016166245,
20464.390412718177686,
20472.373511658894131,
20480.356951721896621,
20488.340732790875336,
20496.324854749575024,
20504.309317481838661,
20512.294120871574705,
20520.279264802775288,
20528.264749159508938,
20536.250573825927859,
20544.236738686260651,
20552.223243624812312,
20560.210088525975152,
20568.197273274206964,
20576.184797754056490,
20584.172661850145232,
20592.160865447163815,
20600.149408429904724,
20608.138290683211380,
20616.127512092029065,
20624.117072541357629,
20632.106971916306065,
20640.097210102023382,
20648.087786983767728,
20656.078702446859097,
20664.069956376701157,
20672.061548658766696,
20680.053479178619455,
20688.045747821892292,
20696.038354474290827,
20704.031299021607992,
20712.024581349713117,
20720.018201344537374,
20728.012158892110165,
20736.006453878526372,
20744.001086189957277,
20751.996055712657835,
20759.991362332948484,
20767.987005937236972,
20775.982986412000173,
20783.979303643798630,
20791.975957519258372,
20799.972947925096378,
20807.970274748095107,
20815.967937875113421,
20823.965937193093851,
20831.964272589044413,
20839.962943950064073,
20847.961951163306367,
20855.961294116019417,
20863.960972695520468,
20871.960986789199524,
20879.961336284522986,
20887.962021069033653,
20895.963041030361637,
20903.964396056184341,
20911.966086034291948,
20919.968110852500104,
20927.970470398751786,
20935.973164561040903,
20943.976193227423209,
20951.979556286052684,
20959.983253625148791,
20967.987285133003752,
20975.991650697978912,
20983.996350208533840,
20992.001383553171763,
21000.006750620479579,
21008.012451299146051,
21016.018485477892682,
21024.024853045546479,
21032.031553890985379,
21040.038587903178268,
21048.045954971163155,
21056.053654984047171,
21064.061687831017480,
21072.070053401326732,
21080.078751584314887,
21088.087782269383752,
21096.097145346018806,
21104.106840703760099,
21112.116868232238630,
21120.127227821161796,
21128.137919360291562,
21136.148942739480844,
21144.160297848640766,
21152.171984577769763,
21160.184002816928114,
21168.196352456256136,
21176.209033385963266,
21184.222045496328064,
21192.235388677712763,
21200.249062820552354,
21208.263067815329123,
21216.277403552630858,
21224.292069923096278,
21232.307066817444138,
21240.322394126465952,
21248.338051741018717,
21256.354039552046743,
21264.370357450552547,
21272.387005327607767,
21280.403983074374992,
21288.421290582064103,
21296.438927741968655,
21304.456894445462240,
21312.475190583980293,
21320.493816049023735,
21328.512770732180797,
21336.532054525094281,
21344.551667319497938,
21352.571609007172810,
21360.591879479990894,
21368.612478629893303,
21376.633406348872086,
21384.654662529013876,
21392.676247062470793,
21400.698159841453162,
21408.720400758265896,
21416.742969705250289,
21424.765866574853135,
21432.789091259572160,
21440.812643651974213,
21448.836523644706176,
21456.860731130484055,
21464.885266002092976,
21472.910128152379912,
21480.935317474271869,
21488.960833860757702,
21496.986677204909938,
21505.012847399855673,
21513.039344338798401,
21521.066167915018013,
21529.093318021852610,
21537.120794552713051,
21545.148597401082952,
21553.176726460518694,
21561.205181624631223,
21569.233962787115161,
21577.263069841737888,
21585.292502682314080,
21593.322261202756636,
21601.352345297025749,
21609.382754859154375,
21617.413489783259138,
21625.444549963493046,
21633.475935294121882,
21641.507645669440535,
21649.539680983838480,
21657.572041131767037,
21665.604726007732097,
21673.637735506334138,
21681.671069522210018,
21689.704727950102097,
21697.738710684781836,
21705.773017621122563,
21713.807648654037621,
21721.842603678545856,
21729.877882589687943,
21737.913485282610054,
21745.949411652498384,
21753.985661594630074,
21762.022235004340473,
21770.059131777026778,
21778.096351808155305,
21786.133894993279682,
21794.171761227989919,
21802.209950407956057,
21810.248462428935454,
21818.287297186721844,
21826.326454577196273,
21834.365934496297996,
21842.405736840031750,
21850.445861504482309,
21858.486308385781740,
21866.527077380156697,
21874.568168383859302,
21882.609581293250812,
21890.651316004743421,
21898.693372414800251,
21906.735750419971737,
21914.778449916873797,
21922.821470802173280,
21930.864812972613436,
21938.908476325002994,
21946.952460756223445,
21954.996766163218126,
21963.041392442984943,
21971.086339492601837,
21979.131607209208596,
21987.177195490010490,
21995.223104232285550,
22003.269333333359100,
22011.315882690640137,
22019.362752201603143,
22027.409941763777169,
22035.457451274753112,
22043.505280632209178,
22051.553429733878147,
22059.601898477547365,
22067.650686761084216,
22075.699794482410653,
22083.749221539514110,
22091.798967830469337,
22099.849033253387461,
22107.899417706456006,
22115.950121087924344,
22124.001143296118244,
22132.052484229403490,
22140.104143786251370,
22148.156121865155001,
22156.208418364691170,
22164.261033183502150,
22172.313966220299335,
22180.367217373848689,
22188.420786542985297,
22196.474673626606091,
22204.528878523669846,
22212.583401133208099,
22220.638241354303318,
22228.693399086121644,
22236.748874227883789,
22244.804666678861395,
22252.860776338409778,
22260.917203105931549,
22268.973946880905714,
22277.031007562869490,
22285.088385051429213,
22293.146079246245790,
22301.204090047049249,
22309.262417353631463,
22317.321061065842514,
22325.380021083612519,
22333.439297306922526,
22341.498889635804517,
22349.558797970385058,
22357.619022210823459,
22365.679562257362704,
22373.740418010296708,
22381.801589369992143,
22389.863076236862980,
22397.924878511403222,
22405.986996094150527,
22414.049428885733505,
22422.112176786824421,
22430.175239698153746,
22438.238617520517437,
22446.302310154787847,
22454.366317501884623,
22462.430639462789259,
22470.495275938566920,
22478.560226830319152,
22486.625492039212986,
22494.691071466499125,
22502.756965013464651,
22510.823172581465769,
22518.889694071935082,
22526.956529386356124,
22535.023678426263359,
22543.091141093278566,
22551.158917289056262,
22559.227006915331003,
22567.295409873899189,
22575.364126066619065,
22583.433155395392532,
22591.502497762205167,
22599.572153069089836,
22607.642121218152170,
22615.712402111545089,
22623.782995651497913,
22631.853901740279980,
22639.925120280255214,
22647.996651173805731,
22656.068494323419145,
22664.140649631601264,
22672.213117000956117,
22680.285896334124118,
22688.358987533825712,
22696.432390502810449,
22704.506105143915192,
22712.580131360045925,
22720.654469054134097,
22728.729118129198469,
22736.804078488316009,
22744.879350034614617,
22752.954932671284041,
22761.030826301579509,
22769.107030828825373,
22777.183546156371449,
22785.260372187669418,
22793.337508826214616,
22801.414955975546036,
22809.492713539279066,
22817.570781421094580,
22825.649159524717106,
22833.727847753951210,
22841.806846012634196,
22849.886154204690683,
22857.965772234078031,
22866.045700004837272,
22874.125937421049457,
22882.206484386879310,
22890.287340806520660,
22898.368506584243733,
22906.449981624380598,
22914.531765831314260,
22922.613859109489567,
22930.696261363424128,
22938.778972497661016,
22946.861992416830617,
22954.945321025617886,
22963.028958228755073,
22971.112903931054461,
22979.197158037361078,
22987.281720452596346,
22995.366591081736260,
23003.451769829804107,
23011.537256601906847,
23019.623051303195098,
23027.709153838855855,
23035.795564114185254,
23043.882282034490345,
23051.969307505154575,
23060.056640431630512,
23068.144280719410744,
23076.232228274049703,
23084.320483001174580,
23092.409044806452584,
23100.497913595609134,
23108.587089274449681,
23116.676571748812421,
23124.766360924601031,
23132.856456707777397,
23140.946859004368889,
23149.037567720457446,
23157.128582762168662,
23165.219904035700893,
23173.311531447299785,
23181.403464903280110,
23189.495704310007568,
23197.588249573895155,
23205.681100601435901,
23213.774257299155579,
23221.867719573656359,
23229.961487331591343,
23238.055560479660926,
23246.149938924634625,
23254.244622573329252,
23262.339611332634377,
23270.434905109483225,
23278.530503810852679,
23286.626407343821484,
23294.722615615472023,
23302.819128532966715,
23310.915946003540739,
23319.013067934462015,
23327.110494233056670,
23335.208224806727230,
23343.306259562898049,
23351.404598409088067,
23359.503241252845328,
23367.602188001790637,
23375.701438563584816,
23383.800992845961446,
23391.900850756697764,
23400.001012203640130,
23408.101477094664006,
23416.202245337739441,
23424.303316840858315,
23432.404691512085265,
23440.506369259539497,
23448.608349991394789,
23456.710633615875850,
23464.813220041265595,
23472.916109175905149,
23481.019300928193843,
23489.122795206574665,
23497.226591919552448,
23505.330690975701145,
23513.435092283616541,
23521.539795751989004,
23529.644801289534371,
23537.750108805044874,
23545.855718207341852,
23553.961629405326676,
23562.067842307951651,
23570.174356824205461,
23578.281172863156826,
23586.388290333903569,
23594.495709145623550,
23602.603429207534646,
23610.711450428909302,
23618.819772719085449,
23626.928395987437398,
23635.037320143415855,
23643.146545096500631,
23651.256070756258850,
23659.365897032272187,
23667.476023834213265,
23675.586451071791089,
23683.697178654765594,
23691.808206492958561,
23699.919534496246342,
23708.031162574552582,
23716.143090637862770,
23724.255318596213328,
23732.367846359691612,
23740.480673838443181,
23748.593800942664529,
23756.707227582610358,
23764.820953668579023,
23772.934979110934364,
23781.049303820091154,
23789.163927706515096,
23797.278850680719188,
23805.394072653281910,
23813.509593534829037,
23821.625413236037275,
23829.741531667648815,
23837.857948740438587,
23845.974664365257922,
23854.091678452998167,
23862.208990914597962,
23870.326601661065069,
23878.444510603447270,
23886.562717652857827,
23894.681222720442747,
23902.800025717420795,
23910.919126555061666,
23919.038525144667801,
23927.158221397628949,
23935.278215225356689,
23943.398506539324444,
23951.519095251060207,
23959.639981272157456,
23967.761164514235134,
23975.882644888981304,
23984.004422308145877,
23992.126496683507867,
24000.248867926911771,
24008.371535950256657,
24016.494500665492524,
24024.617761984616664,
24032.741319819677301,
24040.865174082784506,
24048.989324686088366,
24057.113771541804454,
24065.238514562191995,
24073.363553659561148,
24081.488888746276643,
24089.614519734750502,
24097.740446537460230,
24105.866669066919712,
24113.993187235697405,
24122.120000956423610,
24130.247110141768644,
24138.374514704464673,
24146.502214557283878,
24154.630209613056650,
24162.758499784660671,
24170.887084985035472,
24179.015965127156960,
24187.145140124077443,
24195.274609888856503,
24203.404374334648310,
24211.534433374643413,
24219.664786922076019,
24227.795434890238539,
24235.926377192467044,
24244.057613742163085,
24252.189144452768232,
24260.320969237771351,
24268.453088010730426,
24276.585500685232546,
24284.718207174923009,
24292.851207393505319,
24300.984501254726638,
24309.118088672395970,
24317.251969560340513,
24325.386143832478410,
24333.520611402756913,
24341.655372185174201,
24349.790426093786664,
24357.925773042694345,
24366.061412946048222,
24374.197345718053839,
24382.333571272956760,
24390.470089525078947,
24398.606900388756912,
24406.744003778392653,
24414.881399608453648,
24423.019087793425570,
24431.157068247881398,
24439.295340886405029,
24447.433905623667670,
24455.572762374369631,
24463.711911053247604,
24471.851351575125591,
24479.991083854842145,
24488.131107807304943,
24496.271423347468954,
24504.412030390321888,
24512.552928850931494,
24520.694118644390983,
24528.835599685848138,
24536.977371890505310,
24545.119435173608508,
24553.261789450461947,
24561.404434636395308,
24569.547370646836498,
24577.690597397198871,
24585.834114802994009,
24593.977922779773508,
24602.122021243110794,
24610.266410108652053,
24618.411089292105316,
24626.556058709189529,
24634.701318275710946,
24642.846867907494016,
24650.992707520428667,
24659.138837030452123,
24667.285256353552541,
24675.431965405754454,
24683.578964103144244,
24691.726252361848310,
24699.873830098054896,
24708.021697227977711,
24716.169853667899588,
24724.318299334139738,
24732.467034143079218,
24740.616058011135465,
24748.765370854769571,
24756.914972590504476,
24765.064863134906773,
24773.215042404586711,
24781.365510316212749,
24789.516266786489723,
24797.667311732173403,
24805.818645070074126,
24813.970266717045888,
24822.122176589986339,
24830.274374605844059,
24838.426860681625840,
24846.579634734371211,
24854.732696681170637,
24862.886046439165511,
24871.039683925555437,
24879.193609057561844,
24887.347821752471646,
24895.502321927626326,
24903.657109500396473,
24911.812184388214519,
24919.967546508538362,
24928.123195778902300,
24936.279132116873370,
24944.435355440069543,
24952.591865666148806,
24960.748662712823716,
24968.905746497854125,
24977.063116939039901,
24985.220773954239121,
24993.378717461346241,
25001.536947378299374,
25009.695463623109390,
25017.854266113798076,
25026.013354768470890,
25034.172729505247844,
25042.332390242310794,
25050.492336897892528,
25058.652569390254939,
25066.813087637732679,
25074.973891558685864,
25083.134981071529182,
25091.296356094728253,
25099.458016546785075,
25107.619962346256216,
25115.782193411741900,
25123.944709661875095,
25132.107511015372438,
25140.270597390954208,
25148.433968707409804,
25156.597624883583194,
25164.761565838336537,
25172.925791490604752,
25181.090301759348222,
25189.255096563600091,
25197.420175822397141,
25205.585539454878017,
25213.751187380166812,
25221.917119517489482,
25230.083335786082898,
25238.249836105238501,
25246.416620394295023,
25254.583688572638493,
25262.751040559694957,
25270.918676274941390,
25279.086595637898427,
25287.254798568130354,
25295.423284985256032,
25303.592054808937974,
25311.761107958864159,
25319.930444354791689,
25328.100063916510408,
25336.269966563875641,
25344.440152216753631,
25352.610620795087016,
25360.781372218840261,
25368.952406408043316,
25377.123723282766150,
25385.295322763115109,
25393.467204769240197,
25401.639369221353263,
25409.811816039691621,
25417.984545144561707,
25426.157556456284510,
25434.330849895250140,
25442.504425381881447,
25450.678282836655853,
25458.852422180090798,
25467.026843332732824,
25475.201546215201233,
25483.376530748144432,
25491.551796852258121,
25499.727344448285294,
25507.903173456998047,
25516.079283799233963,
25524.255675395867001,
25532.432348167814780,
25540.609302036042209,
25548.786536921546940,
25556.964052745399385,
25565.141849428677233,
25573.319926892523654,
25581.498285058129113,
25589.676923846724094,
25597.855843179571821,
25606.035042977990088,
25614.214523163351259,
25622.394283657049527,
25630.574324380540929,
25638.754645255317882,
25646.935246202912822,
25655.116127144901839,
25663.297288002930145,
25671.478728698646592,
25679.660449153780064,
25687.842449290070363,
25696.024729029330047,
25704.207288293397141,
25712.390127004164242,
25720.573245083560323,
25728.756642453558015,
25736.940319036177243,
25745.124274753477948,
25753.308509527581919,
25761.493023280610942,
25769.677815934777755,
25777.862887412306009,
25786.048237635484838,
25794.233866526636120,
25802.419774008118111,
25810.605960002343636,
25818.792424431761901,
25826.979167218880320,
25835.166188286224497,
25843.353487556374603,
25851.541064951965382,
25859.728920395660680,
25867.917053810175275,
25876.105465118253051,
25884.294154242699733,
25892.483121106350154,
25900.672365632082801,
25908.861887742830731,
25917.051687361556105,
25925.241764411275653,
25933.432118815042486,
25941.622750495942455,
25949.813659377130534,
25958.004845381772611,
25966.196308433092781,
25974.388048454369709,
25982.580065368907526,
25990.772359100057656,
25998.964929571211542,
26007.157776705807919,
26015.350900427314627,
26023.544300659268629,
26031.737977325221436,
26039.931930348790047,
26048.126159653606010,
26056.320665163373633,
26064.515446801815415,
26072.710504492712062,
26080.905838159873383,
26089.101447727163759,
26097.297333118476672,
26105.493494257763814,
26113.689931068995065,
26121.886643476209429,
26130.083631403467734,
26138.280894774881745,
26146.478433514606877,
26154.676247546824015,
26162.874336795783165,
26171.072701185748883,
26179.271340641047573,
26187.470255086031102,
26195.669444445113186,
26203.868908642718452,
26212.068647603355203,
26220.268661251531739,
26228.468949511814571,
26236.669512308828416,
26244.870349567201629,
26253.071461211646238,
26261.272847166885185,
26269.474507357695984,
26277.676441708892526,
26285.878650145325082,
26294.081132591902133,
26302.283888973557623,
26310.486919215269154,
26318.690223242065258,
26326.893800979003572,
26335.097652351192664,
26343.301777283766569,
26351.506175701913889,
26359.710847530866886,
26367.915792695879645,
26376.121011122279015,
26384.326502735395479,
26392.532267460635921,
26400.738305223410862,
26408.944615949207218,
26417.151199563526461,
26425.358055991924630,
26433.565185159997782,
26441.772586993374716,
26449.980261417731526,
26458.188208358777047,
26466.396427742271044,
26474.604919494013302,
26482.813683539836347,
26491.022719805609086,
26499.232028217254992,
26507.441608700733923,
26515.651461182034836,
26523.861585587197624,
26532.071981842302193,
26540.282649873464834,
26548.493589606841851,
26556.704800968640484,
26564.916283885082521,
26573.128038282455236,
26581.340064087082283,
26589.552361225316417,
26597.764929623543139,
26605.977769208220707,
26614.190879905814654,
26622.404261642852362,
26630.617914345886675,
26638.831837941510457,
26647.046032356363867,
26655.260497517119802,
26663.475233350498456,
26671.690239783263678,
26679.905516742193868,
26688.121064154143824,
26696.336881945979258,
26704.552970044609538,
26712.769328376998601,
26720.985956870128575,
26729.202855451043433,
26737.420024046808976,
26745.637462584541936,
26753.855170991384512,
26762.073149194533471,
26770.291397121218324,
26778.509914698712237,
26786.728701854317478,
26794.947758515369060,
26803.167084609278390,
26811.386680063456879,
26819.606544805366866,
26827.826678762521624,
26836.047081862456253,
26844.267754032749508,
26852.488695201031078,
26860.709905294956116,
26868.931384242223430,
26877.153131970571849,
26885.375148407772940,
26893.597433481645567,
26901.819987120041333,
26910.042809250855498,
26918.265899802019703,
26926.489258701498329,
26934.712885877303052,
26942.936781257485563,
26951.160944770123024,
26959.385376343343523,
26967.610075905307895,
26975.835043384224264,
26984.060278708326223,
26992.285781805894658,
27000.511552605243196,
27008.737591034725483,
27016.963897022742458,
27025.190470497724164,
27033.417311388122471,
27041.644419622472924,
27049.871795129314705,
27058.099437837216101,
27066.327347674810881,
27074.555524570761918,
27082.783968453764828,
27091.012679252558883,
27099.241656895919732,
27107.470901312655769,
27115.700412431615405,
27123.930190181697981,
27132.160234491824667,
27140.390545290960290,
27148.621122508106055,
27156.851966072306823,
27165.083075912632921,
27173.314451958209247,
27181.546094138182525,
27189.778002381746774,
27198.010176618128753,
27206.242616776602517,
27214.475322786460310,
27222.708294577052584,
27230.941532077762531,
27239.175035217991535,
27247.408803927210101,
27255.642838134903286,
27263.877137770599802,
27272.111702763864741,
27280.346533044306852,
27288.581628541563987,
27296.816989185321290,
27305.052614905285736,
27313.288505631211592,
27321.524661292896781,
27329.761081820164691,
27337.997767142878729,
27346.234717190935044,
27354.471931894284353,
27362.709411182899203,
27370.947154986792157,
27379.185163236012158,
27387.423435860640893,
27395.661972790814616,
27403.900773956684134,
27412.139839288454823,
27420.379168716350250,
27428.618762170663103,
27436.858619581678795,
27445.098740879755496,
27453.339125995276845,
27461.579774858651945,
27469.820687400337192,
27478.061863550832641,
27486.303303240660171,
27494.545006400388957,
27502.786972960624553,
27511.029202851997979,
27519.271696005183912,
27527.514452350900683,
27535.757471819888451,
27544.000754342938308,
27552.244299850863172,
27560.488108274526894,
27568.732179544822429,
27576.976513592679112,
27585.221110349062656,
27593.465969744978793,
27601.711091711455083,
27609.956476179573656,
27618.202123080449383,
27626.448032345218962,
27634.694203905080940,
27642.940637691237498,
27651.187333634959941,
27659.434291667523212,
27667.681511720271374,
27675.928993724555767,
27684.176737611782301,
27692.424743313382351,
27700.673010760827310,
27708.921539885632228,
27717.170330619326705,
27725.419382893498550,
27733.668696639761038,
27741.918271789756545,
27750.168108275185659,
27758.418206027752603,
27766.668564979234361,
27774.919185061411554,
27783.170066206112097,
27791.421208345203922,
27799.672611410580430,
27807.924275334189588,
27816.176200047990278,
27824.428385483988677,
27832.680831574238255,
27840.933538250807032,
27849.186505445803050,
27857.439733091385278,
27865.693221119734517,
27873.946969463064306,
27882.200978053628205,
27890.455246823719790,
27898.709775705658103,
27906.964564631805843,
27915.219613534558448,
27923.474922346344101,
27931.730490999631002,
27939.986319426912814,
27948.242407560723223,
27956.498755333643203,
27964.755362678268284,
27973.012229527241288,
27981.269355813241418,
27989.526741468973341,
27997.784386427178106,
28006.042290620647691,
28014.300453982181352,
28022.558876444640191,
28030.817557940903498,
28039.076498403897858,
28047.335697766560770,
28055.595155961895216,
28063.854872922915092,
28072.114848582681589,
28080.375082874292275,
28088.635575730866549,
28096.896327085560188,
28105.157336871583539,
28113.418605022168776,
28121.680131470562628,
28129.941916150080942,
28138.203958994046843,
28146.466259935834387,
28154.728818908839457,
28162.991635846512509,
28171.254710682311270,
28179.518043349751679,
28187.781633782371500,
28196.045481913744879,
28204.309587677471427,
28212.573951007205324,
28220.838571836615301,
28229.103450099413749,
28237.368585729356710,
28245.633978660203866,
28253.899628825784021,
28262.165536159940530,
28270.431700596556766,
28278.698122069537931,
28286.964800512847432,
28295.231735860459594,
28303.498928046385117,
28311.766377004692913,
28320.034082669455529,
28328.302044974792807,
28336.570263854860968,
28344.838739243841701,
28353.107471075956710,
28361.376459285467718,
28369.645703806650999,
28377.915204573830124,
28386.184961521365040,
28394.454974583637522,
28402.725243695073004,
28410.995768790133297,
28419.266549803291127,
28427.537586669084703,
28435.808879322066787,
28444.080427696822881,
28452.352231727971230,
28460.624291350184649,
28468.896606498139590,
28477.169177106567076,
28485.442003110212681,
28493.715084443880187,
28501.988421042391565,
28510.262012840586976,
28518.535859773368429,
28526.809961775663396,
28535.084318782413902,
28543.358930728631094,
28551.633797549315204,
28559.908919179531949,
28568.184295554372511,
28576.459926608942624,
28584.735812278420781,
28593.011952497978200,
28601.288347202840669,
28609.564996328263078,
28617.841899809533061,
28626.119057581963716,
28634.396469580908160,
28642.674135741759528,
28650.952055999940058,
28659.230230290879263,
28667.508658550075779,
28675.787340713050071,
28684.066276715340791,
28692.345466492537525,
28700.624909980248049,
28708.904607114123792,
28717.184557829848927,
28725.464762063122180,
28733.745219749711396,
28742.025930825377145,
28750.306895225927292,
28758.588112887213356,
28766.869583745112323,
28775.151307735519367,
28783.433284794387873,
28791.715514857682138,
28799.997997861413751,
28808.280733741616132,
28816.563722434359079,
28824.846963875745132,
28833.130458001905936,
28841.414204749013152,
28849.698204053260270,
28857.982455850888073,
28866.266960078144621,
28874.551716671332542,
28882.836725566783571,
28891.121986700854904,
28899.407500009932846,
28907.693265430443716,
28915.979282898850215,
28924.265552351629594,
28932.552073725313676,
28940.838846956441557,
28949.125871981606906,
28957.413148737428855,
28965.700677160541090,
28973.988457187620952,
28982.276488755403989,
28990.564771800611197,
28998.853306260025420,
29007.142092070451326,
29015.431129168729967,
29023.720417491727858,
29032.009956976347894,
29040.299747559536627,
29048.589789178236970,
29056.880081769464596,
29065.170625270235178,
29073.461419617618958,
29081.752464748700731,
29090.043760600612586,
29098.335307110486610,
29106.627104215534928,
29114.919151852966024,
29123.211449960032041,
29131.503998474003311,
29139.796797332208371,
29148.089846471972123,
29156.383145830684953,
29164.676695345748158,
29172.970494954592141,
29181.264544594701874,
29189.558844203551416,
29197.853393718694861,
29206.148193077689939,
29214.443242218123487,
29222.738541077622358,
29231.034089593857061,
29239.329887704487192,
29247.625935347256018,
29255.922232459892257,
29264.218778980193747,
29272.515574845972878,
29280.812619995052955,
29289.109914365322766,
29297.407457894671097,
29305.705250521055859,
29314.003292182427685,
29322.301582816784503,
29330.600122362160619,
29338.898910756604892,
29347.197947938228026,
29355.497233845118899,
29363.796768415450060,
29372.096551587401336,
29380.396583299185295,
29388.696863489032694,
29396.997392095236137,
29405.298169056084589,
29413.599194309925224,
29421.900467795116128,
29430.201989450055407,
29438.503759213173907,
29446.805777022920665,
29455.108042817802925,
29463.410556536309741,
29471.713318117013841,
29480.016327498487954,
29488.319584619344823,
29496.623089418211748,
29504.926841833785147,
29513.230841804732336,
29521.535089269815217,
29529.839584167777502,
29538.144326437413838,
29546.449316017555248,
29554.754552847054583,
29563.060036864779249,
29571.365768009654857,
29579.671746220621571,
29587.977971436648659,
29596.284443596752681,
29604.591162639950198,
29612.898128505326895,
29621.205341131957539,
29629.512800458960555,
29637.820506425516214,
29646.128458970779320,
29654.436658033995627,
29662.745103554378147,
29671.053795471219928,
29679.362733723817655,
29687.671918251500756,
29695.981348993635038,
29704.291025889626326,
29712.600948878884083,
29720.911117900872341,
29729.221532895058772,
29737.532193800961977,
29745.843100558133301,
29754.154253106135002,
29762.465651384572993,
29770.777295333067741,
29779.089184891301556,
29787.401319998949475,
29795.713700595733826,
29804.026326621413318,
29812.339198015750299,
29820.652314718576235,
29828.965676669708046,
29837.279283809028129,
29845.593136076429801,
29853.907233411828201,
29862.221575755203958,
29870.536163046519505,
29878.850995225806400,
29887.166072233096202,
29895.481394008475036,
29903.796960492043581,
29912.112771623927983,
29920.428827344287129,
29928.745127593323559,
29937.061672311258008,
29945.378461438329396,
29953.695494914820301,
29962.012772681038768,
29970.330294677325583,
29978.648060844050633,
29986.966071121594723,
29995.284325450393226,
30003.602823770896066,
30011.921566023593186,
30020.240552148978168,
30028.559782087610074,
30036.879255780058884,
30045.198973166905489,
30053.518934188796266,
30061.839138786373951,
30070.159586900328577,
30078.480278471379279,
30086.801213440263382,
30095.122391747750953,
30103.443813334648439,
30111.765478141780477,
30120.087386110011721,
30128.409537180221378,
30136.731931293339585,
30145.054568390292843,
30153.377448412062222,
30161.700571299650619,
30170.023936994086398,
30178.347545436423388,
30186.671396567769989,
30194.995490329223685,
30203.319826661925617,
30211.644405507067859,
30219.969226805827930,
30228.294290499459748,
30236.619596529213595,
30244.945144836368854,
30253.270935362266755,
30261.596968048219424,
30269.923242835615383,
30278.249759665854071,
30286.576518480364030,
30294.903519220610178,
30303.230761828064715,
30311.558246244254406,
30319.885972410709655,
30328.213940269019076,
30336.542149760767643,
30344.870600827591261,
30353.199293411136750,
30361.528227453090949,
30369.857402895166160,
30378.186819679103792,
30386.516477746674354,
30394.846377039673825,
30403.176517499920010,
30411.506899069267092,
30419.837521689601999,
30428.168385302822571,
30436.499489850870305,
30444.830835275723075,
30453.162421519355121,
30461.494248523791612,
30469.826316231079545,
30478.158624583298661,
30486.491173522554163,
30494.823962990962173,
30503.156992930704291,
30511.490263283973036,
30519.823773992950009,
30528.157524999896850,
30536.491516247100662,
30544.825747676837636,
30553.160219231434894,
30561.494930853255937,
30569.829882484671543,
30578.165074068107060,
30586.500505545995111,
30594.836176860786509,
30603.172087954979361,
30611.508238771100878,
30619.844629251696460,
30628.181259339333337,
30636.518128976611479,
30644.855238106178149,
30653.192586670673336,
30661.530174612787960,
30669.868001875238406,
30678.206068400759250,
30686.544374132114172,
30694.882919012106868,
30703.221702983544674,
30711.560725989293132,
30719.899987972210511,
30728.239488875220559,
30736.579228641236114,
30744.919207213231857,
30753.259424534175196,
30761.599880547088105,
30769.940575195018027,
30778.281508421019680,
30786.622680168184161,
30794.964090379653499,
30803.305738998551533,
30811.647625968071225,
30819.989751231405535,
30828.332114731787442,
30836.674716412468115,
30845.017556216738740,
30853.360634087912331,
30861.703949969316454,
30870.047503804315056,
30878.391295536315738,
30886.735325108718826,
30895.079592464982852,
30903.424097548577265,
30911.768840302996978,
30920.113820671769645,
30928.459038598444749,
30936.804494026593602,
30945.150186899856635,
30953.496117161834263,
30961.842284756199660,
30970.188689626636915,
30978.535331716851942,
30986.882210970608867,
30995.229327331639070,
31003.576680743768520,
31011.924271150797722,
31020.272098496578110,
31028.620162724982947,
31036.968463779921876,
31045.317001605304540,
31053.665776145102427,
31062.014787343276112,
31070.364035143840738,
31078.713519490833278,
31087.063240328305255,
31095.413197600337298,
31103.763391251064604,
31112.113821224607818,
31120.464487465123966,
31128.815389916817367,
31137.166528523903253,
31145.517903230622323,
31153.869513981251657,
31162.221360720075609,
31170.573443391433102,
31178.925761939659424,
31187.278316309137153,
31195.631106444256147,
31203.984132289457193,
31212.337393789195630,
31220.690890887937712,
31229.044623530204262,
31237.398591660512466,
31245.752795223430439,
31254.107234163551766,
31262.461908425473666,
31270.816817953829741,
31279.171962693289970,
31287.527342588546162,
31295.882957584304677,
31304.238807625319168,
31312.594892656332377,
31320.951212622167077,
31329.307767467609665,
31337.664557137541124,
31346.021581576802419,
31354.378840730303637,
31362.736334542962140,
31371.094062959731673,
31379.452025925573253,
31387.810223385491554,
31396.168655284531269,
31404.527321567718900,
31412.886222180139157,
31421.245357066898578,
31429.604726173118252,
31437.964329443959286,
31446.324166824600979,
31454.684238260240818,
31463.044543696127221,
31471.405083077494055,
31479.765856349640671,
31488.126863457870058,
31496.488104347510671,
31504.849578963923705,
31513.211287252499460,
31521.573229158642789,
31529.935404627794924,
31538.297813605415286,
31546.660456036981486,
31555.023331868007517,
31563.386441044043750,
31571.749783510640555,
31580.113359213395597,
31588.477168097910180,
31596.841210109825624,
31605.205485194826906,
31613.569993298573536,
31621.934734366797784,
31630.299708345235558,
31638.664915179651871,
31647.030354815844476,
31655.396027199622040,
31663.761932276818698,
31672.128069993319514,
31680.494440294998640,
31688.861043127784797,
31697.227878437610343,
31705.594946170447656,
31713.962246272287302,
31722.329778689152590,
31730.697543367074104,
31739.065540252136998,
31747.433769290408236,
31755.802230428023904,
31764.170923611127364,
31772.539848785872891,
31780.909005898458418,
31789.278394895103702,
31797.648015722057607,
31806.017868325572636,
31814.387952651955857,
31822.758268647499790,
31831.128816258573352,
31839.499595431538182,
31847.870606112777750,
31856.241848248711904,
31864.613321785771404,
31872.985026670441584,
31881.356962849204137,
31889.729130268566223,
31898.101528875078657,
31906.474158615314082,
31914.847019435830589,
31923.220111283269944,
31931.593434104273911,
31939.966987845487893,
31948.340772453604586,
31956.714787875342154,
31965.089034057447861,
31973.463510946661700,
31981.838218489785504,
31990.213156633617473,
31998.588325325003098,
32006.963724510806060,
32015.339354137893679,
32023.715214153195120,
32032.091304503632273,
32040.467625136163406,
32048.844175997783168,
32057.220957035482570,
32065.597968196296279,
32073.975209427288064,
32082.352680675532611,
32090.730381888122793,
32099.108313012202416,
32107.486473994922562,
32115.864864783463418,
32124.243485325016081,
32132.622335566811671,
32141.001415456092218,
32149.380724940147047,
32157.760263966269122,
32166.140032481769595,
32174.520030434010550,
32182.900257770354074,
32191.280714438191353,
32199.661400384957233,
32208.042315558079281,
32216.423459905032360,
32224.804833373309521,
32233.186435910422006,
32241.568267463899247,
32249.950327981332521,
32258.332617410273087,
32266.715135698366794,
32275.097882793230383,
32283.480858642524254,
32291.864063193941547,
32300.247496395179041,
32308.631158193966257,
32317.015048538072733,
32325.399167375264369,
32333.783514653347083,
32342.168090320148622,
32350.552894323511282,
32358.937926611335570,
32367.323187131485611,
32375.708675831905566,
32384.094392660532321,
32392.480337565339141,
32400.866510494317481,
32409.252911395484261,
32417.639540216878231,
32426.026396906567243,
32434.413481412630063,
32442.800793683196389,
32451.188333666385006,
32459.576101310365630,
32467.964096563311614,
32476.352319373425416,
32484.740769688956789,
32493.129447458126378,
32501.518352629242145,
32509.907485150601133,
32518.296844970496750,
32526.686432037316990,
32535.076246299398917,
32543.466287705159630,
32551.856556202994398,
32560.247051741371251,
32568.637774268729117,
32577.028723733572406,
32585.419900084405526,
32593.811303269761993,
32602.202933238204423,
32610.594789938302711,
32618.986873318681319,
32627.379183327953797,
32635.771719914766436,
32644.164483027809183,
32652.557472615761071,
32660.950688627355703,
32669.344131011333957,
32677.737799716469453,
32686.131694691535813,
32694.525815885364864,
32702.920163246788434,
32711.314736724656541,
32719.709536267855583,
32728.104561825297424,
32736.499813345912116,
32744.895290778636991,
32753.290994072463945,
32761.686923176388518,
32770.083078039438988,
32778.479458610629081,
32786.876064839059836,
32795.272896673806827,
32803.669954063989280,
32812.067236958726426,
32820.464745307195699,
32828.862479058574536,
32837.260438162069477,
32845.658622566901613,
32854.057032222328417,
32862.455667077614635,
32870.854527082075947,
32879.253612185006205,
32887.652922335764742,
32896.052457483718172,
32904.452217578233103,
32912.852202568748908,
32921.252412404683128,
32929.652847035489685,
32938.053506410651607,
32946.454390479666472,
32954.855499192060961,
32963.256832497390860,
32971.658390345204680,
32980.060172685109137,
32988.462179466710950,
32996.864410639653215,
33005.266866153600859,
33013.669545958226081,
33022.072450003244739,
33030.475578238372691,
33038.878930613362172,
33047.282507077994524,
33055.686307582058362,
33064.090332075364131,
33072.494580507765932,
33080.899052829117863,
33089.303748989317683,
33097.708668938241317,
33106.113812625852006,
33114.519180002083885,
33122.924771016922023,
33131.330585620358761,
33139.736623762408271,
33148.142885393113829,
33156.549370462540537,
33164.956078920782602,
33173.363010717941506,
33181.770165804147837,
33190.177544129561284,
33198.585145644348813,
33206.992970298706496,
33215.401018042859505,
33223.809288827054843,
33232.217782601546787,
33240.626499316633272,
33249.035438922604953,
33257.444601369803422,
33265.853986608584819,
33274.263594589319837,
33282.673425262415549,
33291.083478578271752,
33299.493754487339174,
33307.904252940090373,
33316.314973886990629,
33324.725917278570705,
33333.137083065332263,
33341.548471197856998,
33349.960081626697502,
33358.371914302450023,
33366.783969175739912,
33375.196246197214350,
33383.608745317513240,
33392.021466487341968,
33400.434409657376818,
33408.847574778366834,
33417.260961801061057,
33425.674570676223084,
33434.088401354638336,
33442.502453787135892,
33450.916727924552106,
33459.331223717723333,
33467.745941117551411,
33476.160880074930901,
33484.576040540778195,
33492.991422466046060,
33501.407025801709096,
33509.822850498734624,
33518.238896508140897,
33526.655163780975272,
33535.071652268256003,
33543.488361921103206,
33551.905292690578790,
33560.322444527810148,
33568.739817383953778,
33577.157411210144346,
33585.575225957582006,
33593.993261577459634,
33602.411518021013762,
33610.829995239495474,
33619.248693184170406,
33627.667611806311470,
33636.086751057249785,
33644.506110888316471,
33652.925691250864475,
33661.345492096275848,
33669.765513375939918,
33678.185755041275115,
33686.606217043736251,
33695.026899334770860,
33703.447801865870133,
33711.868924588539812,
33720.290267454292916,
33728.711830414693395,
33737.133613421305199,
33745.555616425714106,
33753.977839379549550,
33762.400282234411861,
33770.822944941988681,
33779.245827453931270,
33787.668929721941822,
33796.092251697751635,
33804.515793333092006,
33812.939554579708783,
33821.363535389406024,
33829.787735713973234,
33838.212155505236296,
33846.636794715050200,
33855.061653295262659,
33863.486731197757763,
33871.912028374477813,
33880.337544777314179,
33888.763280358238262,
33897.189235069228744,
33905.615408862257027,
33914.041801689352724,
33922.468413502530893,
33930.895244253872079,
33939.322293895434996,
33947.749562379314739,
33956.177049657650059,
33964.604755682572431,
33973.032680406220607,
33981.460823780813371,
33989.889185758518579,
33998.317766291576845,
34006.746565332228784,
34015.175582832744112,
34023.604818745407101,
34032.034273022509296,
34040.463945616393175,
34048.893836479415768,
34057.323945563919551,
34065.754272822319763,
34074.184818207002536,
34082.615581670419488,
34091.046563165014959,
34099.477762643262395,
34107.909180057664344,
34116.340815360716078,
34124.772668504963804,
34133.204739442968275,
34141.637028127290250,
34150.069534510534140,
34158.502258545326185,
34166.935200184299902,
34175.368359380103357,
34183.801736085435550,
34192.235330252966378,
34200.669141835445771,
34209.103170785601833,
34217.537417056191771,
34225.971880600009172,
34234.406561369854899,
34242.841459318544366,
34251.276574398922094,
34259.711906563854427,
34268.147455766236817,
34276.583221958957438,
34285.019205094948120,
34293.455405127155245,
34301.891822008539748,
34310.328455692098942,
34318.765306130830140,
34327.202373277774313,
34335.639657085957879,
34344.077157508465461,
34352.514874498381687,
34360.952808008813008,
34369.390957992887706,
34377.829324403755891,
34386.267907194589498,
34394.706706318582292,
34403.145721728935314,
34411.584953378878708,
34420.024401221671724,
34428.464065210573608,
34436.903945298894541,
34445.344041439915600,
34453.784353586997895,
34462.224881693480711,
34470.665625712732435,
34479.106585598150559,
34487.547761303147126,
34495.989152781141456,
34504.430759985611076,
34512.872582869997132,
34521.314621387806255,
34529.756875492566905,
34538.199345137785713,
34546.642030277027516,
34555.084930863864429,
34563.528046851875843,
34571.971378194692079,
34580.414924845936184,
34588.858686759274860,
34597.302663888367533,
34605.746856186888181,
34614.191263608590816,
34622.635886107163969,
34631.080723636383482,
34639.525776150025195,
34647.971043601872225,
34656.416525945729518,
34664.862223135445674,
34673.308135124854743,
34681.754261867834430,
34690.200603318284266,
34698.647159430096508,
34707.093930157221621,
34715.540915453595517,
34723.988115273183212,
34732.435529569986102,
34740.883158298020135,
34749.331001411308534,
34757.779058863889077,
34766.227330609835917,
34774.675816603252315,
34783.124516798212426,
34791.573431148877717,
34800.022559609380551,
34808.471902133889671,
34816.921458676588372,
34825.371229191681778,
34833.821213633404113,
34842.271411956004158,
34850.721824113723414,
34859.172450060876145,
34867.623289751740231,
34876.074343140651763,
34884.525610181946831,
34892.977090829997906,
34901.428785039184731,
34909.880692763908883,
34918.332813958579209,
34926.785148577640939,
34935.237696575568407,
34943.690457906821393,
34952.143432525910612,
34960.596620387354051,
34969.050021445684251,
34977.503635655455582,
34985.957462971251516,
34994.411503347655525,
35002.865756739309290,
35011.320223100803560,
35019.774902386838221,
35028.229794552047679,
35036.684899551153649,
35045.140217338863295,
35053.595747869883780,
35062.051491098980478,
35070.507446980926034,
35078.963615470507648,
35087.419996522527072,
35095.876590091815160,
35104.333396133217320,
35112.790414601586235,
35121.247645451832796,
35129.705088638846064,
35138.162744117544207,
35146.620611842874496,
35155.078691769798752,
35163.536983853293350,
35171.995488048363768,
35180.454204310022760,
35188.913132593312184,
35197.372272853266622,
35205.831625045000692,
35214.291189123578079,
35222.750965044120676,
35231.210952761779481,
35239.671152231669112,
35248.131563408984221,
35256.592186248919461,
35265.053020706662210,
35273.514066737450776,
35281.975324296538020,
35290.436793339184078,
35298.898473820663639,
35307.360365696302324,
35315.822468921396649,
35324.284783451301337,
35332.747309241378389,
35341.210046246997081,
35349.672994423563068,
35358.136153726489283,
35366.599524111203209,
35375.063105533175985,
35383.526897947864200,
35391.990901310768095,
35400.455115577387915,
35408.919540703267558,
35417.384176643950923,
35425.849023354989185,
35434.314080791991728,
35442.779348910531553,
35451.244827666261699,
35459.710517014813377,
35468.176416911839624,
35476.642527313029859,
35485.108848174073501,
35493.575379450689070,
35502.042121098602365,
35510.509073073590116,
35518.976235331399948,
35527.443607827830419,
35535.911190518694639,
35544.378983359812992,
35552.846986307042243,
35561.315199316239159,
35569.783622343282332,
35578.252255344079458,
35586.721098274552787,
35595.190151090646395,
35603.659413748304360,
35612.128886203507136,
35620.598568412249733,
35629.068460330556263,
35637.538561914443562,
35646.008873119950294,
35654.479393903173332,
35662.950124220173166,
35671.421064027075772,
35679.892213279985299,
35688.363571935056825,
35696.835139948452706,
35705.306917276335298,
35713.778903874917887,
35722.251099700391933,
35730.723504709021654,
35739.196118857049441,
35747.668942100724962,
35756.141974396348814,
35764.615215700243425,
35773.088665968709392,
35781.562325158105523,
35790.036193224783347,
35798.510270125123498,
35806.984555815521162,
35815.459050252400630,
35823.933753392208018,
35832.408665191360342,
35840.883785606361926,
35849.359114593680715,
35857.834652109821036,
35866.310398111330869,
35874.786352554729092,
35883.262515396585513,
35891.738886593484494,
35900.215466102010396,
35908.692253878798510,
35917.169249880455027,
35925.646454063651618,
35934.123866385059955,
35942.601486801358988,
35951.079315269256767,
35959.557351745468623,
35968.035596186746261,
35976.514048549841391,
35984.992708791549376,
35993.471576868636475,
36001.950652737941709,
36010.429936356289545,
36018.909427680519002,
36027.389126667505479,
36035.869033274138928,
36044.349147457309300,
36052.829469173942925,
36061.309998380987963,
36069.790735035399848,
36078.271679094141291,
36086.752830514211382,
36095.234189252623764,
36103.715755266392080,
36112.197528512573626,
36120.679508948240255,
36129.161696530456538,
36137.644091216330708,
36146.126692962970992,
36154.609501727529278,
36163.092517467150174,
36171.575740138985566,
36180.059169700245548,
36188.542806108140212,
36197.026649319879652,
36205.510699292695790,
36213.994955983871478,
36222.479419350660464,
36230.964089350374707,
36239.448965940311609,
36247.934049077804957,
36256.419338720210362,
36264.904834824883437,
36273.390537349208898,
36281.876446250593290,
36290.362561486428604,
36298.848883014186868,
36307.335410791289178,
36315.822144775214838,
36324.309084923464980,
36332.796231193518906,
36341.283583542935958,
36349.771141929217265,
36358.258906309936719,
36366.746876642682764,
36375.235052885022014,
36383.723434994593845,
36392.212022929001250,
36400.700816645890882,
36409.189816102945770,
36417.679021257812565,
36426.168432068217953,
36434.658048491859518,
36443.147870486478496,
36451.637898009830678,
36460.128131019657303,
36468.618569473765092,
36477.109213329946215,
36485.600062546021945,
36494.091117079835385,
36502.582376889222360,
36511.073841932069627,
36519.565512166256667,
36528.057387549684790,
36536.549468040284410,
36545.041753596000490,
36553.534244174777996,
36562.026939734590997,
36570.519840233435389,
36579.012945629321621,
36587.506255880274693,
36595.999770944326883,
36604.493490779561398,
36612.987415344046894,
36621.481544595859305,
36629.975878493132768,
36638.470416993979597,
36646.965160056555760,
36655.460107639031776,
36663.955259699570888,
36672.450616196372721,
36680.946177087673277,
36689.441942331664904,
36697.937911886634538,
36706.434085710825457,
36714.930463762531872,
36723.427046000040718,
36731.923832381682587,
36740.420822865773516,
36748.918017410695029,
36757.415415974770440,
36765.913018516424927,
36774.410824994040013,
36782.908835366048152,
36791.407049590859970,
36799.905467626944301,
36808.404089432784531,
36816.902914966842218,
36825.401944187629852,
36833.901177053667197,
36842.400613523488573,
36850.900253555664676,
36859.400097108751652,
36867.900144141327473,
36876.400394612013770,
36884.900848479417618,
36893.401505702197028,
36901.902366238995455,
36910.403430048478185,
36918.904697089354158,
36927.406167320295935,
36935.907840700048837,
36944.409717187358183,
36952.911796740969294,
36961.414079319649318,
36969.916564882194507,
36978.419253387401113,
36986.922144794116321,
36995.425239061150933,
37003.928536147373961,
37012.432036011661694,
37020.935738612883142,
37029.439643909972801,
37037.943751861843339,
37046.448062427436525,
37054.952575565686857,
37063.457291235587036,
37071.962209396137041,
37080.467330006322300,
37088.972653025164618,
37097.478178411707631,
37105.983906125016802,
37114.489836124157591,
37122.995968368210015,
37131.502302816290467,
37140.008839427515341,
37148.515578161037411,
37157.022518975980347,
37165.529661831547855,
37174.037006686907262,
37182.544553501276823,
37191.052302233860246,
37199.560252843904891,
37208.068405290672672,
37216.576759533425502,
37225.085315531447122,
37233.594073244035826,
37242.103032630533562,
37250.612193650245899,
37259.121556262543891,
37267.631120426791313,
37276.140886102388322,
37284.650853248713247,
37293.161021825188072,
37301.671391791263886,
37310.181963106369949,
37318.692735729986452,
37327.203709621586313,
37335.714884740671550,
37344.226261046773288,
37352.737838499400823,
37361.249617058114382,
37369.761596682474192,
37378.273777332062309,
37386.786158966489893,
37395.298741545346275,
37403.811525028278993,
37412.324509374913760,
37420.837694544941769,
37429.351080498003284,
37437.864667193833156,
37446.378454592115304,
37454.892442652577301,
37463.406631334975827,
37471.921020599060284,
37480.435610404609179,
37488.950400711408292,
37497.465391479272512,
37505.980582668009447,
37514.495974237477640,
37523.011566147535632,
37531.527358358027413,
37540.043350828869734,
37548.559543519950239,
37557.075936391200230,
37565.592529402551008,
37574.109322513933876,
37582.626315685352893,
37591.143508876768465,
37599.660902048184653,
37608.178495159627346,
37616.696288171115157,
37625.214281042703078,
37633.732473734446103,
37642.250866206442879,
37650.769458418762952,
37659.288250331548625,
37667.807241904905823,
37676.326433098984126,
37684.845823873947666,
37693.365414189967851,
37701.885204007223365,
37710.405193285943824,
37718.925381986344291,
37727.445770068647107,
37735.966357493132818,
37744.487144220067421,
37753.008130209724186,
37761.529315422405489,
37770.050699818442808,
37778.572283358153072,
37787.094066001911415,
37795.616047710056591,
37804.138228442985564,
37812.660608161088021,
37821.183186824782752,
37829.705964394488547,
37838.228940830660576,
37846.752116093754012,
37855.275490144245850,
37863.799062942620367,
37872.322834449405491,
37880.846804625092773,
37889.370973430246522,
37897.895340825409221,
37906.419906771159731,
37914.944671228069637,
37923.469634156754182,
37931.994795517821331,
37940.520155271908152,
37949.045713379651716,
37957.571469801725470,
37966.097424498810142,
37974.623577431601007,
37983.149928560800618,
37991.676477847140632,
38000.203225251359981,
38008.730170734219428,
38017.257314256494283,
38025.784655778952583,
38034.312195262427849,
38042.839932667724497,
38051.367867955668771,
38059.896001087123295,
38068.424332022950693,
38076.952860724028142,
38085.481587151261920,
38094.010511265543755,
38102.539633027830860,
38111.068952399044065,
38119.598469340147858,
38128.128183812121279,
38136.658095775943366,
38145.188205192629539,
38153.718512023187941,
38162.249016228663095,
38170.779717770114075,
38179.310616608578130,
38187.841712705157988,
38196.373006020956382,
38204.904496517083317,
38213.436184154648799,
38221.968068894799217,
38230.500150698710058,
38239.032429527549539,
38247.564905342500424,
38256.097578104760032,
38264.630447775554785,
38273.163514316125656,
38281.696777687728172,
38290.230237851603306,
38298.763894769050239,
38307.297748401360877,
38315.831798709848954,
38324.366045655828202,
38332.900489200648735,
38341.435129305675218,
38349.969965932265040,
38358.504999041811971,
38367.040228595709777,
38375.575654555381334,
38384.111276882271341,
38392.647095537802670,
38401.183110483463679,
38409.719321680713620,
38418.255729091055400,
38426.792332675991929,
38435.329132397055218,
38443.866128215762728,
38452.403320093682851,
38460.940707992383977,
38469.478291873449052,
38478.016071698475571,
38486.554047429075581,
38495.092219026875682,
38503.630586453517026,
38512.169149670669867,
38520.707908640004462,
38529.246863323205616,
38537.786013681972690,
38546.325359678026871,
38554.864901273111172,
38563.404638428961334,
38571.944571107342199,
38580.484699270047713,
38589.025022878849995,
38597.565541895572096,
38606.106256282022514,
38614.647166000053403,
38623.188271011524193,
38631.729571278279764,
38640.271066762223199,
38648.812757425243035,
38657.354643229249632,
38665.896724136167904,
38674.439000107944594,
38682.981471106555546,
38691.524137093940226,
38700.066998032103584,
38708.610053883050568,
38717.153304608778853,
38725.696750171344320,
38734.240390532773745,
38742.784225655137561,
38751.328255500506202,
38759.872480030971928,
38768.416899208648829,
38776.961512995636440,
38785.506321354092506,
38794.051324246138392,
38802.596521633960947,
38811.141913479739742,
38819.687499745654350,
38828.233280393927998,
38836.779255386776640,
38845.325424686423503,
38853.871788255135471,
38862.418346055186703,
38870.965098048844084,
38879.512044198410877,
38888.059184466197621,
38896.606518814514857,
38905.154047205731331,
38913.701769602186687,
38922.249685966242396,
38930.797796260296309,
38939.346100446739001,
38947.894598487975600,
38956.443290346454887,
38964.992175984603819,
38973.541255364885728,
38982.090528449756675,
38990.639995201730926,
38999.189655583279091,
39007.739509556929988,
39016.289557085219712,
39024.839798130684358,
39033.390232655881846,
39041.940860623384651,
39050.491681995779800,
39059.042696735668869,
39067.593904805675265,
39076.145306168429670,
39084.696900786555489,
39093.248688622741611,
39101.800669639633270,
39110.352843799948459,
39118.905211066376069,
39127.457771401626815,
39136.010524768440519,
39144.563471129571553,
39153.116610447752464,
39161.669942685781280,
39170.223467806448753,
39178.777185772552912,
39187.331096546899062,
39195.885200092336163,
39204.439496371705900,
39212.993985347864509,
39221.548666983690055,
39230.103541242075153,
39238.658608085919695,
39247.213867478138127,
39255.769319381673995,
39264.324963759470847,
39272.880800574479508,
39281.436829789672629,
39289.993051368059241,
39298.549465272633824,
39307.106071466412686,
39315.662869912404858,
39324.219860573699407,
39332.777043413334468,
39341.334418394377280,
39349.891985479931463,
39358.449744633107912,
39367.007695816995692,
39375.565838994734804,
39384.124174129479798,
39392.682701184392499,
39401.241420122634736,
39409.800330907404714,
39418.359433501900639,
39426.918727869327995,
39435.478213972935919,
39444.037891775951721,
39452.597761241660919,
39461.157822333298100,
39469.718075014170608,
39478.278519247585791,
39486.839154996843718,
39495.399982225280837,
39503.961000896240876,
39512.522210973067558,
39521.083612419155543,
39529.645205197863106,
39538.206989272606734,
39546.768964606802911,
39555.331131163860846,
39563.893488907226129,
39572.456037800366175,
39581.018777806741127,
39589.581708889832953,
39598.144831013130897,
39606.708144140160584,
39615.271648234447639,
39623.835343259510410,
39632.399229178910900,
39640.963305956232944,
39649.527573555038543,
39658.092031938918808,
39666.656681071486673,
39675.221520916369627,
39683.786551437209710,
39692.351772597627132,
39700.917184361314867,
39709.482786691936781,
39718.048579553193122,
39726.614562908776861,
39735.180736722424626,
39743.747100957843941,
39752.313655578800535,
39760.880400549045589,
39769.447335832352110,
39778.014461392522207,
39786.581777193336166,
39795.149283198617923,
39803.716979372213245,
39812.284865677938797,
39820.852942079683999,
39829.421208541280066,
39837.989665026630973,
39846.558311499633419,
39855.127147924198653,
39863.696174264252477,
39872.265390483742522,
39880.834796546609141,
39889.404392416807241,
39897.974178058342659,
39906.544153435206681,
39915.114318511390593,
39923.684673250922060,
39932.255217617828748,
39940.825951576174702,
39949.396875090016692,
39957.967988123418763,
39966.539290640481340,
39975.110782605297572,
39983.682463982011541,
39992.254334734709118,
40000.826394827570766,
40009.398644224726013,
40017.971082890377147,
40026.543710788682802,
40035.116527883845265,
40043.689534140088654,
40052.262729521607980,
40060.836113992678293,
40069.409687517530983,
40077.983450060441101,
40086.557401585669140,
40095.131542057533807,
40103.705871440310148,
40112.280389698353247,
40120.855096795967256,
40129.429992697507259,
40138.005077367342892,
40146.580350769836514,
40155.155812869372312,
40163.731463630363578,
40172.307303017209051,
40180.883330994351127,
40189.459547526217648,
40198.035952577258286,
40206.612546111951815,
40215.189328094791563,
40223.766298490234476,
40232.343457262824813,
40240.920804377055902,
40249.498339797486551,
40258.076063488639193,
40266.653975415094465,
40275.232075541418453,
40283.810363832184521,
40292.388840252031514,
40300.967504765525518,
40309.546357337327208,
40318.125397932068154,
40326.704626514401753,
40335.284043049003230,
40343.863647500540537,
40352.443439833710727,
40361.023420013225405,
40369.603588003810728,
40378.183943770185579,
40386.764487277097942,
40395.345218489317631,
40403.926137371614459,
40412.507243888772791,
40421.088538005598821,
40429.670019686891465,
40438.251688897493295,
40446.833545602232334,
40455.415589765972982,
40463.997821353565087,
40472.580240329902153,
40481.162846659863135,
40489.745640308370639,
40498.328621240318171,
40506.911789420664718,
40515.495144814325613,
40524.078687386288948,
40532.662417101499159,
40541.246333924958890,
40549.830437821663509,
40558.414728756608383,
40566.999206694832537,
40575.583871601374994,
40584.168723441260227,
40592.753762179570913,
40601.338987781375181,
40609.924400211762986,
40618.509999435846112,
40627.095785418729065,
40635.681758125538181,
40644.267917521421623,
40652.854263571527554,
40661.440796241025964,
40670.027515495086845,
40678.614421298923844,
40687.201513617706951,
40695.788792416693468,
40704.376257661089767,
40712.963909316153149,
40721.551747347148194,
40730.139771719317650,
40738.727982397977030,
40747.316379348398186,
40755.904962535896630,
40764.493731925802422,
40773.082687483445625,
40781.671829174178129,
40790.261156963351823,
40798.850670816347701,
40807.440370698546758,
40816.030256575366366,
40824.620328412194795,
40833.210586174463970,
40841.801029827620368,
40850.391659337110468,
40858.982474668395298,
40867.573475786957715,
40876.164662658280577,
40884.756035247875843,
40893.347593521240924,
40901.939337443924160,
40910.531266981459339,
40919.123382099387527,
40927.715682763286168,
40936.308168938739982,
40944.900840591326414,
40953.493697686666565,
40962.086740190366982,
40970.679968068056041,
40979.273381285391224,
40987.866979808008182,
40996.460763601586223,
41005.054732631811930,
41013.648886864364613,
41022.243226264945406,
41030.837750799299101,
41039.432460433148663,
41048.027355132224329,
41056.622434862292721,
41065.217699589135009,
41073.813149278517812,
41082.408783896244131,
41091.004603408109688,
41099.600607779953862,
41108.196796977601480,
41116.793170966891921,
41125.389729713679117,
41133.986473183853377,
41142.583401343290461,
41151.180514157880680,
41159.777811593543447,
41168.375293616183626,
41176.972960191749735,
41185.570811286190292,
41194.168846865453816,
41202.767066895503376,
41211.365471342352976,
41219.964060171958408,
41228.562833350377332,
41237.161790843594645,
41245.760932617660728,
41254.360258638618689,
41262.959768872518907,
41271.559463285448146,
41280.159341843485890,
41288.759404512718902,
41297.359651259270322,
41305.960082049256016,
41314.560696848806401,
41323.161495624088275,
41331.762478341224778,
41340.363644966419088,
41348.964995465845277,
41357.566529805691971,
41366.168247952176898,
41374.770149871525064,
41383.372235529954196,
41391.974504893732956,
41400.576957929100899,
41409.179594602326688,
41417.782414879722637,
41426.385418727542856,
41434.988606112128764,
41443.591976999792678,
41452.195531356846914,
41460.799269149661995,
41469.403190344586619,
41478.007294907998585,
41486.611582806261140,
41495.216054005781189,
41503.820708472980186,
41512.425546174243209,
41521.030567076020816,
41529.635771144756291,
41538.241158346914744,
41546.846728648946737,
41555.452482017339207,
41564.058418418600922,
41572.664537819204270,
41581.270840185687121,
41589.877325484587345,
41598.483993682428263,
41607.090844745776849,
41615.697878641200077,
41624.305095335264923,
41632.912494794560189,
41641.520076985696505,
41650.127841875299055,
41658.735789429993019,
41667.343919616403582,
41675.952232401177753,
41684.560727751006198,
41693.169405632543203,
41701.778266012486711,
41710.387308857527387,
41718.996534134392277,
41727.605941809801152,
41736.215531850481057,
41744.825304223195417,
41753.435258894685830,
41762.045395831752103,
41770.655715001150384,
41779.266216369695030,
41787.876899904207676,
41796.487765571480850,
41805.098813338365289,
41813.710043171719008,
41822.321455038370914,
41830.933048905215401,
41839.544824739110481,
41848.156782506972377,
41856.768922175688203,
41865.381243712203286,
41873.993747083426570,
41882.606432256296102,
41891.219299197779037,
41899.832347874842526,
41908.445578254453721,
41917.058990303608880,
41925.672583989311534,
41934.286359278587042,
41942.900316138424387,
41951.514454535899858,
41960.128774438053370,
41968.743275811932108,
41977.357958624626917,
41985.972822843214090,
41994.587868434806296,
42003.203095366479829,
42011.818503605398291,
42020.434093118674355,
42029.049863873442519,
42037.665815836888214,
42046.281948976160493,
42054.898263258444786,
42063.514758650933800,
42072.131435120834794,
42080.748292635369580,
42089.365331161752692,
42097.982550667235046,
42106.599951119067555,
42115.217532484508411,
42123.835294730852183,
42132.453237825364340,
42141.071361735368555,
42149.689666428137571,
42158.308151871038717,
42166.926818031388393,
42175.545664876539377,
42184.164692373837170,
42192.783900490656379,
42201.403289194386161,
42210.022858452422952,
42218.642608232163184,
42227.262538501032395,
42235.882649226448848,
42244.502940375867183,
42253.123411916742043,
42261.744063816520793,
42270.364896042701730,
42278.985908562754048,
42287.607101344197872,
42296.228474354509672,
42304.850027561260504,
42313.471760931948666,
42322.093674434130662,
42330.715768035377550,
42339.338041703238559,
42347.960495405313850,
42356.583129109189031,
42365.205942782478814,
42373.828936392776086,
42382.452109907717386,
42391.075463294968358,
42399.698996522165544,
42408.322709556952759,
42416.946602367024752,
42425.570674920068996,
42434.194927183780237,
42442.819359125860501,
42451.443970714055467,
42460.068761916059884,
42468.693732699648535,
42477.318883032574377,
42485.944212882597640,
42494.569722217500384,
42503.195411005064670,
42511.821279213108937,
42520.447326809437072,
42529.073553761874791,
42537.699960038262361,
42546.326545606454602,
42554.953310434291780,
42563.580254489665094,
42572.207377740458469,
42580.834680154555826,
42589.462161699870194,
42598.089822344314598,
42606.717662055809342,
42615.345680802318384,
42623.973878551783855,
42632.602255272162438,
42641.230810931425367,
42649.859545497572981,
42658.488458938598342,
42667.117551222516340,
42675.746822317334590,
42684.376272191097087,
42693.005900811840547,
42701.635708147630794,
42710.265694166519097,
42718.895858836593106,
42727.526202125955024,
42736.156724002663395,
42744.787424434871355,
42753.418303390695655,
42762.049360838267603,
42770.680596745718503,
42779.312011081230594,
42787.943603812949732,
42796.575374909079983,
42805.207324337789032,
42813.839452067295497,
42822.471758065817994,
42831.104242301560589,
42839.736904742792831,
42848.369745357733336,
42857.002764114651654,
42865.635960981824610,
42874.269335927536304,
42882.902888920063560,
42891.536619927726861,
42900.170528918839409,
42908.804615861728962,
42917.438880724723276,
42926.073323476201040,
42934.707944084490009,
42943.342742517976149,
42951.977718745052698,
42960.612872734098346,
42969.248204453535436,
42977.883713871764485,
42986.519400957229664,
42995.155265678360593,
43003.791308003615995,
43012.427527901454596,
43021.063925340349670,
43029.700500288781768,
43038.337252715267823,
43046.974182588281110,
43055.611289876367664,
43064.248574548037141,
43072.886036571842851,
43081.523675916338107,
43090.161492550083494,
43098.799486441639601,
43107.437657559603394,
43116.076005872579117,
43124.714531349156459,
43133.353233957976045,
43141.992113667649392,
43150.631170446824399,
43159.270404264148965,
43167.909815088285541,
43176.549402887918404,
43185.189167631717282,
43193.829109288395557,
43202.469227826659335,
43211.109523215200170,
43219.749995422782376,
43228.390644418126612,
43237.031470169975364,
43245.672472647122049,
43254.313651818316430,
43262.955007652366476,
43271.596540118029225,
43280.238249184141750,
43288.880134819519299,
43297.522196992977115,
43306.164435673366825,
43314.806850829540053,
43323.449442430348427,
43332.092210444679949,
43340.735154841393523,
43349.378275589413533,
43358.021572657620709,
43366.665046014946711,
43375.308695630315924,
43383.952521472667286,
43392.596523510947009,
43401.240701714123134,
43409.885056051149149,
43418.529586491036753,
43427.174293002746708,
43435.819175555305264,
43444.464234117716842,
43453.109468659014965,
43461.754879148225882,
43470.400465554412222,
43479.046227846622060,
43487.692165993932576,
43496.338279965420952,
43504.984569730171643,
43513.631035257283656,
43522.277676515885105,
43530.924493475104100,
43539.571486104054202,
43548.218654371892626,
43556.865998247769312,
43565.513517700863304,
43574.161212700346368,
43582.809083215404826,
43591.457129215232271,
43600.105350669058680,
43608.753747546099476,
43617.402319815555529,
43626.051067446715024,
43634.699990408807935,
43643.349088671115169,
43651.998362202881253,
43660.647810973416199,
43669.297434952022741,
43677.947234107981785,
43686.597208410632447,
43695.247357829292014,
43703.897682333314151,
43712.548181892037974,
43721.198856474831700,
43729.849706051056273,
43738.500730590109015,
43747.151930061372695,
43755.803304434251913,
43764.454853678165819,
43773.106577762540837,
43781.758476656817948,
43790.410550330416299,
43799.062798752842355,
43807.715221893507987,
43816.367819721934211,
43825.020592207583832,
43833.673539319985139,
43842.326661028622766,
43850.979957303032279,
43859.633428112741967,
43868.287073427294672,
43876.940893216240511,
43885.594887449151429,
43894.249056095592096,
43902.903399125149008,
43911.557916507430491,
43920.212608212044870,
43928.867474208578642,
43937.522514466691064,
43946.177728955997736,
43954.833117646165192,
43963.488680506852688,
43972.144417507712205,
43980.800328618432104,
43989.456413808729849,
43998.112673048264696,
44006.769106306768663,
44015.425713553966489,
44024.082494759590190,
44032.739449893386336,
44041.396578925094218,
44050.053881824496784,
44058.711358561362431,
44067.369009105474106,
44076.026833426629310,
44084.684831494640093,
44093.343003279303957,
44102.001348750483885,
44110.659867877991928,
44119.318560631683795,
44127.977426981415192,
44136.636466897063656,
44145.295680348492169,
44153.955067305614648,
44162.614627738315903,
44171.274361616517126,
44179.934268910132232,
44188.594349589096964,
44197.254603623354342,
44205.915030982861936,
44214.575631637570041,
44223.236405557458056,
44231.897352712527208,
44240.558473072742345,
44249.219766608133796,
44257.881233288702788,
44266.542873084486928,
44275.204685965516546,
44283.866671901821974,
44292.528830863499024,
44301.191162820578029,
44309.853667743162077,
44318.516345601310604,
44327.179196365155804,
44335.842220004778937,
44344.505416490319476,
44353.168785791887785,
44361.832327879645163,
44370.496042723723804,
44379.159930294285004,
44387.823990561511891,
44396.488223495580314,
44405.152629066673398,
44413.817207245003374,
44422.481958000767918,
44431.146881304222916,
44439.811977125558769,
44448.477245435038640,
44457.142686202918412,
44465.808299399446696,
44474.474084994915756,
44483.140042959596030,
44491.806173263787059,
44500.472475877795659,
44509.138950771921373,
44517.805597916507395,
44526.472417281882372,
44535.139408838389500,
44543.806572556379251,
44552.473908406231203,
44561.141416358303104,
44569.809096382989082,
44578.476948450690543,
44587.144972531816165,
44595.813168596760079,
44604.481536615981895,
44613.150076559890294,
44621.818788398944889,
44630.487672103605291,
44639.156727644331113,
44647.825954991603794,
44656.495354115904775,
44665.164924987737322,
44673.834667577626533,
44682.504581856046570,
44691.174667793580738,
44699.844925360710477,
44708.515354528026364,
44717.185955266075325,
44725.856727545411559,
44734.527671336647472,
44743.198786610329989,
44751.870073337078793,
44760.541531487506290,
44769.213161032232165,
44777.884961941876099,
44786.556934187072329,
44795.229077738491469,
44803.901392566767754,
44812.573878642600903,
44821.246535936639702,
44829.919364419583871,
44838.592364062133129,
44847.265534835016297,
44855.938876708918542,
44864.612389654583239,
44873.286073642761039,
44881.959928644195315,
44890.633954629636719,
44899.308151569857728,
44907.982519435652648,
44916.657058197779406,
44925.331767827083240,
44934.006648294336628,
44942.681699570370256,
44951.356921626007534,
44960.032314432093699,
44968.707877959473990,
44977.383612179030024,
44986.059517061592487,
44994.735592578072101,
45003.411838699350483,
45012.088255396316526,
45020.764842639880953,
45029.441600400969037,
45038.118528650498774,
45046.795627359424543,
45055.472896498686168,
45064.150336039245303,
45072.827945952070877,
45081.505726208139095,
45090.183676778433437,
45098.861797633951937,
45107.540088745714456,
45116.218550084726303,
45124.897181622021890,
45133.575983328635630,
45142.254955175609211,
45150.934097134006151,
45159.613409174904518,
45168.292891269367828,
45176.972543388474151,
45185.652365503345209,
45194.332357585059071,
45203.012519604752015,
45211.692851533545763,
45220.373353342576593,
45229.054025002988055,
45237.734866485930979,
45246.415877762585296,
45255.097058804101835,
45263.778409581682354,
45272.459930066521338,
45281.141620229820546,
45289.823480042781739,
45298.505509476657608,
45307.187708502649912,
45315.870077092033171,
45324.552615216038248,
45333.235322845925111,
45341.918199952990108,
45350.601246508493205,
45359.284462483730749,
45367.967847850006365,
45376.651402578645502,
45385.335126640944509,
45394.019020008257939,
45402.703082651903969,
45411.387314543251705,
45420.071715653648425,
45428.756285954477789,
45437.441025417094352,
45446.125934012925427,
45454.811011713340122,
45463.496258489743923,
45472.181674313578696,
45480.867259156257205,
45489.553012989221315,
45498.238935783905617,
45506.925027511781082,
45515.611288144318678,
45524.297717652982101,
45532.984316009264148,
45541.671083184643066,
45550.358019150648033,
45559.045123878779123,
45567.732397340572788,
45576.419839507543657,
45585.107450351250009,
45593.795229843250127,
45602.483177955080464,
45611.171294658342958,
45619.859579924610443,
45628.548033725448477,
45637.236656032488099,
45645.925446817345801,
45654.614406051608967,
45663.303533706937742,
45671.992829754948616,
45680.682294167301734,
45689.371926915664517,
45698.061727971689834,
45706.751697307052382,
45715.441834893455962,
45724.132140702575271,
45732.822614706135937,
45741.513256875841762,
45750.204067183418374,
45758.895045600605954,
45767.586192099151958,
45776.277506650803844,
45784.968989227309066,
45793.660639800466015,
45802.352458342036698,
45811.044444823826780,
45819.736599217620096,
45828.428921495265968,
45837.121411628533679,
45845.814069589279825,
45854.506895349331899,
45863.199888880561048,
45871.893050154794764,
45880.586379143926024,
45889.279875819804147,
45897.973540154329385,
45906.667372119409265,
45915.361371686929488,
45924.055538828819408,
45932.749873516986554,
45941.444375723360281,
45950.139045419913600,
45958.833882578575867,
45967.528887171312817,
45976.224059170082910,
45984.919398546880984,
45993.614905273694603,
46002.310579322518606,
46011.006420665362384,
46019.702429274249880,
46028.398605121183209,
46037.094948178229970,
46045.791458417421381,
46054.488135810810491,
46063.184980330472172,
46071.881991948459472,
46080.579170636876370,
46089.276516367805016,
46097.974029113342112,
46106.671708845598914,
46115.369555536715779,
46124.067569158789411,
46132.765749683996546,
46141.464097084455716,
46150.162611332321831,
46158.861292399786180,
46167.560140259018226,
46176.259154882172879,
46184.958336241492361,
46193.657684309138858,
46202.357199057354592,
46211.056880458338128,
46219.756728484338964,
46228.456743107599323,
46237.156924300346873,
46245.857272034852940,
46254.557786283403402,
46263.258467018262309,
46271.959314211700985,
46280.660327836027136,
46289.361507863563020,
46298.062854266601789,
46306.764367017472978,
46315.466046088513394,
46324.167891452067124,
46332.869903080470976,
46341.572080946098140,
46350.274425021321804,
46358.976935278507881,
46367.679611690065940,
46376.382454228369170,
46385.085462865827139,
46393.788637574863969,
46402.491978327903780,
46411.195485097392520,
46419.899157855747035,
46428.602996575442376,
46437.307001228924491,
46446.011171788675711,
46454.715508227156533,
46463.420010516885668,
46472.124678630338167,
46480.829512540025462,
46489.534512218458985,
46498.239677638179273,
46506.945008771712310,
46515.650505591598630,
46524.356168070400599,
46533.061996180666029,
46541.767989894971834,
46550.474149185902206,
46559.180474026034062,
46567.886964387980697,
46576.593620244340855,
46585.300441567727830,
46594.007428330791299,
46602.714580506122729,
46611.421898066408176,
46620.129380984253658,
46628.837029232374334,
46637.544842783390777,
46646.252821610010869,
46654.960965684920666,
46663.669274980820774,
46672.377749470397248,
46681.086389126379800,
46689.795193921498139,
46698.504163828481978,
46707.213298820068303,
46715.922598869015928,
46724.632063948083669,
46733.341694030030339,
46742.051489087636583,
46750.761449093712145,
46759.471574021044944,
46768.181863842415623,
46776.892318530663033,
46785.602938058611471,
46794.313722399077960,
46803.024671524915902,
46811.735785408964148,
46820.447064024090650,
46829.158507343170641,
46837.870115339057520,
46846.581887984670175,
46855.293825252883835,
46864.005927116602834,
46872.718193548731506,
46881.430624522203289,
46890.143220009958895,
46898.855979984924488,
46907.568904420040781,
46916.281993288277590,
46924.995246562597458,
46933.708664215984754,
46942.422246221387468,
46951.135992551848176,
46959.849903180351248,
46968.563978079895605,
46977.278217223494721,
46985.992620584205724,
46994.707188135034812,
47003.421919849053666,
47012.136815699297586,
47020.851875658852805,
47029.567099700761901,
47038.282487798132934,
47046.998039924037585,
47055.713756051583914,
47064.429636153879983,
47073.145680204048404,
47081.861888175197237,
47090.578260040470923,
47099.294795773021178,
47108.011495345985168,
47116.728358732529159,
47125.445385905819421,
47134.162576839036774,
47142.879931505383865,
47151.597449878026964,
47160.315131930205098,
47169.032977635099087,
47177.750986965933407,
47186.469159895976190,
47195.187496398422809,
47203.905996446541394,
47212.624660013592802,
47221.343487072830612,
47230.062477597537509,
47238.781631560996175,
47247.500948936518398,
47256.220429697357758,
47264.940073816876975,
47273.659881268344179,
47282.379852025122091,
47291.099986060529773,
47299.820283347929944,
47308.540743860648945,
47317.261367572071322,
47325.982154455567070,
47334.703104484498908,
47343.424217632251384,
47352.145493872245424,
47360.866933177872852,
47369.588535522547318,
47378.310300879689748,
47387.032229222742899,
47395.754320525127696,
47404.476574760301446,
47413.198991901743284,
47421.921571922874136,
47430.644314797202242,
47439.367220498214010,
47448.090288999374025,
47456.813520274205075,
47465.536914296193572,
47474.260471038884134,
47482.984190475792275,
47491.708072580448061,
47500.432117326410662,
47509.156324687210144,
47517.880694636412954,
47526.605227147607366,
47535.329922194345272,
47544.054779750229500,
47552.779799788833770,
47561.504982283804566,
47570.230327208708331,
47578.955834537191549,
47587.681504242878873,
47596.407336299409508,
47605.133330680429935,
47613.859487359579362,
47622.585806310555199,
47631.312287506974826,
47640.038930922586587,
47648.765736531036964,
47657.492704306030646,
47666.219834221272322,
47674.947126250481233,
47683.674580367376620,
47692.402196545685001,
47701.129974759176548,
47709.857914981570502,
47718.586017186637037,
47727.314281348131772,
47736.042707439824881,
47744.771295435530192,
47753.500045309010602,
47762.228957034072664,
47770.958030584515654,
47779.687265934175230,
47788.416663056872494,
47797.146221926428552,
47805.875942516700889,
47814.605824801532435,
47823.335868754773401,
47832.066074350303097,
47840.796441561986285,
47849.526970363738656,
47858.257660729410418,
47866.988512632931815,
47875.719526048203988,
47884.450700949142629,
47893.182037309670704,
47901.913535103733011,
47910.645194305274345,
47919.377014888239501,
47928.108996826587827,
47936.841140094285947,
47945.573444665322313,
47954.305910513670824,
47963.038537613327208,
47971.771325938309019,
47980.504275462597434,
47989.237386160231836,
47997.970658005237055,
48006.704090971659753,
48015.437685033517482,
48024.171440164878732,
48032.905356339811988,
48041.639433532378462,
48050.373671716646641,
48059.108070866706839,
48067.842630956656649,
48076.577351960600936,
48085.312233852644567,
48094.047276606921514,
48102.782480197536643,
48111.517844598645752,
48120.253369784375536,
48128.989055728889070,
48137.724902406349429,
48146.460909790926962,
48155.197077856777469,
48163.933406578107679,
48172.669895929102495,
48181.406545883983199,
48190.143356416934694,
48198.880327502185537,
48207.617459113964287,
48216.354751226506778,
48225.092203814063396,
48233.829816850877251,
48242.567590311198728,
48251.305524169314594,
48260.043618399489787,
48268.781872976025625,
48277.520287873187044,
48286.258863065304467,
48294.997598526664660,
48303.736494231590768,
48312.475550154427765,
48321.214766269491520,
48329.954142551119730,
48338.693678973686474,
48347.433375511522172,
48356.173232139015454,
48364.913248830547673,
48373.653425560463802,
48382.393762303203403,
48391.134259033133276,
48399.874915724671155,
48408.615732352234772,
48417.356708890256414,
48426.097845313153812,
48434.839141595388355,
48443.580597711385053,
48452.322213635605294,
48461.063989342539571,
48469.805924806634721,
48478.548020002381236,
48487.290274904262333,
48496.032689486797608,
48504.775263724455726,
48513.517997591785388,
48522.260891063298914,
48531.003944113523175,
48539.747156716999598,
48548.490528848276881,
48557.234060481896449,
48565.977751592443383,
48574.721602154480934,
48583.465612142572354,
48592.209781531324552,
48600.954110295329883,
48609.698598409180704,
48618.443245847483922,
48627.188052584890102,
48635.933018595998874,
48644.678143855468079,
48653.423428337933728,
48662.168872018017282,
48670.914474870427512,
48679.660236869793152,
48688.406157990830252,
48697.152238208182098,
48705.898477496564738,
48714.644875830672390,
48723.391433185221103,
48732.138149534912372,
48740.885024854484072,
48749.632059118666803,
48758.379252302191162,
48767.126604379816854,
48775.874115326296305,
48784.621785116389219,
48793.369613724877127,
48802.117601126541558,
48810.865747296164045,
48819.614052208540670,
48828.362515838482068,
48837.111138160798873,
48845.859919150308997,
48854.608858781852177,
48863.357957030253601,
48872.107213870353007,
48880.856629277019238,
48889.606203225106583,
48898.355935689469334,
48907.105826645012712,
48915.855876066591009,
48924.606083929116721,
48933.356450207487796,
48942.106974876602180,
48950.857657911386923,
48959.608499286761798,
48968.359498977661133,
48977.110656959026528,
48985.861973205799586,
48994.613447692936461,
49003.365080395415134,
49012.116871288191760,
49020.868820346244320,
49029.620927544579899,
49038.373192858169205,
49047.125616262041149,
49055.878197731180990,
49064.630937240624917,
49073.383834765409119,
49082.136890280540683,
49090.890103761092178,
49099.643475182092516,
49108.397004518599715,
49117.150691745708173,
49125.904536838454078,
49134.658539771960932,
49143.412700521279476,
49152.167019061533210,
49160.921495367816533,
49169.676129415245668,
49178.430921178944118,
49187.185870634035382,
49195.940977755679341,
49204.696242518992221,
49213.451664899133903,
49222.207244871278817,
49230.962982410586847,
49239.718877492239699,
49248.474930091404531,
49257.231140183292155,
49265.987507743098831,
49274.744032746013545,
49283.500715167290764,
49292.257554982112197,
49301.014552165739588,
49309.771706693398301,
49318.529018540335528,
49327.286487681805738,
49336.044114093070675,
49344.801897749406635,
49353.559838626082637,
49362.317936698396807,
49371.076191941625439,
49379.834604331081209,
49388.593173842076794,
49397.351900449917594,
49406.110784129938111,
49414.869824857458298,
49423.629022607827210,
49432.388377356393903,
49441.147889078507433,
49449.907557749545958,
49458.667383344851260,
49467.427365839837876,
49476.187505209862138,
49484.947801430331310,
49493.708254476645379,
49502.468864324218885,
49511.229630948459089,
49519.990554324795085,
49528.751634428670513,
49537.512871235507191,
49546.274264720763313,
49555.035814859889797,
49563.797521628352115,
49572.559385001630289,
49581.321404955189792,
49590.083581464525196,
49598.845914505131077,
49607.608404052509286,
49616.371050082147121,
49625.133852569590090,
49633.896811490361870,
49642.659926819978864,
49651.423198534001131,
49660.186626607952348,
49668.950211017407128,
49677.713951737910975,
49686.477848745045776,
49695.241902014400694,
49704.006111521543062,
49712.770477242069319,
49721.534999151597731,
49730.299677225710184,
49739.064511440039496,
49747.829501770211209,
49756.594648191850865,
49765.359950680591282,
49774.125409212101658,
49782.891023762007535,
49791.656794305999938,
49800.422720819733513,
49809.188803278877458,
49817.955041659122799,
49826.721435936182388,
49835.487986085725424,
49844.254692083486589,
49853.021553905149631,
49861.788571526456508,
49870.555744923149177,
49879.323074070940493,
49888.090558945579687,
49896.858199522845098,
49905.625995778464130,
49914.393947688229673,
49923.162055227912788,
49931.930318373277260,
49940.698737100137805,
49949.467311384280038,
49958.236041201504122,
49967.004926527646603,
49975.773967338514922,
49984.543163609916519,
49993.312515317709767,
50002.082022437753039,
50010.851684945861052,
50019.621502817906730,
50028.391476029770274,
50037.161604557302780,
50045.931888376399002,
50054.702327462953690,
50063.472921792839770,
50072.243671341981099,
50081.014576086279703,
50089.785636001652165,
50098.556851064022339,
50107.328221249343187,
50116.099746533516736,
50124.871426892532327,
50133.643262302328367,
50142.415252738857816,
50151.187398178110016,
50159.959698596037924,
50168.732153968652710,
50177.504764271936438,
50186.277529481885722,
50195.050449574504455,
50203.823524525832909,
50212.596754311860423,
50221.370138908641820,
50230.143678292195546,
50238.917372438569146,
50247.691221323839272,
50256.465224924031645,
50265.239383215244743,
50274.013696173533390,
50282.788163774974237,
50291.562785995680315,
50300.337562811721000,
50309.112494199209323,
50317.887580134272866,
50326.662820593010110,
50335.438215551548637,
50344.213764986037859,
50352.989468872612633,
50361.765327187407820,
50370.541339906594658,
50379.317507006329834,
50388.093828462777310,
50396.870304252122878,
50405.646934350545052,
50414.423718734244176,
50423.200657379420591,
50431.977750262281916,
50440.754997359028494,
50449.532398645897047,
50458.309954099109746,
50467.087663694910589,
50475.865527409529022,
50484.643545219223597,
50493.421717100252863,
50502.200043028889922,
50510.978522981393326,
50519.757156934065279,
50528.535944863164332,
50537.314886744999967,
50546.093982555888942,
50554.873232272118912,
50563.652635870006634,
50572.432193325897970,
50581.211904616109678,
50589.991769716965791,
50598.771788604834001,
50607.551961256052891,
50616.332287647011981,
50625.112767754035303,
50633.893401553541480,
50642.674189021869097,
50651.455130135444051,
50660.236224870648584,
50669.017473203886766,
50677.798875111569942,
50686.580430570116732,
50695.362139555945760,
50704.144002045512025,
50712.926018015234149,
50721.708187441567134,
50730.490510300958704,
50739.272986569892964,
50748.055616224803089,
50756.838399242195010,
50765.621335598552832,
50774.404425270353386,
50783.187668234088051,
50791.971064466277312,
50800.754613943427103,
50809.538316642057907,
50818.322172538704763,
50827.106181609895430,
50835.890343832164945,
50844.674659182070172,
50853.459127636160702,
50862.243749171000673,
50871.028523763168778,
50879.813451389221882,
50888.598532025760505,
50897.383765649377892,
50906.169152236660011,
50914.954691764221934,
50923.740384208671458,
50932.526229546623654,
50941.312227754737251,
50950.098378809598216,
50958.884682687887107,
50967.671139366226271,
50976.457748821281712,
50985.244511029719433,
50994.031425968219992,
51002.818493613434839,
51011.605713942066359,
51020.393086930802383,
51029.180612556338019,
51037.968290795368375,
51046.756121624624939,
51055.544105020824645,
51064.332240960684430,
51073.120529420957610,
51081.908970378361118,
51090.697563809662824,
51099.486309691601491,
51108.275208000937710,
51117.064258714468451,
51125.853461808932479,
51134.642817261141317,
51143.432325047870108,
51152.221985145937651,
51161.011797532126366,
51169.801762183262326,
51178.591879076149780,
51187.382148187629355,
51196.172569494534400,
51204.963142973690992,
51213.753868601954309,
51222.544746356179530,
51231.335776213229110,
51240.126958149958227,
51248.918292143258441,
51257.709778169999481,
51266.501416207094735,
51275.293206231406657,
51284.085148219863186,
51292.877242149377707,
51301.669487996841781,
51310.461885739197896,
51319.254435353373992,
51328.047136816312559,
51336.839990104948811,
51345.632995196239790,
51354.426152067164367,
51363.219460694657755,
51372.012921055713377,
51380.806533127310104,
51389.600296886412252,
51398.394212310049625,
51407.188279375193815,
51415.982498058874626,
51424.776868338100030,
51433.571390189892554,
51442.366063591274724,
51451.160888519283617,
51459.955864950978139,
51468.750992863388092,
51477.546272233579657,
51486.341703038626292,
51495.137285255579627,
51503.933018861527671,
51512.728903833558434,
51521.524940148759924,
51530.321127784234704,
51539.117466717078059,
51547.913956924414379,
51556.710598383353499,
51565.507391071019811,
51574.304334964566806,
51583.101430041118874,
51591.898676277814957,
51600.696073651837651,
51609.493622140318621,
51618.291321720433189,
51627.089172369363951,
51635.887174064286228,
51644.685326782397169,
51653.483630500879372,
51662.282085196951812,
51671.080690847804362,
51679.879447430656001,
51688.678354922747531,
51697.477413301290653,
51706.276622543533449,
51715.075982626716723,
51723.875493528081279,
51732.675155224897026,
51741.474967694426596,
51750.274930913932621,
51759.075044860706839,
51767.875309512004605,
51776.675724845154036,
51785.476290837439592,
51794.277007466153009,
51803.077874708629679,
51811.878892542168614,
51820.680060944105207,
51829.481379891767574,
51838.282849362505658,
51847.084469333640300,
51855.886239782565099,
51864.688160686608171,
51873.490232023148565,
51882.292453769550775,
51891.094825903201126,
51899.897348401478666,
51908.700021241791546,
51917.502844401533366,
51926.305817858112277,
51935.108941588943708,
51943.912215571435809,
51952.715639783033112,
51961.519214201172872,
51970.322938803285069,
51979.126813566814235,
51987.930838469234004,
51996.735013487988908,
52005.539338600545307,
52014.343813784398662,
52023.148439017015335,
52031.953214275876235,
52040.758139538491378,
52049.563214782363502,
52058.368439984995348,
52067.173815123911481,
52075.979340176621918,
52084.785015120658500,
52093.590839933560346,
52102.396814592866576,
52111.202939076138136,
52120.009213360914146,
52128.815637424777378,
52137.622211245288781,
52146.428934800016577,
52155.235808066536265,
52164.042831022474275,
52172.850003645384277,
52181.657325912892702,
52190.464797802604153,
52199.272419292137783,
52208.080190359127300,
52216.888110981184582,
52225.696181135943334,
52234.504400801066367,
52243.312769954180112,
52252.121288572969206,
52260.929956635074632,
52269.738774118181027,
52278.547740999943926,
52287.356857258069795,
52296.166122870243271,
52304.975537814148993,
52313.785102067507978,
52322.594815608012141,
52331.404678413397050,
52340.214690461369173,
52349.024851729664078,
52357.835162196017336,
52366.645621838186344,
52375.456230633892119,
52384.266988560921163,
52393.077895597023598,
52401.888951719964098,
52410.700156907529163,
52419.511511137498019,
52428.323014387664443,
52437.134666635793110,
52445.946467859736003,
52454.758418037272349,
52463.570517146232305,
52472.382765164424200,
52481.195162069700018,
52490.007707839882642,
52498.820402452787675,
52507.633245886318036,
52516.446238118282054,
52525.259379126568092,
52534.072668889035413,
52542.886107383557828,
52551.699694588038255,
52560.513430480335956,
52569.327315038361121,
52578.141348240023945,
52586.955530063205515,
52595.769860485845129,
52604.584339485852979,
52613.398967041161086,
52622.213743129701470,
52631.028667729420704,
52639.843740818272636,
52648.658962374196562,
52657.474332375153608,
52666.289850799119449,
52675.105517624062486,
52683.921332827972947,
52692.737296388826508,
52701.553408284620673,
52710.369668493352947,
52719.186076993035385,
52728.002633761672769,
52736.819338777306257,
52745.636192017926078,
52754.453193461595220,
52763.270343086333014,
52772.087640870195173,
52780.905086791244685,
52789.722680827522709,
52798.540422957092233,
52807.358313158038072,
52816.176351408430492,
52824.994537686354306,
52833.812871969923435,
52842.631354237186315,
52851.449984466293245,
52860.268762635321764,
52869.087688722414896,
52877.906762705686560,
52886.725984563257953,
52895.545354273279372,
52904.364871813879290,
52913.184537163229834,
52922.004350299452199,
52930.824311200740340,
52939.644419845259108,
52948.464676211158803,
52957.285080276640656,
52966.105632019898621,
52974.926331419112103,
52983.747178452489607,
52992.568173098246916,
53001.389315334563435,
53010.210605139713152,
53019.032042491868197,
53027.853627369309834,
53036.675359750232019,
53045.497239612916019,
53054.319266935606720,
53063.141441696549009,
53071.963763874024153,
53080.786233446298866,
53089.608850391632586,
53098.431614688335685,
53107.254526314696705,
53116.077585248989635,
53124.900791469553951,
53133.724144954663643,
53142.547645682665461,
53151.371293631847948,
53160.195088780572405,
53169.019031107156479,
53177.843120589946921,
53186.667357207283203,
53195.491740937555733,
53204.316271759074880,
53213.140949650231050,
53221.965774589414650,
53230.790746554979705,
53239.615865525323898,
53248.441131478837633,
53257.266544393925869,
53266.092104248986288,
53274.917811022438400,
53283.743664692687162,
53292.569665238173911,
53301.395812637318159,
53310.222106868553965,
53319.048547910337220,
53327.875135741101985,
53336.701870339340530,
53345.528751683465089,
53354.355779751967930,
53363.182954523334047,
53372.010275976041157,
53380.837744088559703,
53389.665358839389228,
53398.493120207043830,
53407.321028170023055,
53416.149082706833724,
53424.977283796004485,
53433.805631416056713,
53442.634125545533607,
53451.462766162956541,
53460.291553246861440,
53469.120486775827885,
53477.949566728391801,
53486.778793083118217,
53495.608165818593989,
53504.437684913362318,
53513.267350346053718,
53522.097162095211388,
53530.927120139436738,
53539.757224457360280,
53548.587475027554319,
53557.417871828649368,
53566.248414839268662,
53575.079104038028163,
53583.909939403565659,
53592.740920914526214,
53601.572048549533065,
53610.403322287260380,
53619.234742106345948,
53628.066307985456660,
53636.898019903295790,
53645.729877838486573,
53654.561881769739557,
53663.394031675750739,
53672.226327535187011,
53681.058769326766196,
53689.891357029213395,
53698.724090621195501,
53707.556970081481268,
53716.389995388766692,
53725.223166521784151,
53734.056483459287847,
53742.889946180010156,
53751.723554662705283,
53760.557308886127430,
53769.391208829016250,
53778.225254470191430,
53787.059445788392622,
53795.893782762424962,
53804.728265371042653,
53813.562893593058106,
53822.397667407261906,
53831.232586792481015,
53840.067651727520570,
53848.902862191178428,
53857.738218162296107,
53866.573719619715121,
53875.409366542247881,
53884.245158908743178,
53893.081096698064357,
53901.917179889060208,
53910.753408460579521,
53919.589782391514746,
53928.426301660714671,
53937.262966247071745,
53946.099776129471138,
53954.936731286798022,
53963.773831697966671,
53972.611077341854980,
53981.448468197413604,
53990.286004243520438,
53999.123685459104308,
54007.961511823123146,
54016.799483314469398,
54025.637599912115547,
54034.475861594997696,
54043.314268342081050,
54052.152820132323541,
54060.991516944653995,
54069.830358758088551,
54078.669345551585138,
54087.508477304130793,
54096.347753994719824,
54105.187175602339266,
54114.026742105983431,
54122.866453484697558,
54131.706309717461409,
54140.546310783312947,
54149.386456661253760,
54158.226747330336366,
54167.067182769606006,
54175.907762958100648,
54184.748487874865532,
54193.589357498953177,
54202.430371809423377,
54211.271530785379582,
54220.112834405859758,
54228.954282649960078,
54237.795875496762164,
54246.637612925369467,
54255.479494914870884,
54264.321521444362588,
54273.163692492969858,
54282.006008039817971,
54290.848468064003100,
54299.691072544686904,
54308.533821460972831,
54317.376714792037092,
54326.219752517004963,
54335.062934615023551,
54343.906261065269064,
54352.749731846895884,
54361.593346939072944,
54370.437106320991006,
54379.281009971826279,
54388.125057870769524,
54396.969249997026054,
54405.813586329764803,
54414.658066848227463,
54423.502691531604796,
54432.347460359131219,
54441.192373310048424,
54450.037430363547173,
54458.882631498883711,
54467.727976695307007,
54476.573465932051477,
54485.419099188402470,
54494.264876443579851,
54503.110797676890797,
54511.956862867584277,
54520.803071994938364,
54529.649425038260233,
54538.495921976813406,
54547.342562789912336,
54556.189347456864198,
54565.036275956961617,
54573.883348269519047,
54582.730564373894595,
54591.577924249359057,
54600.425427875277819,
54609.273075231001712,
54618.120866295837914,
54626.968801049173635,
54635.816879470345157,
54644.665101538710587,
54653.513467233642587,
54662.361976534535643,
54671.210629420762416,
54680.059425871673739,
54688.908365866715030,
54697.757449385244399,
54706.606676406678162,
54715.456046910432633,
54724.305560875924130,
54733.155218282554415,
54742.005019109783461,
54750.854963337013032,
54759.705050943710376,
54768.555281909299083,
54777.405656213239126,
54786.256173834983201,
54795.106834753991279,
54803.957638949759712,
54812.808586401726643,
54821.659677089388424,
54830.510910992241406,
54839.362288089752838,
54848.213808361448173,
54857.065471786809212,
54865.917278345375962,
54874.769228016622947,
54883.621320780097449,
54892.473556615324924,
54901.325935501838103,
54910.178457419169717,
54919.031122346881602,
54927.883930264499213,
54936.736881151606212,
54945.589974987749883,
54954.443211752492061,
54963.296591425409133,
54972.150113986099313,
54981.003779414138990,
54989.857587689111824,
54998.711538790630584,
55007.565632698286208,
55016.419869391698739,
55025.274248850459117,
55034.128771054216486,
55042.983435982590890,
55051.838243615209649,
55060.693193931707356,
55069.548286911733157,
55078.403522534943477,
55087.258900780987460,
55096.114421629528806,
55104.970085060223937,
55113.825891052758379,
55122.681839586810383,
55131.537930642050924,
55140.394164198209182,
55149.250540234948858,
55158.107058731955476,
55166.963719668972772,
55175.820523025708098,
55184.677468781861535,
55193.534556917169539,
55202.391787411375844,
55211.249160244202358,
55220.106675395392813,
55228.964332844705496,
55237.822132571884140,
55246.680074556687032,
55255.538158778872457,
55264.396385218249634,
55273.254753854547744,
55282.113264667583280,
55290.971917637143633,
55299.830712742994365,
55308.689649964959244,
55317.548729282840213,
55326.407950676453765,
55335.267314125594567,
55344.126819610122766,
55352.986467109840305,
55361.846256604585506,
55370.706188074196689,
55379.566261498526728,
55388.426476857435773,
55397.286834130762145,
55406.147333298380545,
55415.007974340151122,
55423.868757235963130,
55432.729681965676718,
55441.590748509195691,
55450.451956846409303,
55459.313306957206805,
55468.174798821477452,
55477.036432419168705,
55485.898207730169815,
55494.760124734413694,
55503.622183411796868,
55512.484383742288628,
55521.346725705807330,
55530.209209282293159,
55539.071834451700852,
55547.934601193985145,
55556.797509489100776,
55565.660559317024308,
55574.523750657717756,
55583.387083491150406,
55592.250557797327929,
55601.114173556219612,
55609.977930747816572,
55618.841829352139030,
55627.705869349170825,
55636.570050718939456,
55645.434373441457865,
55654.298837496731721,
55663.163442864810349,
55672.028189525721245,
55680.893077459484630,
55689.758106646178931,
55698.623277065824368,
55707.488588698499370,
55716.354041524260538,
55725.219635523157194,
55734.085370675275044,
55742.951246960699791,
55751.817264359517139,
55760.683422851798241,
55769.549722417650628,
55778.416163037174556,
55787.282744690484833,
55796.149467357681715,
55805.016331018880010,
55813.883335654216353,
55822.750481243812828,
55831.617767767798796,
55840.485195206325443,
55849.352763539529406,
55858.220472747561871,
55867.088322810595855,
55875.956313708775269,
55884.824445422287681,
55893.692717931277002,
55902.561131215952628,
55911.429685256480298,
55920.298380033062131,
55929.167215525900247,
55938.036191715167661,
55946.905308581102872,
55955.774566103900725,
55964.643964263777889,
55973.513503040972864,
55982.383182415709598,
55991.253002368212037,
56000.122962878733233,
56008.993063927518961,
56017.863305494814995,
56026.733687560888939,
56035.604210105993843,
56044.474873110397311,
56053.345676554388774,
56062.216620418214006,
56071.087704682206095,
56079.958929326610814,
56088.830294331746700,
56097.701799677917734,
56106.573445345413347,
56115.445231314559351,
56124.317157565681555,
56133.189224079091218,
56142.061430835121428,
56150.933777814097994,
56159.806264996375830,
56168.678892362295301,
56177.551659892211319,
56186.424567566464248,
56195.297615365445381,
56204.170803269487806,
56213.044131258990092,
56221.917599314321706,
56230.791207415881217,
56239.664955544030818,
56248.538843679190904,
56257.412871801745496,
56266.287039892100438,
56275.161347930690681,
56284.035795897914795,
56292.910383774200454,
56301.785111539960781,
56310.659979175645276,
56319.534986661696166,
56328.410133978548402,
56337.285421106658760,
56346.160848026469466,
56355.036414718473679,
56363.912121163099073,
56372.787967340831528,
56381.663953232156928,
56390.540078817546600,
56399.416344077508256,
56408.292748992520501,
56417.169293543069216,
56426.045977709691215,
56434.922801472865103,
56443.799764813134971,
56452.676867710993974,
56461.554110146986204,
56470.431492101633921,
56479.309013555503043,
56488.186674489094003,
56497.064474882965442,
56505.942414717697829,
56514.820493973827979,
56523.698712631914532,
56532.577070672537957,
56541.455568076278723,
56550.334204823710024,
56559.212980895419605,
56568.091896272002487,
56576.970950934039138,
56585.850144862146408,
56594.729478036933870,
56603.608950439003820,
56612.488562048994936,
56621.368312847502239,
56630.248202815178956,
56639.128231932649214,
56648.008400180558965,
56656.888707539539610,
56665.769153990258928,
56674.649739513348322,
56683.530464089504676,
56692.411327699366666,
56701.292330323623901,
56710.173471942944161,
56719.054752538017055,
56727.936172089532192,
56736.817730578171904,
56745.699427984633076,
56754.581264289648971,
56763.463239473894646,
56772.345353518096090,
56781.227606402986567,
56790.109998109292064,
56798.992528617716744,
56807.875197909022972,
56816.758005963944015,
56825.640952763234964,
56834.524038287636358,
56843.407262517917843,
56852.290625434834510,
56861.174127019163279,
56870.057767251673795,
56878.941546113128425,
56887.825463584333193,
56896.709519646079571,
56905.593714279159030,
56914.478047464348492,
56923.362519182497635,
56932.247129414376104,
56941.131878140826302,
56950.016765342661529,
56958.901791000716912,
56967.786955095813028,
56976.672257608792279,
56985.557698520511622,
56994.443277811777079,
57003.328995463489264,
57012.214851456497854,
57021.100845771645254,
57029.986978389832075,
57038.873249291900720,
57047.759658458737249,
57056.646205871256825,
57065.532891510330955,
57074.419715356845700,
57083.306677391701669,
57092.193777595828578,
57101.081015950127039,
57109.968392435504938,
57118.855907032884716,
57127.743559723217913,
57136.631350487419695,
57145.519279306412500,
57154.407346161162423,
57163.295551032628282,
57172.183893901739793,
57181.072374749455776,
57189.960993556756875,
57198.849750304609188,
57207.738644973978808,
57216.627677545860934,
57225.516848001214385,
57234.406156321056187,
57243.295602486381540,
57252.185186478171090,
57261.074908277449140,
57269.964767865225440,
57278.854765222517017,
57287.744900330326345,
57296.635173169706832,
57305.525583721675503,
57314.416131967285764,
57323.306817887561920,
57332.197641463564651,
57341.088602676354640,
57349.979701506970741,
57358.870937936495466,
57367.762311945967667,
57376.653823516498960,
57385.545472629150026,
57394.437259264996101,
57403.329183405170625,
57412.221245030705177,
57421.113444122747751,
57430.005780662388133,
57438.898254630737938,
57447.790866008908779,
57456.683614778019546,
57465.576500919218233,
57474.469524413609179,
57483.362685242347652,
57492.255983386567095,
57501.149418827415502,
57510.042991546040867,
57518.936701523613010,
57527.830548741279927,
57536.724533180225990,
57545.618654821599193,
57554.512913646605739,
57563.407309636408172,
57572.301842772212694,
57581.196513035181852,
57590.091320406558225,
57598.986264867518912,
57607.881346399277390,
57616.776564983032586,
57625.671920600041631,
57634.567413231488899,
57643.463042858616973,
57652.358809462668432,
57661.254713024893135,
57670.150753526519111,
57679.046930948788940,
57687.943245272967033,
57696.839696480332350,
57705.736284552112920,
57714.633009469616809,
57723.529871214101149,
57732.426869766837626,
57741.324005109141581,
57750.221277222262870,
57759.118686087545939,
57768.016231686247920,
57776.913913999713259,
57785.811733009220916,
57794.709688696108060,
57803.607781041675480,
57812.506010027274897,
57821.404375634228927,
57830.302877843882015,
57839.201516637556779,
57848.100291996626765,
57856.999203902414592,
57865.898252336308360,
57874.797437279637961,
57883.696758713798772,
57892.596216620149789,
57901.495810980079114,
57910.395541774960293,
57919.295408986181428,
57928.195412595145172,
57937.095552583232347,
57945.995828931874712,
57954.896241622453090,
57963.796790636391961,
57972.697475955101254,
57981.598297560005449,
57990.499255432565405,
57999.400349554161949,
58008.301579906277766,
58017.202946470322786,
58026.104449227779696,
58035.006088160094805,
58043.907863248685317,
58052.809774475070299,
58061.711821820688783,
58070.614005267023458,
58079.516324795557011,
58088.418780387757579,
58097.321372025129676,
58106.224099689170544,
58115.126963361370144,
58124.029963023225719,
58132.933098656270886,
58141.836370241988334,
58150.739777761926234,
58159.643321197581827,
58168.547000530517835,
58177.450815742246050,
58186.354766814300092,
58195.258853728235408,
58204.163076465592894,
58213.067435007949825,
58221.971929336861649,
58230.876559433861985,
58239.781325280542660,
58248.686226858480950,
58257.591264149239578,
58266.496437134424923,
58275.401745795621537,
58284.307190114406694,
58293.212770072394051,
58302.118485651197261,
58311.024336832393601,
58319.930323597640381,
58328.836445928522153,
58337.742703806681675,
58346.649097213718051,
58355.555626131324971,
58364.462290541101538,
58373.369090424683236,
58382.276025763749203,
58391.183096539934922,
58400.090302734904981,
58408.997644330309413,
58417.905121307827358,
58426.812733649152506,
58435.720481335949444,
58444.628364349890035,
58453.536382672667969,
58462.444536285984213,
58471.352825171554286,
58480.261249311042775,
58489.169808686187025,
58498.078503278702556,
58506.987333070275781,
58515.896298042673152,
58524.805398177610186,
58533.714633456795127,
58542.624003861994424,
58551.533509374930873,
58560.443149977385474,
58569.352925651066471,
58578.262836377783970,
58587.172882139260764,
58596.083062917270581,
58604.993378693594423,
58613.903829450006015,
58622.814415168308187,
58631.725135830238287,
58640.635991417650075,
58649.546981912310002,
58658.458107296028174,
58667.369367550592870,
58676.280762657843297,
58685.192292599567736,
58694.103957357634499,
58703.015756913817313,
58711.927691249984491,
58720.839760347931588,
58729.751964189570572,
58738.664302756689722,
58747.576776031157351,
58756.489383994819946,
58765.402126629560371,
58774.315003917239665,
58783.228015839711588,
58792.141162378844456,
58801.054443516564788,
58809.967859234726347,
58818.881409515219275,
58827.795094339933712,
58836.708913690796180,
58845.622867549682269,
58854.536955898533051,
58863.451178719231393,
58872.365535993732919,
58881.280027703920496,
58890.194653831735195,
58899.109414359132643,
58908.024309268039360,
58916.939338540403696,
58925.854502158166724,
58934.769800103291345,
58943.685232357711357,
58952.600798903426039,
58961.516499722391018,
58970.432334796547366,
58979.348304107923468,
58988.264407638482226,
58997.180645370208367,
59006.097017285093898,
59015.013523365123547,
59023.930163592318422,
59032.846937948692357,
59041.763846416230081,
59050.680888976981805,
59059.598065612946812,
59068.515376306131657,
59077.432821038608381,
59086.350399792383541,
59095.268112549514626,
59104.185959292037296,
59113.103940002016316,
59122.022054661494622,
59130.940303252529702,
59139.858685757186322,
59148.777202157543798,
59157.695852435674169,
59166.614636573656753,
59175.533554553563590,
59184.452606357495824,
59193.371791967525496,
59202.291111365782854,
59211.210564534369041,
59220.130151455370651,
59229.049872110888828,
59237.969726483104751,
59246.889714554061356,
59255.809836305925273,
59264.730091720848577,
59273.650480780917860,
59282.571003468321578,
59291.491659765175427,
59300.412449653638760,
59309.333373115878203,
59318.254430134024005,
59327.175620690271899,
59336.096944766781235,
59345.018402345718641,
59353.939993409287126,
59362.861717939660593,
59371.783575919012947,
59380.705567329547193,
59389.627692153473618,
59398.549950372966123,
59407.472341970264097,
59416.394866927570547,
59425.317525227095757,
59434.240316851057287,
59443.163241781701799,
59452.086300001246855,
59461.009491491939116,
59469.932816236003418,
59478.856274215700978,
59487.779865413271182,
59496.703589810975245,
59505.627447391081660,
59514.551438135837088,
59523.475562027531851,
59532.399819048398058,
59541.324209180784237,
59550.248732406907948,
59559.173388709110441,
59568.098178069638379,
59577.023100470803911,
59585.948155894933734,
59594.873344324318168,
59603.798665741247532,
59612.724120128084905,
59621.649707467120606,
59630.575427740674058,
59639.501280931093788,
59648.427267020706495,
59657.353385991875257,
59666.279637826904946,
59675.206022508151364,
59684.132540018006694,
59693.059190338797634,
59701.985973452909093,
59710.912889342689596,
59719.839937990509497,
59728.767119378760981,
59737.694433489821677,
59746.621880306076491,
59755.549459809917607,
59764.477171983751759,
59773.405016809956578,
59782.332994270967902,
59791.261104349163361,
59800.189347026993346,
59809.117722286842763,
59818.046230111162004,
59826.974870482365077,
59835.903643382873270,
59844.832548795158800,
59853.761586701642955,
59862.690757084790675,
59871.620059927023249,
59880.549495210834721,
59889.479062918668205,
59898.408763032988645,
59907.338595536253706,
59916.268560410964710,
59925.198657639586600,
59934.128887204598868,
59943.059249088510114,
59951.989743273799832,
59960.920369742976618,
59969.851128478534520,
59978.782019462989410,
59987.713042678842612,
59996.644198108617275,
60005.575485734843824,
60014.506905540045409,
60023.438457506730629,
60032.370141617466288,
60041.301957854775537,
60050.233906201217906,
60059.165986639301991,
60068.098199151645531,
60077.030543720757123,
60085.963020329210849,
60094.895628959588066,
60103.828369594455580,
60112.761242216380197,
60121.694246807972377,
60130.627383351769822,
60139.560651830412098,
60148.494052226458734,
60157.427584522549296,
60166.361248701243312,
60175.295044745180348,
60184.228972636949038,
60193.163032359203498,
60202.097223894554190,
60211.031547225611575,
60219.966002335022495,
60228.900589205397409,
60237.835307819412265,
60246.770158159706625,
60255.705140208912781,
60264.640253949713951,
60273.575499364749703,
60282.510876436674153,
60291.446385148185072,
60300.382025481922028,
60309.317797420597344,
60318.253700946872414,
60327.189736043445009,
60336.125902692976524,
60345.062200878208387,
60353.998630581816542,
60362.935191786498763,
60371.871884474996477,
60380.808708629992907,
60389.745664234222204,
60398.682751270396693,
60407.619969721250527,
60416.557319569525134,
60425.494800797940115,
60434.432413389251451,
60443.370157326186018,
60452.308032591514348,
60461.246039167992421,
60470.184177038347116,
60479.122446185392619,
60488.060846591877635,
60496.999378240550868,
60505.938041114211956,
60514.876835195638705,
60523.815760467645305,
60532.754816912973183,
60541.694004514451080,
60550.633323254864081,
60559.572773117040924,
60568.512354083759419,
60577.452066137833754,
60586.391909262114496,
60595.331883439401281,
60604.271988652544678,
60613.212224884329771,
60622.152592117643508,
60631.093090335300076,
60640.033719520142768,
60648.974479655036703,
60657.915370722817897,
60666.856392706358747,
60675.797545588524372,
60684.738829352158064,
60693.680243980175874,
60702.621789455406542,
60711.563465760780673,
60720.505272879141557,
60729.447210793405247,
60738.389279486444138,
60747.331478941159730,
60756.273809140482626,
60765.216270067307050,
60774.158861704534502,
60783.101584035102860,
60792.044437041906349,
60800.987420707904676,
60809.930535015999340,
60818.873779949128220,
60827.817155490229197,
60836.760661622276530,
60845.704298328178993,
60854.648065590925398,
60863.591963393446349,
60872.535991718679725,
60881.480150549657992,
60890.424439869326307,
60899.368859660615271,
60908.313409906564630,
60917.258090590119537,
60926.202901694283355,
60935.147843202044896,
60944.092915096429351,
60953.038117360367323,
60961.983449976949487,
60970.928912929106446,
60979.874506199928874,
60988.820229772376479,
60997.766083629510831,
61006.712067754342570,
61015.658182129896886,
61024.604426739220798,
61033.550801565368602,
61042.497306591358210,
61051.443941800265748,
61060.390707175138232,
61069.337602699022682,
61078.284628354995220,
61087.231784126117418,
61096.179069995479949,
61105.126485946122557,
61114.074031961165019,
61123.021708023647079,
61131.969514116710343,
61140.917450223416381,
61149.865516326884972,
61158.813712410192238,
61167.762038456457958,
61176.710494448801910,
61185.659080370343872,
61194.607796204196347,
61203.556641933479114,
61212.505617541304673,
61221.454723010851012,
61230.403958325216081,
61239.353323467563314,
61248.302818421041593,
61257.252443168799800,
61266.202197694008646,
61275.152081979773357,
61284.102096009308298,
61293.052239765762351,
61302.002513232320780,
61310.952916392161569,
61319.903449228448153,
61328.854111724387622,
61337.804903863143409,
61346.755825627937156,
61355.706877001975954,
61364.658057968430512,
61373.609368510529748,
61382.560808611466200,
61391.512378254497889,
61400.464077422802802,
61409.415906099631684,
61418.367864268191624,
61427.319951911747921,
61436.272169013507664,
61445.224515556736151,
61454.176991524669575,
61463.129596900565957,
61472.082331667676044,
61481.035195809243305,
61489.988189308569417,
61498.941312148890574,
61507.894564313515730,
61516.847945785681077,
61525.801456548681017,
61534.755096585809952,
61543.708865880362282,
61552.662764415639685,
61561.616792174892907,
61570.570949141467281,
61579.525235298686312,
61588.479650629822572,
61597.434195118214120,
61606.388868747155357,
61615.343671500013443,
61624.298603360082780,
61633.253664310737804,
61642.208854335243814,
61651.164173417018901,
61660.119621539364744,
61669.075198685648502,
61678.030904839215509,
61686.986739983425650,
61695.942704101660638,
61704.898797177273082,
61713.855019193637418,
61722.811370134128083,
61731.767849982126791,
61740.724458721022529,
61749.681196334211563,
61758.638062805053778,
61767.595058116974542,
61776.552182253391948,
61785.509435197673156,
61794.466816933258087,
61803.424327443550283,
61812.381966711975110,
61821.339734721936111,
61830.297631456887757,
61839.255656900248141,
61848.213811035457184,
61857.172093845947529,
61866.130505315180926,
61875.089045426597295,
61884.047714163651108,
61893.006511509782285,
61901.965437448488956,
61910.924491963203764,
61919.883675037410285,
61928.842986654584820,
61937.802426798196393,
61946.761995451750408,
61955.721692598701338,
61964.681518222561863,
61973.641472306844662,
61982.601554835004208,
61991.561765790582285,
62000.522105157084297,
62009.482572917993821,
62018.443169056859915,
62027.403893557195261,
62036.364746402505261,
62045.325727576360805,
62054.286837062245468,
62063.248074843744689,
62072.209440904371149,
62081.170935227681184,
62090.132557797216577,
62099.094308596555493,
62108.056187609239714,
62117.018194818818301,
62125.980330208891246,
62134.942593763007608,
62143.904985464738274,
62152.867505297697790,
62161.830153245442489,
62170.792929291557812,
62179.755833419651026,
62188.718865613322123,
62197.682025856149266,
62206.645314131761552,
62215.608730423751695,
62224.572274715748790,
62233.535946991367382,
62242.499747234222014,
62251.463675427941780,
62260.427731556155777,
62269.391915602522204,
62278.356227550641051,
62287.320667384185072,
62296.285235086790635,
62305.249930642101390,
62314.214754033790086,
62323.179705245514924,
62332.144784260904999,
62341.109991063691268,
62350.075325637488277,
62359.040787965997879,
62368.006378032907378,
62376.972095821889525,
62385.937941316646175,
62394.903914500857354,
62403.870015358210367,
62412.836243872450723,
62421.802600027236622,
62430.769083806291746,
62439.735695193339780,
62448.702434172118956,
62457.669300726294750,
62466.636294839656330,
62475.603416495869169,
62484.570665678715159,
62493.538042371939810,
62502.505546559252252,
62511.473178224412550,
62520.440937351195316,
62529.408823923331511,
62538.376837924566644,
62547.344979338689882,
62556.313248149475839,
62565.281644340677303,
62574.250167896054336,
62583.218818799447035,
62592.187597034579085,
62601.156502585268754,
62610.125535435297934,
62619.094695568477619,
62628.063982968596974,
62637.033397619459720,
62646.002939504891401,
62654.972608608673909,
62663.942404914654617,
62672.912328406659071,
62681.882379068490991,
62690.852556883983198,
62699.822861836961238,
62708.793293911287037,
62717.763853090800694,
62726.734539359335031,
62735.705352700744697,
62744.676293098891620,
62753.647360537630448,
62762.618555000794004,
62771.589876472316973,
62780.561324936010351,
62789.532900375765166,
62798.504602775472449,
62807.476432118994126,
62816.448388390250329,
62825.420471573088435,
62834.392681651457679,
62843.365018609190884,
62852.337482430266391,
62861.310073098538851,
62870.282790597942949,
62879.255634912369715,
62888.228606025768386,
62897.201703922051820,
62906.174928585140151,
62915.148279998997168,
62924.121758147499349,
62933.095363014646864,
62942.069094584359846,
62951.042952840587532,
62960.016937767279160,
62968.991049348391243,
62977.965287567887572,
62986.939652409731934,
62995.914143857888121,
63004.888761896356300,
63013.863506509078434,
63022.838377680054691,
63031.813375393241586,
63040.788499632675666,
63049.763750382327999,
63058.739127626176924,
63067.714631348237162,
63076.690261532530712,
63085.666018163057743,
63094.641901223811146,
63103.617910698842024,
63112.594046572128718,
63121.570308827751433,
63130.546697449688509,
63139.523212422012875,
63148.499853728731978,
63157.476621353904193,
63166.453515281573345,
63175.430535495790537,
63184.407681980585039,
63193.384954720058886,
63202.362353698234074,
63211.339878899205360,
63220.317530307016568,
63229.295307905769732,
63238.273211679515953,
63247.251241612357262,
63256.229397688352037,
63265.207679891638691,
63274.186088206261047,
63283.164622616342967,
63292.143283105979208,
63301.122069659293629,
63310.100982260366436,
63319.080020893321489,
63328.059185542289924,
63337.038476191381051,
63346.017892824711453,
63354.997435426448646,
63363.977103980694665,
63372.956898471587920,
63381.936818883281376,
63390.916865199913445,
63399.897037405637093,
63408.877335484605283,
63417.857759420985531,
63426.838309198916249,
63435.818984802594059,
63444.799786216164648,
63453.780713423810084,
63462.761766409712436,
63471.742945158046496,
63480.724249652994331,
63489.705679878774390,
63498.687235819539637,
63507.668917459523072,
63516.650724782892212,
63525.632657773894607,
63534.614716416683223,
63543.596900695527438,
63552.579210594616598,
63561.561646098176425,
63570.544207190425368,
63579.526893855625531,
63588.509706077958981,
63597.492643841709651,
63606.475707131088711,
63615.458895930350991,
63624.442210223744041,
63633.425649995529966,
63642.409215229963593,
63651.392905911292473,
63660.376722023800539,
63669.360663551728067,
63678.344730479388090,
63687.328922791035438,
63696.313240470939490,
63705.297683503413282,
63714.282251872726192,
63723.266945563147601,
63732.251764559034200,
63741.236708844648092,
63750.221778404287761,
63759.206973222273518,
63768.192293282940227,
63777.177738570542715,
63786.163309069459501,
63795.149004764010897,
63804.134825638488110,
63813.120771677255107,
63822.106842864624923,
63831.093039184961526,
63840.079360622570675,
63849.065807161838165,
63858.052378787113412,
63867.039075482738554,
63876.025897233070282,
63885.012844022479840,
63893.999915835323918,
63902.987112655988312,
63911.974434468844265,
63920.961881258248468,
63929.949453008615819,
63938.937149704324838,
63947.924971329761320,
63956.912917869303783,
63965.900989307388954,
63974.889185628388077,
63983.877506816694222,
63992.865952856773220,
64001.854523732967209,
64010.843219429763849,
64019.832039931556210,
64028.820985222737363,
64037.810055287780415,
64046.799250111107540,
64055.788569677148189,
64064.778013970360917,
64073.767582975160622,
64082.757276676034962,
64091.747095057384286,
64100.737038103710802,
64109.727105799473065,
64118.717298129115079,
64127.707615077109949,
64136.698056627930782,
64145.688622766065237,
64154.679313475993695,
64163.670128742160159,
64172.661068549110496,
64181.652132881303260,
64190.643321723218833,
64199.634635059395805,
64208.626072874329111,
64217.617635152484581,
64226.609321878422634,
64235.601133036638203,
64244.593068611640774,
64253.585128587976214,
64262.577312950154010,
64271.569621682690922,
64280.562054770161922,
64289.554612197054666,
64298.547293947958678,
64307.540100007376168,
64316.533030359896657,
64325.526084990036907,
64334.519263882371888,
64343.512567021476571,
64352.505994391889544,
64361.499545978185779,
64370.493221764932969,
64379.487021736742463,
64388.480945878145576,
64397.474994173760933,
64406.469166608148953,
64415.463463165928260,
64424.457883831695653,
64433.452428590004274,
64442.447097425516404,
64451.441890322799736,
64460.436807266472897,
64469.431848241147236,
64478.427013231470482,
64487.422302222024882,
64496.417715197458165,
64505.413252142403508,
64514.408913041464984,
64523.404697879319428,
64532.400606640578189,
64541.396639309903549,
64550.392795871935959,
64559.389076311337703,
64568.385480612749234,
64577.382008760847384,
64586.378660740272608,
64595.375436535709014,
64604.372336131833436,
64613.369359513322706,
64622.366506664846384,
64631.363777571074024,
64640.361172216711566,
64649.358690586443117,
64658.356332664967340,
64667.354098436975619,
64676.351987887181167,
64685.350001000275370,
64694.348137760978716,
64703.346398153997143,
64712.344782164051139,
64721.343289775861194,
64730.341920974147797,
64739.340675743631436,
64748.339554069061705,
64757.338555935195473,
64766.337681326709571,
64775.336930228389974,
64784.336302624978998,
64793.335798501240788,
64802.335417841903109,
64811.335160631722829,
64820.335026855500473,
64829.335016497949255,
64838.335129543876974,
64847.335365978047776,
64856.335725785240356,
64865.336208950226137,
64874.336815457798366,
64883.337545292728464,
64892.338398439853336,
64901.339374883929850,
64910.340474609736702,
64919.341697602139902,
64928.343043845903594,
64937.344513325864682,
64946.346106026809139,
64955.347821933573869,
64964.349661030959396,
64973.351623303831730,
64982.353708736991393,
64991.355917315275292,
65000.358249023513054,
65009.360703846577962,
65018.363281769263267,
65027.365982776464080,
65036.368806853002752,
65045.371753983774397,
65054.374824153579539,
65063.378017347335117,
65072.381333549878036,
65081.384772746081580,
65090.388334920826310,
65099.392020059007336,
65108.395828145454288,
65117.399759165105934,
65126.403813102835556,
65135.407989943538269,
65144.412289672101906,
65153.416712273436133,
65162.421257732414233,
65171.425926034004078,
65180.430717163071677,
65189.435631104533968,
65198.440667843315168,
65207.445827364346769,
65216.451109652560262,
65225.456514692865312,
65234.462042470229790,
65243.467692969534255,
65252.473466175775684,
65261.479362073863740,
65270.485380648780847,
65279.491521885429393,
65288.497785768820904,
65297.504172283857770,
65306.510681415551517,
65315.517313148862740,
65324.524067468730209,
65333.530944360165449,
65342.537943808121781,
65351.545065797596180,
65360.552310313556518,
65369.559677341014321,
65378.567166864944738,
65387.574778870359296,
65396.582513342233142,
65405.590370265592355,
65414.598349625441188,
65423.606451406776614,
65432.614675594624714,
65441.623022174011567,
65450.631491129948699,
65459.640082447462191,
65468.648796111592674,
65477.657632107351674,
65486.666590419794375,
65495.675671033968683,
65504.684873934886127,
65513.694199107623717,
65522.703646537222085,
65531.713216208729136,
65540.722908107214607,
65549.732722217740957,
65558.742658525385195,
65567.752717015173403,
65576.762897672233521,
65585.773200481606182,
65594.783625428390224,
65603.794172497655381,
65612.804841674500494,
65621.815632944024401,
65630.826546291296836,
65639.837581701445743,
65648.848739159555407,
65657.860018650724669,
65666.871420160096022,
65675.882943672739202,
65684.894589173811255,
65693.906356648396468,
65702.918246081666439,
65711.930257458676351,
65720.942390764641459,
65729.954645984646049,
65738.967023103832616,
65747.979522107358207,
65756.992142980365315,
65766.004885707996436,
65775.017750275394064,
65784.030736667758902,
65793.043844870204339,
65802.057074867901974,
65811.070426646037959,
65820.083900189783890,
65829.097495484296815,
65838.111212514777435,
65847.125051266382798,
65856.139011724313605,
65865.153093873741454,
65874.167297699896153,
65883.181623187963851,
65892.196070323116146,
65901.210639090568293,
65910.225329475535545,
65919.240141463218606,
65928.255075038847281,
65937.270130187651375,
65946.285306894817040,
65955.300605145603186,
65964.316024925195961,
65973.331566218868829,
65982.347229011851596,
65991.363013289344963,
66000.378919036651496,
66009.394946238986449,
66018.411094881594181,
66027.427364949748153,
66036.443756428692723,
66045.460269303686800,
66054.476903560003848,
66063.493659182888223,
66072.510536157657043,
66081.527534469569218,
66090.544654103869107,
66099.561895045888377,
66108.579257280871389,
66117.596740794149810,
66126.614345570967998,
66135.632071596672176,
66144.649918856535805,
66153.667887335861451,
66162.685977019980783,
66171.704187894167262,
66180.722519943767111,
66189.740973154082894,
66198.759547510417178,
66207.778242998159840,
66216.797059602569789,
66225.815997309022350,
66234.835056102834642,
66243.854235969338333,
66252.873536893894197,
66261.892958861863008,
66270.912501858547330,
66279.932165869351593,
66288.951950879622018,
66297.971856874690275,
66306.991883839931688,
66316.012031760736136,
66325.032300622449839,
66334.052690410477226,
66343.073201110164518,
66352.093832706901594,
66361.114585186107433,
66370.135458533128258,
66379.156452733368496,
66388.177567772247130,
66397.198803635110380,
66406.220160307449987,
66415.241637774568517,
66424.263236021943158,
66433.284955034978339,
66442.306794799093041,
66451.328755299677141,
66460.350836522207828,
66469.373038452060428,
66478.395361074697576,
66487.417804375552805,
66496.440368340045097,
66505.463052953651641,
66514.485858201776864,
66523.508784069883404,
66532.531830543448450,
66541.554997607905534,
66550.578285248688189,
66559.601693451317260,
66568.625222201240831,
66577.648871483892435,
66586.672641284792917,
66595.696531589404913,
66604.720542383161956,
66613.744673651628545,
66622.768925380238215,
66631.793297554497258,
66640.817790159911965,
66649.842403181944974,
66658.867136606131680,
66667.891990417978377,
66676.916964602976805,
66685.942059146633255,
66694.967274034497677,
66703.992609252047259,
66713.018064784831950,
66722.043640618358040,
66731.069336738175480,
66740.095153129805112,
66749.121089778782334,
66758.147146670642542,
66767.173323790950235,
66776.199621125226258,
66785.226038659049664,
66794.252576377941296,
66803.279234267494758,
66812.306012313245446,
66821.332910500757862,
66830.359928815596504,
66839.387067243340425,
66848.414325769597781,
66857.441704379874864,
66866.469203059823485,
66875.496821794979041,
66884.524560570964240,
66893.552419373329030,
66902.580398187725223,
66911.608496999702766,
66920.636715794884367,
66929.665054558892734,
66938.693513277277816,
66947.722091935735079,
66956.750790519814473,
66965.779609015167807,
66974.808547407417791,
66983.837605682172580,
66992.866783825069433,
67001.896081821760163,
67010.925499657852924,
67019.955037318999530,
67028.984694790851790,
67038.014472059046966,
67047.044369109236868,
67056.074385927087860,
67065.104522498237202,
67074.134778808365809,
67083.165154843110940,
67092.195650588182616,
67101.226266029203543,
67110.257001151883742,
67119.287855941889575,
67128.318830384887406,
67137.349924466587254,
67146.381138172670035,
67155.412471488816664,
67164.443924400737160,
67173.475496894112439,
67182.507188954652520,
67191.539000568081974,
67200.570931720096269,
67209.602982396376319,
67218.635152582675801,
67227.667442264719284,
67236.699851428216789,
67245.732380058863782,
67254.765028142428491,
67263.797795664635487,
67272.830682611209340,
67281.863688967903727,
67290.896814720457769,
67299.930059854610590,
67308.963424356115866,
67317.996908210727270,
67327.030511404213030,
67336.064233922312269,
67345.098075750778662,
67354.132036875438644,
67363.166117281987681,
67372.200316956237657,
67381.234635883985902,
67390.269074050956988,
67399.303631442991900,
67408.338308045829763,
67417.373103845297010,
67426.408018827161868,
67435.443052977236221,
67444.478206281317398,
67453.513478725217283,
67462.548870294733206,
67471.584380975662498,
67480.620010753846145,
67489.655759615096031,
67498.691627545224037,
67507.727614530056599,
67516.763720555420150,
67525.799945607141126,
67534.836289671075065,
67543.872752733033849,
67552.909334778887569,
67561.946035794433556,
67570.982855765571003,
67580.019794678126345,
67589.056852517969673,
67598.094029270941974,
67607.131324922898784,
67616.168739459724748,
67625.206272867275402,
67634.243925131420838,
67643.281696238045697,
67652.319586173020070,
67661.357594922243152,
67670.395722471570480,
67679.433968806901248,
67688.472333914149203,
67697.510817779184435,
67706.549420387891587,
67715.588141726213507,
67724.626981780020287,
67733.665940535254776,
67742.705017977801617,
67751.744214093560004,
67760.783528868487338,
67769.822962288482813,
67778.862514339474728,
67787.902185007405933,
67796.941974278175621,
67805.981882137755747,
67815.021908572045504,
67824.062053567045950,
67833.102317108641728,
67842.142699182804790,
67851.183199775477988,
67860.223818872647826,
67869.264556460228050,
67878.305412524219719,
67887.346387050565681,
67896.387480025223340,
67905.428691434208304,
67914.470021263434319,
67923.511469498931547,
67932.553036126657389,
67941.594721132612904,
67950.636524502755492,
67959.678446223129868,
67968.720486279664328,
67977.762644658389036,
67986.804921345319599,
67995.847316326457076,
68004.889829587773420,
68013.932461115327897,
68022.975210895121563,
68032.018078913140926,
68041.061065155459801,
68050.104169608050142,
68059.147392256971216,
68068.190733088253182,
68077.234192087926203,
68086.277769242020440,
68095.321464536595158,
68104.365277957665967,
68113.409209491321235,
68122.453259123562020,
68131.497426840491244,
68140.541712628139067,
68149.586116472564754,
68158.630638359856675,
68167.675278276044992,
68176.720036207232624,
68185.764912139493390,
68194.809906058857450,
68203.855017951485934,
68212.900247803394450,
68221.945595600700472,
68230.991061329492368,
68240.036644975843956,
68249.082346525887260,
68258.128165965696098,
68267.174103281373391,
68276.220158459036611,
68285.266331484788680,
68294.312622344776173,
68303.359031025058357,
68312.405557511810912,
68321.452201791093103,
68330.498963849109714,
68339.545843671934563,
68348.592841245714226,
68357.639956556580728,
68366.687189590695198,
68375.734540334175108,
68384.782008773167036,
68393.829594893861213,
68402.877298682346009,
68411.925120124826208,
68420.973059207448387,
68430.021115916359122,
68439.069290237748646,
68448.117582157763536,
68457.165991662608576,
68466.214518738415791,
68475.263163371389965,
68484.311925547721330,
68493.360805253571016,
68502.409802475129254,
68511.458917198615381,
68520.508149410190526,
68529.557499096074025,
68538.606966242470662,
68547.656550835570670,
68556.706252861578832,
68565.756072306714486,
68574.806009157196968,
68583.856063399231061,
68592.906235019065207,
68601.956524002889637,
68611.006930336938240,
68620.057454007459455,
68629.108095000687172,
68638.158853302840726,
68647.209728900154005,
68656.260721778890002,
68665.311831925326260,
68674.363059325638460,
68683.414403966104146,
68692.465865833029966,
68701.517444912620704,
68710.569141191153903,
68719.620954654907109,
68728.672885290128761,
68737.724933083110955,
68746.777098020131234,
68755.829380087438039,
68764.881779271352571,
68773.934295558123267,
68782.986928934085881,
68792.039679385474301,
68801.092546898624278,
68810.145531459827907,
68819.198633055377286,
68828.251851671579061,
68837.305187294754433,
68846.358639911195496,
68855.412209507208900,
68864.465896069144947,
68873.519699583310285,
68882.573620036011562,
68891.627657413584529,
68900.681811702364939,
68909.736082888703095,
68918.790470958905644,
68927.844975899322890,
68936.899597696290584,
68945.954336336173583,
68955.009191805307637,
68964.064164090057602,
68973.119253176773782,
68982.174459051791928,
68991.229781701491447,
69000.285221112237195,
69009.340777270423132,
69018.396450162385008,
69027.452239774502232,
69036.508146093183313,
69045.564169104778557,
69054.620308795667370,
69063.676565152287367,
69072.732938160959748,
69081.789427808122127,
69090.846034080139361,
69099.902756963449065,
69108.959596444445197,
69118.016552509507164,
69127.073625145101687,
69136.130814337579068,
69145.188120073376922,
69154.245542338932864,
69163.303081120669958,
69172.360736404967611,
69181.418508178292541,
69190.476396427067812,
69199.534401137760142,
69208.592522296763491,
69217.650759890530026,
69226.709113905497361,
69235.767584328132216,
69244.826171144886757,
69253.884874342198600,
69262.943693906534463,
69272.002629824346513,
69281.061682082116022,
69290.120850666280603,
69299.180135563336080,
69308.239536759749171,
69317.299054241986596,
69326.358687996544177,
69335.418438009888632,
69344.478304268501233,
69353.538286758892355,
69362.598385467543267,
69371.658600380949792,
69380.718931485593203,
69389.779378767998423,
69398.839942214646726,
69407.900621812077588,
69416.961417546772282,
69426.022329405255732,
69435.083357374038314,
69444.144501439644955,
69453.205761588629684,
69462.267137807459221,
69471.328630082702148,
69480.390238400868839,
69489.451962748513324,
69498.513803112175083,
69507.575759478379041,
69516.637831833679229,
69525.700020164644229,
69534.762324457798968,
69543.824744699682924,
69552.887280876893783,
69561.949932975985575,
69571.012700983497780,
69580.075584885998978,
69589.138584670072305,
69598.201700322300894,
69607.264931829224224,
69616.328279177469085,
69625.391742353589507,
69634.455321344168624,
69643.519016135804122,
69652.582826715079136,
69661.646753068591352,
69670.710795182967559,
69679.774953044761787,
69688.839226640586276,
69697.903615957096918,
69706.968120980847743,
69716.032741698465543,
69725.097478096591658,
69734.162330161809223,
69743.227297880759579,
69752.292381240098621,
69761.357580226394930,
69770.422894826333504,
69779.488325026497478,
69788.553870813586400,
69797.619532174168853,
69806.685309094973491,
69815.751201562583447,
69824.817209563669167,
69833.883333084886544,
69842.949572112876922,
69852.015926634325297,
69861.082396635902114,
69870.148982104219613,
69879.215683025991893,
69888.282499387889402,
69897.349431176568032,
69906.416478378712782,
69915.483640981023200,
69924.550918970155180,
69933.618312332793721,
69942.685821055682027,
69951.753445125446888,
69960.821184528831509,
69969.889039252520888,
69978.957009283200023,
69988.025094607568462,
69997.093295212398516,
70006.161611084346077,
70015.230042210139800,
70024.298588576493785,
70033.367250170151237,
70042.436026977797155,
70051.504918986189296,
70060.573926182041760,
70069.643048552097753,
70078.712286083100480,
70087.781638761778595,
70096.851106574875303,
70105.920689509133808,
70114.990387551311869,
70124.060200688152690,
70133.130128906414029,
70142.200172192839091,
70151.270330534229288,
70160.340603917313274,
70169.410992328848806,
70178.481495755637297,
70187.552114184451057,
70196.622847602033289,
70205.693695995185408,
70214.764659350708826,
70223.835737655346747,
70232.906930895900587,
70241.978239059200860,
70251.049662131976220,
70260.121200101057184,
70269.192852953274269,
70278.264620675385231,
70287.336503254206036,
70296.408500676567201,
70305.480612929255585,
70314.552839999101707,
70323.625181872921530,
70332.697638537531020,
70341.770209979760693,
70350.842896186441067,
70359.915697144402657,
70368.988612840475980,
70378.061643261477002,
70387.134788394279894,
70396.208048225700622,
70405.281422742613358,
70414.354911931834067,
70423.428515780236921,
70432.502234274666989,
70441.576067401983892,
70450.650015149061801,
70459.724077502745786,
70468.798254449910019,
70477.872545977428672,
70486.946952072161366,
70496.021472720982274,
70505.096107910809224,
70514.170857628458180,
70523.245721860890626,
70532.320700594922528,
70541.395793817500817,
70550.471001515485113,
70559.546323675778694,
70568.621760285284836,
70577.697311330906814,
70586.772976799562457,
70595.848756678140489,
70604.924650953573291,
70614.000659612749587,
70623.076782642601756,
70632.153020030062180,
70641.229371762034134,
70650.305837825449998,
70659.382418207256706,
70668.459112894357531,
70677.535921873713960,
70686.612845132243820,
70695.689882656908594,
70704.767034434640664,
70713.844300452386960,
70722.921680697123520,
70731.999175155753619,
70741.076783815267845,
70750.154506662642234,
70759.232343684809166,
70768.310294868730125,
70777.388360201395699,
70786.466539669767371,
70795.544833260821179,
70804.623240961533156,
70813.701762758908444,
70822.780398639879422,
70831.859148591465782,
70840.938012600643560,
70850.016990654432448,
70859.096082739793928,
70868.175288843747694,
70877.254608953269781,
70886.334043055408983,
70895.413591137141339,
70904.493253185471985,
70913.573029187435168,
70922.652919130021473,
70931.732923000265146,
70940.813040785200428,
70949.893272471832461,
70958.973618047180935,
70968.054077498309198,
70977.134650812222390,
70986.215337975983857,
70995.296138976598741,
71004.377053801115835,
71013.458082436627592,
71022.539224870139151,
71031.620481088699307,
71040.701851079385960,
71049.783334829233354,
71058.864932325304835,
71067.946643554692855,
71077.028468504431657,
71086.110407161599142,
71095.192459513287758,
71104.274625546531752,
71113.356905248452676,
71122.439298606099328,
71131.521805606564158,
71140.604426236925065,
71149.687160484303604,
71158.770008335763123,
71167.852969778410625,
71176.936044799338561,
71186.019233385639382,
71195.102535524449195,
71204.185951202860451,
71213.269480407965602,
71222.353123126900755,
71231.436879346758360,
71240.520749054689077,
71249.604732237785356,
71258.688828883183305,
71267.773038978033583,
71276.857362509414088,
71285.941799464490032,
71295.026349830412073,
71304.111013594301767,
71313.195790743309772,
71322.280681264572195,
71331.365685145225143,
71340.450802372448379,
71349.536032933378010,
71358.621376815193798,
71367.706834005017299,
71376.792404490042827,
71385.878088257406489,
71394.963885294331703,
71404.049795587910921,
71413.135819125382113,
71422.221955893895938,
71431.308205880646710,
71440.394569072799641,
71449.481045457549044,
71458.567635022089235,
71467.654337753614527,
71476.741153639304684,
71485.828082666368573,
71494.915124821985955,
71504.002280093394802,
71513.089548467774875,
71522.176929932364146,
71531.264424474342377,
71540.352032080947538,
71549.439752739359392,
71558.527586436830461,
71567.615533160613268,
71576.703592897873023,
71585.791765635876800,
71594.880051361848018,
71603.968450063024648,
71613.056961726630107,
71622.145586339916917,
71631.234323890137603,
71640.323174364530132,
71649.412137750332477,
71658.501214034826262,
71667.590403205234907,
71676.679705248825485,
71685.769120152879623,
71694.858647904635291,
71703.948288491374115,
71713.038041900363169,
71722.127908118869527,
71731.217887134160264,
71740.307978933546110,
71749.398183504294138,
71758.488500833671424,
71767.578930908959592,
71776.669473717483925,
71785.760129246526049,
71794.850897483367589,
71803.941778415290173,
71813.032772029648186,
71822.123878313694149,
71831.215097254753346,
71840.306428840165609,
71849.397873057183460,
71858.489429893175839,
71867.581099335424369,
71876.672881371268886,
71885.764775988049223,
71894.856783173061558,
71903.948902913660277,
71913.041135197141557,
71922.133480010888889,
71931.225937342198449,
71940.318507178453729,
71949.411189506965457,
71958.503984315088019,
71967.596891590161249,
71976.689911319554085,
71985.783043490635464,
71994.876288090730668,
72003.969645107237739,
72013.063114527452854,
72022.156696338832262,
72031.250390528701246,
72040.344197084414191,
72049.438115993369138,
72058.532147242949577,
72067.626290820524446,
72076.720546713477233,
72085.814914909191430,
72094.909395395079628,
72104.003988158525317,
72113.098693186897435,
72122.193510467623128,
72131.288439988085884,
72140.383481735683745,
72149.478635697843856,
72158.573901861978811,
72167.669280215486651,
72176.764770745779970,
72185.860373440285912,
72194.956088286431623,
72204.051915271615144,
72213.147854383278172,
72222.243905608862406,
72231.340068935765885,
72240.436344351459411,
72249.532731843355577,
72258.629231398910633,
72267.725843005566276,
72276.822566650749650,
72285.919402321931557,
72295.016350006553694,
72304.113409692057758,
72313.210581365929102,
72322.307865015623975,
72331.405260628584074,
72340.502768192280200,
72349.600387694183155,
72358.698119121807395,
72367.795962462536409,
72376.893917703928309,
72385.991984833410243,
72395.090163838496665,
72404.188454706672928,
72413.286857425409835,
72422.385371982192737,
72431.483998364536092,
72440.582736559939804,
72449.681586555874674,
72458.780548339884263,
72467.879621899424819,
72476.978807222039904,
72486.078104295214871,
72495.177513106493279,
72504.277033643345931,
72513.376665893360041,
72522.476409843977308,
72531.576265482784947,
72540.676232797268312,
72549.776311774985516,
72558.876502403480117,
72567.976804670222919,
72577.077218562830240,
72586.177744068816537,
72595.278381175681716,
72604.379129871042096,
72613.479990142397583,
72622.580961977320840,
72631.682045363340876,
72640.783240288074012,
72649.884546739020152,
72658.985964703781065,
72668.087494169900310,
72677.189135124936001,
72686.290887556504458,
72695.392751452149241,
72704.494726799428463,
72713.596813585987547,
72722.699011799340951,
72731.801321427104995,
72740.903742456852342,
72750.006274876199313,
72759.108918672733125,
72768.211673834011890,
72777.314540347681032,
72786.417518201327766,
72795.520607382568414,
72804.623807878975640,
72813.727119678194867,
72822.830542767827865,
72831.934077135490952,
72841.037722768785898,
72850.141479655372677,
72859.245347782823956,
72868.349327138814260,
72877.453417710945359,
72886.557619486862677,
72895.661932454197085,
72904.766356600564905,
72913.870891913669766,
72922.975538381069782,
72932.080295990468585,
72941.185164729497046,
72950.290144585829694,
72959.395235547082848,
72968.500437600916484,
72977.605750735048787,
72986.711174937066971,
72995.816710194689222,
73004.922356495531858,
73014.028113827327616,
73023.133982177707367,
73032.239961534360191,
73041.346051884975168,
73050.452253217197722,
73059.558565518746036,
73068.664988777294639,
73077.771522980561713,
73086.878168116192683,
73095.984924171920284,
73105.091791135404492,
73114.198768994392594,
73123.305857736573671,
73132.413057349607698,
73141.520367821271066,
73150.627789139252855,
73159.735321291242144,
73168.842964264971670,
73177.950718048174167,
73187.058582628553268,
73196.166557993856259,
73205.274644131801324,
73214.382841030106647,
73223.491148676504963,
73232.599567058772664,
73241.708096164584276,
73250.816735981745296,
73259.925486497973907,
73269.034347700988292,
73278.143319578564842,
73287.252402118479949,
73296.361595308451797,
73305.470899136242224,
73314.580313589642174,
73323.689838656369830,
73332.799474324230687,
73341.909220580957481,
73351.019077414355706,
73360.129044812201755,
73369.239122762228362,
73378.349311252255575,
73387.459610270045232,
73396.570019803402829,
73405.680539840090205,
73414.791170367912855,
73423.901911374676274,
73433.012762848142302,
73442.123724776145536,
73451.234797146476922,
73460.345979946898296,
73469.457273165273364,
73478.568676789393066,
73487.680190807062900,
73496.791815206102910,
73505.903549974318594,
73515.015395099544548,
73524.127350569615373,
73533.239416372322012,
73542.351592495513614,
73551.463878927010228,
73560.576275654661004,
73569.688782666271436,
73578.801399949734332,
73587.914127492840635,
73597.026965283454047,
73606.139913309423719,
73615.252971558584250,
73624.366140018813894,
73633.479418677947251,
73642.592807523848023,
73651.706306544365361,
73660.819915727377520,
73669.933635060762754,
73679.047464532355662,
73688.161404130034498,
73697.275453841706621,
73706.389613655206631,
73715.503883558427333,
73724.618263539261534,
73733.732753585587488,
73742.847353685268899,
73751.962063826242229,
73761.076883996356628,
73770.191814183519455,
73779.306854375652620,
73788.422004560619825,
73797.537264726328431,
73806.652634860729449,
73815.768114951686584,
73824.883704987121746,
73833.999404954942293,
73843.115214843055583,
73852.231134639427182,
73861.347164331920794,
73870.463303908501985,
73879.579553357078112,
73888.695912665585638,
73897.812381821961026,
73906.928960814126185,
73916.045649630017579,
73925.162448257557116,
73934.279356684754021,
73943.396374899486545,
73952.513502889749361,
73961.630740643464378,
73970.748088148597162,
73979.865545393113280,
73988.983112364920089,
73998.100789052041364,
74007.218575442399015,
74016.336471523987711,
74025.454477284743916,
74034.572592712691403,
74043.690817795752082,
74052.809152521949727,
74061.927596879220800,
74071.046150855559972,
74080.164814438947360,
74089.283587617392186,
74098.402470378903672,
74107.521462711418280,
74116.640564602959785,
74125.759776041508303,
74134.879097015116713,
74143.998527511721477,
74153.118067519375472,
74162.237717026073369,
74171.357476019824389,
74180.477344488637755,
74189.597322420522687,
74198.717409803517512,
74207.837606625646004,
74216.957912874917383,
74226.078328539369977,
74235.198853607013007,
74244.319488065913902,
74253.440231904081884,
74262.561085109555279,
74271.682047670372413,
74280.803119574586162,
74289.924300810249406,
74299.045591365385917,
74308.166991228077677,
74317.288500386348460,
74326.410118828265695,
74335.531846541867708,
74344.653683515265584,
74353.775629736483097,
74362.897685193573125,
74372.019849874646752,
74381.142123767756857,
74390.264506860970869,
74399.386999142385321,
74408.509600600053091,
74417.632311222070712,
74426.755130996520165,
74435.878059911468881,
74445.001097955057048,
74454.124245115322992,
74463.247501380392350,
74472.370866738361656,
74481.494341177312890,
74490.617924685342587,
74499.741617250605486,
74508.865418861169019,
74517.989329505144269,
74527.113349170642323,
74536.237477845803369,
74545.361715518738492,
74554.486062177544227,
74563.610517810360761,
74572.735082405313733,
74581.859755950543331,
74590.984538434146089,
74600.109429844305851,
74609.234430169104598,
74618.359539396711625,
74627.484757515296224,
74636.610084512940375,
74645.735520377827925,
74654.861065098084509,
74663.986718661893974,
74673.112481057396508,
74682.238352272761404,
74691.364332296114299,
74700.490421115639037,
74709.616618719490361,
74718.742925095866667,
74727.869340232908144,
74736.995864118784084,
74746.122496741692885,
74755.249238089789287,
74764.376088151271688,
74773.503046914294828,
74782.630114367086207,
74791.757290497800568,
74800.884575294650858,
74810.011968745791819,
74819.139470839450951,
74828.267081563826650,
74837.394800907117315,
74846.522628857521340,
74855.650565403251676,
74864.778610532506718,
74873.906764233470312,
74883.035026494428166,
74892.163397303520469,
74901.291876649047481,
74910.420464519120287,
74919.549160902068252,
74928.677965786075220,
74937.806879159339587,
74946.935901010147063,
74956.065031326681492,
74965.194270097243134,
74974.323617310001282,
74983.453072953227093,
74992.582637015177170,
75001.712309484093566,
75010.842090348218335,
75019.971979595793528,
75029.101977215090301,
75038.232083194350707,
75047.362297521845903,
75056.492620185832493,
75065.623051174567081,
75074.753590476364479,
75083.884238079423085,
75093.014993972057709,
75102.145858142539510,
75111.276830579125090,
75120.407911270114710,
75129.539100203764974,
75138.670397368419799,
75147.801802752306685,
75156.933316343725892,
75166.064938131006784,
75175.196668102391413,
75184.328506246209145,
75193.460452550760238,
75202.592507004344952,
75211.724669595248997,
75220.856940311816288,
75229.989319142303430,
75239.121806075097993,
75248.254401098442031,
75257.387104200694012,
75266.519915370154195,
75275.652834595151944,
75284.785861864031176,
75293.918997165106703,
75303.052240486693336,
75312.185591817134991,
75321.319051144790137,
75330.452618457944482,
75339.586293745014700,
75348.720076994242845,
75357.853968194060144,
75366.987967332766857,
75376.122074398765108,
75385.256289380340604,
75394.390612265895470,
75403.525043043773621,
75412.659581702333526,
75421.794228229919099,
75430.928982614947017,
75440.063844845761196,
75449.198814910705551,
75458.333892798196757,
75467.469078496578732,
75476.604371994209941,
75485.739773279536166,
75494.875282340886770,
75504.010899166678428,
75513.146623745255056,
75522.282456065062433,
75531.418396114444477,
75540.554443881846964,
75549.690599355642917,
75558.826862524219905,
75567.963233375980053,
75577.099711899369140,
75586.236298082760186,
75595.372991914555314,
75604.509793383200304,
75613.646702477111830,
75622.783719184648362,
75631.920843494299334,
75641.058075394466869,
75650.195414873553091,
75659.332861920003779,
75668.470416522250162,
75677.608078668738017,
75686.745848347884021,
75695.883725548119401,
75705.021710257889936,
75714.159802465655957,
75723.298002159863245,
75732.436309328943025,
75741.574723961355630,
75750.713246045532287,
75759.851875569962431,
75768.990612523077289,
75778.129456893322640,
75787.268408669231576,
75796.407467839206220,
75805.546634391721454,
75814.685908315266715,
75823.825289598302334,
75832.964778229303192,
75842.104374196758727,
75851.244077489129268,
75860.383888094933354,
75869.523806002602214,
75878.663831200668938,
75887.803963677593856,
75896.944203421895509,
75906.084550422048778,
75915.225004666557652,
75924.365566143911565,
75933.506234842658159,
75942.647010751257767,
75951.787893858199823,
75960.928884152046521,
75970.069981621272746,
75979.211186254426138,
75988.352498039996135,
75997.493916966501274,
76006.635443022474647,
76015.777076196449343,
76024.918816476929351,
76034.060663852447760,
76043.202618311566766,
76052.344679842790356,
76061.486848434666172,
76070.629124075727304,
76079.771506754506845,
76088.913996459596092,
76098.056593179469928,
76107.199296902748756,
76116.342107617922011,
76125.485025313580991,
76134.628049978258787,
76143.771181600546697,
76152.914420168977813,
76162.057765672143432,
76171.201218098562094,
76180.344777436868753,
76189.488443675552844,
76198.632216803263873,
76207.776096808534930,
76216.920083679971867,
76226.064177406122326,
76235.208377975592157,
76244.352685376972659,
76253.497099598840578,
76262.641620629758108,
76271.786248458374757,
76280.930983073252719,
76290.075824462983292,
76299.220772616201430,
76308.365827521483880,
76317.510989167436492,
76326.656257542665116,
76335.801632635790156,
76344.947114435417461,
76354.092702930167434,
76363.238398108645924,
76372.384199959502439,
76381.530108471313724,
76390.676123632714734,
76399.822245432384079,
76408.968473858883954,
76418.114808900878415,
76427.261250547016971,
76436.407798785890918,
76445.554453606193420,
76454.701214996530325,
76463.848082945536589,
76472.995057441905374,
76482.142138474242529,
76491.289326031212113,
76500.436620101463632,
76509.584020673675695,
76518.731527736483258,
76527.879141278535826,
76537.026861288526561,
76546.174687755119521,
76555.322620666978764,
76564.470660012753797,
76573.618805781123228,
76582.767057960794773,
76591.915416540403385,
76601.063881508671329,
76610.212452854262665,
76619.361130565841449,
76628.509914632115397,
76637.658805041763117,
76646.807801783506875,
76655.956904846010730,
76665.106114217982395,
76674.255429888115032,
76683.404851845145458,
76692.554380077708629,
76701.704014574555913,
76710.853755324409576,
76720.003602315962780,
76729.153555537923239,
76738.303614979027770,
76747.453780627954984,
76756.604052473470801,
76765.754430504268385,
76774.904914709099103,
76784.055505076656118,
76793.206201595705352,
76802.357004254969070,
76811.507913043169538,
76820.658927949072677,
76829.810048961386201,
76838.961276068861480,
76848.112609260264435,
76857.264048524331884,
76866.415593849786092,
76875.567245225422084,
76884.719002639962127,
76893.870866082172142,
76903.022835540832602,
76912.174911004665773,
76921.327092462466680,
76930.479379903001245,
76939.631773315006285,
76948.784272687305929,
76957.936878008636995,
76967.089589267779957,
76976.242406453500735,
76985.395329554623459,
76994.548358559870394,
77003.701493458080222,
77012.854734238018864,
77022.008080888466793,
77031.161533398233587,
77040.315091756114271,
77049.468755950903869,
77058.622525971382856,
77067.776401806389913,
77076.930383444705512,
77086.084470875110128,
77095.238664086456993,
77104.392963067555684,
77113.547367807201226,
77122.701878294217749,
77131.856494517414831,
77141.011216465645703,
77150.166044127676287,
77159.320977492374368,
77168.476016548564075,
77177.631161285069538,
77186.786411690729437,
77195.941767754397006,
77205.097229464852717,
77214.252796810993459,
77223.408469781614258,
77232.564248365597450,
77241.720132551781717,
77250.876122329005739,
77260.032217686137301,
77269.188418612000532,
77278.344725095477770,
77287.501137125436799,
77296.657654690716299,
77305.814277780184057,
77314.971006382722408,
77324.127840487170033,
77333.284780082423822,
77342.441825157351559,
77351.598975700791925,
77360.756231701685465,
77369.913593148870859,
77379.071060031244997,
77388.228632337675663,
77397.386310057074297,
77406.544093178308685,
77415.701981690290268,
77424.859975581872277,
77434.018074842009810,
77443.176279459570651,
77452.334589423451689,
77461.493004722549813,
77470.651525345805567,
77479.810151282101288,
77488.968882520333864,
77498.127719049443840,
77507.286660858328105,
77516.445707935941755,
77525.604860271138023,
77534.764117852886557,
77543.923480670098797,
77553.082948711686186,
77562.242521966618369,
77571.402200423792237,
77580.561984072148334,
77589.721872900641756,
77598.881866898183944,
77608.041966053715441,
77617.202170356191345,
77626.362479794566752,
77635.522894357767655,
77644.683414034763700,
77653.844038814495434,
77663.004768685932504,
77672.165603637971799,
77681.326543659670278,
77690.487588739924831,
77699.648738867705106,
77708.809994032009854,
77717.971354221750516,
77727.132819425940397,
77736.294389633534593,
77745.456064833546407,
77754.617845014887280,
77763.779730166614172,
77772.941720277638524,
77782.103815336988191,
77791.266015333647374,
77800.428320256585721,
77809.590730094801984,
77818.753244837309467,
77827.915864473063266,
77837.078588991091237,
77846.241418380392133,
77855.404352629964706,
77864.567391728822258,
77873.730535665978095,
77882.893784430416417,
77892.057138011165080,
77901.220596397251938,
77910.384159577646642,
77919.547827541449806,
77928.711600277616526,
77937.875477775174659,
77947.039460023166612,
77956.203547010620241,
77965.367738726577954,
77974.532035160053056,
77983.696436300087953,
77992.860942135710502,
78002.025552655992215,
78011.190267849946395,
78020.355087706615450,
78029.520012215056340,
78038.685041364326025,
78047.850175143452361,
78057.015413541521411,
78066.180756547575584,
78075.346204150657286,
78084.511756339838030,
78093.677413104189327,
78102.843174432768137,
78112.009040314645972,
78121.175010738894343,
78130.341085694555659,
78139.507265170759638,
78148.673549156519584,
78157.839937640979770,
78167.006430613182602,
78176.173028062214144,
78185.339729977145907,
78194.506536347107613,
78203.673447161156218,
78212.840462408392341,
78222.007582077902043,
78231.174806158815045,
78240.342134640217409,
78249.509567511166097,
78258.677104760805378,
78267.844746378250420,
78277.012492352587287,
78286.180342672960251,
78295.348297328426270,
78304.516356308158720,
78313.684519601229113,
78322.852787196781719,
78332.021159083946259,
78341.189635251808795,
78350.358215689542703,
78359.526900386263151,
78368.695689331056201,
78377.864582513124333,
78387.033579921582714,
78396.202681545531959,
78405.371887374159996,
78414.541197396567441,
78423.710611601927667,
78432.880129979384947,
78442.049752518098103,
78451.219479207182303,
78460.389310035825474,
78469.559244993157336,
78478.729284068365814,
78487.899427250580629,
78497.069674528989708,
78506.240025892751873,
78515.410481331025949,
78524.581040832999861,
78533.751704387817881,
78542.922471984682488,
78552.093343612752506,
78561.264319261186756,
78570.435398919216823,
78579.606582576001529,
78588.777870220714249,
78597.949261842528358,
78607.120757430689991,
78616.292356974357972,
78625.464060462705675,
78634.635867884964682,
78643.807779230337474,
78652.979794487982872,
78662.151913647147012,
78671.324136697003269,
78680.496463626768673,
78689.668894425689359,
78698.841429082924151,
78708.014067587719182,
78717.186809929276933,
78726.359656096828985,
78735.532606079592369,
78744.705659866769565,
78753.878817447621259,
78763.052078811364481,
78772.225443947201711,
78781.398912844422739,
78790.572485492186388,
78799.746161879797000,
78808.919941996457055,
78818.093825831412687,
78827.267813373939134,
78836.441904613238876,
78845.616099538587150,
78854.790398139200988,
78863.964800404370180,
78873.139306323355413,
78882.313915885359165,
78891.488629079700331,
78900.663445895595942,
78909.838366322350339,
78919.013390349209658,
78928.188517965420033,
78937.363749160285806,
78946.539083923067665,
78955.714522243026295,
78964.890064109451487,
78974.065709511647583,
78983.241458438846166,
78992.417310880351579,
79001.593266825482715,
79010.769326263471157,
79019.945489183635800,
79029.121755575295538,
79038.298125427711057,
79047.474598730172147,
79056.651175471997703,
79065.827855642462964,
79075.004639230915927,
79084.181526226646383,
79093.358516618929571,
79102.535610397113487,
79111.712807550502475,
79120.890108068400878,
79130.067511940127588,
79139.245019155016053,
79148.422629702356062,
79157.600343571495614,
79166.778160751782707,
79175.956081232492579,
79185.134105002987781,
79194.312232052601757,
79203.490462370667956,
79212.668795946505270,
79221.847232769476250,
79231.025772828899790,
79240.204416114138439,
79249.383162614511093,
79258.562012319409405,
79267.740965218137717,
79276.920021300073131,
79286.099180554563645,
79295.278442970971810,
79304.457808538631070,
79313.637277246962185,
79322.816849085240392,
79331.996524042915553,
79341.176302109335666,
79350.356183273805073,
79359.536167525773635,
79368.716254854574800,
79377.896445249600220,
79387.076738700212445,
79396.257135195832234,
79405.437634725807584,
79414.618237279501045,
79423.798942846347927,
79432.979751415725332,
79442.160662977010361,
79451.341677519594668,
79460.522795032899012,
79469.704015506300493,
79478.885338929219870,
79488.066765291005140,
79497.248294581149821,
79506.429926788987359,
79515.611661903953063,
79524.793499915453140,
79533.975440812908346,
79543.157484585739439,
79552.339631223323522,
79561.521880715139559,
79570.704233050593757,
79579.886688219063217,
79589.069246210041456,
79598.251907012905576,
79607.434670617105439,
79616.617537012061803,
79625.800506187268184,
79634.983578132072580,
79644.166752835983061,
79653.350030288405833,
79662.533410478805308,
79671.716893396616797,
79680.900479031275609,
79690.084167372260708,
79699.267958408992854,
79708.451852130951011,
79717.635848527614144,
79726.819947588359355,
79736.004149302738369,
79745.188453660171945,
79754.372860650109942,
79763.557370262060431,
79772.741982485458720,
79781.926697309798328,
79791.111514724543667,
79800.296434719173703,
79809.481457283167401,
79818.666582405989175,
79827.851810077147093,
79837.037140286134672,
79846.222573022401775,
79855.408108275456470,
79864.593746034792275,
79873.779486289902707,
79882.965329030266730,
79892.151274245421519,
79901.337321924831485,
79910.523472057990148,
79919.709724634449231,
79928.896079643673147,
79938.082537075184518,
79947.269096918491414,
79956.455759163131006,
79965.642523798567709,
79974.829390814353246,
79984.016360200010240,
79993.203431945046759,
80002.390606038999977,
80011.577882471377961,
80020.765261231703334,
80029.952742309527821,
80039.140325694403145,
80048.328011375793722,
80057.515799343309482,
80066.703689586443943,
80075.891682094748830,
80085.079776857775869,
80094.267973865091335,
80103.456273106188746,
80112.644674570634379,
80121.833178247994510,
80131.021784127820865,
80140.210492199665168,
80149.399302453122800,
80158.588214877672726,
80167.777229462924879,
80176.966346198460087,
80186.155565073815524,
80195.344886078557465,
80204.534309202266741,
80213.723834434553282,
80222.913461764925160,
80232.103191182977753,
80241.293022678350098,
80250.482956240535714,
80259.672991859173635,
80268.863129523830139,
80278.053369224100607,
80287.243710949565866,
80296.434154689835850,
80305.624700434462284,
80314.815348173113307,
80324.006097895311541,
80333.196949590710574,
80342.387903248891234,
80351.578958859448903,
80360.770116412008065,
80369.961375896178652,
80379.152737301556044,
80388.344200617764727,
80397.535765834400081,
80406.727432941115694,
80415.919201927506947,
80425.111072783212876,
80434.303045497828862,
80443.495120060993941,
80452.687296462332597,
80461.879574691498419,
80471.071954738101340,
80480.264436591780395,
80489.457020242145518,
80498.649705678893952,
80507.842492891635629,
80517.035381869980483,
80526.228372603625758,
80535.421465082181385,
80544.614659295344609,
80553.807955232696258,
80563.001352883948130,
80572.194852238739259,
80581.388453286708682,
80590.582156017524539,
80599.775960420854972,
80608.969866486368119,
80618.163874203717569,
80627.357983562600566,
80636.552194552641595,
80645.746507163537899,
80654.940921384972171,
80664.135437206597999,
80673.330054618098075,
80682.524773609169642,
80691.719594169495394,
80700.914516288743471,
80710.109539956611115,
80719.304665162766469,
80728.499891896950430,
80737.695220148787485,
80746.890649908047635,
80756.086181164369918,
80765.281813907451578,
80774.477548127048067,
80783.673383812827524,
80792.869320954487193,
80802.065359541753423,
80811.261499564323458,
80820.457741011938197,
80829.654083874265780,
80838.850528141061659,
80848.047073802008526,
80857.243720846847282,
80866.440469265304273,
80875.637319047091296,
80884.834270181963802,
80894.031322659604484,
80903.228476469768793,
80912.425731602197629,
80921.623088046602788,
80930.820545792768826,
80940.018104830378434,
80949.215765149201616,
80958.413526738950168,
80967.611389589423197,
80976.809353690332500,
80986.007419031433528,
80995.205585602496285,
81004.403853393232566,
81013.602222393412376,
81022.800692592834821,
81031.999263981197146,
81041.197936548313010,
81050.396710283908760,
81059.595585177768953,
81068.794561219678144,
81077.993638399377232,
81087.192816706650774,
81096.392096131268772,
81105.591476663001231,
81114.790958291632705,
81123.990541006976855,
81133.190224798760028,
81142.390009656795883,
81151.589895570854424,
81160.789882530763862,
81169.989970526279649,
81179.190159547186340,
81188.390449583312147,
81197.590840624427074,
81206.791332660373882,
81215.991925680878921,
81225.192619675799506,
81234.393414634934743,
81243.594310548069188,
81252.795307405060157,
81261.996405195663101,
81271.197603909720783,
81280.398903537046863,
81289.600304067440447,
81298.801805490773404,
81308.003407796801184,
81317.205110975381103,
81326.406915016326820,
81335.608819909481099,
81344.810825644672150,
81354.012932211728184,
81363.215139600477414,
81372.417447800748050,
81381.619856802397408,
81390.822366595268250,
81400.024977169174235,
81409.227688513987232,
81418.430500619535451,
81427.633413475676207,
81436.836427072266815,
81446.039541399164591,
81455.242756446197745,
81464.446072203238145,
81473.649488660128554,
81482.853005806755391,
81492.056623632975970,
81501.260342128647608,
81510.464161283642170,
81519.668081087802420,
81528.872101531043882,
81538.076222603194765,
81547.280444294170593,
81556.484766593828681,
81565.689189492026344,
81574.893712978679105,
81584.098337043673382,
81593.303061676851939,
81602.507886868130299,
81611.712812607423984,
81620.917838884575758,
81630.122965689501143,
81639.328193012086558,
81648.533520842247526,
81657.738949169841362,
81666.944477984841797,
81676.150107277077041,
81685.355837036506273,
81694.561667252986808,
81703.767597916477825,
81712.973629016880295,
81722.179760544066085,
81731.385992487994372,
81740.592324838566128,
81749.798757585740532,
81759.005290719360346,
81768.211924229399301,
81777.418658105787472,
81786.625492338440381,
81795.832426917273551,
81805.039461832246161,
81814.246597073273733,
81823.453832630286342,
81832.661168493257719,
81841.868604652074282,
81851.076141096738866,
81860.283777817123337,
81869.491514803230530,
81878.699352044990519,
81887.907289532347932,
81897.115327255261946,
81906.323465203677188,
81915.531703367567388,
81924.740041736862622,
81933.948480301551172,
81943.157019051577663,
81952.365657976915827,
81961.574397067539394,
81970.783236313392990,
81979.992175704450347,
81989.201215230685193,
81998.410354882085812,
82007.619594648625934,
82016.828934520264738,
82026.038374487005058,
82035.247914538791520,
82044.457554665656062,
82053.667294857572415,
82062.877135104499757,
82072.087075396455475,
82081.297115723427851,
82090.507256075405166,
82099.717496442375705,
82108.927836814356851,
82118.138277181336889,
82127.348817533304100,
82136.559457860290422,
82145.770198152269586,
82154.981038399302633,
82164.191978591348743,
82173.403018718454405,
82182.614158770578797,
82191.825398737797514,
82201.036738610113389,
82210.248178377529257,
82219.459718030091608,
82228.671357557803276,
82237.883096950696199,
82247.094936198787764,
82256.306875292124460,
82265.518914220738225,
82274.731052974646445,
82283.943291543910163,
82293.155629918546765,
82302.368068088602740,
82311.580606044124579,
82320.793243775158771,
82330.005981271737255,
82339.218818523891969,
82348.431755521727609,
82357.644792255247012,
82366.857928714540321,
82376.071164889610372,
82385.284500770590967,
82394.497936347455834,
82403.711471610338776,
82412.925106549257180,
82422.138841154301190,
82431.352675415531849,
82440.566609323010198,
82449.780642866826383,
82458.994776037055999,
82468.209008823745535,
82477.423341216985136,
82486.637773206850397,
82495.852304783460568,
82505.066935936862137,
82514.281666657145252,
82523.496496934385505,
82532.711426758673042,
82541.926456120156217,
82551.141585008866969,
82560.356813414909993,
82569.572141328404541,
82578.787568739411654,
82588.003095638094237,
82597.218722014498780,
82606.434447858744534,
82615.650273160950746,
82624.866197911222116,
82634.082222099663340,
82643.298345716379117,
82652.514568751532352,
82661.730891195169534,
82670.947313037482672,
82680.163834268518258,
82689.380454878468299,
82698.597174857393838,
82707.813994195472333,
82717.030912882808479,
82726.247930909550632,
82735.465048265803489,
82744.682264941715403,
82753.899580927420175,
82763.116996213066159,
82772.334510788787156,
82781.552124644731521,
82790.769837771047605,
82799.987650157854659,
82809.205561795344693,
82818.423572673607850,
82827.641682782850694,
82836.859892113207025,
82846.078200654825196,
82855.296608397897216,
82864.515115332513233,
82873.733721448938013,
82882.952426737232599,
82892.171231187603553,
82901.390134790257434,
82910.609137535313494,
82919.828239412963740,
82929.047440413371078,
82938.266740526712965,
82947.486139743181411,
82956.705638052939321,
82965.925235446178704,
82975.144931913062464,
82984.364727443826268,
82993.584622028603917,
83002.804615657601971,
83012.024708321026992,
83021.244900009041885,
83030.465190711882315,
83039.685580419711187,
83048.906069122749614,
83058.126656811160501,
83067.347343475194066,
83076.568129105042317,
83085.789013690882712,
83095.009997222965467,
83104.231079691482591,
83113.452261086640647,
83122.673541398660745,
83131.894920617734897,
83141.116398734127870,
83150.337975738031673,
83159.559651619667420,
83168.781426369270775,
83178.003299977048300,
83187.225272433264763,
83196.447343728097621,
83205.669513851840748,
83214.891782794686151,
83224.114150546854944,
83233.336617098641000,
83242.559182440236327,
83251.781846561905695,
83261.004609453884768,
83270.227471106423764,
83279.450431509758346,
83288.673490654124180,
83297.896648529815138,
83307.119905127037782,
83316.343260436115088,
83325.566714447239065,
83334.790267150674481,
83344.013918536715209,
83353.237668595611467,
83362.461517317598918,
83371.685464692985988,
83380.909510712008341,
83390.133655364974402,
83399.357898642105283,
83408.582240533709410,
83417.806681030080654,
83427.031220121454680,
83436.255857798125362,
83445.480594050386571,
83454.705428868517629,
83463.930362242812407,
83473.155394163521123,
83482.380524620952201,
83491.605753605443169,
83500.831081107215141,
83510.056507116634748,
83519.282031623923103,
83528.507654619432287,
83537.733376093427069,
83546.959196036244975,
83556.185114438208984,
83565.411131289569312,
83574.637246580648934,
83583.863460301785381,
83593.089772443272523,
83602.316182995433337,
83611.542691948561696,
83620.769299293009681,
83629.996005019071163,
83639.222809117083671,
83648.449711577355629,
83657.676712390210014,
83666.903811546013458,
83676.131009035059833,
83685.358304847686668,
83694.585698974231491,
83703.813191405002726,
83713.040782130396110,
83722.268471140720067,
83731.496258426283021,
83740.724143977480708,
83749.952127784636104,
83759.180209838072187,
83768.408390128184692,
83777.636668645296595,
83786.865045379759977,
83796.093520321941469,
83805.322093462164048,
83814.550764790823450,
83823.779534298271756,
83833.008401974875596,
83842.237367810957949,
83851.466431796943652,
83860.695593923141132,
83869.924854179975227,
83879.154212557798019,
83888.383669046961586,
83897.613223637861665,
83906.842876320864889,
83916.072627086381544,
83925.302475924734608,
83934.532422826348920,
83943.762467781591113,
83952.992610780871473,
83962.222851814556634,
83971.453190873027779,
83980.683627946724300,
83989.914163025983726,
83999.144796101216343,
84008.375527162846993,
84017.606356201242306,
84026.837283206841676,
84036.068308170011733,
84045.299431081148214,
84054.530651930719614,
84063.761970709078014,
84072.993387406648253,
84082.224902013840619,
84091.456514521094505,
84100.688224918805645,
84109.920033197427983,
84119.151939347313601,
84128.383943358901888,
84137.616045222675893,
84146.848244929002249,
84156.080542468334897,
84165.312937831098679,
84174.545431007703883,
84183.778021988619003,
84193.010710764239775,
84202.243497325063800,
84211.476381661472260,
84220.709363763919100,
84229.942443622858264,
84239.175621228758246,
84248.408896572000231,
84257.642269643096370,
84266.875740432471503,
84276.109308930565021,
84285.342975127830869,
84294.576739014766645,
84303.810600581782637,
84313.044559819361893,
84322.278616717943805,
84331.512771268011420,
84340.747023460047785,
84349.981373284477741,
84359.215820731798885,
84368.450365792479715,
84377.685008456974174,
84386.919748715779861,
84396.154586559350719,
84405.389521978184348,
84414.624554962749244,
84423.859685503543005,
84433.094913591005025,
84442.330239215691108,
84451.565662368040648,
84460.801183038522140,
84470.036801217691391,
84479.272516896002344,
84488.508330063938047,
84497.744240712025203,
84506.980248830746859,
84516.216354410629719,
84525.452557442127727,
84534.688857915782137,
84543.925255822061445,
84553.161751151506905,
84562.398343894645222,
84571.635034041944891,
84580.871821583932615,
84590.108706511120545,
84599.345688814064488,
84608.582768483218388,
84617.819945509152603,
84627.057219882364734,
84636.294591593410587,
84645.532060632744106,
84654.769626990979305,
84664.007290658613783,
84673.245051626174245,
84682.482909884187393,
84691.720865423179930,
84700.958918233736767,
84710.197068306340952,
84719.435315631577396,
84728.673660199972801,
84737.912102002039319,
84747.150641028361861,
84756.389277269510785,
84765.628010715983692,
84774.866841358350939,
84784.105769187197438,
84793.344794193020789,
84802.583916366435005,
84811.823135697966791,
84821.062452178171952,
84830.301865797635401,
84839.541376546912943,
84848.780984416560386,
84858.020689397177193,
84867.260491479304619,
84876.500390653513023,
84885.740386910401867,
84894.980480240526958,
84904.220670634473208,
84913.460958082796424,
84922.701342576139723,
84931.941824105015257,
84941.182402660051594,
84950.423078231833642,
84959.663850810946315,
84968.904720387945417,
84978.145686953474069,
84987.386750498088077,
84996.627911012386903,
85005.869168487013667,
85015.110522912509623,
85024.351974279517890,
85033.593522578608827,
85042.835167800411000,
85052.076909935523872,
85061.318748974576010,
85070.560684908137773,
85079.802717726837727,
85089.044847421304439,
85098.287073982151924,
85107.529397399979644,
85116.771817665401613,
85126.014334769090055,
85135.256948701615329,
85144.499659453620552,
85153.742467015734292,
85162.985371378585114,
85172.228372532787034,
85181.471470469012274,
85190.714665177860297,
85199.957956649959669,
85209.201344875997165,
85218.444829846557695,
85227.688411552328034,
85236.932089983907645,
85246.175865131968749,
85255.419736987154465,
85264.663705540107912,
85273.907770781501313,
85283.151932701948681,
85292.396191292122239,
85301.640546542679658,
85310.884998444293160,
85320.129546987591311,
85329.374192163260886,
85338.618933961945004,
85347.863772374344990,
85357.108707391074859,
85366.353739002850489,
85375.598867200314999,
85384.844091974126059,
85394.089413315014099,
85403.334831213607686,
85412.580345660579042,
85421.825956646629493,
85431.071664162445813,
85440.317468198700226,
85449.563368746079504,
85458.809365795270423,
85468.055459336959757,
85477.301649361834279,
85486.547935860580765,
85495.794318823885988,
85505.040798242480378,
85514.287374107036158,
85523.534046408269205,
85532.780815136851743,
85542.027680283499649,
85551.274641838957905,
85560.521699793855078,
85569.768854138936149,
85579.016104864946101,
85588.263451962557156,
85597.510895422485191,
85606.758435235446086,
85616.006071392170270,
85625.253803883373621,
85634.501632699742913,
85643.749557832052233,
85652.997579271002905,
85662.245697007310810,
85671.493911031706375,
85680.742221334934584,
85689.990627907725866,
85699.239130740796099,
85708.487729824875714,
85717.736425150738796,
85726.985216709101223,
85736.234104490693426,
85745.483088486245833,
85754.732168686547084,
85763.981345082313055,
85773.230617664274178,
85782.479986423219088,
85791.729451349892770,
85800.979012435011100,
85810.228669669362716,
85819.478423043692601,
85828.728272548760287,
85837.978218175325310,
85847.228259914161754,
85856.478397756014601,
85865.728631691672490,
85874.978961711851298,
85884.229387807354215,
85893.479909968955326,
85902.730528187428718,
85911.981242453548475,
85921.232052758088685,
85930.482959091823432,
85939.733961445512250,
85948.985059809943778,
85958.236254175935755,
85967.487544534247718,
85976.738930875653750,
85985.990413190957042,
85995.241991470946232,
86004.493665706439060,
86013.745435888165957,
86022.997302006959217,
86032.249264053636580,
86041.501322018972132,
86050.753475893754512,
86060.005725668816012,
86069.258071334930719,
86078.510512882916373,
86087.763050303605269,
86097.015683587771491,
86106.268412726232782,
86115.521237709821435,
86124.774158529311535,
86134.027175175593584,
86143.280287639412563,
86152.533495911600767,
86161.786799983019591,
86171.040199844457675,
86180.293695486732759,
86189.547286900720792,
86198.800974077195860,
86208.054757007019361,
86217.308635681009036,
86226.562610090011731,
86235.816680224830634,
86245.070846076356247,
86254.325107635391760,
86263.579464892769465,
86272.833917839365313,
86282.088466466011596,
86291.343110763540608,
86300.597850722828298,
86309.852686334692407,
86319.107617589994334,
86328.362644479580922,
86337.617766994342674,
86346.872985125082778,
86356.128298862691736,
86365.383708198060049,
86374.639213121961802,
86383.894813625360257,
86393.150509699073154,
86402.406301333932788,
86411.662188520858763,
86420.918171250712476,
86430.174249514384428,
86439.430423302692361,
86448.686692606555880,
86457.943057416850934,
86467.199517724438920,
86476.456073520210339,
86485.712724795055692,
86494.969471539836377,
86504.226313745471998,
86513.483251402823953,
86522.740284502797294,
86531.997413036253420,
86541.254636994141038,
86550.511956367306993,
86559.769371146670892,
86569.026881323123234,
86578.284486887569074,
86587.542187830913463,
86596.799984144046903,
86606.057875817874447,
86615.315862843315699,
86624.573945211275714,
86633.832122912674095,
86643.090395938415895,
86652.348764279377065,
86661.607227926549967,
86670.865786870781449,
86680.124441103049321,
86689.383190614229534,
86698.642035395270796,
86707.900975437078159,
86717.160010730585782,
86726.419141266742372,
86735.678367036438431,
86744.937688030651771,
86754.197104240258341,
86763.456615656235954,
86772.716222269533318,
86781.975924071040936,
86791.235721051722066,
86800.495613202539971,
86809.755600514399703,
86819.015682978279074,
86828.275860585097689,
86837.536133325804258,
86846.796501191391144,
86856.056964172777953,
86865.317522260913393,
86874.578175446746172,
86883.838923721268657,
86893.099767075429554,
86902.360705500177573,
86911.621738986461423,
86920.882867525258916,
86930.144091107562417,
86939.405409724306082,
86948.666823366453173,
86957.928332025025156,
86967.189935690927086,
86976.451634355165879,
86985.713428008733899,
86994.975316642594407,
87004.237300247710664,
87013.499378815104137,
87022.761552335708984,
87032.023820800546673,
87041.286184200609569,
87050.548642526831827,
87059.811195770278573,
87069.073843921854859,
87078.336586972625810,
87087.599424913554685,
87096.862357735662954,
87106.125385429913877,
87115.388507987328921,
87124.651725398915005,
87133.915037655635388,
87143.178444748555194,
87152.441946668652236,
87161.705543406933430,
87170.969234954391140,
87180.233021302075940,
87189.496902440980193,
87198.760878362125368,
87208.024949056532932,
87217.289114515195251,
87226.553374729177449,
87235.817729689442785,
87245.082179387056385,
87254.346723813068820,
87263.611362958443351,
87272.876096814245102,
87282.140925371510093,
87291.405848621259793,
87300.670866554530221,
87309.935979162342846,
87319.201186435762793,
87328.466488365811529,
87337.731884943554178,
87346.997376159968553,
87356.262962006163434,
87365.528642473189393,
87374.794417552067898,
87384.060287233820418,
87393.326251509541180,
87402.592310370280757,
87411.858463807075168,
87421.124711810989538,
87430.391054373074439,
87439.657491484409547,
87448.924023136030883,
87458.190649319018121,
87467.457370024436386,
87476.724185243321699,
87485.991094966797391,
87495.258099185899482,
87504.525197891693097,
87513.792391075257910,
87523.059678727688151,
87532.327060840034392,
87541.594537403376307,
87550.862108408808126,
87560.129773847394972,
87569.397533710245625,
87578.665387988396105,
87587.933336672998848,
87597.201379755104426,
87606.469517225807067,
87615.737749076215550,
87625.006075297380448,
87634.274495880439645,
87643.543010816443712,
87652.811620096545084,
87662.080323711837991,
87671.349121653387556,
87680.618013912317110,
87689.887000479720882,
87699.156081346722203,
87708.425256504415302,
87717.694525943923509,
87726.963889656341053,
87736.233347632805817,
87745.502899864412029,
87754.772546342283022,
87764.042287057542126,
87773.312122001298121,
87782.582051164688892,
87791.852074538837769,
87801.122192114838981,
87810.392403883859515,
87819.662709837008151,
87828.933109965393669,
87838.203604260197608,
87847.474192712528748,
87856.744875313510420,
87866.015652054280508,
87875.286522925991449,
87884.557487919795676,
87893.828547026816523,
87903.099700238177320,
87912.370947545059607,
87921.642288938586717,
87930.913724409940187,
87940.185253950199694,
87949.456877550604986,
87958.728595202250290,
87968.000406896317145,
87977.272312623957987,
87986.544312376325252,
87995.816406144571374,
88005.088593919863342,
88014.360875693382695,
88023.633251456296421,
88032.905721199727850,
88042.178284914887627,
88051.450942592913634,
88060.723694225031068,
88069.996539802363259,
88079.269479316091747,
88088.542512757398072,
88097.815640117478324,
88107.088861387499492,
88116.362176558643114,
88125.635585622061626,
88134.909088569009327,
88144.182685390624101,
88153.456376078087487,
88162.730160622624680,
88172.004039015402668,
88181.278011247617542,
88190.552077310479945,
88199.826237195171416,
88209.100490892888047,
88218.374838394840481,
88227.649279692224809,
88236.923814776251675,
88246.198443638102617,
88255.473166268988280,
88264.747982660162961,
88274.022892802793649,
88283.297896688090987,
88292.572994307280169,
88301.848185651571839,
88311.123470712191192,
88320.398849480334320,
88329.674321947255521,
88338.949888104165439,
88348.225547942274716,
88357.501301452779444,
88366.777148626948474,
88376.053089456036105,
88385.329123931209324,
88394.605252043707878,
88403.881473784800619,
88413.157789145698189,
88422.434198117654887,
88431.710700691866805,
88440.987296859631897,
88450.263986612131703,
88459.540769940664177,
88468.817646836439962,
88478.094617290713359,
88487.371681294738664,
88496.648838839741074,
88505.926089917003992,
88515.203434517767164,
88524.480872633270337,
88533.758404254796915,
88543.036029373586643,
88552.313747980879270,
88561.591560067958198,
88570.869465626092278,
88580.147464646564913,
88589.425557120586745,
88598.703743039455730,
88607.982022394440719,
88617.260395176796010,
88626.538861377834110,
88635.817420988780214,
88645.096074000946828,
88654.374820405588252,
88663.653660193987889,
88672.932593357443693,
88682.211619887195411,
88691.490739774570102,
88700.769953010822064,
88710.049259587263805,
88719.328659495178727,
88728.608152725835680,
88737.887739270561724,
88747.167419120611157,
88756.447192267296487,
88765.727058701930218,
88775.007018415781204,
88784.287071400147397,
88793.567217646399513,
88802.847457145733642,
88812.127789889549604,
88821.408215869072592,
88830.688735075687873,
88839.969347500649747,
88849.250053135270718,
88858.530851970892400,
88867.811743998841848,
88877.092729210387915,
88886.373807596857660,
88895.654979149578139,
88904.936243859905517,
88914.217601719108643,
88923.499052718543680,
88932.780596849537687,
88942.062234103403171,
88951.343964471452637,
88960.625787945071352,
88969.907704515513615,
88979.189714174193796,
88988.471816912395298,
88997.754012721459731,
89007.036301592757809,
89016.318683517602040,
89025.601158487334033,
89034.883726493309950,
89044.166387526856852,
89053.449141579345451,
89062.731988642117358,
89072.014928706499632,
89081.297961763862986,
89090.581087805578136,
89099.864306822957587,
89109.147618807401159,
89118.431023750221357,
89127.714521642817999,
89136.998112476532697,
89146.281796242736164,
89155.565572932784562,
89164.849442538048606,
89174.133405049899011,
89183.417460459677386,
89192.701608758827206,
89201.985849938631873,
89211.270183990505757,
89220.554610905834124,
89229.839130676002242,
89239.123743292351719,
89248.408448746267823,
89257.693247029164922,
89266.978138132413733,
89276.263122047399520,
89285.548198765492998,
89294.833368278079433,
89304.118630576573196,
89313.403985652345000,
89322.689433496809215,
89331.974974101336556,
89341.260607457370497,
89350.546333556252648,
89359.832152389382827,
89369.118063948204508,
89378.404068224117509,
89387.690165208492544,
89396.976354892729432,
89406.262637268286198,
89415.549012326562661,
89424.835480058914982,
89434.122040456815739,
89443.408693511650199,
89452.695439214803628,
89461.982277557748603,
89471.269208531884942,
89480.556232128627016,
89489.843348339418299,
89499.130557155658607,
89508.417858568762313,
89517.705252570201992,
89526.992739151333808,
89536.280318303659442,
89545.567990018564160,
89554.855754287491436,
89564.143611101899296,
89573.431560453216662,
89582.719602332843351,
89592.007736732266494,
89601.295963642900460,
89610.584283056188724,
89619.872694963589311,
89629.161199356531142,
89638.449796226486797,
89647.738485564856092,
89657.027267363155261,
89666.316141612798674,
89675.605108305229805,
89684.894167431906681,
89694.183318984301877,
89703.472562953873421,
89712.761899332079338,
89722.051328110377654,
89731.340849280211842,
89740.630462833083584,
89749.920168760450906,
89759.209967053757282,
89768.499857704504393,
89777.789840704121161,
89787.079916044109268,
89796.370083715941291,
89805.660343711118912,
89814.950696021041949,
89824.241140637270291,
89833.531677551232860,
89842.822306754445890,
89852.113028238367406,
89861.403841994499089,
89870.694748014298966,
89879.985746289312374,
89889.276836810968234,
89898.568019570782781,
89907.859294560257695,
89917.150661770880106,
89926.442121194137144,
89935.733672821530490,
89945.025316644576378,
89954.317052654747386,
89963.608880843574298,
89972.900801202558796,
89982.192813723173458,
89991.484918396949070,
90000.777115215387312,
90010.069404170004418,
90019.361785252331174,
90028.654258453840157,
90037.946823766076704,
90047.239481180542498,
90056.532230688753771,
90065.825072282241308,
90075.118005952521344,
90084.411031691124663,
90093.704149489552947,
90102.997359339366085,
90112.290661232036655,
90121.584055159168201,
90130.877541112204199,
90140.171119082777295,
90149.464789062316413,
90158.758551042454201,
90168.052405014634132,
90177.346350970474305,
90186.640388901476399,
90195.934518799171201,
90205.228740655147703,
90214.523054460893036,
90223.817460207981640,
90233.111957887958852,
90242.406547492399113,
90251.701229012804106,
90260.996002440748271,
90270.290867767806049,
90279.585824985493673,
90288.880874085385585,
90298.176015059070778,
90307.471247898050933,
90316.766572593929595,
90326.061989138266654,
90335.357497522621998,
90344.653097738526412,
90353.948789777627098,
90363.244573631425737,
90372.540449291496770,
90381.836416749429191,
90391.132475996797439,
90400.428627025205060,
90409.724869826182839,
90419.021204391319770,
90428.317630712204846,
90437.614148780427058,
90446.910758587560849,
90456.207460125180660,
90465.504253384890035,
90474.801138358263415,
90484.098115036889794,
90493.395183412387269,
90502.692343476315727,
90511.989595220278716,
90521.286938635879778,
90530.584373714707908,
90539.881900448366650,
90549.179518828444998,
90558.477228846546495,
90567.775030494303792,
90577.072923763291328,
90586.370908645112650,
90595.668985131400404,
90604.967153213729034,
90614.265412883760291,
90623.563764133054065,
90632.862206953243003,
90642.160741335959756,
90651.459367272793315,
90660.758084755390882,
90670.056893775356002,
90679.355794324321323,
90688.654786393875838,
90697.953869975695852,
90707.253045061384910,
90716.552311642546556,
90725.851669710827991,
90735.151119257890969,
90744.450660275324481,
90753.750292754790280,
90763.050016687891912,
90772.349832066291128,
90781.649738881620578,
90790.949737125527463,
90800.249826789644430,
90809.550007865633233,
90818.850280345111969,
90828.150644219742389,
90837.451099481142592,
90846.751646121003432,
90856.052284130957560,
90865.353013502681279,
90874.653834227778134,
90883.954746297938982,
90893.255749704796472,
90902.556844440056011,
90911.858030495321145,
90921.159307862297283,
90930.460676532602520,
90939.762136497971369,
90949.063687749992823,
90958.365330280386843,
90967.667064080771524,
90976.968889142895932,
90986.270805458349059,
90995.572813018850866,
91004.874911816077656,
91014.177101841662079,
91023.479383087324095,
91032.781755544754560,
91042.084219205615227,
91051.386774061582400,
91060.689420104346937,
91069.992157325570588,
91079.294985717002419,
91088.597905270260526,
91097.900915977093973,
91107.204017829149961,
91116.507210818133899,
91125.810494935751194,
91135.113870173692703,
91144.417336523663835,
91153.720893977340893,
91163.024542526458390,
91172.328282162678079,
91181.632112877763575,
91190.936034663347527,
91200.240047511208104,
91209.544151412963402,
91218.848346360420692,
91228.152632345241727,
91237.457009359146468,
91246.761477393854875,
91256.066036441057804,
91265.370686492504319,
91274.675427539899829,
91283.980259574964293,
91293.285182589446777,
91302.590196575023583,
91311.895301523443777,
91321.200497426427319,
91330.505784275708720,
91339.811162063022493,
91349.116630780088599,
91358.422190418641549,
91367.727840970430407,
91377.033582427175133,
91386.339414780595689,
91395.645338022470241,
91404.951352144518751,
91414.257457138475729,
91423.563652996090241,
91432.869939709125902,
91442.176317269302672,
91451.482785668369615,
91460.789344898090349,
91470.095994950199383,
91479.402735816460336,
91488.709567488622270,
91498.016489958448801,
91507.323503217688994,
91516.630607258091914,
91525.937802071435726,
91535.245087649469497,
91544.552463983942289,
91553.859931066661375,
91563.167488889361266,
91572.475137443805579,
91581.782876721772482,
91591.090706715011038,
91600.398627415328519,
91609.706638814473990,
91619.014740904240170,
91628.322933676376124,
91637.631217122689122,
91646.939591234928230,
91656.248056004900718,
91665.556611424384755,
91674.865257485129405,
91684.173994178971043,
91693.482821497644181,
91702.791739432999748,
91712.100747976772254,
91721.409847120768973,
91730.719036856797175,
91740.028317176620476,
91749.337688072046149,
91758.647149534896016,
91767.956701556948246,
91777.266344130010111,
91786.576077245859778,
91795.885900896319072,
91805.195815073180711,
91814.505819768266520,
91823.815914973383769,
91833.126100680354284,
91842.436376880927128,
91851.746743566967780,
91861.057200730268960,
91870.367748362667044,
91879.678386455969303,
91888.989115001953905,
91898.299933992515435,
91907.610843419417506,
91916.921843274496496,
91926.232933549559675,
91935.544114236457972,
91944.855385327013209,
91954.166746813061764,
91963.478198686396354,
91972.789740938867908,
91982.101373562327353,
91991.413096548611065,
92000.724909889497212,
92010.036813576894929,
92019.348807602596935,
92028.660891958468710,
92037.973066636302974,
92047.285331628023414,
92056.597686925408198,
92065.910132520308252,
92075.222668404603610,
92084.535294570130645,
92093.848011008711183,
92103.160817712239805,
92112.473714672538335,
92121.786701881472254,
92131.099779330892488,
92140.412947012679069,
92149.726204918653821,
92159.039553040711326,
92168.352991370687960,
92177.666519900449202,
92186.980138621904189,
92196.293847526845639,
92205.607646607197239,
92214.921535854795366,
92224.235515261520050,
92233.549584819265874,
92242.863744519883767,
92252.177994355239207,
92261.492334317241330,
92270.806764397726511,
92280.121284588618437,
92289.435894881768036,
92298.750595269084442,
92308.065385742418584,
92317.380266293679597,
92326.695236914703855,
92336.010297597473254,
92345.325448333795066,
92354.640689115592977,
92363.956019934761571,
92373.271440783180878,
92382.586951652774587,
92391.902552535393625,
92401.218243422990781,
92410.534024307416985,
92419.849895180610474,
92429.165856034436729,
92438.481906860848540,
92447.798047651696834,
92457.114278398948954,
92466.430599094455829,
92475.747009730155696,
92485.063510297957691,
92494.380100789770950,
92503.696781197519158,
92513.013551513126004,
92522.330411728471518,
92531.647361835508491,
92540.964401826131507,
92550.281531692307908,
92559.598751425903174,
92568.916061018870096,
92578.233460463146912,
92587.550949750642758,
92596.868528873295872,
92606.186197823015391,
92615.503956591754104,
92624.821805171450251,
92634.139743554012966,
92643.457771731424145,
92652.775889695563819,
92662.094097438413883,
92671.412394951898023,
92680.730782227954478,
92690.049259258521488,
92699.367826035580947,
92708.686482551041991,
92718.005228796871961,
92727.324064764994546,
92736.642990447391639,
92745.962005835986929,
92755.281110922747757,
92764.600305699626915,
92773.919590158606297,
92783.238964291595039,
92792.558428090604139,
92801.877981547528179,
92811.197624654407264,
92820.517357403165079,
92829.837179785754415,
92839.157091794157168,
92848.477093420340680,
92857.797184656272293,
92867.117365493919351,
92876.437635925278300,
92885.757995942287380,
92895.078445536943036,
92904.398984701227164,
92913.719613427107106,
92923.040331706550205,
92932.361139531552908,
92941.682036894082557,
92951.003023786150152,
92960.324100199723034,
92969.645266126783099,
92978.966521559341345,
92988.287866489350563,
92997.609300908850855,
93006.930824809780461,
93016.252438184164930,
93025.574141023986158,
93034.895933321255143,
93044.217815067953779,
93053.539786256122170,
93062.861846877698554,
93072.183996924737585,
93081.506236389192054,
93090.828565263131168,
93100.150983538507717,
93109.473491207347251,
93118.796088261675322,
93128.118774693488376,
93137.441550494797411,
93146.764415657613426,
93156.087370173947420,
93165.410414035854046,
93174.733547235300648,
93184.056769764327328,
93193.380081614974188,
93202.703482779223123,
93212.026973249128787,
93221.350553016716731,
93230.674222073997953,
93239.997980412998004,
93249.321828025771538,
93258.645764904329553,
93267.969791040712153,
93277.293906426959438,
93286.618111055082409,
93295.942404917135718,
93305.266788005144917,
93314.591260311179212,
93323.915821827235050,
93333.240472545381635,
93342.565212457688176,
93351.890041556136566,
93361.214959832825116,
93370.539967279779376,
93379.865063889039448,
93389.190249652689090,
93398.515524562753853,
93407.840888611302944,
93417.166341790376464,
93426.491884092043620,
93435.817515508373617,
93445.143236031362903,
93454.469045653153444,
93463.794944365756237,
93473.120932161269593,
93482.447009031733614,
93491.773174969202955,
93501.099429965790478,
93510.425774013507180,
93519.752207104480476,
93529.078729230735917,
93538.405340384386363,
93547.732040557471919,
93557.058829742090893,
93566.385707930312492,
93575.712675114235026,
93585.039731285898597,
93594.366876437416067,
93603.694110560871195,
93613.021433648318634,
93622.348845691900351,
93631.676346683641896,
93641.003936615656130,
93650.331615480070468,
93659.659383268910460,
93668.987239974318072,
93678.315185588362510,
93687.643220103156636,
93696.971343510798761,
93706.299555803372641,
93715.627856972976588,
93724.956247011708911,
93734.284725911726127,
93743.613293665068340,
93752.941950263862964,
93762.270695700193755,
93771.599529966217233,
93780.928453054031706,
93790.257464955720934,
93799.586565663412330,
93808.915755169247859,
93818.245033465267625,
93827.574400543671800,
93836.903856396544143,
93846.233401016012067,
93855.563034394159331,
93864.892756523157004,
93874.222567395117949,
93883.552467002140475,
93892.882455336410203,
93902.212532389981789,
93911.542698155026301,
93920.872952623642050,
93930.203295788029209,
93939.533727640271536,
93948.864248172481894,
93958.194857376845903,
93967.525555245490978,
93976.856341770544532,
93986.187216944163083,
93995.518180758459494,
94004.849233205604833,
94014.180374277711962,
94023.511603966966504,
94032.842922265495872,
94042.174329165456584,
94051.505824658990605,
94060.837408738254453,
94070.169081395390094,
94079.500842622568598,
94088.832692411961034,
94098.164630755680264,
94107.496657645897358,
94116.828773074812489,
94126.160977034538519,
94135.493269517275621,
94144.825650515165762,
94154.158120020365459,
94163.490678025074885,
94172.823324521421455,
94182.156059501605341,
94191.488882957797614,
94200.821794882169343,
94210.154795266877045,
94219.487884104120894,
94228.821061386042857,
94238.154327104857657,
94247.487681252750917,
94256.821123821835499,
94266.154654804369784,
94275.488274192510289,
94284.821981978442636,
94294.155778154337895,
94303.489662712410791,
94312.823635644846945,
94322.157696943788324,
94331.491846601493307,
94340.826084610118414,
94350.160410961892921,
94359.494825648958795,
94368.829328663559863,
94378.163919997867197,
94387.498599644110072,
94396.833367594445008,
94406.168223841144936,
94415.503168376322719,
94424.838201192280394,
94434.173322281130822,
94443.508531635161489,
94452.843829246543464,
94462.179215107491473,
94471.514689210234792,
94480.850251546944492,
94490.185902109893505,
94499.521640891267452,
94508.857467883281060,
94518.193383078178158,
94527.529386468144367,
94536.865478045423515,
94546.201657802230329,
94555.537925730823190,
94564.874281823358615,
94574.210726072153193,
94583.547258469348890,
94592.883879007247742,
94602.220587678020820,
94611.557384473970160,
94620.894269387252280,
94630.231242410169216,
94639.568303534921142,
94648.905452753766440,
94658.242690058948938,
94667.580015442668810,
94676.917428897213540,
94686.254930414797855,
94695.592519987680134,
94704.930197608118760,
94714.267963268357562,
94723.605816960640368,
94732.943758677196456,
94742.281788410313311,
94751.619906152249314,
94760.958111895219190,
94770.296405631495873,
94779.634787353352294,
94788.973257053032285,
94798.311814722808776,
94807.650460354954703,
94816.989193941684789,
94826.328015475315624,
94835.666924948091037,
94845.005922352269408,
94854.345007680123672,
94863.684180923955864,
94873.023442076009815,
94882.362791128558456,
94891.702228073845617,
94901.041752904216992,
94910.381365611901856,
94919.721066189173143,
94929.060854628347442,
94938.400730921654031,
94947.740695061424049,
94957.080747039930429,
94966.420886849431554,
94975.761114482258563,
94985.101429930655286,
94994.441833186923759,
95003.782324243366020,
95013.122903092269553,
95022.463569725921843,
95031.804324136610376,
95041.145166316651739,
95050.486096258333419,
95059.827113953957451,
95069.168219395825872,
95078.509412576182513,
95087.850693487431272,
95097.192062121815979,
95106.533518471653224,
95115.875062529245042,
95125.216694286893471,
95134.558413736929651,
95143.900220871655620,
95153.242115683373413,
95162.584098164399620,
95171.926168307050830,
95181.268326103658183,
95190.610571546509163,
95199.952904627949465,
95209.295325340266572,
95218.637833675820730,
95227.980429626928526,
95237.323113185891998,
95246.665884345042286,
95256.008743096725084,
95265.351689433256979,
95274.694723346954561,
95284.037844830163522,
95293.381053875215002,
95302.724350474440143,
95312.067734620155534,
95321.411206304750522,
95330.754765520498040,
95340.098412259787438,
95349.442146514935303,
95358.785968278287328,
95368.129877542189206,
95377.473874298972078,
95386.817958540996187,
95396.162130260621780,
95405.506389450165443,
95414.850736101987422,
95424.195170208433410,
95433.539691761863651,
95442.884300754652941,
95452.228997179146972,
95461.573781027647783,
95470.918652292588376,
95480.263610966285341,
95489.608657041113474,
95498.953790509447572,
95508.299011363604222,
95517.644319596001878,
95526.989715198971680,
95536.335198164873873,
95545.680768486112356,
95555.026426155047375,
95564.372171164039173,
95573.718003505433444,
95583.063923171663191,
95592.409930155059556,
95601.756024448011885,
95611.102206042909529,
95620.448474932083627,
95629.794831107981736,
95639.141274562949548,
95648.487805289390963,
95657.834423279651674,
95667.181128526135581,
95676.527921021275688,
95685.874800757388584,
95695.221767726878170,
95704.568821922192001,
95713.915963335646666,
95723.263191959689721,
95732.610507786681410,
95741.957910809054738,
95751.305401019184501,
95760.652978409460047,
95770.000642972285277,
95779.348394700107747,
95788.696233585244045,
95798.044159620199935,
95807.392172797277453,
95816.740273108982365,
95826.088460547631257,
95835.436735105715343,
95844.785096775580314,
95854.133545549673727,
95863.482081420399481,
95872.830704380176030,
95882.179414421421825,
95891.528211536540766,
95900.877095717965858,
95910.226066958101001,
95919.575125249393750,
95928.924270584248006,
95938.273502955096774,
95947.622822354358505,
95956.972228774451651,
95966.321722207809216,
95975.671302646878758,
95985.020970084064174,
95994.370724511842127,
96003.720565922601963,
96013.070494308791240,
96022.420509662813856,
96031.770611977175577,
96041.120801244280301,
96050.471077456531930,
96059.821440606436227,
96069.171890686397091,
96078.522427688876633,
96087.873051606293302,
96097.223762431123760,
96106.574560155786457,
96115.925444772772607,
96125.276416274457006,
96134.627474653374520,
96143.978619901943603,
96153.329852012640913,
96162.681170977870352,
96172.032576790108578,
96181.384069441846805,
96190.735648925518035,
96200.087315233598929,
96209.439068358522491,
96218.790908292779932,
96228.142835028804257,
96237.494848559101229,
96246.846948876103852,
96256.199135972303338,
96265.551409840190900,
96274.903770472170436,
96284.256217860776815,
96293.608751998457592,
96302.961372877689428,
96312.314080490934430,
96321.666874830712914,
96331.019755889486987,
96340.372723659704207,
96349.725778133870335,
96359.078919304491137,
96368.432147164014168,
96377.785461704930640,
96387.138862919746316,
96396.492350800952408,
96405.845925341025577,
96415.199586532442481,
96424.553334367694333,
96433.907168839330552,
96443.261089939769590,
96452.615097661560867,
96461.969191997181042,
96471.323372939121327,
96480.677640479916590,
96490.031994612043491,
96499.386435327978688,
96508.740962620271603,
96518.095576481413445,
96527.450276903895428,
96536.805063880223315,
96546.159937402946525,
96555.514897464512615,
96564.869944057500106,
96574.225077174371108,
96583.580296807675040,
96592.935602949888562,
96602.290995593575644,
96611.646474731212948,
96621.002040355335339,
96630.357692458463134,
96639.713431033160305,
96649.069256071874406,
96658.425167567154858,
96667.781165511551080,
96677.137249897583388,
96686.493420717772096,
96695.849677964652074,
96705.206021630743635,
96714.562451708596200,
96723.918968190744636,
96733.275571069680154,
96742.632260337995831,
96751.989035988197429,
96761.345898012834368,
96770.702846404456068,
96780.059881155568291,
96789.417002258764114,
96798.774209706534748,
96808.131503491473268,
96817.488883606099989,
96826.846350042978884,
96836.203902794630267,
96845.561541853618110,
96854.919267212506384,
96864.277078863844508,
96873.634976800181903,
96882.992961014067987,
96892.351031498066732,
96901.709188244756660,
96911.067431246643537,
96920.425760496334988,
96929.784175986365881,
96939.142677709329291,
96948.501265657760086,
96957.859939824236790,
96967.218700201337924,
96976.577546781612909,
96985.936479557640268,
96995.295498521983973,
97004.654603667237097,
97014.013794985934510,
97023.373072470698389,
97032.732436114049051,
97042.091885908637778,
97051.451421846970334,
97060.811043921668897,
97070.170752125282888,
97079.530546450434485,
97088.890426889658556,
97098.250393435606384,
97107.610446080798283,
97116.970584817870986,
97126.330809639388463,
97135.691120537929237,
97145.051517506130040,
97154.412000536525738,
97163.772569621753064,
97173.133224754390540,
97182.493965927031240,
97191.854793132282794,
97201.215706362723722,
97210.576705610990757,
97219.937790869662422,
97229.298962131346343,
97238.660219388635596,
97248.021562634166912,
97257.382991860489710,
97266.744507060269825,
97276.106108226085780,
97285.467795350545202,
97294.829568426284823,
97304.191427445912268,
97313.553372402006062,
97322.915403287232039,
97332.277520094183274,
97341.639722815452842,
97351.002011443706579,
97360.364385971523006,
97369.726846391582512,
97379.089392696434516,
97388.452024878744851,
97397.814742931135697,
97407.177546846258338,
97416.540436616691295,
97425.903412235100404,
97435.266473694064189,
97444.629620986306691,
97453.992854104377329,
97463.356173040956492,
97472.719577788651804,
97482.083068340143654,
97491.446644687996013,
97500.810306824932923,
97510.174054743518354,
97519.537888436461799,
97528.901807896370883,
97538.265813115882338,
97547.629904087676550,
97556.994080804375699,
97566.358343258601963,
97575.722691443050280,
97585.087125350386486,
97594.451644973189104,
97603.816250304182176,
97613.180941335987882,
97622.545718061272055,
97631.910580472700531,
97641.275528562895488,
97650.640562324551865,
97660.005681750306394,
97669.370886832839460,
97678.736177564831451,
97688.101553938904544,
97697.467015947739128,
97706.832563584015588,
97716.198196840399760,
97725.563915709557477,
97734.929720184154576,
97744.295610256871441,
97753.661585920373909,
97763.027647167356918,
97772.393793990457198,
97781.760026382384240,
97791.126344335847534,
97800.492747843454708,
97809.859236897915252,
97819.225811491924105,
97828.592471618147101,
97837.959217269279179,
97847.326048438015277,
97856.692965117035783,
97866.059967299021082,
97875.427054976680665,
97884.794228142680367,
97894.161486789729679,
97903.528830910508987,
97912.896260497727781,
97922.263775544081000,
97931.631376042263582,
97940.999061984970467,
97950.366833364867489,
97959.734690174722346,
97969.102632407229976,
97978.470660055027111,
97987.838773110866896,
97997.206971567473374,
98006.575255417483277,
98015.943624653707957,
98025.312079268769594,
98034.680619255421334,
98044.049244606372667,
98053.417955314333085,
98062.786751371982973,
98072.155632772090030,
98081.524599507363746,
98090.893651570469956,
98100.262788954190910,
98109.632011651236098,
98119.001319654315012,
98128.370712956122588,
98137.740191549470183,
98147.109755426994525,
98156.479404581434210,
98165.849139005556935,
98175.218958692086744,
98184.588863633747678,
98193.958853823234676,
98203.328929253329989,
98212.699089916757657,
98222.069335806241725,
98231.439666914535337,
98240.810083234362537,
98250.180584758447367,
98259.551171479586628,
98268.921843390446156,
98278.292600483837305,
98287.663442752455012,
98297.034370189081528,
98306.405382786455448,
98315.776480537300813,
98325.147663434414426,
98334.518931470491225,
98343.890284638298908,
98353.261722930634278,
98362.633246340192272,
98372.004854859769694,
98381.376548482105136,
98390.748327199951746,
98400.120191006091773,
98409.492139893263811,
98418.864173854235560,
98428.236292881774716,
98437.608496968648979,
98446.980786107596941,
98456.353160291429958,
98465.725619512886624,
98475.098163764734636,
98484.470793039756245,
98493.843507330719149,
98503.216306630391045,
98512.589190931539633,
98521.962160226976266,
98531.335214509454090,
98540.708353771711700,
98550.081578006589552,
98559.454887206855346,
98568.828281365262228,
98578.201760474606999,
98587.575324527686462,
98596.948973517268314,
98606.322707436178462,
98615.696526277126395,
98625.070430032981676,
98634.444418696482899,
98643.818492260441417,
98653.192650717654033,
98662.566894060888444,
98671.941222282956005,
98681.315635376697173,
98690.690133334835991,
98700.064716150198365,
98709.439383815595647,
98718.814136323824641,
98728.188973667682149,
98737.563895839979523,
98746.938902833499014,
98756.313994641066529,
98765.689171255507972,
98775.064432669591042,
98784.439778876156197,
98793.815209868000238,
98803.190725637934520,
98812.566326178770396,
98821.942011483333772,
98831.317781544436002,
98840.693636354888440,
98850.069575907516992,
98859.445600195147563,
98868.821709210576955,
98878.197902946631075,
98887.574181396135828,
98896.950544551946223,
98906.326992406859063,
98915.703524953700253,
98925.080142185281147,
98934.456844094485859,
98943.833630674111191,
98953.210501916968497,
98962.587457815912785,
98971.964498363784514,
98981.341623553395038,
98990.718833377613919,
99000.096127829267061,
99009.473506901165820,
99018.850970586165204,
99028.228518877134775,
99037.606151766871335,
99046.983869248258998,
99056.361671314123669,
99065.739557957320358,
99075.117529170674970,
99084.495584947057068,
99093.873725279307109,
99103.251950160280103,
99112.630259582831059,
99122.008653539785882,
99131.387132024057792,
99140.765695028458140,
99150.144342545827385,
99159.523074569064192,
99168.901891091009020,
99178.280792104531429,
99187.659777602486429,
99197.038847577729030,
99206.418002023128793,
99215.797240931584383,
99225.176564295907156,
99234.555972109010327,
99243.935464363734354,
99253.315041052934248,
99262.694702169552329,
99272.074447706385399,
99281.454277656346676,
99290.834192012276617,
99300.214190767102991,
99309.594273913666257,
99318.974441444850527,
99328.354693353554467,
99337.735029632618534,
99347.115450274970499,
99356.495955273465370,
99365.876544621016365,
99375.257218310463941,
99384.637976334735868,
99394.018818686701707,
99403.399745359245571,
99412.780756345280679,
99422.161851637691143,
99431.543031229331973,
99440.924295113160042,
99450.305643282015808,
99459.687075728841592,
99469.068592446521507,
99478.450193427925115,
99487.831878665980184,
99497.213648153570830,
99506.595501883624820,
99515.977439849040820,
99525.359462042702944,
99534.741568457538961,
99544.123759086447535,
99553.506033922327333,
99562.888392958106124,
99572.270836186682573,
99581.653363600955345,
99591.035975193866761,
99600.418670958344592,
99609.801450887258397,
99619.184314973550499,
99628.567263210119563,
99637.950295589922462,
99647.333412105843308,
99656.716612750838976,
99666.099897517779027,
99675.483266399649438,
99684.866719389334321,
99694.250256479775999,
99703.633877663887688,
99713.017582934626262,
99722.401372284875833,
99731.785245707622380,
99741.169203195750015,
99750.553244742230163,
99759.937370339990593,
99769.321579981944524,
99778.705873661034275,
99788.090251370231272,
99797.474713102419628,
99806.859258850585320,
99816.243888607670669,
99825.628602366588893,
99835.013400120296865,
99844.398281861751457,
99853.783247583894990,
99863.168297279669787,
99872.553430942003615,
99881.938648563897004,
99891.323950138248620,
99900.709335658044438,
99910.094805116241332,
99919.480358505781624,
99928.865995819593081,
99938.251717050705338,
99947.637522192002507,
99957.023411236485117,
99966.409384177095490,
99975.795441006819601,
99985.181581718599773,
99994.567806305407430,
100003.954114760199445,
100013.340507075976348,
100022.726983245636802,
100032.113543262224994,
100041.500187118668691,
100050.886914807953872,
100060.273726323037408,
100069.660621656905278,
100079.047600802543457,
100088.434663752937922,
100097.821810501016444,
100107.209041039764998,
100116.596355362227769,
100125.983753461317974,
100135.371235330050695,
100144.758800961397355,
100154.146450348343933,
100163.534183483861852,
100172.922000360966194,
100182.309900972628384,
100191.697885311863502,
100201.085953371642972,
100210.474105144923669,
100219.862340624749777,
100229.250659804063616,
100238.639062675923924,
100248.027549233273021,
100257.416119469140540,
100266.804773376512458,
100276.193510948389303,
100285.582332177786157,
100294.971237057674443,
100304.360225581083796,
100313.749297741000191,
100323.138453530438710,
100332.527692942414433,
100341.917015969913336,
100351.306422605965054,
100360.695912843584665,
100370.085486675729044,
100379.475144095486030,
100388.864885095827049,
100398.254709669781732,
100407.644617810350610,
100417.034609510563314,
100426.424684763434925,
100435.814843562009628,
100445.205085899229744,
100454.595411768226768,
100463.985821161943022,
100473.376314073422691,
100482.766890495709958,
100492.157550421790802,
100501.548293844723958,
100510.939120757553610,
100520.330031153294840,
100529.721025024962728,
100539.112102365601459,
100548.503263168269768,
100557.894507425924530,
100567.285835131697240,
100576.677246278559323,
100586.068740859569516,
100595.460318867786555,
100604.851980296210968,
100614.243725137930596,
100623.635553385960520,
100633.027465033315821,
100642.419460073113441,
100651.811538498324808,
100661.203700302081415,
100670.595945477340138,
100679.988274017203366,
100689.380685914715286,
100698.773181162934634,
100708.165759754905594,
100717.558421683672350,
100726.951166942322743,
100736.343995523857302,
100745.736907421392971,
100755.129902627944830,
100764.522981136586168,
100773.916142940404825,
100783.309388032430434,
100792.702716405750834,
100802.096128053410212,
100811.489622968496406,
100820.883201144039049,
100830.276862573155086,
100839.670607248859596,
100849.064435164298629,
100858.458346312443609,
100867.852340686440584,
100877.246418279377394,
100886.640579084254568,
100896.034823094203603,
100905.429150302297785,
100914.823560701595852,
100924.218054285200196,
100933.612631046184106,
100943.007290977591765,
100952.402034072598326,
100961.796860324189765,
100971.191769725497579,
100980.586762269595056,
100989.981837949613691,
100999.376996758568566,
101008.772238689591177,
101018.167563735798467,
101027.562971890220069,
101036.958463146016584,
101046.354037496217643,
101055.749694933983847,
101065.145435452344827,
101074.541259044461185,
101083.937165703391656,
101093.333155422267737,
101102.729228194148163,
101112.125384012178984,
101121.521622869433486,
101130.917944759043166,
101140.314349674081313,
101149.710837607679423,
101159.107408552939887,
101168.504062503008754,
101177.900799450915656,
101187.297619389850297,
101196.694522312871413,
101206.091508213110501,
101215.488577083713608,
101224.885728917754022,
101234.282963708363241,
101243.680281448687310,
101253.077682131799520,
101262.475165750831366,
101271.872732298943447,
101281.270381769223604,
101290.668114154817886,
101300.065929448814131,
101309.463827644387493,
101318.861808734640363,
101328.259872712689685,
101337.658019571696059,
101347.056249304776429,
101356.454561905047740,
101365.852957365656039,
101375.251435679761926,
101384.649996840453241,
101394.048640840905136,
101403.447367674249108,
101412.846177333572996,
101422.245069812110160,
101431.644045102933887,
101441.043103199219331,
101450.442244094068883,
101459.841467780686799,
101469.240774252175470,
101478.640163501695497,
101488.039635522378376,
101497.439190307413810,
101506.838827849933296,
101516.238548143068328,
101525.638351179994061,
101535.038236953827436,
101544.438205457787262,
101553.838256684990483,
101563.238390628612251,
101572.638607281798613,
101582.038906637681066,
101591.439288689463865,
101600.839753430293058,
101610.240300853358349,
101619.640930951776681,
101629.041643718723208,
101638.442439147402183,
101647.843317230945104,
101657.244277962527121,
101666.645321335308836,
101676.046447342494503,
101685.447655977230170,
101694.848947232705541,
101704.250321102081216,
101713.651777578532347,
101723.053316655219533,
101732.454938325376133,
101741.856642582148197,
101751.258429418696323,
101760.660298828210216,
101770.062250803894131,
101779.464285338894115,
101788.866402426443528,
101798.268602059688419,
101807.670884231847594,
101817.073248936067102,
101826.475696165565751,
101835.878225913547794,
101845.280838173159282,
101854.683532937648124,
101864.086310200145817,
101873.489169953900273,
101882.892112192057539,
101892.295136907865526,
101901.698244094499387,
101911.101433745148825,
101920.504705853032647,
101929.908060411355109,
101939.311497413291363,
101948.715016852060216,
101958.118618720851373,
101967.522303012912744,
101976.926069721434033,
101986.329918839590391,
101995.733850360615179,
102005.137864277712652,
102014.541960584130720,
102023.946139273015433,
102033.350400337614701,
102042.754743771176436,
102052.159169566861237,
102061.563677717917017,
102070.968268217548029,
102080.372941058973083,
102089.777696235440089,
102099.182533740138751,
102108.587453566316981,
102117.992455707149929,
102127.397540155914612,
102136.802706905829837,
102146.207955950114410,
102155.613287281987141,
102165.018700894681388,
102174.424196781430510,
102183.829774935467867,
102193.235435350012267,
102202.641178018340725,
102212.047002933613840,
102221.452910089166835,
102230.858899478131207,
102240.264971093813074,
102249.671124929445796,
102259.077360978233628,
102268.483679233439034,
102277.890079688324477,
102287.296562336123316,
102296.703127170039807,
102306.109774183394620,
102315.516503369377460,
102324.923314721236238,
102334.330208232262521,
102343.737183895675116,
102353.144241704721935,
102362.551381652694545,
102371.958603732768097,
102381.365907938292366,
102390.773294262457057,
102400.180762698539183,
102409.588313239786658,
102418.995945879505598,
102428.403660610900261,
102437.811457427247660,
102447.219336321824812,
102456.627297287894180,
102466.035340318689123,
102475.443465407501208,
102484.851672547636554,
102494.259961732270312,
102503.668332954752259,
102513.076786208315752,
102522.485321486266912,
102531.893938781809993,
102541.302638088280219,
102550.711419398925500,
102560.120282707051956,
102569.529228005878394,
102578.938255288740038,
102588.347364548870246,
102597.756555779604241,
102607.165828974189935,
102616.575184125889791,
102625.984621228024480,
102635.394140273856465,
102644.803741256691865,
102654.213424169778591,
102663.623189006466419,
102673.033035759988707,
102682.442964423651574,
102691.852974990761140,
102701.263067454594420,
102710.673241808457533,
102720.083498045612941,
102729.493836159395869,
102738.904256143097882,
102748.314757989981445,
102757.725341693396331,
102767.136007246590452,
102776.546754642913584,
102785.957583875642740,
102795.368494938084041,
102804.779487823543604,
102814.190562525327550,
102823.601719036727445,
102833.012957351063960,
102842.424277461657766,
102851.835679361800430,
102861.247163044812623,
102870.658728504000464,
102880.070375732684624,
102889.482104724171222,
102898.893915471766377,
102908.305807968834415,
102917.717782208637800,
102927.129838184497203,
102936.541975889776950,
102945.954195317768608,
102955.366496461792849,
102964.778879315155791,
102974.191343871236313,
102983.603890123311430,
102993.016518064730917,
103002.429227688800893,
103011.842018988842028,
103021.254891958247754,
103030.667846590280533,
103040.080882878304692,
103049.494000815640902,
103058.907200395624386,
103068.320481611590367,
103077.733844456888619,
103087.147288924839813,
103096.560815008808277,
103105.974422702085576,
103115.388111998036038,
103124.801882890009438,
103134.215735371355549,
103143.629669435409596,
103153.043685075506801,
103162.457782284982386,
103171.871961057229782,
103181.286221385569661,
103190.700563263308140,
103200.114986683867755,
103209.529491640554625,
103218.944078126747627,
103228.358746135796537,
103237.773495661007473,
103247.188326695802971,
103256.603239233503700,
103266.018233267473988,
103275.433308791092713,
103284.848465797695098,
103294.263704280616366,
103303.679024233293603,
103313.094425649047480,
103322.509908521213220,
103331.925472843198804,
103341.341118608354009,
103350.756845810057712,
103360.172654441659688,
103369.588544496567920,
103379.004515968117630,
103388.420568849687697,
103397.836703134642448,
103407.252918816360761,
103416.669215888221515,
103426.085594343603589,
103435.502054175900412,
103444.918595378447208,
103454.335217944666510,
103463.751921867908095,
103473.168707141579944,
103482.585573759046383,
103492.002521713700844,
103501.419550998922205,
103510.836661608089344,
103520.253853534595692,
103529.671126771834679,
103539.088481313199736,
103548.505917152069742,
103557.923434281852678,
103567.341032695898321,
103576.758712387643754,
103586.176473350467859,
103595.594315577764064,
103605.012239062940353,
103614.430243799404707,
103623.848329780506901,
103633.266496999684023,
103642.684745450314949,
103652.103075125865871,
103661.521486019642907,
103670.939978125112248,
103680.358551435652771,
103689.777205944716115,
103699.195941645652056,
103708.614758531897678,
103718.033656596846413,
103727.452635833949898,
103736.871696236572461,
103746.290837798136636,
103755.710060512079508,
103765.129364371809061,
103774.548749370704172,
103783.968215502231033,
103793.387762759797624,
103802.807391136768274,
103812.227100626638276,
103821.646891222801059,
103831.066762918679160,
103840.486715707680560,
103849.906749583271449,
103859.326864538787049,
103868.747060567751760,
103878.167337663588114,
103887.587695819645887,
103897.008135029420373,
103906.428655286319554,
103915.849256583780516,
103925.269938915240346,
103934.690702274107025,
103944.111546653846744,
103953.532472047896590,
103962.953478449664544,
103972.374565852631349,
103981.795734250175883,
103991.216983635807992,
104000.638314002921106,
104010.059725344981416,
104019.481217655426008,
104028.902790927706519,
104038.324445155216381,
104047.746180331479991,
104057.167996449905331,
104066.589893503944040,
104076.011871487062308,
104085.433930392682669,
104094.856070214271313,
104104.278290945279878,
104113.700592579160002,
104123.122975109392428,
104132.545438529385137,
104141.967982832647976,
104151.390608012588928,
104160.813314062703284,
104170.236100976442685,
104179.658968747287872,
104189.081917368661379,
104198.504946834029397,
104207.928057136916323,
104217.351248270701035,
104226.774520228922484,
104236.197873005017755,
104245.621306592438486,
104255.044820984694525,
104264.468416175237508,
104273.892092157533625,
104283.315848925092723,
104292.739686471308232,
104302.163604789733654,
104311.587603873806074,
104321.011683717035339,
104330.435844312873087,
104339.860085654785507,
104349.284407736282446,
104358.708810550844646,
104368.133294091923744,
104377.557858353044139,
104386.982503327672021,
104396.407229009288130,
104405.832035391373211,
104415.256922467451659,
104424.681890230960562,
104434.106938675438869,
104443.532067794338218,
104452.957277581183007,
104462.382568029439426,
104471.807939132617321,
104481.233390884211985,
104490.658923277733265,
104500.084536306632799,
104509.510229964449536,
104518.936004244664218,
104528.361859140801243,
104537.787794646326802,
104547.213810754779843,
104556.639907459612004,
104566.066084754391341,
104575.492342632598593,
104584.918681087714504,
104594.345100113307126,
104603.771599702798994,
104613.198179849787266,
104622.624840547723579,
104632.051581790146884,
104641.478403570567025,
104650.905305882493849,
104660.332288719437202,
104669.759352074936032,
104679.186495942471083,
104688.613720315581304,
104698.041025187791092,
104707.468410552624846,
104716.895876403606962,
104726.323422734218184,
104735.751049538026564,
104745.178756808527396,
104754.606544539288734,
104764.034412723776768,
104773.462361355574103,
104782.890390428190585,
104792.318499935136060,
104801.746689869978582,
104811.174960226213443,
104820.603310997394146,
104830.031742177030537,
104839.460253758705221,
104848.888845735913492,
104858.317518102194299,
104867.746270851115696,
104877.175103976202081,
104886.604017470977851,
104896.033011328996508,
104905.462085543797002,
104914.891240108918282,
104924.320475017913850,
104933.749790264351759,
104943.179185841727303,
104952.608661743608536,
104962.038217963578063,
104971.467854495131178,
104980.897571331835934,
104990.327368467289489,
104999.757245894987136,
105009.187203608482378,
105018.617241601372370,
105028.047359867196064,
105037.477558399492409,
105046.907837191844010,
105056.338196237775264,
105065.768635530883330,
105075.199155064692604,
105084.629754832829349,
105094.060434828774305,
105103.491195046153734,
105112.922035478492035,
105122.352956119371811,
105131.783956962361117,
105141.215038001042558,
105150.646199228969635,
105160.077440639710403,
105169.508762226832914,
105178.940163983905222,
105188.371645904539037,
105197.803207982258755,
105207.234850210690638,
105216.666572583344532,
105226.098375093861250,
105235.530257735779742,
105244.962220502697164,
105254.394263388196123,
105263.826386385830119,
105273.258589489239966,
105282.690872691950062,
105292.123235987572116,
105301.555679369688733,
105310.988202831882518,
105320.420806367736077,
105329.853489970861119,
105339.286253634825698,
105348.719097353226971,
105358.152021119662095,
105367.585024927713675,
105377.018108770964318,
105386.451272643069387,
105395.884516537553282,
105405.317840448027709,
105414.751244368133484,
105424.184728291409556,
105433.618292211525841,
105443.051936121992185,
105452.485660016493057,
105461.919463888596511,
105471.353347731914255,
105480.787311540028895,
105490.221355306581245,
105499.655479025168461,
105509.089682689373149,
105518.523966292836121,
105527.958329829139984,
105537.392773291954654,
105546.827296674804529,
105556.261899971374078,
105565.696583175231353,
105575.131346280017169,
105584.566189279328682,
105594.001112166850362,
105603.436114936092054,
105612.871197580738226,
105622.306360094415140,
105631.741602470719954,
105641.176924703293480,
105650.612326785718324,
105660.047808711678954,
105669.483370474757976,
105678.919012068596203,
105688.354733486834448,
105697.790534723084420,
105707.226415770972380,
105716.662376624139142,
105726.098417276210967,
105735.534537720828666,
105744.970737951618503,
105754.407017962192185,
105763.843377746248734,
105773.279817297356203,
105782.716336609199061,
105792.152935675389017,
105801.589614489552332,
105811.026373045388027,
105820.463211336478707,
105829.900129356508842,
105839.337127099104691,
105848.774204557892517,
105858.211361726571340,
105867.648598598723765,
105877.085915168034262,
105886.523311428158195,
105895.960787372692721,
105905.398342995351413,
105914.835978289789637,
105924.273693249604548,
105933.711487868466065,
105943.149362140073208,
105952.587316058023134,
105962.025349616014864,
105971.463462807703763,
105980.901655626701540,
105990.339928066721768,
105999.778280121419812,
106009.216711784436484,
106018.655223049441702,
106028.093813910105382,
106037.532484360068338,
106046.971234393029590,
106056.410064002659055,
106065.848973182597547,
106075.287961926514981,
106084.727030228081276,
106094.166178081010003,
106103.605405478941975,
106113.044712415518006,
106122.484098884451669,
106131.923564879427431,
106141.363110394100659,
106150.802735422155820,
106160.242439957262832,
106169.682223993077059,
106179.122087523355731,
106188.562030541710556,
106198.002053041840554,
106207.442155017430196,
106216.882336462149397,
106226.322597369726282,
106235.762937733816216,
106245.203357548103668,
106254.643856806287658,
106264.084435502067208,
106273.525093629083131,
106282.965831181107205,
106292.406648151751142,
106301.847544534743065,
106311.288520323811099,
106320.729575512581505,
106330.170710094811511,
106339.611924064171035,
106349.053217414359096,
106358.494590139060165,
106367.936042232002364,
106377.377573686870164,
106386.819184497377137,
106396.260874657207751,
106405.702644160090131,
106415.144492999708746,
106424.586421169791720,
106434.028428664008970,
106443.470515476103174,
106452.912681599787902,
106462.354927028733073,
106471.797251756695914,
106481.239655777346343,
106490.682139084441587,
106500.124701671666116,
106509.567343532748055,
106519.010064661400975,
106528.452865051338449,
106537.895744696259499,
106547.338703589906800,
106556.781741725993925,
106566.224859098263551,
106575.668055700400146,
106585.111331526160939,
106594.554686569244950,
106603.998120823380305,
106613.441634282309678,
106622.885226939746644,
106632.328898789419327,
106641.772649825070403,
106651.216480040398892,
106660.660389429205679,
106670.104377985131578,
106679.548445701962919,
106688.992592573427828,
106698.436818593239877,
106707.881123755156295,
106717.325508052919758,
106726.769971480243839,
106736.214514030914870,
106745.659135698602768,
106755.103836477093864,
106764.548616360130836,
106773.993475341441808,
106783.438413414769457,
106792.883430573885562,
106802.328526812503696,
106811.773702124381089,
106821.218956503260415,
106830.664289942913456,
106840.109702437082888,
106849.555193979482283,
106859.000764563897974,
106868.446414184101741,
106877.892142833821708,
106887.337950506800553,
106896.783837196795503,
106906.229802897578338,
106915.675847602935391,
106925.121971306565683,
106934.568174002240994,
106944.014455683762208,
106953.460816344901104,
106962.907255979342153,
106972.353774580886238,
106981.800372143348795,
106991.247048660399741,
107000.693804125883617,
107010.140638533557649,
107019.587551877164515,
107029.034544150475995,
107038.481615347263869,
107047.928765461329021,
107057.375994486399577,
107066.823302416290971,
107076.270689244745881,
107085.718154965565191,
107095.165699572491576,
107104.613323059340473,
107114.061025419840007,
107123.508806647849269,
107132.956666737096384,
107142.404605681338580,
107151.852623474420398,
107161.300720110069960,
107170.748895582102705,
107180.197149884304963,
107189.645483010433963,
107199.093894954319694,
107208.542385709719383,
107217.990955270419363,
107227.439603630220518,
107236.888330782952835,
107246.337136722315336,
107255.786021442196216,
107265.234984936323599,
107274.684027198542026,
107284.133148222608725,
107293.582348002339131,
107303.031626531519578,
107312.480983803980052,
107321.930419813477783,
107331.379934553828207,
107340.829528018875862,
107350.279200202363427,
107359.728951098091784,
107369.178780699934578,
107378.628689001634484,
107388.078675997036044,
107397.528741679896484,
107406.978886044074898,
107416.429109083386720,
107425.879410791603732,
107435.329791162555921,
107444.780250190044171,
107454.230787867913023,
107463.681404189919704,
107473.132099149952410,
107482.582872741782921,
107492.033724959226674,
107501.484655796113657,
107510.935665246273857,
107520.386753303522710,
107529.837919961661100,
107539.289165214533568,
107548.740489055955550,
107558.191891479742480,
107567.643372479724349,
107577.094932049745694,
107586.546570183621952,
107595.998286875183112,
107605.450082118230057,
107614.901955906650983,
107624.353908234203118,
107633.805939094789210,
107643.258048482210143,
107652.710236390295904,
107662.162502812876482,
107671.614847743825521,
107681.067271176900249,
107690.519773106032517,
107699.972353525023209,
107709.425012427702313,
107718.877749807914370,
107728.330565659503918,
107737.783459976300946,
107747.236432752208202,
107756.689483980968362,
107766.142613656498725,
107775.595821772643831,
107785.049108323233668,
107794.502473302112776,
107803.955916703140247,
107813.409438520175172,
107822.863038747047540,
107832.316717377630994,
107841.770474405755522,
107851.224309825280216,
107860.678223630093271,
107870.132215814024676,
107879.586286370918970,
107889.040435294649797,
107898.494662579076248,
107907.948968218071968,
107917.403352205452393,
107926.857814535134821,
107936.312355200949241,
107945.766974196769297,
107955.221671516468632,
107964.676447153877234,
107974.131301102897851,
107983.586233357404126,
107993.041243911240599,
108002.496332758295466,
108011.951499892413267,
108021.406745307482197,
108030.862068997390452,
108040.317470955968020,
108049.772951177132200,
108059.228509654727532,
108068.684146382671315,
108078.139861354808090,
108087.595654564996948,
108097.051526007184293,
108106.507475675185560,
108115.963503562903497,
108125.419609664240852,
108134.875793973042164,
108144.332056483224733,
108153.788397188647650,
108163.244816083228216,
108172.701313160840073,
108182.157888415342313,
108191.614541840666789,
108201.071273430672591,
108210.528083179277019,
108219.984971080339164,
108229.441937127776328,
108238.898981315476703,
108248.356103637313936,
108257.813304087234428,
108267.270582659111824,
108276.727939346805215,
108286.185374144246452,
108295.642887045338284,
108305.100478043968906,
108314.558147134041064,
108324.015894309457508,
108333.473719564150088,
108342.931622891977895,
108352.389604286872782,
108361.847663742722943,
108371.305801253460231,
108380.764016812972841,
108390.222310415163520,
108399.680682053978671,
108409.139131723262835,
108418.597659416991519,
108428.056265129052917,
108437.514948853364331,
108446.973710583857610,
108456.432550314406399,
108465.891468038942548,
108475.350463751412462,
108484.809537445689784,
108494.268689115735469,
108503.727918755437713,
108513.187226358742919,
108522.646611919553834,
108532.106075431802310,
108541.565616889420198,
108551.025236286324798,
108560.484933616418857,
108569.944708873663330,
108579.404562051960966,
108588.864493145301822,
108598.324502147515886,
108607.784589052593219,
108617.244753854465671,
108626.704996547050541,
108636.165317124308785,
108645.625715580143151,
108655.086191908485489,
108664.546746103311307,
108674.007378158523352,
108683.468088068038924,
108692.928875825877185,
108702.389741425911780,
108711.850684862089111,
108721.311706128370133,
108730.772805218701251,
108740.233982126999763,
108749.695236847241176,
108759.156569373342791,
108768.617979699280113,
108778.079467818984995,
108787.541033726403839,
108797.002677415483049,
108806.464398880183580,
108815.926198114437284,
108825.388075112234219,
108834.850029867506237,
108844.312062374214293,
108853.774172626304789,
108863.236360617709579,
108872.698626342404168,
108882.160969794407720,
108891.623390967579326,
108901.085889855952701,
108910.548466453416040,
108920.011120754017611,
108929.473852751674713,
108938.936662440348300,
108948.399549814013881,
108957.862514866603306,
108967.325557592150290,
108976.788677984557580,
108986.251876037844340,
108995.715151745927869,
109005.178505102827330,
109014.641936102489126,
109024.105444738888764,
109033.569031005987199,
109043.032694897789042,
109052.496436408226145,
109061.960255531332223,
109071.424152261010022,
109080.888126591322361,
109090.352178516157437,
109099.816308029563515,
109109.280515125501552,
109118.744799797947053,
109128.209162040875526,
109137.673601848277031,
109147.138119214156177,
109156.602714132459369,
109166.067386597234872,
109175.532136602385435,
109184.996964141959324,
109194.461869209902943,
109203.926851800279110,
109213.391911906990572,
109222.857049524100148,
109232.322264645539690,
109241.787557265342912,
109251.252927377499873,
109260.718374976015184,
109270.183900054820697,
109279.649502608008333,
109289.115182629509945,
109298.580940113359247,
109308.046775053531746,
109317.512687444017502,
109326.978677278879331,
109336.444744552063639,
109345.910889257589588,
109355.377111389476340,
109364.843410941728507,
109374.309787908292492,
109383.776242283260217,
109393.242774060607189,
109402.709383234323468,
109412.176069798428216,
109421.642833746955148,
109431.109675073908875,
109440.576593773279455,
109450.043589839100605,
109459.510663265391486,
109468.977814046171261,
109478.445042175386334,
109487.912347647172282,
109497.379730455475510,
109506.847190594315180,
109516.314728057725006,
109525.782342839738703,
109535.250034934360883,
109544.717804335610708,
109554.185651037521893,
109563.653575034128153,
109573.121576319463202,
109582.589654887502547,
109592.057810732338112,
109601.526043847930850,
109610.994354228372686,
109620.462741867682780,
109629.931206759865745,
109639.399748898984399,
109648.868368279028800,
109658.337064894076320,
109667.805838738146122,
109677.274689805271919,
109686.743618089501979,
109696.212623584855464,
109705.681706285395194,
109715.150866185125778,
109724.620103278124589,
109734.089417558425339,
109743.558809020032641,
109753.028277657052968,
109762.497823463476379,
109771.967446433394798,
109781.437146560827387,
109790.906923839793308,
109800.376778264413588,
109809.846709828663734,
109819.316718526650220,
109828.786804352377658,
109838.256967299908865,
109847.727207363335765,
109857.197524536677520,
109866.667918813996948,
109876.138390189327765,
109885.608938656761893,
109895.079564210333047,
109904.550266844089492,
109914.021046552137705,
109923.491903328496846,
109932.962837167215184,
109942.433848062399193,
109951.904936008097138,
109961.376100998357288,
109970.847343027242459,
109980.318662088830024,
109989.790058177197352,
109999.261531286392710,
110008.733081410493469,
110018.204708543562447,
110027.676412679647910,
110037.148193812899990,
110046.620051937308745,
110056.091987046966096,
110065.563999135963968,
110075.036088198365178,
110084.508254228247097,
110093.980497219672543,
110103.452817166762543,
110112.925214063550811,
110122.397687904129270,
110131.870238682589843,
110141.342866392980795,
110150.815571029437706,
110160.288352586008841,
110169.761211056771572,
110179.234146435803268,
110188.707158717239508,
110198.180247895128559,
110207.653413963547791,
110217.126656916618231,
110226.599976748402696,
110236.073373453007662,
110245.546847024510498,
110255.020397457017680,
110264.494024744592025,
110273.967728881383664,
110283.441509861440863,
110292.915367678870098,
110302.389302327763289,
110311.863313802226912,
110321.337402096367441,
110330.811567204262246,
110340.285809120017802,
110349.760127837740583,
110359.234523351537064,
110368.708995655528270,
110378.183544743747916,
110387.658170610375237,
110397.132873249487602,
110406.607652655191487,
110416.082508821578813,
110425.557441742770607,
110435.032451412916998,
110444.507537826051703,
110453.982700976339402,
110463.457940857857466,
110472.933257464756025,
110482.408650791127002,
110491.884120831076871,
110501.359667578741210,
110510.835291028226493,
110520.310991173653747,
110529.786768009129446,
110539.262621528789168,
110548.738551726724836,
110558.214558597115683,
110567.690642134024529,
110577.166802331586950,
110586.643039183953078,
110596.119352685200283,
110605.595742829507799,
110615.072209610982100,
110624.548753023729660,
110634.025373061871505,
110643.502069719615974,
110652.978842990982230,
110662.455692870207713,
110671.932619351311587,
110681.409622428516741,
110690.886702095929650,
110700.363858347671339,
110709.841091177906492,
110719.318400580727030,
110728.795786550326739,
110738.273249080782989,
110747.750788166245911,
110757.228403800923843,
110766.706095978865051,
110776.183864694263320,
110785.661709941254230,
110795.139631713973358,
110804.617630006585387,
110814.095704813182238,
110823.573856127972249,
110833.052083945090999,
110842.530388258630410,
110852.008769062813371,
110861.487226351746358,
110870.965760119579500,
110880.444370360506582,
110889.923057068619528,
110899.401820238112123,
110908.880659863119945,
110918.359575937807676,
110927.838568456325447,
110937.317637412823387,
110946.796782801495283,
110956.276004616462160,
110965.755302851903252,
110975.234677501954138,
110984.714128560779500,
110994.193656022587675,
111003.673259881470585,
111013.152940131651121,
111022.632696767279413,
111032.112529782505590,
111041.592439171508886,
111051.072424928439432,
111060.552487047505565,
111070.032625522842864,
111079.512840348616010,
111088.993131519018789,
111098.473499028186779,
111107.953942870357423,
111117.434463039637194,
111126.915059530248982,
111136.395732336328365,
111145.876481452112785,
111155.357306871694163,
111164.838208589309943,
111174.319186599153909,
111183.800240895332536,
111193.281371472083265,
111202.762578323585331,
111212.243861444017966,
111221.725220827545854,
111231.206656468377332,
111240.688168360691634,
111250.169756498624338,
111259.651420876456541,
111269.133161488309270,
111278.614978328390862,
111288.096871390880551,
111297.578840670001227,
111307.060886159903021,
111316.543007854779717,
111326.025205748868757,
111335.507479836320272,
111344.989830111371703,
111354.472256568173179,
111363.954759200933040,
111373.437338003874174,
111382.919992971161264,
111392.402724097031751,
111401.885531375635765,
111411.368414801225299,
111420.851374367979588,
111430.334410070106969,
111439.817521901815780,
111449.300709857256152,
111458.783973930709180,
111468.267314116354100,
111477.750730408413801,
111487.234222801038413,
111496.717791288509034,
111506.201435864975792,
111515.685156524690683,
111525.168953261862043,
111534.652826070698211,
111544.136774945378420,
111553.620799880169216,
111563.104900869249832,
111572.589077906872262,
111582.073330987186637,
111591.557660104503157,
111601.042065252957400,
111610.526546426815912,
111620.011103620301583,
111629.495736827608198,
111638.980446042973199,
111648.465231260619476,
111657.950092474784469,
111667.435029679647414,
111676.920042869489407,
111686.405132038504235,
111695.890297180929338,
111705.375538290987606,
111714.860855362916482,
111724.346248390953406,
111733.831717369306716,
111743.317262292228406,
111752.802883153941366,
111762.288579948668485,
111771.774352670661756,
111781.260201314173173,
111790.746125873396522,
111800.232126342598349,
111809.718202716001542,
111819.204354987872648,
111828.690583152390900,
111838.176887203866499,
111847.663267136507784,
111857.149722944566747,
111866.636254622295382,
111876.122862163916579,
111885.609545563682332,
111895.096304815815529,
111904.583139914626372,
111914.070050854279543,
111923.557037629085244,
111933.044100233280915,
111942.531238661103998,
111952.018452906821040,
111961.505742964654928,
111970.993108828901313,
111980.480550493783085,
111989.968067953552236,
111999.455661202518968,
112008.943330234862515,
112018.431075044878526,
112027.918895626818994,
112037.406791974950465,
112046.894764083510381,
112056.382811946794391,
112065.870935559054487,
112075.359134914528113,
112084.847410007510916,
112094.335760832225787,
112103.824187382982927,
112113.312689654019778,
112122.801267639617436,
112132.289921334056999,
112141.778650731561356,
112151.267455826455262,
112160.756336612947052,
112170.245293085361482,
112179.734325237979647,
112189.223433065009885,
112198.712616560776951,
112208.201875719518284,
112217.691210535573191,
112227.180621003135457,
112236.670107116544386,
112246.159668870051973,
112255.649306257939315,
112265.139019274502061,
112274.628807914006757,
112284.118672170719947,
112293.608612038951833,
112303.098627512983512,
112312.588718587096082,
112322.078885255541536,
112331.569127512673731,
112341.059445352701005,
112350.549838769962662,
112360.040307758725248,
112369.530852313313517,
112379.021472427979461,
112388.512168097004178,
112398.002939314741525,
112407.493786075399839,
112416.984708373332978,
112426.475706202851143,
112435.966779558162671,
112445.457928433650522,
112454.949152823581244,
112464.440452722250484,
112473.931828123939340,
112483.423279023016221,
112492.914805413674912,
112502.406407290327479,
112511.898084647182259,
112521.389837478593108,
112530.881665778870229,
112540.373569542265614,
112549.865548763162224,
112559.357603435797500,
112568.849733554525301,
112578.341939113655826,
112587.834220107426518,
112597.326576530249440,
112606.819008376347483,
112616.311515640089056,
112625.804098315769807,
112635.296756397699937,
112644.789489880189649,
112654.282298757563694,
112663.775183024132275,
112673.268142674234696,
112682.761177702137502,
112692.254288102209102,
112701.747473868745146,
112711.240734996055835,
112720.734071478524129,
112730.227483310372918,
112739.720970485999715,
112749.214532999714720,
112758.708170845828135,
112768.201884018664714,
112777.695672512563760,
112787.189536321835476,
112796.683475440833718,
112806.177489863854134,
112815.671579585265135,
112825.165744599362370,
112834.659984900499694,
112844.154300482972758,
112853.648691341179074,
112863.143157469414291,
112872.637698861988611,
112882.132315513284993,
112891.627007417628192,
112901.121774569342961,
112910.616616962739499,
112920.111534592229873,
112929.606527452095179,
112939.101595536689274,
112948.596738840380567,
112958.091957357464707,
112967.587251082339208,
112977.082620009299717,
112986.578064132700092,
112996.073583446923294,
113005.569177946264972,
113015.064847625108087,
113024.560592477791943,
113034.056412498655845,
113043.552307682068204,
113053.048278022382874,
113062.544323513924610,
113072.040444151047268,
113081.536639928148361,
113091.032910839523538,
113100.529256879584864,
113110.025678042642539,
113119.522174323064974,
113129.018745715235127,
113138.515392213477753,
113148.012113812175812,
113157.508910505668609,
113167.005782288353657,
113176.502729154555709,
113185.999751098643173,
113195.496848114984459,
113204.994020197977079,
113214.491267341945786,
113223.988589541244437,
113233.485986790285097,
113242.983459083392518,
113252.481006414978765,
113261.978628779397695,
113271.476326170988614,
113280.974098584163585,
113290.471946013291017,
113299.969868452710216,
113309.467865896818694,
113318.965938339999411,
113328.464085776606225,
113337.962308201065753,
113347.460605607659090,
113356.958977990856511,
113366.457425344997318,
113375.955947664479027,
113385.454544943670044,
113394.953217176924227,
113404.451964358653640,
113413.950786483255797,
113423.449683545099106,
113432.948655538566527,
113442.447702458041022,
113451.946824297920102,
113461.446021052557626,
113470.945292716380209,
113480.444639283770812,
113489.944060749126947,
113499.443557106802473,
113508.943128351224004,
113518.442774476774503,
113527.942495477851480,
113537.442291348823346,
113546.942162084102165,
113556.442107678085449,
113565.942128125170711,
113575.442223419755464,
113584.942393556251773,
113594.442638529013493,
113603.942958332510898,
113613.443352961083292,
113622.943822409157292,
113632.444366671130410,
113641.944985741400160,
113651.445679614378605,
113660.946448284492362,
113670.447291746095289,
113679.948209993643104,
113689.449203021518770,
113698.950270824119798,
113708.451413395887357,
113717.952630731204408,
113727.453922824497567,
113736.955289670193451,
113746.456731262675021,
113755.958247596354340,
113765.459838665643474,
113774.961504464998143,
113784.463244988801307,
113793.965060231465031,
113803.466950187401380,
113812.968914851066074,
113822.470954216842074,
113831.973068279185100,
113841.475257032492664,
113850.977520471147727,
113860.479858589649666,
113869.982271382395993,
113879.484758843784221,
113888.987320968226413,
113898.489957750236499,
113907.992669184139231,
113917.495455264419434,
113926.998315985489171,
113936.501251341760508,
113946.004261327703716,
113955.507345937730861,
113965.010505166268558,
113974.513739007728873,
113984.017047456611181,
113993.520430507283891,
114003.023888154188171,
114012.527420391779742,
114022.031027214514324,
114031.534708616774878,
114041.038464593060780,
114050.542295137784095,
114060.046200245356886,
114069.550179910263978,
114079.054234126931988,
114088.558362889787531,
114098.062566193286330,
114107.566844031869550,
114117.071196399992914,
114126.575623292112141,
114136.080124702610192,
114145.584700626015547,
114155.089351056754822,
114164.594075989225530,
114174.098875417927047,
114183.603749337300542,
114193.108697741830838,
114202.613720625900896,
114212.118817983995541,
114221.623989810555940,
114231.129236100081471,
114240.634556846998748,
114250.139952045748942,
114259.645421690802323,
114269.150965776643716,
114278.656584297685185,
114288.162277248411556,
114297.668044623307651,
114307.173886416785535,
114316.679802623315481,
114326.185793237382313,
114335.691858253485407,
114345.197997666007723,
114354.704211469448637,
114364.210499658307526,
114373.716862227011006,
114383.223299170029350,
114392.729810481876484,
114402.236396156935371,
114411.743056189763593,
114421.249790574802319,
114430.756599306521821,
114440.263482379363268,
114449.770439787869691,
114459.277471526438603,
114468.784577589598484,
114478.291757971834159,
114487.799012667543138,
114497.306341671312111,
114506.813744977538590,
114516.321222580736503,
114525.828774475390674,
114535.336400655956822,
114544.844101116934326,
114554.351875852808007,
114563.859724858048139,
114573.367648127168650,
114582.875645654596156,
114592.383717434902792,
114601.891863462486072,
114611.400083731889026,
114620.908378237581928,
114630.416746974049602,
114639.925189935835078,
114649.433707117350423,
114658.942298513124115,
114668.450964117640979,
114677.959703925414942,
114687.468517930916278,
114696.977406128658913,
114706.486368513127672,
114715.995405078821932,
114725.504515820226516,
114735.013700731869903,
114744.522959808222367,
114754.032293043797836,
114763.541700433081132,
114773.051181970600737,
114782.560737650841475,
114792.070367468317272,
114801.580071417512954,
114811.089849492957001,
114820.599701689134235,
114830.109628000573139,
114839.619628421743982,
114849.129702947204350,
114858.639851571453619,
114868.150074288932956,
114877.660371094258153,
114887.170741981841275,
114896.681186946283560,
114906.191705982026178,
114915.702299083626713,
114925.212966245599091,
114934.723707462442690,
114944.234522728671436,
114953.745412038799259,
114963.256375387354637,
114972.767412768866052,
114982.278524177818326,
114991.789709608798148,
115001.300969056261238,
115010.812302514736075,
115020.323709978765692,
115029.835191442878568,
115039.346746901603183,
115048.858376349424361,
115058.370079780899687,
115067.881857190572191,
115077.393708572955802,
115086.905633922535344,
115096.417633233882952,
115105.929706501541659,
115115.441853720025392,
115124.954074883862631,
115134.466369987567305,
115143.978739025711548,
115153.491181992780184,
115163.003698883359903,
115172.516289691979182,
115182.028954413122847,
115191.541693041392136,
115201.054505571300979,
115210.567391997377854,
115220.080352314165793,
115229.593386516207829,
115239.106494598061545,
115248.619676554255420,
115258.132932379303384,
115267.646262067806674,
115277.159665614279220,
115286.673143013264053,
115296.186694259304204,
115305.700319346957258,
115315.214018270766246,
115324.727791025303304,
115334.241637605082360,
115343.755558004646446,
115353.269552218582248,
115362.783620241432800,
115372.297762067755684,
115381.811977692079381,
115391.326267108946922,
115400.840630312959547,
115410.355067298660288,
115419.869578060606727,
115429.384162593327346,
115438.898820891408832,
115448.413552949394216,
115457.928358761841082,
115467.443238323336118,
115476.958191628436907,
115486.473218671686482,
115495.988319447642425,
115505.503493950891425,
115515.018742176005617,
115524.534064117484377,
115534.049459769972600,
115543.564929128013318,
115553.080472186164116,
115562.596088938968023,
115572.111779381055385,
115581.627543506969232,
115591.143381311267149,
115600.659292788521270,
115610.175277933332836,
115619.691336740244878,
115629.207469203844084,
115638.723675318717142,
115648.239955079392530,
115657.756308480500593,
115667.272735516628018,
115676.789236182288732,
115686.305810472113080,
115695.822458380673197,
115705.339179902541218,
115714.855975032289280,
115724.372843764518620,
115733.889786093801376,
115743.406802014724235,
115752.923891521859332,
115762.441054609822459,
115771.958291273171199,
115781.475601506506791,
115790.992985304386821,
115800.510442661470734,
115810.027973572287010,
115819.545578031451441,
115829.063256033521611,
115838.581007573127863,
115848.098832644856884,
115857.616731243280810,
115867.134703363015433,
115876.652748998661991,
115886.170868144778069,
115895.689060796023114,
115905.207326946911053,
115914.725666592101334,
115924.244079726180644,
115933.762566343750223,
115943.281126439396758,
115952.799760007721488,
115962.318467043340206,
115971.837247540854150,
115981.356101494850009,
115990.875028899958124,
116000.394029750779737,
116009.913104041901533,
116019.432251767939306,
116028.951472923523397,
116038.470767503225943,
116047.990135501662735,
116057.509576913464116,
116067.029091733216774,
116076.548679955565603,
116086.068341575111845,
116095.588076586427633,
116105.107884984186967,
116114.627766762976535,
116124.147721917397575,
116133.667750442080433,
116143.187852331670001,
116152.708027580723865,
116162.228276183901471,
116171.748598135833163,
116181.268993431134732,
116190.789462064392865,
116200.310004030223354,
116209.830619323314750,
116219.351307938239188,
116228.872069869656116,
116238.392905112137669,
116247.913813660343294,
116257.434795508903335,
116266.955850652448134,
116276.476979085593484,
116285.998180802969728,
116295.519455799192656,
116305.040804068936268,
116314.562225606787251,
116324.083720407390501,
116333.605288465390913,
116343.126929775433382,
116352.648644332119147,
116362.170432130107656,
116371.692293164000148,
116381.214227428485174,
116390.736234918163973,
116400.258315627666889,
116409.780469551697024,
116419.302696684797411,
116428.824997021700256,
116438.347370557006798,
116447.869817285318277,
116457.392337201352348,
116466.914930299724801,
116476.437596575065982,
116485.960336022006231,
116495.483148635263206,
116505.006034409423592,
116514.528993339117733,
116524.052025419048732,
116533.575130643861485,
116543.098309008157230,
116552.621560506624519,
116562.144885133937350,
116571.668282884696964,
116581.191753753577359,
116590.715297735223430,
116600.238914824323729,
116609.762605015479494,
116619.286368303422933,
116628.810204682740732,
116638.334114148092340,
116647.858096694224514,
116657.382152315680287,
116666.906281007191865,
116676.430482763404143,
116685.954757578991121,
116695.479105448583141,
116705.003526366839651,
116714.528020328507409,
116724.052587328144000,
116733.577227360481629,
116743.101940420136089,
116752.626726501854137,
116762.151585600237013,
116771.676517709958716,
116781.201522825722350,
116790.726600942172809,
116800.251752053998644,
116809.776976155844750,
116819.302273242399679,
116828.827643308381084,
116838.353086348361103,
116847.878602357101045,
116857.404191329245805,
116866.929853259469382,
116876.455588142460329,
116885.981395972892642,
116895.507276745454874,
116905.033230454821023,
116914.559257095650537,
116924.085356662661070,
116933.611529150512069,
116943.137774553877534,
116952.664092867460568,
116962.190484085920616,
116971.716948203989887,
116981.243485216313275,
116990.770095117593883,
117000.296777902534814,
117009.823533565766411,
117019.350362102049985,
117028.877263506015879,
117038.404237772410852,
117047.931284895879799,
117057.458404871111270,
117066.985597692852025,
117076.512863355732406,
117086.040201854499173,
117095.567613183826325,
117105.095097338387859,
117114.622654312945087,
117124.150284102128353,
117133.677986700655310,
117143.205762103229063,
117152.733610304552712,
117162.261531299314811,
117171.789525082247565,
117181.317591648024973,
117190.845730991350138,
117200.373943106940715,
117209.902227989499806,
117219.430585633715964,
117228.959016034306842,
117238.487519186004647,
117248.016095083454275,
117257.544743721446139,
117267.073465094639687,
117276.602259197708918,
117286.131126025429694,
117295.660065572505118,
117305.189077833623742,
117314.718162803488667,
117324.247320476861205,
117333.776550848415354,
117343.305853912868770,
117352.835229664968210,
117362.364678099358571,
117371.894199210873921,
117381.423792994130054,
117390.953459443902830,
117400.483198554866249,
117410.013010321796173,
117419.542894739395706,
117429.072851802324294,
117438.602881505372352,
117448.132983843286638,
117457.663158810741152,
117467.193406402468099,
117476.723726613156032,
117486.254119437624468,
117495.784584870532854,
117505.315122906671604,
117514.845733540685615,
117524.376416767350747,
117533.907172581399209,
117543.438000977548654,
117552.968901950531290,
117562.499875495108427,
117572.030921606012271,
117581.562040277916822,
117591.093231505597942,
117600.624495283787837,
117610.155831607262371,
117619.687240470695542,
117629.218721868877765,
117638.750275796541246,
117648.281902248359984,
117657.813601219138945,
117667.345372703610337,
117676.877216696491814,
117686.409133192573790,
117695.941122186588473,
117705.473183673238964,
117715.005317647286574,
117724.537524103521719,
117734.069803036618396,
117743.602154441367020,
117753.134578312528902,
117762.667074644807144,
117772.199643432948506,
117781.732284671801608,
117791.264998356025899,
117800.797784480382688,
117810.330643039633287,
117819.863574028568109,
117829.396577441890258,
117838.929653274361044,
117848.462801520770881,
117857.996022175851976,
117867.529315234365640,
117877.062680691044079,
117886.596118540706811,
117896.129628778056940,
117905.663211397913983,
117915.196866394951940,
117924.730593763990328,
117934.264393499790458,
117943.798265597113641,
117953.332210050721187,
117962.866226855374407,
117972.400316005863715,
117981.934477496863110,
117991.468711323250318,
118001.003017479786649,
118010.537395961175207,
118020.071846762206405,
118029.606369877685211,
118039.140965302343830,
118048.675633030958124,
118058.210373058318510,
118067.745185379186296,
118077.280069988381001,
118086.815026880576625,
118096.350056050650892,
118105.885157493306906,
118115.420331203364185,
118124.955577175584040,
118134.490895404756884,
118144.026285885658581,
118153.561748613050440,
118163.097283581766533,
118172.632890786480857,
118182.168570222100243,
118191.704321883356897,
118201.240145765012130,
118210.776041861827252,
118220.312010168694542,
118229.848050680317101,
118239.384163391470793,
118248.920348297004239,
118258.456605391678750,
118267.992934670226532,
118277.529336127568968,
118287.065809758350952,
118296.602355557464762,
118306.138973519642605,
118315.675663639703998,
118325.212425912439357,
118334.749260332653648,
118344.286166895166389,
118353.823145594695234,
118363.360196426059701,
118372.897319384108414,
118382.434514463646337,
118391.971781659376575,
118401.509120966162300,
118411.046532378852135,
118420.584015892163734,
118430.121571500916616,
118439.659199199944851,
118449.196898984009749,
118458.734670847974485,
118468.272514786571264,
118477.810430794663262,
118487.348418867011787,
118496.886478998509119,
118506.424611183858360,
118515.962815417922684,
118525.501091695492505,
118535.039440011401894,
118544.577860360441264,
118554.116352737386478,
118563.654917137158918,
118573.193553554461687,
118582.732261984157958,
118592.271042421081802,
118601.809894859994529,
118611.348819295730209,
118620.887815723122912,
118630.426884136963054,
118639.966024532142910,
118649.505236903380137,
118659.044521245552460,
118668.583877553464845,
118678.123305821980466,
118687.662806045860634,
118697.202378219939419,
118706.742022339050891,
118716.281738398014568,
118725.821526391664520,
118735.361386314863921,
118744.901318162344978,
118754.441321929043625,
118763.981397609692067,
118773.521545199153479,
118783.061764692291035,
118792.602056083895150,
118802.142419368799892,
118811.682854541897541,
118821.223361597920302,
118830.763940531775006,
118840.304591338222963,
118849.845314012200106,
118859.386108548467746,
118868.926974941889057,
118878.467913187298109,
118888.008923279514420,
118897.550005213415716,
118907.091158983792411,
118916.632384585522232,
118926.173682013453799,
118935.715051262377528,
118945.256492327171145,
118954.798005202654167,
118964.339589883704321,
118973.881246365170227,
118983.422974641842302,
118992.964774708612822,
119002.506646560315858,
119012.048590191800031,
119021.590605597943068,
119031.132692773506278,
119040.674851713454700,
119050.217082412535092,
119059.759384865668835,
119069.301759067646344,
119078.844205013374449,
119088.386722697701771,
119097.929312115447829,
119107.471973261490348,
119117.014706130677951,
119126.557510717888363,
119136.100387017941102,
119145.643335025713895,
119155.186354736084468,
119164.729446143872337,
119174.272609243998886,
119183.815844031239976,
119193.359150500502437,
119202.902528646634892,
119212.445978464544169,
119221.989499949035235,
119231.533093095000368,
119241.076757897317293,
119250.620494350805529,
119260.164302450401010,
119269.708182190908701,
119279.252133567220881,
119288.796156574244378,
119298.340251206740504,
119307.884417459703400,
119317.428655327938031,
119326.972964806278469,
119336.517345889689750,
119346.061798573005944,
119355.606322851090226,
119365.150918718791218,
119374.695586171059404,
119384.240325202699751,
119393.785135808648192,
119403.330017983709695,
119412.874971722805640,
119422.419997020871961,
119431.965093872655416,
119441.510262273164699,
119451.055502217190224,
119460.600813699653372,
119470.146196715446422,
119479.691651259447099,
119489.237177326518577,
119498.782774911567685,
119508.328444009457598,
119517.874184615066042,
119527.419996723343502,
119536.965880329124047,
119546.511835427314509,
119556.057862012778060,
119565.603960080450634,
119575.150129625166301,
119584.696370641860995,
119594.242683125426993,
119603.789067070742021,
119613.335522472698358,
119622.882049326173728,
119632.428647626089514,
119641.975317367352545,
119651.522058544840547,
119661.068871153416694,
119670.615755188046023,
119680.162710643591709,
119689.709737514931476,
119699.256835797015810,
119708.804005484693334,
119718.351246572914533,
119727.898559056557133,
119737.445942930527963,
119746.993398189733853,
119756.540924829037976,
119766.088522843405372,
119775.636192227757419,
119785.183932976913638,
119794.731745085839066,
119804.279628549440531,
119813.827583362610312,
119823.375609520284343,
119832.923707017325796,
119842.471875848699710,
119852.020116009283811,
119861.568427494014031,
119871.116810297797201,
119880.665264415496495,
119890.213789842106053,
119899.762386572489049,
119909.311054601581418,
119918.859793924289988,
119928.408604535550694,
119937.957486430299468,
119947.506439603414037,
119957.055464049772127,
119966.604559764426085,
119976.153726742166327,
119985.702964977943338,
119995.252274466751260,
120004.801655203453265,
120014.351107182985288,
120023.900630400283262,
120033.450224850224913,
120042.999890527833486,
120052.549627427957603,
120062.099435545504093,
120071.649314875496202,
120081.199265412782552,
120090.749287152328179,
120100.299380089039914,
120109.849544217882794,
120119.399779533763649,
120128.950086031589308,
120138.500463706353912,
120148.050912552964292,
120157.601432566356380,
120167.152023741466110,
120176.702686073200312,
120186.253419556553126,
120195.804224186402280,
120205.355099957741913,
120214.906046865478856,
120224.457064904578147,
120234.008154069917509,
120243.559314356520190,
120253.110545759263914,
120262.661848273142823,
120272.213221893063746,
120281.764666613991722,
120291.316182430833578,
120300.867769338612561,
120310.419427332206396,
120319.971156406580121,
120329.522956556713325,
120339.074827777483733,
120348.626770063900040,
120358.178783410898177,
120367.730867813443183,
120377.283023266441887,
120386.835249764859327,
120396.387547303689644,
120405.939915877868771,
120415.492355482332641,
120425.044866112031741,
120434.597447761960211,
120444.150100427039433,
120453.702824102234445,
120463.255618782510282,
120472.808484462831984,
120482.361421138135483,
120491.914428803385817,
120501.467507453562575,
120511.020657083630795,
120520.573877688511857,
120530.127169263199903,
120539.680531802659971,
120549.233965301857097,
120558.787469755741768,
120568.341045159308123,
120577.894691507477546,
120587.448408795244177,
120597.002197017573053,
120606.556056169458316,
120616.109986245821347,
120625.663987241627183,
120635.218059151913621,
120644.772201971616596,
120654.326415695686592,
120663.880700319146854,
120673.435055836889660,
120682.989482243996463,
120692.543979535315884,
120702.098547705943929,
120711.653186750787427,
120721.207896664825967,
120730.762677443082794,
120740.317529080508393,
120749.872451572067803,
120759.427444912740611,
120768.982509097550064,
120778.537644121446647,
120788.092849979395396,
120797.648126666375902,
120807.203474177440512,
120816.758892507510609,
120826.314381651609438,
120835.869941604658379,
120845.425572361724335,
120854.981273917714134,
120864.537046267694677,
120874.092889406616450,
120883.648803329473594,
120893.204788031216594,
120902.760843506897800,
120912.316969751467695,
120921.873166759949527,
120931.429434527322883,
120940.985773048596457,
120950.542182318706182,
120960.098662332704407,
120969.655213085556170,
120979.211834572299267,
120988.768526787913288,
120998.325289727363270,
121007.882123385657906,
121017.439027757834992,
121026.996002838830464,
121036.553048623711220,
121046.110165107456851,
121055.667352285061497,
121065.224610151519300,
121074.781938701853505,
121084.339337931014597,
121093.896807834098581,
121103.454348406041390,
121113.011959641895373,
121122.569641536625568,
121132.127394085240667,
121141.685217282807571,
121151.243111124262214,
121160.801075604686048,
121170.359110718985903,
121179.917216462301440,
121189.475392829539487,
121199.033639815737843,
121208.591957415963407,
121218.150345625181217,
121227.708804438414518,
121237.267333850686555,
121246.825933856991469,
121256.384604452381609,
121265.943345631851116,
121275.502157390394132,
121285.061039723092108,
121294.619992624910083,
121304.179016090900404,
121313.738110116057214,
121323.297274695432861,
121332.856509824021487,
121342.415815496875439,
121351.975191708988859,
121361.534638455414097,
121371.094155731145293,
121380.653743531234795,
121390.213401850676746,
121399.773130684552598,
121409.332930027827388,
121418.892799875611672,
121428.452740222841385,
121438.012751064627082,
121447.572832395904697,
121457.132984211813891,
121466.693206507334253,
121476.253499277445371,
121485.813862517316011,
121495.374296221853001,
121504.934800386166899,
121514.495375005222741,
121524.056020074160188,
121533.616735587929725,
121543.177521541569149,
121552.738377930159913,
121562.299304748725262,
121571.860301992317545,
121581.421369655974559,
121590.982507734734099,
121600.543716223590309,
121610.104995117653743,
121619.666344411976752,
121629.227764101538924,
121638.789254181378055,
121648.350814646633808,
121657.912445492256666,
121667.474146713342634,
121677.035918304944062,
121686.597760262084194,
121696.159672579815378,
121705.721655253204517,
121715.283708277289406,
121724.845831647122395,
121734.408025357770384,
121743.970289404242067,
121753.532623781647999,
121763.095028485025978,
121772.657503509384696,
121782.220048849834711,
121791.782664501413819,
121801.345350459145266,
121810.908106718125055,
121820.470933273434639,
121830.033830120082712,
121839.596797253136174,
121849.159834667705582,
121858.722942358799628,
121868.286120321470662,
121877.849368550814688,
121887.412687041884055,
121896.976075789745664,
121906.539534789437312,
121916.103064036069554,
121925.666663524680189,
121935.230333250350668,
121944.794073208118789,
121954.357883393095108,
121963.921763800317422,
121973.485714424867183,
121983.049735261825845,
121992.613826306202100,
122002.177987553135608,
122011.742218997693271,
122021.306520634898334,
122030.870892459846800,
122040.435334467649227,
122049.999846653343411,
122059.564429012039909,
122069.129081538761966,
122078.693804228591034,
122088.258597076652222,
122097.823460077997879,
122107.388393227724009,
122116.953396520882961,
122126.518469952556188,
122136.083613517839694,
122145.648827211785829,
122155.214111029548803,
122164.779464966122760,
122174.344889016676461,
122183.910383176204050,
122193.475947439845186,
122203.041581802666769,
122212.607286259779357,
122222.173060806278954,
122231.738905437188805,
122241.304820147677674,
122250.870804932797910,
122260.436859787601861,
122270.002984707229189,
122279.569179686746793,
122289.135444721265230,
122298.701779805909609,
122308.268184935674071,
122317.834660105727380,
122327.401205311180092,
122336.967820547084557,
122346.534505808522226,
122356.101261090632761,
122365.668086388497613,
122375.234981697227340,
122384.801947011874290,
122394.368982327578124,
122403.936087639463949,
122413.503262942555011,
122423.070508232034626,
122432.637823502955143,
122442.205208750427119,
122451.772663969590212,
122461.340189155496773,
122470.907784303257358,
122480.475449408040731,
122490.043184464870137,
122499.610989468885236,
122509.178864415225689,
122518.746809298972948,
122528.314824115208467,
122537.882908859101008,
122547.451063525673817,
122557.019288110153866,
122566.587582607564400,
122576.155947013045079,
122585.724381321706460,
122595.292885528688203,
122604.861459629071760,
122614.430103617967688,
122623.998817490515648,
122633.567601241826196,
122643.136454867009888,
122652.705378361206385,
122662.274371719511691,
122671.843434937036363,
122681.412568008905509,
122690.981770930287894,
122700.551043696235865,
122710.120386301889084,
122719.689798742445419,
122729.259281012899010,
122738.828833108476829,
122748.398455024260329,
122757.968146755360067,
122767.537908296944806,
122777.107739644096000,
122786.677640792026068,
122796.247611735758255,
122805.817652470475878,
122815.387762991289492,
122824.957943293367862,
122834.528193371763336,
122844.098513221702888,
122853.668902838238864,
122863.239362216554582,
122872.809891351789702,
122882.380490239011124,
122891.951158873445820,
122901.521897250146139,
122911.092705364309950,
122920.663583211047808,
122930.234530785484822,
122939.805548082789755,
122949.376635098058614,
122958.947791826460161,
122968.519018263163161,
122978.090314403278171,
122987.661680241930299,
122997.233115774259204,
123006.804620995448204,
123016.376195900607854,
123025.947840484906919,
123035.519554743499612,
123045.091338671496487,
123054.663192264022655,
123064.235115516305086,
123073.807108423425234,
123083.379170980566414,
123092.951303182868287,
123102.523505025485065,
123112.095776503527304,
123121.668117612207425,
123131.240528346635983,
123140.813008701967192,
123150.385558673398918,
123159.958178256027168,
123169.530867445020704,
123179.103626235562842,
123188.676454622793244,
123198.249352601895225,
123207.822320167950238,
123217.395357316199807,
123226.968464041725383,
123236.541640339768492,
123246.114886205439689,
123255.688201633893186,
123265.261586620341404,
123274.835041159865796,
123284.408565247707884,
123293.982158878992777,
123303.555822048860136,
123313.129554752493277,
123322.703356985133723,
123332.277228741833824,
123341.851170017805998,
123351.425180808262667,
123360.999261108285282,
123370.573410913086263,
123380.147630217878032,
123389.721919017742039,
123399.296277307890705,
123408.870705083536450,
123418.445202339789830,
123428.019769071863266,
123437.594405274940073,
123447.169110944101703,
123456.743886074647889,
123466.318730661674635,
123475.893644700379809,
123485.468628185961279,
123495.043681113558705,
123504.618803478355403,
123514.193995275563793,
123523.769256500294432,
123533.344587147832499,
123542.919987213288550,
123552.495456691845902,
123562.070995578687871,
123571.646603869041428,
123581.222281558046234,
123590.798028640871053,
123600.373845112728304,
123609.949730968830409,
123619.525686204317026,
123629.101710814386024,
123638.677804794235271,
123648.253968139033532,
123657.830200844007777,
123667.406502904297668,
123676.982874315130175,
123686.559315071703168,
123696.135825169185409,
123705.712404602760216,
123715.289053367640008,
123724.865771459022653,
123734.442558872106019,
123744.019415602029767,
123753.596341644079075,
123763.173336993335397,
123772.750401645142119,
123782.327535594551591,
123791.904738836878096,
123801.482011367232190,
123811.059353180869948,
123820.636764272931032,
123830.214244638729724,
123839.791794273347477,
123849.369413172040367,
123858.947101330049918,
123868.524858742472134,
123878.102685404621297,
123887.680581311607966,
123897.258546458717319,
123906.836580841089017,
123916.414684454008238,
123925.992857292614644,
123935.571099352135207,
123945.149410627811449,
123954.727791114797583,
123964.306240808349685,
123973.884759703636519,
123983.463347795885056,
123993.042005080351373,
124002.620731552204234,
124012.199527206656057,
124021.778392038919264,
124031.357326044220827,
124040.936329217787716,
124050.515401554817799,
124060.094543050523498,
124069.673753700117231,
124079.253033498884179,
124088.832382441978552,
124098.411800524583668,
124107.991287742013810,
124117.570844089408638,
124127.150469562024227,
124136.730164155102102,
124146.309927863825578,
124155.889760683450731,
124165.469662609219085,
124175.049633636284852,
124184.629673759933212,
124194.209782975347480,
124203.789961277769180,
124213.370208662454388,
124222.950525124615524,
124232.530910659465007,
124242.111365262258914,
124251.691888928180560,
124261.272481652529677,
124270.853143430475029,
124280.433874257287243,
124290.014674128164188,
124299.595543038376491,
124309.176480983122019,
124318.757487957671401,
124328.338563957207953,
124337.919708977045957,
124347.500923012339626,
124357.082206058403244,
124366.663558110420126,
124376.244979163646349,
124385.826469213323435,
124395.408028254678356,
124404.989656282952637,
124414.571353293402353,
124424.153119281269028,
124433.734954241779633,
124443.316858170175692,
124452.898831061742385,
124462.480872911663027,
124472.062983715222799,
124481.645163467663224,
124491.227412164211273,
124500.809729800137575,
124510.392116370669100,
124519.974571871061926,
124529.557096296586678,
124539.139689642470330,
124548.722351903983508,
124558.305083076324081,
124567.887883154806332,
124577.470752134642680,
124587.053690011089202,
124596.636696779416525,
124606.219772434880724,
124615.802916972708772,
124625.386130388214951,
124634.969412676582579,
124644.552763833125937,
124654.136183853057446,
124663.719672731662286,
124673.303230464225635,
124682.886857045916258,
124692.470552472062991,
124702.054316737951012,
124711.638149838792742,
124721.222051769873360,
124730.806022526419838,
124740.390062103746459,
124749.974170497109299,
124759.558347701735329,
124769.142593712909729,
124778.726908525874023,
124788.311292135942494,
124797.895744538414874,
124807.480265728430822,
124817.064855701362831,
124826.649514452466974,
124836.234241976999328,
124845.819038270186866,
124855.403903327372973,
124864.988837143784622,
124874.573839714706992,
124884.158911035425263,
124893.744051101210061,
124903.329259907288360,
124912.914537449018098,
124922.499883721597143,
124932.085298720368883,
124941.670782440589392,
124951.256334877485642,
124960.841956026371918,
124970.427645882591605,
124980.013404441313469,
124989.599231697880896,
124999.185127647549962,
125008.771092285605846,
125018.357125607362832,
125027.943227608076995,
125037.529398283033515,
125047.115637627517572,
125056.701945636814344,
125066.288322306209011,
125075.874767630986753,
125085.461281606461853,
125095.047864227861282,
125104.634515490499325,
125114.221235389661160,
125123.808023920690175,
125133.394881078842445,
125142.981806859374046,
125152.568801257570158,
125162.155864268803271,
125171.742995888300356,
125181.330196111375699,
125190.917464933314477,
125200.504802349445526,
125210.092208354995819,
125219.679682945323293,
125229.267226115684025,
125238.854837861392298,
125248.442518177762395,
125258.030267060050392,
125267.618084503599675,
125277.205970503666322,
125286.793925055593718,
125296.381948154667043,
125305.970039796200581,
125315.558199975450407,
125325.146428687803564,
125334.734725928457920,
125344.323091692771413,
125353.911525976058329,
125363.500028773632948,
125373.088600080722244,
125382.677239892756916,
125392.265948204963934,
125401.854725012672134,
125411.443570311166695,
125421.032484095776454,
125430.621466361815692,
125440.210517104569590,
125449.799636319410638,
125459.388824001594912,
125468.978080146451248,
125478.567404749293928,
125488.156797805422684,
125497.746259310151800,
125507.335789258824661,
125516.925387646770105,
125526.515054469215102,
125536.104789721575798,
125545.694593399108271,
125555.284465497155907,
125564.874406011062092,
125574.464414936097455,
125584.054492267605383,
125593.644638000914711,
125603.234852131339721,
125612.825134654180147,
125622.415485564808478,
125632.005904858495342,
125641.596392530584126,
125651.186948576418217,
125660.777572991326451,
125670.368265770594007,
125679.959026909549721,
125689.549856403566082,
125699.140754247971927,
125708.731720438023331,
125718.322754969107336,
125727.913857836538227,
125737.505029035673942,
125747.096268561828765,
125756.687576410302427,
125766.278952576438314,
125775.870397055652575,
125785.461909843172180,
125795.053490934325964,
125804.645140324559179,
125814.236858009098796,
125823.828643983331858,
125833.420498242616304,
125843.012420782237314,
125852.604411597552826,
125862.196470683906227,
125871.788598036626354,
125881.380793651085696,
125890.973057522583986,
125900.565389646479161,
125910.157790018114611,
125919.750258632819168,
125929.342795485979877,
125938.935400572881917,
125948.528073888912331,
125958.120815429399954,
125967.713625189688173,
125977.306503165134927,
125986.899449351069052,
125996.492463742863038,
126006.085546335831168,
126015.678697125345934,
126025.271916106765275,
126034.865203275418025,
126044.458558626676677,
126054.051982155840960,
126063.645473858356127,
126073.239033729507355,
126082.832661764638033,
126092.426357959120651,
126102.020122308342252,
126111.613954807602568,
126121.207855452303193,
126130.801824237787514,
126140.395861159369815,
126149.989966212480795,
126159.584139392463840,
126169.178380694618681,
126178.772690114361467,
126188.367067647035583,
126197.961513287984417,
126207.556027032609563,
126217.150608876225306,
126226.745258814233239,
126236.339976841991302,
126245.934762954857433,
126255.529617148175021,
126265.124539417345659,
126274.719529757712735,
126284.314588164677843,
126293.909714633526164,
126303.504909159717499,
126313.100171738566132,
126322.695502365473658,
126332.290901035783463,
126341.886367744868039,
126351.481902488128981,
126361.077505260909675,
126370.673176058524405,
126380.268914876491181,
126389.864721710036974,
126399.460596554650692,
126409.056539405646618,
126418.652550258397241,
126428.248629108304158,
126437.844775950739859,
126447.440990781018627,
126457.037273594629369,
126466.633624386886368,
126476.230043153191218,
126485.826529888872756,
126495.423084589376231,
126505.019707250015927,
126514.616397866251646,
126524.213156433426775,
126533.809982946884702,
126543.406877402085229,
126553.003839794371743,
126562.600870119116735,
126572.197968371765455,
126581.795134547617636,
126591.392368642103975,
126600.989670650626067,
126610.587040568556404,
126620.184478391282028,
126629.781984114189981,
126639.379557732681860,
126648.977199242159259,
126658.574908637965564,
126668.172685915546026,
126677.770531070229481,
126687.368444097504835,
126696.966424992628163,
126706.564473751146579,
126716.162590368388919,
126725.760774839684018,
126735.359027160520782,
126744.957347326286254,
126754.555735332309268,
126764.154191174078733,
126773.752714846923482,
126783.351306346259662,
126792.949965667517972,
126802.548692806085455,
126812.147487757290946,
126821.746350516681559,
126831.345281079557026,
126840.944279441304388,
126850.543345597383450,
126860.142479543181253,
126869.741681274084840,
126879.340950785524910,
126888.940288072917610,
126898.539693131620879,
126908.139165957094519,
126917.738706544725574,
126927.338314889886533,
126936.937990988051752,
126946.537734834564617,
126956.137546424928587,
126965.737425754443393,
126975.337372818597942,
126984.937387612779276,
126994.537470132388989,
127004.137620372872334,
127013.737838329630904,
127023.338123998051742,
127032.938477373565547,
127042.538898451617570,
127052.139387227580301,
127061.739943696898990,
127071.340567855004338,
127080.941259697254281,
127090.542019219137728,
127100.142846416041721,
127109.743741283396957,
127119.344703816634137,
127128.945734011111199,
127138.546831862317049,
127148.147997365667834,
127157.749230516565149,
127167.350531310439692,
127176.951899742736714,
127186.553335808857810,
127196.154839504248230,
127205.756410824280465,
127215.358049764472526,
127224.959756320153247,
127234.561530486811535,
127244.163372259892640,
127253.765281634798157,
127263.367258606929681,
127272.969303171776119,
127282.571415324724512,
127292.173595061205560,
127301.775842376693618,
127311.378157266619382,
127320.980539726384450,
127330.582989751390414,
127340.185507337155286,
127349.788092479066108,
127359.390745172582683,
127368.993465413092053,
127378.596253196112229,
127388.199108517044806,
127397.802031371291378,
127407.405021754311747,
127417.008079661565716,
127426.611205088513088,
127436.214398030540906,
127445.817658483108971,
127455.420986441662535,
127465.024381901690504,
127474.627844858594472,
127484.231375307776034,
127493.834973244782304,
127503.438638664956670,
127513.042371563817142,
127522.646171936765313,
127532.250039779290091,
127541.853975086793071,
127551.457977854748606,
127561.062048078616499,
127570.666185753842001,
127580.270390875855810,
127589.874663440146833,
127599.479003442116664,
127609.083410877254209,
127618.687885740961065,
127628.292428028784343,
127637.897037736125640,
127647.501714858415653,
127657.106459391114186,
127666.711271329739247,
127676.316150669692433,
127685.921097406462650,
127695.526111535451491,
127705.131193052176968,
127714.736341952069779,
127724.341558230589726,
127733.946841883196612,
127743.552192905379343,
127753.157611292597721,
127762.763097040282446,
127772.368650143893319,
127781.974270598919247,
127791.579958400820033,
127801.185713545026374,
127810.791536027085385,
127820.397425842369557,
127830.003382986396900,
127839.609407454627217,
127849.215499242520309,
127858.821658345565083,
127868.427884759206790,
127878.034178478905233,
127887.640539500163868,
127897.246967818442499,
127906.853463429200929,
127916.460026327898959,
127926.066656510054599,
127935.673353971127653,
127945.280118706548819,
127954.886950711836107,
127964.493849982449319,
127974.100816513848258,
127983.707850301565486,
127993.314951341002597,
128002.922119627692155,
128012.529355157123064,
128022.136657924696920,
128031.744027925960836,
128041.351465156374616,
128050.958969611427165,
128060.566541286578286,
128070.174180177331436,
128079.781886279175524,
128089.389659587526694,
128098.997500097961165,
128108.605407805938739,
128118.213382706890116,
128127.821424796362408,
128137.429534069829970,
128147.037710522723501,
128156.645954150633770,
128166.254264948933269,
128175.862642913169111,
128185.471088038859307,
128195.079600321478210,
128204.688179756456520,
128214.296826339326799,
128223.905540065607056,
128233.514320930757094,
128243.123168930280372,
128252.732084059665794,
128262.341066314387717,
128271.950115689964150,
128281.559232181883999,
128291.168415785679827,
128300.777666496811435,
128310.386984310753178,
128319.996369223023066,
128329.605821229139110,
128339.215340324575664,
128348.824926504850737,
128358.434579765438684,
128368.044300101886620,
128377.654087509639794,
128387.263941984216217,
128396.873863521148451,
128406.483852115925401,
128416.093907764021424,
128425.704030460983631,
128435.314220202257275,
128444.924476983418572,
128454.534800799941877,
128464.145191647345200,
128473.755649521073792,
128483.366174416703871,
128492.976766329738894,
128502.587425255667767,
128512.198151190023054,
128521.808944128279109,
128531.419804065983044,
128541.030730998609215,
128550.641724921690184,
128560.252785830743960,
128569.863913721274002,
128579.475108588798321,
128589.086370428849477,
128598.697699236916378,
128608.309095008502482,
128617.920557739140349,
128627.532087424347992,
128637.143684059687075,
128646.755347640602849,
128656.367078162642429,
128665.978875621338375,
128675.590740012194146,
128685.202671330698649,
128694.814669572457206,
128704.426734732929617,
128714.038866807648446,
128723.651065792117151,
128733.263331681941054,
128742.875664472507196,
128752.488064159464557,
128762.100530738272937,
128771.713064204494003,
128781.325664553631213,
128790.938331781202578,
128800.551065882784314,
128810.163866853836225,
128819.776734689949080,
128829.389669386597234,
128839.002670939356904,
128848.615739343746100,
128858.228874595253728,
128867.842076689470559,
128877.455345621929155,
128887.068681388089317,
128896.682083983556367,
128906.295553403848317,
128915.909089644483174,
128925.522692701008054,
128935.136362568970071,
128944.750099243887234,
128954.363902721263003,
128963.977772996746353,
128973.591710065767984,
128983.205713923933217,
128992.819784566672752,
129002.433921989708324,
129012.048126188383321,
129021.662397158419481,
129031.276734895189293,
129040.891139394385391,
129050.505610651438474,
129060.120148661982967,
129069.734753421478672,
129079.349424925501808,
129088.964163169628591,
129098.578968149377033,
129108.193839860323351,
129117.808778297956451,
129127.423783457881655,
129137.038855335616972,
129146.653993926709518,
129156.269199226706405,
129165.884471231183852,
129175.499809935688972,
129185.115215335739776,
129194.730687426912482,
129204.346226204783306,
129213.961831664870260,
129223.577503802720457,
129233.193242613910115,
129242.809048093986348,
129252.424920238554478,
129262.040859043059754,
129271.656864503151155,
129281.272936614375794,
129290.889075372222578,
129300.505280772325932,
129310.121552810203866,
129319.737891481461702,
129329.354296781632002,
129338.970768706276431,
129348.587307250913000,
129358.203912411176134,
129367.820584182583843,
129377.437322560712346,
129387.054127541137859,
129396.670999119392945,
129406.287937291097478,
129415.904942051740363,
129425.522013396941475,
129435.139151322262478,
129444.756355823279591,
129454.373626895510824,
129463.990964534590603,
129473.608368736022385,
129483.225839495396940,
129492.843376808348694,
129502.460980670381105,
129512.078651077084942,
129521.696388024007319,
129531.314191506753559,
129540.932061520899879,
129550.549998061978840,
129560.168001125610317,
129569.786070707341423,
129579.404206802762928,
129589.022409407436498,
129598.640678516967455,
129608.259014126917464,
129617.877416232862743,
129627.495884830364957,
129637.114419915000326,
129646.733021482388722,
129656.351689528091811,
129665.970424047685810,
129675.589225036761491,
129685.208092490851413,
129694.827026405619108,
129704.446026776597137,
129714.065093599390821,
129723.684226869561826,
129733.303426582715474,
129742.922692734413431,
129752.542025320275570,
129762.161424335878110,
129771.780889776811819,
129781.400421638623811,
129791.020019916933961,
129800.639684607376694,
129810.259415705455467,
129819.879213206804707,
129829.499077106986078,
129839.119007401692215,
129848.739004086368368,
129858.359067156736273,
129867.979196608270286,
129877.599392436663038,
129887.219654637476197,
129896.839983206285979,
129906.460378138697706,
129916.080839430302149,
129925.701367076733732,
129935.321961073554121,
129944.942621416368638,
129954.563348100738949,
129964.184141122357687,
129973.805000476757414,
129983.425926159543451,
129993.046918166321120,
130002.667976492666639,
130012.289101134228986,
130021.910292086584377,
130031.531549345352687,
130041.152872906110133,
130050.774262764491141,
130060.395718916101032,
130070.017241356501472,
130079.638830081326887,
130089.260485086197150,
130098.882206366717583,
130108.503993918493506,
130118.125847737130243,
130127.747767818160355,
130137.369754157334683,
130146.991806750185788,
130156.613925592348096,
130166.236110679397825,
130175.858362006940297,
130185.480679570668144,
130195.103063366099377,
130204.725513388912077,
130214.348029634653358,
130223.970612099030404,
130233.593260777619435,
130243.215975665996666,
130252.838756759825628,
130262.461604054682539,
130272.084517546230927,
130281.707497230076115,
130291.330543101823423,
130300.953655157063622,
130310.576833391489345,
130320.200077800662257,
130329.823388380245888,
130339.446765125816455,
130349.070208033037488,
130358.693717097514309,
130368.317292314823135,
130377.940933680714807,
130387.564641190678230,
130397.188414840406040,
130406.812254625532660,
130416.436160541634308,
130426.060132584403618,
130435.684170749416808,
130445.308275032308302,
130454.932445428741630,
130464.556681934293010,
130474.180984544625971,
130483.805353255374939,
130493.429788062145235,
130503.054288960600388,
130512.678855946374824,
130522.303489015044761,
130531.928188162302831,
130541.552953383754357,
130551.177784675033763,
130560.802682031804579,
130570.427645449672127,
130580.052674924270832,
130589.677770451293327,
130599.302932026286726,
130608.928159644943662,
130618.553453302898561,
130628.178812995785847,
130637.804238719269051,
130647.429730468953494,
130657.055288240473601,
130666.680912029463798,
130676.306601831645821,
130685.932357642566785,
130695.558179457919323,
130705.184067273337860,
130714.810021084485925,
130724.436040886968840,
130734.062126676479238,
130743.688278448622441,
130753.314496199061978,
130762.940779923417722,
130772.567129617382307,
130782.193545276575605,
130791.820026896661147,
130801.446574473273358,
130811.073188002075767,
130820.699867478702799,
130830.326612898817984,
130839.953424258084851,
130849.580301552166929,
130859.207244776611333,
130868.834253927241662,
130878.461328999575926,
130888.088469989321311,
130897.715676892141346,
130907.342949703641352,
130916.970288419543067,
130926.597693035480916,
130936.225163547089323,
130945.852699950075475,
130955.480302240015590,
130965.107970412675058,
130974.735704463600996,
130984.363504388544243,
130993.991370183110121,
131003.619301843005815,
131013.247299363836646,
131022.875362741338904,
131032.503491971088806,
131042.131687048808089,
131051.759947970189387,
131061.388274730823468,
131071.016667326402967,
131080.645125752605963,
131090.273650005110539,
131099.902240079536568,
131109.530895971605787,
131119.159617676952621,
131128.788405191269703,
131138.417258510220563,
131148.046177629439626,
131157.675162544648629,
131167.304213251452893,
131176.933329745603260,
131186.562512022763258,
131196.191760078509105,
131205.821073908620747,
131215.450453508703504,
131225.079898874508217,
131234.709410001698416,
131244.338986885821214,
131253.968629522714764,
131263.598337907984387,
131273.228112037264509,
131282.857951906335074,
131292.487857510772301,
131302.117828846327029,
131311.747865908662789,
131321.377968693443108,
131331.008137196389725,
131340.638371413137065,
131350.268671339319553,
131359.899036970775342,
131369.529468303051544,
131379.159965331869898,
131388.790528052923037,
131398.421156461903593,
131408.051850554475095,
131417.682610326359281,
131427.313435773190577,
131436.944326890690718,
131446.575283674552338,
131456.206306120409863,
131465.837394224043237,
131475.468547981086886,
131485.099767387204338,
131494.731052438146435,
131504.362403129576705,
131513.993819457158679,
131523.625301416643197,
131533.256849003664684,
131542.888462213944877,
131552.520141043205513,
131562.151885487110121,
131571.783695541293127,
131581.415571201563580,
131591.047512463585008,
131600.679519322962733,
131610.311591775534907,
131619.943729816906853,
131629.575933442800306,
131639.208202648907900,
131648.840537430951372,
131658.472937784623355,
131668.105403705587378,
131677.737935189594282,
131687.370532232307596,
131697.003194829478161,
131706.635922976740403,
131716.268716669874266,
131725.901575904485071,
131735.534500676381867,
131745.167490981257288,
131754.800546814745758,
131764.433668172569014,
131774.066855050477898,
131783.700107444194146,
131793.333425349352183,
131802.966808761702850,
131812.600257676967885,
131822.233772090810817,
131831.867351999011589,
131841.500997397233732,
131851.134708281198982,
131860.768484646570869,
131870.402326489158440,
131880.036233804625226,
131889.670206588663859,
131899.304244836996077,
131908.938348545343615,
131918.572517709457316,
131928.206752325000707,
131937.841052387724631,
131947.475417893350823,
131957.109848837571917,
131966.744345216051443,
131976.378907024627551,
131986.013534258963773,
131995.648226914781844,
132005.282984987745294,
132014.917808473692276,
132024.552697368228110,
132034.187651667161845,
132043.822671366186114,
132053.457756460993551,
132063.092906947334995,
132072.728122820932185,
132082.363404077535961,
132091.998750712838955,
132101.634162722533802,
132111.269640102429548,
132120.905182848218828,
132130.540790955565171,
132140.176464420277625,
132149.812203238077927,
132159.448007404687814,
132169.083876915799920,
132178.719811767165083,
132188.355811954534147,
132197.991877473628847,
132207.628008320200024,
132217.264204489911208,
132226.900465978571447,
132236.536792781873373,
132246.173184895596933,
132255.809642315376550,
132265.446165037050378,
132275.082753056311049,
132284.719406368938508,
132294.356124970596284,
132303.992908857064322,
132313.629758024064358,
132323.266672467347234,
132332.903652182692895,
132342.540697165764868,
132352.177807412313996,
132361.814982918149326,
132371.452223678934388,
132381.089529690478230,
132390.726900948502589,
132400.364337448700098,
132410.001839186850702,
132419.639406158734346,
132429.277038360014558,
132438.914735786558595,
132448.552498434000881,
132458.190326298092259,
132467.828219374612672,
132477.466177659342065,
132487.104201148031279,
132496.742289836343843,
132506.380443720117910,
132516.018662795017008,
132525.656947056850186,
132535.295296501368284,
132544.933711124322144,
132554.572190921433503,
132564.210735888511408,
132573.849346021219390,
132583.488021315395599,
132593.126761766761774,
132602.765567371097859,
132612.404438124125591,
132622.043374021566706,
132631.682375059259357,
132641.321441232925281,
132650.960572538315319,
132660.599768971180310,
132670.239030527300201,
132679.878357202454936,
132689.517748992337147,
132699.157205892755883,
132708.796727899432881,
132718.436315008206293,
132728.075967214768752,
132737.715684514871100,
132747.355466904351488,
132756.995314378931653,
132766.635226934362436,
132776.275204566365574,
132785.915247270837426,
132795.555355043441523,
132805.195527879957808,
132814.835765776224434,
132824.476068727875827,
132834.116436730808346,
132843.756869780714624,
132853.397367873403709,
132863.037931004597340,
132872.678559170104563,
132882.319252365705324,
132891.960010587150464,
132901.600833830190822,
132911.241722090664553,
132920.882675364264287,
132930.523693646857282,
132940.164776934107067,
132949.805925221880898,
132959.447138505842304,
132969.088416781916749,
132978.729760045796866,
132988.371168293262599,
132998.012641520093894,
133007.654179722070694,
133017.295782894972945,
133026.937451034609694,
133036.579184136702679,
133046.220982197060948,
133055.862845211464446,
133065.504773175664013,
133075.146766085526906,
133084.788823936745757,
133094.430946725158719,
133104.073134446487529,
133113.715387096628547,
133123.357704671245301,
133133.000087166146841,
133142.642534577200422,
133152.285046900127782,
133161.927624130708864,
133171.570266264752718,
133181.212973298010183,
133190.855745226348517,
133200.498582045489457,
133210.141483751212945,
133219.784450339328032,
133229.427481805672869,
133239.070578145998297,
133248.713739356084261,
133258.356965431768913,
133268.000256368773989,
133277.643612162966747,
133287.287032810098026,
133296.930518305947771,
133306.574068646325031,
133316.217683827067958,
133325.861363843898289,
133335.505108692683280,
133345.148918369173771,
133354.792792869207915,
133364.436732188507449,
133374.080736322968733,
133383.724805268313503,
133393.368939020379912,
133403.013137574977009,
133412.657400927884737,
133422.301729074883042,
133431.946122011839179,
133441.590579734474886,
133451.235102238686522,
133460.879689520195825,
133470.524341574811842,
133480.169058398401830,
133489.813839986716630,
133499.458686335594393,
133509.103597440815065,
133518.748573298187694,
133528.393613903550431,
133538.038719252683222,
133547.683889341395115,
133557.329124165524263,
133566.974423720821505,
133576.619788003154099,
133586.265217008301988,
133595.910710732074222,
133605.556269170308951,
133615.201892318815226,
133624.847580173372990,
133634.493332729820395,
133644.139149983995594,
133653.785031931620324,
133663.430978568649152,
133673.076989890774712,
133682.723065893864259,
133692.369206573755946,
133702.015411926229717,
133711.661681947123725,
133721.308016632217914,
133730.954415977408644,
133740.600879978446756,
133750.247408631170401,
133759.894001931417733,
133769.540659874968696,
133779.187382457690546,
133788.834169675392332,
133798.481021523883101,
133808.127937998971902,
133817.774919096525991,
133827.421964812354418,
133837.069075142266229,
133846.716250082070474,
133856.363489627663512,
133866.010793774796184,
133875.658162519364851,
133885.305595857120352,
133894.953093783929944,
133904.600656295631779,
133914.248283388034906,
133923.895975057006581,
133933.543731298326747,
133943.191552107833559,
133952.839437481365167,
133962.487387414759723,
133972.135401903884485,
133981.783480944519397,
133991.431624532502610,
134001.079832663701382,
134010.728105333924759,
134020.376442539039999,
134030.024844274797942,
134039.673310537153156,
134049.321841321856482,
134058.970436624746071,
134068.619096441689180,
134078.267820768523961,
134087.916609601117671,
134097.565462935221149,
134107.214380766759859,
134116.863363091542851,
134126.512409905379172,
134136.161521204223391,
134145.810696983768139,
134155.459937239938881,
134165.109241968544666,
134174.758611165452749,
134184.408044826530386,
134194.057542947586626,
134203.707105524459621,
134213.356732553016627,
134223.006424029124901,
134232.656179948564386,
134242.306000307231443,
134251.955885100964224,
134261.605834325629985,
134271.255847977037774,
134280.905926051054848,
134290.556068543548463,
134300.206275450356770,
134309.856546767347027,
134319.506882490299176,
134329.157282615167787,
134338.807747137732804,
134348.458276053861482,
134358.108869359421078,
134367.759527050278848,
134377.410249122272944,
134387.061035571270622,
134396.711886393110035,
134406.362801583600231,
134416.013781138695776,
134425.664825054205721,
134435.315933325997321,
134444.967105949908728,
134454.618342921836302,
134464.269644237618195,
134473.921009893092560,
134483.572439884126652,
134493.223934206616832,
134502.875492856401252,
134512.527115829347167,
134522.178803121380042,
134531.830554728221614,
134541.482370645797346,
134551.134250870032702,
134560.786195396736730,
134570.438204221776687,
134580.090277341078036,
134589.742414750391617,
134599.394616445672000,
134609.046882422728231,
134618.699212677485775,
134628.351607205811888,
134638.004066003573826,
134647.656589066580636,
134657.309176390757784,
134666.961827971943421,
134676.614543806033907,
134686.267323888954706,
134695.920168216456659,
134705.573076784494333,
134715.226049588905880,
134724.879086625587661,
134734.532187890436035,
134744.185353379289154,
134753.838583088014275,
134763.491877012507757,
134773.145235148636857,
134782.798657492268831,
134792.452144039329141,
134802.105694785685046,
134811.759309727145592,
134821.412988859665347,
134831.066732179082464,
134840.720539681322407,
134850.374411362194223,
134860.028347217652481,
134869.682347243506229,
134879.336411435710033,
134888.990539790131152,
134898.644732302636839,
134908.298988969094353,
134917.953309785370948,
134927.607694747450296,
134937.262143851112342,
134946.916657092311652,
134956.571234466886381,
134966.225875970732886,
134975.880581599805737,
134985.535351349913981,
134995.190185216924874,
135004.845083196822088,
135014.500045285472879,
135024.155071478715399,
135033.810161772446008,
135043.465316162590170,
135053.120534645044245,
135062.775817215675488,
135072.431163870380260,
135082.086574605054921,
135091.742049415595829,
135101.397588297899347,
135111.053191247832729,
135120.708858261350542,
135130.364589334320044,
135140.020384462637594,
135149.676243642170448,
135159.332166868844070,
135168.988154138554819,
135178.644205447228160,
135188.300320790702244,
135197.956500164902536,
135207.612743565783603,
135217.269050989183597,
135226.925422430998879,
135236.581857887154911,
135246.238357353606261,
135255.894920826161979,
135265.551548300747527,
135275.208239773288369,
135284.864995239739073,
135294.521814695879584,
135304.178698137722677,
135313.835645561135607,
135323.492656962072942,
135333.149732336314628,
135342.806871679873439,
135352.464074988674838,
135362.121342258556979,
135371.778673485474428,
135381.436068665352650,
135391.093527794000693,
135400.751050867460435,
135410.408637881540926,
135420.066288832225837,
135429.724003715411527,
135439.381782526965253,
135449.039625262841582,
135458.697531918907771,
135468.355502491147490,
135478.013536975486204,
135487.671635367732961,
135497.329797663900536,
135506.988023859914392,
135516.646313951554475,
135526.304667934920872,
135535.963085805793526,
135545.621567560156109,
135555.280113193904981,
135564.938722702994710,
135574.597396083321655,
135584.256133330782177,
135593.914934441301739,
135603.573799410864012,
135613.232728235307150,
135622.891720910643926,
135632.550777432741597,
135642.209897797496524,
135651.869082000863273,
135661.528330038825516,
135671.187641907192301,
135680.847017601976404,
135690.506457119103288,
135700.165960454440210,
135709.825527603941737,
135719.485158563591540,
135729.144853329227772,
135738.804611896834103,
135748.464434262335999,
135758.124320421658922,
135767.784270370728336,
135777.444284105498809,
135787.104361621837597,
135796.764502915757475,
135806.424707983154804,
135816.084976819925942,
135825.745309422054561,
135835.405705785466125,
135845.066165906086098,
135854.726689779898152,
135864.387277402740438,
135874.047928770567523,
135883.708643879450392,
135893.369422725139884,
135903.030265303736087,
135912.691171611048048,
135922.352141643059440,
135932.013175395783037,
135941.674272865056992,
135951.335434046835871,
135960.996658937132452,
135970.657947531813988,
135980.319299826864153,
135989.980715818179306,
135999.642195501772221,
136009.303738873539260,
136018.965345929434989,
136028.627016665355768,
136038.288751077343477,
136047.950549161265371,
136057.612410913134227,
136067.274336328817299,
136076.936325404298259,
136086.598378135589883,
136096.260494518501218,
136105.922674549103249,
136115.584918223292334,
136125.247225537023041,
136134.909596486279042,
136144.572031066927593,
136154.234529275039677,
136163.897091106424341,
136173.559716557152569,
136183.222405623178929,
136192.885158300341573,
136202.547974584711483,
136212.210854472185019,
136221.873797958745854,
136231.536805040290346,
136241.199875712831272,
136250.863009972323198,
136260.526207814720692,
136270.189469235949218,
136279.852794231992448,
136289.516182798863156,
136299.179634932399495,
136308.843150628614239,
136318.506729883520165,
136328.170372693013633,
136337.834079053078312,
136347.497848959668772,
136357.161682408768684,
136366.825579396303510,
136376.489539918315131,
136386.153563970641699,
136395.817651549325092,
136405.481802650348982,
136415.146017269667936,
136424.810295403207419,
136434.474637046921998,
136444.139042196853552,
136453.803510848927544,
136463.468042999127647,
136473.132638643379323,
136482.797297777637141,
136492.462020397972083,
136502.126806500251405,
136511.791656080546090,
136521.456569134723395,
136531.121545658825198,
136540.786585648776963,
136550.451689100591466,
136560.116856010223273,
136569.782086373626953,
136579.447380186786177,
136589.112737445742823,
136598.778158146335045,
136608.443642284662928,
136618.109189856622834,
136627.774800858285744,
136637.440475285489811,
136647.106213134335121,
136656.772014400718035,
136666.437879080680432,
136676.103807170176879,
136685.769798665161943,
136695.435853561648401,
136705.101971855619922,
136714.768153543031076,
136724.434398619865533,
136734.100707082136068,
136743.767078925768146,
136753.433514146832749,
136763.100012741226237,
136772.766574704990489,
136782.433200034050969,
136792.099888724449556,
136801.766640772140818,
136811.433456173137529,
136821.100334923394257,
136830.767277018952882,
136840.434282455738867,
136850.101351229735883,
136859.768483337014914,
136869.435678773472318,
136879.102937535149977,
136888.770259618002456,
136898.437645018071635,
136908.105093731282977,
136917.772605753707467,
136927.440181081270566,
136937.107819710014155,
136946.775521635863697,
136956.443286854890175,
136966.111115363019053,
136975.779007156321313,
136985.446962230693316,
136995.114980582264252,
137004.783062206901377,
137014.451207100675674,
137024.119415259570815,
137033.787686679512262,
137043.456021356658312,
137053.124419286847115,
137062.792880466149654,
137072.461404890578706,
137082.129992556088837,
137091.798643458721926,
137101.467357594461646,
137111.136134959291667,
137120.804975549282972,
137130.473879360361025,
137140.142846388538601,
137149.811876629857579,
137159.480970080301631,
137169.150126735912636,
137178.819346592616057,
137188.488629646511981,
137198.157975893496769,
137207.827385329641402,
137217.496857951016864,
137227.166393753490411,
137236.835992733191233,
137246.505654886044795,
137256.175380208151182,
137265.845168695435859,
137275.515020343911601,
137285.184935149678495,
137294.854913108632900,
137304.524954216845799,
137314.195058470329968,
137323.865225865127286,
137333.535456397192320,
137343.205750062566949,
137352.876106857234845,
137362.546526777237887,
137372.217009818647057,
137381.887555977416923,
137391.558165249502053,
137401.228837631031638,
137410.899573117960244,
137420.570371706358856,
137430.241233392182039,
137439.912158171471674,
137449.583146040298743,
137459.254196994588710,
137468.925311030441662,
137478.596488143841270,
137488.267728330800310,
137497.939031587389763,
137507.610397909535095,
137517.281827293394599,
137526.953319734922843,
137536.624875230074394,
137546.296493775007548,
137555.968175365647767,
137565.639919998036930,
137575.311727668246021,
137584.983598372258712,
137594.655532106116880,
137604.327528865862405,
137613.999588647478959,
137623.671711447066627,
137633.343897260579979,
137643.016146084089996,
137652.688457913609454,
137662.360832745180232,
137672.033270574815106,
137681.705771398585057,
137691.378335212473758,
137701.050962012552191,
137710.723651794891339,
137720.396404555358458,
137730.069220290170051,
137739.742098995309789,
137749.415040666761342,
137759.088045300595695,
137768.761112892854726,
137778.434243439522106,
137788.107436936727026,
137797.780693380453158,
137807.454012766742380,
137817.127395091636572,
137826.800840351148508,
137836.474348541378276,
137846.147919658309547,
137855.821553697984200,
137865.495250656502321,
137875.169010529847583,
137884.842833314061863,
137894.516719005245250,
137904.190667599381413,
137913.864679092512233,
137923.538753480737796,
137933.212890760041773,
137942.887090926524252,
137952.561353976139799,
137962.235679905075813,
137971.910068709228653,
137981.584520384756615,
137991.259034927643370,
138000.933612333989004,
138010.608252599777188,
138020.282955721108010,
138029.957721694023348,
138039.632550514565082,
138049.307442178775091,
138058.982396682724357,
138068.657414022396551,
138078.332494193949969,
138088.007637193397386,
138097.682843016780680,
138107.358111660112627,
138117.033443119522417,
138126.708837391022826,
138136.384294470655732,
138146.059814354521222,
138155.735397038661176,
138165.411042519059265,
138175.086750791902887,
138184.762521853146609,
138194.438355698919622,
138204.114252325176494,
138213.790211728075519,
138223.466233903658576,
138233.142318847996648,
138242.818466557044303,
138252.494677027018042,
138262.170950253872434,
138271.847286233649356,
138281.523684962536208,
138291.200146436516661,
138300.876670651603490,
138310.553257603954989,
138320.229907289583934,
138329.906619704590412,
138339.583394844987197,
138349.260232706874376,
138358.937133286322933,
138368.614096579374745,
138378.291122582129901,
138387.968211290630279,
138397.645362700946862,
138407.322576809179736,
138416.999853611341678,
138426.677193103532773,
138436.354595281824004,
138446.032060142315459,
138455.709587681019912,
138465.387177894037450,
138475.064830777468160,
138484.742546327324817,
138494.420324539736612,
138504.098165410716319,
138513.776068936393131,
138523.454035112838028,
138533.132063936121995,
138542.810155402286910,
138552.488309507432859,
138562.166526247659931,
138571.844805618980899,
138581.523147617524955,
138591.201552239363082,
138600.880019480595365,
138610.558549337234581,
138620.237141805409919,
138629.915796881192364,
138639.594514560682001,
138649.273294839920709,
138658.952137715037679,
138668.631043182045687,
138678.310011237073923,
138687.989041876222473,
138697.668135095562320,
138707.347290891164448,
138717.026509259070735,
138726.705790195439477,
138736.385133696341654,
138746.064539757848252,
138755.744008376001148,
138765.423539546958636,
138775.103133266791701,
138784.782789531600429,
138794.462508337426698,
138804.142289680370595,
138813.822133556532208,
138823.502039962040726,
138833.182008892938029,
138842.862040345324203,
138852.542134315328440,
138862.222290798934409,
138871.902509792358615,
138881.582791291613830,
138891.263135292829247,
138900.943541792134056,
138910.624010785511928,
138920.304542269150261,
138929.985136239149142,
138939.665792691550450,
138949.346511622483376,
138959.027293028018903,
138968.708136904257117,
138978.389043247356312,
138988.070012053358369,
138997.751043318363372,
139007.432137038471410,
139017.113293209811673,
139026.794511828396935,
139036.475792890472803,
139046.157136392022949,
139055.838542329205666,
139065.520010698091937,
139075.201541494781850,
139084.883134715375490,
139094.564790356031153,
139104.246508412790718,
139113.928288881812477,
139123.610131759167416,
139133.292037040955620,
139142.974004723277176,
139152.656034802290378,
139162.338127274037106,
139172.020282134675654,
139181.702499380277004,
139191.384779006970348,
139201.067121010855772,
139210.749525388004258,
139220.431992134632310,
139230.114521246752702,
139239.797112720523728,
139249.479766552045476,
139259.162482737418031,
139268.845261272741482,
139278.528102154174121,
139288.211005377786933,
139297.893970939709106,
139307.576998836040730,
139317.260089062940096,
139326.943241616478190,
139336.626456492813304,
139346.309733688016422,
139355.993073198216734,
139365.676475019514328,
139375.359939148096601,
139385.043465580005432,
139394.727054311370011,
139404.410705338348635,
139414.094418657070491,
139423.778194263577461,
139433.462032154056942,
139443.145932324638125,
139452.829894771333784,
139462.513919490389526,
139472.198006477905437,
139481.882155729952501,
139491.566367242718115,
139501.250641012273263,
139510.934977034776239,
139520.619375306327129,
139530.303835823084228,
139539.988358581089415,
139549.672943576588295,
139559.357590805593645,
139569.042300264321966,
139578.727071948902449,
139588.411905855376972,
139598.096801979962038,
139607.781760318728629,
139617.466780867835041,
139627.151863623410463,
139636.837008581584087,
139646.522215738455998,
139656.207485090213595,
139665.892816632927861,
139675.578210362786194,
139685.263666275888681,
139694.949184368364513,
139704.634764636401087,
139714.320407076098491,
139724.006111683527706,
139733.691878454934340,
139743.377707386389375,
139753.063598474051105,
139762.749551714048721,
139772.435567102482310,
139782.121644635568373,
139791.807784309406998,
139801.493986120098270,
139811.180250063829590,
139820.866576136730146,
139830.552964334929129,
139840.239414654555731,
139849.925927091826452,
139859.612501642783172,
139869.299138303642394,
139878.985837070504203,
139888.672597939526895,
139898.359420906810556,
139908.046305968600791,
139917.733253120968584,
139927.420262360043125,
139937.107333682011813,
139946.794467083003838,
139956.481662559177494,
139966.168920106691075,
139975.856239721644670,
139985.543621400196571,
139995.231065138563281,
140004.918570932786679,
140014.606138779112371,
140024.293768673640443,
140033.981460612500086,
140043.669214591907803,
140053.357030607963679,
140063.044908656855114,
140072.732848734711297,
140082.420850837661419,
140092.108914961892879,
140101.797041103563970,
140111.485229258832987,
140121.173479423800018,
140130.861791594652459,
140140.550165767577710,
140150.238601938675856,
140159.927100104163401,
140169.615660260169534,
140179.304282402852550,
140188.992966528341640,
140198.681712632824201,
140208.370520712458529,
140218.059390763373813,
140227.748322781815659,
140237.437316763825947,
140247.126372705650283,
140256.815490603417857,
140266.504670453316066,
140276.193912251474103,
140285.883215994079364,
140295.572581677261041,
140305.262009297235636,
140314.951498850103235,
140324.641050332080340,
140334.330663739296142,
140344.020339067938039,
140353.710076314193429,
140363.399875474191504,
140373.089736544090556,
140382.779659520107089,
140392.469644398341188,
140402.159691175038461,
140411.849799846298993,
140421.539970408339286,
140431.230202857346740,
140440.920497189392336,
140450.610853400750784,
140460.301271487522172,
140469.991751445952104,
140479.682293272140669,
140489.372896962304367,
140499.063562512601493,
140508.754289919190342,
140518.445079178258311,
140528.135930286021903,
140537.826843238610309,
140547.517818032181822,
140557.208854662952945,
140566.899953127081972,
140576.591113420727197,
140586.282335540076019,
140595.973619481344940,
140605.664965240663150,
140615.356372814247152,
140625.047842198255239,
140634.739373388874810,
140644.430966382264160,
140654.122621174639789,
140663.814337762159994,
140673.506116140983067,
140683.197956307383720,
140692.889858257403830,
140702.581821987376316,
140712.273847493343055,
140721.965934771578759,
140731.658083818270825,
140741.350294629548443,
140751.042567201628117,
140760.734901530697243,
140770.427297612943221,
140780.119755444553448,
140789.812275021715323,
140799.504856340587139,
140809.197499397414504,
140818.890204188326607,
140828.582970709539950,
140838.275798957271036,
140847.968688927649055,
140857.661640616919613,
140867.354654021211900,
140877.047729136800626,
140886.740865959844086,
140896.434064486471470,
140906.127324712957488,
140915.820646635489538,
140925.514030250225915,
140935.207475553354016,
140944.900982541119447,
140954.594551209651399,
140964.288181555195479,
140973.981873573909979,
140983.675627262069611,
140993.369442615774460,
141003.063319631328341,
141012.757258304802235,
141022.451258632470854,
141032.145320610521594,
141041.839444235141855,
141051.533629502548138,
141061.227876408927841,
141070.922184950497467,
141080.616555123473518,
141090.310986924014287,
141100.005480348336278,
141109.700035392626887,
141119.394652053160826,
141129.089330326067284,
141138.784070207562763,
141148.478871693863766,
141158.173734781215899,
141167.868659465748351,
141177.563645743677625,
141187.258693611278431,
141196.953803064679960,
141206.648974100156920,
141216.344206713896710,
141226.039500902057625,
141235.734856660914375,
141245.430273986625252,
141255.125752875406761,
141264.821293323533610,
141274.516895327134989,
141284.212558882456506,
141293.908283985743765,
141303.604070633155061,
141313.299918820906896,
141322.995828545215772,
141332.691799802327296,
141342.387832588399760,
141352.083926899766084,
141361.780082732497249,
141371.476300082838861,
141381.172578947094735,
141390.868919321394060,
141400.565321201982442,
141410.261784585105488,
141419.958309466921492,
141429.654895843676059,
141439.351543711614795,
141449.048253066895995,
141458.745023905823473,
141468.441856224555522,
141478.138750019337749,
141487.835705286328448,
141497.532722021860536,
141507.229800222063204,
141516.926939883211162,
141526.624141001520911,
141536.321403573150747,
141546.018727594433585,
141555.716113061498618,
141565.413559970620554,
141575.111068318015896,
141584.808638099901145,
141594.506269312521908,
141604.203961952036479,
141613.901716014777776,
141623.599531496962300,
141633.297408394719241,
141642.995346704294207,
141652.693346422020113,
141662.391407544055255,
141672.089530066645239,
141681.787713985977462,
141691.485959298326634,
141701.184265999938361,
141710.882634087000042,
141720.581063555757282,
141730.279554402455688,
141739.978106623340864,
141749.676720214600209,
141759.375395172537537,
141769.074131493311143,
141778.772929173166631,
141788.471788208378712,
141798.170708595222095,
141807.869690329825971,
141817.568733408465050,
141827.267837827384938,
141836.967003582860343,
141846.666230671078665,
141856.365519088314613,
141866.064868830755586,
141875.764279894676292,
141885.463752276351443,
141895.163285971968435,
141904.862880977772875,
141914.562537290010368,
141924.262254904984729,
141933.962033818854252,
141943.661874027864542,
141953.361775528290309,
141963.061738316377159,
141972.761762388370698,
141982.461847740516532,
141992.161994369031163,
142001.862202270189300,
142011.562471440207446,
142021.262801875360310,
142030.963193571893498,
142040.663646526023513,
142050.364160734054167,
142060.064736192172859,
142069.765372896654299,
142079.466070843773196,
142089.166830029746052,
142098.867650450818473,
142108.568532103236066,
142118.269474983302644,
142127.970479087205604,
142137.671544411248760,
142147.372670951619511,
142157.073858704679878,
142166.775107666588156,
142176.476417833619053,
142186.177789202018175,
142195.879221768060233,
142205.580715528019937,
142215.282270478113787,
142224.983886614616495,
142234.685563933773665,
142244.387302431860007,
142254.089102105121128,
142263.790962949773530,
142273.492884962150129,
142283.194868138467427,
142292.896912475029239,
142302.599017968022963,
142312.301184613752412,
142322.003412408492295,
142331.705701348430011,
142341.408051429927582,
142351.110462649172405,
142360.812935002497397,
142370.515468486089958,
142380.218063096224796,
142389.920718829234829,
142399.623435681278352,
142409.326213648717385,
142419.029052727797534,
142428.731952914706198,
142438.434914205805399,
142448.137936597340740,
142457.841020085499622,
142467.544164666644065,
142477.247370336990571,
142486.950637092872057,
142496.653964930446818,
142506.357353846105980,
142516.060803836036939,
142525.764314896543510,
142535.467887023871299,
142545.171520214353222,
142554.875214464176679,
142564.578969769645482,
142574.282786127063446,
142583.986663532676175,
142593.690601982729277,
142603.394601473555667,
142613.098662001400953,
142622.802783562539844,
142632.506966153247049,
142642.211209769768175,
142651.915514408436138,
142661.619880065525649,
142671.324306737224106,
142681.028794419922633,
142690.733343109837733,
142700.437952803273220,
142710.142623496474698,
142719.847355185775086,
142729.552147867390886,
142739.257001537655015,
142748.961916192813078,
142758.666891829139786,
142768.371928442997159,
142778.077026030514389,
142787.782184588111704,
142797.487404112034710,
142807.192684598558117,
142816.898026043927530,
142826.603428444504971,
142836.308891796506941,
142846.014416096266359,
142855.720001340028830,
142865.425647524127271,
142875.131354644807288,
142884.837122698372696,
142894.542951681098202,
142904.248841589316726,
142913.954792419244768,
142923.660804167186143,
142933.366876829502871,
142943.073010402411455,
142952.779204882244812,
142962.485460265219444,
142972.191776547697373,
142981.898153725982411,
142991.604591796320165,
143001.311090755014447,
143011.017650598369073,
143020.724271322658751,
143030.430952924216399,
143040.137695399287622,
143049.844498744205339,
143059.551362955244258,
143069.258288028679090,
143078.965273960842751,
143088.672320748009952,
143098.379428386513609,
143108.086596872628434,
143117.793826202600030,
143127.501116372819524,
143137.208467379532522,
143146.915879219042836,
143156.623351887654280,
143166.330885381670669,
143176.038479697395815,
143185.746134831104428,
143195.453850779100321,
143205.161627537745517,
143214.869465103256516,
143224.577363472024444,
143234.285322640236700,
143243.993342604313511,
143253.701423360529589,
143263.409564905130537,
143273.117767234478379,
143282.826030344876926,
143292.534354232629994,
143302.242738893983187,
143311.951184325327631,
143321.659690522908932,
143331.368257483089110,
143341.076885202113772,
143350.785573676344939,
143360.494322902115528,
143370.203132875612937,
143379.912003593257396,
143389.620935051352717,
143399.329927246144507,
143409.038980174023891,
143418.748093831207370,
143428.457268214115174,
143438.166503318992909,
143447.875799142144388,
143457.585155679931631,
143467.294572928629350,
143477.004050884599565,
143486.713589544087881,
143496.423188903485425,
143506.132848959037801,
143515.842569707107032,
143525.552351143967826,
143535.262193265982205,
143544.972096069424879,
143554.682059550657868,
143564.392083705984987,
143574.102168531710049,
143583.812314024195075,
143593.522520179656567,
143603.232786994572962,
143612.943114465102553,
143622.653502587694675,
143632.363951358594932,
143642.074460774136242,
143651.785030830651522,
143661.495661524502793,
143671.206352851935662,
143680.917104809341254,
143690.627917392994277,
143700.338790599285858,
143710.049724424432497,
143719.760718864854425,
143729.471773916826351,
143739.182889576739399,
143748.894065840839176,
143758.605302705487702,
143768.316600167047000,
143778.027958221791778,
143787.739376866113162,
143797.450856096227653,
143807.162395908613689,
143816.873996299429564,
143826.585657265211921,
143836.297378802119056,
143846.009160906542093,
143855.721003574813949,
143865.432906803296646,
143875.144870588235790,
143884.856894926080713,
143894.568979813077021,
143904.281125245615840,
143913.993331219971878,
143923.705597732565366,
143933.417924779612804,
143943.130312357563525,
143952.842760462663136,
143962.555269091331866,
143972.267838239844423,
143981.980467904562829,
143991.693158081820002,
144001.405908767948858,
144011.118719959340524,
144020.831591652211500,
144030.544523843040224,
144040.257516528101405,
144049.970569703757064,
144059.683683366281912,
144069.396857512096176,
144079.110092137503671,
144088.823387238866417,
144098.536742812517332,
144108.250158854818437,
144117.963635362044442,
144127.677172330673784,
144137.390769756864756,
144147.104427637124900,
144156.818145967758028,
144166.531924745038850,
144176.245763965416700,
144185.959663625137182,
144195.673623720678734,
144205.387644248228753,
144215.101725204265676,
144224.815866585035110,
144234.530068387015490,
144244.244330606394215,
144253.958653239678824,
144263.673036283144029,
144273.387479733093642,
144283.101983585976996,
144292.816547838068800,
144302.531172485789284,
144312.245857525413157,
144321.960602953360649,
144331.675408765935572,
144341.390274959558155,
144351.105201530532213,
144360.820188475161558,
144370.535235789895523,
144380.250343471037922,
144389.965511515008984,
144399.680739918054314,
144409.396028676652350,
144419.111377787077799,
144428.826787245692685,
144438.542257048917236,
144448.257787192997057,
144457.973377674439689,
144467.689028489461634,
144477.404739634541329,
144487.120511105982587,
144496.836342900118325,
144506.552235013368772,
144516.268187442037743,
144525.984200182545464,
144535.700273231195752,
144545.416406584437937,
144555.132600238575833,
144564.848854189942358,
144574.565168434957741,
144584.281542969954899,
144593.997977791295853,
144603.714472895400831,
144613.431028278602753,
144623.147643937234534,
144632.864319867716404,
144642.581056066381279,
144652.297852529620286,
144662.014709253737237,
144671.731626235210570,
144681.448603470285889,
144691.165640955470735,
144700.882738687068922,
144710.599896661384264,
144720.317114874866093,
144730.034393323876429,
144739.751732004806399,
144749.469130913988920,
144759.186590047786012,
144768.904109402617905,
144778.621688974817516,
144788.339328760775970,
144798.057028756884392,
144807.774788959475700,
144817.492609364970122,
144827.210489969758783,
144836.928430770087289,
144846.646431762521388,
144856.364492943306686,
144866.082614308863413,
144875.800795855553588,
144885.519037579768337,
144895.237339477898786,
144904.955701546336059,
144914.674123781383969,
144924.392606179491850,
144934.111148737050826,
144943.829751450364711,
144953.548414315911941,
144963.267137329996331,
144972.985920489038108,
144982.704763789399294,
144992.423667227500118,
145002.142630799702602,
145011.861654502368765,
145021.580738331889734,
145031.299882284685737,
145041.019086357118795,
145050.738350545550929,
145060.457674846431473,
145070.177059256093344,
145079.896503770927666,
145089.616008387325564,
145099.335573101736372,
145109.055197910434799,
145118.774882809899282,
145128.494627796491841,
145138.214432866574498,
145147.934298016596586,
145157.654223242920125,
145167.374208541907137,
145177.094253909977851,
145186.814359343552496,
145196.534524838905782,
145206.254750392574351,
145215.975036000920227,
145225.695381660247222,
145235.415787367062876,
145245.136253117700107,
145254.856778908520937,
145264.577364736003801,
145274.298010596510721,
145284.018716486432822,
145293.739482402161229,
145303.460308340087067,
145313.181194296601461,
145322.902140268153744,
145332.623146251105936,
145342.344212241849164,
145352.065338236774551,
145361.786524232331431,
145371.507770224881824,
145381.229076210787753,
145390.950442186527653,
145400.671868148492649,
145410.393354093044763,
145420.114900016604224,
145429.836505915591260,
145439.558171786367893,
145449.279897625354351,
145459.001683428999968,
145468.723529193637660,
145478.445434915716760,
145488.167400591628393,
145497.889426217792789,
145507.611511790571967,
145517.333657306415262,
145527.055862761742901,
145536.778128152946010,
145546.500453476386610,
145556.222838728543138,
145565.945283905777615,
145575.667789004568476,
145585.390354021190433,
145595.112978952151025,
145604.835663793899585,
145614.558408542710822,
145624.281213195150485,
145634.004077747522388,
145643.727002196275862,
145653.449986537802033,
145663.173030768579338,
145672.896134884969797,
145682.619298883393640,
145692.342522760212887,
145702.065806511935079,
145711.789150134922238,
145721.512553625594592,
145731.236016980401473,
145740.959540195704903,
145750.683123267954215,
145760.406766193569638,
145770.130468968971400,
145779.854231590579730,
145789.578054054756649,
145799.301936357980594,
145809.025878496700898,
145818.749880467250478,
145828.473942266107770,
145838.198063889634795,
145847.922245334368199,
145857.646486596640898,
145867.370787672873121,
145877.095148559485096,
145886.819569252955262,
145896.544049749703845,
145906.268590046063764,
145915.993190138513455,
145925.717850023531355,
145935.442569697479485,
145945.167349156807177,
145954.892188397905556,
145964.617087417223956,
145974.342046211211709,
145984.067064776289044,
145993.792143108847085,
146003.517281205306062,
146013.242479062173516,
146022.967736675811466,
146032.693054042698350,
146042.418431159196189,
146052.143868021783419,
146061.869364626909373,
146071.594920970936073,
146081.320537050371058,
146091.046212861576350,
146100.771948401030386,
146110.497743665153394,
146120.223598650394706,
146129.949513353174552,
146139.675487769884057,
146149.401521897030761,
146159.127615731005790,
146168.853769268287579,
146178.579982505238149,
146188.306255438306835,
146198.032588064001175,
146207.758980378741398,
146217.485432378918631,
146227.211944060982205,
146236.938515421381453,
146246.665146456565708,
146256.391837162926095,
146266.118587536970153,
146275.845397575118113,
146285.572267273731995,
146295.299196629377548,
146305.026185638445895,
146314.753234297328163,
146324.480342602531891,
146334.207510550477309,
146343.934738137584645,
146353.662025360332336,
146363.389372215140611,
146373.116778698487906,
146382.844244806794450,
146392.571770536480471,
146402.299355883995304,
146412.027000845846487,
146421.754705418425146,
146431.482469598238822,
146441.210293381620431,
146450.938176765106618,
146460.666119745117612,
146470.394122318102745,
146480.122184480569558,
146489.850306228880072,
146499.578487559512723,
146509.306728468916845,
146519.035028953541769,
146528.763389009895036,
146538.491808634338668,
146548.220287823380204,
146557.948826573468978,
146567.677424880996114,
146577.406082742498256,
146587.134800154395634,
146596.863577113137580,
146606.592413615202531,
146616.321309656981612,
146626.050265235040570,
146635.779280345683219,
146645.508354985504411,
146655.237489150895271,
146664.966682838305132,
146674.695936044270638,
146684.425248765124707,
146694.154620997433085,
146703.884052737586899,
146713.613543982064584,
146723.343094727344578,
146733.072704969847109,
146742.802374706108822,
146752.532103932520840,
146762.261892645532498,
146771.991740841651335,
146781.721648517326685,
146791.451615669036983,
146801.181642293202458,
146810.911728386330651,
146820.641873944841791,
146830.372078965243418,
146840.102343443984864,
146849.832667377486359,
146859.563050762255443,
146869.293493594799656,
146879.023995871510124,
146888.754557588894386,
146898.485178743430879,
146908.215859331539832,
146917.946599349728785,
146927.677398794417968,
146937.408257662114920,
146947.139175949327182,
146956.870153652445879,
146966.601190767978551,
146976.332287292403635,
146986.063443222170463,
146995.794658553786576,
147005.525933283672202,
147015.257267408334883,
147024.988660924194846,
147034.720113827788737,
147044.451626115594991,
147054.183197784062941,
147063.914828829671023,
147073.646519248868572,
147083.378269038134022,
147093.110078194004018,
147102.841946712869685,
147112.573874591238564,
147122.305861825618194,
147132.037908412457909,
147141.770014348236145,
147151.502179629460443,
147161.234404252521927,
147170.966688214015448,
147180.699031510361237,
147190.431434138008626,
147200.163896093494259,
147209.896417373296572,
147219.628997973835794,
147229.361637891677674,
147239.094337123242440,
147248.827095665008528,
147258.559913513454376,
147268.292790665145731,
147278.025727116473718,
147287.758722863974981,
147297.491777904127957,
147307.224892233352875,
147316.958065848244587,
147326.691298745194217,
147336.424590920738410,
147346.157942371355603,
147355.891353093495127,
147365.624823083693627,
147375.358352338458644,
147385.091940854181303,
147394.825588627427351,
147404.559295654675225,
147414.293061932403361,
147424.026887457061093,
147433.760772225214168,
147443.494716233282816,
147453.228719477832783,
147462.962781955313403,
147472.696903662232216,
147482.431084595038556,
147492.165324750298169,
147501.899624124402180,
147511.633982713945443,
147521.368400515377289,
147531.102877525147051,
147540.837413739820477,
147550.572009155905107,
147560.306663769821171,
147570.041377578105312,
147579.776150577265071,
147589.510982763778884,
147599.245874134125188,
147608.980824684840627,
147618.715834412403638,
147628.450903313292656,
147638.186031384073431,
147647.921218621195294,
147657.656465021136682,
147667.391770580434240,
147677.127135295566404,
147686.862559163040714,
147696.598042179393815,
147706.333584341075039,
147716.069185644621029,
147725.804846086510224,
147735.540565663250163,
147745.276344371348387,
147755.012182207312435,
147764.748079167678952,
147774.484035248897271,
147784.220050447474932,
147793.956124759948580,
147803.692258182796650,
147813.428450712584890,
147823.164702345704427,
147832.901013078779215,
147842.637382908287691,
147852.373811830679188,
147862.110299842519453,
147871.846846940316027,
147881.583453120547347,
147891.320118379750056,
147901.056842714373488,
147910.793626121041598,
147920.530468596174615,
147930.267370136280078,
147940.004330737952841,
147949.741350397613132,
147959.478429111797595,
147969.215566877071979,
147978.952763689885614,
147988.690019546775147,
147998.427334444248118,
148008.164708378841169,
148017.902141347032739,
148027.639633345388575,
148037.377184370387113,
148047.114794418535894,
148056.852463486371562,
148066.590191570430761,
148076.327978667162824,
148086.065824773162603,
148095.803729884908535,
148105.541693998937262,
148115.279717111756327,
148125.017799219873268,
148134.755940319795627,
148144.494140408089152,
148154.232399481261382,
148163.970717535819858,
148173.709094568301225,
148183.447530575213023,
148193.186025553062791,
148202.924579498416279,
148212.663192407751922,
148222.401864277635468,
148232.140595104516251,
148241.879384885076433,
148251.618233615648933,
148261.357141292886809,
148271.096107913239393,
148280.835133473272435,
148290.574217969522579,
148300.313361398497364,
148310.052563756704330,
148319.791825040738331,
148329.531145247077802,
148339.270524372201180,
148349.009962412732420,
148358.749459365149960,
148368.489015226019546,
148378.228629991819616,
148387.968303659115918,
148397.708036224445095,
148407.447827684343792,
148417.187678035261342,
148426.927587273821700,
148436.667555396532407,
148446.407582399901003,
148456.147668280493235,
148465.887813034816645,
148475.628016659466084,
148485.368279150861781,
148495.108600505656796,
148504.848980720329564,
148514.589419791416731,
148524.329917715454940,
148534.070474489009939,
148543.811090108589269,
148553.551764570700470,
148563.292497871938394,
148573.033290008810582,
148582.774140977853676,
148592.515050775662530,
148602.256019398657372,
148611.997046843520366,
148621.738133106671739,
148631.479278184764553,
148641.220482074248139,
148650.961744771688245,
148660.703066273621516,
148670.444446576613700,
148680.185885677201441,
148689.927383571921382,
148699.668940257310169,
148709.410555729904445,
148719.152229986269958,
148728.893963022943353,
148738.635754836519482,
148748.377605423447676,
148758.119514780322788,
148767.861482903710566,
148777.603509790118551,
148787.345595436083386,
148797.087739838199923,
148806.829942993033910,
148816.572204897092888,
148826.314525546884397,
148836.056904939032393,
148845.799343070044415,
148855.541839936486213,
148865.284395534894429,
148875.027009861863917,
148884.769682913873112,
148894.512414687545970,
148904.255205179360928,
148913.998054385941941,
148923.740962303796550,
148933.483928929519607,
148943.226954259589547,
148952.970038290630328,
148962.713181019178592,
148972.456382441800088,
148982.199642555002356,
148991.942961355409352,
149001.686338839499513,
149011.429775003896793,
149021.173269845108734,
149030.916823359701084,
149040.660435544326901,
149050.404106395377312,
149060.147835909534479,
149069.891624083305942,
149079.635470913257450,
149089.379376395983854,
149099.123340527992696,
149108.867363305907929,
149118.611444726207992,
149128.355584785516839,
149138.099783480342012,
149147.844040807336569,
149157.588356762978947,
149167.332731343863998,
149177.077164546557469,
149186.821656367596006,
149196.566206803545356,
149206.310815851000370,
149216.055483506526798,
149225.800209766719490,
149235.544994628056884,
149245.289838087133830,
149255.034740140574286,
149264.779700784856686,
149274.524720016604988,
149284.269797832414042,
149294.014934228820493,
149303.760129202331882,
149313.505382749630371,
149323.250694867165294,
149332.996065551618813,
149342.741494799498469,
149352.486982607399113,
149362.232528971857391,
149371.978133889468154,
149381.723797356797149,
149391.469519370468333,
149401.215299926989246,
149410.961139022896532,
149420.707036654872354,
149430.452992819395149,
149440.199007513147080,
149449.945080732577480,
149459.691212474339409,
149469.437402734969510,
149479.183651511091739,
149488.929958799242740,
149498.676324595988262,
149508.422748897952260,
149518.169231701671379,
149527.915773003769573,
149537.662372800783487,
149547.409031089278869,
149557.155747865850572,
149566.902523127093446,
149576.649356869573239,
149586.396249089913908,
149596.143199784593889,
149605.890208950324450,
149615.637276583584026,
149625.384402680996573,
149635.131587239127839,
149644.878830254601780,
149654.626131723925937,
149664.373491643724265,
149674.120910010649823,
149683.868386821181048,
149693.615922071912792,
149703.363515759498114,
149713.111167880473658,
149722.858878431492485,
149732.606647409003926,
149742.354474809690146,
149752.102360630116891,
149761.850304866908118,
149771.598307516600471,
149781.346368575788802,
149791.094488041097065,
149800.842665909091011,
149810.590902176365489,
149820.339196839515353,
149830.087549895077245,
149839.835961339704227,
149849.584431169991149,
149859.332959382474655,
149869.081545973778702,
149878.830190940498142,
149888.578894279286033,
149898.327655986591708,
149908.076476059126435,
149917.825354493426858,
149927.574291286146035,
149937.323286433762405,
149947.072339933016337,
149956.821451780386269,
149966.570621972583467,
149976.319850506057264,
149986.069137377518928,
149995.818482583505102,
150005.567886120668845,
150015.317347985575907,
150025.066868174821138,
150034.816446684999391,
150044.566083512705518,
150054.315778654592577,
150064.065532107139006,
150073.815343867056072,
150083.565213930967730,
150093.315142295352416,
150103.065128956921399,
150112.815173912211321,
150122.565277157875244,
150132.315438690420706,
150142.065658506558975,
150151.815936602855800,
150161.566272975876927,
150171.316667622304522,
150181.067120538646122,
150190.817631721612997,
150200.568201167712687,
150210.318828873598250,
150220.069514835893642,
150229.820259051164612,
150239.571061516035115,
150249.321922227070900,
150259.072841181012336,
150268.823818374308757,
150278.574853803642327,
150288.325947465637000,
150298.077099356887629,
150307.828309473959962,
150317.579577813536162,
150327.330904372152872,
150337.082289146463154,
150346.833732133090962,
150356.585233328631148,
150366.336792729707668,
150376.088410332915373,
150385.840086134849116,
150395.591820132161956,
150405.343612321448745,
150415.095462699333439,
150424.847371262410888,
150434.599338007305050,
150444.351362930610776,
150454.103446028981125,
150463.855587299040053,
150473.607786737353308,
150483.360044340573950,
150493.112360105325934,
150502.864734028175008,
150512.617166105774231,
150522.369656334776664,
150532.122204711748054,
150541.874811233341461,
150551.627475896122633,
150561.380198696744628,
150571.132979631889611,
150580.885818698065123,
150590.638715891982429,
150600.391671210236382,
150610.144684649392730,
150619.897756206133636,
150629.650885877112160,
150639.404073658864945,
150649.157319548074156,
150658.910623541334644,
150668.663985635299468,
150678.417405826563481,
150688.170884111779742,
150697.924420487543102,
150707.678014950506622,
150717.431667497323360,
150727.185378124500858,
150736.939146828779485,
150746.692973606783198,
150756.446858455106849,
150766.200801370316185,
150775.954802349180682,
150785.708861388207879,
150795.462978484109044,
150805.217153633479029,
150814.971386832883582,
150824.725678079063073,
150834.480027368583251,
150844.234434698126279,
150853.988900064257905,
150863.743423463660292,
150873.498004892928293,
150883.252644348773174,
150893.007341827702476,
150902.762097326398361,
150912.516910841572098,
150922.271782369789435,
150932.026711907674326,
150941.781699451908935,
150951.536744999088114,
150961.291848545835819,
150971.047010088863317,
150980.802229624707252,
150990.557507150107995,
151000.312842661602190,
151010.068236155901104,
151019.823687629599590,
151029.579197079379810,
151039.334764501807513,
151049.090389893623069,
151058.846073251363123,
151068.601814571768045,
151078.357613851374481,
151088.113471086951904,
151097.869386275007855,
151107.625359412253601,
151117.381390495342202,
151127.137479520839406,
151136.893626485485584,
151146.649831385875586,
151156.406094218633370,
151166.162414980470203,
151175.918793667980935,
151185.675230277789524,
151195.431724806578131,
151205.188277250999818,
151214.944887607678538,
151224.701555873267353,
151234.458282044390216,
151244.215066117758397,
151253.971908089937642,
151263.728807957639219,
151273.485765717487084,
151283.242781366105191,
151292.999854900204809,
151302.756986316351686,
151312.514175611257087,
151322.271422781603178,
151332.028727823955705,
151341.786090735025937,
151351.543511511408724,
151361.300990149815334,
151371.058526646840619,
151380.816120999195846,
151390.573773203534074,
151400.331483256450156,
151410.089251154655358,
151419.847076894744532,
151429.604960473428946,
151439.362901887332555,
151449.120901133108418,
151458.878958207438700,
151468.637073106947355,
151478.395245828316547,
151488.153476368199335,
151497.911764723248780,
151507.670110890117940,
151517.428514865459874,
151527.186976645956747,
151536.945496228232514,
151546.704073608998442,
151556.462708784820279,
151566.221401752467500,
151575.980152508564061,
151585.738961049704812,
151595.497827372630127,
151605.256751473963959,
151615.015733350417577,
151624.774772998556728,
151634.533870415150886,
151644.293025596794905,
151654.052238540200051,
151663.811509241961176,
151673.570837698760442,
151683.330223907338222,
151693.089667864289368,
151702.849169566325145,
151712.608729010011302,
151722.368346192146419,
151732.128021109325346,
151741.887753758230247,
151751.647544135485077,
151761.407392237830209,
151771.167298061889596,
151780.927261604316300,
151790.687282861850690,
151800.447361831087619,
151810.207498508709250,
151819.967692891426850,
151829.727944975893479,
151839.488254758733092,
151849.248622236686060,
151859.009047406376339,
151868.769530264515197,
151878.530070807697484,
151888.290669032692676,
151898.051324936124729,
151907.812038514704909,
151917.572809764998965,
151927.333638683805475,
151937.094525267748395,
151946.855469513480784,
151956.616471417743014,
151966.377530977100832,
151976.138648188381921,
151985.899823048152030,
151995.661055553093320,
152005.422345699917059,
152015.183693485305412,
152024.945098905911436,
152034.706561958417296,
152044.468082639505155,
152054.229660945857177,
152063.991296874155523,
152073.752990421082359,
152083.514741583290743,
152093.276550357491942,
152103.038416740368120,
152112.800340728572337,
152122.562322318815859,
152132.324361507780850,
152142.086458292120369,
152151.848612668545684,
152161.610824633709854,
152171.373094184324145,
152181.135421317070723,
152190.897806028631749,
152200.660248315689387,
152210.422748174925800,
152220.185305603023153,
152229.947920596692711,
152239.710593152558431,
152249.473323267389787,
152259.236110937810736,
152268.998956160561647,
152278.761858932295581,
152288.524819249694701,
152298.287837109441170,
152308.050912508275360,
152317.814045442850329,
152327.577235909819137,
152337.340483905951260,
152347.103789427870652,
152356.867152472346788,
152366.630573035974521,
152376.394051115523325,
152386.157586707617156,
152395.921179808996385,
152405.684830416314071,
152415.448538526368793,
152425.212304135697195,
152434.976127241097856,
152444.740007839252939,
152454.503945926873712,
152464.267941500584129,
152474.031994557095459,
152483.796105093177175,
152493.560273105482338,
152503.324498590693111,
152513.088781545491656,
152522.853121966618346,
152532.617519850726239,
152542.381975194584811,
152552.146487994818017,
152561.911058248166228,
152571.675685951340711,
152581.440371100994525,
152591.205113693868043,
152600.969913726585219,
152610.734771196002839,
152620.499686098628445,
152630.264658431347925,
152640.029688190697925,
152649.794775373506127,
152659.559919976425590,
152669.325121996167582,
152679.090381429443369,
152688.855698272876907,
152698.621072523324983,
152708.386504177353345,
152718.151993231702363,
152727.917539683112409,
152737.683143528323853,
152747.448804763931548,
152757.214523386734072,
152766.980299393384485,
152776.746132780623157,
152786.512023545132251,
152796.277971683652140,
152806.043977192894090,
152815.810040069511160,
152825.576160310214618,
152835.342337911832146,
152845.108572870929493,
152854.874865184276132,
152864.641214848641539,
152874.407621860591462,
152884.174086216953583,
152893.940607914410066,
152903.707186949672177,
152913.473823319451185,
152923.240517020516563,
152933.007268049463164,
152942.774076403089566,
152952.540942078077933,
152962.307865071139531,
152972.074845379043836,
152981.841882998443907,
152991.608977926051011,
153001.376130158663727,
153011.143339692876907,
153020.910606525518233,
153030.677930653240765,
153040.445312072784873,
153050.212750780832721,
153059.980246774153784,
153069.747800049459329,
153079.515410603460623,
153089.283078432839829,
153099.050803534366423,
153108.818585904722568,
153118.586425540706841,
153128.354322438914096,
153138.122276596142910,
153147.890288009133656,
153157.658356674568495,
153167.426482589158695,
153177.194665749673732,
153186.962906152766664,
153196.731203795265174,
153206.499558673822321,
153216.267970785149373,
153226.036440126015805,
153235.804966693132883,
153245.573550483182771,
153255.342191492934944,
153265.110889719129773,
153274.879645158449421,
153284.648457807692466,
153294.417327663482865,
153304.186254722648300,
153313.955238981841831,
153323.724280437803827,
153333.493379087303765,
153343.262534927023808,
153353.031747953762533,
153362.801018164202105,
153372.570345554995583,
153382.339730123028858,
153392.109171864954988,
153401.878670777485240,
153411.648226857359987,
153421.417840101348702,
153431.187510506133549,
153440.957238068542210,
153450.727022785169538,
153460.496864652843215,
153470.266763668274507,
153480.036719828145579,
153489.806733129313216,
153499.576803568401374,
153509.346931142150424,
153519.117115847417153,
153528.887357680796413,
153538.657656639086781,
153548.428012718999526,
153558.198425917333225,
153567.968896230740938,
153577.739423656021245,
153587.510008189914515,
153597.280649829073809,
153607.051348570385017,
153616.822104410472093,
153626.592917346075410,
153636.363787373993546,
153646.134714490995975,
153655.905698693677550,
153665.676739978924161,
153675.447838343417970,
153685.218993783899350,
153694.990206297108671,
153704.761475879844511,
153714.532802528759930,
153724.304186240653507,
153734.075627012294717,
153743.847124840336619,
153753.618679721635999,
153763.390291652845917,
153773.161960630764952,
153782.933686652133474,
153792.705469713691855,
153802.477309812151361,
153812.249206944281468,
153822.021161106851650,
153831.793172296631383,
153841.565240510273725,
153851.337365744635463,
153861.109547996398760,
153870.881787262303988,
153880.654083539149724,
153890.426436823676340,
153900.198847112566000,
153909.971314402675489,
153919.743838690686971,
153929.516419973399024,
153939.289058247493813,
153949.061753509769915,
153958.834505756996805,
153968.607314985885751,
153978.380181193206226,
153988.153104375727708,
153997.926084530190565,
154007.699121653335169,
154017.472215741930995,
154027.245366792747518,
154037.018574802525109,
154046.791839768004138,
154056.565161685983185,
154066.338540553144412,
154076.111976366344607,
154085.885469122265931,
154095.659018817677861,
154105.432625449349871,
154115.206289014051436,
154124.980009508552030,
154134.753786929533817,
154144.527621273853583,
154154.301512538251700,
154164.075460719410330,
154173.849465814157156,
154183.623527819232550,
154193.397646731435088,
154203.171822547505144,
154212.946055264183087,
154222.720344878267497,
154232.494691386469640,
154242.269094785617199,
154252.043555072392337,
154261.818072243622737,
154271.592646296077874,
154281.367277226469014,
154291.141965031623840,
154300.916709708282724,
154310.691511253186036,
154320.466369663132355,
154330.241284934832947,
154340.016257065144600,
154349.791286050807685,
154359.566371888504364,
154369.341514575091423,
154379.116714107338339,
154388.891970481985481,
154398.667283695802325,
154408.442653745529242,
154418.218080628023017,
154427.993564339994919,
154437.769104878185317,
154447.544702239392791,
154457.320356420445023,
154467.096067418053281,
154476.871835228957934,
154486.647659849986667,
154496.423541277938057,
154506.199479509552475,
154515.975474541541189,
154525.751526370790089,
154535.527634993981337,
154545.303800407971721,
154555.080022609472508,
154564.856301595224068,
154574.632637362112291,
154584.409029906848446,
154594.185479226231109,
154603.961985316971550,
154613.738548175926553,
154623.515167799894698,
154633.291844185558148,
154643.068577329715481,
154652.845367229223484,
154662.622213880793424,
154672.399117281223880,
154682.176077427284326,
154691.953094315744238,
154701.730167943431297,
154711.507298307114979,
154721.284485403535655,
154731.061729229491903,
154740.839029781753197,
154750.616387057147222,
154760.393801052472554,
154770.171271764411358,
154779.948799189791316,
154789.726383325440111,
154799.504024168127216,
154809.281721714593004,
154819.059475961665157,
154828.837286906113150,
154838.615154544735560,
154848.393078874301864,
154858.171059891610639,
154867.949097593402257,
154877.727191976533504,
154887.505343037773855,
154897.283550773863681,
154907.061815181659767,
154916.840136257902486,
154926.618513999390416,
154936.396948402922135,
154946.175439465296222,
154955.953987183253048,
154965.732591553649399,
154975.511252573254751,
154985.289970238809474,
154995.068744547199458,
155004.847575495106867,
155014.626463079417590,
155024.405407296901103,
155034.184408144268673,
155043.963465618464397,
155053.742579716170439,
155063.521750434185378,
155073.300977769336896,
155083.080261718423571,
155092.859602278185776,
155102.638999445480295,
155112.418453217134811,
155122.197963589831488,
155131.977530560456216,
155141.757154125749366,
155151.536834282567725,
155161.316571027680766,
155171.096364357857965,
155180.876214269956108,
155190.656120760715567,
155200.436083826934919,
155210.216103465500055,
155219.996179673122242,
155229.776312446629163,
155239.556501782819396,
155249.336747678491520,
155259.117050130473217,
155268.897409135504859,
155278.677824690472335,
155288.458296792086912,
155298.238825437234482,
155308.019410622684518,
155317.800052345206495,
155327.580750601657201,
155337.361505388806108,
155347.142316703480901,
155356.923184542480158,
155366.704108902602457,
155376.485089780617272,
155386.266127173410496,
155396.047221077693393,
155405.828371490380960,
155415.609578408213565,
155425.390841827989789,
155435.172161746566417,
155444.953538160712924,
155454.734971067256993,
155464.516460462938994,
155474.298006344673922,
155484.079608709231252,
155493.861267553409562,
155503.642982874036534,
155513.424754667881643,
155523.206582931772573,
155532.988467662566109,
155542.770408857002622,
155552.552406511968002,
155562.334460624202620,
155572.116571190592367,
155581.898738207877614,
155591.680961672915146,
155601.463241582532646,
155611.245577933528693,
155621.027970722643659,
155630.810419946821639,
155640.592925602803007,
155650.375487687386340,
155660.158106197457528,
155669.940781129756942,
155679.723512481170474,
155689.506300248496700,
155699.289144428505097,
155709.072045018023346,
155718.855002013937337,
155728.638015412987443,
155738.421085212059552,
155748.204211407923140,
155757.987393997405889,
155767.770632977364585,
155777.553928344568703,
155787.337280095875030,
155797.120688228053041,
155806.904152737988625,
155816.687673622480361,
155826.471250878355931,
155836.254884502413915,
155846.038574491481995,
155855.822320842416957,
155865.606123552017380,
155875.389982617110945,
155885.173898034496233,
155894.957869801030029,
155904.741897913481807,
155914.525982368795667,
155924.310123163711978,
155934.094320295058424,
155943.878573759691790,
155953.662883554381551,
155963.447249676013598,
155973.231672121386509,
155983.016150887327967,
155992.800685970723862,
156002.585277368314564,
156012.369925076927757,
156022.154629093449330,
156031.939389414736070,
156041.724206037557451,
156051.509078958741156,
156061.294008175173076,
156071.078993683622684,
156080.864035480975872,
156090.649133564002113,
156100.434287929587299,
156110.219498574530007,
156120.004765495657921,
156129.790088689856930,
156139.575468153954716,
156149.360903884691652,
156159.146395878953626,
156168.931944133626530,
156178.717548645508941,
156188.503209411428543,
156198.288926428242121,
156208.074699692748254,
156217.860529201832833,
156227.646414952294435,
156237.432356940989848,
156247.218355164717650,
156257.004409620334627,
156266.790520304726670,
156276.576687214692356,
156286.362910347059369,
156296.149189698713599,
156305.935525266453624,
156315.721917047078023,
156325.508365037530893,
156335.294869234552607,
156345.081429635087261,
156354.868046235875227,
156364.654719033802394,
156374.441448025754653,
156384.228233208501479,
156394.015074578928761,
156403.801972133835079,
156413.588925870135427,
156423.375935784599278,
156433.163001874112524,
156442.950124135531951,
156452.737302565656137,
156462.524537161370972,
156472.311827919533243,
156482.099174836912425,
156491.886577910452615,
156501.674037136894185,
156511.461552513210336,
156521.249124036112335,
156531.036751702573383,
156540.824435509362957,
156550.612175453337841,
156560.399971531413030,
156570.187823740328895,
156579.975732077029534,
156589.763696538284421,
156599.551717121037655,
156609.339793822058709,
156619.127926638233475,
156628.916115566389635,
156638.704360603442183,
156648.492661746131489,
156658.281018991430756,
156668.069432336109458,
156677.857901777082589,
156687.646427311148727,
156697.435008935193764,
156707.223646646016277,
156717.012340440560365,
156726.801090315653710,
156736.589896268094890,
156746.378758294798899,
156756.167676392593421,
156765.956650558335241,
156775.745680788881145,
156785.534767081117025,
156795.323909431870561,
156805.113107837998541,
156814.902362296386855,
156824.691672803834081,
156834.481039357255213,
156844.270461953536142,
156854.059940589446342,
156863.849475261929911,
156873.639065967756324,
156883.428712703869678,
156893.218415467097657,
156903.008174254326150,
156912.797989062353736,
156922.587859888095409,
156932.377786728407955,
156942.167769580148160,
156951.957808440201916,
156961.747903305367799,
156971.538054172560805,
156981.328261038637720,
156991.118523900455330,
157000.908842754899524,
157010.699217598797986,
157020.489648429036606,
157030.280135242472170,
157040.070678036019672,
157049.861276806477690,
157059.651931550732115,
157069.442642265668837,
157079.233408948115539,
157089.024231594987214,
157098.815110203169752,
157108.606044769461732,
157118.397035290778149,
157128.188081763946684,
157137.979184185882332,
157147.770342553441878,
157157.561556863511214,
157167.352827112918021,
157177.144153298548190,
157186.935535417316714,
157196.726973466080381,
157206.518467441666871,
157216.310017340932973,
157226.101623160822783,
157235.893284898163984,
157245.685002549871569,
157255.476776112773223,
157265.268605583754834,
157275.060490959702292,
157284.852432237501489,
157294.644429414009210,
157304.436482486111345,
157314.228591450635577,
157324.020756304555107,
157333.812977044639410,
157343.605253667861689,
157353.397586171020521,
157363.189974551030900,
157372.982418804778717,
157382.774918929149862,
157392.567474920972018,
157402.360086777160177,
157412.152754494571127,
157421.945478070119862,
157431.738257500663167,
157441.531092783116037,
157451.323983914306154,
157461.116930891148513,
157470.909933710499899,
157480.702992369275307,
157490.496106864302419,
157500.289277192554437,
157510.082503350829938,
157519.875785336043919,
157529.669123145082267,
157539.462516774830874,
157549.255966222175630,
157559.049471483944217,
157568.843032557109836,
157578.636649438500172,
157588.430322125030216,
157598.224050613585860,
157608.017834900994785,
157617.811674984230194,
157627.605570860148873,
157637.399522525665816,
157647.193529977579601,
157656.987593212805223,
157666.781712228286779,
157676.575887020881055,
157686.370117587473942,
157696.164403924951330,
157705.958746030228212,
157715.753143900190480,
157725.547597531694919,
157735.342106921656523,
157745.136672067019390,
157754.931292964523891,
157764.725969611230539,
157774.520702003908809,
157784.315490139531903,
157794.110334014927503,
157803.905233627068810,
157813.700188972754404,
157823.495200048957486,
157833.290266852563946,
157843.085389380430570,
157852.880567629443249,
157862.675801596546080,
157872.471091278566746,
157882.266436672507552,
157892.061837775167078,
157901.857294583489420,
157911.652807094389573,
157921.448375304695219,
157931.243999211350456,
157941.039678811270278,
157950.835414101311471,
157960.631205078359926,
157970.427051739388844,
157980.222954081255011,
157990.018912100844318,
157999.814925795071758,
158009.610995160852326,
158019.407120195071911,
158029.203300894587301,
158038.999537256400799,
158048.795829277310986,
158058.592176954291062,
158068.388580284197815,
158078.185039264033549,
158087.981553890538635,
158097.778124160715379,
158107.574750071478775,
158117.371431619685609,
158127.168168802279979,
158136.964961616147775,
158146.761810058203992,
158156.558714125334518,
158166.355673814454349,
158176.152689122507581,
158185.949760046321899,
158195.746886582841398,
158205.544068729010178,
158215.341306481743231,
158225.138599837839138,
158234.935948794329306,
158244.733353348070523,
158254.530813495919574,
158264.328329234907869,
158274.125900561833987,
158283.923527473671129,
158293.721209967334289,
158303.518948039680254,
158313.316741687653121,
158323.114590908167884,
158332.912495698139537,
158342.710456054453971,
158352.508471974026179,
158362.306543453829363,
158372.104670490691205,
158381.902853081584908,
158391.701091223396361,
158401.499384913069662,
158411.297734147490701,
158421.096138923545368,
158430.894599238206865,
158440.693115088390186,
158450.491686470981222,
158460.290313382894965,
158470.088995821075514,
158479.887733782408759,
158489.686527263838798,
158499.485376262251521,
158509.284280774591025,
158519.083240797801409,
158528.882256328739459,
158538.681327364349272,
158548.480453901545843,
158558.279635937244166,
158568.078873468417441,
158577.878166491951561,
158587.677515004761517,
158597.476919003762305,
158607.276378485868918,
158617.075893447996350,
158626.875463887117803,
158636.675089800119167,
158646.474771183915436,
158656.274508035479812,
158666.074300351639977,
158675.874148129485548,
158685.674051365727792,
158695.474010057427222,
158705.274024201498833,
158715.074093794828514,
158724.874218834418571,
158734.674399317038478,
158744.474635239777854,
158754.274926599493483,
158764.075273393100360,
158773.875675617571687,
158783.676133269793354,
158793.476646346680354,
158803.277214845205890,
158813.077838762284955,
158822.878518094861647,
158832.679252839821856,
158842.480042994080577,
158852.280888554640114,
158862.081789518415462,
158871.882745882321615,
158881.683757643273566,
158891.484824798186310,
158901.285947344062151,
158911.087125277786981,
158920.888358596275793,
158930.689647296472685,
158940.490991375321755,
158950.292390829796204,
158960.093845656781923,
158969.895355853193905,
158979.696921416005353,
158989.498542342102155,
158999.300218628515722,
159009.101950272102840,
159018.903737269836711,
159028.705579618574120,
159038.507477315375581,
159048.309430357097881,
159058.111438740685116,
159067.913502463052282,
159077.715621521201683,
159087.517795912048314,
159097.320025632507168,
159107.122310679551447,
159116.924651050067041,
159126.727046741027152,
159136.529497749404982,
159146.332004072086420,
159156.134565706015565,
159165.937182648136513,
159175.739854895480676,
159185.542582444875734,
159195.345365293265786,
159205.148203437653137,
159214.951096874981886,
159224.754045602108818,
159234.557049616065342,
159244.360108913766453,
159254.163223492185352,
159263.966393348178826,
159273.769618478778284,
159283.572898880898720,
159293.376234551513335,
159303.179625487478916,
159312.983071685855975,
159322.786573143472197,
159332.590129857417196,
159342.393741824518656,
159352.197409041778883,
159362.001131506141974,
159371.804909214493819,
159381.608742163865827,
159391.412630351202097,
159401.216573773417622,
159411.020572427456500,
159420.824626310262829,
159430.628735418780707,
159440.432899750070646,
159450.237119300931226,
159460.041394068393856,
159469.845724049402634,
159479.650109240901656,
159489.454549639835022,
159499.259045243146829,
159509.063596047839383,
159518.868202050827676,
159528.672863249026705,
159538.477579639467876,
159548.282351219066186,
159558.087177984794835,
159567.892059933539713,
159577.696997062361334,
159587.501989368145587,
159597.307036847865675,
159607.112139498465694,
159616.917297316977056,
159626.722510300227441,
159636.527778445277363,
159646.333101749012712,
159656.138480208435794,
159665.943913820490707,
159675.749402582121547,
159685.554946490359725,
159695.360545542062027,
159705.166199734259862,
159714.971909063897328,
159724.777673527860316,
159734.583493123238441,
159744.389367846917594,
159754.195297695870977,
159764.001282667013584,
159773.807322757376824,
159783.613417963933898,
159793.419568283570698,
159803.225773713289527,
159813.032034250063589,
159822.838349890866084,
159832.644720632612007,
159842.451146472332766,
159852.257627406914253,
159862.064163433358772,
159871.870754548639525,
159881.677400749758817,
159891.484102033602539,
159901.290858397172997,
159911.097669837443391,
159920.904536351357820,
159930.711457935918588,
159940.518434588069795,
159950.325466304813745,
159960.132553083036328,
159969.939694919827161,
159979.746891812013928,
159989.554143756657140,
159999.361450750759104,
160009.168812791176606,
160018.976229874970159,
160028.783701999083860,
160038.591229160432704,
160048.398811356077204,
160058.206448583019665,
160068.014140838058665,
160077.821888118312927,
160087.629690420726547,
160097.437547742243623,
160107.245460079837358,
160117.053427430539159,
160126.861449791234918,
160136.669527158985147,
160146.477659530704841,
160156.285846903367201,
160166.094089274003636,
160175.902386639529141,
160185.710738996946020,
160195.519146343227476,
160205.327608675375814,
160215.136125990335131,
160224.944698285049526,
160234.753325556579512,
160244.562007801840082,
160254.370745017804438,
160264.179537201533094,
160273.988384349882836,
160283.797286459914176,
160293.606243528629420,
160303.415255552943563,
160313.224322529829806,
160323.033444456377765,
160332.842621329444228,
160342.651853146031499,
160352.461139903170988,
160362.270481597806793,
160372.079878226941219,
160381.889329787547467,
160391.698836276598740,
160401.508397691126447,
160411.318014028045582,
160421.127685284387553,
160430.937411457125563,
160440.747192543203710,
160450.557028539653402,
160460.366919443476945,
160470.176865251560230,
160479.986865961021977,
160489.796921568777179,
160499.607032071799040,
160509.417197467148071,
160519.227417751651956,
160529.037692922458518,
160538.848022976540960,
160548.658407910814276,
160558.468847722309874,
160568.279342408000957,
160578.089891964860726,
160587.900496389920590,
160597.711155680153752,
160607.521869832533412,
160617.332638844061876,
160627.143462711741449,
160636.954341432545334,
160646.765275003475836,
160656.576263421477051,
160666.387306683667703,
160676.198404786875471,
160686.009557728248183,
160695.820765504642623,
160705.632028113119304,
160715.443345550709637,
160725.254717814299511,
160735.066144901036751,
160744.877626807749039,
160754.689163531525992,
160764.500755069369916,
160774.312401418224908,
160784.124102575180586,
160793.935858537093736,
160803.747669301053975,
160813.559534864034504,
160823.371455223037628,
160833.183430375065655,
160842.995460317120887,
160852.807545046176529,
160862.619684559263987,
160872.431878853356466,
160882.244127925456269,
160892.056431772565702,
160901.868790391687071,
160911.681203779822681,
160921.493671933974838,
160931.306194851174951,
160941.118772528308909,
160950.931404962524539,
160960.744092150736833,
160970.556834089948097,
160980.369630777218845,
160990.182482209493173,
160999.995388383802492,
161009.808349297120003,
161019.621364946477115,
161029.434435328905238,
161039.247560441348469,
161049.060740280896425,
161058.873974844434997,
161068.687264129082905,
161078.500608131726040,
161088.314006849483121,
161098.127460279298248,
161107.940968418231932,
161117.754531263257377,
161127.568148811347783,
161137.381821059563663,
161147.195548004878219,
161157.009329644322861,
161166.823165974899894,
161176.637056993640726,
161186.451002697518561,
161196.265003083535703,
161206.079058148752665,
161215.893167890142649,
161225.707332304737065,
161235.521551389480010,
161245.335825141461100,
161255.150153557711747,
161264.964536635146942,
161274.778974370856304,
161284.593466761813033,
161294.408013805048540,
161304.222615497594234,
161314.037271836423315,
161323.851982818567194,
161333.666748441028176,
161343.481568700866774,
161353.296443595056189,
161363.111373120598728,
161372.926357274554903,
161382.741396053897915,
161392.556489455659175,
161402.371637476899195,
161412.186840114562074,
161422.002097365708323,
161431.817409227340249,
161441.632775696460158,
161451.448196770128561,
161461.263672445318662,
161471.079202719120076,
161480.894787588476902,
161490.710427050478756,
161500.526121102040634,
161510.341869740252150,
161520.157672962144716,
161529.973530764720635,
161539.789443145011319,
161549.605410100019071,
161559.421431626746198,
161569.237507722282317,
161579.053638383600628,
161588.869823607761646,
161598.686063391738571,
161608.502357732562814,
161618.318706627265783,
161628.135110072878888,
161637.951568066462642,
161647.768080604990246,
161657.584647685493110,
161667.401269304973539,
161677.217945460550254,
161687.034676149138249,
161696.851461367856245,
161706.668301113648340,
161716.485195383633254,
161726.302144174755085,
161736.119147484045243,
161745.936205308564240,
161755.753317645372590,
161765.570484491443494,
161775.387705843837466,
161785.204981699527707,
161795.022312055632938,
161804.839696909097256,
161814.657136257039383,
161824.474630096374312,
161834.292178424249869,
161844.109781237639254,
161853.927438533544773,
161863.745150309056044,
161873.562916561204474,
161883.380737286963267,
161893.198612483451143,
161903.016542147612199,
161912.834526276565157,
161922.652564867225010,
161932.470657916768687,
161942.288805422140285,
161952.107007380400319,
161961.925263788551092,
161971.743574643682223,
161981.561939942796016,
161991.380359682982089,
162001.198833861155435,
162011.017362474492984,
162020.835945519909728,
162030.654582994553493,
162040.473274895339273,
162050.292021219443996,
162060.110821963840863,
162069.929677125532180,
162079.748586701607564,
162089.567550689040218,
162099.386569084948860,
162109.205641886335798,
162119.024769090261543,
162128.843950693757506,
162138.663186693825992,
162148.482477087556617,
162158.301821872009896,
162168.121221044129925,
162177.940674601093633,
162187.760182539786911,
162197.579744857415790,
162207.399361550895264,
162217.219032617344055,
162227.038758053793572,
162236.858537857246120,
162246.678372024820419,
162256.498260553460568,
162266.318203440285288,
162276.138200682325987,
162285.958252276614076,
162295.778358220180962,
162305.598518510145368,
162315.418733143480495,
162325.239002117275959,
162335.059325428534066,
162344.879703074373538,
162354.700135051767575,
162364.520621357805794,
162374.341161989519605,
162384.161756943969522,
162393.982406218186952,
162403.803109809232410,
162413.623867714166408,
162423.444679930020357,
162433.265546453883871,
162443.086467282788362,
162452.907442413736135,
162462.728471843845909,
162472.549555570090888,
162482.370693589647999,
162492.191885899432236,
162502.013132496591425,
162511.834433378127869,
162521.655788541131187,
162531.477197982632788,
162541.298661699664081,
162551.120179689372890,
162560.941751948674209,
162570.763378474744968,
162580.585059264558367,
162590.406794315233128,
162600.228583623800660,
162610.050427187292371,
162619.872325002797879,
162629.694277067348594,
162639.516283378034132,
162649.338343931827694,
162659.160458725935314,
162668.982627757271985,
162678.804851022985531,
162688.627128520078259,
162698.449460245668888,
162708.271846196759725,
162718.094286370498594,
162727.916780763771385,
162737.739329373813234,
162747.561932197597343,
162757.384589232213330,
162767.207300474721706,
162777.030065922212088,
162786.852885571657680,
162796.675759420206305,
162806.498687464889372,
162816.321669702738291,
162826.144706130929990,
162835.967796746350359,
162845.790941546205431,
162855.614140527555719,
162865.437393687345320,
162875.260701022751164,
162885.084062530833762,
162894.907478208595421,
162904.730948053154862,
162914.554472061514389,
162924.378050230850931,
162934.201682558137691,
162944.025369040464284,
162953.849109674920328,
162963.672904458508128,
162973.496753388404613,
162983.320656461612089,
162993.144613675161963,
163002.968625026231166,
163012.792690511763794,
163022.616810128936777,
163032.440983874752419,
163042.265211746300338,
163052.089493740670150,
163061.913829854864161,
163071.738220086030196,
163081.562664431257872,
163091.387162887520390,
163101.211715451994678,
163111.036322121683042,
163120.860982893645996,
163130.685697765031364,
163140.510466732841451,
163150.335289794224082,
163160.160166946152458,
163169.985098185774405,
163179.810083510179538,
163189.635122916399268,
163199.460216401465004,
163209.285363962553674,
163219.110565596696688,
163228.935821300983662,
163238.761131072475109,
163248.586494908202440,
163258.411912805313477,
163268.237384760868736,
163278.062910771928728,
163287.888490835583070,
163297.714124948950484,
163307.539813109004172,
163317.365555312950164,
163327.191351557732560,
163337.017201840557391,
163346.843106158456067,
163356.669064508459996,
163366.495076887717005,
163376.321143293287605,
163386.147263722232310,
163395.973438171669841,
163405.799666638689814,
163415.625949120265432,
163425.452285613631830,
163435.278676115820417,
163445.105120623804396,
163454.931619134848006,
163464.758171645866241,
163474.584778154094238,
163484.411438656534301,
163494.238153150246944,
163504.064921632379992,
163513.891744099993957,
163523.718620550149353,
163533.545550979935797,
163543.372535386501113,
163553.199573766876711,
163563.026666118152207,
163572.853812437446322,
163582.681012721790466,
163592.508266968332464,
163602.335575174132828,
163612.162937336252071,
163621.990353451867122,
163631.817823517980287,
163641.645347531710286,
163651.472925490146736,
163661.300557390379254,
163671.128243229497457,
163680.955983004590962,
163690.783776712749386,
163700.611624351062346,
163710.439525916677667,
163720.267481406597653,
163730.095490817941027,
163739.923554147826508,
163749.751671393372817,
163759.579842551553156,
163769.408067619602662,
163779.236346594552742,
163789.064679473463912,
163798.893066253513098,
163808.721506931731710,
163818.550001505238470,
163828.378549971093889,
163838.207152326445794,
163848.035808568354696,
163857.864518693939317,
163867.693282700260170,
163877.522100584494183,
163887.350972343614558,
163897.179897974827327,
163907.008877475222107,
163916.837910841801204,
163926.666998071770649,
163936.496139162161853,
163946.325334110122640,
163956.154582912713522,
163965.983885567053221,
163975.813242070260458,
163985.642652419395745,
163995.472116611577803,
164005.301634643954458,
164015.131206513498910,
164024.960832217446296,
164034.790511752857128,
164044.620245116762817,
164054.450032306398498,
164064.279873318766477,
164074.109768151014578,
164083.939716800203314,
164093.769719263451407,
164103.599775537906680,
164113.429885620658752,
164123.260049508797238,
164133.090267199382652,
164142.920538689591922,
164152.750863976485562,
164162.581243057211395,
164172.411675928829936,
164182.242162588459905,
164192.072703033249127,
164201.903297260287218,
164211.733945266663795,
164221.564647049439372,
164231.395402605819982,
164241.226211932866136,
164251.057075027667452,
164260.887991887371754,
164270.718962509097764,
164280.549986889905995,
164290.381065026886063,
164300.212196917214897,
164310.043382558011217,
164319.874621946364641,
164329.705915079306578,
164339.537261954101268,
164349.368662567721913,
164359.200116917374544,
164369.031625000119675,
164378.863186813076027,
164388.694802353391424,
164398.526471618097275,
164408.358194604428718,
164418.189971309417160,
164428.021801730210427,
164437.853685863839928,
164447.685623707540799,
164457.517615258402657,
164467.349660513456911,
164477.181759469909593,
164487.013912124850322,
164496.846118475397816,
164506.678378518641694,
164516.510692251700675,
164526.343059671751689,
164536.175480775855249,
164546.007955561159179,
164555.840484024694888,
164565.673066163755720,
164575.505701975314878,
164585.338391456578393,
164595.171134604519466,
164605.003931416460546,
164614.836781889403937,
164624.669686020497466,
164634.502643806830747,
164644.335655245551607,
164654.168720333807869,
164664.001839068660047,
164673.835011447285069,
164683.668237466801656,
164693.501517124299426,
164703.334850416897098,
164713.168237341742497,
164723.001677895983448,
164732.835172076709569,
164742.668719881039578,
164752.502321306121303,
164762.335976349015255,
164772.169685006956570,
164782.003447277005762,
164791.837263156310655,
164801.671132641931763,
164811.505055731104221,
164821.339032420888543,
164831.173062708374346,
164841.007146590767661,
164850.841284065187210,
164860.675475128722610,
164870.509719778521685,
164880.344018011703156,
164890.178369825385744,
164900.012775216717273,
164909.847234182874672,
164919.681746720918454,
164929.516312827996444,
164939.350932501198258,
164949.185605737759033,
164959.020332534710178,
164968.855112889257725,
164978.689946798462188,
164988.524834259500494,
164998.359775269520469,
165008.194769825582625,
165018.029817924922099,
165027.864919564570300,
165037.700074741733260,
165047.535283453529701,
165057.370545697078342,
165067.205861469527008,
165077.041230767936213,
165086.876653589570196,
165096.712129931489471,
165106.547659790841863,
165116.383243164804298,
165126.218880050408188,
165136.054570444859564,
165145.890314345306251,
165155.726111748896074,
165165.561962652718648,
165175.397867053921800,
165185.233824949653354,
165195.069836337090237,
165204.905901213292964,
165214.742019575467566,
165224.578191420674557,
165234.414416746178176,
165244.250695549038937,
165254.087027826375561,
165263.923413575394079,
165273.759852793155005,
165283.596345476864371,
165293.432891623670002,
165303.269491230661515,
165313.106144294986734,
165322.942850813851692,
165332.779610784316901,
165342.616424203559291,
165352.453291068784893,
165362.290211377054220,
165372.127185125515098,
165381.964212311373558,
165391.801292931690114,
165401.638426983670797,
165411.475614464463433,
165421.312855371186743,
165431.150149700988550,
165440.987497451045783,
165450.824898618418956,
165460.662353200343205,
165470.499861193966353,
165480.337422596378019,
165490.175037404784234,
165500.012705616303720,
165509.850427227996988,
165519.688202237186488,
165529.526030640932731,
165539.363912436354440,
165549.201847620657645,
165559.039836190961068,
165568.877878144412534,
165578.715973478188971,
165588.554122189409100,
165598.392324275249848,
165608.230579732859042,
165618.068888559384504,
165627.907250751915853,
165637.745666307717329,
165647.584135223878548,
165657.422657497576438,
165667.261233125958825,
165677.099862106144428,
165686.938544435310178,
165696.777280110603897,
165706.616069129173411,
165716.454911488224752,
165726.293807184876641,
165736.132756216306007,
165745.971758579602465,
165755.810814272001153,
165765.649923290562583,
165775.489085632550996,
165785.328301295056008,
165795.167570275283651,
165805.006892570352647,
165814.846268177410820,
165824.685697093664203,
165834.525179316173308,
165844.364714842231479,
165854.204303668928333,
165864.043945793411694,
165873.883641212887596,
165883.723389924445655,
165893.563191925321007,
165903.403047212574165,
165913.242955783469370,
165923.082917635125341,
165932.922932764719008,
165942.763001169369090,
165952.603122846281622,
165962.443297792633530,
165972.283526005543536,
165982.123807482130360,
165991.964142219658243,
166001.804530215275008,
166011.644971466099378,
166021.485465969279176,
166031.326013722049538,
166041.166614721558290,
166051.007268964895047,
166060.847976449353155,
166070.688737171934918,
166080.529551129933679,
166090.370418320497265,
166100.211338740773499,
166110.052312387910206,
166119.893339259113418,
166129.734419351501856,
166139.575552662310656,
166149.416739188658539,
166159.257978927693330,
166169.099271876650164,
166178.940618032618659,
166188.782017392863054,
166198.623469954502070,
166208.464975714683533,
166218.306534670584369,
166228.148146819410613,
166237.989812158310087,
166247.831530684459722,
166257.673302394978236,
166267.515127287129872,
166277.357005358033348,
166287.198936604865594,
166297.040921024832642,
166306.882958615053212,
166316.725049372704234,
166326.567193295020843,
166336.409390379092656,
166346.251640622183913,
166356.093944021355128,
166365.936300573899643,
166375.778710276907077,
166385.621173127612565,
166395.463689123163931,
166405.306258260738105,
166415.148880537512014,
166424.991555950633483,
166434.834284497337649,
166444.677066174772335,
166454.519900980056264,
166464.362788910482777,
166474.205729963112390,
166484.048724135238444,
166493.891771423979662,
166503.734871826483868,
166513.578025339986198,
166523.421231961634476,
166533.264491688605631,
166543.107804518105695,
166552.951170447340701,
166562.794589473429369,
166572.638061593548628,
166582.481586804904509,
166592.325165104673943,
166602.168796490062959,
166612.012480958219385,
166621.856218506320147,
166631.700009131600382,
166641.543852831237018,
166651.387749602348777,
166661.231699442170793,
166671.075702347879997,
166680.919758316653315,
166690.763867345638573,
166700.608029432129115,
166710.452244573185453,
166720.296512766071828,
166730.140834007936064,
166739.985208295984194,
166749.829635627393145,
166759.674115999368951,
166769.518649409001227,
166779.363235853612423,
166789.207875330379466,
166799.052567836362869,
166808.897313368914183,
166818.742111925093923,
166828.586963502166327,
166838.431868097250117,
166848.276825707609532,
166858.121836330363294,
166867.966899962775642,
166877.812016601965297,
166887.657186245196499,
166897.502408889587969,
166907.347684532374842,
166917.193013170734048,
166927.038394801900722,
166936.883829422964482,
166946.729317031218670,
166956.574857623811113,
166966.420451197889633,
166976.266097750747576,
166986.111797279532766,
166995.957549781363923,
167005.803355253563495,
167015.649213693279307,
167025.495125097688287,
167035.341089463967364,
167045.187106789351674,
167055.033177071018144,
167064.879300306143705,
167074.725476491992595,
167084.571705625683535,
167094.417987704451662,
167104.264322725473903,
167114.110710686014500,
167123.957151583163068,
167133.803645414154744,
167143.650192176253768,
167153.496791866578860,
167163.343444482365157,
167173.190150020789588,
167183.036908479087288,
167192.883719854435185,
167202.730584144039312,
167212.577501345076598,
167222.424471454753075,
167232.271494470332982,
167242.118570388905937,
167251.965699207823491,
167261.812880924146157,
167271.660115535109071,
167281.507403037947370,
167291.354743429896189,
167301.202136708045145,
167311.049582869716687,
167320.897081912058638,
167330.744633832247928,
167340.592238627548795,
167350.439896295109065,
167360.287606832163874,
167370.135370235948358,
167379.983186503610341,
167389.831055632355856,
167399.678977619449142,
167409.526952462038025,
167419.374980157386744,
167429.223060702672228,
167439.071194095071405,
167448.919380331819411,
167458.767619410122279,
167468.615911327186041,
167478.464256080216728,
167488.312653666449478,
167498.161104083032114,
167508.009607327228878,
167517.858163396274904,
167527.706772287288914,
167537.555433997535147,
167547.404148524219636,
167557.252915864577517,
167567.101736015756615,
167576.950608975021169,
167586.799534739548108,
167596.648513306601672,
167606.497544673329685,
167616.346628836996388,
167626.195765794778708,
167636.044955543911783,
167645.894198081572540,
167655.743493405054323,
167665.592841511475854,
167675.442242398101371,
167685.291696062136907,
167695.141202500817599,
167704.990761711320374,
167714.840373690909473,
167724.690038436732721,
167734.539755946054356,
167744.389526216109516,
167754.239349244016921,
167764.089225027128123,
167773.939153562561842,
167783.789134847611422,
167793.639168879395584,
167803.489255655236775,
167813.339395172224613,
167823.189587427739752,
167833.039832418871811,
167842.890130142885027,
167852.740480597014539,
167862.590883778466377,
167872.441339684446575,
167882.291848312190268,
167892.142409658903489,
167901.993023721821373,
167911.843690498179058,
167921.694409985153470,
167931.545182180067059,
167941.396007079980336,
167951.246884682273958,
167961.097814984066645,
167970.948797982622636,
167980.799833675147966,
167990.650922058877768,
168000.502063131018076,
168010.353256888862234,
168020.204503329558065,
168030.055802450369811,
168039.907154248474399,
168049.758558721136069,
168059.610015865589958,
168069.461525679012993,
168079.313088158669416,
168089.164703301794361,
168099.016371105622966,
168108.868091567332158,
168118.719864684186177,
168128.571690453361953,
168138.423568872181932,
168148.275499937793938,
168158.127483647462213,
168167.979519998392789,
168177.831608987908112,
168187.683750613068696,
168197.535944871196989,
168207.388191759586334,
168217.240491275326349,
168227.092843415739480,
168236.945248178060865,
168246.797705559467431,
168256.650215557223419,
168266.502778168593068,
168276.355393390753306,
168286.208061220939271,
168296.060781656444306,
168305.913554694445338,
168315.766380332177505,
168325.619258566875942,
168335.472189395804890,
168345.325172816170380,
168355.178208825207548,
168365.031297420180636,
168374.884438598237466,
168384.737632356729591,
168394.590878692833940,
168404.444177603785647,
168414.297529086819850,
168424.150933139200788,
168434.004389758163597,
168443.857898940856103,
168453.711460684629856,
168463.565074986661784,
168473.418741844216129,
168483.272461254528025,
168493.126233214803506,
168502.980057722335914,
168512.833934774302179,
168522.687864367995644,
168532.541846500622341,
168542.395881169446511,
168552.249968371703289,
168562.104108104598708,
168571.958300365426112,
168581.812545151362428,
168591.666842459730105,
168601.521192287676968,
168611.375594632525463,
168621.230049491481623,
168631.084556861809688,
168640.939116740744794,
168650.793729125463869,
168660.648394013289362,
168670.503111401456408,
168680.357881287200144,
168690.212703667755704,
168700.067578540387331,
168709.922505902271951,
168719.777485750761116,
168729.632518083060859,
168739.487602896348108,
168749.342740187945310,
168759.197929955058498,
168769.053172194981016,
168778.908466904918896,
168788.763814082136378,
168798.619213723839493,
168808.474665827350691,
168818.330170389876002,
168828.185727408650564,
168838.041336880938616,
168847.896998803975293,
168857.752713175024837,
168867.608479991380591,
168877.464299250190379,
168887.320170948805753,
168897.176095084403642,
168907.032071654306492,
168916.888100655662129,
168926.744182085793000,
168936.600315941934241,
168946.456502221350092,
168956.312740921275690,
168966.169032039004378,
168976.025375571742188,
168985.881771516724257,
168995.738219871243928,
169005.594720632594544,
169015.451273797894828,
169025.307879364554537,
169035.164537329750601,
169045.021247690718155,
169054.878010444721440,
169064.734825589082902,
169074.591693120979471,
169084.448613037675386,
169094.305585336493095,
169104.162610014638631,
169114.019687069347128,
169123.876816497911932,
169133.733998297539074,
169143.591232465551002,
169153.448518999211956,
169163.305857895698864,
169173.163249152392382,
169183.020692766411230,
169192.878188735077856,
169202.735737055685604,
169212.593337725440506,
169222.450990741635906,
169232.308696101536043,
169242.166453802376054,
169252.024263841391075,
169261.882126215903554,
169271.740040923148626,
169281.598007960361429,
169291.456027324864408,
169301.314099013863597,
169311.172223024652340,
169321.030399354465771,
169330.888628000626341,
169340.746908960281871,
169350.605242230813019,
169360.463627809454920,
169370.322065693442710,
169380.180555880040629,
169390.039098366542021,
169399.897693150182022,
169409.756340228224872,
169419.615039597993018,
169429.473791256692493,
169439.332595201587537,
169449.191451429971494,
169459.050359939108603,
169468.909320726292208,
169478.768333788699238,
169488.627399123710347,
169498.486516728502465,
169508.345686600368936,
169518.204908736603102,
169528.064183134498307,
169537.923509791231481,
169547.782888704125071,
169557.642319870443316,
169567.501803287479561,
169577.361338952498045,
169587.220926862763008,
169597.080567015538691,
169606.940259408060228,
169616.800004037650069,
169626.659800901543349,
169636.519649997062515,
169646.379551321442705,
169656.239504871948157,
169666.099510645843111,
169675.959568640450016,
169685.819678853033110,
169695.679841280856635,
169705.540055921155727,
169715.400322771223728,
169725.260641828353982,
169735.121013089839835,
169744.981436552916421,
169754.841912214818876,
169764.702440072898753,
169774.563020124449395,
169784.423652366705937,
169794.284336796903517,
169804.145073412364582,
169814.005862210353371,
169823.866703188163228,
169833.727596343116602,
169843.588541672361316,
169853.449539173248922,
169863.310588843101868,
169873.171690679126186,
169883.032844678673428,
169892.894050838949624,
169902.755309157277225,
169912.616619630891364,
169922.477982257114490,
169932.339397033269051,
169942.200863956531975,
169952.062383024254814,
169961.923954233672703,
169971.785577582137194,
169981.647253066825215,
169991.508980685117422,
170001.370760434248950,
170011.232592311542248,
170021.094476314232452,
170030.956412439583801,
170040.818400684947846,
170050.680441047559725,
170060.542533524741884,
170070.404678113729460,
170080.266874811874004,
170090.129123616352445,
170099.991424524574541,
170109.853777533746324,
170119.716182641219348,
170129.578639844170539,
170139.441149139980553,
170149.303710525942734,
170159.166323999234010,
170169.028989557264140,
170178.891707197239157,
170188.754476916510612,
170198.617298712342745,
170208.480172581970692,
170218.343098522746004,
170228.206076531932922,
170238.069106606853893,
170247.932188744744053,
170257.795322942925850,
170267.658509198692627,
170277.521747509337729,
170287.385037872096291,
170297.248380284319865,
170307.111774743272690,
170316.975221246306319,
170326.838719790626783,
170336.702270373556530,
170346.565872992388904,
170356.429527644417249,
170366.293234326964011,
170376.156993037235225,
170386.020803772640647,
170395.884666530415416,
170405.748581307823770,
170415.612548102188157,
170425.476566910801921,
170435.340637730987510,
170445.204760560009163,
170455.068935395131120,
170464.933162233704934,
170474.797441073053051,
170484.661771910381503,
170494.526154743041843,
170504.390589568298310,
170514.255076383473352,
170524.119615185889415,
170533.984205972781638,
170543.848848741530674,
170553.713543489342555,
170563.578290213539731,
170573.443088911473751,
170583.307939580408856,
170593.172842217609286,
170603.037796820426593,
170612.902803386125015,
170622.767861912085209,
170632.632972395484103,
170642.498134833673248,
170652.363349223975092,
170662.228615563712083,
170672.093933850148460,
170681.959304080548463,
170691.824726252292749,
170701.690200362616451,
170711.555726408871124,
170721.421304388321005,
170731.286934298346750,
170741.152616136096185,
170751.018349899037275,
170760.884135584405158,
170770.749973189493176,
170780.615862711594673,
170790.481804148061201,
170800.347797496157000,
170810.213842753233621,
170820.079939916555304,
170829.946088983473601,
170839.812289951194543,
170849.678542817098787,
170859.544847578508779,
170869.411204232717864,
170879.277612777019385,
170889.144073208706686,
170899.010585525160423,
170908.877149723557523,
170918.743765801307745,
170928.610433755704435,
170938.477153584040934,
170948.343925283639692,
170958.210748851823155,
170968.077624285855563,
170977.944551583088469,
170987.811530740786111,
170997.678561756329145,
171007.545644626952708,
171017.412779350037454,
171027.279965922818519,
171037.147204342705663,
171047.014494606934022,
171056.881836712796940,
171066.749230657733278,
171076.616676438919967,
171086.484174053679453,
171096.351723499450600,
171106.219324773410335,
171116.086977872939315,
171125.954682795330882,
171135.822439537936589,
171145.690248097991571,
171155.558108472876484,
171165.426020659913775,
171175.293984656367684,
171185.162000459589763,
171195.030068066931562,
171204.898187475599116,
171214.766358683031285,
171224.634581686463207,
171234.502856483246433,
171244.371183070674306,
171254.239561446127482,
171264.107991606841097,
171273.976473550166702,
171283.845007273455849,
171293.713592773943674,
171303.582230049069040,
171313.450919096067082,
171323.319659912260249,
171333.188452495000092,
171343.057296841609059,
171352.926192949351389,
171362.795140815607738,
171372.664140437729657,
171382.533191812894074,
171392.402294938568957,
171402.271449812047649,
171412.140656430594390,
171422.009914791560732,
171431.879224892327329,
171441.748586730129318,
171451.618000302318251,
171461.487465606274782,
171471.356982639204944,
171481.226551398547599,
171491.096171881566988,
171500.965844085614663,
171510.835568008013070,
171520.705343646084657,
171530.575170997122768,
171540.445050058420748,
171550.314980827475665,
171560.184963301435346,
171570.054997477680445,
171579.925083353620721,
171589.795220926462207,
171599.665410193585558,
171609.535651152371429,
171619.405943800025852,
171629.276288133987691,
171639.146684151521185,
171649.017131849977886,
171658.887631226680242,
171668.758182278979803,
171678.628785004140809,
171688.499439399631228,
171698.370145462627988,
171708.240903190511744,
171718.111712580692256,
171727.982573630375555,
171737.853486336971400,
171747.724450697831344,
171757.595466710219625,
171767.466534371487796,
171777.337653678987408,
171787.208824630040908,
171797.080047221999848,
171806.951321452186676,
171816.822647317894734,
171826.694024816533783,
171836.565453945368063,
171846.436934701749124,
171856.308467083057621,
171866.180051086557796,
171876.051686709659407,
171885.923373949655797,
171895.795112803869415,
171905.666903269680915,
171915.538745344383642,
171925.410639025300043,
171935.282584309839876,
171945.154581195296487,
171955.026629678992322,
171964.898729758278932,
171974.770881430478767,
171984.643084693030687,
171994.515339543111622,
172004.387645978160435,
172014.260003995499574,
172024.132413592480589,
172034.004874766396824,
172043.877387514658039,
172053.749951834499370,
172063.622567723417887,
172073.495235178590519,
172083.367954197427025,
172093.240724777308060,
172103.113546915526967,
172112.986420609464403,
172122.859345856413711,
172132.732322653755546,
172142.605350998812355,
172152.478430888906587,
172162.351562321448000,
172172.224745293700835,
172182.097979803045746,
172191.971265846863389,
172201.844603422447108,
172211.717992527177557,
172221.591433158348082,
172231.464925313339336,
172241.338468989531975,
172251.212064184161136,
172261.085710894723888,
172270.959409118455369,
172280.833158852678025,
172290.706960094859824,
172300.580812842265004,
172310.454717092274223,
172320.328672842180822,
172330.202680089394562,
172340.076738831266994,
172349.950849065091461,
172359.825010788248619,
172369.699223998060916,
172379.573488691909006,
172389.447804867173545,
172399.322172521147877,
172409.196591651183553,
172419.071062254661229,
172428.945584328874247,
172438.820157871261472,
172448.694782879087143,
172458.569459349790122,
172468.444187280634651,
172478.318966669059591,
172488.193797512329184,
172498.068679807853187,
172507.943613552983152,
172517.818598745041527,
172527.693635381408967,
172537.568723459437024,
172547.443862976477249,
172557.319053929822985,
172567.194296316913096,
172577.069590135099133,
172586.944935381674441,
172596.820332054048777,
172606.695780149515485,
172616.571279665542534,
172626.446830599394161,
172636.322432948421920,
172646.198086710064672,
172656.073791881557554,
172665.949548460368533,
172675.825356443790952,
172685.701215829234570,
172695.577126613963628,
172705.453088795416988,
172715.329102370975306,
172725.205167337931925,
172735.081283693696605,
172744.957451435562689,
172754.833670560939936,
172764.709941067179898,
172774.586262951663230,
172784.462636211683275,
172794.339060844678897,
172804.215536848001648,
172814.092064218915766,
172823.968642954918323,
172833.845273053273559,
172843.721954511391232,
172853.598687326593790,
172863.475471496320097,
172873.352307017834391,
172883.229193888575537,
172893.106132105895085,
172902.983121667115483,
172912.860162569646491,
172922.737254810810555,
172932.614398388046538,
172942.491593298589578,
172952.368839539936744,
172962.246137109381380,
172972.123486004304141,
172982.000886222085683,
172991.878337760048453,
173001.755840615631314,
173011.633394786127610,
173021.511000268947100,
173031.388657061470440,
173041.266365161020076,
173051.144124564947560,
173061.021935270720860,
173070.899797275604215,
173080.777710577065591,
173090.655675172340125,
173100.533691058924887,
173110.411758234084118,
173120.289876695256680,
173130.168046439823229,
173140.046267465106212,
173149.924539768515388,
173159.802863347344100,
173169.681238199118525,
173179.559664321044693,
173189.438141710561467,
173199.316670365078608,
173209.195250281918561,
173219.073881458432879,
173228.952563892060425,
173238.831297580094542,
173248.710082520032302,
173258.588918709108839,
173268.467806144763017,
173278.346744824404595,
173288.225734745326918,
173298.104775904939743,
173307.983868300681934,
173317.863011929817731,
173327.742206789785996,
173337.621452877967386,
173347.500750191684347,
173357.380098728346638,
173367.259498485334916,
173377.138949460088043,
173387.018451649841154,
173396.898005052120425,
173406.777609664160991,
173416.657265483430820,
173426.536972507310566,
173436.416730733151780,
173446.296540158306016,
173456.176400780241238,
173466.056312596221687,
173475.936275603744434,
173485.816289800073719,
173495.696355182677507,
173505.576471748878248,
173515.456639496114803,
173525.336858421709621,
173535.217128523072461,
173545.097449797554873,
173554.977822242624825,
173564.858245855546556,
173574.738720633788034,
173584.619246574671706,
173594.499823675607331,
173604.380451934004668,
173614.261131347215269,
173624.141861912619788,
173634.022643627657089,
173643.903476489591412,
173653.784360495919827,
173663.665295643993886,
173673.546281931194244,
173683.427319354901556,
173693.308407912496477,
173703.189547601359664,
173713.070738418871770,
173722.951980362471659,
173732.833273429481778,
173742.714617617370095,
173752.596012923429953,
173762.477459345100215,
173772.358956879703328,
173782.240505524765467,
173792.122105277492665,
173802.003756135440199,
173811.885458095872309,
173821.767211156315170,
173831.649015314003918,
173841.530870566406520,
173851.412776910903631,
173861.294734344934113,
173871.176742865762208,
173881.058802470884984,
173890.940913157653995,
173900.823074923478998,
173910.705287765740650,
173920.587551681790501,
173930.469866669125622,
173940.352232725039357,
173950.234649846970569,
173960.117118032270810,
173969.999637278378941,
173979.882207582646515,
173989.764828942512395,
173999.647501355299028,
174009.530224818532588,
174019.412999329448212,
174029.295824885542970,
174039.178701484168414,
174049.061629122734303,
174058.944607798650395,
174068.827637509268243,
174078.710718252026709,
174088.593850024306448,
174098.477032823517220,
174108.360266646981472,
174118.243551492254483,
174128.126887356542284,
174138.010274237371050,
174147.893712132121436,
174157.777201038144995,
174167.660740952880587,
174177.544331873679766,
174187.427973798039602,
174197.311666723195231,
174207.195410646701930,
174217.079205565882148,
174226.963051478203852,
174236.846948380960384,
174246.730896271619713,
174256.614895147591596,
174266.498945006227586,
174276.383045844966546,
174286.267197661189130,
174296.151400452305097,
174306.035654215724207,
174315.919958948856220,
174325.804314649081789,
174335.688721313781571,
174345.573178940452635,
174355.457687526330119,
174365.342247068998404,
174375.226857565779937,
174385.111519014084479,
174394.996231411263580,
174404.880994754814310,
174414.765809042088222,
174424.650674270465970,
174434.535590437444625,
174444.420557540317532,
174454.305575576581759,
174464.190644543617964,
174474.075764438806800,
174483.960935259558028,
174493.846157003281405,
174503.731429667386692,
174513.616753249254543,
174523.502127746382030,
174533.387553156149806,
174543.273029475851217,
174553.158556703012437,
174563.044134835014120,
174572.929763869236922,
174582.815443803119706,
174592.701174634072231,
174602.586956359475153,
174612.472788976767333,
174622.358672483358532,
174632.244606876629405,
174642.130592154077021,
174652.016628312994726,
174661.902715350821381,
174671.788853265054058,
174681.675042053015204,
174691.561281712114578,
174701.447572239791043,
174711.333913633541670,
174721.220305890659802,
174731.106749008555198,
174740.993242984724930,
174750.879787816520547,
174760.766383501410019,
174770.653030036744894,
174780.539727419964038,
174790.426475648506312,
174800.313274719723267,
174810.200124631141080,
174820.087025380082196,
174829.973976963956375,
174839.860979380231583,
174849.748032626288477,
174859.635136699595023,
174869.522291597502772,
174879.409497317450587,
174889.296753856906435,
174899.184061213192763,
174909.071419383806642,
174918.958828366128728,
174928.846288157597883,
174938.733798755594762,
174948.621360157616436,
174958.508972360985354,
174968.396635363169480,
174978.284349161578575,
174988.172113753651502,
174998.059929136798019,
175007.947795308427885,
175017.835712265979964,
175027.723680006834911,
175037.611698528489796,
175047.499767828296172,
175057.387887903692899,
175067.276058752118843,
175077.164280371012865,
175087.052552757697413,
175096.940875909727765,
175106.829249824455474,
175116.717674499319401,
175126.606149931729306,
175136.494676119094947,
175146.383253058913397,
175156.271880748594413,
175166.160559185431339,
175176.049288367008558,
175185.938068290677620,
175195.826898953848286,
175205.715780353988521,
175215.604712488508085,
175225.493695354874944,
175235.382728950440651,
175245.271813272644067,
175255.160948318953160,
175265.050134086777689,
175274.939370573556516,
175284.828657776699401,
175294.717995693587000,
175304.607384321774589,
175314.496823658613721,
175324.386313701455947,
175334.275854447885649,
175344.165445895254379,
175354.055088040942792,
175363.944780882447958,
175373.834524417179637,
175383.724318642576691,
175393.614163556107087,
175403.504059155122377,
175413.394005437090527,
175423.284002399450401,
175433.174050039582653,
175443.064148355042562,
175452.954297343152575,
175462.844497001322452,
175472.734747327107470,
175482.625048317830078,
175492.515399970958242,
175502.405802283959929,
175512.296255254244898,
175522.186758879193803,
175532.077313156303717,
175541.967918083042605,
175551.858573656732915,
175561.749279874929925,
175571.640036734956084,
175581.530844234337565,
175591.421702370513231,
175601.312611140805529,
175611.203570542769739,
175621.094580573815620,
175630.985641231294721,
175640.876752512791427,
175650.767914415628184,
175660.659126937302062,
175670.550390075222822,
175680.441703826829325,
175690.333068189560436,
175700.224483160825912,
175710.115948738151928,
175720.007464918919140,
175729.899031700566411,
175739.790649080503499,
175749.682317056256579,
175759.574035625206307,
175769.465804784791544,
175779.357624532422051,
175789.249494865653105,
175799.141415781807154,
175809.033387278352166,
175818.925409352785209,
175828.817482002516044,
175838.709605225012638,
175848.601779017626541,
175858.494003377883928,
175868.386278303194558,
175878.278603791026399,
175888.170979838818312,
175898.063406444009161,
175907.955883604008704,
175917.848411316314014,
175927.740989578334847,
175937.633618387510069,
175947.526297741336748,
175957.419027637253748,
175967.311808072612621,
175977.204639044968644,
175987.097520551731577,
175996.990452590311179,
176006.883435158204520,
176016.776468252850464,
176026.669551871687872,
176036.562686012097402,
176046.455870671663433,
176056.349105847737519,
176066.242391537787626,
176076.135727739281720,
176086.029114449629560,
176095.922551666299114,
176105.816039386758348,
176115.709577608446125,
176125.603166328772204,
176135.496805545291863,
176145.390495255356655,
176155.284235456434544,
176165.178026145964395,
176175.071867321472382,
176184.965758980310056,
176194.859701120003592,
176204.753693737991853,
176214.647736831713701,
176224.541830398578895,
176234.435974436113611,
176244.330168941756710,
176254.224413912917953,
176264.118709347065305,
176274.013055241695838,
176283.907451594219310,
176293.801898402103689,
176303.696395662816940,
176313.590943373797927,
176323.485541532456409,
176333.380190136347665,
176343.274889182852348,
176353.169638669438427,
176363.064438593573868,
176372.959288952726638,
176382.854189744306495,
176392.749140965810511,
176402.644142614706652,
176412.539194688404677,
176422.434297184430761,
176432.329450100165559,
176442.224653433106141,
176452.119907180749578,
176462.015211340447422,
176471.910565909754951,
176481.805970886111027,
176491.701426266954513,
176501.596932049753377,
176511.492488231917378,
176521.388094811030896,
176531.283751784474589,
176541.179459149658214,
176551.075216904137051,
176560.971025045320857,
176570.866883570706705,
176580.762792477733456,
176590.658751763839973,
176600.554761426523328,
176610.450821463222383,
176620.346931871405104,
176630.243092648597667,
176640.139303792122519,
176650.035565299593145,
176659.931877168331994,
176669.828239395952551,
176679.724651979835471,
176689.621114917448722,
176699.517628206231166,
176709.414191843679873,
176719.310805827291915,
176729.207470154506154,
176739.104184822703246,
176749.000949829496676,
176758.897765172267100,
176768.794630848482484,
176778.691546855639899,
176788.588513191178208,
176798.485529852623586,
176808.382596837327583,
176818.279714142845478,
176828.176881766645238,
176838.074099706194829,
176847.971367958874907,
176857.868686522269854,
176867.766055393818533,
176877.663474570959806,
176887.560944051161641,
176897.458463831921108,
176907.356033910677070,
176917.253654284926597,
176927.151324952166760,
176937.049045909807319,
176946.946817155345343,
176956.844638686277904,
176966.742510500014760,
176976.640432594052982,
176986.538404965889640,
176996.436427613021806,
177006.334500532888342,
177016.232623722869903,
177026.130797180609079,
177036.029020903486526,
177045.927294888941105,
177055.825619134528097,
177065.723993637686362,
177075.622418395883869,
177085.520893406588584,
177095.419418667268474,
177105.317994175449712,
177115.216619928571163,
177125.115295924100792,
177135.014022159564774,
177144.912798632343765,
177154.811625339993043,
177164.710502279980574,
177174.609429449803429,
177184.508406846842263,
177194.407434468623251,
177204.306512312701670,
177214.205640376458177,
177224.104818657418946,
177234.004047152993735,
177243.903325860766927,
177253.802654778148280,
177263.702033902634867,
177273.601463231723756,
177283.500942762882914,
177293.400472493522102,
177303.300052421254804,
177313.199682543490781,
177323.099362857639790,
177332.999093361315317,
177342.898874051927123,
177352.798704926943174,
177362.698585983918747,
177372.598517220205395,
177382.498498633445706,
177392.398530221020337,
177402.298611980426358,
177412.198743909131736,
177422.098926004633540,
177431.999158264457947,
177441.899440686043818,
177451.799773266917327,
177461.700156004488235,
177471.600588896253612,
177481.501071939797839,
177491.401605132501572,
177501.302188471890986,
177511.202821955463151,
177521.103505580627825,
177531.004239344969392,
177540.905023245955817,
177550.805857281055069,
177560.706741447676904,
177570.607675743434811,
177580.508660165767651,
177590.409694712143391,
177600.310779380059103,
177610.211914167040959,
177620.113099070469616,
177630.014334087958559,
177639.915619216917548,
177649.816954454901861,
177659.718339799350360,
177669.619775247760117,
177679.521260797628202,
177689.422796446451684,
177699.324382191698533,
177709.226018030865816,
177719.127703961479710,
177729.029439980920870,
177738.931226086831884,
177748.833062276622513,
177758.734948547789827,
177768.636884897860000,
177778.538871324242791,
177788.440907824493479,
177798.342994396138238,
177808.245131036615930,
177818.147317743423628,
177828.049554514058400,
177837.951841345988214,
177847.854178236768348,
177857.756565183895873,
177867.659002184780547,
177877.561489237006754,
177887.464026337984251,
177897.366613485268317,
177907.269250676356023,
177917.171937908715336,
177927.074675179872429,
177936.977462487266166,
177946.880299828480929,
177956.783187200926477,
177966.686124602187192,
177976.589112029672833,
177986.492149480909575,
177996.395236953452695,
178006.298374444711953,
178016.201561952242628,
178026.104799473512685,
178036.008087006048299,
178045.911424547346542,
178055.814812094875379,
178065.718249646160984,
178075.621737198700430,
178085.525274749961682,
178095.428862297529122,
178105.332499838812510,
178115.236187371338019,
178125.139924892631825,
178135.043712400190998,
178144.947549891483504,
178154.851437364035519,
178164.755374815344112,
178174.659362242935458,
178184.563399644277524,
178194.467487016896484,
178204.371624358260306,
178214.275811665895162,
178224.180048937356332,
178234.084336170053575,
178243.988673361542169,
178253.893060509319184,
178263.797497610881692,
178273.701984663726762,
178283.606521665409673,
178293.511108613427496,
178303.415745505189989,
178313.320432338339742,
178323.225169110257411,
178333.129955818556482,
178343.034792460588505,
178352.939679034054279,
178362.844615536363563,
178372.749601965042530,
178382.654638317559147,
178392.559724591468694,
178402.464860784268240,
178412.370046893396648,
178422.275282916467404,
178432.180568850948475,
178442.085904694336932,
178451.991290444188053,
178461.896726097940700,
178471.802211653121049,
178481.707747107284376,
178491.613332457869546,
178501.518967702490045,
178511.424652838555630,
178521.330387863650685,
178531.236172775243176,
178541.142007570859278,
178551.047892247966956,
178560.953826804179698,
178570.859811236878159,
178580.765845543675823,
178590.671929722069763,
178600.578063769527944,
178610.484247683634749,
178620.390481461829040,
178630.296765101695200,
178640.203098600672092,
178650.109481956285890,
178660.015915166120976,
178669.922398227645317,
178679.828931138385087,
178689.735513895808253,
178699.642146497499198,
178709.548828940925887,
178719.455561223614495,
178729.362343343149405,
178739.269175296882167,
178749.176057082484476,
178759.082988697424298,
178768.989970139227808,
178778.897001405392075,
178788.804082493443275,
178798.711213400878478,
178808.618394125282066,
178818.525624664092902,
178828.432905014895368,
178838.340235175099224,
178848.247615142347058,
178858.155044914106838,
178868.062524487933842,
178877.970053861296037,
178887.877633031748701,
178897.785261996759800,
178907.692940753942821,
178917.600669300736627,
178927.508447634667391,
178937.416275753319496,
178947.324153654131806,
178957.232081334717805,
178967.140058792516356,
178977.048086025111843,
178986.956163029943127,
178996.864289804652799,
179006.772466346679721,
179016.680692653579172,
179026.588968722819118,
179036.497294551954838,
179046.405670138570713,
179056.314095480163814,
179066.222570574202109,
179076.131095418269979,
179086.039670009835390,
179095.948294346482726,
179105.856968425738160,
179115.765692245040555,
179125.674465802032501,
179135.583289094181964,
179145.492162118956912,
179155.401084873970831,
179165.310057356749894,
179175.219079564791173,
179185.128151495620841,
179195.037273146765074,
179204.946444515779149,
179214.855665600160137,
179224.764936397434212,
179234.674256905156653,
179244.583627120882738,
179254.493047042080434,
179264.402516666305019,
179274.312035991024459,
179284.221605013881344,
179294.131223732343642,
179304.040892143966630,
179313.950610246247379,
179323.860378036741167,
179333.770195513003273,
179343.680062672472559,
179353.589979512791615,
179363.499946031428408,
179373.409962225909112,
179383.320028093789006,
179393.230143632652471,
179403.140308839938371,
179413.050523713231087,
179422.960788250027690,
179432.871102447941666,
179442.781466304382775,
179452.691879816964502,
179462.602342983242124,
179472.512855800712714,
179482.423418266931549,
179492.334030379395699,
179502.244692135718651,
179512.155403533310164,
179522.066164569841931,
179531.976975242752815,
179541.887835549627198,
179551.798745487991255,
179561.709705055371160,
179571.620714249293087,
179581.531773067312315,
179591.442881506984122,
179601.354039565805579,
179611.265247241390171,
179621.176504531147657,
179631.087811432749731,
179640.999167943664361,
179650.910574061446823,
179660.822029783623293,
179670.733535107749049,
179680.645090031379368,
179690.556694552011322,
179700.468348667229293,
179710.380052374501247,
179720.291805671498878,
179730.203608555602841,
179740.115461024484830,
179750.027363075612811,
179759.939314706600271,
179769.851315914886072,
179779.763366698083701,
179789.675467053719331,
179799.587616979348240,
179809.499816472467501,
179819.412065530661494,
179829.324364151485497,
179839.236712332465686,
179849.149110071128234,
179859.061557365028420,
179868.974054211750627,
179878.886600608704612,
179888.799196553649381,
179898.711842043965589,
179908.624537077208515,
179918.537281651049852,
179928.450075762899360,
179938.362919410370523,
179948.275812590960413,
179958.188755302253412,
179968.101747541804798,
179978.014789307169849,
179987.927880595845636,
179997.841021405416541,
180007.754211733408738,
180017.667451577377506,
180027.580740934878122,
180037.494079803494969,
180047.407468180696014,
180057.320906064094743,
180067.234393451217329,
180077.147930339560844,
180087.061516726796981,
180096.975152610364603,
180106.888837987906300,
180116.802572856890038,
180126.716357214900199,
180136.630191059492063,
180146.544074388220906,
180156.458007198612904,
180166.371989488252439,
180176.286021254665684,
180186.200102495407918,
180196.114233208034420,
180206.028413390100468,
180215.942643039190443,
180225.856922152772313,
180235.771250728517771,
180245.685628763894783,
180255.600056256458629,
180265.514533203764586,
180275.429059603455244,
180285.343635452969465,
180295.258260749891633,
180305.172935491835233,
180315.087659676297335,
180325.002433300862322,
180334.917256363056367,
180344.832128860434750,
180354.747050790640060,
180364.662022151111159,
180374.577042939461535,
180384.492113153217360,
180394.407232789963018,
180404.322401847282890,
180414.237620322703151,
180424.152888213749975,
180434.068205518065952,
180443.983572233119048,
180453.898988356522750,
180463.814453885832336,
180473.729968818573980,
180483.645533152332064,
180493.561146884690970,
180503.476810013147769,
180513.392522535315948,
180523.308284448692575,
180533.224095750920242,
180543.139956439554226,
180553.055866512062494,
180562.971825966087636,
180572.887834799184930,
180582.803893008909654,
180592.720000592758879,
180602.636157548404299,
180612.552363873342983,
180622.468619565159315,
180632.384924621408572,
180642.301279039675137,
180652.217682817456080,
180662.134135952393990,
180672.050638442015043,
180681.967190283874515,
180691.883791475556791,
180701.800442014617147,
180711.717141898610862,
180721.633891125151422,
180731.550689691765001,
180741.467537596006878,
180751.384434835461434,
180761.301381407654844,
180771.218377310229698,
180781.135422540683066,
180791.052517096657539,
180800.969660975621082,
180810.886854175216286,
180820.804096692969324,
180830.721388526464580,
180840.638729673315538,
180850.556120131019270,
180860.473559897160158,
180870.391048969351687,
180880.308587345090928,
180890.226175022020470,
180900.143811997637386,
180910.061498269584263,
180919.979233835387276,
180929.897018692601705,
180939.814852838841034,
180949.732736271660542,
180959.650668988615507,
180969.568650987319415,
180979.486682265327545,
180989.404762820166070,
180999.322892649419373,
181009.241071750730043,
181019.159300121595152,
181029.077577759569976,
181038.995904662297107,
181048.914280827331822,
181058.832706252229400,
181068.751180934603326,
181078.669704871921567,
181088.588278061884921,
181098.506900502019562,
181108.425572189880768,
181118.344293123052921,
181128.263063299120404,
181138.181882715638494,
181148.100751370220678,
181158.019669260393130,
181167.938636383798439,
181177.857652737962781,
181187.776718320499640,
181197.695833128906088,
181207.614997160824714,
181217.534210413810797,
181227.453472885448718,
181237.372784573322861,
181247.292145475017605,
181257.211555588088231,
181267.131014910148224,
181277.050523438752862,
181286.970081171457423,
181296.889688105875393,
181306.809344239562051,
181316.729049570159987,
181326.648804095166270,
181336.568607812165283,
181346.488460718799615,
181356.408362812595442,
181366.328314091137145,
181376.248314552067313,
181386.168364192883018,
181396.088463011255953,
181406.008611004683189,
181415.928808170749107,
181425.849054507096298,
181435.769350011280039,
181445.689694680913817,
181455.610088513523806,
181465.530531506723491,
181475.451023658068152,
181485.371564965142170,
181495.292155425588135,
181505.212795036932221,
181515.133483796787914,
181525.054221702710493,
181534.975008752342546,
181544.895844943210250,
181554.816730272927089,
181564.737664739106549,
181574.658648339245701,
181584.579681070987135,
181594.500762931915233,
181604.421893919643480,
181614.343074031698052,
181624.264303265692433,
181634.185581619240111,
181644.106909089925466,
181654.028285675274674,
181663.949711372930324,
181673.871186180535005,
181683.792710095527582,
181693.714283115608850,
181703.635905238334090,
181713.557576461316785,
181723.479296782083111,
181733.401066198333865,
181743.322884707537014,
181753.244752307364251,
181763.166668995341752,
181773.088634769170312,
181783.010649626317900,
181792.932713564398000,
181802.854826581082307,
181812.776988673867891,
181822.699199840397341,
181832.621460078255041,
181842.543769384996267,
181852.466127758292714,
181862.388535195699660,
181872.310991694801487,
181882.233497253182577,
181892.156051868427312,
181902.078655538149178,
181912.001308259932557,
181921.924010031390935,
181931.846760850137798,
181941.769560713670217,
181951.692409619688988,
181961.615307565749390,
181971.538254549435806,
181981.461250568390824,
181991.384295620111516,
182001.307389702298678,
182011.230532812478486,
182021.153724948293529,
182031.076966107299086,
182041.000256287166849,
182050.923595485422993,
182060.846983699651901,
182070.770420927496161,
182080.693907166540157,
182090.617442414339166,
182100.541026668564882,
182110.464659926830791,
182120.388342186663067,
182130.312073445675196,
182140.235853701480664,
182150.159682951663854,
182160.083561193867354,
182170.007488425646443,
182179.931464644556399,
182189.855489848327124,
182199.779564034513896,
182209.703687200642889,
182219.627859344356693,
182229.552080463268794,
182239.476350554963574,
182249.400669617083622,
182259.325037647213321,
182269.249454642907949,
182279.173920601868303,
182289.098435521620559,
182299.022999399749096,
182308.947612233896507,
182318.872274021647172,
182328.796984760614578,
182338.721744448412210,
182348.646553082682658,
182358.571410660952097,
182368.496317180863116,
182378.421272640029201,
182388.346277036034735,
182398.271330366464099,
182408.196432628988987,
182418.121583821164677,
182428.046783940575551,
182437.972032984922407,
182447.897330951702315,
182457.822677838616073,
182467.748073643189855,
182477.673518363066250,
182487.599011995887849,
182497.524554539180826,
182507.450145990645979,
182517.375786347780377,
182527.301475608313922,
182537.227213769801892,
182547.153000829857774,
182557.078836786036845,
182567.004721636010800,
182576.930655377393123,
182586.856638007739093,
182596.782669524691300,
182606.708749925892334,
182616.634879208926577,
182626.561057371378411,
182636.487284410890425,
182646.413560325017897,
182656.339885111461626,
182666.266258767805994,
182676.192681291606277,
182686.119152680534171,
182696.045672932174057,
182705.972242044197628,
182715.898860014072852,
182725.825526839558734,
182735.752242518239655,
182745.679007047699997,
182755.605820425524144,
182765.532682649383787,
182775.459593716863310,
182785.386553625576198,
182795.313562373165041,
182805.240619957214221,
182815.167726375337224,
182825.094881625176640,
182835.022085704287747,
182844.949338610400446,
182854.876640341040911,
182864.803990893851733,
182874.731390266417293,
182884.658838456409285,
182894.586335461382987,
182904.513881279068301,
182914.441475906933192,
182924.369119342649356,
182934.296811583888484,
182944.224552628234960,
182954.152342473273166,
182964.080181116703898,
182974.008068556053331,
182983.936004788964055,
182993.863989813107764,
183003.792023626068840,
183013.720106225460768,
183023.648237608926138,
183033.576417774049332,
183043.504646718472941,
183053.432924439810449,
183063.361250935704447,
183073.289626203739317,
183083.218050241557648,
183093.146523046802031,
183103.075044617056847,
183113.003614949935582,
183122.932234043139033,
183132.860901894222479,
183142.789618500770302,
183152.718383860483300,
183162.647197970974958,
183172.576060829858761,
183182.504972434719093,
183192.433932783227647,
183202.362941872968804,
183212.291999701614259,
183222.221106266806601,
183232.150261566101108,
183242.079465597111266,
183252.008718357537873,
183261.938019844965311,
183271.867370057036169,
183281.796768991363933,
183291.726216645562090,
183301.655713017273229,
183311.585258104110835,
183321.514851903746603,
183331.444494413794018,
183341.374185631779255,
183351.303925555490423,
183361.233714182424592,
183371.163551510282559,
183381.093437536706915,
183391.023372259223834,
183400.953355675563216,
183410.883387783367652,
183420.813468580134213,
183430.743598063621903,
183440.673776231415104,
183450.604003081156407,
183460.534278610459296,
183470.464602816937258,
183480.394975698261987,
183490.325397252017865,
183500.255867475876585,
183510.186386367480736,
183520.116953924443806,
183530.047570144321071,
183539.978235024871537,
183549.908948563708691,
183559.839710758329602,
183569.770521606522379,
183579.701381105842302,
183589.632289253961062,
183599.563246048521250,
183609.494251487049041,
183619.425305567332543,
183629.356408286868827,
183639.287559643416898,
183649.218759634532034,
183659.150008257856825,
183669.081305511004757,
183679.012651391647523,
183688.944045897456817,
183698.875489026017021,
183708.806980774941621,
183718.738521141931415,
183728.670110124570783,
183738.601747720531421,
183748.533433927426813,
183758.465168742870446,
183768.396952164621325,
183778.328784190147417,
183788.260664817178622,
183798.192594043328427,
183808.124571866297629,
183818.056598283612402,
183827.988673293002648,
183837.920796892081853,
183847.852969078492606,
183857.785189849848393,
183867.717459203791805,
183877.649777137994533,
183887.582143650099169,
183897.514558737719199,
183907.447022398468107,
183917.379534630046692,
183927.312095430039335,
183937.244704796205042,
183947.177362726011779,
183957.110069217218552,
183967.042824267409742,
183976.975627874315251,
183986.908480035461253,
183996.841380748577649,
184006.774330011219718,
184016.707327821204672,
184026.640374175942270,
184036.573469073220622,
184046.506612510653213,
184056.439804485940840,
184066.373044996609678,
184076.306334040389629,
184086.239671614865074,
184096.173057717795018,
184106.106492346676532,
184116.039975499210414,
184125.973507173155667,
184135.907087365980260,
184145.840716075414093,
184155.774393299099756,
184165.708119034738047,
184175.641893279855140,
184185.575716032180935,
184195.509587289358024,
184205.443507049028995,
184215.377475308807334,
184225.311492066364735,
184235.245557319372892,
184245.179671065474395,
184255.113833302253624,
184265.048044027440483,
184274.982303238648456,
184284.916610933520133,
184294.850967109727208,
184304.785371764883166,
184314.719824896659702,
184324.654326502757613,
184334.588876580732176,
184344.523475128313294,
184354.458122143085347,
184364.392817622720031,
184374.327561564889038,
184384.262353967293166,
184394.197194827487692,
184404.132084143144311,
184414.067021911963820,
184424.002008131559705,
184433.937042799603660,
184443.872125913709169,
184453.807257471577032,
184463.742437470878940,
184473.677665909170173,
184483.612942784209736,
184493.548268093611114,
184503.483641834987793,
184513.419064006040571,
184523.354534604441142,
184533.290053627832094,
184543.225621073826915,
184553.161236940068193,
184563.096901224314934,
184573.032613924151519,
184582.968375037278747,
184592.904184561251896,
184602.840042493829969,
184612.775948832626455,
184622.711903575283941,
184632.647906719503226,
184642.583958262897795,
184652.520058203168446,
184662.456206537957769,
184672.392403264879249,
184682.328648381662788,
184692.264941885892767,
184702.201283775357297,
184712.137674047553446,
184722.074112700211117,
184732.010599731031107,
184741.947135137597797,
184751.883718917611986,
184761.820351068745367,
184771.757031588640530,
184781.693760474940063,
184791.630537725344766,
184801.567363337526331,
184811.504237309069140,
184821.441159637673991,
184831.378130321070785,
184841.315149356785696,
184851.252216742577730,
184861.189332476118580,
184871.126496555021731,
184881.063708976929775,
184891.000969739572611,
184900.938278840592830,
184910.875636277633021,
184920.813042048364878,
184930.750496150489198,
184940.687998581590364,
184950.625549339398276,
184960.563148421584629,
184970.500795825733803,
184980.438491549604805,
184990.376235590869328,
185000.314027947082650,
185010.251868615945568,
185020.189757595246192,
185030.127694882481592,
185040.065680475439876,
185050.003714371734532,
185059.941796569008147,
185069.879927064990625,
185079.818105857295450,
185089.756332943681628,
185099.694608321704436,
185109.632931989064673,
185119.571303943463136,
185129.509724182513310,
185139.448192703945097,
185149.386709505430190,
185159.325274584582075,
185169.263887939072447,
185179.202549566631205,
185189.141259464871837,
185199.080017631466035,
185209.018824064143701,
185218.957678760518320,
185228.896581718261587,
185238.835532935074298,
185248.774532408599043,
185258.713580136536621,
185268.652676116558723,
185278.591820346307941,
185288.531012823455967,
185298.470253545703599,
185308.409542510722531,
185318.348879716126248,
185328.288265159644652,
185338.227698838978540,
185348.167180751741398,
185358.106710895604920,
185368.046289268299006,
185377.985915867437143,
185387.925590690749232,
185397.865313735877862,
185407.805085000465624,
185417.744904482271522,
185427.684772178879939,
185437.624688088020775,
185447.564652207394829,
185457.504664534615586,
185467.444725067383843,
185477.384833803371293,
185487.324990740278736,
185497.265195875748759,
185507.205449207453057,
185517.145750733092427,
185527.086100450396771,
185537.026498356921365,
185546.966944450454321,
185556.907438728638226,
185566.847981189086568,
185576.788571829587454,
185586.729210647783475,
185596.669897641288117,
185606.610632807831280,
185616.551416145142866,
185626.492247650807258,
185636.433127322583459,
185646.374055158084957,
185656.315031155012548,
185666.256055311096134,
185676.197127623949200,
185686.138248091272544,
185696.079416710825171,
185706.020633480162360,
185715.961898397014011,
185725.903211459110025,
185735.844572664063890,
185745.785982009576401,
185755.727439493406564,
185765.668945113080554,
185775.610498866444686,
185785.552100751083344,
185795.493750764726428,
185805.435448905045632,
185815.377195169683546,
185825.318989556340966,
185835.260832062776899,
185845.202722686575726,
185855.144661425496452,
185865.086648277210770,
185875.028683239332167,
185884.970766309648752,
185894.912897485744907,
185904.855076765379636,
185914.797304146282841,
185924.739579626009800,
185934.681903202348622,
185944.624274872883689,
185954.566694635432214,
185964.509162487636786,
185974.451678427139996,
185984.394242451642640,
185994.336854558903724,
186004.279514746478526,
186014.222223012184259,
186024.164979353634408,
186034.107783768529771,
186044.050636254600249,
186053.993536809488432,
186063.936485430924222,
186073.879482116550207,
186083.822526864038082,
186093.765619671205059,
186103.708760535606416,
186113.651949454972055,
186123.595186427060980,
186133.538471449457575,
186143.481804519920843,
186153.425185636122478,
186163.368614795763278,
186173.312091996514937,
186183.255617236078251,
186193.199190512154019,
186203.142811822413933,
186213.086481164587894,
186223.030198536347598,
186232.973963935364736,
186242.917777359398315,
186252.861638806061819,
186262.805548273085151,
186272.749505758169107,
186282.693511259014485,
186292.637564773322083,
186302.581666298734490,
186312.525815832981607,
186322.470013373764232,
186332.414258918783162,
186342.358552465710090,
186352.302894012245815,
186362.247283556120237,
186372.191721094975946,
186382.136206626542844,
186392.080740148521727,
186402.025321658584289,
186411.969951154431328,
186421.914628633821849,
186431.859354094340233,
186441.804127533832798,
186451.748948949825717,
186461.693818340136204,
186471.638735702465056,
186481.583701034425758,
186491.528714333835524,
186501.473775598220527,
186511.418884825485293,
186521.364042013185099,
186531.309247159078950,
186541.254500260838540,
186551.199801316193771,
186561.145150322845438,
186571.090547278436134,
186581.035992180724861,
186590.981485027412418,
186600.927025816170499,
186610.872614544758108,
186620.818251210759627,
186630.763935812021373,
186640.709668346127728,
186650.655448810866801,
186660.601277203910286,
186670.547153522929875,
186680.493077765655471,
186690.439049929816974,
186700.385070013027871,
186710.331138013105374,
186720.277253927692072,
186730.223417754546972,
186740.169629491283558,
186750.115889135689940,
186760.062196685379604,
186770.008552138111554,
186779.954955491644796,
186789.901406743592815,
186799.847905891772825,
186809.794452933740104,
186819.741047867311863,
186829.687690690159798,
186839.634381400013808,
186849.581119994545588,
186859.527906471455935,
186869.474740828532958,
186879.421623063361039,
186889.368553173757391,
186899.315531157335499,
186909.262557011883473,
186919.209630735102110,
186929.156752324663103,
186939.103921778296353,
186949.051139093673555,
186958.998404268582817,
186968.945717300637625,
186978.893078187626088,
186988.840486927219899,
186998.787943517148960,
187008.735447955084965,
187018.683000238757813,
187028.630600365926512,
187038.578248334233649,
187048.525944141467335,
187058.473687785241054,
187068.421479263342917,
187078.369318573415512,
187088.317205713305157,
187098.265140680567129,
187108.213123472960433,
187118.161154088273179,
187128.109232524118852,
187138.057358778285561,
187148.005532848415896,
187157.953754732297966,
187167.902024427603465,
187177.850341932091396,
187187.798707243404351,
187197.747120359301334,
187207.695581277454039,
187217.644089995621471,
187227.592646511533530,
187237.541250822891016,
187247.489902927365620,
187257.438602822687244,
187267.387350506673101,
187277.336145976907574,
187287.284989231149666,
187297.233880267158384,
187307.182819082605420,
187317.131805675191572,
187327.080840042704949,
187337.029922182788141,
187346.979052093229257,
187356.928229771641782,
187366.877455215842929,
187376.826728423562599,
187386.776049392443383,
187396.725418120244285,
187406.674834604636999,
187416.624298843409633,
187426.573810834262986,
187436.523370574926957,
187446.472978063044138,
187456.422633296460845,
187466.372336272790562,
187476.322086989792297,
187486.271885445195949,
187496.221731636702316,
187506.171625562012196,
187516.121567218971904,
187526.071556605136720,
187536.021593718323857,
187545.971678556263214,
187555.921811116597382,
187565.871991397114471,
187575.822219395544380,
187585.772495109587908,
187595.722818536945852,
187605.673189675406320,
187615.623608522582799,
187625.574075076350709,
187635.524589334294433,
187645.475151294231182,
187655.425760953832651,
187665.376418310857844,
187675.327123363007559,
187685.277876108011696,
187695.228676543571055,
187705.179524667473743,
187715.130420477391453,
187725.081363971112296,
187735.032355146307964,
187744.983394000708358,
187754.934480531985173,
187764.885614738042932,
187774.836796616407810,
187784.788026164955227,
187794.739303381298669,
187804.690628263284452,
187814.642000808496960,
187824.593421014840715,
187834.544888879870996,
187844.496404401434120,
187854.447967577143572,
187864.399578404903878,
187874.351236882270314,
187884.302943007118301,
187894.254696777032223,
187904.206498189887498,
187914.158347243268508,
187924.110243935021572,
187934.062188262818381,
187944.014180224417942,
187953.966219817521051,
187963.918307039915817,
187973.870441889244830,
187983.822624363354407,
187993.774854459828930,
188003.727132176543819,
188013.679457511199871,
188023.631830461439677,
188033.584251025080448,
188043.536719199852087,
188053.489234983455390,
188063.441798373649362,
188073.394409368105698,
188083.347067964670714,
188093.299774160986999,
188103.252527954871766,
188113.205329343938502,
188123.158178326004418,
188133.111074898770312,
188143.064019060053397,
188153.017010807496263,
188162.970050138857914,
188172.923137051868252,
188182.876271544315387,
188192.829453613871010,
188202.782683258294128,
188212.735960475343745,
188222.689285262720659,
188232.642657618212979,
188242.596077539463295,
188252.549545024317922,
188262.503060070448555,
188272.456622675614199,
188282.410232837573858,
188292.363890554028330,
188302.317595822707517,
188312.271348641399527,
188322.225149007805157,
188332.178996919654310,
188342.132892374735093,
188352.086835370777408,
188362.040825905452948,
188371.994863976637134,
188381.948949581885245,
188391.903082719130907,
188401.857263385929400,
188411.811491580214351,
188421.765767299599247,
188431.720090541813988,
188441.674461304675788,
188451.628879585885443,
188461.583345383172855,
188471.537858694326133,
188481.492419517016970,
188491.447027849062579,
188501.401683688192861,
188511.356387032079510,
188521.311137878597947,
188531.265936225332553,
188541.220782070158748,
188551.175675410777330,
188561.130616244889097,
188571.085604570311261,
188581.040640384686412,
188590.995723685889971,
188600.950854471535422,
188610.906032739498187,
188620.861258487449959,
188630.816531713120639,
188640.771852414298337,
188650.727220588712953,
188660.682636234123493,
188670.638099348230753,
188680.593609928851947,
188690.549167973658768,
188700.504773480468430,
188710.460426446981728,
188720.416126870986773,
188730.371874750184361,
188740.327670082304394,
188750.283512865193188,
188760.239403096522437,
188770.195340774080250,
188780.151325895538321,
188790.107358458772069,
188800.063438461394981,
188810.019565901282476,
188819.975740776077146,
188829.931963083596202,
188839.888232821598649,
188849.844549987756182,
188859.800914579915116,
188869.757326595747145,
188879.713786033040378,
188889.670292889553821,
188899.626847163046477,
188909.583448851190042,
188919.540097951859934,
188929.496794462756952,
188939.453538381581893,
188949.410329706151970,
188959.367168434167979,
188969.324054563418031,
188979.280988091632025,
188989.237969016627176,
188999.194997336133383,
189009.152073047822341,
189019.109196149540367,
189029.066366638959153,
189039.023584513954120,
189048.980849772138754,
189058.938162411359372,
189068.895522429404082,
189078.852929823886370,
189088.810384592739865,
189098.767886733548949,
189108.725436244189041,
189118.683033122360939,
189128.640677365852753,
189138.598368972423486,
189148.556107939773938,
189158.513894265721319,
189168.471727947995532,
189178.429608984384686,
189188.387537372560473,
189198.345513110369211,
189208.303536195569905,
189218.261606625892455,
189228.219724399095867,
189238.177889512880938,
189248.136101965123089,
189258.094361753523117,
189268.052668875810923,
189278.011023329774616,
189287.969425113173202,
189297.927874223794788,
189307.886370659340173,
189317.844914417655673,
189327.803505496442085,
189337.762143893400207,
189347.720829606405459,
189357.679562633187743,
189367.638342971447855,
189377.597170619032113,
189387.556045573670417,
189397.514967833092669,
189407.473937395057874,
189417.432954257412348,
189427.392018417827785,
189437.351129874092294,
189447.310288624023087,
189457.269494665291859,
189467.228747995744925,
189477.188048613112187,
189487.147396515123546,
189497.106791699625319,
189507.066234164260095,
189517.025723906961503,
189526.985260925343027,
189536.944845217221882,
189546.904476780386176,
189556.864155612536706,
189566.823881711548893,
189576.783655075094430,
189586.743475700961426,
189596.703343586967094,
189606.663258730754023,
189616.623221130226739,
189626.583230783086037,
189636.543287687120028,
189646.503391840087716,
189656.463543239719002,
189666.423741883860203,
189676.383987770212116,
189686.344280896562850,
189696.304621260671411,
189706.265008860355010,
189716.225443693343550,
189726.185925757425139,
189736.146455050300574,
189746.107031569845276,
189756.067655313730938,
189766.028326279803878,
189775.989044465764891,
189785.949809869460296,
189795.910622488619993,
189805.871482321002986,
189815.832389364426490,
189825.793343616620405,
189835.754345075314632,
189845.715393738384591,
189855.676489603560185,
189865.637632668571314,
189875.598822931235190,
189885.560060389339924,
189895.521345040586311,
189905.482676882791566,
189915.444055913772900,
189925.405482131231111,
189935.366955532983411,
189945.328476116730599,
189955.290043880348094,
189965.251658821594901,
189975.213320938142715,
189985.175030227866955,
189995.136786688555731,
190005.098590317880735,
190015.060441113746492,
190025.022339073795592,
190034.984284195932560,
190044.946276477829088,
190054.908315917331493,
190064.870402512169676,
190074.832536260131747,
190084.794717159005813,
190094.756945206579985,
190104.719220400584163,
190114.681542738835560,
190124.643912219122285,
190134.606328839174239,
190144.568792596837739,
190154.531303489813581,
190164.493861515918979,
190174.456466672912939,
190184.419118958641775,
190194.381818370806286,
190204.344564907194581,
190214.307358565623872,
190224.270199343882268,
190234.233087239699671,
190244.196022250864189,
190254.159004375193035,
190264.122033610445214,
190274.085109954408836,
190284.048233404813800,
190294.011403959564632,
190303.974621616274817,
190313.937886372819776,
190323.901198227016721,
190333.864557176566450,
190343.827963219344383,
190353.791416353051318,
190363.754916575504467,
190373.718463884462835,
190383.682058277714532,
190393.645699753047666,
190403.609388308337657,
190413.573123941168888,
190423.536906649504090,
190433.500736431073165,
190443.464613283576909,
190453.428537204948952,
190463.392508192860987,
190473.356526245130226,
190483.320591359544778,
190493.284703533921856,
190503.248862765962258,
190513.213069053570507,
190523.177322394418297,
190533.141622786322841,
190543.105970227130456,
190553.070364714571042,
190563.034806246432709,
190572.999294820503565,
190582.963830434629926,
190592.928413086541696,
190602.893042774026981,
190612.857719494873891,
190622.822443246841431,
190632.787214027805021,
190642.752031835494563,
190652.716896667698165,
190662.681808522233041,
190672.646767396887299,
190682.611773289361736,
190692.576826197589980,
190702.541926119243726,
190712.507073052169289,
190722.472266994125675,
190732.437507942959201,
190742.402795896370662,
190752.368130852264585,
190762.333512808312662,
190772.298941762390314,
190782.264417712256545,
190792.229940655728569,
190802.195510590565391,
190812.161127514555119,
190822.126791425514966,
190832.092502321233042,
190842.058260199497454,
190852.024065058125416,
190861.989916894846829,
190871.955815707508009,
190881.921761493867962,
190891.887754251743900,
190901.853793978923932,
190911.819880673254374,
190921.786014332436025,
190931.752194954315200,
190941.718422536680009,
190951.684697077289457,
190961.651018573989859,
190971.617387024598429,
190981.583802426845068,
190991.550264778488781,
191001.516774077463197,
191011.483330321469111,
191021.449933508323738,
191031.416583635786083,
191041.383280701760668,
191051.350024703948293,
191061.316815640137065,
191071.283653508144198,
191081.250538305816008,
191091.217470030911500,
191101.184448681247886,
191111.151474254555069,
191121.118546748708468,
191131.085666161496192,
191141.052832490706351,
191151.020045734156156,
191160.987305889575509,
191170.954612954810727,
191180.921966927679023,
191190.889367805968504,
191200.856815587438177,
191210.824310269963462,
191220.791851851274259,
191230.759440329216886,
191240.727075701579452,
191250.694757966208272,
191260.662487120775040,
191270.630263163184281,
191280.598086091253208,
191290.565955902682617,
191300.533872595377034,
191310.501836167095462,
191320.469846615684219,
191330.437903938844102,
191340.406008134450531,
191350.374159200349823,
191360.342357134213671,
191370.310601933975704,
191380.278893597394926,
191390.247232122259447,
191400.215617506328272,
191410.184049747535028,
191420.152528843580512,
191430.121054792281939,
191440.089627591456519,
191450.058247238979675,
191460.026913732523099,
191469.995627069991315,
191479.964387249114225,
191489.933194267796353,
191499.902048123767599,
191509.870948814845178,
191519.839896338846302,
191529.808890693588182,
191539.777931876829825,
191549.747019886475755,
191559.716154720255872,
191569.685336375958286,
191579.654564851487521,
191589.623840144544374,
191599.593162253033370,
191609.562531174655305,
191619.531946907314705,
191629.501409448770573,
191639.470918796869228,
191649.440474949369673,
191659.410077904118225,
191669.379727658873890,
191679.349424211512087,
191689.319167559850030,
191699.288957701646723,
191709.258794634719379,
191719.228678356856108,
191729.198608865961432,
191739.168586159794359,
191749.138610236113891,
191759.108681092766346,
191769.078798727627145,
191779.048963138426188,
191789.019174323038897,
191798.989432279166067,
191808.959737004770432,
191818.930088497523684,
191828.900486755330348,
191838.870931776007637,
191848.841423557314556,
191858.811962097126525,
191868.782547393173445,
191878.753179443330737,
191888.723858245444717,
191898.694583797245286,
191908.665356096549658,
191918.636175141262356,
191928.607040929142386,
191938.577953458006959,
191948.548912725615082,
191958.519918729900382,
191968.490971468621865,
191978.462070939538535,
191988.433217140525812,
191998.404410069459118,
192008.375649724039249,
192018.346936102141626,
192028.318269201583462,
192038.289649020152865,
192048.261075555667048,
192058.232548806030536,
192068.204068768915022,
192078.175635442283237,
192088.147248823894188,
192098.118908911535982,
192108.090615703025833,
192118.062369196210057,
192128.034169388964074,
192138.006016279017786,
192147.977909864217509,
192157.949850142380456,
192167.921837111323839,
192177.893870768893976,
192187.865951112878975,
192197.838078141183360,
192207.810251851507928,
192217.782472241728101,
192227.754739309602883,
192237.727053053095005,
192247.699413469905267,
192257.671820557938190,
192267.644274314923678,
192277.616774738737149,
192287.589321827224921,
192297.561915578175103,
192307.534555989375804,
192317.507243058702443,
192327.479976783972234,
192337.452757163031492,
192347.425584193668328,
192357.398457873641746,
192367.371378200885374,
192377.344345173216425,
192387.317358788393904,
192397.290419044293230,
192407.263525938702514,
192417.236679469468072,
192427.209879634436220,
192437.183126431365963,
192447.156419858132722,
192457.129759912553709,
192467.103146592504345,
192477.076579895714531,
192487.050059820059687,
192497.023586363386130,
192506.997159523481969,
192516.970779298222624,
192526.944445685396204,
192536.918158682790818,
192546.891918288310990,
192556.865724499773933,
192566.839577314967755,
192576.813476731738774,
192586.787422747904202,
192596.761415361339459,
192606.735454569861759,
192616.709540371230105,
192626.683672763290815,
192636.657851743977517,
192646.632077311020112,
192656.606349462264916,
192666.580668195587350,
192676.555033508717315,
192686.529445399617543,
192696.503903866017936,
192706.478408905823017,
192716.452960516762687,
192726.427558696741471,
192736.402203443576582,
192746.376894755143439,
192756.351632629201049,
192766.326417063595727,
192776.301248056231998,
192786.276125604868867,
192796.251049707323546,
192806.226020361500559,
192816.201037565158913,
192826.176101316174027,
192836.151211612392217,
192846.126368451630697,
192856.101571831706678,
192866.076821750437375,
192876.052118205727311,
192886.027461195393698,
192896.002850717224646,
192905.978286769066472,
192915.953769348765491,
192925.929298454168020,
192935.904874083091272,
192945.880496233381564,
192955.856164902885212,
192965.831880089390324,
192975.807641790801426,
192985.783450004935730,
192995.759304729581345,
193005.735205962642794,
193015.711153701937292,
193025.687147945223842,
193035.663188690465176,
193045.639275935420301,
193055.615409677935531,
193065.591589915886289,
193075.567816647031577,
193085.544089869305026,
193095.520409580465639,
193105.496775778417941,
193115.473188461008249,
193125.449647625966463,
193135.426153271255316,
193145.402705394633813,
193155.379303993977373,
193165.355949067161418,
193175.332640611974057,
193185.309378626203397,
193195.286163107812172,
193205.262994054588489,
193215.239871464349562,
193225.216795334970811,
193235.193765664240345,
193245.170782450062688,
193255.147845690255053,
193265.124955382663757,
193275.102111525164219,
193285.079314115486341,
193295.056563151563751,
193305.033858631271869,
193315.011200552398805,
193324.988588912761770,
193334.966023710236186,
193344.943504942639265,
193354.921032607904635,
193364.898606703820406,
193374.876227228174685,
193384.853894178901101,
193394.831607553816866,
193404.809367350680986,
193414.787173567456193,
193424.765026201959699,
193434.742925252008718,
193444.720870715478668,
193454.698862590186764,
193464.676900873950217,
193474.654985564731760,
193484.633116660232190,
193494.611294158414239,
193504.589518057066016,
193514.567788354062941,
193524.546105047193123,
193534.524468134390190,
193544.502877613442251,
193554.481333482224727,
193564.459835738554830,
193574.438384380337084,
193584.416979405301390,
193594.395620811468689,
193604.374308596568881,
193614.353042758477386,
193624.331823295069626,
193634.310650204162812,
193644.289523483603261,
193654.268443131237291,
193664.247409144911217,
193674.226421522558667,
193684.205480261967750,
193694.184585360955680,
193704.163736817397876,
193714.142934629169758,
193724.122178794088541,
193734.101469310029643,
193744.080806174897589,
193754.060189386393176,
193764.039618942537345,
193774.019094841059996,
193783.998617079865653,
193793.978185656829737,
193803.957800569740357,
193813.937461816472933,
193823.917169394931989,
193833.896923302934738,
193843.876723538327496,
193853.856570098927477,
193863.836462982668309,
193873.816402187396307,
193883.796387710899580,
193893.776419551082654,
193903.756497705791844,
193913.736622172873467,
193923.716792950144736,
193933.697010035539279,
193943.677273426874308,
193953.657583122025244,
193963.637939118838403,
193973.618341415160103,
193983.598790008807555,
193993.579284897656180,
194003.559826079668710,
194013.540413552575046,
194023.521047314308817,
194033.501727362658130,
194043.482453695527511,
194053.463226310792379,
194063.444045206269948,
194073.424910379806533,
194083.405821829306660,
194093.386779552616645,
194103.367783547611907,
194113.348833812080557,
194123.329930343956221,
194133.311073141056113,
194143.292262201226549,
194153.273497522372054,
194163.254779102368047,
194173.236106939031743,
194183.217481030209456,
194193.198901373834815,
194203.180367967695929,
194213.161880809639115,
194223.143439897598000,
194233.125045229389798,
194243.106696802919032,
194253.088394615973812,
194263.070138666487765,
194273.051928952278104,
194283.033765471220249,
194293.015648221189622,
194302.997577200032538,
194312.979552405595314,
194322.961573835782474,
194332.943641488469439,
194342.925755361444317,
194352.907915452669840,
194362.890121759875910,
194372.872374281054363,
194382.854673014022410,
194392.837017956626369,
194402.819409106741659,
194412.801846462272806,
194422.784330021037022,
194432.766859780909726,
194442.749435739766341,
194452.732057895453181,
194462.714726245903876,
194472.697440788877429,
194482.680201522307470,
194492.663008444040315,
194502.645861551951384,
194512.628760843916098,
194522.611706317780772,
194532.594697971420828,
194542.577735802740790,
194552.560819809528766,
194562.543949989747489,
194572.527126341185067,
194582.510348861687817,
194592.493617549247574,
194602.476932401652448,
194612.460293416748755,
194622.443700592499226,
194632.427153926633764,
194642.410653417173307,
194652.394199061876861,
194662.377790858678054,
194672.361428805394098,
194682.345112899900414,
194692.328843140101526,
194702.312619523872854,
194712.296442049060715,
194722.280310713540530,
194732.264225515187718,
194742.248186451877700,
194752.232193521485897,
194762.216246721887728,
194772.200346050900407,
194782.184491506428458,
194792.168683086376404,
194802.152920788590563,
194812.137204610975459,
194822.121534551348304,
194832.105910607613623,
194842.090332777675940,
194852.074801059323363,
194862.059315450489521,
194872.043875949078938,
194882.028482552879723,
194892.013135259854607,
194901.997834067791700,
194911.982578974624630,
194921.967369978228817,
194931.952207076479681,
194941.937090267194435,
194951.922019548306707,
194961.906994917721022,
194971.892016373225488,
194981.877083912753733,
194991.862197534210281,
195001.847357235354139,
195011.832563014206244,
195021.817814868496498,
195031.803112796245841,
195041.788456795271486,
195051.773846863419749,
195061.759282998595154,
195071.744765198702225,
195081.730293461558176,
195091.715867785096634,
195101.701488167163916,
195111.687154605693650,
195121.672867098473944,
195131.658625643409323,
195141.644430238462519,
195151.630280881392537,
195161.616177570162108,
195171.602120302646654,
195181.588109076692490,
195191.574143890204141,
195201.560224741027923,
195211.546351627097465,
195221.532524546259083,
195231.518743496417301,
195241.505008475389332,
195251.491319481137907,
195261.477676511480240,
195271.464079564349959,
195281.450528637593379,
195291.437023729144130,
195301.423564836761216,
195311.410151958494680,
195321.396785092103528,
195331.383464235521387,
195341.370189386652783,
195351.356960543314926,
195361.343777703441447,
195371.330640864878660,
195381.317550025589298,
195391.304505183332367,
195401.291506336099701,
195411.278553481766721,
195421.265646618150640,
195431.252785743185086,
195441.239970854716375,
195451.227201950707240,
195461.214479028974893,
195471.201802087423857,
195481.189171123929555,
195491.176586136396509,
195501.164047122700140,
195511.151554080774076,
195521.139107008406427,
195531.126705903530819,
195541.114350764080882,
195551.102041587902931,
195561.089778372901492,
195571.077561116922880,
195581.065389817929827,
195591.053264473710442,
195601.041185082227457,
195611.029151641356293,
195621.017164148943266,
195631.005222602951108,
195640.993327001167927,
195650.981477341614664,
195660.969673622050323,
195670.957915840466740,
195680.946203994681127,
195690.934538082627114,
195700.922918102180120,
195710.911344051215565,
195720.899815927667078,
195730.888333729380975,
195740.876897454261780,
195750.865507100243121,
195760.854162665142212,
195770.842864146892680,
195780.831611543399049,
195790.820404852478532,
195800.809244072122965,
195810.798129200178664,
195820.787060234550154,
195830.776037173112854,
195840.765060013713082,
195850.754128754371777,
195860.743243392877048,
195870.732403927133419,
195880.721610355074517,
195890.710862674575765,
195900.700160883541685,
195910.689504979818594,
195920.678894961340120,
195930.668330826039892,
195940.657812571764225,
195950.647340196388541,
195960.636913697817363,
195970.626533073984319,
195980.616198322764831,
195990.605909442063421,
196000.595666429697303,
196010.585469283716520,
196020.575318001880078,
196030.565212582150707,
196040.555153022432933,
196050.545139320573071,
196060.535171474533854,
196070.525249482132494,
196080.515373341302620,
196090.505543049977859,
196100.495758606062736,
196110.486020007374464,
196120.476327251846669,
196130.466680337412981,
196140.457079261948820,
196150.447524023329606,
196160.438014619488968,
196170.428551048302324,
196180.419133307703305,
196190.409761395567330,
196200.400435309769819,
196210.391155048244400,
196220.381920608924702,
196230.372731989598833,
196240.363589188258629,
196250.354492202808615,
196260.345441031124210,
196270.336435671080835,
196280.327476120641222,
196290.318562377680792,
196300.309694440045860,
196310.300872305699158,
196320.292095972545212,
196330.283365438488545,
196340.274680701346369,
196350.266041759139625,
196360.257448609714629,
196370.248901251005009,
196380.240399680857081,
196390.231943897204474,
196400.223533897951711,
196410.215169681032421,
196420.206851244292920,
196430.198578585666837,
196440.190351703087799,
196450.182170594402123,
196460.174035257572541,
196470.165945690445369,
196480.157901890983339,
196490.149903857032768,
196500.141951586556388,
196510.134045077429619,
196520.126184327556984,
196530.118369334843010,
196540.110600097250426,
196550.102876612567343,
196560.095198878814699,
196570.087566893867915,
196580.079980655573308,
196590.072440161922714,
196600.064945410762448,
196610.057496400026139,
196620.050093127647415,
196630.042735591501696,
196640.035423789493507,
196650.028157719527371,
196660.020937379566021,
196670.013762767455773,
196680.006633881101152,
196689.999550718464889,
196699.992513277451508,
196709.985521555907326,
196719.978575551795075,
196729.971675263019279,
196739.964820687513566,
196749.958011823095148,
196759.951248667784967,
196769.944531219458440,
196779.937859475961886,
196789.931233435316244,
196799.924653095338726,
196809.918118454021169,
196819.911629509180784,
196829.905186258838512,
196839.898788700782461,
196849.892436833004467,
196859.886130653496366,
196869.879870159988059,
196879.873655350500485,
196889.867486222938169,
196899.861362775205635,
196909.855285005236510,
196919.849252910906216,
196929.843266490119277,
196939.837325740838423,
196949.831430660968181,
196959.825581248383969,
196969.819777501048520,
196979.814019416837255,
196989.808306993712904,
196999.802640229550889,
197009.797019122255733,
197019.791443669819273,
197029.785913870058721,
197039.780429720936809,
197049.774991220358061,
197059.769598366256105,
197069.764251156564569,
197079.758949589158874,
197089.753693661943544,
197099.748483372881310,
197109.743318719905801,
197119.738199700834230,
197129.733126313716639,
197139.728098556370242,
197149.723116426757770,
197159.718179922783747,
197169.713289042381803,
197179.708443783427356,
197189.703644143883139,
197199.698890121653676,
197209.694181714643491,
197219.689518920815317,
197229.684901738073677,
197239.680330164293991,
197249.675804197438993,
197259.671323835442308,
197269.666889076150255,
197279.662499917583773,
197289.658156357589178,
197299.653858394100098,
197309.649606025079265,
197319.645399248402100,
197329.641238062002230,
197339.637122463813284,
197349.633052451710682,
197359.629028023686260,
197369.625049177615438,
197379.621115911460947,
197389.617228223098209,
197399.613386110519059,
197409.609589571540710,
197419.605838604154997,
197429.602133206266444,
197439.598473375808680,
197449.594859110715333,
197459.591290408861823,
197469.587767268269090,
197479.584289686754346,
197489.580857662309427,
197499.577471192867961,
197509.574130276276264,
197519.570834910526173,
197529.567585093493108,
197539.564380823168904,
197549.561222097458085,
197559.558108914294280,
197569.555041271494702,
197579.552019167109393,
197589.549042599071981,
197599.546111565228784,
197609.543226063542534,
197619.540386091917753,
197629.537591648346279,
197639.534842730674427,
197649.532139336864930,
197659.529481464909622,
197669.526869112625718,
197679.524302277975949,
197689.521780958952149,
197699.519305153400637,
197709.516874859342352,
197719.514490074565401,
197729.512150797148934,
197739.509857024939265,
197749.507608755840920,
197759.505405987845734,
197769.503248718858231,
197779.501136946812039,
197789.499070669640787,
197799.497049885248998,
197809.495074591628509,
197819.493144786654739,
197829.491260468261316,
197839.489421634410974,
197849.487628283037338,
197859.485880411986727,
197869.484178019309184,
197879.482521102851024,
197889.480909660604084,
197899.479343690414680,
197909.477823190303752,
197919.476348158175824,
197929.474918591935420,
197939.473534489574376,
197949.472195848997217,
197959.470902668079361,
197969.469654944870854,
197979.468452677218011,
197989.467295863054460,
197999.466184500372037,
198009.465118587075267,
198019.464098121039569,
198029.463123100285884,
198039.462193522718735,
198049.461309386242647,
198059.460470688878559,
198069.459677428443683,
198079.458929602958960,
198089.458227210387122,
198099.457570248545380,
198109.456958715454675,
198119.456392609048635,
198129.455871927231783,
198139.455396667966852,
198149.454966829158366,
198159.454582408798160,
198169.454243404790759,
198179.453949815069791,
198189.453701637597987,
198199.453498870250769,
198209.453341511049075,
198219.453229557868326,
198229.453163008642150,
198239.453141861391487,
198249.453166113962652,
198259.453235764347482,
198269.453350810450502,
198279.453511250234442,
198289.453717081691138,
198299.453968302637804,
198309.454264911124483,
198319.454606905026594,
198329.454994282306870,
198339.455427040869836,
198349.455905178736430,
198359.456428693782073,
198369.456997583940392,
198379.457611847232329,
198389.458271481533302,
198399.458976484776940,
198409.459726854925975,
198419.460522589943139,
198429.461363687732955,
198439.462250146258157,
198449.463181963452371,
198459.464159137249226,
198469.465181665611453,
198479.466249546530889,
198489.467362777853850,
198499.468521357543068,
198509.469725283590378,
198519.470974553871201,
198529.472269166435581,
198539.473609119129833,
198549.474994409945793,
198559.476425036787987,
198569.477900997706456,
198579.479422290518414,
198589.480988913186593,
198599.482600863702828,
198609.484258140000748,
198619.485960740072187,
198629.487708661763463,
198639.489501903095515,
198649.491340461943764,
198659.493224336387357,
198669.495153524185298,
198679.497128023474943,
198689.499147832044400,
198699.501212947972817,
198709.503323369077407,
198719.505479093408212,
198729.507680118898861,
198739.509926443482982,
198749.512218065065099,
198759.514554981607944,
198769.516937191132456,
198779.519364691514056,
198789.521837480744580,
198799.524355556728551,
198809.526918917457806,
198819.529527560895076,
198829.532181484915782,
198839.534880687511759,
198849.537625166645739,
198859.540414920280455,
198869.543249946291326,
198879.546130242699292,
198889.549055807496188,
198899.552026638499228,
198909.555042733700247,
198919.558104091178393,
198929.561210708721774,
198939.564362584380433,
198949.567559716058895,
198959.570802101719892,
198969.574089739326155,
198979.577422626840416,
198989.580800762225408,
198999.584224143356550,
199009.587692768254783,
199019.591206634853734,
199029.594765741087031,
199039.598370085004717,
199049.602019664424006,
199059.605714477365837,
199069.609454521792941,
199079.613239795609843,
199089.617070296895690,
199099.620946023438592,
199109.624866973317694,
199119.628833144408418,
199129.632844534702599,
199139.636901142162969,
199149.641002964752261,
199159.645150000374997,
199169.649342247023014,
199179.653579702659044,
199189.657862365245819,
199199.662190232687863,
199209.666563303006114,
199219.670981574105099,
199229.675445044005755,
199239.679953710612608,
199249.684507571888389,
199259.689106625766726,
199269.693750870268559,
199279.698440303298412,
199289.703174922848120,
199299.707954726880416,
199309.712779713328928,
199319.717649880156387,
199329.722565225325525,
199339.727525746769970,
199349.732531442481559,
199359.737582310452126,
199369.742678348586196,
199379.747819554846501,
199389.753005927195773,
199399.758237463625846,
199409.763514162041247,
199419.768836020492017,
199429.774203036795370,
199439.779615209117765,
199449.785072535218205,
199459.790575013146736,
199469.796122640895192,
199479.801715416338993,
199489.807353337528184,
199499.813036402396392,
199509.818764608877245,
199519.824537954933476,
199529.830356438586023,
199539.836220057739411,
199549.842128810356371,
199559.848082694457844,
199569.854081707948353,
199579.860125848790631,
199589.866215115005616,
199599.872349504497834,
199609.878529015230015,
199619.884753645252204,
199629.891023392410716,
199639.897338254755596,
199649.903698230220471,
199659.910103316738969,
199669.916553512361133,
199679.923048814962385,
199689.929589222563663,
199699.936174733069493,
199709.942805344529916,
199719.949481054849457,
199729.956201862049056,
199739.962967764062341,
199749.969778758822940,
199759.976634844322689,
199769.983536018553423,
199779.990482279419666,
199789.997473625029670,
199800.004510053171543,
199810.011591561924433,
199820.018718149192864,
199830.025889812997775,
199840.033106551360106,
199850.040368362097070,
199860.047675243258709,
199870.055027192778653,
199880.062424208736047,
199890.069866288977209,
199900.077353431552183,
199910.084885634394595,
199920.092462895408971,
199930.100085212674458,
199940.107752584153786,
199950.115465007751482,
199960.123222481459379,
199970.131025003240211,
199980.138872571114916,
199990.146765182988020,
200000.154702836938668,
200010.162685530784074,
200020.170713262574282,
200030.178786030301126,
200040.186903831898235,
200050.195066665386548,
200060.203274528728798,
200070.211527419800404,
200080.219825336680515,
200090.228168277331861,
200100.236556239688070,
200110.244989221740980,
200120.253467221511528,
200130.261990236846032,
200140.270558265823638,
200150.279171306407079,
200160.287829356588190,
200170.296532414242392,
200180.305280477419728,
200190.314073544082930,
200200.322911612252938,
200210.331794679834275,
200220.340722744847881,
200230.349695805227384,
200240.358713858964620,
200250.367776904051425,
200260.376884938450530,
200270.386037960124668,
200280.395235967094777,
200290.404478957294486,
200300.413766928715631,
200310.423099879320944,
200320.432477807131363,
200330.441900710080517,
200340.451368586131139,
200350.460881433333270,
200360.470439249620540,
200370.480042032897472,
200380.489689781272318,
200390.499382492649602,
200400.509120165050263,
200410.518902796378825,
200420.528730384714436,
200430.538602927932516,
200440.548520424083108,
200450.558482871158049,
200460.568490267061861,
200470.578542609786382,
200480.588639897410758,
200490.598782127839513,
200500.608969299035380,
200510.619201408961089,
200520.629478455695789,
200530.639800437144004,
200540.650167351326672,
200550.660579196177423,
200560.671035969717195,
200570.681537669879617,
200580.692084294714732,
200590.702675842185272,
200600.713312310224865,
200610.723993696854450,
200620.734720000065863,
200630.745491217850940,
200640.756307348114206,
200650.767168388905702,
200660.778074338217266,
200670.789025194011629,
200680.800020954280626,
200690.811061616957886,
200700.822147180093452,
200710.833277641620953,
200720.844452999590430,
200730.855673251935514,
200740.866938396618934,
200750.878248431690736,
200760.889603355113650,
200770.901003164850408,
200780.912447858863743,
200790.923937435232801,
200800.935471891862107,
200810.947051226743497,
200820.958675437868806,
200830.970344523229869,
200840.982058480876731,
200850.993817308684811,
200861.005621004704153,
200871.017469566897489,
200881.029362993256655,
200891.041301281860797,
200901.053284430527128,
200911.065312437334796,
200921.077385300246533,
200931.089503017370589,
200941.101665586524177,
200951.113873005757341,
200961.126125273091020,
200971.138422386517050,
200981.150764343939954,
200991.163151143409777,
201001.175582782947458,
201011.188059260515729,
201021.200580574077321,
201031.213146721624071,
201041.225757701147813,
201051.238413510727696,
201061.251114148210036,
201071.263859611673979,
201081.276649899111362,
201091.289485008484917,
201101.302364937815582,
201111.315289685036987,
201121.328259248199174,
201131.341273625264876,
201141.354332814225927,
201151.367436813074164,
201161.380585619801423,
201171.393779232428642,
201181.407017648947658,
201191.420300867292099,
201201.433628885482904,
201211.447001701570116,
201221.460419313516468,
201231.473881719226483,
201241.487388916779310,
201251.500940904195886,
201261.514537679409841,
201271.528179240442114,
201281.541865585284540,
201291.555596711928956,
201301.569372618367197,
201311.583193302561995,
201321.597058762592496,
201331.610968996363226,
201341.624924001924228,
201351.638923777238233,
201361.652968320326181,
201371.667057629209012,
201381.681191701849457,
201391.695370536210248,
201401.709594130370533,
201411.723862482234836,
201421.738175589853199,
201431.752533451217460,
201441.766936064348556,
201451.781383427180117,
201461.795875537762186,
201471.810412394086597,
201481.824993994116085,
201491.839620335900690,
201501.854291417374043,
201511.869007236586185,
201521.883767791528953,
201531.898573080223287,
201541.913423100573709,
201551.928317850717576,
201561.943257328530308,
201571.958241532091051,
201581.973270459362539,
201591.988344108336605,
201602.003462477063295,
201612.018625563476235,
201622.033833365596365,
201632.049085881502833,
201642.064383109071059,
201652.079725046380190,
201662.095111691422062,
201672.110543042188510,
201682.126019096671371,
201692.141539852862479,
201702.157105308840983,
201712.172715462482302,
201722.188370311894687,
201732.204069855069974,
201742.219814089941792,
201752.235603014560184,
201762.251436626916984,
201772.267314925033133,
201782.283237906900467,
201792.299205570539925,
201802.315217913856031,
201812.331274935073452,
201822.347376631951192,
201832.363523002626607,
201842.379714045091532,
201852.395949757308699,
201862.412230137299048,
201872.428555183083517,
201882.444924892712152,
201892.461339264089474,
201902.477798295294633,
201912.494301984290360,
201922.510850329097593,
201932.527443327737274,
201942.544080978230340,
201952.560763278510422,
201962.577490226656664,
201972.594261820631800,
201982.611078058514977,
201992.627938938210718,
202002.644844457769068,
202012.661794615210965,
202022.678789408557350,
202032.695828835741850,
202042.712912894872716,
202052.730041583912680,
202062.747214900824474,
202072.764432843687246,
202082.781695410521934,
202092.799002599262167,
202102.816354407928884,
202112.833750834572129,
202122.851191877212841,
202132.868677533784648,
202142.886207802337594,
202152.903782680921722,
202162.921402167499764,
202172.939066260092659,
202182.956774956750451,
202192.974528255406767,
202202.992326154111652,
202213.010168650915148,
202223.028055743721779,
202233.045987430668902,
202243.063963709690142,
202253.081984578835545,
202263.100050036038738,
202273.118160079437075,
202283.136314706935082,
202293.154513916611904,
202303.172757706430275,
202313.191046074440237,
202323.209379018662730,
202333.227756537060486,
202343.246178627712652,
202353.264645288552856,
202363.283156517660245,
202373.301712313026655,
202383.320312672643922,
202393.338957594562089,
202403.357647076802095,
202413.376381117355777,
202423.395159714214969,
202433.413982865429716,
202443.432850569050061,
202453.451762822951423,
202463.470719625271158,
202473.489720974059310,
202483.508766867220402,
202493.527857302804478,
202503.546992278919788,
202513.566171793412650,
202523.585395844449522,
202533.604664429993136,
202543.623977548006224,
202553.643335196597036,
202563.662737373728305,
202573.682184077420970,
202583.701675305725075,
202593.721211056632455,
202603.740791328164050,
202613.760416118311696,
202623.780085425125435,
202633.799799246626208,
202643.819557580834953,
202653.839360425772611,
202663.859207779431017,
202673.879099639802007,
202683.899036005022936,
202693.919016872969223,
202703.939042241749121,
202713.959112109383568,
202723.979226473835297,
202733.999385333183454,
202744.019588685448980,
202754.039836528536398,
202764.060128860641271,
202774.080465679668123,
202784.100846983696101,
202794.121272770687938,
202804.141743038693676,
202814.162257785763359,
202824.182817009859718,
202834.203420709120110,
202844.224068881390849,
202854.244761524809292,
202864.265498637454584,
202874.286280217173044,
202884.307106262102025,
202894.327976770320674,
202904.348891739733517,
202914.369851168390596,
202924.390855054371059,
202934.411903395608533,
202944.432996190211270,
202954.454133436229313,
202964.475315131567186,
202974.496541274304036,
202984.517811862548115,
202994.539126894174842,
203004.560486367292469,
203014.581890279951040,
203024.603338630113285,
203034.624831415829249,
203044.646368635148974,
203054.667950286035193,
203064.689576366625261,
203074.711246874823701,
203084.732961808709661,
203094.754721166333184,
203104.776524945715209,
203114.798373144818470,
203124.820265761722112,
203134.842202794447076,
203144.864184241043404,
203154.886210099502932,
203164.908280367875705,
203174.930395044153556,
203184.952554126444738,
203194.974757612653775,
203204.997005500888918,
203215.019297789200209,
203225.041634475579485,
203235.064015558076790,
203245.086441034683958,
203255.108910903421929,
203265.131425162369851,
203275.153983809577767,
203285.176586843008408,
203295.199234260711819,
203305.221926060708938,
203315.244662241078913,
203325.267442799813580,
203335.290267734904774,
203345.313137044489849,
203355.336050726502435,
203365.359008779021678,
203375.382011200039415,
203385.405057987634791,
203395.428149139799643,
203405.451284654642222,
203415.474464530096157,
203425.497688764211489,
203435.520957355096471,
203445.544270300684730,
203455.567627599055413,
203465.591029248287668,
203475.614475246315124,
203485.637965591216926,
203495.661500281101326,
203505.685079313901952,
203515.708702687697951,
203525.732370400452055,
203535.756082450272515,
203545.779838835209375,
203555.803639553254470,
203565.827484602457844,
203575.851373980869539,
203585.875307686452288,
203595.899285717343446,
203605.923308071505744,
203615.947374746960122,
203625.971485741843935,
203635.995641054149019,
203646.019840681809001,
203656.044084623019444,
203666.068372875655768,
203676.092705437913537,
203686.117082307726378,
203696.141503483144334,
203706.165968962246552,
203716.190478742995765,
203726.215032823529327,
203736.239631201809971,
203746.264273875887739,
203756.288960843841778,
203766.313692103693029,
203776.338467653404223,
203786.363287491141818,
203796.388151614810340,
203806.413060022576246,
203816.438012712373165,
203826.463009682309348,
203836.488050930434838,
203846.513136454683263,
203856.538266253221082,
203866.563440323981922,
203876.588658665074036,
203886.613921274576569,
203896.639228150423151,
203906.664579290722031,
203916.689974693494150,
203926.715414356760448,
203936.740898278629174,
203946.766426457063062,
203956.791998890141258,
203966.817615575913806,
203976.843276512430748,
203986.868981697683921,
203996.894731129723368,
204006.920524806628237,
204016.946362726506777,
204026.972244887205306,
204036.998171286948491,
204047.024141923699062,
204057.050156795536168,
204067.076215900451643,
204077.102319236524636,
204087.128466801776085,
204097.154658594314242,
204107.180894612072734,
204117.207174853247125,
204127.233499315712834,
204137.259867997636320,
204147.286280896980315,
204157.312738011882175,
204167.339239340333734,
204177.365784880355932,
204187.392374630027916,
204197.419008587341523,
204207.445686750463210,
204217.472409117326606,
204227.499175686039962,
204237.525986454624217,
204247.552841421071207,
204257.579740583518287,
204267.606683939986397,
204277.633671488525579,
204287.660703227156773,
204297.687779153930023,
204307.714899266895372,
204317.742063564161072,
204327.769272043660749,
204337.796524703531759,
204347.823821541795041,
204357.851162556500640,
204367.878547745698597,
204377.905977107409853,
204387.933450639771763,
204397.960968340688851,
204407.988530208327575,
204418.016136240679771,
204428.043786435853690,
204438.071480791841168,
204448.099219306721352,
204458.127001978515182,
204468.154828805301804,
204478.182699785189470,
204488.210614916053601,
204498.238574196089758,
204508.266577623347985,
204518.294625195820117,
204528.322716911585303,
204538.350852768664481,
204548.379032765165903,
204558.407256899110507,
204568.435525168519234,
204578.463837571529439,
204588.492194106074749,
204598.520594770350726,
204608.549039562291000,
204618.577528479945613,
204628.606061521481024,
204638.634638684859965,
204648.663259968161583,
204658.691925369406817,
204668.720634886733023,
204678.749388518073829,
204688.778186261566589,
204698.807028115261346,
204708.835914077208145,
204718.864844145427924,
204728.893818317999830,
204738.922836593003012,
204748.951898968429305,
204758.981005442386959,
204769.010156012926018,
204779.039350678038318,
204789.068589435948525,
204799.097872284473851,
204809.127199221897172,
204819.156570246093906,
204829.185985355230514,
204839.215444547327934,
204849.244947820465313,
204859.274495172634488,
204869.304086601972813,
204879.333722106501227,
204889.363401684269775,
204899.393125333357602,
204909.422893051814754,
204919.452704837662168,
204929.482560689008096,
204939.512460603902582,
204949.542404580424773,
204959.572392616566503,
204969.602424710436026,
204979.632500860054279,
204989.662621063529514,
204999.692785318882670,
205009.722993624192895,
205019.753245977510232,
205029.783542376913829,
205039.813882820453728,
205049.844267306150869,
205059.874695832171710,
205069.905168396449881,
205079.935684997122735,
205089.966245632211212,
205099.996850299823564,
205110.027498997980729,
205120.058191724732751,
205130.088928478216985,
205140.119709256367059,
205150.150534057378536,
205160.181402879214147,
205170.212315720040351,
205180.243272577819880,
205190.274273450631881,
205200.305318336613709,
205210.336407233728096,
205220.367540140083292,
205230.398717053787550,
205240.429937972803600,
205250.461202895297902,
205260.492511819291394,
205270.523864742805017,
205280.555261664005229,
205290.586702580796555,
205300.618187491432764,
205310.649716393818380,
205320.681289286119863,
205330.712906166387256,
205340.744567032612395,
205350.776271882961737,
205360.808020715427119,
205370.839813528145896,
205380.871650319080800,
205390.903531086427392,
205400.935455828119302,
205410.967424542322988,
205420.999437227030285,
205431.031493880378548,
205441.063594500388717,
205451.095739085081732,
205461.127927632653154,
205471.160160141094821,
205481.192436608456774,
205491.224757032847265,
205501.257121412316337,
205511.289529744914034,
205521.321982028719503,
205531.354478261782788,
205541.387018442299450,
205551.419602568115806,
205561.452230637514731,
205571.484902648400748,
205581.517618598969420,
205591.550378487183480,
205601.583182311209384,
205611.616030069068074,
205621.648921758780489,
205631.681857378513087,
205641.714836926286807,
205651.747860400180798,
205661.780927798215998,
205671.814039118558867,
205681.847194359230343,
205691.880393518280471,
205701.913636593817500,
205711.946923583891476,
205721.980254486552440,
205732.013629299966851,
205742.047048022097442,
205752.080510651023360,
205762.114017184881959,
205772.147567621723283,
205782.181161959568271,
205792.214800196612487,
205802.248482330789557,
205812.282208360295044,
205822.315978283062577,
205832.349792097287718,
205842.383649800962303,
205852.417551392194582,
205862.451496869092807,
205872.485486229677917,
205882.519519472087268,
205892.553596594341798,
205902.587717594520655,
205912.621882470673881,
205922.656091220938833,
205932.690343843365554,
205942.724640336004086,
205952.758980696962681,
205962.793364924291382,
205972.827793016098440,
205982.862264970433898,
205992.896780785376905,
206002.931340459006606,
206012.965943989373045,
206023.000591374642681,
206033.035282612778246,
206043.070017701946199,
206053.104796640109271,
206063.139619425492128,
206073.174486056115711,
206083.209396530000959,
206093.244350845285226,
206103.279349000018556,
206113.314390992280096,
206123.349476820207201,
206133.384606481762603,
206143.419779975141864,
206153.454997298365925,
206163.490258449513931,
206173.525563426665030,
206183.560912227927474,
206193.596304851293098,
206203.631741294986568,
206213.667221556970617,
206223.702745635411702,
206233.738313528272556,
206243.773925233748741,
206253.809580749832094,
206263.845280074689072,
206273.881023206340615,
206283.916810142924078,
206293.952640882402193,
206303.988515423028730,
206314.024433762708213,
206324.060395899694413,
206334.096401831891853,
206344.132451557496097,
206354.168545074586291,
206364.204682381241582,
206374.240863475453807,
206384.277088355476735,
206394.313357019214891,
206404.349669464863837,
206414.386025690473616,
206424.422425694152480,
206434.458869473921368,
206444.495357027917635,
206454.531888354191324,
206464.568463450879790,
206474.605082316062180,
206484.641744947730331,
206494.678451344050700,
206504.715201503073331,
206514.751995422964683,
206524.788833101658383,
206534.825714537379099,
206544.862639728118666,
206554.899608672043541,
206564.936621367174666,
206574.973677811649395,
206585.010778003488667,
206595.047921940858942,
206605.085109621752053,
206615.122341044363566,
206625.159616206685314,
206635.196935106854653,
206645.234297742979834,
206655.271704113081796,
206665.309154215268791,
206675.346648047649069,
206685.384185608301777,
206695.421766895306064,
206705.459391906799283,
206715.497060640773270,
206725.534773095423589,
206735.572529268742073,
206745.610329158895183,
206755.648172763932962,
206765.686060081992764,
206775.723991111037321,
206785.761965849262197,
206795.799984294746537,
206805.838046445627697,
206815.876152299868409,
206825.914301855664235,
206835.952495111036114,
206845.990732064121403,
206856.029012712970143,
206866.067337055748794,
206876.105705090507399,
206886.144116815296002,
206896.182572228251956,
206906.221071327396203,
206916.259614110953407,
206926.298200576915406,
206936.336830723419553,
206946.375504548457684,
206956.414222050283570,
206966.452983226830838,
206976.491788076353259,
206986.530636596813565,
206996.569528786349110,
207006.608464643097250,
207016.647444165049819,
207026.686467350373277,
207036.725534197146771,
207046.764644703478552,
207056.803798867418664,
207066.842996687133564,
207076.882238160644192,
207086.921523286087904,
207096.960852061485639,
207107.000224485091167,
207117.039640554809012,
207127.079100268863840,
207137.118603625363903,
207147.158150622271933,
207157.197741257812595,
207167.237375530006830,
207177.277053437021095,
207187.316774976876331,
207197.356540147709893,
207207.396348947571823,
207217.436201374628581,
207227.476097426959313,
207237.516037102643168,
207247.556020399788395,
207257.596047316445038,
207267.636117850779556,
207277.676232000812888,
207287.716389764740597,
207297.756591140583623,
207307.796836126508424,
207317.837124720535940,
207327.877456920774421,
207337.917832725390326,
207347.958252132404596,
207357.998715139954584,
207368.039221746148542,
207378.079771949065616,
207388.120365746814059,
207398.161003137531225,
207408.201684119267156,
207418.242408690071898,
207428.283176848170115,
207438.323988591553643,
207448.364843918359838,
207458.405742826696951,
207468.446685314731440,
207478.487671380455140,
207488.528701021976303,
207498.569774237490492,
207508.610891024989542,
207518.652051382698119,
207528.693255308549851,
207538.734502800827613,
207548.775793857494136,
207558.817128476744983,
207568.858506656601094,
207578.899928395199822,
207588.941393690736732,
207598.982902541145450,
207609.024454944650643,
207619.066050899302354,
207629.107690403208835,
207639.149373454507440,
207649.191100051248213,
207659.232870191568509,
207669.274683873605682,
207679.316541095409775,
207689.358441855089040,
207699.400386150780832,
207709.442373980564298,
207719.484405342518585,
207729.526480234868359,
207739.568598655518144,
207749.610760602750815,
207759.652966074587312,
207769.695215069194091,
207779.737507584592095,
207789.779843618947780,
207799.822223170340294,
207809.864646236936096,
207819.907112816727022,
207829.949622907908633,
207839.992176508560078,
207850.034773616818711,
207860.077414230734576,
207870.120098348474130,
207880.162825968087418,
207890.205597087711794,
207900.248411705484614,
207910.291269819455920,
207920.334171427792171,
207930.377116528543411,
207940.420105119876098,
207950.463137199840276,
207960.506212766573299,
207970.549331818241626,
207980.592494352837093,
207990.635700368526159,
208000.678949863417074,
208010.722242835676298,
208020.765579283324769,
208030.808959204528946,
208040.852382597338874,
208050.895849459921010,
208060.939359790383605,
208070.982913586834911,
208081.026510847324971,
208091.070151570049347,
208101.113835753058083,
208111.157563394546742,
208121.201334492507158,
208131.245149045163998,
208141.289007050567307,
208151.332908506796230,
208161.376853412017226,
208171.420841764338547,
208181.464873561868444,
208191.508948802744271,
208201.553067484986968,
208211.597229606792098,
208221.641435166267911,
208231.685684161522659,
208241.729976590664592,
208251.774312451743754,
208261.818691743013915,
208271.863114462466910,
208281.907580608240096,
208291.952090178529033,
208301.996643171296455,
208312.041239584825234,
208322.085879417078104,
208332.130562666308833,
208342.175289330567466,
208352.220059407904046,
208362.264872896543238,
208372.309729794535087,
208382.354630100016948,
208392.399573811155278,
208402.444560925912810,
208412.489591442601522,
208422.534665359155042,
208432.579782673856243,
208442.624943384696962,
208452.670147489843657,
208462.715394987433683,
208472.760685875546187,
208482.806020152260317,
208492.851397815829841,
208502.896818864217494,
208512.942283295706147,
208522.987791108258534,
208533.033342300041113,
208543.078936869191239,
208553.124574813846266,
208563.170256132056238,
208573.215980822045822,
208583.261748881835956,
208593.307560309593100,
208603.353415103396401,
208613.399313261412317,
208623.445254781749099,
208633.491239662515000,
208643.537267901818268,
208653.583339497854467,
208663.629454448586330,
208673.675612752296729,
208683.721814407035708,
208693.768059410940623,
208703.814347762119723,
208713.860679458681261,
208723.907054498733487,
208733.953472880501067,
208743.999934601975838,
208754.046439661353361,
208764.092988056712784,
208774.139579786191462,
208784.186214847984957,
208794.232893240114208,
208804.279614960745675,
208814.326380007987609,
208824.373188379919156,
208834.420040074823191,
208844.466935090604238,
208854.513873425545171,
208864.560855077754240,
208874.607880045281490,
208884.654948326293379,
208894.702059918927262,
208904.749214821262285,
208914.796413031464908,
208924.843654547643382,
208934.890939367935061,
208944.938267490419094,
208954.985638913261937,
208965.033053634630051,
208975.080511652602581,
208985.128012965258677,
208995.175557570793899,
209005.223145467316499,
209015.270776652905624,
209025.318451125785941,
209035.366168883949285,
209045.413929925678531,
209055.461734248965513,
209065.509581852005795,
209075.557472732936731,
209085.605406889837468,
209095.653384320816258,
209105.701405024126871,
209115.749468997790245,
209125.797576239914633,
209135.845726748666493,
209145.893920522183180,
209155.942157558660256,
209165.990437856089557,
209176.038761412637541,
209186.087128226499772,
209196.135538295755396,
209206.183991618541768,
209216.232488192996243,
209226.281028017227072,
209236.329611089371610,
209246.378237407596316,
209256.426906969980337,
209266.475619774631923,
209276.524375819775742,
209286.573175103490939,
209296.622017623856664,
209306.670903379097581,
209316.719832367292838,
209326.768804586550687,
209336.817820035095792,
209346.866878710919991,
209356.915980612247949,
209366.965125737217022,
209377.014314083906356,
209387.063545650511514,
209397.112820435140748,
209407.162138435844099,
209417.211499650904443,
209427.260904078342719,
209437.310351716296282,
209447.359842562960694,
209457.409376616444206,
209467.458953874855069,
209477.508574336330639,
209487.558237999008270,
209497.607944861054420,
209507.657694920548238,
209517.707488175656181,
209527.757324624573812,
209537.807204265322071,
209547.857127096096519,
209557.907093115005409,
209567.957102320215199,
209578.007154709775932,
209588.057250281999586,
209598.107389034877997,
209608.157570966519415,
209618.207796075177612,
209628.258064358960837,
209638.308375815890031,
209648.358730444248067,
209658.409128242084989,
209668.459569207567256,
209678.510053338861326,
209688.560580634046346,
209698.611151091259671,
209708.661764708725968,
209718.712421484495280,
209728.763121416675858,
209738.813864503492368,
209748.864650743053062,
209758.915480133466190,
209768.966352672927314,
209779.017268359544687,
209789.068227191426558,
209799.119229166768491,
209809.170274283649633,
209819.221362540236441,
209829.272493934724480,
209839.323668465134688,
209849.374886129691731,
209859.426146926532965,
209869.477450853766641,
209879.528797909530113,
209889.580188091989839,
209899.631621399283176,
209909.683097829518374,
209919.734617380861891,
209929.786180051509291,
209939.837785839481512,
209949.889434743003221,
209959.941126760153566,
209969.992861889157211,
209980.044640128093306,
209990.096461475186516,
210000.148325928428676,
210010.200233486073557,
210020.252184146258514,
210030.304177907120902,
210040.356214766710764,
210050.408294723310973,
210060.460417775000678,
210070.512583919888129,
210080.564793156168889,
210090.617045481980313,
210100.669340895459754,
210110.721679394715466,
210120.774060977943009,
210130.826485643221531,
210140.878953388804803,
210150.931464212742867,
210160.984018113173079,
210171.036615088291001,
210181.089255136263091,
210191.141938255139394,
210201.194664443144575,
210211.247433698386885,
210221.300246019061888,
210231.353101403248729,
210241.405999849084765,
210251.458941354823764,
210261.511925918486668,
210271.564953538269037,
210281.618024212366436,
210291.671137938858010,
210301.724294715910219,
210311.777494541689521,
210321.830737414333271,
210331.884023331920616,
210341.937352292705327,
210351.990724294795655,
210362.044139336270746,
210372.097597415413475,
210382.151098530273885,
210392.204642679018434,
210402.258229859784478,
210412.311860070767580,
210422.365533310075989,
210432.419249575847061,
210442.473008866247255,
210452.526811179443030,
210462.580656513571739,
210472.634544866770739,
210482.688476237235591,
210492.742450622987235,
210502.796468022337649,
210512.850528433365980,
210522.904631854209583,
210532.958778283005813,
210543.012967717979336,
210553.067200157209300,
210563.121475598862162,
210573.175794041104382,
210583.230155482102418,
210593.284559919935418,
210603.339007352828048,
210613.393497778859455,
210623.448031196312513,
210633.502607603179058,
210643.557226997741964,
210653.611889378080377,
210663.666594742360758,
210673.721343088749563,
210683.776134415384149,
210693.830968720401870,
210703.885846001969185,
210713.940766258252552,
210723.995729487447534,
210734.050735687575070,
210744.105784856947139,
210754.160876993613783,
210764.216012095770566,
210774.271190161525737,
210784.326411189103965,
210794.381675176613498,
210804.436982122191694,
210814.492332024034113,
210824.547724880278111,
210834.603160689119250,
210844.658639448636677,
210854.714161156996852,
210864.769725812424440,
210874.825333413056796,
210884.880983956973068,
210894.936677442397922,
210904.992413867468713,
210915.048193230322795,
210925.104015529184835,
210935.159880762133980,
210945.215788927365793,
210955.271740023017628,
210965.327734047285048,
210975.383770998305408,
210985.439850874157855,
210995.495973673154367,
211005.552139393286780,
211015.608348032867070,
211025.664599589916179,
211035.720894062716980,
211045.777231449377723,
211055.833611748006660,
211065.890034956799354,
211075.946501073951367,
211086.003010097541846,
211096.059562025824562,
211106.116156856878661,
211116.172794588928809,
211126.229475220083259,
211136.286198748508468,
211146.342965172370896,
211156.399774489895208,
211166.456626699131448,
211176.513521798275178,
211186.570459785521962,
211196.627440659038257,
211206.684464416932315,
211216.741531057399698,
211226.798640578577761,
211236.855792978662066,
211246.912988255760865,
211256.970226408127928,
211267.027507433842402,
211277.084831331099849,
211287.142198098037625,
211297.199607732851291,
211307.257060233678203,
211317.314555598713923,
211327.372093826037599,
211337.429674913932104,
211347.487298860476585,
211357.544965663866606,
211367.602675322239520,
211377.660427833790891,
211387.718223196658073,
211397.776061409065733,
211407.833942469034810,
211417.891866374906385,
211427.949833124730503,
211438.007842716702726,
211448.065895148989512,
211458.123990419728216,
211468.182128527143504,
211478.240309469372733,
211488.298533244553255,
211498.356799850880634,
211508.415109286521329,
211518.473461549583590,
211528.531856638292084,
211538.590294550871477,
211548.648775285371812,
211558.707298840017756,
211568.765865212917561,
211578.824474402324995,
211588.883126406377414,
211598.941821223212173,
211609.000558850995731,
211619.059339287952753,
211629.118162532220595,
211639.177028581936611,
211649.235937435267260,
211659.294889090408105,
211669.353883545583813,
211679.412920798844425,
211689.472000848414609,
211699.531123692519031,
211709.590289329207735,
211719.649497756763594,
211729.708748973265756,
211739.768042976938887,
211749.827379765920341,
211759.886759338463889,
211769.946181692561368,
211780.005646826582961,
211790.065154738607816,
211800.124705426773289,
211810.184298889274942,
211820.243935124250129,
211830.303614129981725,
211840.363335904577980,
211850.423100446147146,
211860.482907752942992,
211870.542757823102875,
211880.602650654822355,
211890.662586246209685,
211900.722564595489530,
211910.782585700857453,
211920.842649560479913,
211930.902756172406953,
211940.962905534979654,
211951.023097646248061,
211961.083332504495047,
211971.143610107770655,
211981.203930454328656,
211991.264293542306405,
212001.324699369957671,
212011.385147935361601,
212021.445639236684656,
212031.506173272122396,
212041.566750039957697,
212051.627369538182393,
212061.688031765079359,
212071.748736718844157,
212081.809484397585038,
212091.870274799497565,
212101.931107922748197,
212111.991983765532495,
212122.052902326016920,
212132.113863602338824,
212142.174867592781084,
212152.235914295422845,
212162.297003708459670,
212172.358135830087122,
212182.419310658471659,
212192.480528191750636,
212202.541788428148720,
212212.603091365832370,
212222.664437002997147,
212232.725825337809511,
212242.787256368377712,
212252.848730092955520,
212262.910246509738499,
212272.971805616834899,
212283.033407412469387,
212293.095051894779317,
212303.156739062018460,
212313.218468912265962,
212323.280241443775594,
212333.342056654713815,
212343.403914543188876,
212353.465815107483650,
212363.527758345735492,
212373.589744256081758,
212383.651772836747114,
212393.713844085898018,
212403.775958001730032,
212413.838114582380513,
212423.900313826074125,
212433.962555730977328,
212444.024840295227477,
212454.087167517078342,
212464.149537394696381,
212474.211949926218949,
212484.274405109841609,
212494.336902943759924,
212504.399443426140351,
212514.462026555178454,
212524.524652329098899,
212534.587320745922625,
212544.650031804048922,
212554.712785501527833,
212564.775581836525816,
212574.838420807296643,
212584.901302411948564,
212594.964226648735348,
212605.027193515852559,
212615.090203011408448,
212625.153255133627681,
212635.216349880647613,
212645.279487250692910,
212655.342667241988238,
212665.405889852670953,
212675.469155080907512,
212685.532462924864376,
212695.595813382766210,
212705.659206452837680,
212715.722642133187037,
212725.786120422038948,
212735.849641317530768,
212745.913204817916267,
212755.976810921361903,
212766.040459626034135,
212776.104150930099422,
212786.167884831753327,
212796.231661329249619,
212806.295480420696549,
212816.359342104289681,
212826.423246378224576,
212836.487193240667693,
212846.551182689872803,
212856.615214723948156,
212866.679289341176627,
212876.743406539608259,
212886.807566317525925,
212896.871768673125189,
212906.936013604543405,
212917.000301109976135,
212927.064631187648047,
212937.129003835667390,
212947.193419052317040,
212957.257876835734351,
212967.322377184143988,
212977.386920095654204,
212987.451505568547873,
212997.516133600933244,
213007.580804191093193,
213017.645517337135971,
213027.710273037228035,
213037.775071289681364,
213047.839912092546001,
213057.904795444104820,
213067.969721342524281,
213078.034689785999944,
213088.099700772698270,
213098.164754300843924,
213108.229850368545158,
213118.294988974055741,
213128.360170115600340,
213138.425393791345414,
213148.490659999428317,
213158.555968738073716,
213168.621320005506277,
213178.686713799892459,
213188.752150119427824,
213198.817628962278832,
213208.883150326699251,
213218.948714210739126,
213229.014320612768643,
213239.079969530896051,
213249.145660963316914,
213259.211394908226794,
213269.277171363792149,
213279.342990328237647,
213289.408851799758850,
213299.474755776551319,
213309.540702256781515,
213319.606691238674102,
213329.672722720395541,
213339.738796700112289,
213349.804913176107220,
213359.871072146517690,
213369.937273609539261,
213380.003517563396599,
213390.069804006256163,
213400.136132936255308,
213410.202504351676907,
213420.268918250716524,
213430.335374631569721,
213440.401873492344748,
213450.468414831324480,
213460.534998646646272,
213470.601624936534790,
213480.668293699243804,
213490.735004932881566,
213500.801758635672741,
213510.868554805812892,
213520.935393441497581,
213531.002274540980579,
213541.069198102370137,
213551.136164123890921,
213561.203172603738494,
213571.270223540166626,
213581.337316931283567,
213591.404452775343088,
213601.471631070540752,
213611.538851815072121,
213621.606115007132757,
213631.673420644889120,
213641.740768726536771,
213651.808159250358585,
213661.875592214491917,
213671.943067617132328,
213682.010585456475383,
213692.078145730774850,
213702.145748438138980,
213712.213393576821545,
213722.281081145076314,
213732.348811141011538,
213742.416583562851883,
213752.484398408792913,
213762.552255677088397,
213772.620155365846585,
213782.688097473350354,
213792.756081997766159,
213802.824108937318670,
213812.892178290174343,
213822.960290054528741,
213833.028444228664739,
213843.096640810632380,
213853.164879798772745,
213863.233161191223189,
213873.301484986208379,
213883.369851181952981,
213893.438259776594350,
213903.506710768357152,
213913.575204155495157,
213923.643739936145721,
213933.712318108533509,
213943.780938670912292,
213953.849601621419424,
213963.918306958279572,
213973.987054679659195,
213984.055844783812063,
213994.124677268904634,
214004.193552133219782,
214014.262469374894863,
214024.331428992125439,
214034.400430983165279,
214044.469475346122636,
214054.538562079309486,
214064.607691180863185,
214074.676862649066607,
214084.746076482057106,
214094.815332678030245,
214104.884631235239794,
214114.953972151881317,
214125.023355426121270,
214135.092781056184322,
214145.162249040295137,
214155.231759376678383,
214165.301312063471414,
214175.370907098957105,
214185.440544481301913,
214195.510224208672298,
214205.579946279351134,
214215.649710691504879,
214225.719517443358200,
214235.789366533135762,
214245.859257959004026,
214255.929191719158553,
214265.999167811882216,
214276.069186235254165,
214286.139246987615479,
214296.209350067161722,
214306.279495472001145,
214316.349683200416621,
214326.419913250632817,
214336.490185620816192,
214346.560500309162308,
214356.630857313924935,
214366.701256633328740,
214376.771698265540181,
214386.842182208783925,
214396.912708461255534,
214406.983277021179674,
214417.053887886722805,
214427.124541056138696,
214437.195236527652014,
214447.265974299458321,
214457.336754369782284,
214467.407576736819465,
214477.478441398765426,
214487.549348353844834,
214497.620297600253252,
214507.691289136215346,
214517.762322959984886,
214527.833399069699226,
214537.904517463641241,
214547.975678139948286,
214558.046881096932339,
214568.118126332701650,
214578.189413845509989,
214588.260743633582024,
214598.332115695084212,
214608.403530028270325,
214618.474986631394131,
214628.546485502622090,
214638.618026640120661,
214648.689610042201821,
214658.761235707032029,
214668.832903632806847,
214678.904613817750942,
214688.976366260088980,
214699.048160958016524,
214709.119997909787344,
214719.191877113567898,
214729.263798567611957,
214739.335762270144187,
214749.407768219331047,
214759.479816413426306,
214769.551906850625528,
214779.624039529095171,
214789.696214447176317,
214799.768431602977216,
214809.840690994751640,
214819.912992620724253,
214829.985336479090620,
214840.057722568104509,
214850.130150885932380,
214860.202621430828003,
214870.275134200957837,
214880.347689194633858,
214890.420286409993423,
214900.492925845261198,
214910.565607498720055,
214920.638331368507352,
214930.711097452847753,
214940.783905749995029,
214950.856756258173846,
214960.929648975579767,
214971.002583900408354,
214981.075561030913377,
214991.148580365348607,
215001.221641901822295,
215011.294745638646418,
215021.367891574045643,
215031.441079706215532,
215041.514310033322545,
215051.587582553620450,
215061.660897265392123,
215071.734254166774917,
215081.807653256022604,
215091.881094531359849,
215101.954577991011320,
215112.028103633143473,
215122.101671456068289,
215132.175281457952224,
215142.248933637019945,
215152.322627991467016,
215162.396364519576309,
215172.470143219543388,
215182.543964089534711,
215192.617827127862256,
215202.691732332663378,
215212.765679702220950,
215222.839669234759640,
215232.913700928475009,
215242.987774781591725,
215253.061890792334452,
215263.136048958927859,
215273.210249279596610,
215283.284491752536269,
215293.358776376000606,
215303.433103148243390,
215313.507472067401977,
215323.581883131788345,
215333.656336339568952,
215343.730831688997569,
215353.805369178298861,
215363.879948805697495,
215373.954570569389034,
215384.029234467598144,
215394.103940498578595,
215404.178688660555054,
215414.253478951723082,
215424.328311370365554,
215434.403185914648930,
215444.478102582797874,
215454.553061373066157,
215464.628062283678446,
215474.703105312859407,
215484.778190458833706,
215494.853317719826009,
215504.928487094060984,
215515.003698579734191,
215525.078952175128506,
215535.154247878439492,
215545.229585687862709,
215555.304965601739241,
215565.380387618177338,
215575.455851735430770,
215585.531357951782411,
215595.606906265427824,
215605.682496674533468,
215615.758129177411320,
215625.833803772256942,
215635.909520457324106,
215645.985279230808374,
215656.061080090934411,
215666.136923035985092,
215676.212808064068668,
215686.288735173526220,
215696.364704362582415,
215706.440715629432816,
215716.516768972331192,
215726.592864389473107,
215736.669001879112329,
215746.745181439473527,
215756.821403068781365,
215766.897666765260510,
215776.973972527164733,
215787.050320352689596,
215797.126710240147077,
215807.203142187645426,
215817.279616193496622,
215827.356132255896227,
215837.432690373098012,
215847.509290543355746,
215857.585932764835889,
215867.662617035850417,
215877.739343354594894,
215887.816111719293986,
215897.892922128143255,
215907.969774579425575,
215918.046669071365613,
215928.123605602188036,
215938.200584170117509,
215948.277604773407802,
215958.354667410341790,
215968.431772079027724,
215978.508918777777581,
215988.586107504786924,
215998.663338258338626,
216008.740611036657356,
216018.817925837967778,
216028.895282660436351,
216038.972681502433261,
216049.050122362095863,
216059.127605237677926,
216069.205130127404118,
216079.282697029586416,
216089.360305942333071,
216099.437956863956060,
216109.515649792680051,
216119.593384726729710,
216129.671161664387910,
216139.748980603821110,
216149.826841543312185,
216159.904744481085800,
216169.982689415337518,
216180.060676344321109,
216190.138705266348552,
216200.216776179586304,
216210.294889082288137,
216220.373043972707819,
216230.451240848982707,
216240.529479709482985,
216250.607760552404216,
216260.686083376000170,
216270.764448178408202,
216280.842854957998497,
216290.921303712937515,
216300.999794441420818,
216311.078327141818590,
216321.156901812239084,
216331.235518450965174,
216341.314177056308836,
216351.392877626378322,
216361.471620159514714,
216371.550404653913574,
216381.629231107799569,
216391.708099519455573,
216401.787009887106251,
216411.865962208918063,
216421.944956483261194,
216432.023992708331207,
216442.103070882323664,
216452.182191003492335,
216462.261353070090991,
216472.340557080373401,
216482.419803032564232,
216492.499090924917255,
216502.578420755628031,
216512.657792523008538,
216522.737206225254340,
216532.816661860590102,
216542.896159427327802,
216552.975698923633900,
216563.055280347820371,
216573.134903698053677,
216583.214568972616689,
216593.294276169763179,
216603.374025287717814,
216613.453816324705258,
216623.533649279008387,
216633.613524148880970,
216643.693440932489466,
216653.773399628145853,
216663.853400234074797,
216673.933442748500966,
216684.013527169678127,
216694.093653495860053,
216704.173821725300513,
216714.254031856224174,
216724.334283886913909,
216734.414577815536177,
216744.494913640402956,
216754.575291359738912,
216764.655710971768713,
216774.736172474804334,
216784.816675867012236,
216794.897221146646189,
216804.977808312018169,
216815.058437361294637,
216825.139108292787569,
216835.219821104663424,
216845.300575795263285,
216855.381372362724505,
216865.462210805417271,
216875.543091121508041,
216885.624013309279690,
216895.704977366956882,
216905.785983292764286,
216915.867031085013878,
216925.948120741872117,
216936.029252261650981,
216946.110425642575137,
216956.191640882898355,
216966.272897980845300,
216976.354196934727952,
216986.435537742712768,
216996.516920403082622,
217006.598344914091285,
217016.679811274021631,
217026.761319481040118,
217036.842869533458725,
217046.924461429502117,
217057.006095167394960,
217067.087770745449234,
217077.169488161889603,
217087.251247414911631,
217097.333048502885504,
217107.414891423919471,
217117.496776176354615,
217127.578702758415602,
217137.660671168385306,
217147.742681404459290,
217157.824733464891324,
217167.906827347993385,
217177.988963051961036,
217188.071140575048048,
217198.153359915508190,
217208.235621071624337,
217218.317924041650258,
217228.400268823781516,
217238.482655416300986,
217248.565083817462437,
217258.647554025548743,
217268.730066038755467,
217278.812619855365483,
217288.895215473603457,
217298.977852891810471,
217309.060532108094776,
217319.143253120855661,
217329.226015928230481,
217339.308820528560318,
217349.391666920011630,
217359.474555100954603,
217369.557485069497488,
217379.640456824039575,
217389.723470362689113,
217399.806525683845393,
217409.889622785674874,
217419.972761666431325,
217430.055942324397620,
217440.139164757885737,
217450.222428965003928,
217460.305734944122378,
217470.389082693465753,
217480.472472211258719,
217490.555903495784150,
217500.639376545324922,
217510.722891358134802,
217520.806447932380252,
217530.890046266431455,
217540.973686358454870,
217551.057368206791580,
217561.141091809608042,
217571.224857165245339,
217581.308664271899033,
217591.392513127851998,
217601.476403731358005,
217611.560336080641719,
217621.644310174044222,
217631.728326009761076,
217641.812383586046053,
217651.896482901182026,
217661.980623953422764,
217672.064806741022039,
217682.149031262204517,
217692.233297515282175,
217702.317605498450575,
217712.401955210050801,
217722.486346648307517,
217732.570779811445391,
217742.655254697776400,
217752.739771305525210,
217762.824329632945592,
217772.908929678349523,
217782.993571439903462,
217793.078254915948492,
217803.162980104709277,
217813.247747004439589,
217823.332555613451404,
217833.417405929969391,
217843.502297952218214,
217853.587231678509852,
217863.672207107098075,
217873.757224236236652,
217883.842283064179355,
217893.927383589150850,
217904.012525809521321,
217914.097709723457228,
217924.182935329270549,
217934.268202625127742,
217944.353511609428097,
217954.438862280367175,
217964.524254636198748,
217974.609688675176585,
217984.695164395641768,
217994.780681795731653,
218004.866240873845527,
218014.951841628149850,
218025.037484056927497,
218035.123168158432236,
218045.208893930976046,
218055.294661372725386,
218065.380470482079545,
218075.466321257204982,
218085.552213696413673,
218095.638147797901183,
218105.724123560008593,
218115.810140980989672,
218125.896200059069088,
218135.982300792558817,
218146.068443179683527,
218156.154627218696987,
218166.240852907911176,
218176.327120245550759,
218186.413429229956819,
218196.499779859266710,
218206.586172131850617,
218216.672606045962311,
218226.759081599855563,
218236.845598791755037,
218246.932157620001817,
218257.018758082791464,
218267.105400178406853,
218277.192083905130858,
218287.278809261275455,
218297.365576245007105,
218307.452384854725096,
218317.539235088566784,
218327.626126944815041,
218337.713060421810951,
218347.800035517779179,
218357.887052230973495,
218367.974110559705878,
218378.061210502230097,
218388.148352056799922,
218398.235535221698228,
218408.322759995120578,
218418.410026375495363,
218428.497334360930836,
218438.584683949797181,
218448.672075140319066,
218458.759507930721156,
218468.846982319373637,
218478.934498304501176,
218489.022055884357542,
218499.109655057254713,
218509.197295821388252,
218519.284978175099241,
218529.372702116612345,
218539.460467644297751,
218549.548274756292813,
218559.636123450909508,
218569.724013726459816,
218579.811945581168402,
218589.899919013289036,
218599.987934021191904,
218610.075990603072569,
218620.164088757184800,
218630.252228481840575,
218640.340409775322769,
218650.428632635856047,
218660.516897061810596,
218670.605203051294666,
218680.693550602736650,
218690.781939714332111,
218700.870370384363923,
218710.958842611085856,
218721.047356392868096,
218731.135911727877101,
218741.224508614395745,
218751.313147050765110,
218761.401827035180759,
218771.490548565954668,
218781.579311641340610,
218791.668116259650560,
218801.756962419167394,
218811.845850118057569,
218821.934779354749480,
218832.023750127409585,
218842.112762434349861,
218852.201816273853183,
218862.290911644144217,
218872.380048543564044,
218882.469226970395539,
218892.558446922863368,
218902.647708399250405,
218912.737011397810420,
218922.826355916884495,
218932.915741954697296,
218943.005169509560801,
218953.094638579728780,
218963.184149163484108,
218973.273701259051450,
218983.363294864800991,
218993.452929978986504,
219003.542606599803548,
219013.632324725651415,
219023.722084354725666,
219033.811885485338280,
219043.901728115713922,
219053.991612244222779,
219064.081537869089516,
219074.171504988567904,
219084.261513600940816,
219094.351563704520231,
219104.441655297589023,
219114.531788378400961,
219124.621962945238920,
219134.712178996385774,
219144.802436530095292,
219154.892735544737661,
219164.983076038479339,
219175.073458009632304,
219185.163881456508534,
219195.254346377390902,
219205.344852770562284,
219215.435400634189136,
219225.525989966728957,
219235.616620766319102,
219245.707293031300651,
219255.798006759956479,
219265.888761950598564,
219275.979558601422468,
219286.070396710769273,
219296.161276276921853,
219306.252197298133979,
219316.343159772688523,
219326.434163698926568,
219336.525209075072780,
219346.616295899380930,
219356.707424170162994,
219366.798593885760056,
219376.889805044367677,
219386.981057644326938,
219397.072351683891611,
219407.163687161315465,
219417.255064074968686,
219427.346482423017733,
219437.437942203890998,
219447.529443415784044,
219457.620986056950642,
219467.712570125702769,
219477.804195620381506,
219487.895862539153313,
219497.987570880446583,
219508.079320642456878,
219518.171111823408864,
219528.262944421760039,
219538.354818435647758,
219548.446733863413101,
219558.538690703338943,
219568.630688953679055,
219578.722728612774517,
219588.814809678849997,
219598.906932150217472,
219608.999096025218023,
219619.091301302047214,
219629.183547979017021,
219639.275836054439424,
219649.368165526597295,
219659.460536393773509,
219669.552948654250940,
219679.645402306283358,
219689.737897348211845,
219699.830433778290171,
219709.923011594801210,
219720.015630796056939,
219730.108291380311130,
219740.200993345933966,
219750.293736691121012,
219760.386521414155141,
219770.479347513435641,
219780.572214987158077,
219790.665123833576217,
219800.758074051031144,
219810.851065637863940,
219820.944098592270166,
219831.037172912590904,
219841.130288597109029,
219851.223445644078311,
219861.316644051839830,
219871.409883818676462,
219881.503164942841977,
219891.596487422677455,
219901.689851256378461,
219911.783256442315178,
219921.876702978799585,
219931.970190864056349,
219942.063720096397446,
219952.157290674105752,
219962.250902595493244,
219972.344555858871900,
219982.438250462495489,
219992.531986404646887,
220002.625763683608966,
220012.719582297722809,
220022.813442245213082,
220032.907343524420867,
220043.001286133687245,
220053.095270071207779,
220063.189295335323550,
220073.283361924317433,
220083.377469836414093,
220093.471619070012821,
220103.565809623396490,
220113.660041494789766,
220123.754314682533732,
220133.848629184882157,
220143.942985000205226,
220154.037382126727607,
220164.131820562761277,
220174.226300306618214,
220184.320821356523084,
220194.415383710846072,
220204.509987367840949,
220214.604632325819694,
220224.699318583065178,
220234.794046137918485,
220244.888814988633385,
220254.983625133492751,
220265.078476570837665,
220275.173369298892794,
220285.268303315970115,
220295.363278620410711,
220305.458295210526558,
220315.553353084513219,
220325.648452240799088,
220335.743592677550623,
220345.838774393138010,
220355.933997385785915,
220366.029261653893627,
220376.124567195714917,
220386.219914009503555,
220396.315302093629725,
220406.410731446376303,
220416.506202065967955,
220426.601713950745761,
220436.697267099021701,
220446.792861509049544,
220456.888497179199476,
220466.984174107725266,
220477.079892292938894,
220487.175651733123232,
220497.271452426590258,
220507.367294371622847,
220517.463177566532977,
220527.559102009603521,
220537.655067699117353,
220547.751074633444659,
220557.847122810810106,
220567.943212229554774,
220578.039342887961539,
220588.135514784313273,
220598.231727916951058,
220608.327982284157770,
220618.424277884187177,
220628.520614715409465,
220638.616992776078405,
220648.713412064535078,
220658.809872579033254,
220668.906374317884911,
220679.002917279460235,
220689.099501461925684,
220699.196126863680547,
220709.292793483036803,
220719.389501318248222,
220729.486250367626781,
220739.583040629426250,
220749.679872102045920,
220759.776744783710456,
220769.873658672760939,
220779.970613767538453,
220790.067610066267662,
220800.164647567289649,
220810.261726268858183,
220820.358846169314347,
220830.456007266999222,
220840.553209560137475,
220850.650453047099290,
220860.747737726167543,
220870.845063595625106,
220880.942430653813062,
220891.039838898985181,
220901.137288329424337,
220911.234778943529818,
220921.332310739555396,
220931.429883715842152,
220941.527497870585648,
220951.625153202214278,
220961.722849708981812,
220971.820587389171124,
220981.918366241123294,
220992.016186263121199,
221002.114047453476815,
221012.211949810473016,
221022.309893332479987,
221032.407878017751500,
221042.505903864541324,
221052.603970871306956,
221062.702079036214855,
221072.800228357635206,
221082.898418833821779,
221092.996650463173864,
221103.094923243887024,
221113.193237174389651,
221123.291592252877308,
221133.389988477720181,
221143.488425847201142,
221153.586904359632172,
221163.685424013296142,
221173.783984806563240,
221183.882586737687234,
221193.981229804980103,
221204.079914006841136,
221214.178639341407688,
221224.277405807108153,
221234.376213402225403,
221244.475062125042314,
221254.573951973929070,
221264.672882947139442,
221274.771855042985408,
221284.870868259808049,
221294.969922595890239,
221305.069018049543956,
221315.168154619052075,
221325.267332302813884,
221335.366551099083154,
221345.465811006113654,
221355.565112022275571,
221365.664454145909986,
221375.763837375270668,
221385.863261708669597,
221395.962727144447854,
221406.062233680888312,
221416.161781316332053,
221426.261370049091056,
221436.360999877419090,
221446.460670799686341,
221456.560382814233890,
221466.660135919228196,
221476.759930113155860,
221486.859765394212445,
221496.959641760709928,
221507.059559211076703,
221517.159517743508331,
221527.259517356345896,
221537.359558047930477,
221547.459639816544950,
221557.559762660501292,
221567.659926578140585,
221577.760131567803910,
221587.860377627686830,
221597.960664756159531,
221608.060992951621301,
221618.161362212267704,
221628.261772536468925,
221638.362223922566045,
221648.462716368783731,
221658.563249873492168,
221668.663824435061542,
221678.764440051687416,
221688.865096721769078,
221698.965794443589402,
221709.066533215489471,
221719.167313035752159,
221729.268133902689442,
221739.368995814613299,
221749.469898769952124,
221759.570842766872374,
221769.671827803715132,
221779.772853878821479,
221789.873920990561601,
221799.975029137160163,
221810.076178317016456,
221820.177368528384250,
221830.278599769575521,
221840.379872038960457,
221850.481185334821930,
221860.582539655530127,
221870.683934999338817,
221880.785371364501771,
221890.886848749476485,
221900.988367152487626,
221911.089926571905380,
221921.191527006070828,
221931.293168453208636,
221941.394850911688991,
221951.496574379823869,
221961.598338855925249,
221971.700144338392420,
221981.801990825420944,
221991.903878315381007,
222002.005806806555483,
222012.107776297343662,
222022.209786786028417,
222032.311838270892622,
222042.413930750306463,
222052.516064222552814,
222062.618238685972756,
222072.720454138849163,
222082.822710579610430,
222092.925008006423013,
222103.027346417686203,
222113.129725811741082,
222123.232146186870523,
222133.334607541444711,
222143.437109873717418,
222153.539653182029724,
222163.642237464722712,
222173.744862720079254,
222183.847528946498642,
222193.950236142205540,
222204.052984305628343,
222214.155773434962612,
222224.258603528607637,
222234.361474584933603,
222244.464386602165177,
222254.567339578643441,
222264.670333512709476,
222274.773368402733468,
222284.876444246998290,
222294.979561043757712,
222305.082718791411025,
222315.185917488270206,
222325.289157132676337,
222335.392437722941395,
222345.495759257348254,
222355.599121734237997,
222365.702525151951704,
222375.805969508830458,
222385.909454803186236,
222396.012981033301912,
222406.116548197547672,
222416.220156294235494,
222426.323805321706459,
222436.427495278272545,
222446.531226162245730,
222456.634997971937992,
222466.738810705719516,
222476.842664361844072,
222486.946558938740054,
222497.050494434661232,
222507.154470847919583,
222517.258488176914398,
222527.362546419899445,
222537.466645575244911,
222547.570785641291877,
222557.674966616294114,
222567.779188498621807,
222577.883451286616037,
222587.987754978588782,
222598.092099572881125,
222608.196485067746835,
222618.300911461614305,
222628.405378752766410,
222638.509886939544231,
222648.614436020230642,
222658.719025993224932,
222668.823656856780872,
222678.928328609297751,
222689.033041249029338,
222699.137794774374925,
222709.242589183588279,
222719.347424475068692,
222729.452300647128141,
222739.557217698078603,
222749.662175626232056,
222759.767174429929582,
222769.872214107570471,
222779.977294657379389,
222790.082416077755624,
222800.187578366982052,
222810.292781523428857,
222820.398025545408018,
222830.503310431231512,
222840.608636179269524,
222850.714002787775826,
222860.819410255178809,
222870.924858579790452,
222881.030347759864526,
222891.135877793829422,
222901.241448679909809,
222911.347060416592285,
222921.452713002043311,
222931.558406434662174,
222941.664140712789958,
222951.769915834767744,
222961.875731798907509,
222971.981588603521232,
222982.087486246979097,
222992.193424727622187,
223002.299404043733375,
223012.405424193711951,
223022.511485175811686,
223032.617586988402763,
223042.723729629797162,
223052.829913098394172,
223062.936137392476667,
223073.042402510385728,
223083.148708450404229,
223093.255055210960563,
223103.361442790337605,
223113.467871186847333,
223123.574340398889035,
223133.680850424774690,
223143.787401262758067,
223153.893992911296664,
223164.000625368615147,
223174.107298633112805,
223184.214012703159824,
223194.320767576980870,
223204.427563253004337,
223214.534399729513098,
223224.641277004877338,
223234.748195077467244,
223244.855153945478378,
223254.962153607397340,
223265.069194061477901,
223275.176275306061143,
223285.283397339546354,
223295.390560160187306,
223305.497763766412390,
223315.605008156417171,
223325.712293328659143,
223335.819619281450287,
223345.926986013131682,
223356.034393522044411,
223366.141841806442244,
223376.249330864811782,
223386.356860695319483,
223396.464431296481052,
223406.572042666433845,
223416.679694803693565,
223426.787387706513982,
223436.895121373294387,
223447.002895802288549,
223457.110710991866654,
223467.218566940398887,
223477.326463646226330,
223487.434401107602753,
223497.542379322956549,
223507.650398290570593,
223517.758458008873276,
223527.866558476089267,
223537.974699690617854,
223548.082881650771014,
223558.191104354977142,
223568.299367801490007,
223578.407671988621587,
223588.516016914800275,
223598.624402578279842,
223608.732828977459576,
223618.841296110738767,
223628.949803976283874,
223639.058352572639706,
223649.166941897972720,
223659.275571950682206,
223669.384242729138350,
223679.492954231711337,
223689.601706456684042,
223699.710499402397545,
223709.819333067192929,
223719.928207449469483,
223730.037122547510080,
223740.146078359655803,
223750.255074884305941,
223760.364112119801575,
223770.473190064396476,
223780.582308716460830,
223790.691468074423028,
223800.800668136536842,
223810.909908901201561,
223821.019190366700059,
223831.128512531431625,
223841.237875393708237,
223851.347278951900080,
223861.456723204319132,
223871.566208149335580,
223881.675733785290504,
223891.785300110495882,
223901.894907123351004,
223912.004554822167847,
223922.114243205287494,
223932.223972271051025,
223942.333742017828627,
223952.443552443990484,
223962.553403547761263,
223972.663295327598462,
223982.773227781814057,
223992.883200908807339,
224002.993214706832077,
224013.103269174287561,
224023.213364309456665,
224033.323500110796886,
224043.433676576561993,
224053.543893705151277,
224063.654151494876714,
224073.764449944108492,
224083.874789051187690,
224093.985168814455392,
224104.095589232281782,
224114.206050302920630,
224124.316552024858538,
224134.427094396378379,
224144.537677415821236,
224154.648301081499085,
224164.758965391840320,
224174.869670345127815,
224184.980415939760860,
224195.091202174051432,
224205.202029046311509,
224215.312896554998588,
224225.423804698395543,
224235.534753474814352,
224245.645742882683408,
224255.756772920343792,
224265.867843586078379,
224275.978954878257355,
224286.090106795250904,
224296.201299335400108,
224306.312532497075154,
224316.423806278617121,
224326.535120678367093,
224336.646475694666151,
224346.757871325855376,
224356.869307570304954,
224366.980784426414175,
224377.092301892436808,
224387.203859966801247,
224397.315458647819469,
224407.427097933861660,
224417.538777823239798,
224427.650498314353172,
224437.762259405542864,
224447.874061095149955,
224457.985903381515527,
224468.097786263009766,
224478.209709738002857,
224488.321673804806778,
224498.433678461791715,
224508.545723707356956,
224518.657809539727168,
224528.769935957359849,
224538.882102958596079,
224548.994310541835148,
224559.106558705301723,
224569.218847447424196,
224579.331176766543649,
224589.443546661059372,
224599.555957129254239,
224609.668408169498434,
224619.780899780249456,
224629.893431959731970,
224640.006004706316162,
224650.118618018430425,
224660.231271894299425,
224670.343966332409764,
224680.456701331044314,
224690.569476888660574,
224700.682293003454106,
224710.795149673882406,
224720.908046898286557,
224731.020984675007639,
224741.133963002386736,
224751.246981878794031,
224761.360041302657919,
224771.473141272203065,
224781.586281785857864,
224791.699462842021603,
224801.812684438977158,
224811.925946575036505,
224822.039249248686247,
224832.152592458209256,
224842.265976201975718,
224852.379400478326716,
224862.492865285690641,
224872.606370622321265,
224882.719916486559669,
224892.833502876950661,
224902.947129791660700,
224913.060797229089076,
224923.174505187664181,
224933.288253665697994,
224943.402042661531596,
224953.515872173535172,
224963.629742200108012,
224973.743652739503887,
224983.857603790238500,
224993.971595350507414,
225004.085627418768127,
225014.199699993419927,
225024.313813072687481,
225034.427966655057389,
225044.542160738754319,
225054.656395322264871,
225064.770670403930126,
225074.884985982062062,
225084.999342055001762,
225095.113738621206721,
225105.228175678959815,
225115.342653226602124,
225125.457171262620250,
225135.571729785209754,
225145.686328792828135,
225155.800968283816474,
225165.915648256603163,
225176.030368709383765,
225186.145129640703090,
225196.259931048756698,
225206.374772932031192,
225216.489655288838549,
225226.604578117578058,
225236.719541416590801,
225246.834545184188755,
225256.949589418800315,
225267.064674118737457,
225277.179799282399472,
225287.294964908156544,
225297.410170994320652,
225307.525417539261980,
225317.640704541438026,
225327.756031999160768,
225337.871399910683976,
225347.986808274523355,
225358.102257088990882,
225368.217746352398535,
225378.333276063174708,
225388.448846219689585,
225398.564456820226042,
225408.680107863241574,
225418.795799347077264,
225428.911531270074192,
225439.027303630544338,
225449.143116426974302,
225459.258969657676062,
225469.374863320932491,
225479.490797415259294,
225489.606771938910242,
225499.722786890284624,
225509.838842267781729,
225519.954938069713535,
225530.071074294508435,
225540.187250940391095,
225550.303468005935429,
225560.419725489336997,
225570.536023389082402,
225580.652361703454517,
225590.768740430794423,
225600.885159569559619,
225611.001619118091185,
225621.118119074759306,
225631.234659437905066,
225641.351240205927752,
225651.467861377110239,
225661.584522949939128,
225671.701224922697293,
225681.817967293813126,
225691.934750061569503,
225702.051573224482127,
225712.168436780746561,
225722.285340728820302,
225732.402285067131743,
225742.519269793876447,
225752.636294907599222,
225762.753360406582942,
225772.870466289226897,
225782.987612553843064,
225793.104799198888941,
225803.222026222647401,
225813.339293623546837,
225823.456601399957435,
225833.573949550220277,
225843.691338072676444,
225853.808766965783434,
225863.926236227824120,
225874.043745857226895,
225884.161295852361945,
225894.278886211512145,
225904.396516933193197,
225914.514188015687978,
225924.631899457366671,
225934.749651256570360,
225944.867443411785644,
225954.985275921266293,
225965.103148783440702,
225975.221061996679055,
225985.339015559293330,
225995.457009469741024,
226005.575043726421427,
226015.693118327588309,
226025.811233271640958,
226035.929388557007769,
226046.047584182058927,
226056.165820145106409,
226066.284096444578608,
226076.402413078845711,
226086.520770046219695,
226096.639167345158057,
226106.757604973972775,
226116.876082931034034,
226126.994601214828435,
226137.113159823551541,
226147.231758755689953,
226157.350398009642959,
226167.469077583693434,
226177.587797476240667,
226187.706557685771259,
226197.825358210480772,
226207.944199048855808,
226218.063080199266551,
226228.182001659995876,
226238.300963429588592,
226248.419965506269364,
226258.539007888466585,
226268.658090574579546,
226278.777213562920224,
226288.896376851917012,
226299.015580439969199,
226309.134824325359659,
226319.254108506545890,
226329.373432981898077,
226339.492797749728197,
226349.612202808493748,
226359.731648156535812,
226369.851133792224573,
226379.970659713901114,
226390.090225920052035,
226400.209832408960210,
226410.329479179083137,
226420.449166228645481,
226430.568893556192052,
226440.688661160063930,
226450.808469038514886,
226460.928317190089729,
226471.048205613071332,
226481.168134305888088,
226491.288103266822873,
226501.408112494391389,
226511.528161986876512,
226521.648251742677530,
226531.768381760193733,
226541.888552037766203,
226552.008762573823333,
226562.129013366706204,
226572.249304414814105,
226582.369635716517223,
226592.490007270185743,
226602.610419074218953,
226612.730871127016144,
226622.851363426860189,
226632.971895972266793,
226643.092468761547934,
226653.213081793073798,
226663.333735065214569,
226673.454428576427745,
226683.575162324996199,
226693.695936309377430,
226703.816750527912518,
226713.937604978942545,
226724.058499660954112,
226734.179434572288301,
226744.300409711257089,
226754.421425076317973,
226764.542480665841140,
226774.663576478225878,
226784.784712511755060,
226794.905888764944393,
226805.027105236105854,
226815.148361923580524,
226825.269658825825900,
226835.390995941270376,
226845.512373268138617,
226855.633790804946329,
226865.755248550034594,
226875.876746501773596,
226885.998284658533521,
226896.119863018742763,
226906.241481580771506,
226916.363140342989936,
226926.484839303826448,
226936.606578461563913,
226946.728357814718038,
226956.850177361542592,
226966.972037100524176,
226977.093937030003872,
226987.215877148351865,
226997.337857453996548,
227007.459877945308108,
227017.581938620627625,
227027.704039478383493,
227037.826180516945897,
227047.948361734772334,
227058.070583130087471,
227068.192844701436115,
227078.315146447159350,
227088.437488365540048,
227098.559870455123018,
227108.682292714191135,
227118.804755141172791,
227128.927257734467275,
227139.049800492386566,
227149.172383413388161,
227159.295006495842244,
227169.417669738089899,
227179.540373138675932,
227189.663116695766803,
227199.785900407878216,
227209.908724273409462,
227220.031588290672516,
227230.154492458095774,
227240.277436774136731,
227250.400421237049159,
227260.523445845290553,
227270.646510597289307,
227280.769615491299191,
227290.892760525894118,
227301.015945699327858,
227311.139171010087011,
227321.262436456425348,
227331.385742036858574,
227341.509087749756873,
227351.632473593403120,
227361.755899566342123,
227371.879365666827653,
227382.002871893346310,
227392.126418244210072,
227402.250004717876436,
227412.373631312686484,
227422.497298027097713,
227432.621004859392997,
227442.744751808088040,
227452.868538871523924,
227462.992366047983523,
227473.116233336040750,
227483.240140733949374,
227493.364088240225101,
227503.488075853092596,
227513.612103571067564,
227523.736171392520191,
227533.860279315791558,
227543.984427339397371,
227554.108615461562295,
227564.232843680802034,
227574.357111995457672,
227584.481420403957600,
227594.605768904672004,
227604.730157495941967,
227614.854586176224984,
227624.979054943949450,
227635.103563797398238,
227645.228112735057948,
227655.352701755240560,
227665.477330856432673,
227675.602000036975369,
227685.726709295267938,
227695.851458629709668,
227705.976248038670747,
227716.101077520579565,
227726.225947073835414,
227736.350856696750270,
227746.475806387839839,
227756.600796145416098,
227766.725825967849232,
227776.850895853684051,
227786.976005801116116,
227797.101155808719341,
227807.226345874776598,
227817.351575997687178,
227827.476846175937681,
227837.602156407781877,
227847.727506691735471,
227857.852897026168648,
227867.978327409393387,
227878.103797839954495,
227888.229308316163952,
227898.354858836391941,
227908.480449399037752,
227918.606080002587987,
227928.731750645296415,
227938.857461325707845,
227948.983212042163359,
227959.109002792974934,
227969.234833576658275,
227979.360704391583567,
227989.486615236091893,
227999.612566108640749,
228009.738557007600321,
228019.864587931399001,
228029.990658878377872,
228040.116769846965326,
228050.242920835589757,
228060.369111842592247,
228070.495342866459396,
228080.621613905474078,
228090.747924958151998,
228100.874276022776030,
228111.000667097774567,
228121.127098181634210,
228131.253569272666937,
228141.380080369359348,
228151.506631469994318,
228161.633222573029343,
228171.759853676921921,
228181.886524779954925,
228192.013235880614957,
228202.139986977272201,
228212.266778068325948,
228222.393609152146382,
228232.520480227191001,
228242.647391291888198,
228252.774342344520846,
228262.901333383604651,
228273.028364407451591,
228283.155435414577369,
228293.282546403235756,
228303.409697371942457,
228313.536888319038553,
228323.664119242952438,
228333.791390142112505,
228343.918701014888939,
228354.046051859651925,
228364.173442674829857,
228374.300873458822025,
228384.428344210115029,
228394.555854926962638,
228404.683405607880559,
228414.810996251238976,
228424.938626855408074,
228435.066297418845352,
228445.194007939920994,
228455.321758417063393,
228465.449548848613631,
228475.577379233029205,
228485.705249568738509,
228495.833159854053520,
228505.961110087460838,
228516.089100267359754,
228526.217130392120453,
228536.345200460142223,
228546.473310469940770,
228556.601460419711657,
228566.729650308057899,
228576.857880133233266,
228586.986149893753463,
228597.114459588017780,
228607.242809214338195,
228617.371198771201307,
228627.499628257006407,
228637.628097670152783,
228647.756607008981518,
228657.885156271979213,
228668.013745457516052,
228678.142374564049533,
228688.271043589949841,
228698.399752533587161,
228708.528501393418992,
228718.657290167815518,
228728.786118855176028,
228738.914987453987123,
228749.043895962560782,
228759.172844379383605,
228769.301832702796673,
228779.430860931286588,
228789.559929063165328,
228799.689037096890388,
228809.818185030831955,
228819.947372863476630,
228830.076600593165495,
228840.205868218356045,
228850.335175737418467,
228860.464523148752050,
228870.593910450814292,
228880.723337641917169,
228890.852804720605491,
228900.982311685191235,
228911.111858534131898,
228921.241445265826769,
228931.371071878646035,
228941.500738371047191,
228951.630444741429528,
228961.760190988192335,
228971.889977109705796,
228982.019803104485618,
228992.149668970785569,
229002.279574707179563,
229012.409520312008681,
229022.539505783643108,
229032.669531120569445,
229042.799596321186982,
229052.929701383836800,
229063.059846306976397,
229073.190031089063268,
229083.320255728409393,
229093.450520223472267,
229103.580824572680285,
229113.711168774432736,
229123.841552827158011,
229133.971976729284506,
229144.102440479153302,
229154.232944075192790,
229164.363487515889574,
229174.494070799555629,
229184.624693924677558,
229194.755356889625546,
229204.886059692886192,
229215.016802332742373,
229225.147584807738895,
229235.278407116216840,
229245.409269256575499,
229255.540171227301471,
229265.671113026706735,
229275.802094653306995,
229285.933116105443332,
229296.064177381573245,
229306.195278480096022,
229316.326419399469160,
229326.457600137975533,
229336.588820694130845,
229346.720081066421699,
229356.851381253101863,
229366.982721252687043,
229377.114101063518319,
229387.245520684140502,
229397.376980112807360,
229407.508479348092806,
229417.640018388279714,
229427.771597231854685,
229437.903215877246112,
229448.034874322795076,
229458.166572566988179,
229468.298310608195607,
229478.430088444874855,
229488.561906075396109,
229498.693763498216867,
229508.825660711736418,
229518.957597714383155,
229529.089574504556367,
229539.221591080713551,
229549.353647441166686,
229559.485743584460579,
229569.617879508936312,
229579.750055213051382,
229589.882270695175976,
229600.014525953796692,
229610.146820987283718,
229620.279155794036342,
229630.411530372541165,
229640.543944721139269,
229650.676398838259047,
229660.808892722387100,
229670.941426371922716,
229681.073999785177875,
229691.206612960726488,
229701.339265896880534,
229711.471958592126612,
229721.604691044834908,
229731.737463253462920,
229741.870275216351729,
229752.003126932046143,
229762.136018398858141,
229772.268949615245219,
229782.401920579635771,
229792.534931290458189,
229802.667981746082660,
229812.801071945053991,
229822.934201885625953,
229833.067371566343354,
229843.200580985518172,
229853.333830141637009,
229863.467119033186464,
229873.600447658449411,
229883.733816015970660,
229893.867224104091292,
229904.000671921297908,
229914.134159465931589,
229924.267686736508040,
229934.401253731339239,
229944.534860448940890,
229954.668506887624972,
229964.802193045994500,
229974.935918922274141,
229985.069684515066911,
229995.203489822655683,
230005.337334843527060,
230015.471219576051226,
230025.605144018772990,
230035.739108169975225,
230045.873112028173637,
230056.007155591709306,
230066.141238859097939,
230076.275361828709720,
230086.409524499002146,
230096.543726868316298,
230106.677968935196986,
230116.812250697956188,
230126.946572155109607,
230137.080933305027429,
230147.215334146138048,
230157.349774676869856,
230167.484254895651247,
230177.618774800939718,
230187.753334391076351,
230197.887933664605953,
230208.022572619840503,
230218.157251255295705,
230228.291969569283538,
230238.426727560348809,
230248.561525226890808,
230258.696362567308825,
230268.831239579973044,
230278.966156263428275,
230289.101112616015598,
230299.236108636221616,
230309.371144322387408,
230319.506219673057785,
230329.641334686515620,
230339.776489361334825,
230349.911683695827378,
230360.046917688508984,
230370.182191337720724,
230380.317504641949199,
230390.452857599622803,
230400.588250209140824,
230410.723682468931656,
230420.859154377481900,
230430.994665933132637,
230441.130217134341365,
230451.265807979536476,
230461.401438467233675,
230471.537108595686732,
230481.672818363469560,
230491.808567768981447,
230501.944356810621684,
230512.080185486818664,
230522.216053796029883,
230532.351961736654630,
230542.487909307150403,
230552.623896505945595,
230562.759923331439495,
230572.895989782031393,
230583.032095856266096,
230593.168241552455584,
230603.304426869144663,
230613.440651804645313,
230623.576916357473237,
230633.713220526027726,
230643.849564308737172,
230653.985947704000864,
230664.122370710334508,
230674.258833326050080,
230684.395335549750598,
230694.531877379718935,
230704.668458814383484,
230714.805079852259951,
230724.941740491747623,
230735.078440731303999,
230745.215180569270160,
230755.351960004220018,
230765.488779034436448,
230775.625637658435153,
230785.762535874673631,
230795.899473681522068,
230806.036451077408856,
230816.173468060791492,
230826.310524630098371,
230836.447620783845196,
230846.584756520285737,
230856.721931838052114,
230866.859146735427203,
230876.996401210926706,
230887.133695262920810,
230897.271028889954323,
230907.408402090310119,
230917.545814862533007,
230927.683267204993172,
230937.820759116206318,
230947.958290594484424,
230958.095861638372298,
230968.233472246269230,
230978.371122416632716,
230988.508812147803837,
230998.646541438356508,
231008.784310286602704,
231018.922118691029027,
231029.059966650092974,
231039.197854162193835,
231049.335781225789106,
231059.473747839307180,
231069.611754001205554,
231079.749799709854415,
231089.887884963769466,
231100.026009761291789,
231110.164174100966193,
231120.302377981191967,
231130.440621400368400,
231140.578904356952989,
231150.717226849403232,
231160.855588876089314,
231170.993990435585147,
231181.132431526144501,
231191.270912146370392,
231201.409432294633007,
231211.547991969360737,
231221.686591168981977,
231231.825229891954223,
231241.963908136676764,
231252.102625901694410,
231262.241383185319137,
231272.380179986066651,
231282.519016302394448,
231292.657892132643610,
231302.796807475329842,
231312.935762328852434,
231323.074756691727089,
231333.213790562236682,
231343.352863939042436,
231353.491976820369018,
231363.631129204761237,
231373.770321090618381,
231383.909552476514364,
231394.048823360644747,
231404.188133741670754,
231414.327483617962571,
231424.466872987890383,
231434.606301849998999,
231444.745770202600397,
231454.885278044268489,
231465.024825373431668,
231475.164412188431015,
231485.304038487753132,
231495.443704269855516,
231505.583409533195663,
231515.723154276231071,
231525.862938497273717,
231536.002762194926618,
231546.142625367530854,
231556.282528013602132,
231566.422470131510636,
231576.562451719713863,
231586.702472776727518,
231596.842533300834475,
231606.982633290695958,
231617.122772744565737,
231627.262951660959516,
231637.403170038305689,
231647.543427875061752,
231657.683725169656100,
231667.824061920604436,
231677.964438126218738,
231688.104853785072919,
231698.245308895508060,
231708.385803456010763,
231718.526337465038523,
231728.666910920990631,
231738.807523822353687,
231748.948176167585189,
231759.088867955084424,
231769.229599183279788,
231779.370369850716088,
231789.511179955763509,
231799.652029496821342,
231809.792918472405290,
231819.933846880943747,
231830.074814720894210,
231840.215821990685072,
231850.356868688802933,
231860.497954813617980,
231870.639080363587709,
231880.780245337227825,
231890.921449732966721,
231901.062693549145479,
231911.203976784338010,
231921.345299436972709,
231931.486661505361553,
231941.628062988165766,
231951.769503883639118,
231961.910984190355521,
231972.052503906656057,
231982.194063031143742,
231992.335661562101450,
232002.477299498044886,
232012.618976837460650,
232022.760693578718929,
232032.902449720306322,
232043.044245260680327,
232053.186080198269337,
232063.327954531501746,
232073.469868258893257,
232083.611821378872264,
232093.753813889808953,
232103.895845790248131,
232114.037917078589089,
232124.180027753289323,
232134.322177812777227,
232144.464367255568504,
232154.606596080033341,
232164.748864284658339,
232174.891171867930098,
232185.033518828189699,
232195.175905164011056,
232205.318330873706145,
232215.460795955907088,
232225.603300408867653,
232235.745844231219962,
232245.888427421246888,
232256.031049977522343,
232266.173711898474721,
232276.316413182474207,
232286.459153828065610,
232296.601933833677322,
232306.744753197708633,
232316.887611918617040,
232327.030509994947352,
232337.173447425098857,
232347.316424207470845,
232357.459440340579022,
232367.602495822822675,
232377.745590652746614,
232387.888724828662816,
232398.031898349116091,
232408.175111212622141,
232418.318363417434739,
232428.461654962215107,
232438.604985845246119,
232448.748356065130793,
232458.891765620210208,
232469.035214508970967,
232479.178702729957877,
232489.322230281395605,
232499.465797162003582,
232509.609403370035579,
232519.753048904036405,
232529.896733762434451,
232540.040457943745423,
232550.184221446339507,
232560.328024268674199,
232570.471866409294307,
232580.615747866511811,
232590.759668638929725,
232600.903628724860027,
232611.047628122905735,
232621.191666831407929,
232631.335744848824106,
232641.479862173699075,
232651.624018804432126,
232661.768214739451651,
232671.912449977215147,
232682.056724516267423,
232692.201038355007768,
232702.345391491806367,
232712.489783925237134,
232722.634215653728461,
232732.778686675708741,
232742.923196989635471,
232753.067746594024356,
232763.212335487216478,
232773.356963667756645,
232783.501631134102354,
232793.646337884652894,
232803.791083917953074,
232813.935869232431287,
232824.080693826428615,
232834.225557698548073,
232844.370460847188951,
232854.515403270808747,
232864.660384967835853,
232874.805405936815077,
232884.950466176145710,
232895.095565684285248,
232905.240704459691187,
232915.385882500821026,
232925.531099806161365,
232935.676356374169700,
232945.821652203216217,
232955.966987291874830,
232966.112361638486618,
232976.257775241654599,
232986.403228099748958,
232996.548720211227192,
233006.694251574634109,
233016.839822188252583,
233026.985432050714735,
233037.131081160390750,
233047.276769515767228,
233057.422497115330771,
233067.568263957451563,
233077.714070040761726,
233087.859915363456821,
233098.005799924285384,
233108.151723721501185,
233118.297686753649032,
233128.443689019186422,
233138.589730516512645,
233148.735811244201614,
233158.881931200652616,
233169.028090384323150,
233179.174288793670712,
233189.320526427123696,
233199.466803283226909,
233209.613119360408746,
233219.759474657097599,
233229.905869171809172,
233240.052302902942756,
233250.198775849014055,
233260.345288008480566,
233270.491839379741577,
233280.638429961341899,
233290.785059751680819,
233300.931728749244940,
233311.078436952579068,
233321.225184359995183,
233331.371970970038092,
233341.518796781136189,
233351.665661791834282,
233361.812566000589868,
233371.959509405714925,
233382.106492005812470,
233392.253513799281791,
233402.400574784667697,
233412.547674960369477,
233422.694814324844629,
233432.841992876579752,
233442.989210614003241,
233453.136467535689007,
233463.283763640007237,
233473.431098925386323,
233483.578473390371073,
233493.725887033389881,
233503.873339852929348,
233514.020831847505178,
233524.168363015400246,
233534.315933355275774,
233544.463542865501950,
233554.611191544594476,
233564.758879390981747,
233574.906606403092155,
233585.054372579499613,
233595.202177918574307,
233605.350022418802837,
233615.497906078730011,
233625.645828896696912,
233635.793790871248348,
233645.941792000841815,
233656.089832283905707,
233666.237911718955729,
233676.386030304449378,
233686.534188038844150,
233696.682384920655750,
233706.830620948196156,
233716.978896120155696,
233727.127210434788140,
233737.275563890754711,
233747.423956486367388,
233757.572388220200082,
233767.720859090622980,
233777.869369096209994,
233788.017918235360412,
233798.166506506560836,
233808.315133908239659,
233818.463800438970793,
233828.612506097182631,
233838.761250881245360,
233848.910034789761994,
233859.058857821102720,
233869.207719973783242,
233879.356621246319264,
233889.505561637051869,
233899.654541144584073,
233909.803559767286060,
233919.952617503731744,
233930.101714352262206,
233940.250850311480463,
233950.400025379756698,
233960.549239555606619,
233970.698492837516824,
233980.847785223915707,
233990.997116713260766,
234001.146487304067705,
234011.295896994823124,
234021.445345783926314,
234031.594833669922082,
234041.744360651267925,
234051.893926726421341,
234062.043531893810723,
234072.193176151951775,
234082.342859499360202,
234092.492581934435293,
234102.642343455634546,
234112.792144061560975,
234122.941983750555664,
234133.091862521105213,
234143.241780371725326,
234153.391737300873501,
234163.541733307065442,
234173.691768388671335,
234183.841842544235988,
234193.991955772246001,
234204.142108071158873,
234214.292299439373892,
234224.442529875494074,
234234.592799377889605,
234244.743107945105294,
234254.893455575569533,
234265.043842267768923,
234275.194268020190066,
234285.344732831348665,
234295.495236699585803,
234305.645779623475391,
234315.796361601474928,
234325.946982632041909,
234336.097642713721143,
234346.248341844882816,
234356.399080024071736,
234366.549857249774504,
234376.700673520361306,
234386.851528834464261,
234397.002423190453555,
234407.153356586815789,
234417.304329022066668,
234427.455340494663687,
234437.606391003035242,
234447.757480545726139,
234457.908609121135669,
234468.059776727866847,
234478.210983364289859,
234488.362229028891306,
234498.513513720186893,
234508.664837436634116,
234518.816200176690472,
234528.967601938900771,
234539.119042721664300,
234549.270522523438558,
234559.422041342826560,
234569.573599178198492,
234579.725196028040955,
234589.876831890898757,
234600.028506765142083,
234610.180220649373950,
234620.331973542051855,
234630.483765441516880,
234640.635596346430248,
234650.787466255133040,
234660.939375166140962,
234671.091323077969719,
234681.243309989135014,
234691.395335897948826,
234701.547400803072378,
234711.699504702875856,
234721.851647595904069,
234732.003829480614513,
234742.156050355464686,
234752.308310218941187,
234762.460609069530619,
234772.612946905719582,
234782.765323725994676,
234792.917739528813399,
234803.070194312662352,
234813.222688076057239,
234823.375220817426452,
234833.527792535256594,
234843.680403228092473,
234853.833052894362481,
234863.985741532495013,
234874.138469141093083,
234884.291235718555981,
234894.444041263370309,
234904.596885774051771,
234914.749769248999655,
234924.902691686846083,
234935.055653085932136,
234945.208653444773518,
234955.361692761915037,
234965.514771035814192,
234975.667888264899375,
234985.821044447657187,
234995.974239582661539,
235006.127473668282619,
235016.280746703094337,
235026.434058685495984,
235036.587409614061471,
235046.740799487219192,
235056.894228303455748,
235067.047696061257739,
235077.201202759140870,
235087.354748395533534,
235097.508332968893228,
235107.661956477822969,
235117.815618920751149,
235127.969320296106162,
235138.123060602403712,
235148.276839838217711,
235158.430658001918346,
235168.584515091992216,
235178.738411106984131,
235188.892346045351587,
235199.046319905639393,
235209.200332686188631,
235219.354384385631420,
235229.508475002367049,
235239.662604534969432,
235249.816772981779650,
235259.970980341400718,
235270.125226612319238,
235280.279511792905396,
235290.433835881791310,
235300.588198877376271,
235310.742600778146880,
235320.897041582647944,
235331.051521289336961,
235341.206039896642324,
235351.360597403137945,
235361.515193807281321,
235371.669829107529949,
235381.824503302428639,
235391.979216390405782,
235402.133968369947979,
235412.288759239629144,
235422.443588997848565,
235432.598457643063739,
235442.753365173877683,
235452.908311588718789,
235463.063296886015451,
235473.218321064399788,
235483.373384122271091,
235493.528486058086855,
235503.683626870362787,
235513.838806557643693,
235523.994025118328864,
235534.149282550963107,
235544.304578854062129,
235554.459914025996113,
235564.615288065397181,
235574.770700970693724,
235584.926152740314137,
235595.081643372861436,
235605.237172866793117,
235615.392741220479365,
235625.548348432581406,
235635.703994501527632,
235645.859679425746435,
235656.015403203811729,
235666.171165834151907,
235676.326967315311776,
235686.482807645748835,
235696.638686823949683,
235706.794604848430026,
235716.950561717618257,
235727.106557430175599,
235737.262591984384926,
235747.418665378820151,
235757.574777611967875,
235767.730928682372905,
235777.887118588434532,
235788.043347328755772,
235798.199614901735913,
235808.355921305890661,
235818.512266539735720,
235828.668650601728586,
235838.825073490384966,
235848.981535204162356,
235859.138035741634667,
235869.294575101201190,
235879.451153281435836,
235889.607770280766999,
235899.764426097710384,
235909.921120730781695,
235920.077854178467533,
235930.234626439254498,
235940.391437511600088,
235950.548287394049112,
235960.705176085029962,
235970.862103583087446,
235981.019069886766374,
235991.176074994495139,
236001.333118904731236,
236011.490201616019476,
236021.647323126875563,
236031.804483435727889,
236041.961682541179471,
236052.118920441629598,
236062.276197135564871,
236072.433512621559203,
236082.590866898099193,
236092.748259963584132,
236102.905691816529725,
236113.063162455568090,
236123.220671879069414,
236133.378220085549401,
236143.535807073523756,
236153.693432841479080,
236163.851097387872869,
236174.008800711308140,
236184.166542810213286,
236194.324323682987597,
236204.482143328321399,
236214.640001744613983,
236224.797898930351948,
236234.955834883992793,
236245.113809604139533,
236255.271823089220561,
236265.429875337722478,
236275.587966348190093,
236285.746096119110007,
236295.904264648939716,
236306.062471936253132,
236316.220717979478650,
236326.379002777102869,
236336.537326327699702,
236346.695688629755750,
236356.854089681641199,
236367.012529481988167,
236377.171008029283257,
236387.329525322042173,
236397.488081358664203,
236407.646676137694158,
236417.805309657647740,
236427.963981917069759,
236438.122692914359504,
236448.281442648119992,
236458.440231116808718,
236468.599058318824973,
236478.757924252829980,
236488.916828917223029,
236499.075772310548928,
236509.234754431265173,
236519.393775277974783,
236529.552834849018836,
236539.711933142971247,
236549.871070158376824,
236560.030245893751271,
236570.189460347493878,
236580.348713518091245,
236590.508005404233700,
236600.667336004233221,
236610.826705316663720,
236620.986113340070006,
236631.145560072851367,
236641.305045513523510,
236651.464569660689449,
236661.624132512806682,
236671.783734068303602,
236681.943374325754121,
236692.103053283703048,
236702.262770940549672,
236712.422527294751490,
236722.582322345027933,
236732.742156089720083,
236742.902028527314542,
236753.061939656385221,
236763.221889475418720,
236773.381877982930746,
236783.541905177378794,
236793.701971057365881,
236803.862075621291297,
236814.022218867641641,
236824.182400794990826,
236834.342621401825454,
236844.502880686719436,
236854.663178648013854,
236864.823515284311725,
236874.983890594186960,
236885.144304575951537,
236895.304757228295784,
236905.465248549648095,
236915.625778538465966,
236925.786347193323309,
236935.946954512764933,
236946.107600495190127,
236956.268285139201907,
236966.429008443199564,
236976.589770405786112,
236986.750571025419049,
236996.911410300555872,
237007.072288229799597,
237017.233204811607720,
237027.394160044525051,
237037.555153927009087,
237047.716186457546428,
237057.877257634710986,
237068.038367456989363,
237078.199515922897262,
237088.360703030863078,
237098.521928779518930,
237108.683193167205900,
237118.844496192585211,
237129.005837854114361,
237139.167218150279950,
237149.328637079597684,
237159.490094640612369,
237169.651590831839712,
237179.813125651708106,
237189.974699098733254,
237200.136311171459965,
237210.297961868403945,
237220.459651188080898,
237230.621379128977424,
237240.783145689609228,
237250.944950868462911,
237261.106794664054178,
237271.268677074927837,
237281.430598099541385,
237291.592557736439630,
237301.754555984138278,
237311.916592841123929,
237322.078668305912288,
237332.240782377019059,
237342.402935052930843,
237352.565126332163345,
237362.727356213261373,
237372.889624694682425,
237383.051931774971308,
237393.214277452643728,
237403.376661726244492,
237413.539084594172891,
237423.701546055031940,
237433.864046107337344,
237444.026584749517497,
237454.189161980088102,
237464.351777797710383,
237474.514432200725423,
237484.677125187678030,
237494.839856757142115,
237505.002626907662489,
237515.165435637580231,
237525.328282945585670,
237535.491168830107199,
237545.654093289660523,
237555.817056322732242,
237565.980057927896269,
237576.143098103639204,
237586.306176848476753,
237596.469294160895515,
237606.632450039411196,
237616.795644482568605,
237626.958877488883445,
237637.122149056842318,
237647.285459184902720,
237657.448807871667668,
237667.612195115652867,
237677.775620915403124,
237687.939085269259522,
237698.102588175883284,
237708.266129633731907,
237718.429709641350200,
237728.593328197253868,
237738.756985299987718,
237748.920680947951041,
237759.084415139805060,
237769.248187873890856,
237779.411999148869654,
237789.575848963198951,
237799.739737315423554,
237809.903664204000961,
237820.067629627534188,
237830.231633584422525,
237840.395676073268987,
237850.559757092531072,
237860.723876640811795,
237870.888034716568654,
237881.052231318288250,
237891.216466444544494,
237901.380740093823988,
237911.545052264642436,
237921.709402955544647,
237931.873792164988117,
237942.038219891517656,
237952.202686133648967,
237962.367190889926860,
237972.531734158867039,
237982.696315938956104,
237992.860936228709761,
238003.025595026672818,
238013.190292331302771,
238023.355028141202638,
238033.519802454829914,
238043.684615270671202,
238053.849466587358620,
238064.014356403320562,
238074.179284717130940,
238084.344251527218148,
238094.509256832156098,
238104.674300630547805,
238114.839382920792559,
238125.004503701435169,
238135.169662970933132,
238145.334860727947671,
238155.500096970936283,
238165.665371698385570,
238175.830684908753028,
238185.996036600758089,
238196.161426772741834,
238206.326855423307279,
238216.492322550999233,
238226.657828154216986,
238236.823372231563553,
238246.988954781554639,
238257.154575802676845,
238267.320235293504084,
238277.485933252522955,
238287.651669678249164,
238297.817444569227519,
238307.983257923915517,
238318.149109740916174,
238328.315000018745195,
238338.480928755889181,
238348.646895950805629,
238358.812901602097554,
238368.978945708338870,
238379.145028267957969,
238389.311149279470555,
238399.477308741479646,
238409.643506652442738,
238419.809743010904640,
238429.976017815351952,
238440.142331064300379,
238450.308682756352937,
238460.475072890025331,
238470.641501463745954,
238480.807968476030510,
238490.974473925540224,
238501.141017810732592,
238511.307600130094215,
238521.474220882140798,
238531.640880065417150,
238541.807577678497182,
238551.974313719867496,
238562.141088188014692,
238572.307901081454474,
238582.474752398848068,
238592.641642138536554,
238602.808570299181156,
238612.975536879181163,
238623.142541877139593,
238633.309585291630356,
238643.476667121081846,
238653.643787364097079,
238663.810946019075345,
238673.978143084677868,
238684.145378559420351,
238694.312652441731188,
238704.479964730184292,
238714.647315423266264,
238724.814704519609222,
238734.982132017641561,
238745.149597915966297,
238755.317102212982718,
238765.484644907352049,
238775.652225997531787,
238785.819845482066739,
238795.987503359472612,
238806.155199628236005,
238816.322934286989039,
238826.490707334160106,
238836.658518768352224,
238846.826368588022888,
238856.994256791716907,
238867.162183377949987,
238877.330148345325142,
238887.498151692241663,
238897.666193417360773,
238907.834273519139970,
238918.002391996065853,
238928.170548846799647,
238938.338744069682434,
238948.506977663375437,
238958.675249626394361,
238968.843559957284015,
238979.011908654414583,
238989.180295716534602,
238999.348721142072463,
239009.517184929485666,
239019.685687077435432,
239029.854227584321052,
239040.022806448803749,
239050.191423669341020,
239060.360079244419467,
239070.528773172642104,
239080.697505452466430,
239090.866276082553668,
239101.035085061303107,
239111.203932387317764,
239121.372818059084238,
239131.541742075060029,
239141.710704433964565,
239151.879705134197138,
239162.048744174244348,
239172.217821552825626,
239182.386937268223846,
239192.556091319158440,
239202.725283704086905,
239212.894514421554049,
239223.063783470075577,
239233.233090848254506,
239243.402436554490123,
239253.571820587443653,
239263.741242945543490,
239273.910703627305338,
239284.080202631390421,
239294.249739956256235,
239304.419315600447590,
239314.588929562451085,
239324.758581840840634,
239334.928272434219252,
239345.098001340928022,
239355.267768559686374,
239365.437574088922702,
239375.607417927152710,
239385.777300073037622,
239395.947220524976728,
239406.117179281573044,
239416.287176341313170,
239426.457211702771019,
239436.627285364433192,
239446.797397324931808,
239456.967547582695261,
239467.137736136268359,
239477.307962984225014,
239487.478228125110036,
239497.648531557380920,
239507.818873279698892,
239517.989253290492343,
239528.159671588276979,
239538.330128171655815,
239548.500623039144557,
239558.671156189346220,
239568.841727620689198,
239579.012337331630988,
239589.182985320949228,
239599.353671586985001,
239609.524396128341323,
239619.695158943563001,
239629.865960031165741,
239640.036799389636144,
239650.207677017664537,
239660.378592913562898,
239670.549547076050658,
239680.720539503614418,
239690.891570194769884,
239701.062639148003655,
239711.233746362006059,
239721.404891835205490,
239731.576075566030340,
239741.747297553229146,
239751.918557795230299,
239762.089856290607713,
239772.261193037789781,
239782.432568035495933,
239792.603981282067252,
239802.775432776223170,
239812.946922516362974,
239823.118450501147890,
239833.290016728948103,
239843.461621198453940,
239853.633263908152003,
239863.804944856557995,
239873.976664042158518,
239884.148421463673003,
239894.320217119471636,
239904.492051008128328,
239914.663923128246097,
239924.835833478282439,
239935.007782056840369,
239945.179768862406490,
239955.351793893554714,
239965.523857148771640,
239975.695958626689389,
239985.868098325794563,
239996.040276244573761,
240006.212492381630000,
240016.384746735478984,
240026.557039304723730,
240036.729370087821735,
240046.901739083376015,
240057.074146289873170,
240067.246591705828905,
240077.419075329904445,
240087.591597160499077,
240097.764157196215820,
240107.936755435599480,
240118.109391877253074,
240128.282066519575892,
240138.454779361229157,
240148.627530400728574,
240158.800319636502536,
240168.973147067270475,
240179.146012691431679,
240189.318916507647373,
240199.491858514316846,
240209.664838710101321,
240219.837857093545608,
240230.010913663078099,
240240.184008417330915,
240250.357141354848864,
240260.530312474147649,
240270.703521773772081,
240280.876769252208760,
240291.050054908118909,
240301.223378739960026,
240311.396740746276919,
240321.570140925701708,
240331.743579276662786,
240341.917055797704961,
240352.090570487489458,
240362.264123344415566,
240372.437714367144508,
240382.611343554133782,
240392.785010903986404,
240402.958716415247181,
240413.132460086402716,
240423.306241916026920,
240433.480061902664602,
240443.653920044889674,
240453.827816341188736,
240464.001750790223014,
240474.175723390362691,
240484.349734140239889,
240494.523783038399415,
240504.697870083415182,
240514.871995273831999,
240525.046158608136466,
240535.220360084873391,
240545.394599702616688,
240555.568877459969372,
240565.743193355330732,
240575.917547387420200,
240586.091939554666169,
240596.266369855671655,
240606.440838288952364,
240616.615344852994895,
240626.789889546489576,
240636.964472367864801,
240647.139093315723585,
240657.313752388552530,
240667.488449584925547,
240677.663184903503861,
240687.837958342686761,
240698.012769901077263,
240708.187619577220175,
240718.362507369602099,
240728.537433276855154,
240738.712397297465941,
240748.887399430066580,
240759.062439673143672,
240769.237518025242025,
240779.412634484848240,
240789.587789050652646,
240799.762981721083634,
240809.938212494831532,
240820.113481370266527,
240830.288788346049841,
240840.464133420668077,
240850.639516592782456,
240860.814937860792270,
240870.990397223300533,
240881.165894678881159,
240891.341430226137163,
240901.517003863496939,
240911.692615589563502,
240921.868265402852558,
240932.043953302025329,
240942.219679285481106,
240952.395443351881113,
240962.571245499770157,
240972.747085727634840,
240982.922964034078177,
240993.098880417557666,
241003.274834876734531,
241013.450827410124475,
241023.626858016272308,
241033.802926693722839,
241043.979033441020874,
241054.155178256740328,
241064.331361139426008,
241074.507582087651826,
241084.683841099846177,
241094.860138174728490,
241105.036473310785368,
241115.212846506503411,
241125.389257760514738,
241135.565707071364159,
241145.742194437567377,
241155.918719857669203,
241166.095283330272650,
241176.271884853922529,
241186.448524427105440,
241196.625202048453502,
241206.801917716453318,
241216.978671429737005,
241227.155463186791167,
241237.332292986160610,
241247.509160826448351,
241257.686066706140991,
241267.863010623870650,
241278.039992578123929,
241288.217012567532947,
241298.394070590584306,
241308.571166645822814,
241318.748300731851486,
241328.925472847215133,
241339.102682990429457,
241349.279931160039268,
241359.457217354676686,
241369.634541572828311,
241379.811903813097160,
241389.989304074028041,
241400.166742354107555,
241410.344218651967822,
241420.521732966153650,
241430.699285295122536,
241440.876875637593912,
241451.054503992054379,
241461.232170356990537,
241471.409874731063610,
241481.587617112701992,
241491.765397500537802,
241501.943215893174056,
241512.121072289126460,
241522.298966686881613,
241532.476899085100740,
241542.654869482270442,
241552.832877876993734,
241563.010924267786322,
241573.189008653193014,
241583.367131031845929,
241593.545291402289877,
241603.723489762953250,
241613.901726112555480,
241624.080000449524960,
241634.258312772522913,
241644.436663080006838,
241654.615051370696165,
241664.793477642990183,
241674.971941895433702,
241685.150444126746152,
241695.328984335355926,
241705.507562519807834,
241715.686178678675788,
241725.864832810621010,
241736.043524914042791,
241746.222254987660563,
241756.401023029960925,
241766.579829039488686,
241776.758673014730448,
241786.937554954405641,
241797.116474856913555,
241807.295432720973622,
241817.474428545043338,
241827.653462327667512,
241837.832534067449160,
241848.011643762991298,
241858.190791412722319,
241868.369977015361656,
241878.549200569250388,
241888.728462073166156,
241898.907761525566457,
241909.087098925112514,
241919.266474270232720,
241929.445887559471885,
241939.625338791549439,
241949.804827964835567,
241959.984355078078806,
241970.163920129649341,
241980.343523118295707,
241990.523164042446297,
242000.702842900762334,
242010.882559691672213,
242021.062314413895365,
242031.242107065860182,
242041.421937646169681,
242051.601806153397774,
242061.781712586089270,
242071.961656942818081,
242082.141639222187223,
242092.321659422683297,
242102.501717542909319,
242112.681813581468305,
242122.861947536846856,
242133.042119407560676,
242143.222329192329198,
242153.402576889609918,
242163.582862498005852,
242173.763186016061809,
242183.943547442322597,
242194.123946775391232,
242204.304384013812523,
242214.484859156131279,
242224.665372200921411,
242234.845923146785935,
242245.026511992240557,
242255.207138735830085,
242265.387803376157535,
242275.568505911767716,
242285.749246341263643,
242295.930024663248332,
242306.110840876150178,
242316.291694978601299,
242326.472586969146505,
242336.653516846417915,
242346.834484608902130,
242357.015490255260374,
242367.196533783921041,
242377.377615193545353,
242387.558734482678119,
242397.739891649864148,
242407.921086693735560,
242418.102319612778956,
242428.283590405568248,
242438.464899070764659,
242448.646245606738375,
242458.827630012267036,
242469.009052285779035,
242479.190512425935594,
242489.372010431281524,
242499.553546300216112,
242509.735120031575207,
242519.916731623758096,
242530.098381075309590,
242540.280068384949118,
242550.461793551105075,
242560.643556572380476,
242570.825357447378337,
242581.007196174643468,
242591.189072752662469,
242601.370987180125667,
242611.552939455577871,
242621.734929577592993,
242631.916957544599427,
242642.099023355374811,
242652.281127008347539,
242662.463268502120627,
242672.645447835297091,
242682.827665006363532,
242693.009920013981173,
242703.192212856607512,
242713.374543532961980,
242723.556912041502073,
242733.739318380772602,
242743.921762549492996,
242754.104244546091650,
242764.286764369171578,
242774.469322017335799,
242784.651917489070911,
242794.834550783038139,
242805.017221897753188,
242815.199930831819074,
242825.382677583809709,
242835.565462152211694,
242845.748284535802668,
242855.931144732894609,
242866.114042742206948,
242876.296978562284494,
242886.479952191672055,
242896.662963628972648,
242906.846012872672873,
242917.029099921463057,
242927.212224773829803,
242937.395387428463437,
242947.578587883792352,
242957.761826138477772,
242967.945102191064507,
242978.128416040039156,
242988.311767684121151,
242998.495157121738885,
243008.678584351669997,
243018.862049372226465,
243029.045552182156825,
243039.229092780005885,
243049.412671164231142,
243059.596287333610235,
243069.779941286513349,
243079.963633021659916,
243090.147362537536537,
243100.331129832804436,
243110.514934905892005,
243120.698777755518677,
243130.882658380171051,
243141.066576778452145,
243151.250532948935870,
243161.434526890108828,
243171.618558600719552,
243181.802628079196438,
243191.986735324200708,
243202.170880334218964,
243212.355063107912429,
243222.539283643884119,
243232.723541940562427,
243242.907837996608578,
243253.092171810567379,
243263.276543381100055,
243273.460952706605894,
243283.645399785804329,
243293.829884617327480,
243304.014407199603738,
243314.198967531177914,
243324.383565610827645,
243334.568201436981326,
243344.752875008241972,
243354.937586323096184,
243365.122335380321601,
243375.307122178346617,
243385.491946715774247,
243395.676808991178405,
243405.861709003162105,
243416.046646750241052,
243426.231622231076472,
243436.416635444213171,
243446.601686388195958,
243456.786775061569642,
243466.971901462995447,
243477.157065591076389,
243487.342267444299068,
243497.527507021208294,
243507.712784320523497,
243517.898099340731278,
243528.083452080434654,
243538.268842538091121,
243548.454270712420112,
243558.639736601995537,
243568.825240205362206,
243579.010781521123135,
243589.196360547823133,
243599.381977284007007,
243609.567631728277775,
243619.753323879296659,
243629.939053735550260,
243640.124821295583388,
243650.310626558057265,
243660.496469521574909,
243670.682350184622919,
243680.868268545775209,
243691.054224603693001,
243701.240218357008416,
243711.426249804149847,
243721.612318943720311,
243731.798425774381030,
243741.984570294676814,
243752.170752503152471,
243762.356972398411017,
243772.543229978997260,
243782.729525243572425,
243792.915858190681320,
243803.102228818868753,
243813.288637126795948,
243823.475083112891298,
243833.661566775903339,
243843.848088114318671,
243854.034647126769414,
243864.221243811742170,
243874.407878167956369,
243884.594550193898613,
243894.781259888171917,
243904.968007249379298,
243915.154792276007356,
243925.341614966775523,
243935.528475320170401,
243945.715373334795004,
243955.902309009252349,
243966.089282342145452,
243976.276293331960915,
243986.463341977359960,
243996.650428276858293,
244006.837552229175344,
244017.024713832768612,
244027.211913086241111,
244037.399149988195859,
244047.586424537235871,
244057.773736731905956,
244067.961086570809130,
244078.148474052490201,
244088.335899175610393,
244098.523361938598100,
244108.710862340230960,
244118.898400378995575,
244129.085976053436752,
244139.273589362273924,
244149.461240303935483,
244159.648928877082653,
244169.836655080260243,
244180.024418912100373,
244190.212220371206058,
244200.400059456093004,
244210.587936165422434,
244220.775850497680949,
244230.963802451529773,
244241.151792025455507,
244251.339819218177581,
244261.527884028240805,
244271.715986454189988,
244281.904126494569937,
244292.092304148100084,
244302.280519413267029,
244312.468772288673790,
244322.657062772923382,
244332.845390864560613,
244343.033756562246708,
244353.222159864468267,
244363.410600769886514,
244373.599079277017154,
244383.787595384521410,
244393.976149091031402,
244404.164740395033732,
244414.353369295073207,
244424.542035789811052,
244434.730739877850283,
244444.919481557793915,
244455.108260828186758,
244465.297077687573619,
244475.485932134615723,
244485.674824167799670,
244495.863753785903100,
244506.052720987354405,
244516.241725770756602,
244526.430768134712707,
244536.619848077825736,
244546.808965598698705,
244556.998120695934631,
244567.187313368020114,
244577.376543613558169,
244587.565811431326438,
244597.755116819607792,
244607.944459777296288,
244618.133840302791214,
244628.323258394753793,
244638.512714051728835,
244648.702207272377564,
244658.891738055186579,
244669.081306398758898,
244679.270912301755743,
244689.460555762780132,
244699.650236780318664,
244709.839955353090772,
244720.029711479583057,
244730.219505158340326,
244740.409336388082011,
244750.599205167352920,
244760.789111494697863,
244770.979055368778063,
244781.169036788138328,
244791.359055751381675,
244801.549112257052911,
244811.739206303813262,
244821.929337890207535,
244832.119507014896953,
244842.309713676368119,
244852.499957873282256,
244862.690239604242379,
244872.880558867764194,
244883.070915662450716,
244893.261309986992273,
244903.451741839875467,
244913.642211219761521,
244923.832718125195242,
244934.023262554779649,
244944.213844507175963,
244954.404463980812579,
244964.595120974408928,
244974.785815486568026,
244984.976547515834682,
244995.167317060811911,
245005.358124120044522,
245015.548968692251947,
245025.739850775862578,
245035.930770369654056,
245046.121727471996564,
245056.312722081667744,
245066.503754197212402,
245076.694823817233555,
245086.885930940276012,
245097.077075564942788,
245107.268257689836901,
245117.459477313561365,
245127.650734434777405,
245137.842029051971622,
245148.033361163688824,
245158.224730768706650,
245168.416137865511701,
245178.607582452706993,
245188.799064528837334,
245198.990584092563950,
245209.182141142548062,
245219.373735677218065,
245229.565367695293389,
245239.757037195289740,
245249.948744175926549,
245260.140488635603106,
245270.332270573097048,
245280.524089986953186,
245290.715946875716327,
245300.907841237989487,
245311.099773072433891,
245321.291742377594346,
245331.483749152073869,
245341.675793394475477,
245351.867875103402184,
245362.059994277457008,
245372.252150915184757,
245382.444345015188446,
245392.636576576129301,
245402.828845596552128,
245413.021152075118152,
245423.213496010255767,
245433.405877400800819,
245443.598296245181700,
245453.790752542059636,
245463.983246289979434,
245474.175777487660525,
245484.368346133531304,
245494.560952226311201,
245504.753595764545025,
245514.946276746835792,
245525.138995171844726,
245535.331751038058428,
245545.524544344196329,
245555.717375088745030,
245565.910243270365754,
245576.103148887661519,
245586.296091939177131,
245596.489072423573816,
245606.682090339454589,
245616.875145685364259,
245627.068238459905842,
245637.261368661682354,
245647.454536289296811,
245657.647741341410438,
245667.840983816626249,
245678.034263713372638,
245688.227581030427245,
245698.420935766334878,
245708.614327919640345,
245718.807757489063079,
245729.001224473089678,
245739.194728870381368,
245749.388270679482957,
245759.581849899113877,
245769.775466527702520,
245779.969120564026525,
245790.162812006572494,
245800.356540853943443,
245810.550307104742387,
245820.744110757630551,
245830.937951811210951,
245841.131830263970187,
245851.325746114685899,
245861.519699361786479,
245871.713690003933152,
245881.907718039787142,
245892.101783467835048,
245902.295886286854511,
245912.490026495273924,
245922.684204091812717,
245932.878419074899284,
245943.072671443369472,
245953.266961195709882,
245963.461288330523530,
245973.655652846413432,
245983.850054741982603,
245994.044494015834061,
246004.238970666570822,
246014.433484692795901,
246024.628036093112314,
246034.822624866181286,
246045.017251010547625,
246055.211914524756139,
246065.406615407468053,
246075.601353657344589,
246085.796129272930557,
246095.990942252828972,
246106.185792595701059,
246116.380680300033418,
246126.575605364545481,
246136.770567787782056,
246146.965567568346160,
246157.160604704840807,
246167.355679195869016,
246177.550791040092008,
246187.745940236112801,
246197.941126782476204,
246208.136350677727023,
246218.331611920642899,
246228.526910509768641,
246238.722246443590848,
246248.917619720887160,
246259.113030340085970,
246269.308478299906710,
246279.503963598952396,
246289.699486235826043,
246299.895046209130669,
246310.090643517469289,
246320.286278159444919,
246330.481950133602368,
246340.677659438602859,
246350.873406073107617,
246361.069190035603242,
246371.265011324867373,
246381.460869939328404,
246391.656765877647558,
246401.852699138486059,
246412.048669720446924,
246422.244677622016752,
246432.440722842031391,
246442.636805378802819,
246452.832925231166882,
246463.029082397726597,
246473.225276876910357,
246483.421508667495800,
246493.617777768085944,
246503.814084177109180,
246514.010427893459564,
246524.206808915449074,
246534.403227241855348,
246544.599682871281402,
246554.796175802330254,
246564.992706033546710,
246575.189273563591996,
246585.385878391069127,
246595.582520514581120,
246605.779199932730990,
246615.975916644121753,
246626.172670647472842,
246636.369461941212649,
246646.566290524002397,
246656.763156394561520,
246666.960059551376617,
246677.156999993167119,
246687.353977718419628,
246697.550992725795368,
246707.748045013955561,
246717.945134581445018,
246728.142261426924961,
246738.339425548940198,
246748.536626946210163,
246758.733865617250558,
246768.931141560606193,
246779.128454775083810,
246789.325805259111803,
246799.523193011467811,
246809.720618030580226,
246819.918080315168481,
246830.115579863893799,
246840.313116675242782,
246850.510690747934859,
246860.708302080514841,
246870.905950671527535,
246881.103636519750580,
246891.301359623728786,
246901.499119982065167,
246911.696917593304534,
246921.894752456166316,
246932.092624569195323,
246942.290533931052778,
246952.488480540341698,
246962.686464395606890,
246972.884485495567787,
246983.082543838769197,
246993.280639423814137,
247003.478772249363828,
247013.676942313963082,
247023.875149616331328,
247034.073394155013375,
247044.271675928612240,
247054.469994935730938,
247064.668351175030693,
247074.866744645056315,
247085.065175344527233,
247095.263643272046465,
247105.462148426100612,
247115.660690805409104,
247125.859270408604061,
247136.057887234288501,
247146.256541280919919,
247156.455232547275955,
247166.653961032017833,
247176.852726733573945,
247187.051529650721932,
247197.250369782064809,
247207.449247126089176,
247217.648161681572674,
247227.847113447001902,
247238.046102421038086,
247248.245128602284240,
247258.444191989401588,
247268.643292580934940,
247278.842430375603726,
247289.041605371894548,
247299.240817568526836,
247309.440066964103607,
247319.639353557169670,
247329.838677346444456,
247340.038038330414565,
247350.237436507857637,
247360.436871877260273,
247370.636344437283697,
247380.835854186530923,
247391.035401123663178,
247401.234985247254372,
247411.434606555907521,
247421.634265048312955,
247431.833960723015480,
247442.033693578676321,
247452.233463613840286,
247462.433270827226806,
247472.633115217438899,
247482.832996783021372,
247493.032915522635449,
247503.232871434942354,
247513.432864518486895,
247523.632894771872088,
247533.832962193759158,
247544.033066782809328,
247554.233208537625615,
247564.433387456752826,
247574.633603538852185,
247584.833856782584917,
247595.034147186554037,
247605.234474749304354,
247615.434839469497092,
247625.635241345851682,
247635.835680376825621,
247646.036156561196549,
247656.236669897421962,
247666.437220384163084,
247676.637808020197554,
247686.838432803953765,
247697.039094734092942,
247707.239793809334515,
247717.440530028165085,
247727.641303389304085,
247737.842113891296322,
247748.042961532803020,
247758.243846312485402,
247768.444768228946486,
247778.645727280731080,
247788.846723466558615,
247799.047756784973899,
247809.248827234638156,
247819.449934814154403,
247829.651079522125656,
247839.852261357300449,
247850.053480318165384,
247860.254736403294373,
247870.456029611523263,
247880.657359941222239,
247890.858727391168941,
247901.060131960024592,
247911.261573646275792,
247921.463052448583767,
247931.664568365609739,
247941.866121396014933,
247952.067711538285948,
247962.269338791200425,
247972.471003153244965,
247982.672704623197205,
247992.874443199543748,
248003.076218880945817,
248013.278031666035531,
248023.479881553415908,
248033.681768541777274,
248043.883692629606230,
248054.085653815738624,
248064.287652098544640,
248074.489687476860126,
248084.691759949171683,
248094.893869514256949,
248105.096016170544317,
248115.298199916811427,
248125.500420751603087,
248135.702678673580522,
248145.904973681346746,
248156.107305773562985,
248166.309674948774045,
248176.512081205699360,
248186.714524542912841,
248196.917004959075712,
248207.119522452703677,
248217.322077022574376,
248227.524668667232618,
248237.727297385339625,
248247.929963175440207,
248258.132666036253795,
248268.335405966325197,
248278.538182964373846,
248288.740997028944548,
248298.943848158756737,
248309.146736352297012,
248319.349661608284805,
248329.552623925323132,
248339.755623302102322,
248349.958659737167181,
248360.161733229149831,
248370.364843776711496,
248380.567991378455190,
248390.771176032983931,
248400.974397739017149,
248411.177656495099654,
248421.380952299892670,
248431.584285151941003,
248441.787655050080502,
248451.991061992681352,
248462.194505978521192,
248472.397987006261246,
248482.601505074475426,
248492.805060181708541,
248503.008652326709125,
248513.212281508080196,
248523.415947724366561,
248533.619650974287651,
248543.823391256446484,
248554.027168569446076,
248564.230982912005857,
248574.434834282612428,
248584.638722680043429,
248594.842648102785461,
248605.046610549557954,
248615.250610019022133,
248625.454646509751910,
248635.658720020350302,
248645.862830549449427,
248656.066978095710510,
248666.271162657736568,
248676.475384234247031,
248686.679642823786708,
248696.883938424958615,
248707.088271036540391,
248717.292640656960430,
248727.497047284938162,
248737.701490919193020,
248747.905971558211604,
248758.110489200742450,
248768.315043845330365,
248778.519635490665678,
248788.724264135351405,
248798.928929777990561,
248809.133632417244371,
248819.338372051774058,
248829.543148680182640,
248839.747962301073130,
248849.952812913106754,
248860.157700514886528,
248870.362625105131883,
248880.567586682416731,
248890.772585245344089,
248900.977620792604284,
248911.182693322742125,
248921.387802834477043,
248931.592949326353846,
248941.798132797091966,
248952.003353245294420,
248962.208610669622431,
248972.413905068620807,
248982.619236441067187,
248992.824604785389965,
249003.030010100512300,
249013.235452384746168,
249023.440931636898313,
249033.646447855571751,
249043.852001039485913,
249054.057591187127400,
249064.263218297157437,
249074.468882368295453,
249084.674583399144467,
249094.880321388307493,
249105.086096334445756,
249115.291908236191375,
249125.497757092147367,
249135.703642900945852,
249145.909565661248052,
249156.115525371715194,
249166.321522030950291,
249176.527555637614569,
249186.733626190252835,
249196.939733687642729,
249207.145878128270851,
249217.352059510914842,
249227.558277834090404,
249237.764533096516971,
249247.970825296768453,
249258.177154433506075,
249268.383520505391061,
249278.589923510968219,
249288.796363449015189,
249299.002840318018571,
249309.209354116814211,
249319.415904843830504,
249329.622492497786880,
249339.829117077373667,
249350.035778581077466,
249360.242477007705020,
249370.449212355801137,
249380.655984623968834,
249390.862793810985750,
249401.069639915338485,
249411.276522935746470,
249421.483442870812723,
249431.690399719198467,
249441.897393479535822,
249452.104424150486011,
249462.311491730622947,
249472.518596218607854,
249482.725737613101956,
249492.932915912708268,
249503.140131116146222,
249513.347383221960627,
249523.554672228812706,
249533.761998135421891,
249543.969360940303886,
249554.176760642178124,
249564.384197239560308,
249574.591670731286285,
249584.799181115842657,
249595.006728391948855,
249605.214312558207894,
249615.421933613222791,
249625.629591555771185,
249635.837286384339677,
249646.045018097618595,
249656.252786694269162,
249666.460592172865290,
249676.668434532126412,
249686.876313770713750,
249697.084229887172114,
249707.292182880220935,
249717.500172748405021,
249727.708199490443803,
249737.916263104969403,
249748.124363590643043,
249758.332500946038635,
249768.540675169875612,
249778.748886260698782,
249788.957134217227576,
249799.165419038065011,
249809.373740721872309,
249819.582099267252488,
249829.790494672954082,
249839.998926937521901,
249850.207396059529856,
249860.415902037755586,
249870.624444870802108,
249880.833024557330646,
249891.041641095886007,
249901.250294485245831,
249911.458984723896720,
249921.667711810703622,
249931.876475744036725,
249942.085276522760978,
249952.294114145362983,
249962.502988610620378,
249972.711899917077972,
249982.920848063396988,
249993.129833048296859,
250003.338854870264186,
250013.547913528105710,
250023.757009020366240,
250033.966141345794313,
250044.175310502876528,
250054.384516490332317,
250064.593759306822903,
250074.803038950951304,
250085.012355421436951,
250095.221708716882858,
250105.431098835921148,
250115.640525777154835,
250125.849989539274247,
250136.059490120998817,
250146.269027520815143,
250156.478601737442659,
250166.688212769600796,
250176.897860615834361,
250187.107545274833683,
250197.317266745201778,
250207.527025025628973,
250217.736820114776492,
250227.946652011189144,
250238.156520713586360,
250248.366426220629364,
250258.576368530979380,
250268.786347643152112,
250278.996363555983407,
250289.206416267988971,
250299.416505777771818,
250309.626632084167795,
250319.836795185605297,
250330.046995080861961,
250340.257231768599013,
250350.467505247390363,
250360.677815515897237,
250370.888162572809961,
250381.098546416673344,
250391.308967046265025,
250401.519424460188020,
250411.729918657045346,
250421.940449635498226,
250432.151017394236987,
250442.361621931806440,
250452.572263247013325,
250462.782941338402452,
250472.993656204635045,
250483.204407844314119,
250493.415196256159106,
250503.626021438831231,
250513.836883390962612,
250524.047782111098059,
250534.258717597986106,
250544.469689850287978,
250554.680698866606690,
250564.891744645545259,
250575.102827185881324,
250585.313946486159693,
250595.525102545128902,
250605.736295361333760,
250615.947524933406385,
250626.158791260124417,
250636.370094339974457,
250646.581434171792353,
250656.792810754122911,
250667.004224085598253,
250677.215674164879601,
250687.427160990599077,
250697.638684561476111,
250707.850244876113720,
250718.061841933114920,
250728.273475731257349,
250738.485146269085817,
250748.696853545348858,
250758.908597558591282,
250769.120378307445208,
250779.332195790630067,
250789.544050006807083,
250799.755940954637481,
250809.967868632666068,
250820.179833039699588,
250830.391834174224641,
250840.603872035047971,
250850.815946620714385,
250861.028057929885108,
250871.240205961279571,
250881.452390713500790,
250891.664612185180886,
250901.876870374981081,
250912.089165281591704,
250922.301496903615771,
250932.513865239772713,
250942.726270288607338,
250952.938712048839079,
250963.151190519216470,
250973.363705698226113,
250983.576257584558334,
250993.788846176932566,
251004.001471473951824,
251014.214133474277332,
251024.426832176570315,
251034.639567579462891,
251044.852339681616286,
251055.065148481779033,
251065.277993978437735,
251075.490876170311822,
251085.703795056062518,
251095.916750634351047,
251106.129742903867736,
251116.342771863215603,
251126.555837511026766,
251136.768939846078865,
251146.982078866858501,
251157.195254572085105,
251167.408466960419901,
251177.621716030553216,
251187.835001781088067,
251198.048324210772989,
251208.261683318094583,
251218.475079101830488,
251228.688511560641928,
251238.901980693131918,
251249.115486497990787,
251259.329028973821551,
251269.542608119314536,
251279.756223933130968,
251289.969876413932070,
251300.183565560320858,
251310.397291371016763,
251320.611053844710113,
251330.824852979945717,
251341.038688775413902,
251351.252561229775893,
251361.466470341751119,
251371.680416109942598,
251381.894398532982450,
251392.108417609590106,
251402.322473338339478,
251412.536565717949998,
251422.750694747082889,
251432.964860424341168,
251443.179062748444267,
251453.393301718024304,
251463.607577331742505,
251473.821889588230988,
251484.036238486150978,
251494.250624024221906,
251504.465046200988581,
251514.679505015199538,
251524.894000465516001,
251535.108532550511882,
251545.323101268964820,
251555.537706619419623,
251565.752348600537516,
251575.967027211125242,
251586.181742449727608,
251596.396494314918527,
251606.611282805533847,
251616.826107920060167,
251627.040969657333335,
251637.255868015898159,
251647.470802994444966,
251657.685774591634981,
251667.900782806100324,
251678.115827636502218,
251688.330909081501886,
251698.546027139818761,
251708.761181810143171,
251718.976373090903508,
251729.191600980964722,
251739.406865478988038,
251749.622166583576472,
251759.837504293333041,
251770.052878607064486,
251780.268289523257408,
251790.483737040718552,
251800.699221158050932,
251810.914741873857565,
251821.130299186916091,
251831.345893095858628,
251841.561523599229986,
251851.777190695836907,
251861.992894384282408,
251872.208634663169505,
251882.424411531275837,
251892.640224987175316,
251902.856075029587373,
251913.071961657085922,
251923.287884868390393,
251933.503844662220217,
251943.719841037120204,
251953.935873991838889,
251964.151943525037495,
251974.368049635289935,
251984.584192321315641,
251994.800371581775835,
252005.016587415389949,
252015.232839820790105,
252025.449128796579316,
252035.665454341389704,
252045.881816453998908,
252056.098215133068152,
252066.314650377142243,
252076.531122185086133,
252086.747630555328215,
252096.964175486558815,
252107.180756977671990,
252117.397375027037924,
252127.614029633637983,
252137.830720795842353,
252148.047448512457777,
252158.264212782029063,
252168.481013603392057,
252178.697850975149777,
252188.914724895934341,
252199.131635364406975,
252209.348582379258005,
252219.565565939148655,
252229.782586042740149,
252239.999642688635504,
252250.216735875641461,
252260.433865602244623,
252270.651031867309939,
252280.868234669324011,
252291.085474007064477,
252301.302749879134353,
252311.520062284223968,
252321.737411220994545,
252331.954796688136412,
252342.172218684252584,
252352.389677208062494,
252362.607172258198261,
252372.824703833379317,
252383.042271932237782,
252393.259876553376671,
252403.477517695573624,
252413.695195357489865,
252423.912909537699306,
252434.130660234921379,
252444.348447447788203,
252454.566271175019210,
252464.784131415275624,
252475.002028167131357,
252485.219961429422256,
252495.437931200664025,
252505.655937479634304,
252515.873980264936108,
252526.092059555172455,
252536.310175349150086,
252546.528327645472018,
252556.746516442828579,
252566.964741739822784,
252577.183003535174066,
252587.401301827572752,
252597.619636615621857,
252607.838007898069918,
252618.056415673461743,
252628.274859940574970,
252638.493340698099928,
252648.711857944581425,
252658.930411678767996,
252669.149001899379073,
252679.367628604959464,
252689.586291794228600,
252699.804991465935018,
252710.023727618623525,
252720.242500251042657,
252730.461309361853637,
252740.680154949659482,
252750.899037013208726,
252761.117955551162595,
252771.336910562153207,
252781.555902044841787,
252791.774929998005973,
252801.993994420161471,
252812.213095310085919,
252822.432232666411437,
252832.651406487799250,
252842.870616772910580,
252853.089863520493964,
252863.309146729152417,
252873.528466397576267,
252883.747822524368530,
252893.967215108365053,
252904.186644148081541,
252914.406109642179217,
252924.625611589464825,
252934.845149988483172,
252945.064724837953690,
252955.284336136537604,
252965.503983882983448,
252975.723668075894238,
252985.943388713902095,
252996.163145795726450,
253006.382939320028527,
253016.602769285556860,
253026.822635690856259,
253037.042538534675259,
253047.262477815675084,
253057.482453532458749,
253067.702465683891205,
253077.922514268400846,
253088.142599284794414,
253098.362720731733134,
253108.582878607878229,
253118.803072911920026,
253129.023303642461542,
253139.243570798251312,
253149.463874378008768,
253159.684214380278718,
253169.904590803867904,
253180.125003647262929,
253190.345452909328742,
253200.565938588610152,
253210.786460683884798,
253221.007019193784799,
253231.227614116971381,
253241.448245452134870,
253251.668913197878283,
253261.889617353008362,
253272.110357916069916,
253282.331134885811480,
253292.551948260836070,
253302.772798039979534,
253312.993684221757576,
253323.214606804889627,
253333.435565788066015,
253343.656561169947963,
253353.877592949196696,
253364.098661124560749,
253374.319765694643138,
253384.540906658075983,
253394.762084013695130,
253404.983297759987181,
253415.204547895817086,
253425.425834419671446,
253435.647157330357004,
253445.868516626476776,
253456.089912306808401,
253466.311344369925791,
253476.532812814490171,
253486.754317639308283,
253496.975858842924936,
253507.197436424088664,
253517.419050381460693,
253527.640700713731349,
253537.862387419503648,
253548.084110497555230,
253558.305869946460007,
253568.527665765053825,
253578.749497951794183,
253588.971366505516926,
253599.193271424883278,
253609.415212708525360,
253619.637190355220810,
253629.859204363485333,
253640.081254732038360,
253650.303341459657531,
253660.525464545033174,
253670.747623986651888,
253680.969819783378625,
253691.192051933816401,
253701.414320436713751,
253711.636625290615484,
253721.858966494270135,
253732.081344046338927,
253742.303757945599500,
253752.526208190683974,
253762.748694780137157,
253772.971217712765792,
253783.193776987231104,
253793.416372602281626,
253803.639004556403961,
253813.861672848463058,
253824.084377477061935,
253834.307118440861814,
253844.529895738611231,
253854.752709368913202,
253864.975559330516262,
253875.198445622081636,
253885.421368242241442,
253895.644327189715113,
253905.867322463192977,
253916.090354061336257,
253926.313421982806176,
253936.536526226351270,
253946.759666790574556,
253956.982843674224569,
253967.206056875846116,
253977.429306394304149,
253987.652592228201684,
253997.875914376229048,
254008.099272836989257,
254018.322667609259952,
254028.546098691789666,
254038.769566083006794,
254048.993069781834492,
254059.216609786817571,
254069.440186096820980,
254079.663798710273113,
254089.887447625980712,
254100.111132842663210,
254110.334854358923621,
254120.558612173510483,
254130.782406285085017,
254141.006236692279344,
254151.230103393871104,
254161.454006388492417,
254171.677945674804505,
254181.901921251497697,
254192.125933117291424,
254202.349981270876015,
254212.574065710854484,
254222.798186435975367,
254233.022343444958096,
254243.246536736347480,
254253.470766308950260,
254263.695032161427662,
254273.919334292470012,
254284.143672700738534,
254294.368047384923557,
254304.592458343686303,
254314.816905575717101,
254325.041389079735382,
254335.265908854431473,
254345.490464898408391,
254355.715057210414670,
254365.939685789169744,
254376.164350633276626,
254386.389051741483854,
254396.613789112452650,
254406.838562744815135,
254417.063372637348948,
254427.288218788744416,
254437.513101197546348,
254447.738019862619694,
254457.962974782451056,
254468.187965955963591,
254478.412993381643901,
254488.638057058298727,
254498.863156984531088,
254509.088293159089517,
254519.313465580635238,
254529.538674247916788,
254539.763919159420766,
254549.989200314041227,
254560.214517710410291,
254570.439871347130975,
254580.665261223068228,
254590.890687336708652,
254601.116149686858989,
254611.341648272180464,
254621.567183091305196,
254631.792754143069033,
254642.018361425987678,
254652.244004938780563,
254662.469684680312639,
254672.695400649012299,
254682.921152843744494,
254693.146941263112240,
254703.372765905864071,
254713.598626770603005,
254723.824523856193991,
254734.050457161065424,
254744.276426684082253,
254754.502432423905702,
254764.728474379226100,
254774.954552548704669,
254785.180666931031737,
254795.406817524926737,
254805.633004329021787,
254815.859227342094528,
254826.085486562718870,
254836.311781989730662,
254846.538113621645607,
254856.764481457299553,
254866.990885495324619,
254877.217325734382030,
254887.443802173220320,
254897.670314810442505,
254907.896863644797122,
254918.123448675061809,
254928.350069899752270,
254938.576727317704353,
254948.803420927491970,
254959.030150727834553,
254969.256916717538843,
254979.483718895149650,
254989.710557259415509,
254999.937431808997644,
255010.164342542644590,
255020.391289459075779,
255030.618272556806915,
255040.845291834673844,
255051.072347291425103,
255061.299438925547292,
255071.526566735847155,
255081.753730721102329,
255091.980930879886728,
255102.208167210861575,
255112.435439712833613,
255122.662748384405859,
255132.890093224385055,
255143.117474231316010,
255153.344891404005466,
255163.572344741056440,
255173.799834241217468,
255184.027359903207980,
255194.254921725601889,
255204.482519707176834,
255214.710153846681351,
255224.937824142660247,
255235.165530593978474,
255245.393273199180840,
255255.621051957074087,
255265.848866866261233,
255276.076717925432604,
255286.304605133365840,
255296.532528488693060,
255306.760487990133697,
255316.988483636319870,
255327.216515426029218,
255337.444583358010277,
255347.672687430749647,
255357.900827643112279,
255368.129003993730294,
255378.357216481323121,
255388.585465104522882,
255398.813749862107215,
255409.042070752766449,
255419.270427775103599,
255429.498820927867200,
255439.727250209718477,
255449.955715619522380,
255460.184217155765509,
255470.412754817196401,
255480.641328602592694,
255490.869938510528300,
255501.098584539780859,
255511.327266689040698,
255521.555984956969041,
255531.784739342314424,
255542.013529843679862,
255552.242356459872099,
255562.471219189465046,
255572.700118031294551,
255582.929052983934525,
255593.158024046162609,
255603.387031216610922,
255613.616074494057102,
255623.845153877075063,
255634.074269364500651,
255644.303420954965986,
255654.532608647132292,
255664.761832439748105,
255674.991092331416439,
255685.220388321002247,
255695.449720407108543,
255705.679088588367449,
255715.908492863585707,
255726.137933231482748,
255736.367409690574277,
255746.596922239696141,
255756.826470877596876,
255767.056055602879496,
255777.285676414234331,
255787.515333310380811,
255797.745026290067472,
255807.974755351897329,
255818.204520494618919,
255828.434321717038983,
255838.664159017644124,
255848.894032395241084,
255859.123941848549293,
255869.353887376259081,
255879.583868976973463,
255889.813886649557389,
255900.043940392555669,
255910.274030204775045,
255920.504156084876740,
255930.734318031609291,
255940.964516043546610,
255951.194750119466335,
255961.425020258117002,
255971.655326458043419,
255981.885668718168745,
255992.116047036979580,
256002.346461413282668,
256012.576911845739232,
256022.807398333156016,
256033.037920874077827,
256043.268479467311408,
256053.499074111547088,
256063.729704805387883,
256073.960371547582326,
256084.191074336908059,
256094.421813172055408,
256104.652588051627390,
256114.883398974372540,
256125.114245939039392,
256135.345128944230964,
256145.576047988753999,
256155.807003071211511,
256166.037994190410245,
256176.269021344953217,
256186.500084533647168,
256196.731183755066013,
256206.962319007987389,
256217.193490291101625,
256227.424697603128152,
256237.655940942757297,
256247.887220308708493,
256258.118535699555650,
256268.349887114221929,
256278.581274551252136,
256288.812698009336600,
256299.044157487311168,
256309.275652983749751,
256319.507184497429989,
256329.738752026984002,
256339.970355571276741,
256350.201995128794806,
256360.433670698344940,
256370.665382278646575,
256380.897129868390039,
256391.128913466294762,
256401.360733070992865,
256411.592588681232883,
256421.824480295734247,
256432.056407913187286,
256442.288371532311430,
256452.520371151855215,
256462.752406770363450,
256472.984478386642877,
256483.216585999442032,
256493.448729607422138,
256503.680909209215315,
256513.913124803599203,
256524.145376389380544,
256534.377663965104148,
256544.609987529518548,
256554.842347081314074,
256565.074742619210156,
256575.307174141955329,
256585.539641648152610,
256595.772145136666950,
256606.004684606101364,
256616.237260055146180,
256626.469871482549934,
256636.702518886973849,
256646.935202267166460,
256657.167921621818095,
256667.400676949648187,
256677.633468249288853,
256687.866295519517735,
256698.099158759083366,
256708.332057966588764,
256718.564993140840670,
256728.797964280412998,
256739.030971384199802,
256749.264014450716786,
256759.497093478741590,
256769.730208467080956,
256779.963359414279694,
256790.196546319202753,
256800.429769180365838,
256810.663027996663004,
256820.896322766668163,
256831.129653489246266,
256841.363020162942121,
256851.596422786504263,
256861.829861358652124,
256872.063335878134239,
256882.296846343611833,
256892.530392753775232,
256902.763975107372971,
256912.997593403182691,
256923.231247639778303,
256933.464937815908343,
256943.698663930350449,
256953.932425981678534,
256964.166223968728445,
256974.400057890132302,
256984.633927744667744,
256994.867833530966891,
257005.101775247836486,
257015.335752893821336,
257025.569766467844602,
257035.803815968480194,
257046.037901394389337,
257056.272022744436981,
257066.506180017255247,
257076.740373211505357,
257086.974602325935848,
257097.208867359266151,
257107.443168310244801,
257117.677505177533021,
257127.911877959762933,
257138.146286655799486,
257148.380731264303904,
257158.615211783908308,
257168.849728213419439,
257179.084280551498523,
257189.318868796835886,
257199.553492948238272,
257209.788153004250489,
257220.022848963737488,
257230.257580825360492,
257240.492348587751621,
257250.727152249804931,
257260.961991810094332,
257271.196867267368361,
257281.431778620230034,
257291.666725867602509,
257301.901709008001490,
257312.136728040321032,
257322.371782963076839,
257332.606873775133863,
257342.842000475153327,
257353.077163061825559,
257363.312361533869989,
257373.547595890035154,
257383.782866128982278,
257394.018172249401687,
257404.253514250216540,
257414.488892129797023,
257424.724305887124501,
257434.959755520831095,
257445.195241029578028,
257455.430762412113836,
257465.666319667187054,
257475.901912793458905,
257486.137541789677925,
257496.373206654592650,
257506.608907386864303,
257516.844643985125003,
257527.080416448239703,
257537.316224774869625,
257547.552068963646889,
257557.787949013407342,
257568.023864922841312,
257578.259816690610023,
257588.495804315403802,
257598.731827796029393,
257608.967887131148018,
257619.203982319450006,
257629.440113359742099,
257639.676280250685522,
257649.912482990970602,
257660.148721579345874,
257670.384996014472563,
257680.621306295099203,
257690.857652420032537,
257701.094034387817373,
257711.330452197260456,
257721.566905847110320,
257731.803395336028188,
257742.039920662704390,
257752.276481825945666,
257762.513078824413242,
257772.749711656797444,
257782.986380321905017,
257793.223084818280768,
257803.459825144789647,
257813.696601300180191,
257823.933413283055415,
257834.170261092163855,
257844.407144726254046,
257854.644064183987211,
257864.881019464111887,
257875.118010565289296,
257885.355037486384390,
257895.592100226000184,
257905.829198782885214,
257916.066333155729808,
257926.303503343311604,
257936.540709344262723,
257946.777951157389907,
257957.015228781296173,
257967.252542214788264,
257977.489891456614714,
257987.727276505378541,
257997.964697359886486,
258008.202154018828878,
258018.439646480866941,
258028.677174744865624,
258038.914738809398841,
258049.152338673244230,
258059.389974335150328,
258069.627645793807460,
258079.865353047935059,
258090.103096096165245,
258100.340874937362969,
258110.578689570189454,
258120.816539993335027,
258131.054426205548225,
258141.292348205548478,
258151.530305992026115,
258161.768299563729670,
258172.006328919378575,
258182.244394057663158,
258192.482494977390161,
258202.720631677104393,
258212.958804155670805,
258223.197012411837932,
258233.435256444150582,
258243.673536251531914,
258253.911851832555840,
258264.150203186058206,
258274.388590310612926,
258284.627013205084950,
258294.865471868135501,
258305.103966298454907,
258315.342496494791703,
258325.581062455923529,
258335.819664180424297,
258346.058301667217165,
258356.296974914788734,
258366.535683922062162,
258376.774428687669570,
258387.013209210330388,
258397.252025488793151,
258407.490877521806397,
258417.729765307973139,
258427.968688846100122,
258438.207648134935880,
258448.446643173141638,
258458.685673959495034,
258468.924740492715500,
258479.163842771435156,
258489.402980794460746,
258499.642154560453491,
258509.881364068220137,
258520.120609316509217,
258530.359890303807333,
258540.599207029154059,
258550.838559491065098,
258561.077947688318091,
258571.317371619661571,
258581.556831283785868,
258591.796326679352205,
258602.035857805225533,
258612.275424660096178,
258622.515027242567157,
258632.754665551503422,
258642.994339585537091,
258653.234049343445804,
258663.473794823919889,
258673.713576025707880,
258683.953392947529210,
258694.193245588074205,
258704.433133946149610,
258714.673058020358440,
258724.913017809507437,
258735.153013312345138,
258745.393044527474558,
258755.633111453789752,
258765.873214089893736,
258776.113352434535045,
258786.353526486462215,
258796.593736244336469,
258806.833981706964551,
258817.074262873094995,
258827.314579741330817,
258837.554932310478762,
258847.795320579229156,
258858.035744546359638,
258868.276204210560536,
258878.516699570580386,
258888.757230625138618,
258898.997797372867353,
258909.238399812631542,
258919.479037943092408,
258929.719711762940278,
258939.960421271040104,
258950.201166465936694,
258960.441947346524103,
258970.682763911347138,
258980.923616159328958,
258991.164504089043476,
259001.405427699268330,
259011.646386988752056,
259021.887381956243189,
259032.128412600402953,
259042.369478919979883,
259052.610580913722515,
259062.851718580350280,
259073.092891918553505,
259083.334100927138934,
259093.575345604767790,
259103.816625950159505,
259114.057941962033510,
259124.299293639254756,
259134.540680980397156,
259144.782103984238347,
259155.023562649526866,
259165.265056974953040,
259175.506586959294509,
259185.748152601241600,
259195.989753899542848,
259206.231390852917684,
259216.473063460056437,
259226.714771719765849,
259236.956515630707145,
259247.198295191657962,
259257.440110401279526,
259267.681961258378578,
259277.923847761645447,
259288.165769909857772,
259298.407727701647673,
259308.649721135821892,
259318.891750211158069,
259329.133814926230116,
259339.375915279902983,
259349.618051270837896,
259359.860222897754284,
259370.102430159487994,
259380.344673054671148,
259390.586951581994072,
259400.829265740350820,
259411.071615528315306,
259421.314000944694271,
259431.556421988207148,
259441.798878657544265,
259452.041370951454155,
259462.283898868714459,
259472.526462408044608,
259482.769061568164034,
259493.011696347763063,
259503.254366745648440,
259513.497072760423180,
259523.739814390952233,
259533.982591635925928,
259544.225404494063696,
259554.468252964143176,
259564.711137044825591,
259574.954056734859478,
259585.197012032993371,
259595.440002937946701,
259605.683029448468005,
259615.926091563305818,
259626.169189281121362,
259636.412322600779589,
259646.655491520854412,
259656.898696040123468,
259667.141936157480814,
259677.385211871413048,
259687.628523180785123,
259697.871870084403781,
259708.115252580813831,
259718.358670668822015,
259728.602124347293284,
259738.845613614772446,
259749.089138470095349,
259759.332698911981424,
259769.576294939120999,
259779.819926550320815,
259790.063593744242098,
259800.307296519691590,
259810.551034875359619,
259820.794808809965616,
259831.038618322287221,
259841.282463411043864,
259851.526344074925873,
259861.770260312739993,
259872.014212123147445,
259882.258199504954973,
259892.502222456852905,
259902.746280977560673,
259912.990375065855915,
259923.234504720458062,
259933.478669940057443,
259943.722870723519009,
259953.967107069445774,
259964.211378976586275,
259974.455686443805462,
259984.700029469648143,
259994.944408052950166,
260005.188822192430962,
260015.433271886839066,
260025.677757134981221,
260035.922277935402235,
260046.166834287054371,
260056.411426188598853,
260066.656053638667800,
260076.900716636067955,
260087.145415179576958,
260097.390149267885135,
260107.634918899741024,
260117.879724073922262,
260128.124564789090073,
260138.369441044022096,
260148.614352837466868,
260158.859300168114714,
260169.104283034743275,
260179.349301436130190,
260189.594355370878475,
260199.839444837911287,
260210.084569835802540,
260220.329730363329872,
260230.574926419300027,
260240.820158002403332,
260251.065425111359218,
260261.310727744887117,
260271.556065901822876,
260281.801439580798615,
260292.046848780621076,
260302.292293500067899,
260312.537773737742100,
260322.783289492479526,
260333.028840762970503,
260343.274427548021777,
260353.520049846236361,
260363.765707656479208,
260374.011400977557059,
260384.257129808014724,
260394.502894146688050,
260404.748693992354674,
260414.994529343646718,
260425.240400199429132,
260435.486306558363140,
260445.732248419226380,
260455.978225780680077,
260466.224238641530974,
260476.470287000527605,
260486.716370856389403,
260496.962490207894007,
260507.208645053702639,
260517.454835392622044,
260527.701061223342549,
260537.947322544641793,
260548.193619355297415,
260558.439951653912431,
260568.686319439351792,
260578.932722710422240,
260589.179161465610377,
260599.425635703926673,
260609.672145423857728,
260619.918690624443116,
260630.165271304140333,
260640.411887461901642,
260650.658539096359164,
260660.905226206261432,
260671.151948790327879,
260681.398706847452559,
260691.645500376180280,
260701.892329375317786,
260712.139193843671819,
260722.386093779903604,
260732.633029182790779,
260742.880000051052775,
260753.127006383496337,
260763.374048178811790,
260773.621125435718568,
260783.868238152994309,
260794.115386329416651,
260804.362569963617716,
260814.609789054520661,
260824.857043600670295,
260835.104333600960672,
260845.351659053994808,
260855.599019958637655,
260865.846416313579539,
260876.093848117539892,
260886.341315369354561,
260896.588818067684770,
260906.836356211279053,
260917.083929798944155,
260927.331538829341298,
260937.579183301306330,
260947.826863213442266,
260958.074578564584954,
260968.322329353570240,
260978.570115578913828,
260988.817937239538878,
260999.065794334164821,
261009.313686861511087,
261019.561614820355317,
261029.809578209329629,
261040.057577027298976,
261050.305611272982787,
261060.553680945158703,
261070.801786042429740,
261081.049926563660847,
261091.298102507600561,
261101.546313872910105,
261111.794560658483533,
261122.042842862923862,
261132.291160485037835,
261142.539513523573987,
261152.787901977251749,
261163.036325844848761,
261173.284785124968039,
261183.533279816649156,
261193.781809918349609,
261204.030375428963453,
261214.278976347268326,
261224.527612671867246,
261234.776284401654266,
261245.024991535290610,
261255.273734071524814,
261265.522512009105412,
261275.771325346810045,
261286.020174083416350,
261296.269058217585552,
261306.517977748124395,
261316.766932673723204,
261327.015922993217828,
261337.264948705327697,
261347.514009808655828,
261357.763106302183587,
261368.012238184484886,
261378.261405454395572,
261388.510608110693283,
261398.759846151981037,
261409.009119577152887,
261419.258428384811850,
261429.507772573881084,
261439.757152142992709,
261450.006567090866156,
261460.256017416395480,
261470.505503118183697,
261480.755024195095757,
261491.004580645763781,
261501.254172469023615,
261511.503799663594691,
261521.753462228196440,
261532.003160161664709,
261542.252893462602515,
261552.502662129874807,
261562.752466162288329,
261573.002305558387889,
261583.252180317125749,
261593.502090437163133,
261603.752035917132162,
261614.002016756072408,
261624.252032952470472,
261634.502084505220409,
261644.752171412983444,
261655.002293674595421,
261665.252451288659358,
261675.502644254098414,
261685.752872569602914,
261696.003136233892292,
261706.253435245802393,
261716.503769603936234,
261726.754139307187870,
261737.004544354160316,
261747.254984743718524,
261757.505460474611027,
261767.755971545557259,
261778.006517955334857,
261788.257099702663254,
261798.507716786320088,
261808.758369204966584,
261819.009056957467692,
261829.259780042571947,
261839.510538458940573,
261849.761332205467625,
261860.012161280697910,
261870.263025683554588,
261880.513925412786193,
261890.764860467053950,
261901.015830845193705,
261911.266836545866681,
261921.517877567850519,
261931.768953909981064,
261942.020065570948645,
261952.271212549501797,
261962.522394844359951,
261972.773612454358954,
261983.024865378218237,
261993.276153614715440,
262003.527477162453579,
262013.778836020414019,
262024.030230187228881,
262034.281659661559388,
262044.533124442357803,
262054.784624528227141,
262065.036159918061458,
262075.287730610463768,
262085.539336604269920,
262095.790977898141136,
262106.042654490942368,
262116.294366381480359,
262126.546113568299916,
262136.797896050353302,
262147.049713826272637,
262157.301566894922871,
262167.553455254936125,
262177.805378905148245,
262188.057337844278663,
262198.309332071105018,
262208.561361584288534,
262218.813426382781472,
262229.065526465128642,
262239.317661830224097,
262249.569832476729061,
262259.822038403479382,
262270.074279609310906,
262280.326556092710234,
262290.578867852629628,
262300.831214887788519,
262311.083597196964547,
262321.336014778877143,
262331.588467632303946,
262341.840955755964387,
262352.093479148577899,
262362.346037809096742,
262372.598631736065727,
262382.851260928320698,
262393.103925384581089,
262403.356625103740953,
262413.609360084345099,
262423.862130325287580,
262434.114935825287830,
262444.367776583123486,
262454.620652597630396,
262464.873563867353369,
262475.126510391186457,
262485.379492167907301,
262495.632509196177125,
262505.885561474889982,
262516.138649002707098,
262526.391771778406110,
262536.644929800706450,
262546.898123068443965,
262557.151351580338087,
262567.404615335050039,
262577.657914331473876,
262587.911248568329029,
262598.164618044509552,
262608.418022758502048,
262618.671462709200568,
262628.924937895324547,
262639.178448315768037,
262649.431993969134055,
262659.685574854258448,
262669.939190969860647,
262680.192842314718291,
262690.446528887609020,
262700.700250687252264,
262710.954007712367456,
262721.207799961848650,
262731.461627434415277,
262741.715490128728561,
262751.969388043624349,
262762.223321177822072,
262772.477289530215785,
262782.731293099350296,
262792.985331884061452,
262803.239405883185100,
262813.493515095440671,
262823.747659519664012,
262834.001839154399931,
262844.256053998600692,
262854.510304050985724,
262864.764589310332667,
262875.018909775302745,
262885.273265444731805,
262895.527656317339279,
262905.782082392019220,
262916.036543667374644,
262926.291040142241400,
262936.545571815338917,
262946.800138685386628,
262957.054740751395002,
262967.309378011792433,
262977.564050465472974,
262987.818758111272473,
262998.073500947910361,
263008.328278974106070,
263018.583092188637238,
263028.837940590281505,
263039.092824177816510,
263049.347742949961685,
263059.602696905494668,
263069.857686043193098,
263080.112710361776408,
263090.367769860196859,
263100.622864536882844,
263110.877994390844833,
263121.133159420744050,
263131.388359625358135,
263141.643595003522933,
263151.898865553957876,
263162.154171275324188,
263172.409512166515924,
263182.664888226252515,
263192.920299453311600,
263203.175745846412610,
263213.431227404333185,
263223.686744125967380,
263233.942296009860002,
263244.197883054905105,
263254.453505259822123,
263264.709162623388693,
263274.964855144440662,
263285.220582821639255,
263295.476345653703902,
263305.732143639586866,
263315.987976777891163,
263326.243845067452639,
263336.499748506990727,
263346.755687095283065,
263357.011660831223708,
263367.267669713357463,
263377.523713740520179,
263387.779792911547702,
263398.035907225159463,
263408.292056680191308,
263418.548241275246255,
263428.804461009276565,
263439.060715880885255,
263449.317005889024585,
263459.573331032239366,
263469.829691309423652,
263480.086086719296873,
263490.342517260694876,
263500.598982932337094,
263510.855483732942957,
263521.112019661406521,
263531.368590716330800,
263541.625196896609850,
263551.881838200963102,
263562.138514628168195,
263572.395226176944561,
263582.651972846186254,
263592.908754634438083,
263603.165571540652309,
263613.422423563548364,
263623.679310701903887,
263633.936232954496518,
263644.193190319987480,
263654.450182797212619,
263664.707210385065991,
263674.964273082092404,
263685.221370887185913,
263695.478503799124155,
263705.735671816568356,
263715.992874938470777,
263726.250113163434435,
263736.507386490295175,
263746.764694917714223,
263757.022038444702048,
263767.279417069745250,
263777.536830791854300,
263787.794279609574005,
263798.051763521856628,
263808.309282527421601,
263818.566836624930147,
263828.824425813276321,
263839.082050091295969,
263849.339709457475692,
263859.597403910825960,
263869.855133450066205,
263880.112898073915858,
263890.370697781268973,
263900.628532570728566,
263910.886402441130485,
263921.144307391194161,
263931.402247419871856,
263941.660222525766585,
263951.918232707655989,
263962.176277964317705,
263972.434358294645790,
263982.692473697243258,
263992.950624170945957,
264003.208809714531526,
264013.467030326777603,
264023.725286006461829,
264033.983576752303634,
264044.241902563080657,
264054.500263437628746,
264064.758659374609124,
264075.017090372915845,
264085.275556431268342,
264095.534057548444252,
264105.792593723163009,
264116.051164954202250,
264126.309771240397822,
264136.568412580527365,
264146.827088973310310,
264157.085800417466089,
264167.344546911888756,
264177.603328455297742,
264187.862145046470687,
264198.120996684185229,
264208.379883367102593,
264218.638805094175041,
264228.897761864063796,
264239.156753675488289,
264249.415780527400784,
264259.674842418404296,
264269.933939347392879,
264280.193071313085966,
264290.452238314202987,
264300.711440349521581,
264310.970677417994011,
264321.229949518165085,
264331.489256648870651,
264341.748598809004761,
264352.007975997228641,
264362.267388212319929,
264372.526835453056265,
264382.786317718157079,
264393.045835006632842,
264403.305387316911947,
264413.564974648004863,
264423.824596998631023,
264434.084254367568064,
264444.343946753535420,
264454.603674155368935,
264464.863436571846250,
264475.123234001686797,
264485.383066443726420,
264495.642933896626346,
264505.902836359338835,
264516.162773830466904,
264526.422746308904607,
264536.682753793371376,
264546.942796282644849,
264557.202873775502667,
264567.462986270722467,
264577.723133767081890,
264587.983316263358574,
264598.243533758271951,
264608.503786250774283,
264618.764073739410378,
264629.024396223074291,
264639.284753700485453,
264649.545146170596126,
264659.805573631951120,
264670.066036083444487,
264680.326533523795661,
264690.587065951840486,
264700.847633366298396,
264711.108235766005237,
264721.368873149738647,
264731.629545516218059,
264741.890252864221111,
264752.150995192583650,
264762.411772499966901,
264772.672584785323124,
264782.933432047313545,
264793.194314284657594,
264803.455231496307533,
264813.716183680924587,
264823.977170837228186,
264834.238192964112386,
264844.499250060296617,
264854.760342124674935,
264865.021469155792147,
264875.282631152600516,
264885.543828113819472,
264895.805060038284864,
264906.066326924657915,
264916.327628771774471,
264926.588965578470379,
264936.850337343523279,
264947.111744065652601,
264957.373185743635986,
264967.634662376192864,
264977.896173962275498,
264988.157720500545111,
264998.419301989721134,
265008.680918428755831,
265018.942569816310424,
265029.204256151162554,
265039.465977432089858,
265049.727733657811768,
265059.989524827338755,
265070.251350939215627,
265080.513211992336437,
265090.775107985420618,
265101.037038917362224,
265111.299004786764272,
265121.561005592520814,
265131.823041333351284,
265142.085112008091528,
265152.347217615519185,
265162.609358154353686,
265172.871533623430878,
265183.133744021528400,
265193.395989347423892,
265203.658269599836785,
265213.920584777602926,
265224.182934879616369,
265234.445319904421922,
265244.707739850913640,
265254.970194717869163,
265265.232684504182544,
265275.495209208398592,
265285.757768829469569,
265296.020363366114907,
265306.282992817170452,
265316.545657181413844,
265326.808356457506306,
265337.071090644400101,
265347.333859740698244,
265357.596663745294791,
265367.859502656967379,
265378.122376474493649,
265388.385285196651239,
265398.648228822159581,
265408.911207349854521,
265419.174220778630115,
265429.437269107089378,
265439.700352334068157,
265449.963470458344091,
265460.226623478811234,
265470.489811394014396,
265480.753034202964045,
265491.016291904321406,
265501.279584496980533,
265511.542911979544442,
265521.806274351023603,
265532.069671609962825,
265542.333103755372576,
265552.596570785855874,
265562.860072700190358,
265573.123609497328289,
265583.387181175930891,
265593.650787734834012,
265603.914429172698874,
265614.178105488477740,
265624.441816680948250,
265634.705562748771627,
265644.969343690667301,
265655.233159505703952,
265665.497010192426387,
265675.760895749670453,
265686.024816176271997,
265696.288771471066866,
265706.552761632658076,
265716.816786659939680,
265727.080846551689319,
265737.344941306684632,
265747.609070923761465,
265757.873235401522834,
265768.137434738979209,
265778.401668934908230,
265788.665937987912912,
265798.930241896829102,
265809.194580660550855,
265819.458954277855810,
265829.723362747405190,
265839.987806068034843,
265850.252284238580614,
265860.516797257820144,
265870.781345124472864,
265881.045927837316412,
265891.310545395244844,
265901.575197797035798,
265911.839885041408706,
265922.104607127083000,
265932.369364053069148,
265942.634155817911960,
265952.898982420447282,
265963.163843859569170,
265973.428740134113468,
265983.693671242624987,
265993.958637184055988,
266004.223637957125902,
266014.488673560786992,
266024.753743993584067,
266035.018849254411180,
266045.283989342104178,
266055.549164255440701,
266065.814373993081972,
266076.079618553980254,
266086.344897936796769,
266096.610212140483782,
266106.875561163644306,
266117.140945005114190,
266127.406363663787488,
266137.671817138267215,
266147.937305427505635,
266158.202828530222178,
266168.468386445252690,
266178.733979171316605,
266188.999606707133353,
266199.265269051713403,
266209.530966203717981,
266219.796698161924724,
266230.062464925053064,
266240.328266492055263,
266250.594102861708961,
266260.859974032558966,
266271.125880003673956,
266281.391820773773361,
266291.657796341576613,
266301.923806705861352,
266312.189851865463424,
266322.455931819276884,
266332.722046565846540,
266342.988196104182862,
266353.254380432947073,
266363.520599551091436,
266373.786853457160760,
266384.053142150107305,
266394.319465628650505,
266404.585823891742621,
266414.852216937928461,
266425.118644766160287,
266435.385107375215739,
266445.651604763814248,
266455.918136930733453,
266466.184703874925617,
266476.451305594993755,
266486.717942089831922,
266496.984613358217757,
266507.251319398987107,
266517.518060210801195,
266527.784835792554077,
266538.051646143023390,
266548.318491260928567,
266558.585371145163663,
266568.852285794448107,
266579.119235207617749,
266589.386219383450225,
266599.653238320723176,
266609.920292018214241,
266620.187380474701058,
266630.454503689135890,
266640.721661660121754,
266650.988854386552703,
266661.256081867089961,
266671.523344100685790,
266681.790641086117830,
266692.057972821989097,
266702.325339307368267,
266712.592740540858358,
266722.860176521295216,
266733.127647247456480,
266743.395152718236204,
266753.662692932295613,
266763.930267888412345,
266774.197877585538663,
266784.465522022452205,
266794.733201197697781,
266805.000915110344067,
266815.268663759110495,
266825.536447142658290,
266835.804265259997919,
266846.072118109732401,
266856.340005690755788,
266866.607928001845721,
266876.875885041779839,
266887.143876809335779,
266897.411903303407598,
266907.679964522656519,
266917.948060465918388,
266928.216191132087260,
266938.484356519882567,
266948.752556627965532,
266959.020791455346625,
266969.289061000745278,
266979.557365262880921,
266989.825704240589403,
267000.094077932764776,
267010.362486338126473,
267020.630929455393925,
267030.899407283461187,
267041.167919821164105,
267051.436467067163903,
267061.705049020354636,
267071.973665679397527,
267082.242317043361254,
267092.511003110732418,
267102.779723880463280,
267113.048479351331480,
267123.317269522172865,
267133.586094391765073,
267143.854953958827537,
267154.123848222254310,
267164.392777180764824,
267174.661740833136719,
267184.930739178264048,
267195.199772214982659,
267205.468839941895567,
267215.737942357896827,
267226.007079461938702,
267236.276251252565999,
267246.545457728672773,
267256.814698889036663,
267267.083974732609931,
267277.353285257937387,
267287.622630463913083,
267297.892010349431075,
267308.161424913327210,
267318.430874154146295,
267328.700358070840593,
267338.969876662362367,
267349.239429927140009,
267359.509017864300404,
267369.778640472446568,
267380.048297750530764,
267390.317989697156008,
267400.587716311274562,
267410.857477591722272,
267421.127273537160363,
267431.397104146424681,
267441.666969418351073,
267451.936869351775385,
267462.206803945417050,
267472.476773198053706,
267482.746777108579408,
267493.016815675713588,
267503.286888898292091,
267513.556996775092557,
267523.827139304950833,
267534.097316486644559,
267544.367528318951372,
267554.637774800707120,
267564.908055930747651,
267575.178371707792394,
267585.448722130618989,
267595.719107198179699,
267605.989526909077540,
267616.259981262206566,
267626.530470256402623,
267636.800993890443351,
267647.071552163106389,
267657.342145073169377,
267667.612772619526368,
267677.883434800896794,
267688.154131616000086,
267698.424863063846715,
267708.695629143156111,
267718.966429852589499,
267729.237265191157348,
267739.508135157520883,
267749.779039750515949,
267760.049978968978394,
267770.320952811627649,
267780.591961277415976,
267790.863004364946391,
267801.134082073171157,
267811.405194400867913,
267821.676341346756089,
267831.947522909729742,
267842.218739088508300,
267852.489989882044028,
267862.761275288881734,
267873.032595308031887,
267883.303949938330334,
267893.575339178380091,
267903.846763027133420,
267914.118221483367961,
267924.389714545861352,
267934.661242213391233,
267944.932804484793451,
267955.204401358962059,
267965.476032834500074,
267975.747698910359759,
267986.019399585260544,
267996.291134858154692,
268006.562904727645218,
268016.834709192626178,
268027.106548251991626,
268037.378421904344577,
268047.650330148695502,
268057.922272983647417,
268068.194250408210792,
268078.466262421046849,
268088.738309020933229,
268099.010390206880402,
268109.282505977433175,
268119.554656331543811,
268129.826841267989948,
268140.099060785607435,
268150.371314883115701,
268160.643603559408803,
268170.915926813206170,
268181.188284643343650,
268191.460677048715297,
268201.733104027982336,
268212.005565580097027,
268222.278061703720596,
268232.550592397688888,
268242.823157660954166,
268253.095757492119446,
268263.368391890078783,
268273.641060853551608,
268283.913764381606597,
268294.186502472788561,
268304.459275125933345,
268314.732082339993212,
268325.004924113571178,
268335.277800445619505,
268345.550711334974039,
268355.823656780296005,
268366.096636780479457,
268376.369651334302034,
268386.642700440657791,
268396.915784098149743,
268407.188902305788361,
268417.462055062351283,
268427.735242366557941,
268438.008464217244182,
268448.281720613129437,
268458.555011553282384,
268468.828337036306038,
268479.101697060978040,
268489.375091626308858,
268499.648520730843302,
268509.921984373533633,
268520.195482553157490,
268530.469015268608928,
268540.742582518549170,
268551.016184301930480,
268561.289820617472287,
268571.563491463952232,
268581.837196840206161,
268592.110936745128129,
268602.384711177437566,
268612.658520135912113,
268622.932363619445823,
268633.206241626874544,
268643.480154156801291,
268653.754101208236534,
268664.028082780016121,
268674.302098870743066,
268684.576149479369633,
268694.850234604615252,
268705.124354245490395,
268715.398508400539868,
268725.672697068774141,
268735.946920248970855,
268746.221177939791232,
268756.495470140129328,
268766.769796848879196,
268777.044158064760268,
268787.318553786550183,
268797.592984013142996,
268807.867448743374553,
268818.141947975906078,
268828.416481709748041,
268838.691049943561666,
268848.965652676124591,
268859.240289906330872,
268869.514961633016355,
268879.789667854900472,
268890.064408570935484,
268900.339183779724408,
268910.613993480277713,
268920.888837671256624,
268931.163716351555195,
268941.438629520067479,
268951.713577175338287,
268961.988559316436294,
268972.263575942022726,
268982.538627050991636,
268992.813712642120663,
269003.088832714187447,
269013.363987266202457,
269023.639176296652295,
269033.914399804489221,
269044.189657788665500,
269054.464950247784145,
269064.740277180797420,
269075.015638586424757,
269085.291034463560209,
269095.566464810981415,
269105.841929627407808,
269116.117428911849856,
269126.392962662910577,
269136.668530879542232,
269146.944133560522459,
269157.219770704628900,
269167.495442310639191,
269177.771148377505597,
269188.046888904005755,
269198.322663888800889,
269208.598473330843262,
269218.874317228968721,
269229.150195581838489,
269239.426108388404828,
269249.702055647445377,
269259.978037357795984,
269270.254053518176079,
269280.530104127479717,
269290.806189184484538,
269301.082308688026387,
269311.358462636882905,
269321.634651030006353,
269331.910873865999747,
269342.187131143815350,
269352.463422862230800,
269362.739749020023737,
269373.016109616088215,
269383.292504649085458,
269393.568934118084144,
269403.845398021687288,
269414.121896358788945,
269424.398429128224961,
269434.674996328656562,
269444.951597959094215,
269455.228234018315561,
269465.504904504981823,
269475.781609418045264,
269486.058348756341729,
269496.335122518648859,
269506.611930703744292,
269516.888773310463876,
269527.165650337643456,
269537.442561784060672,
269547.719507648551371,
269557.996487930009607,
269568.273502627096605,
269578.550551738706417,
269588.827635263733100,
269599.104753200837877,
269609.381905548856594,
269619.659092306741513,
269629.936313473270275,
269640.213569047220517,
269650.490859027311672,
269660.768183412554208,
269671.045542201551143,
269681.322935393254738,
269691.600362986559048,
269701.877824980067089,
269712.155321372731123,
269722.432852163386997,
269732.710417350812349,
269742.988016933784820,
269753.265650911198463,
269763.543319281830918,
269773.821022044401616,
269784.098759197921026,
269794.376530741050374,
269804.654336672741920,
269814.932176991656888,
269825.210051696747541,
269835.487960786791518,
269845.765904260508250,
269856.043882116850000,
269866.321894354594406,
269876.599940972577315,
269886.878021969518159,
269897.156137344369199,
269907.434287095849868,
269917.712471222796012,
269927.990689724101685,
269938.268942598486319,
269948.547229844785761,
269958.825551461894065,
269969.103907448588870,
269979.382297803647816,
269989.660722525906749,
269999.939181614259724,
270010.217675067367963,
270020.496202884183731,
270030.774765063601080,
270041.053361604164820,
270051.331992504885420,
270061.610657764656935,
270071.889357382082380,
270082.168091356172226,
270092.446859685645904,
270102.725662369339261,
270113.004499406088144,
270123.283370794728398,
270133.562276533979457,
270143.841216622793581,
270154.120191059831996,
270164.399199844163377,
270174.678242974390741,
270184.957320449408144,
270195.236432268051431,
270205.515578429098241,
270215.794758931384422,
270226.073973773804028,
270236.353222955076490,
270246.632506474037655,
270256.911824329581577,
270267.191176520427689,
270277.470563045470044,
270287.749983903544489,
270298.029439093428664,
270308.308928613900207,
270318.588452463794965,
270328.868010642065201,
270339.147603147372138,
270349.427229978668038,
270359.706891134672333,
270369.986586614279076,
270380.266316416207701,
270390.546080539410468,
270400.825878982606810,
270411.105711744632572,
270421.385578824440017,
270431.665480220632162,
270441.945415932219476,
270452.225385957979597,
270462.505390296690166,
270472.785428947128821,
270483.065501908189617,
270493.345609178824816,
270503.625750757579226,
270513.905926643405110,
270524.186136835254729,
270534.466381331731100,
270544.746660131786484,
270555.026973234198522,
270565.307320637861267,
270575.587702341494150,
270585.868118343991227,
270596.148568644188344,
270606.429053240804933,
270616.709572132851463,
270626.990125318930950,
270637.270712797995657,
270647.551334568939637,
270657.831990630424116,
270668.112680981343146,
270678.393405620474368,
270688.674164546770044,
270698.954957758949604,
270709.235785255907103,
270719.516647036361974,
270729.797543099208269,
270740.078473443281837,
270750.359438067418523,
270760.640436970337760,
270770.921470150991809,
270781.202537608158309,
270791.483639340673108,
270801.764775347255636,
270812.045945626974571,
270822.327150178374723,
270832.608389000408351,
270842.889662092027720,
270853.170969451777637,
270863.452311078726780,
270873.733686971652787,
270884.015097129275091,
270894.296541550487746,
270904.578020234126598,
270914.859533179027494,
270925.141080383968074,
270935.422661847784184,
270945.704277569369879,
270955.985927547502797,
270966.267611780960578,
270976.549330268695485,
270986.831083009368740,
270997.112870001932606,
271007.394691245222930,
271017.676546738017350,
271027.958436479093507,
271038.240360467345454,
271048.522318701609038,
271058.804311180603690,
271069.086337903339881,
271079.368398868478835,
271089.650494074972812,
271099.932623521657661,
271110.214787207136396,
271120.496985130477697,
271130.779217290459201,
271141.061483685800340,
271151.343784315453377,
271161.626119178137742,
271171.908488272863906,
271182.190891598234884,
271192.473329153202940,
271202.755800936662126,
271213.038306947273668,
271223.320847183989827,
271233.603421645588242,
271243.886030330846552,
271254.168673238658812,
271264.451350367977284,
271274.734061717346776,
271285.016807285835966,
271295.299587072222494,
271305.582401075225789,
271315.865249293798115,
271326.148131726717111,
271336.431048372818623,
271346.713999230880290,
271356.996984299854375,
271367.280003578460310,
271377.563057065592147,
271387.846144760085735,
271398.129266660660505,
271408.412422766268719,
271418.695613075688016,
271428.978837587754242,
271439.262096301303245,
271449.545389215229079,
271459.828716328192968,
271470.112077639147174,
271480.395473146927543,
271490.678902850369923,
271500.962366748193745,
271511.245864839409478,
271521.529397122678347,
271531.812963596952613,
271542.096564260951709,
271552.380199113627896,
271562.663868153700605,
271572.947571380122099,
271583.231308791670017,
271593.515080387121998,
271603.798886165313888,
271614.082726125139743,
271624.366600265435409,
271634.650508584920317,
271644.934451082604937,
271655.218427757150494,
271665.502438607451040,
271675.786483632458840,
271686.070562830835115,
271696.354676201415714,
271706.638823743152898,
271716.923005454824306,
271727.207221335207578,
271737.491471383196767,
271747.775755597627722,
271758.060073977278080,
271768.344426521100104,
271778.628813227813225,
271788.913234096253291,
271799.197689125314355,
271809.482178313774057,
271819.766701660526451,
271830.051259164290968,
271840.335850824078079,
271850.620476638490800,
271860.905136606656015,
271871.189830727118533,
271881.474558998830616,
271891.759321420686319,
271902.044117991463281,
271912.328948709997348,
271922.613813575124368,
271932.898712585621979,
271943.183645740442444,
271953.468613038363401,
271963.753614478162490,
271974.038650058733765,
271984.323719778913073,
271994.608823637536261,
272004.893961633439176,
272015.179133765399456,
272025.464340032311156,
272035.749580433010124,
272046.034854966332205,
272056.320163630996831,
272066.605506426014472,
272076.890883350162767,
272087.176294402161147,
272097.461739581020083,
272107.747218885517213,
272118.032732314430177,
272128.318279866594821,
272138.603861540963408,
272148.889477336197160,
272159.175127251248341,
272169.460811285011005,
272179.746529436146375,
272190.032281703664921,
272200.318068086227868,
272210.603888582845684,
272220.889743192179594,
272231.175631913298275,
272241.461554744804744,
272251.747511685651261,
272262.033502734673675,
272272.319527890707832,
272282.605587152473163,
272292.891680519038346,
272303.177807989006396,
272313.463969561329577,
272323.750165234901942,
272334.036395008442923,
272344.322658880788367,
272354.608956850890536,
272364.895288917468861,
272375.181655079417396,
272385.468055335630197,
272395.754489684826694,
272406.040958125900943,
272416.327460657746997,
272426.613997279084288,
272436.900567988806870,
272447.187172785808798,
272457.473811668867711,
272467.760484636761248,
272478.047191688558087,
272488.333932822803035,
272498.620708038506564,
272508.907517334504519,
272519.194360709516332,
272529.481238162494265,
272539.768149692274164,
272550.055095297691878,
272560.342074977466837,
272570.629088730667718,
272580.916136555839330,
272591.203218452108558,
272601.490334418136626,
272611.777484452817589,
272622.064668554987293,
272632.351886723481584,
272642.639138957136311,
272652.926425254787318,
272663.213745615386870,
272673.501100037537981,
272683.788488520251121,
272694.075911062362138,
272704.363367662648670,
272714.650858320004772,
272724.938383033266291,
272735.225941801210865,
272745.513534622674342,
272755.801161496667191,
272766.088822421850637,
272776.376517397118732,
272786.664246421249118,
272796.952009493194055,
272807.239806611847598,
272817.527637775812764,
272827.815502984100021,
272838.103402235603426,
272848.391335528984200,
272858.679302863252815,
272868.967304237186909,
272879.255339649505913,
272889.543409099278506,
272899.831512585165910,
272910.119650106120389,
272920.407821660861373,
272930.696027248399332,
272940.984266867453698,
272951.272540516860317,
272961.560848195571452,
272971.849189902306534,
272982.137565635901410,
272992.425975395366549,
273002.714419179363176,
273013.002896986785345,
273023.291408816585317,
273033.579954667424317,
273043.868534538254607,
273054.157148427970242,
273064.445796335232444,
273074.734478258993477,
273085.023194198205601,
273095.311944151471835,
273105.600728117860854,
273115.889546096033882,
273126.178398085001390,
273136.467284083482809,
273146.756204090372194,
273157.045158104505390,
273167.334146124776453,
273177.623168149904814,
273187.912224178842735,
273198.201314210426062,
273208.490438243374228,
273218.779596276697703,
273229.068788309232332,
273239.358014339639340,
273249.647274366929196,
273259.936568389937747,
273270.225896407384425,
273280.515258418279700,
273290.804654421401210,
273301.094084415526595,
273311.383548399608117,
273321.673046372423414,
273331.962578332866542,
273342.252144279656932,
273352.541744211805053,
273362.831378128030337,
273373.121046027285047,
273383.410747908346821,
273393.700483770051505,
273403.990253611234948,
273414.280057430849411,
273424.569895227614325,
273434.859767000481952,
273445.149672748229932,
273455.439612469635904,
273465.729586163710337,
273476.019593829172663,
273486.309635464916937,
273496.599711069837213,
273506.889820642652921,
273517.179964182316326,
273527.470141687663272,
273537.760353157471400,
273548.050598590634763,
273558.340877986047417,
273568.631191342428792,
273578.921538658731151,
273589.211919933848549,
273599.502335166442208,
273609.792784355580807,
273620.083267499925569,
273630.373784598428756,
273640.664335649809800,
273650.954920653137378,
273661.245539607072715,
273671.536192510626279,
273681.826879362401087,
273692.117600161465816,
273702.408354906598106,
273712.699143596633803,
273722.989966230350547,
273733.280822806758806,
273743.571713324578013,
273753.862637782702222,
273764.153596179967280,
273774.444588515267242,
273784.735614787321538,
273795.026674995140638,
273805.317769137443975,
273815.608897213183809,
273825.900059221195988,
273836.191255160258152,
273846.482485029264353,
273856.773748827050440,
273867.065046552394051,
273877.356378204363864,
273887.647743781562895,
273897.939143282885198,
273908.230576707341243,
273918.522044053650461,
273928.813545320648700,
273939.105080507288221,
273949.396649612288456,
273959.688252634601668,
273969.979889573121909,
273980.271560426452197,
273990.563265193661209,
274000.855003873584792,
274011.146776465058792,
274021.438582966919057,
274031.730423377885018,
274042.022297696967144,
274052.314205923059490,
274062.606148054823279,
274072.898124091268983,
274083.190134031232446,
274093.482177873433102,
274103.774255616823211,
274114.066367260355037,
274124.358512802631594,
274134.650692242663354,
274144.942905579344369,
274155.235152811394073,
274165.527433937764727,
274175.819748957292177,
274186.112097868754063,
274196.404480671102647,
274206.696897363057360,
274216.989347943570465,
274227.281832411536016,
274237.574350765731651,
274247.866903004935011,
274258.159489128214773,
274268.452109134232160,
274278.744763021881226,
274289.037450790114235,
274299.330172437650617,
274309.622927963326219,
274319.915717366151512,
274330.208540644904133,
274340.501397798419930,
274350.794288825476542,
274361.087213724968024,
274371.380172495963052,
274381.673165137006436,
274391.966191647050437,
274402.259252025047317,
274412.552346269774716,
274422.845474380010273,
274433.138636354822665,
274443.431832192814909,
274453.725061892939266,
274464.018325454206206,
274474.311622875218745,
274484.604954154929146,
274494.898319292231463,
274505.191718285961542,
274515.485151134955231,
274525.778617838106584,
274536.072118394135032,
274546.365652802109253,
274556.659221060690470,
274566.952823168772738,
274577.246459125366528,
274587.540128929133061,
274597.833832579024602,
274608.127570073876996,
274618.421341412467882,
274628.715146593865938,
274639.008985616615973,
274649.302858479786664,
274659.596765182272065,
274669.890705722791608,
274680.184680100239348,
274690.478688313567545,
274700.772730361437425,
274711.066806242801249,
274721.360915956669487,
274731.655059501586948,
274741.949236876680516,
274752.243448080669623,
274762.537693112390116,
274772.831971970910672,
274783.126284654776100,
274793.420631162996870,
274803.715011494467035,
274814.009425648022443,
274824.303873622382525,
274834.598355416674167,
274844.892871029500384,
274855.187420459871646,
274865.482003706507385,
274875.776620768418070,
274886.071271644323133,
274896.365956333174836,
274906.660674833750818,
274916.955427145061549,
274927.250213265768252,
274937.545033194823191,
274947.839886931062210,
274958.134774473437574,
274968.429695820668712,
274978.724650971649680,
274989.019639925274532,
274999.314662680379115,
275009.609719235857483,
275019.904809590487275,
275030.199933743220754,
275040.495091692835558,
275050.790283438167535,
275061.085508978227153,
275071.380768311733846,
275081.676061437639873,
275091.971388354664668,
275102.266749061760493,
275112.562143557879608,
275122.857571841683239,
275133.153033912065439,
275143.448529768094886,
275153.744059408374596,
275164.039622831915040,
275174.335220037552062,
275184.630851024005096,
275194.926515790401027,
275205.222214335342869,
275215.517946657782886,
275225.813712756673340,
275236.109512630733661,
275246.405346278857905,
275256.701213699998334,
275266.997114892816171,
275277.293049856438302,
275287.589018589467742,
275297.885021090914961,
275308.181057359674014,
275318.477127394464333,
275328.773231194296386,
275339.069368757889606,
275349.365540084138047,
275359.661745171935763,
275369.957984020176809,
275380.254256627697032,
275390.550562993332278,
275400.846903115860187,
275411.143276994291227,
275421.439684627461247,
275431.736126014206093,
275442.032601153245196,
275452.329110043705441,
275462.625652684306260,
275472.922229073883500,
275483.218839211331215,
275493.515483095485251,
275503.812160725297872,
275514.108872099488508,
275524.405617217009421,
275534.702396076754667,
275544.999208677443676,
275555.296055018086918,
275565.592935097520240,
275575.889848914579488,
275586.186796468100511,
275596.483777756977361,
275606.780792779987678,
275617.077841536200140,
275627.374924024275970,
275637.672040243051015,
275647.969190191652160,
275658.266373868682422,
275668.563591273152269,
275678.860842403781135,
275689.158127259521279,
275699.455445839324966,
275709.752798141853418,
275720.050184166117106,
275730.347603910951875,
275740.645057375251781,
275750.942544557794463,
275761.240065457415767,
275771.537620073126163,
275781.835208403761499,
275792.132830448041204,
275802.430486204859335,
275812.728175673284568,
275823.025898851978127,
275833.323655739834066,
275843.621446335804649,
275853.919270638609305,
275864.217128647316713,
275874.515020360588096,
275884.812945777317509,
275895.110904896515422,
275905.408897716901265,
275915.706924237485509,
275926.004984456929378,
275936.303078374243341,
275946.601205988263246,
275956.899367297766730,
275967.197562301764265,
275977.495790999091696,
275987.794053388526663,
275998.092349469021428,
276008.390679239353631,
276018.689042698359117,
276028.987439845106564,
276039.285870678315405,
276049.584335196763277,
276059.882833399577066,
276070.181365285359789,
276080.479930853121914,
276090.778530101699289,
276101.077163029927760,
276111.375829636701383,
276121.674529920856003,
276131.973263881285675,
276142.272031516942661,
276152.570832826546393,
276162.869667808990926,
276173.168536463170312,
276183.467438787920401,
276193.766374782193452,
276204.065344444767106,
276214.364347774593625,
276224.663384770450648,
276234.962455431232229,
276245.261559755890630,
276255.560697743087076,
276265.859869391832035,
276276.159074701077770,
276286.458313669485506,
276296.757586296123918,
276307.056892579712439,
276317.356232519145124,
276327.655606113432441,
276337.955013361119200,
276348.254454261390492,
276358.553928813082166,
276368.853437014855444,
276379.152978865720797,
276389.452554364572279,
276399.752163510245737,
276410.051806301577017,
276420.351482737401966,
276430.651192816672847,
276440.950936538225506,
276451.250713900953997,
276461.550524903635960,
276471.850369545281865,
276482.150247824669350,
276492.450159740634263,
276502.750105292128865,
276513.050084477930795,
276523.350097297050525,
276533.650143748207483,
276543.950223830412142,
276554.250337542383932,
276564.550484883016907,
276574.850665851321537,
276585.150880445959046,
276595.451128665939905,
276605.751410510216374,
276616.051725977391470,
276626.352075066650286,
276636.652457776595838,
276646.952874106180388,
276657.253324054356199,
276667.553807619900908,
276677.854324801766779,
276688.154875598731451,
276698.455460009630769,
276708.756078033533413,
276719.056729669158813,
276729.357414915342815,
276739.658133771095891,
276749.958886235137470,
276760.259672306478024,
276770.560491983836982,
276780.861345266224816,
276791.162232152535580,
276801.463152641430497,
276811.764106731920037,
276822.065094422956463,
276832.366115713259205,
276842.667170601838734,
276852.968259087414481,
276863.269381168880500,
276873.570536845305469,
276883.871726115292404,
276894.172948977909982,
276904.474205431877635,
276914.775495476205833,
276925.076819109730422,
276935.378176331170835,
276945.679567139595747,
276955.980991533841006,
276966.282449512684252,
276976.583941075194161,
276986.885466219973750,
276997.187024946033489,
277007.488617252267431,
277017.790243137511425,
277028.091902600717731,
277038.393595640605781,
277048.695322256186046,
277058.997082446236163,
277069.298876209650189,
277079.600703545322176,
277089.902564452204388,
277100.204458929016255,
277110.506386974710040,
277120.808348588238005,
277131.110343768203165,
277141.412372513848823,
277151.714434823777992,
277162.016530696942937,
277172.318660132237710,
277182.620823128556367,
277192.923019684734754,
277203.225249799550511,
277213.527513472072314,
277223.829810701019596,
277234.132141485344619,
277244.434505823941436,
277254.736903715645894,
277265.039335159235634,
277275.341800153779332,
277285.644298697996419,
277295.946830790897366,
277306.249396431201603,
277316.551995617861394,
277326.854628349770792,
277337.157294625707436,
277347.459994444740005,
277357.762727805529721,
277368.065494707087055,
277378.368295148306061,
277388.671129127906170,
277398.973996644897852,
277409.276897698116954,
277419.579832286457531,
277429.882800408755429,
277440.185802063904703,
277450.488837250741199,
277460.791905968217179,
277471.095008215226699,
277481.398143990605604,
277491.701313293131534,
277502.004516121698543,
277512.307752475433517,
277522.611022352881264,
277532.914325753168669,
277543.217662675073370,
277553.521033117373008,
277563.824437079078052,
277574.127874559024349,
277584.431345556164160,
277594.734850069216918,
277605.038388097134884,
277615.341959638870321,
277625.645564693142660,
277635.949203259020578,
277646.252875335165299,
277656.556580920587294,
277666.860320014238823,
277677.164092614781111,
277687.467898721282836,
277697.771738332521636,
277708.075611447449774,
277718.379518064903095,
277728.683458183717448,
277738.987431802786887,
277749.291438921063673,
277759.595479537325446,
277769.899553650582675,
277780.203661259554792,
277790.507802363135852,
277800.811976960336324,
277811.116185049933847,
277821.420426630938891,
277831.724701701954473,
277842.029010262107477,
277852.333352310233749,
277862.637727845110931,
277872.942136865749490,
277883.246579370927066,
277893.551055359479506,
277903.855564830475487,
277914.160107782576233,
277924.464684214850422,
277934.769294126017485,
277945.073937515087891,
277955.378614380897488,
277965.683324722223915,
277975.988068538135849,
277986.292845827410929,
277996.597656588826794,
278006.902500821452122,
278017.207378524064552,
278027.512289695558138,
278037.817234334826935,
278048.122212440648582,
278058.427224012091756,
278068.732269047934096,
278079.037347546953242,
278089.342459508276079,
278099.647604930505622,
278109.952783812826965,
278120.257996153784916,
278130.563241952506360,
278140.868521207827143,
278151.173833918466698,
278161.479180083551910,
278171.784559701802209,
278182.089972772169858,
278192.395419293548912,
278202.700899264658801,
278213.006412684568204,
278223.311959552054759,
278233.617539866012521,
278243.923153625451960,
278254.228800829034299,
278264.534481475886423,
278274.840195564611349,
278285.145943094335962,
278295.451724063837901,
278305.757538471953012,
278316.063386317691766,
278326.369267599773593,
278336.675182317208964,
278346.981130468833726,
278357.287112053541932,
278367.593127070169430,
278377.899175517668482,
278388.205257394874934,
278398.511372700741049,
278408.817521433986258,
278419.123703593737446,
278429.429919178655837,
278439.736168187693693,
278450.042450619803276,
278460.348766473762225,
278470.655115748581011,
278480.961498442979064,
278491.267914556025062,
278501.574364086438436,
278511.880847033113241,
278522.187363395118155,
278532.493913171172608,
278542.800496360112447,
278553.107112961006351,
278563.413762972573750,
278573.720446393825114,
278584.027163223479874,
278594.333913460548501,
278604.640697103925049,
278614.947514152445365,
278625.254364605003502,
278635.561248460493516,
278645.868165717693046,
278656.175116375728976,
278666.482100433204323,
278676.789117889304180,
278687.096168742631562,
278697.403252992255148,
278707.710370636952575,
278718.017521675617900,
278728.324706107261591,
278738.631923930544872,
278748.939175144536421,
278759.246459748072084,
278769.553777740045916,
278779.861129119351972,
278790.168513884767890,
278800.475932035362348,
278810.783383569912985,
278821.090868487197440,
278831.398386786400806,
278841.705938466067892,
278852.013523525383789,
278862.321141962951515,
278872.628793777839746,
278882.936478968942538,
278893.244197535095736,
278903.551949475135189,
278913.859734788071364,
278924.167553472623695,
278934.475405527802650,
278944.783290952502284,
278955.091209745616652,
278965.399161905865185,
278975.707147432316560,
278986.015166323806625,
278996.323218579229433,
279006.631304197479039,
279016.939423177333083,
279027.247575517802034,
279037.555761217779946,
279047.863980276044458,
279058.172232691606041,
279068.480518463184126,
279078.788837589905597,
279089.097190070489887,
279099.405575903831050,
279109.713995088881347,
279120.022447624418419,
279130.330933509510942,
279140.639452742936555,
279150.948005323531106,
279161.256591250305064,
279171.565210522036068,
279181.873863137676381,
279192.182549096061848,
279202.491268396144733,
279212.800021036819089,
279223.108807016862556,
279233.417626335285604,
279243.726478990982287,
279254.035364982672036,
279264.344284309481736,
279274.653236970072612,
279284.962222963513341,
279295.271242288639769,
279305.580294944229536,
279315.889380929351319,
279326.198500242724549,
279336.507652883417904,
279346.816838850150816,
279357.126058141991962,
279367.435310757544357,
279377.744596695993096,
279388.053915956115816,
279398.363268536748365,
279408.672654436901212,
279418.982073655351996,
279429.291526190994773,
279439.601012042898219,
279449.910531209665351,
279460.220083690423053,
279470.529669483890757,
279480.839288589020725,
279491.148941004881635,
279501.458626730018295,
279511.768345763674006,
279522.078098104451783,
279532.387883751420304,
279542.697702703357209,
279553.007554959214758,
279563.317440517945215,
279573.627359378326219,
279583.937311539310031,
279594.247296999790706,
279604.557315758604091,
279614.867367814702448,
279625.177453166863415,
279635.487571814272087,
279645.797723755415063,
279656.107908989477437,
279666.418127515236847,
279676.728379331587348,
279687.038664437481202,
279697.348982831754256,
279707.659334513358772,
279717.969719481072389,
279728.280137733847369,
279738.590589270635974,
279748.901074090274051,
279759.211592191655654,
279769.522143573674839,
279779.832728235167451,
279790.143346175143961,
279800.453997392382007,
279810.764681885950267,
279821.075399654509965,
279831.386150697071571,
279841.696935012587346,
279852.007752599776722,
279862.318603457766585,
279872.629487585218158,
279882.940404981200118,
279893.251355644490104,
279903.562339574040379,
279913.873356768744998,
279924.184407227498014,
279934.495490949135274,
279944.806607932550833,
279955.117758176813368,
279965.428941680584103,
279975.740158442873508,
279986.051408462575637,
279996.362691738526337,
280006.674008269736078,
280016.985358054982498,
280027.296741093159653,
280037.608157383278012,
280047.919606924115214,
280058.231089714623522,
280068.542605753638782,
280078.854155040171463,
280089.165737572999205,
280099.477353351016063,
280109.789002373290714,
280120.100684638426173,
280130.412400145607535,
280140.724148893554229,
280151.035930881160311,
280161.347746107436251,
280171.659594571159687,
280181.971476271341089,
280192.283391206816304,
280202.595339376421180,
280212.907320779107977,
280223.219335413770750,
280233.531383279361762,
280243.843464374658652,
280254.155578698613681,
280264.467726250179112,
280274.779907028190792,
280285.092121031542774,
280295.404368259129114,
280305.716648709843867,
280316.028962382581085,
280326.341309276351240,
280336.653689389873762,
280346.966102722100914,
280357.278549272043165,
280367.591029038478155,
280377.903542020299938,
280388.216088216460776,
280398.528667625854723,
280408.841280247317627,
280419.153926079801749,
280429.466605122201145,
280439.779317373351660,
280450.092062832263764,
280460.404841497715097,
280470.717653368657921,
280481.030498444044497,
280491.343376722710673,
280501.656288203550503,
280511.969232885458041,
280522.282210767385550,
280532.595221848110668,
280542.908266126643866,
280553.221343601879198,
280563.534454272710718,
280573.847598137974273,
280584.160775196563918,
280594.473985447490122,
280604.787228889530525,
280615.100505521637388,
280625.413815342762973,
280635.727158351684920,
280646.040534547413699,
280656.353943928726949,
280666.667386494635139,
280676.980862244032323,
280687.294371175754350,
280697.607913288753480,
280707.921488581865560,
280718.235097054101061,
280728.548738704179414,
280738.862413531111088,
280749.176121533906553,
280759.489862711227033,
280769.803637062199414,
280780.117444585543126,
280790.431285280326847,
280800.745159145211801,
280811.059066179324873,
280821.373006381501909,
280831.686979750578757,
280842.000986285565887,
280852.315025985240936,
280862.629098848614376,
280872.943204874463845,
280883.257344061741605,
280893.571516409516335,
280903.885721916391049,
280914.199960581492633,
280924.514232403598726,
280934.828537381719798,
280945.142875514633488,
280955.457246801233850,
280965.771651240589563,
280976.086088831420057,
280986.400559572852217,
280996.715063463489059,
281007.029600502457470,
281017.344170688535087,
281027.658774020615965,
281037.973410497768782,
281048.288080118771177,
281058.602782882517204,
281068.917518787900917,
281079.232287833874580,
281089.547090019274037,
281099.861925343051553,
281110.176793804159388,
281120.491695401375182,
281130.806630133767612,
281141.121597999997903,
281151.436598999251146,
281161.751633130188566,
281172.066700391762424,
281182.381800783041399,
281192.696934302744921,
281203.012100949883461,
281213.327300723234657,
281223.642533621867187,
281233.957799644558690,
281244.273098790261429,
281254.588431057927664,
281264.903796446335036,
281275.219194954435807,
281285.534626581124030,
281295.850091325468384,
281306.165589186130092,
281316.481120162119623,
281326.796684252331033,
281337.112281455716584,
281347.427911771112122,
281357.743575197469909,
281368.059271733625792,
281378.375001378532033,
281388.690764131082688,
281399.006559990288224,
281409.322388954809867,
281419.638251023774501,
281429.954146195959765,
281440.270074470376130,
281450.586035845801234,
281460.902030321245547,
281471.218057895603124,
281481.534118567651603,
281491.850212336517870,
281502.166339200863149,
281512.482499159756117,
281522.798692212090828,
281533.114918356644921,
281543.431177592487074,
281553.747469918453135,
281564.063795333437156,
281574.380153836333193,
281584.696545426093508,
281595.012970101495739,
281605.329427861666773,
281615.645918705326039,
281625.962442631484009,
281636.278999639034737,
281646.595589726755861,
281656.912212893774267,
281667.228869138751179,
281677.545558460813481,
281687.862280858680606,
281698.179036331363022,
281708.495824877812993,
281718.812646496808156,
281729.129501187417191,
281739.446388948301319,
281749.763309778645635,
281760.080263677111361,
281770.397250642825384,
281780.714270674565341,
281791.031323771225289,
281801.348409931757487,
281811.665529155055992,
281821.982681440073065,
281832.299866785644554,
281842.617085190664511,
281852.934336654201616,
281863.251621174917091,
281873.568938751937822,
281883.886289383983240,
281894.203673070122022,
281904.521089809190016,
281914.838539600081276,
281925.156022441689856,
281935.473538332968019,
281945.791087272926234,
281956.108669260167517,
281966.426284293935169,
281976.743932372948620,
281987.061613496160135,
281997.379327662463766,
282007.697074870811775,
282018.014855120040011,
282028.332668409100734,
282038.650514736946207,
282048.968394102412276,
282059.286306504451204,
282069.604251941898838,
282079.922230413765647,
282090.240241918887477,
282100.558286456158385,
282110.876364024647046,
282121.194474623014685,
282131.512618250388186,
282141.830794905603398,
282152.149004587496165,
282162.467247295076959,
282172.785523027123418,
282183.103831782704219,
282193.422173560597003,
282203.740548359928653,
282214.058956179302186,
282224.377397017786279,
282234.695870874333195,
282245.014377747778781,
282255.332917637017090,
282265.651490541058592,
282275.970096458797343,
282286.288735388952773,
282296.607407330709975,
282306.926112282730173,
282317.244850244140252,
282327.563621213776059,
282337.882425190415233,
282348.201262173184659,
282358.520132160745561,
282368.839035152283031,
282379.157971146516502,
282389.476940142514650,
282399.795942138938699,
282410.114977134973742,
282420.434045129339211,
282430.753146121045575,
282441.072280108986888,
282451.391447091998998,
282461.710647069092374,
282472.029880039161071,
282482.349146001099143,
282492.668444953742437,
282502.987776896217838,
282513.307141827186570,
282523.626539745659102,
282533.945970650587697,
282544.265434540808201,
282554.584931415389292,
282564.904461273050401,
282575.224024112860207,
282585.543619933538139,
282595.863248734269291,
282606.182910513656680,
282616.502605270827189,
282626.822333004674874,
282637.142093713977374,
282647.461887397861574,
282657.781714054988697,
282668.101573684543837,
282678.421466285188217,
282688.741391855990514,
282699.061350395844784,
282709.381341903528664,
282719.701366378227249,
282730.021423818543553,
282740.341514223662671,
282750.661637592245825,
282760.981793923419900,
282771.301983215962537,
282781.622205468942411,
282791.942460681020748,
282802.262748851324432,
282812.583069978689309,
282822.903424062067643,
282833.223811100353487,
282843.544231092440896,
282853.864684037282132,
282864.185169933713041,
282874.505688780744094,
282884.826240577269346,
282895.146825322066434,
282905.467443014262244,
282915.788093652634416,
282926.108777236193419,
282936.429493763775099,
282946.750243234273512,
282957.071025646640919,
282967.391840999887791,
282977.712689292791765,
282988.033570524246898,
282998.354484693321865,
283008.675431798794307,
283018.996411839674693,
283029.317424814798869,
283039.638470723177306,
283049.959549563529436,
283060.280661335040350,
283070.601806036429480,
283080.922983666707296,
283091.244194224767853,
283101.565437709446996,
283111.886714119813405,
283122.208023454644717,
283132.529365712951403,
283142.850740893569309,
283153.172148995567113,
283163.493590017606039,
283173.815063958754763,
283184.136570818023756,
283194.458110594132449,
283204.779683286207728,
283215.101288892910816,
283225.422927413426805,
283235.744598846416920,
283246.066303191008046,
283256.388040446036030,
283266.709810610394925,
283277.031613683036994,
283287.353449662739877,
283297.675318548688665,
283307.997220339602791,
283318.319155034550931,
283328.641122632194310,
283338.963123131718021,
283349.285156531899702,
283359.607222831691615,
283369.929322030046023,
283380.251454125740565,
283390.573619117902126,
283400.895817005250137,
283411.218047786853276,
283421.540311461547390,
283431.862608028342947,
283442.184937485959381,
283452.507299833581783,
283462.829695069871377,
283473.152123193955049,
283483.474584204610437,
283493.797078100789804,
283504.119604881445412,
283514.442164545471314,
283524.764757091877982,
283535.087382519443054,
283545.410040827176999,
283555.732732013915665,
283566.055456078611314,
283576.378213020216208,
283586.701002837740816,
283597.023825529846363,
283607.346681095601525,
283617.669569534074981,
283627.992490843927953,
283638.315445024170913,
283648.638432073756121,
283658.961451991694048,
283669.284504776645917,
283679.607590427855030,
283689.930708943924401,
283700.253860323980916,
283710.577044566860422,
283720.900261671456974,
283731.223511636839248,
283741.546794461784884,
283751.870110145304352,
283762.193458686233498,
283772.516840083524585,
283782.840254336071666,
283793.163701442885213,
283803.487181402801070,
283813.810694214829709,
283824.134239877748769,
283834.457818390568718,
283844.781429752241820,
283855.105073961603921,
283865.428751017665491,
283875.752460919262376,
283886.076203665405046,
283896.399979254871141,
283906.723787686845753,
283917.047628959931899,
283927.371503073314670,
283937.695410025655292,
283948.019349816138856,
283958.343322443543002,
283968.667327906819992,
283978.991366204863880,
283989.315437336626928,
283999.639541301061399,
284009.963678097061347,
284020.287847723520827,
284030.612050179333892,
284040.936285463511012,
284051.260553574946243,
284061.584854512533639,
284071.909188275225461,
284082.233554861857556,
284092.557954271556810,
284102.882386503042653,
284113.206851555325557,
284123.531349427299574,
284133.855880117858760,
284144.180443626071792,
284154.505039950716309,
284164.829669090744574,
284175.154331045050640,
284185.479025812644977,
284195.803753392363433,
284206.128513783274684,
284216.453306984098163,
284226.778132993960753,
284237.102991811581887,
284247.427883436088450,
284257.752807866199873,
284268.077765100984834,
284278.402755139279179,
284288.727777980151586,
284299.052833622321486,
284309.377922064857557,
284319.703043306712061,
284330.028197346662637,
284340.353384183777962,
284350.678603816835675,
284361.003856244962662,
284371.329141466878355,
284381.654459481593221,
284391.979810288059525,
284402.305193885171320,
284412.630610271822661,
284422.956059447082225,
284433.281541409611236,
284443.607056158536579,
284453.932603692752309,
284464.258184011152480,
284474.583797112747561,
284484.909442996315192,
284495.235121660865843,
284505.560833105293568,
284515.886577328608837,
284526.212354329647496,
284536.538164107361808,
284546.864006660704035,
284557.189881988510024,
284567.515790089790244,
284577.841730963496957,
284588.167704608466011,
284598.493711023649666,
284608.819750208058394,
284619.145822160469834,
284629.471926879952662,
284639.798064365342725,
284650.124234615650494,
284660.450437629653607,
284670.776673406420741,
284681.102941944787744,
284691.429243243823294,
284701.755577302305028,
284712.081944119127002,
284722.408343693416100,
284732.734776023949962,
284743.061241109680850,
284753.387738949561026,
284763.714269542484544,
284774.040832887345459,
284784.367428983154241,
284794.694057828804944,
284805.020719423249830,
284815.347413765382953,
284825.674140854156576,
284836.000900688406546,
284846.327693267201539,
284856.654518589377403,
284866.981376653828193,
284877.308267459680792,
284887.635191005596425,
284897.962147290701978,
284908.289136313833296,
284918.616158073942643,
284928.943212569924071,
284939.270299800846260,
284949.597419765370432,
284959.924572462681681,
284970.251757891557645,
284980.578976051067002,
284990.906226939987391,
285001.233510557329282,
285011.560826901928522,
285021.888175972853787,
285032.215557768999133,
285042.542972289200407,
285052.870419532526284,
285063.197899497754406,
285073.525412183953449,
285083.852957589901052,
285094.180535714724101,
285104.508146557200234,
285114.835790116223507,
285125.163466390862595,
285135.491175380069762,
285145.818917082564440,
285156.146691497473512,
285166.474498623632826,
285176.802338459936436,
285187.130211005453020,
285197.458116258960217,
285207.786054219584912,
285218.114024886046536,
285228.442028257413767,
285238.770064332464244,
285249.098133110324852,
285259.426234589831438,
285269.754368769878056,
285280.082535649475176,
285290.410735227458645,
285300.738967502838932,
285311.067232474568300,
285321.395530141424388,
285331.723860502475873,
285342.052223556675017,
285352.380619302799460,
285362.709047740034293,
285373.037508866982535,
285383.366002682887483,
285393.694529186410364,
285404.023088376794476,
285414.351680252642836,
285424.680304813082330,
285435.008962057007011,
285445.337651983369142,
285455.666374591004569,
285465.995129878981970,
285476.323917846078984,
285486.652738491422497,
285496.981591813731939,
285507.310477812017780,
285517.639396485406905,
285527.968347832502332,
285538.297331852430943,
285548.626348544145003,
285558.955397906538565,
285569.284479938389268,
285579.613594638940413,
285589.942742006853223,
285600.271922041138168,
285610.601134740805719,
285620.930380104749929,
285631.259658131806646,
285641.588968821044546,
285651.918312171299476,
285662.247688181698322,
285672.577096850844100,
285682.906538177980110,
285693.236012161767576,
285703.565518801391590,
285713.895058095629793,
285724.224630043492652,
285734.554234643874224,
285744.883871895668563,
285755.213541797886137,
285765.543244349479210,
285775.872979549283627,
285786.202747396309860,
285796.532547889510170,
285806.862381027720403,
285817.192246809951030,
285827.522145235096104,
285837.852076302166097,
285848.182040009996854,
285858.512036357540637,
285868.842065343807917,
285879.172126967692748,
285889.502221228030976,
285899.832348123833071,
285910.162507654225919,
285920.492699817812536,
285930.822924613778014,
285941.153182040841784,
285951.483472098188940,
285961.813794784597121,
285972.144150098960381,
285982.474538040347397,
285992.804958607594017,
286003.135411799652502,
286013.465897615533322,
286023.796416054072324,
286034.126967114338186,
286044.457550795050338,
286054.788167095393874,
286065.118816014088225,
286075.449497550202068,
286085.780211702571250,
286096.110958470322657,
286106.441737852175720,
286116.772549847082701,
286127.103394454228692,
286137.434271672274917,
286147.765181500290055,
286158.096123937109951,
286168.427098981803283,
286178.758106633205898,
286189.089146890386473,
286199.420219752064440,
286209.751325217424892,
286220.082463285129052,
286230.413633954362012,
286240.744837223959621,
286251.076073092815932,
286261.407341559941415,
286271.738642624230124,
286282.069976284692530,
286292.401342540222686,
286302.732741389656439,
286313.064172832062468,
286323.395636866276618,
286333.727133491425775,
286344.058662706229370,
286354.390224509697873,
286364.721818900841754,
286375.053445878496859,
286385.385105441673659,
286395.716797589324415,
286406.048522320343181,
286416.380279633624014,
286426.712069528119173,
286437.043892002897337,
286447.375747056736145,
286457.707634688646067,
286468.039554897637572,
286478.371507682488300,
286488.703493042266928,
286499.035510975809302,
286509.367561482184101,
286519.699644560168963,
286530.031760208832566,
286540.363908427127171,
286550.696089213888627,
286561.028302568069194,
286571.360548488737550,
286581.692826974613126,
286592.025138024881016,
286602.357481638318859,
286612.689857813937124,
286623.022266550571658,
286633.354707847232930,
286643.687181702931412,
286654.019688116561156,
286664.352227087016217,
286674.684798613190651,
286685.017402694269549,
286695.350039328855928,
286705.682708516134880,
286716.015410254942253,
286726.348144544288516,
286736.680911383009516,
286747.013710770115722,
286757.346542704501189,
286767.679407185292803,
286778.012304211093578,
286788.345233781205025,
286798.678195894230157,
286809.011190549412277,
286819.344217745470814,
286829.677277481474448,
286840.010369756317232,
286850.343494568951428,
286860.676651918271091,
286871.009841803286690,
286881.343064222892281,
286891.676319176098332,
286902.009606661740690,
286912.342926678829826,
286922.676279226259794,
286933.009664303099271,
286943.343081908125896,
286953.676532040350139,
286964.010014698724262,
286974.343529882200528,
286984.677077589731198,
286995.010657820152119,
287005.344270572531968,
287015.677915845764801,
287026.011593638861086,
287036.345303950540256,
287046.679046780045610,
287057.012822126096580,
287067.346629987761844,
287077.680470363935456,
287088.014343253569677,
287098.348248655500356,
287108.682186568912584,
287119.016156992467586,
287129.350159925292246,
287139.684195366338827,
287150.018263314443175,
287160.352363768615760,
287170.686496727808844,
287181.020662190916482,
287191.354860156949144,
287201.689090624742676,
287212.023353593365755,
287222.357649061654229,
287232.691977028676774,
287243.026337493211031,
287253.360730454383884,
287263.695155910972971,
287274.029613862046972,
287284.364104306441732,
287294.698627243225928,
287305.033182671177201,
287315.367770589480642,
287325.702390996855684,
287336.037043892371003,
287346.371729274920654,
287356.706447143398691,
287367.041197496873792,
287377.375980334181804,
287387.710795654391404,
287398.045643456280231,
287408.380523738975171,
287418.715436501312070,
287429.050381742243189,
287439.385359460662585,
287449.720369655638933,
287460.055412326008081,
287470.390487470896915,
287480.725595088966656,
287491.060735179344192,
287501.395907740981784,
287511.731112772831693,
287522.066350273671560,
287532.401620242628269,
287542.736922678595874,
287553.072257580526639,
287563.407624947314616,
287573.743024778028484,
287584.078457071387675,
287594.413921826635487,
287604.749419042491354,
287615.084948717965744,
287625.420510852010921,
287635.756105443637352,
287646.091732491680887,
287656.427391995210201,
287666.763083953002933,
287677.098808364185970,
287687.434565227536950,
287697.770354542124551,
287708.106176306901034,
287718.442030520760454,
287728.777917182596866,
287739.113836291537154,
287749.449787846300751,
287759.785771846014541,
287770.121788289514370,
287780.457837175868917,
287790.793918503914028,
287801.130032272601966,
287811.466178480884992,
287821.802357127831783,
287832.138568212219980,
287842.474811733118258,
287852.811087689420674,
287863.147396080079488,
287873.483736903988756,
287883.820110160275362,
287894.156515847658738,
287904.492953965265770,
287914.829424511932302,
287925.165927486668807,
287935.502462888369337,
287945.839030716102570,
287956.175630968587939,
287966.512263645068742,
287976.848928744264413,
287987.185626265243627,
287997.522356206842232,
288007.859118568187114,
288018.195913348055910,
288028.532740545400884,
288038.869600159290712,
288049.206492188677657,
288059.543416632397566,
288069.880373489402700,
288080.217362758761737,
288090.554384439310525,
288100.891438530117739,
288111.228525029961020,
288121.565643937967252,
288131.902795252914075,
288142.239978973928373,
288152.577195099845994,
288162.914443629677407,
288173.251724562258460,
288183.589037896716036,
288193.926383631827775,
288204.263761766662356,
288214.601172300113831,
288224.938615231192671,
288235.276090558734722,
288245.613598281808663,
288255.951138399308547,
288266.288710910244845,
288276.626315813395195,
288286.963953107944690,
288297.301622792670969,
288307.639324866642710,
288317.977059328637552,
288328.314826177840587,
288338.652625412971247,
288348.990457033214625,
288359.328321037290152,
288369.666217424382921,
288380.004146193270572,
288390.342107342963573,
288400.680100872355979,
288411.018126780516468,
288421.356185066280887,
288431.694275728659704,
288442.032398766605183,
288452.370554179069586,
288462.708741964946967,
288473.046962123306002,
288483.385214652982540,
288493.723499552987050,
288504.061816822271794,
288514.400166459789034,
288524.738548464432824,
288535.076962835271843,
288545.415409571083728,
288555.753888670995366,
288566.092400133900810,
288576.430943958752323,
288586.769520144443959,
288597.108128689986188,
288607.446769594331272,
288617.785442856431473,
288628.124148475239053,
288638.462886449648067,
288648.801656778668985,
288659.140459461370483,
288669.479294496471994,
288679.818161883042194,
288690.157061620033346,
288700.495993706455920,
288710.834958141145762,
288721.173954923113342,
288731.512984051369131,
288741.852045524807181,
288752.191139342321549,
288762.530265503039118,
288772.869424005679321,
288783.208614849427249,
288793.547838033060543,
288803.887093555647880,
288814.226381416083314,
288824.565701613319106,
288834.905054146365728,
288845.244439014175441,
288855.583856215584092,
288865.923305749660358,
288876.262787615356501,
288886.602301811566576,
288896.941848337301053,
288907.281427191512194,
288917.621038373094052,
288927.960681881057099,
288938.300357714295387,
288948.640065871935803,
288958.979806352697778,
288969.319579155708198,
288979.659384279861115,
288989.999221724050585,
289000.339091487403493,
289010.678993568639271,
289021.018927966884803,
289031.358894681034144,
289041.698893710097764,
289052.038925052969716,
289062.378988708660472,
289072.719084676064085,
289083.059212954191025,
289093.399373541935347,
289103.739566438365728,
289114.079791642259806,
289124.420049152744468,
289134.760338968713768,
289145.100661089119967,
289155.441015512857120,
289165.781402238993905,
289176.121821266424377,
289186.462272594217211,
289196.802756221033633,
289207.143272146175150,
289217.483820368361194,
289227.824400886718649,
289238.165013700083364,
289248.505658807465807,
289258.846336207818240,
289269.187045900092926,
289279.527787883183919,
289289.868562156218104,
289300.209368717914913,
289310.550207567459438,
289320.891078703629319,
289331.231982125493232,
289341.572917832003441,
289351.913885822112206,
289362.254886094713584,
289372.595918648818042,
289382.936983483319636,
289393.278080597403459,
289403.619209989672527,
289413.960371659370139,
289424.301565605332144,
289434.642791826510802,
289444.984050321858376,
289455.325341090443544,
289465.666664131102152,
289476.008019442902878,
289486.349407024623360,
289496.690826875448693,
289507.032278994214721,
289517.373763379815500,
289527.715280031319708,
289538.056828947679605,
289548.398410127789248,
289558.740023570717312,
289569.081669275299646,
289579.423347240604926,
289589.765057465468999,
289600.106799948960543,
289610.448574689915404,
289620.790381687460467,
289631.132220940373372,
289641.474092447839212,
289651.815996208635624,
289662.157932221773081,
289672.499900486203842,
289682.841901000880171,
289693.183933764754329,
289703.525998776836786,
289713.868096036138013,
289724.210225541435648,
289734.552387291856576,
289744.894581286236644,
289755.236807523702737,
289765.579066003032494,
289775.921356723294593,
289786.263679683383089,
289796.606034882366657,
289806.948422319081146,
289817.290841992537025,
289827.633293901744764,
289837.975778045598418,
289848.318294423050247,
289858.660843033110723,
289869.003423874790315,
289879.346036946924869,
289889.688682248466648,
289900.031359778600745,
289910.374069535988383,
289920.716811519814655,
289931.059585728857201,
289941.402392162359320,
289951.745230818924028,
289962.088101697852835,
289972.431004797806963,
289982.773940117971506,
289993.116907657182310,
290003.459907414449845,
290013.802939388726372,
290024.146003579022363,
290034.489099984231871,
290044.832228603423573,
290055.175389435375109,
290065.518582479096949,
290075.861807733774185,
290086.205065198068041,
290096.548354871163610,
290106.891676751838531,
290117.235030839219689,
290127.578417132142931,
290137.921835629676934,
290148.265286330773961,
290158.608769234328065,
290168.952284339233302,
290179.295831644674763,
290189.639411149488296,
290199.983022852626164,
290210.326666753040627,
290220.670342849742156,
290231.014051141683012,
290241.357791627815459,
290251.701564307091758,
290262.045369178522378,
290272.389206241001375,
290282.733075493539218,
290293.076976935088169,
290303.420910564658698,
290313.764876381086651,
290324.108874383557122,
290334.452904570789542,
290344.796966941852588,
290355.141061495698523,
290365.485188231454231,
290375.829347147780936,
290386.173538243863732,
290396.517761518596672,
290406.862016970932018,
290417.206304599880241,
290427.550624404335395,
290437.894976383366156,
290448.239360535808373,
290458.583776860730723,
290468.928225357085466,
290479.272706023824867,
290489.617218859784771,
290499.961763864150271,
290510.306341035757214,
290520.650950373616070,
290530.995591876679100,
290541.340265543898568,
290551.684971374226734,
290562.029709366674069,
290572.374479520192835,
290582.719281833735295,
290593.064116306253709,
290603.408982936758548,
290613.753881724202074,
290624.098812667478342,
290634.443775765656028,
290644.788771017629188,
290655.133798422466498,
290665.478857978945598,
290675.823949686251581,
290686.169073543162085,
290696.514229548803996,
290706.859417702013161,
290717.204638001858257,
290727.549890447233338,
290737.895175037090667,
290748.240491770440713,
290758.585840646352153,
290768.931221663602628,
290779.276634821202606,
290789.622080118162557,
290799.967557553551160,
290810.313067126146052,
290820.658608835074119,
290831.004182679113001,
290841.349788657447789,
290851.695426768856123,
290862.041097012464888,
290872.386799387109932,
290882.732533891859930,
290893.078300525608938,
290903.424099287425634,
290913.769930176204070,
290924.115793190780096,
290934.461688330397010,
290944.807615593832452,
290955.153574980038684,
290965.499566488084383,
290975.845590117038228,
290986.191645865619648,
290996.537733732955530,
291006.883853717881721,
291017.230005819525104,
291027.576190036721528,
291037.922406368597876,
291048.268654813989997,
291058.614935371908359,
291068.961248041247018,
291079.307592821191065,
291089.653969710401725,
291100.000378708122298,
291110.346819813188631,
291120.693293024611194,
291131.039798341284040,
291141.386335762275849,
291151.732905286480673,
291162.079506912850775,
291172.426140640513040,
291182.772806468245108,
291193.119504395173863,
291203.466234420135152,
291213.812996542197652,
291224.159790760255419,
291234.506617073318921,
291244.853475480340421,
291255.200365980388597,
291265.547288572241087,
291275.894243255024776,
291286.241230027633719,
291296.588248889078386,
291306.935299838311039,
291317.282382874342147,
291327.629497996123973,
291337.976645202492364,
291348.323824492690619,
291358.671035865379963,
291369.018279319803696,
291379.365554854739457,
291389.712862469314132,
291400.060202162305359,
291410.407573932898231,
291420.754977779870387,
291431.102413702406920,
291441.449881699227262,
291451.797381769458298,
291462.144913912110496,
291472.492478126019705,
291482.840074410254601,
291493.187702763767447,
291503.535363185510505,
291513.883055674494244,
291524.230780229670927,
291534.578536849934608,
291544.926325534353964,
291555.274146281881258,
291565.621999091526959,
291575.969883962126914,
291586.317800892808009,
291596.665749882464297,
291607.013730930048041,
291617.361744034569710,
291627.709789195039775,
291638.057866410352290,
291648.405975679575931,
291658.754117001546547,
291669.102290375332814,
291679.450495799945202,
291689.798733274219558,
291700.147002797282767,
291710.495304367970675,
291720.843637985351961,
291731.192003648320679,
291741.540401355945505,
291751.888831107120495,
291762.237292900914326,
291772.585786736104637,
291782.934312611934729,
291793.282870527124032,
291803.631460480857641,
291813.980082471854985,
291824.328736499417573,
291834.677422562264837,
291845.026140659465455,
291855.374890789971687,
291865.723672952794004,
291876.072487146942876,
291886.421333371137735,
291896.770211624738295,
291907.119121906405780,
291917.468064215267077,
291927.817038550274447,
291938.166044910380151,
291948.515083294536453,
291958.864153701812029,
291969.213256130984519,
291979.562390581355430,
291989.911557051585987,
292000.260755540803075,
292010.609986047900748,
292020.959248572005890,
292031.308543111954350,
292041.657869666698389,
292052.007228235306684,
292062.356618816731498,
292072.706041409983300,
292083.055496013839729,
292093.404982627602294,
292103.754501249932218,
292114.104051880072802,
292124.453634516743477,
292134.803249159129336,
292145.152895806124434,
292155.502574456681032,
292165.852285109751392,
292176.202027764462400,
292186.551802419649903,
292196.901609074207954,
292207.251447727379855,
292217.601318377885036,
292227.951221024908591,
292238.301155667228159,
292248.651122303970624,
292259.001120934030041,
292269.351151556475088,
292279.701214170141611,
292290.051308774156496,
292300.401435367355589,
292310.751593948807567,
292321.101784517406486,
292331.452007072337437,
292341.802261612319853,
292352.152548136422411,
292362.502866643713787,
292372.853217133029830,
292383.203599603439216,
292393.554014053894207,
292403.904460483405273,
292414.254938890866470,
292424.605449275346473,
292434.955991635739338,
292445.306565971055534,
292455.657172280247323,
292466.007810562383384,
292476.358480816474184,
292486.709183041297365,
292497.059917235921603,
292507.410683399415575,
292517.761481530615129,
292528.112311628588941,
292538.463173692347482,
292548.814067720726598,
292559.164993712911382,
292569.515951667679474,
292579.866941584157757,
292590.217963461123873,
292600.569017297821119,
292610.920103093027137,
292621.271220845927019,
292631.622370555240195,
292641.973552220093552,
292652.324765839497559,
292662.676011412288062,
292673.027288937591948,
292683.378598414303269,
292693.729939841432497,
292704.081313217931893,
292714.432718542811926,
292724.784155815083068,
292735.135625033697579,
292745.487126197549514,
292755.838659305707552,
292766.190224357182160,
292776.541821350925602,
292786.893450285831932,
292797.245111160911620,
292807.596803975291550,
292817.948528727749363,
292828.300285417411942,
292838.652074043173343,
292849.003894604044035,
292859.355747098976281,
292869.707631527038757,
292880.059547887125518,
292890.411496178247035,
292900.763476399355568,
292911.115488549403381,
292921.467532627517357,
292931.819608632533345,
292942.171716563461814,
292952.523856419371441,
292962.876028199156281,
292973.228231901710387,
292983.580467526277062,
292993.932735071517527,
293004.285034536733292,
293014.637365920701995,
293024.989729222375900,
293035.342124440940097,
293045.694551575114019,
293056.047010624082759,
293066.399501586682163,
293076.752024462039117,
293087.104579249047674,
293097.457165946776513,
293107.809784554061480,
293118.162435069971252,
293128.515117493516300,
293138.867831823532470,
293149.220578059204854,
293159.573356199369300,
293169.926166243094485,
293180.279008189274464,
293190.631882037036121,
293200.984787785157096,
293211.337725432822481,
293221.690694978809915,
293232.043696422304492,
293242.396729762200266,
293252.749794997507706,
293263.102892127120867,
293273.456021150050219,
293283.809182065422647,
293294.162374872015789,
293304.515599568956532,
293314.868856155138928,
293325.222144629631657,
293335.575464991328772,
293345.928817239240743,
293356.282201372436248,
293366.635617389751133,
293376.989065290312283,
293387.342545072955545,
293397.696056736807805,
293408.049600280763116,
293418.403175703890156,
293428.756783005082980,
293439.110422183352057,
293449.464093237649649,
293459.817796167102642,
293470.171530970488675,
293480.525297646934632,
293490.879096195392776,
293501.232926614757162,
293511.586788904212881,
293521.940683062479366,
293532.294609088799916,
293542.648566981952172,
293553.002556741121225,
293563.356578365084715,
293573.710631853027735,
293584.064717203727923,
293594.418834416370373,
293604.772983489790931,
293615.127164423058275,
293625.481377215124667,
293635.835621864942368,
293646.189898371580057,
293656.544206733873580,
293666.898546951008029,
293677.252919021819253,
293687.607322945434134,
293697.961758720630314,
293708.316226346534677,
293718.670725822215900,
293729.025257146451622,
293739.379820318310522,
293749.734415336861275,
293760.089042200997937,
293770.443700909730978,
293780.798391462070867,
293791.153113856911659,
293801.507868093380239,
293811.862654170428868,
293822.217472086835187,
293832.572321841900703,
293842.927203434403054,
293853.282116863469128,
293863.637062127934769,
293873.992039226926863,
293884.347048159281258,
293894.702088924124837,
293905.057161520351656,
293915.412265947030392,
293925.767402203113306,
293936.122570287552662,
293946.477770199417137,
293956.833001937484369,
293967.188265501055866,
293977.543560888851061,
293987.898888100055046,
293998.254247133503668,
294008.609637988265604,
294018.965060663293116,
294029.320515157538466,
294039.676001470186748,
294050.031519599899184,
294060.387069545977283,
294070.742651307140477,
294081.098264882632066,
294091.453910271287896,
294101.809587472118437,
294112.165296484017745,
294122.521037306170911,
294132.876809937530197,
294143.232614376931451,
294153.588450623443350,
294163.944318676134571,
294174.300218533899169,
294184.656150195631199,
294195.012113660632167,
294205.368108927563298,
294215.724135995609686,
294226.080194863723591,
294236.436285530799069,
294246.792407995963003,
294257.148562258051243,
294267.504748316190671,
294277.860966169217136,
294288.217215816373937,
294298.573497256380506,
294308.929810488421936,
294319.286155511392280,
294329.642532324243803,
294339.998940926045179,
294350.355381315748673,
294360.711853492422961,
294371.068357454845682,
294381.424893202260137,
294391.781460733502172,
294402.138060047640465,
294412.494691143569071,
294422.851354020356666,
294433.208048677013721,
294443.564775112492498,
294453.921533325745258,
294464.278323315782472,
294474.635145081672817,
294484.991998622193933,
294495.348883936647326,
294505.705801023752429,
294516.062749882752541,
294526.419730512425303,
294536.776742911781184,
294547.133787079947069,
294557.490863015700597,
294567.847970718285069,
294578.205110186478123,
294588.562281419348437,
294598.919484415906481,
294609.276719175279140,
294619.633985696185846,
294629.991283977753483,
294640.348614018934313,
294650.705975818797015,
294661.063369376293849,
294671.420794690318871,
294681.778251760115381,
294692.135740584344603,
294702.493261162249837,
294712.850813492783345,
294723.208397574780975,
294733.566013407427818,
294743.923660989617929,
294754.281340320419986,
294764.639051398611628,
294774.996794223436154,
294785.354568793787621,
294795.712375108676497,
294806.070213167055044,
294816.428082967933733,
294826.785984510323033,
294837.143917793116998,
294847.501882815500721,
294857.859879576310050,
294868.217908074613661,
294878.575968309422024,
294888.934060279570986,
294899.292183984245639,
294909.650339422281832,
294920.008526592864655,
294930.366745494771749,
294940.724996127188206,
294951.083278488891665,
294961.441592579067219,
294971.799938396725338,
294982.158315940643661,
294992.516725210065488,
295002.875166203768458,
295013.233638920879457,
295023.592143360292539,
295033.950679521192797,
295044.309247402299661,
295054.667847002914641,
295065.026478321815375,
295075.385141357954126,
295085.743836110574193,
295096.102562578336801,
295106.461320760543458,
295116.820110656030010,
295127.178932263806928,
295137.537785583001096,
295147.896670612331945,
295158.255587351042777,
295168.614535797969438,
295178.973515952238813,
295189.332527812744956,
295199.691571378556546,
295210.050646648567636,
295220.409753621905111,
295230.768892297463026,
295241.128062674193643,
295251.487264751340263,
295261.846498527564108,
295272.205764002108481,
295282.565061173867434,
295292.924390041793231,
295303.283750604954548,
295313.643142862420063,
295324.002566813025624,
295334.362022455839906,
295344.721509789873380,
295355.081028814020101,
295365.440579527523369,
295375.800161929044407,
295386.159776017826516,
295396.519421792821959,
295406.879099252982996,
295417.238808397261892,
295427.598549224727321,
295437.958321734389756,
295448.318125925143249,
295458.677961796114687,
295469.037829346198123,
295479.397728574462235,
295489.757659479975700,
295500.117622061457951,
295510.477616318152286,
295520.837642249010969,
295531.197699852986261,
295541.557789128972217,
295551.917910076270346,
295562.278062693658285,
295572.638246980030090,
295582.998462934687268,
295593.358710556291044,
295603.718989844142925,
295614.079300797078758,
295624.439643414109014,
295634.800017694302369,
295645.160423636494670,
295655.520861239812803,
295665.881330503209028,
295676.241831425752025,
295686.602364006335847,
295696.962928244087379,
295707.323524137842469,
295717.684151686728001,
295728.044810889754444,
295738.405501745757647,
295748.766224253922701,
295759.126978413085453,
295769.487764222372789,
295779.848581680736970,
295790.209430787188467,
295800.570311540679540,
295810.931223940278869,
295821.292167984938715,
295831.653143673611339,
295842.014151005365420,
295852.375189979211427,
295862.736260594159830,
295873.097362849104684,
295883.458496743114665,
295893.819662275258452,
295904.180859444371890,
295914.542088249581866,
295924.903348689782433,
295935.264640764158685,
295945.625964471488260,
295955.987319811014459,
295966.348706781514920,
295976.710125382058322,
295987.071575611655135,
295997.433057469257619,
296007.794570954050869,
296018.156116064754315,
296028.517692800553050,
296038.879301160457544,
296049.240941143361852,
296059.602612748334650,
296069.964315974386409,
296080.326050820469391,
296090.687817285594065,
296101.049615368770901,
296111.411445069068577,
296121.773306385322940,
296132.135199316719081,
296142.497123862092849,
296152.859080020571128,
296163.221067791047972,
296173.583087172650266,
296183.945138164330274,
296194.307220765040256,
296204.669334973790683,
296215.031480789592024,
296225.393658211454749,
296235.755867238389328,
296246.118107869464438,
296256.480380103515927,
296266.842683939670678,
296277.205019376880955,
296287.567386414157227,
296297.929785050509963,
296308.292215284891427,
296318.654677116428502,
296329.017170544015244,
296339.379695566603914,
296349.742252183379605,
296360.104840393119957,
296370.467460195068270,
296380.830111588002183,
296391.192794571106788,
296401.555509143276140,
296411.918255303520709,
296422.281033050850965,
296432.643842384219170,
296443.006683302810416,
296453.369555805402342,
296463.732459891180042,
296474.095395559037570,
296484.458362807927188,
296494.821361637092195,
296505.184392045193817,
296515.547454031475354,
296525.910547594889067,
296536.273672734445427,
296546.636829449154902,
296557.000017737911548,
296567.363237599842250,
296577.726489033841062,
296588.089772039034870,
296598.453086614375934,
296608.816432758874726,
296619.179810471483506,
296629.543219751270954,
296639.906660597189330,
296650.270133008249104,
296660.633636983518954,
296670.997172521892935,
296681.360739622497931,
296691.724338284286205,
296702.087968506210018,
296712.451630287338048,
296722.815323626622558,
296733.179048523132224,
296743.542804975761101,
296753.906592983636074,
296764.270412545825820,
296774.634263661049772,
296784.998146328551229,
296795.362060547282454,
296805.726006316195708,
296816.089983634359669,
296826.453992500726599,
296836.818032914423384,
296847.182104874227662,
296857.546208379382733,
296867.910343428724445,
296878.274510021379683,
296888.638708156242501,
296899.002937832381576,
296909.367199048807379,
296919.731491804413963,
296930.095816098386422,
296940.460171929618809,
296950.824559297121596,
296961.188978200021666,
296971.553428637096658,
296981.917910607589874,
296992.282424110278953,
297002.646969144348986,
297013.011545708694030,
297023.376153802440967,
297033.740793424542062,
297044.105464573949575,
297054.470167249673977,
297064.834901450783946,
297075.199667176289950,
297085.564464425086044,
297095.929293196240906,
297106.294153488881420,
297116.659045301843435,
297127.023968634253833,
297137.388923484948464,
297147.753909853170626,
297158.118927737756167,
297168.483977137715556,
297178.849058052117471,
297189.214170479972381,
297199.579314420290757,
297209.944489872024860,
297220.309696834185161,
297230.674935305898543,
297241.040205286000855,
297251.405506773618981,
297261.770839767646976,
297272.136204267269932,
297282.501600271265488,
297292.867027778876945,
297303.232486788998358,
297313.597977300581988,
297323.963499312696513,
297334.329052824352402,
297344.694637834560126,
297355.060254342330154,
297365.425902346672956,
297375.791581846540794,
297386.157292841002345,
297396.523035329068080,
297406.888809309690259,
297417.254614781937562,
297427.620451744820457,
297437.986320197174791,
297448.352220138302073,
297458.718151567038149,
297469.084114482393488,
297479.450108883378562,
297489.816134769062046,
297500.182192138454411,
297510.548280990449712,
297520.914401324174833,
297531.280553138523828,
297541.646736432623584,
297552.012951205484569,
297562.379197455942631,
297572.745475183241069,
297583.111784386215732,
297593.478125063993502,
297603.844497215468436,
297614.210900839825626,
297624.577335935842711,
297634.943802502646577,
297645.310300539364107,
297655.676830044772942,
297666.043391018058173,
297676.409983458172064,
297686.776607364066876,
297697.143262734869495,
297707.509949569473974,
297717.876667867007200,
297728.243417626305018,
297738.610198846610729,
297748.977011526760180,
297759.343855665763840,
297769.710731262748595,
297780.077638316666707,
297790.444576826528646,
297800.811546791344881,
297811.178548210067675,
297821.545581081765704,
297831.912645405507647,
297842.279741180187557,
297852.646868404874112,
297863.014027078635991,
297873.381217200367246,
297883.748438769078348,
297894.115691783896182,
297904.482976243773010,
297914.850292147719301,
297925.217639494803734,
297935.585018283862155,
297945.952428514079656,
297956.319870184350293,
297966.687343293859158,
297977.054847841442097,
297987.422383826167788,
297997.789951247104909,
298008.157550103147514,
298018.525180393480696,
298028.892842116823886,
298039.260535272536799,
298049.628259859455284,
298059.996015876531601,
298070.363803322950844,
298080.731622197548859,
298091.099472499452531,
298101.467354227614123,
298111.835267381044105,
298122.203211958869360,
298132.571187959983945,
298142.939195383398328,
298153.307234228122979,
298163.675304493284784,
298174.043406177777797,
298184.411539280670695,
298194.779703800915740,
298205.147899737581611,
298215.516127089736983,
298225.884385856275912,
298236.252676036267076,
298246.620997628720943,
298256.989350632589776,
298267.357735047058668,
298277.726150870963465,
298288.094598103372846,
298298.463076743355487,
298308.831586789863650,
298319.200128241907805,
298329.568701098498423,
298339.937305358762387,
298350.305941021535546,
298360.674608085944783,
298371.043306551000569,
298381.412036415655166,
298391.780797679035459,
298402.149590340035502,
298412.518414397665765,
298422.887269851111341,
298433.256156699149869,
298443.625074940966442,
298453.994024575513322,
298464.363005601800978,
298474.732018018898088,
298485.101061825698707,
298495.470137021446135,
298505.839243604859803,
298516.208381575124804,
298526.577550931309815,
298536.946751672250684,
298547.315983797132503,
298557.685247304907534,
298568.054542194528040,
298578.423868465179112,
298588.793226115638390,
298599.162615145090967,
298609.532035552489106,
298619.901487336901482,
298630.270970497280359,
298640.640485032636207,
298651.010030942037702,
298661.379608224553522,
298671.749216879019514,
298682.118856904562563,
298692.488528300309554,
298702.858231065096334,
298713.227965197991580,
298723.597730698063970,
298733.967527564207558,
298744.337355795607436,
298754.707215391157661,
298765.077106349868700,
298775.447028670867439,
298785.816982353047933,
298796.186967395478860,
298806.556983797287103,
298816.927031557250302,
298827.297110674553551,
298837.667221148149110,
298848.037362977105659,
298858.407536160433665,
298868.777740697085392,
298879.147976586187724,
298889.518243826576509,
298899.888542417553253,
298910.258872357779182,
298920.629233646555804,
298930.999626282893587,
298941.370050265570171,
298951.740505593828857,
298962.110992266563699,
298972.481510282901581,
298982.852059641736560,
298993.222640342137311,
299003.593252383230720,
299013.963895763794426,
299024.334570483129937,
299034.705276540073100,
299045.076013933634385,
299055.446782662940677,
299065.817582726827823,
299076.188414124597330,
299086.559276855026837,
299096.930170917185023,
299107.301096310140565,
299117.672053032903932,
299128.043041084427387,
299138.414060463837814,
299148.785111170087475,
299159.156193202186842,
299169.527306559146382,
299179.898451240092982,
299190.269627243920695,
299200.640834569639992,
299211.012073216435965,
299221.383343183086254,
299231.754644468834158,
299242.125977072573733,
299252.497340993315447,
299262.868736230186187,
299273.240162782021798,
299283.611620648065582,
299293.983109827095177,
299304.354630318412092,
299314.726182120793965,
299325.097765233367682,
299335.469379655143712,
299345.841025385132525,
299356.212702422344591,
299366.584410765848588,
299376.956150414538570,
299387.327921367599629,
299397.699723623867612,
299408.071557182585821,
299418.443422042648308,
299428.815318203065544,
299439.187245662906207,
299449.559204421064351,
299459.931194476725068,
299470.303215828840621,
299480.675268476421479,
299491.047352418536320,
299501.419467654079199,
299511.791614182235207,
299522.163792002014816,
299532.536001112253871,
299542.908241512137465,
299553.280513200617861,
299563.652816176763736,
299574.025150439643767,
299584.397515988093801,
299594.769912821298931,
299605.142340938211419,
299615.514800337899942,
299625.887291019374970,
299636.259812981588766,
299646.632366223610006,
299657.004950744507369,
299667.377566543291323,
299677.750213618914131,
299688.122891970444471,
299698.495601596892811,
299708.868342497211415,
299719.241114670585375,
299729.613918115966953,
299739.986752832308412,
299750.359618818678427,
299760.732516074145678,
299771.105444597662427,
299781.478404388297349,
299791.851395445002709,
299802.224417766905390,
299812.597471352957655,
299822.970556202228181,
299833.343672313669231,
299843.716819686349481,
299854.089998319279402,
299864.463208211469464,
299874.836449362046551,
299885.209721769904718,
299895.583025434054434,
299905.956360353680793,
299916.329726527619641,
299926.703123955056071,
299937.076552634884138,
299947.450012566114310,
299957.823503747931682,
299968.197026179172099,
299978.570579859078862,
299988.944164786487818,
299999.317780960409436,
300009.691428380028810,
300020.065107044123579,
300030.438816952053457,
300040.812558102596086,
300051.186330494820140,
300061.560134127794299,
300071.933969000470825,
300082.307835111976601,
300092.681732461322099,
300103.055661047401372,
300113.429620869341306,
300123.803611926152371,
300134.177634216845036,
300144.551687740546186,
300154.925772496091668,
300165.299888482666574,
300175.674035699223168,
300186.048214144713711,
300196.422423818381503,
300206.796664719004184,
300217.170936845766846,
300227.545240197738167,
300237.919574773695786,
300248.293940572883002,
300258.668337594252080,
300269.042765836871695,
300279.417225299752317,
300289.791715981846210,
300300.166237882280257,
300310.540790999948513,
300320.915375334094279,
300331.289990883495193,
300341.664637647336349,
300352.039315624628216,
300362.414024814264849,
300372.788765215489548,
300383.163536827196367,
300393.538339648337569,
300403.913173678098246,
300414.288038915430661,
300424.662935359403491,
300435.037863008968998,
300445.412821863137651,
300455.787811921094544,
300466.162833181675524,
300476.537885644065682,
300486.912969307159074,
300497.288084170024376,
300507.663230231788475,
300518.038407491345424,
300528.413615947763901,
300538.788855600112583,
300549.164126447285526,
300559.539428488526028,
300569.914761722728144,
300580.290126148844138,
300590.665521766059101,
300601.040948573208880,
300611.416406569594983,
300621.791895754053257,
300632.167416125594173,
300642.542967683402821,
300652.918550426315051,
300663.294164353515953,
300673.669809463899583,
300684.045485756534617,
300694.421193230547942,
300704.796931884833612,
300715.172701718576718,
300725.548502730671316,
300735.924334920127876,
300746.300198286073282,
300756.676092827459797,
300767.052018543356098,
300777.427975432772655,
300787.803963494719937,
300798.179982728324831,
300808.556033132539596,
300818.932114706316497,
300829.308227448840626,
300839.684371358947828,
300850.060546435881406,
300860.436752678593621,
300870.812990085978527,
300881.189258657279424,
300891.565558391332161,
300901.941889287321828,
300912.318251344258897,
300922.694644560979214,
300933.071068936784286,
300943.447524470509961,
300953.824011161341332,
300964.200529008114245,
300974.577078009955585,
300984.953658165992238,
300995.330269475060049,
301005.706911936344113,
301016.083585548854899,
301026.460290311544668,
301036.837026223482098,
301047.213793283735868,
301057.590591491258238,
301067.967420845117886,
301078.344281344325282,
301088.721172988007311,
301099.098095775116235,
301109.475049704604317,
301119.852034775714856,
301130.229050987225492,
301140.606098338379525,
301150.983176828071009,
301161.360286455368623,
301171.737427219282836,
301182.114599118882325,
301192.491802153235767,
301202.869036321353633,
301213.246301622071769,
301223.623598054749891,
301234.000925618223846,
301244.378284311562311,
301254.755674133717548,
301265.133095083816443,
301275.510547160927672,
301285.888030363945290,
301296.265544692054391,
301306.643090144149028,
301317.020666719297878,
301327.398274416627828,
301337.775913235032931,
301348.153583173640072,
301358.531284231459722,
301368.909016407560557,
301379.286779700778425,
301389.664574110414833,
301400.042399635305628,
301410.420256274635904,
301420.798144027299713,
301431.176062892423943,
301441.554012868960854,
301451.931993955979124,
301462.310006152605638,
301472.688049457734451,
301483.066123870434240,
301493.444229389773682,
301503.822366014821455,
301514.200533744471613,
301524.578732577792834,
301534.956962513970211,
301545.335223551839590,
301555.713515690586064,
301566.091838929103687,
301576.470193266577553,
301586.848578701901715,
301597.226995234261267,
301607.605442862550262,
301617.983921585837379,
301628.362431403133087,
301638.740972313622478,
301649.119544316141400,
301659.498147409816738,
301669.876781593600754,
301680.255446866736747,
301690.634143228002358,
301701.012870676582679,
301711.391629211429972,
301721.770418831671122,
301732.149239536316600,
301742.528091324318666,
301752.906974194745999,
301763.285888146783691,
301773.664833179209381,
301784.043809291208163,
301794.422816481790505,
301804.801854750025086,
301815.180924094864167,
301825.560024515492842,
301835.939156010688748,
301846.318318579753395,
301856.697512221464422,
301867.076736935123336,
301877.455992719682399,
301887.835279573977459,
301898.214597497368231,
301908.593946488574147,
301918.973326546838507,
301929.352737671113573,
301939.732179860468023,
301950.111653113970533,
301960.491157430573367,
301970.870692809286993,
301981.250259249296505,
301991.629856749554165,
302002.009485309012234,
302012.389144926797599,
302022.768835601920728,
302033.148557333450299,
302043.528310120454989,
302053.908093961828854,
302064.287908856756985,
302074.667754804133438,
302085.047631803143304,
302095.427539852738846,
302105.807478951872326,
302116.187449099787045,
302126.567450295377057,
302136.947482537711039,
302147.327545825799461,
302157.707640158711001,
302168.087765535514336,
302178.467921955161728,
302188.848109416721854,
302199.228327919263393,
302209.608577461796813,
302219.988858043332584,
302230.369169662939385,
302240.749512319627684,
302251.129886012582574,
302261.510290740639903,
302271.890726502868347,
302282.271193298452999,
302292.651691126171499,
302303.032219985325355,
302313.412779874808621,
302323.793370793689974,
302334.173992741096299,
302344.554645715863444,
302354.935329717176501,
302365.316044744104147,
302375.696790795482229,
302386.077567870612256,
302396.458375968388282,
302406.839215087820776,
302417.220085227978416,
302427.600986387988087,
302437.981918566685636,
302448.362881763314363,
302458.743875976768322,
302469.124901206232607,
302479.505957450659480,
302489.887044709059410,
302500.268162980501074,
302510.649312264053151,
302521.030492558667902,
302531.411703863472212,
302541.792946177476551,
302552.174219499691390,
302562.555523829185404,
302572.936859164969064,
302583.318225506169256,
302593.699622851680033,
302604.081051200628281,
302614.462510552082676,
302624.844000904995482,
302635.225522258493584,
302645.607074611587450,
302655.988657963229343,
302666.370272312546149,
302676.751917658664752,
302687.133594000362791,
302697.515301336941775,
302707.897039667295758,
302718.278808990493417,
302728.660609305661637,
302739.042440611694474,
302749.424302907777019,
302759.806196192803327,
302770.188120465842076,
302780.570075726078358,
302790.952061972406227,
302801.334079203894362,
302811.716127419611439,
302822.098206618626136,
302832.480316799832508,
302842.862457962473854,
302853.244630105386022,
302863.626833227812313,
302874.009067328588571,
302884.391332407016307,
302894.773628461873159,
302905.155955492286012,
302915.538313497323543,
302925.920702476112638,
302936.303122427489143,
302946.685573350638151,
302957.068055244570132,
302967.450568108353764,
302977.833111940941308,
302988.215686741459649,
302998.598292508919258,
303008.980929242388811,
303019.363596940820571,
303029.746295603341423,
303040.129025228961837,
303050.511785816808697,
303060.894577365776058,
303071.277399874932598,
303081.660253343463410,
303092.043137770320754,
303102.426053154398687,
303112.808999494940508,
303123.191976790956687,
303133.574985041399486,
303143.958024245453998,
303154.341094401956070,
303164.724195510149002,
303175.107327568985056,
303185.490490577474702,
303195.873684534744825,
303206.256909439747687,
303216.640165291610174,
303227.023452089284547,
303237.406769831839483,
303247.790118518401869,
303258.173498147982173,
303268.556908719532657,
303278.940350232180208,
303289.323822684935294,
303299.707326076808386,
303310.090860406926367,
303320.474425674241502,
303330.858021877880674,
303341.241649016912561,
303351.625307090231217,
303362.008996097021736,
303372.392716036178172,
303382.776466906885616,
303393.160248708212748,
303403.544061438995413,
303413.927905098535120,
303424.311779685725924,
303434.695685199578293,
303445.079621639219113,
303455.463589003658853,
303465.847587291966192,
303476.231616503151599,
303486.615676636341959,
303496.999767690373119,
303507.383889664546587,
303517.768042557756416,
303528.152226369013079,
303538.536441097559873,
303548.920686742174439,
303559.304963302158285,
303569.689270776405465,
303580.073609163868241,
303590.457978463848121,
303600.842378675180953,
303611.226809796993621,
303621.611271828354802,
303631.995764768216759,
303642.380288615706377,
303652.764843369892333,
303663.149429029726889,
303673.534045594278723,
303683.918693062674720,
303694.303371433750726,
303704.688080706866458,
303715.072820880741347,
303725.457591954735108,
303735.842393927683588,
303746.227226798539050,
303756.612090566719417,
303766.996985230944119,
303777.381910790281836,
303787.766867243975867,
303798.151854590803850,
303808.536872830067296,
303818.921921960776672,
303829.307001981709618,
303839.692112892284058,
303850.077254691277631,
303860.462427377759013,
303870.847630950971507,
303881.232865409809165,
303891.618130753224250,
303902.003426980460063,
303912.388754090527073,
303922.774112082377542,
303933.159500955138355,
303943.544920707703568,
303953.930371339374688,
303964.315852848987561,
303974.701365235669073,
303985.086908498546109,
303995.472482636454515,
304005.858087648579385,
304016.243723534047604,
304026.629390291753225,
304037.015087920823134,
304047.400816420267802,
304057.786575789097697,
304068.172366026497912,
304078.558187131420709,
304088.944039102818351,
304099.329921939992346,
304109.715835641778540,
304120.101780207245611,
304130.487755635578651,
304140.873761925555300,
304151.259799076593481,
304161.645867087470833,
304172.031965957256034,
304182.418095685075969,
304192.804256269941106,
304203.190447710920125,
304213.576670007139910,
304223.962923157494515,
304234.349207160994411,
304244.735522016882896,
304255.121867724054027,
304265.508244281692896,
304275.894651688751765,
304286.281089944299310,
304296.667559047346003,
304307.054058997076936,
304317.440589792269748,
304327.827151432284154,
304338.213743916014209,
304348.600367242470384,
304358.987021410779562,
304369.373706419952214,
304379.760422269057017,
304390.147168957162648,
304400.533946483279578,
304410.920754846418276,
304421.307594045763835,
304431.694464080152102,
304442.081364948884584,
304452.468296650913544,
304462.855259185133036,
304473.242252550844569,
304483.629276746883988,
304494.016331772378180,
304504.403417626454029,
304514.790534308005590,
304525.177681816276163,
304535.564860150218010,
304545.952069308725186,
304556.339309291157406,
304566.726580096350517,
304577.113881723373197,
304587.501214171352331,
304597.888577439356595,
304608.275971526280046,
304618.663396431307774,
304629.050852153392043,
304639.438338691659737,
304649.825856045237742,
304660.213404212961905,
304670.600983194075525,
304680.988592987589072,
304691.376233592454810,
304701.763905007799622,
304712.151607232750393,
304722.539340266084764,
304732.927104107220657,
304743.314898754993919,
304753.702724208356813,
304764.090580466669053,
304774.478467528650071,
304784.866385393601377,
304795.254334060533438,
304805.642313528340310,
304816.030323796265293,
304826.418364863260649,
304836.806436728336848,
304847.194539390620776,
304857.582672849122901,
304867.970837102911901,
304878.359032151056454,
304888.747257992625237,
304899.135514626570512,
304909.523802052077372,
304919.912120268039871,
304930.300469273759518,
304940.688849068013951,
304951.077259649988264,
304961.465701018751133,
304971.854173173254821,
304982.242676112684421,
304992.631209835992195,
305003.019774342305027,
305013.408369630575180,
305023.796995699929539,
305034.185652549436782,
305044.574340178049169,
305054.963058584951796,
305065.351807769096922,
305075.740587729611434,
305086.129398465505801,
305096.518239975848701,
305106.907112259708811,
305117.296015316096600,
305127.684949143964332,
305138.073913742613513,
305148.462909110938199,
305158.851935248007067,
305169.240992152888793,
305179.630079824652057,
305190.019198262307327,
305200.408347464981489,
305210.797527431568597,
305221.186738161311951,
305231.575979653222021,
305241.965251906251069,
305252.354554919584189,
305262.743888692231849,
305273.133253223088104,
305283.522648511454463,
305293.912074556283187,
305304.301531356526539,
305314.691018911427818,
305325.080537219939288,
305335.470086281013209,
305345.859666093951091,
305356.249276657646988,
305366.638917971053161,
305377.028590033471119,
305387.418292843794916,
305397.808026401151437,
305408.197790704551153,
305418.587585753004532,
305428.977411545696668,
305439.367268081521615,
305449.757155359606259,
305460.147073379135691,
305470.537022139003966,
305480.927001638221554,
305491.317011876031756,
305501.707052851445042,
305512.097124563239049,
305522.487227010889910,
305532.877360193175264,
305543.267524109280203,
305553.657718758156989,
305564.047944138874300,
305574.438200250617228,
305584.828487092338037,
305595.218804663047194,
305605.609152961871587,
305615.999531987938099,
305626.389941740140785,
305636.780382217606530,
305647.170853419462219,
305657.561355344660114,
305667.951887992268894,
305678.342451361473650,
305688.733045451110229,
305699.123670260363724,
305709.514325788419228,
305719.905012033996172,
305730.295728996454272,
305740.686476674687583,
305751.077255067881197,
305761.468064174987376,
305771.858903995133005,
305782.249774527270347,
305792.640675770642702,
305803.031607724027708,
305813.422570386726875,
305823.813563757750671,
305834.204587836051360,
305844.595642620814033,
305854.986728111049160,
305865.377844305650797,
305875.768991203920450,
305886.160168804926798,
305896.551377107389271,
305906.942616110784002,
305917.333885813946836,
305927.725186215888243,
305938.116517315793317,
305948.507879112672526,
305958.899271605594549,
305969.290694793628063,
305979.682148675783537,
305990.073633251129650,
306000.465148518793285,
306010.856694477668498,
306021.248271126998588,
306031.639878465794027,
306042.031516493007075,
306052.423185207881033,
306062.814884609309956,
306073.206614696420729,
306083.598375468223821,
306093.990166923904326,
306104.381989062298089,
306114.773841882706620,
306125.165725384082180,
306135.557639565377031,
306145.949584425834473,
306156.341559964406770,
306166.733566180220805,
306177.125603072228841,
306187.517670639557764,
306197.909768881334458,
306208.301897796511184,
306218.694057384156622,
306229.086247643397655,
306239.478468573244754,
306249.870720172650181,
306260.263002440915443,
306270.655315376992803,
306281.047658979834523,
306291.440033248625696,
306301.832438182376791,
306312.224873780098278,
306322.617340040975250,
306333.009836964018177,
306343.402364548237529,
306353.794922792760190,
306364.187511696596630,
306374.580131258757319,
306384.972781478485558,
306395.365462354675401,
306405.758173886395525,
306416.150916072772816,
306426.543688912875950,
306436.936492405657191,
306447.329326550301630,
306457.722191345761530,
306468.115086791163776,
306478.508012885635253,
306488.900969628070015,
306499.293957017653156,
306509.686975053453352,
306520.080023734364659,
306530.473103059746791,
306540.866213028435595,
306551.259353639383335,
306561.652524892007932,
306572.045726785087027,
306582.438959317747504,
306592.832222489116248,
306603.225516298203729,
306613.618840744078625,
306624.012195825751405,
306634.405581542407162,
306644.798997892998159,
306655.192444876593072,
306665.585922492376994,
306675.979430739127565,
306686.372969616262708,
306696.766539122676477,
306707.160139257321134,
306717.553770019439980,
306727.947431408043485,
306738.341123422142118,
306748.734846060804557,
306759.128599323157687,
306769.522383208211977,
306779.916197715036105,
306790.310042842640541,
306800.703918590268586,
306811.097824956814293,
306821.491761941288132,
306831.885729542933404,
306842.279727760760579,
306852.673756593721919,
306863.067816041002516,
306873.461906101671048,
306883.856026774621569,
306894.250178059155587,
306904.644359954108950,
306915.038572458666749,
306925.432815571955871,
306935.827089292986784,
306946.221393620653544,
306956.615728554257657,
306967.010094092809595,
306977.404490235203411,
306987.798916980740614,
306998.193374328315258,
307008.587862277054228,
307018.982380826026201,
307029.376929974299856,
307039.771509720885660,
307050.166120064910501,
307060.560761005501263,
307070.955432541493792,
307081.350134672131389,
307091.744867396482732,
307102.139630713500082,
307112.534424622368533,
307122.929249122098554,
307133.324104211700615,
307143.718989890359808,
307154.113906157028396,
307164.508853010833263,
307174.903830450901296,
307185.298838476126548,
307195.693877085635904,
307206.088946278614458,
307216.484046054014470,
307226.879176410846412,
307237.274337348295376,
307247.669528865430038,
307258.064750961202662,
307268.460003634740133,
307278.855286885111127,
307289.250600711442530,
307299.645945112744812,
307310.041320087970234,
307320.436725636420306,
307330.832161756930873,
307341.227628448628820,
307351.623125710699242,
307362.018653542152606,
307372.414211941882968,
307382.809800909191836,
307393.205420443089679,
307403.601070542528760,
307413.996751206694171,
307424.392462434596382,
307434.788204225304071,
307445.183976577944122,
307455.579779491468798,
307465.975612965004984,
307476.371476997621357,
307486.767371588444803,
307497.163296736369375,
307507.559252440696582,
307517.955238700320479,
307528.351255514309742,
307538.747302881791256,
307549.143380801891908,
307559.539489273447543,
307569.935628295817878,
307580.331797867896967,
307590.727997988695279,
307601.124228657514323,
307611.520489873189945,
307621.916781634907238,
307632.313103941676673,
307642.709456792625133,
307653.105840186763089,
307663.502254123159219,
307673.898698600940406,
307684.295173619058914,
307694.691679176758043,
307705.088215272990055,
307715.484781906765420,
307725.881379077269230,
307736.278006783511955,
307746.674665024504066,
307757.071353799547069,
307767.468073107418604,
307777.864822947303765,
307788.261603318329435,
307798.658414219564293,
307809.055255649902392,
307819.452127608587034,
307829.849030094628688,
307840.245963107096031,
307850.642926645115949,
307861.039920707640704,
307871.436945293797180,
307881.834000402654056,
307892.231086033280008,
307902.628202184743714,
307913.025348856113851,
307923.422526046459097,
307933.819733754906338,
307944.216971980407834,
307954.614240722090472,
307965.011539979022928,
307975.408869750273880,
307985.806230034970213,
307996.203620832005981,
308006.601042140624486,
308016.998493959894404,
308027.395976288767997,
308037.793489126430359,
308048.191032471891958,
308058.588606324105058,
308068.986210682371166,
308079.383845545642544,
308089.781510912987869,
308100.179206783475820,
308110.576933156116866,
308120.974690030096099,
308131.372477404423989,
308141.770295278285630,
308152.168143650458660,
308162.566022520361003,
308172.963931886828505,
308183.361871748988051,
308193.759842105966527,
308204.157842956774402,
308214.555874300480355,
308224.953936136269476,
308235.352028462977614,
308245.750151279906277,
308256.148304586007725,
308266.546488380408846,
308276.944702662061900,
308287.342947430210188,
308297.741222683805972,
308308.139528421976138,
308318.537864643789362,
308328.936231348256115,
308339.334628534386866,
308349.733056201483123,
308360.131514348555356,
308370.530002974439412,
308380.928522078495007,
308391.327071659616195,
308401.725651716929860,
308412.124262249504682,
308422.522903256409336,
308432.921574736712500,
308443.320276689482853,
308453.719009113789070,
308464.117772008758038,
308474.516565373400226,
308484.915389206842519,
308495.314243508037180,
308505.713128276227508,
308516.112043510307558,
308526.510989209462423,
308536.909965372818988,
308547.308971999329515,
308557.708009088004474,
308568.107076638087165,
308578.506174648588058,
308588.905303118575830,
308599.304462047060952,
308609.703651433286723,
308620.102871276147198,
308630.502121574711055,
308640.901402328221593,
308651.300713535631076,
308661.700055195949972,
308672.099427308421582,
308682.498829871998169,
308692.898262885748409,
308703.297726348799188,
308713.697220260277390,
308724.096744619018864,
308734.496299424325116,
308744.895884675264824,
308755.295500370790251,
308765.695146510086488,
308776.094823092164006,
308786.494530115975067,
308796.894267580879387,
308807.294035485771019,
308817.693833829660434,
308828.093662611790933,
308838.493521831114776,
308848.893411486758851,
308859.293331577791832,
308869.693282103224192,
308880.093263062182814,
308890.493274453794584,
308900.893316277128179,
308911.293388531077653,
308921.693491214886308,
308932.093624327622820,
308942.493787868297659,
308952.893981836095918,
308963.294206229969859,
308973.694461048929952,
308984.094746292335913,
308994.495061958965380,
309004.895408048061654,
309015.295784558576997,
309025.696191489754710,
309036.096628840488847,
309046.497096609964501,
309056.897594797250349,
309067.298123401298653,
309077.698682421410922,
309088.099271856539417,
309098.499891705694608,
309108.900541968003381,
309119.301222642592620,
309129.701933728531003,
309140.102675224770792,
309150.503447130497079,
309160.904249444836751,
309171.305082166683860,
309181.705945295281708,
309192.106838829640765,
309202.507762768771499,
309212.908717111859005,
309223.309701857971959,
309233.710717006062623,
309244.111762555432506,
309254.512838504975662,
309264.913944853702560,
309275.315081600914709,
309285.716248745622579,
309296.117446286662016,
309306.518674223450944,
309316.919932554941624,
309327.321221280086320,
309337.722540398070123,
309348.123889908019919,
309358.525269808829762,
309368.926680099859368,
309379.328120779944584,
309389.729591848154087,
309400.131093303789385,
309410.532625145686325,
309420.934187373088207,
309431.335779985005502,
309441.737402980506886,
309452.139056358602829,
309462.540740118536633,
309472.942454259260558,
309483.344198779901490,
309493.745973679469898,
309504.147778957150877,
309514.549614611954894,
309524.951480642892420,
309535.353377049206756,
309545.755303829850163,
309556.157260983949527,
309566.559248510573525,
309576.961266408790834,
309587.363314677670132,
309597.765393316280097,
309608.167502323805820,
309618.569641699083149,
309628.971811441471800,
309639.374011549924035,
309649.776242023450322,
309660.178502861235756,
309670.580794062349014,
309680.983115625742357,
309691.385467550659087,
309701.787849836167879,
309712.190262481220998,
309722.592705484887119,
309732.995178846351337,
309743.397682564740535,
309753.800216638948768,
309764.202781068277545,
309774.605375851620920,
309785.008000988105778,
309795.410656476859003,
309805.813342316949274,
309816.216058507387061,
309826.618805047299247,
309837.021581935812719,
309847.424389171879739,
309857.827226754743606,
309868.230094683356583,
309878.632992956787348,
309889.035921574279200,
309899.438880534784403,
309909.841869837255217,
309920.244889481051359,
309930.647939465125091,
309941.051019788486883,
309951.454130450263619,
309961.857271449523978,
309972.260442785453051,
309982.663644456944894,
309993.066876463301014,
310003.470138803415466,
310013.873431476415135,
310024.276754481426906,
310034.680107817461248,
310045.083491483645048,
310055.486905479163397,
310065.890349802852143,
310076.293824453896377,
310086.697329431539401,
310097.100864734675270,
310107.504430362372659,
310117.908026313874871,
310128.311652588192374,
310138.715309184277430,
310149.118996101315133,
310159.522713338432368,
310169.926460894639604,
310180.330238768947311,
310190.734046960656997,
310201.137885468662716,
310211.541754292091355,
310221.945653430011589,
310232.349582881608512,
310242.753542645776179,
310253.157532721816096,
310263.561553108622320,
310273.965603805321734,
310284.369684811099432,
310294.773796124907676,
310305.177937745873351,
310315.582109673123341,
310325.986311905668117,
310336.390544442634564,
310346.794807283033151,
310357.199100426107179,
310367.603423870808911,
310378.007777616148815,
310388.412161661428399,
310398.816576005541719,
310409.221020647615660,
310419.625495586777106,
310430.030000822094735,
310440.434536352637224,
310450.839102177473251,
310461.243698295787908,
310471.648324706417043,
310482.052981408662163,
310492.457668401650153,
310502.862385684216861,
310513.267133255722001,
310523.671911115001421,
310534.076719261356629,
310544.481557693623472,
310554.886426411103457,
310565.291325412807055,
310575.696254697744735,
310586.101214265159797,
310596.506204114004504,
310606.911224243347533,
310617.316274652432185,
310627.721355340210721,
310638.126466305635404,
310648.531607548065949,
310658.936779066512827,
310669.341980859870091,
310679.747212927439250,
310690.152475268288981,
310700.557767881371547,
310710.963090765755624,
310721.368443920742720,
310731.773827345285099,
310742.179241038335022,
310752.584684999193996,
310762.990159226814285,
310773.395663720322773,
310783.801198478788137,
310794.206763501395471,
310804.612358786980622,
310815.017984334961511,
310825.423640144173987,
310835.829326213744935,
310846.235042542801239,
310856.640789130527992,
310867.046565975819249,
310877.452373077743687,
310887.858210435602814,
310898.264078048290685,
310908.669975914992392,
310919.075904034776613,
310929.481862406712025,
310939.887851029809099,
310950.293869903311133,
310960.699919026170392,
310971.105998397571966,
310981.512108016526327,
310991.918247882160358,
311002.324417993542738,
311012.730618349683937,
311023.136848949827254,
311033.543109792983159,
311043.949400878103916,
311054.355722204491030,
311064.762073771154974,
311075.168455577106215,
311085.574867621529847,
311095.981309903436340,
311106.387782421894372,
311116.794285176147241,
311127.200818165147211,
311137.607381387962960,
311148.013974843779579,
311158.420598531607538,
311168.827252450515516,
311179.233936599630397,
311189.640650978079066,
311200.047395584930200,
311210.454170419136062,
311220.860975479998160,
311231.267810766410548,
311241.674676277558319,
311252.081572012510151,
311262.488497970451135,
311272.895454150275327,
311283.302440551167820,
311293.709457172255497,
311304.116504012548830,
311314.523581071174704,
311324.930688347260002,
311335.337825839873403,
311345.744993547967169,
311356.152191470784601,
311366.559419607336167,
311376.966677956748754,
311387.373966518149246,
311397.781285290548112,
311408.188634273072239,
311418.596013464732096,
311429.003422864770982,
311439.410862472082954,
311449.818332285853103,
311460.225832305266522,
311470.633362529275473,
311481.040922956948634,
311491.448513587529305,
311501.856134419969749,
311512.263785453396849,
311522.671466686879285,
311533.079178119602147,
311543.486919750517700,
311553.894691578752827,
311564.302493603434414,
311574.710325823689345,
311585.118188238469884,
311595.526080846961122,
311605.934003648231737,
311616.341956641350407,
311626.749939825444017,
311637.157953199639451,
311647.565996762947179,
311657.974070514435880,
311668.382174453290645,
311678.790308578521945,
311689.198472889198456,
311699.606667384505272,
311710.014892063452862,
311720.423146925168112,
311730.831431968719698,
311741.239747193234507,
311751.648092597723007,
311762.056468181428500,
311772.464873943245038,
311782.873309882415924,
311793.281775997893419,
311803.690272288862616,
311814.098798754392192,
311824.507355393550824,
311834.915942205523606,
311845.324559189321008,
311855.733206343953498,
311866.141883668606170,
311876.550591162464116,
311886.959328824421391,
311897.368096653604880,
311907.776894649257883,
311918.185722810332663,
311928.594581135897897,
311939.003469625196885,
311949.412388277240098,
311959.821337090979796,
311970.230316065659281,
311980.639325200463645,
311991.048364494170528,
312001.457433946197852,
312011.866533555439673,
312022.275663321081083,
312032.684823242074344,
312043.094013317662757,
312053.503233546973206,
312063.912483928899746,
312074.321764462685678,
312084.731075147399679,
312095.140415982052218,
312105.549786965828389,
312115.959188097738661,
312126.368619376968127,
312136.778080802469049,
312147.187572373542935,
312157.597094089142047,
312168.006645948276855,
312178.416227950248867,
312188.825840093952138,
312199.235482378571760,
312209.645154803292826,
312220.054857367009390,
312230.464590069022961,
312240.874352908111177,
312251.284145883750170,
312261.693968994775787,
312272.103822240314912,
312282.513705619552638,
312292.923619131557643,
312303.333562775282189,
312313.743536550027784,
312324.153540454688482,
312334.563574488565791,
312344.973638650553767,
312355.383732939895708,
312365.793857355543878,
312376.204011896625161,
312386.614196562382858,
312397.024411351827439,
312407.434656263911165,
312417.844931297819130,
312428.255236452794634,
312438.665571727731731,
312449.075937121757306,
312459.486332633998245,
312469.896758263581432,
312480.307214009575546,
312490.717699871049263,
312501.128215847129468,
312511.538761936826631,
312521.949338139325846,
312532.359944453695789,
312542.770580879063345,
312553.181247414380778,
312563.591944058949593,
312574.002670811722055,
312584.413427671766840,
312594.824214638327248,
312605.235031710413750,
312615.645878887036815,
312626.056756167439744,
312636.467663550574798,
312646.878601035685278,
312657.289568621723447,
312667.700566307816189,
312678.111594093148597,
312688.522651976731140,
312698.933739957632497,
312709.344858035037760,
312719.756006208073813,
312730.167184475634713,
312740.578392836963758,
312750.989631291187834,
312761.400899837259203,
312771.812198474421166,
312782.223527201684192,
312792.634886018116958,
312803.046274922904558,
312813.457693915115669,
312823.869142993877176,
312834.280622158083133,
312844.692131407035049,
312855.103670739801601,
312865.515240155451465,
312875.926839653053321,
312886.338469231734052,
312896.750128890504129,
312907.161818628665060,
312917.573538445110898,
312927.985288339026738,
312938.397068309423048,
312948.808878355543129,
312959.220718476339243,
312969.632588670996483,
312980.044488938583527,
312990.456419278169051,
313000.868379688938148,
313011.280370169784874,
313021.692390720068943,
313032.104441338800825,
313042.516522024874575,
313052.928632777591702,
313063.340773596137296,
313073.752944479230791,
313084.165145426406525,
313094.577376436558552,
313104.989637508697342,
313115.401928642007988,
313125.814249835675582,
313136.226601088710595,
313146.638982400065288,
313157.051393769099377,
313167.463835194765124,
313177.876306676189415,
313188.288808212440927,
313198.701339802704751,
313209.113901445991360,
313219.526493141311221,
313229.939114888024051,
313240.351766685023904,
313250.764448531379458,
313261.177160426275805,
313271.589902368898038,
313282.002674358198419,
313292.415476393245626,
313302.828308473282959,
313313.241170597320888,
313323.654062764428090,
313334.066984973789658,
313344.479937224532478,
313354.892919515608810,
313365.305931846087333,
313375.718974215327762,
313386.132046622165944,
313396.545149065786973,
313406.958281545375939,
313417.371444059943315,
313427.784636608557776,
313438.197859190346207,
313448.611111804493703,
313459.024394450010732,
313469.437707125965971,
313479.851049831486307,
313490.264422565756831,
313500.677825327729806,
313511.091258116590325,
313521.504720931465272,
313531.918213771365117,
313542.331736635416746,
313552.745289522805251,
313563.158872432541102,
313573.572485363692977,
313583.986128315445967,
313594.399801286868751,
313604.813504277030006,
313615.227237285056617,
313625.641000310075469,
313636.054793351155240,
313646.468616407306399,
313656.882469477830455,
313667.296352561679669,
313677.710265657980926,
313688.124208765744697,
313698.538181884214282,
313708.952185012516566,
313719.366218149603810,
313729.780281294661108,
313740.194374446757138,
313750.608497605018783,
313761.022650768456515,
313771.436833936313633,
313781.851047107658815,
313792.265290281502530,
313802.679563457029872,
313813.093866633309517,
313823.508199809410144,
313833.922562984342221,
313844.336956157465465,
313854.751379327790346,
313865.165832494210918,
313875.580315656145103,
313885.994828812428750,
313896.409371962305158,
313906.823945104784798,
313917.238548239052761,
313927.653181364177726,
313938.067844479228370,
313948.482537583331577,
313958.897260675614234,
313969.312013755145017,
313979.726796820992604,
313990.141609872400295,
314000.556452908262145,
314010.971325927821454,
314021.386228930146899,
314031.801161914365366,
314042.216124879429117,
314052.631117824639659,
314063.046140749065671,
314073.461193651717622,
314083.876276531722397,
314094.291389388148673,
314104.706532220239751,
314115.121705026947893,
314125.536907807399984,
314135.952140560781118,
314146.367403286159970,
314156.782695982488804,
314167.198018649127334,
314177.613371285027824,
314188.028753889200743,
314198.444166461005807,
314208.859608999337070,
314219.275081503437832,
314229.690583972202148,
314240.106116404873319,
314250.521678800578229,
314260.937271158327349,
314271.352893477363978,
314281.768545756698586,
314292.184227995458059,
314302.599940192594659,
314313.015682347468100,
314323.431454458972439,
314333.847256526292767,
314344.263088548614178,
314354.678950524888933,
314365.094842454302125,
314375.510764335922431,
314385.926716168876737,
314396.342697952350136,
314406.758709685236681,
314417.174751366779674,
314427.590822996164206,
314438.006924572342541,
314448.423056094383355,
314458.839217561588157,
314469.255408972909208,
314479.671630327531602,
314490.087881624407601,
314500.504162862780504,
314510.920474041777197,
314521.336815160408150,
314531.753186217800248,
314542.169587213080376,
314552.586018145375419,
314563.002479013695847,
314573.418969817284960,
314583.835490555153228,
314594.252041226369329,
314604.668621830118354,
314615.085232365468983,
314625.501872831548098,
314635.918543227366172,
314646.335243552224711,
314656.751973805017769,
314667.168733984930441,
314677.585524091147818,
314688.002344122622162,
314698.419194078654982,
314708.836073958140332,
314719.252983760263305,
314729.669923484150786,
314740.086893128871452,
314750.503892693610396,
314760.920922177494504,
314771.337981579417828,
314781.755070898623671,
314792.172190134297125,
314802.589339285448659,
314813.006518351146951,
314823.423727330518886,
314833.840966222749557,
314844.258235026907641,
314854.675533742003608,
314865.092862367280759,
314875.510220901807770,
314885.927609344595112,
314896.345027694827877,
314906.762475951691158,
314917.179954114137217,
314927.597462181292940,
314938.015000152343418,
314948.432568026357330,
314958.850165802519768,
314969.267793479782995,
314979.685451057332102,
314990.103138534235768,
315000.520855909679085,
315010.938603182730731,
315021.356380352459382,
315031.774187418050133,
315042.192024378513452,
315052.609891233034432,
315063.027787980623543,
315073.445714620465878,
315083.863671151688322,
315094.281657573417760,
315104.699673884606455,
315115.117720084439497,
315125.535796172160190,
315135.953902146720793,
315146.372038007248193,
315156.790203752811067,
315167.208399382710923,
315177.626624895841815,
315188.044880291272420,
315198.463165568362456,
315208.881480726064183,
315219.299825763504487,
315229.718200679693837,
315240.136605473933741,
315250.555040145234670,
315260.973504692548886,
315271.391999115294311,
315281.810523412365001,
315292.229077582946047,
315302.647661626106128,
315313.066275540972129,
315323.484919326612726,
315333.903592982213013,
315344.322296506783459,
315354.741029899509158,
315365.159793159575202,
315375.578586285817437,
315385.997409277595580,
315396.416262133978307,
315406.835144854034297,
315417.254057436774019,
315427.672999881498981,
315438.091972187161446,
315448.510974352888297,
315458.930006377922837,
315469.349068261275534,
315479.768160002073273,
315490.187281599268317,
315500.606433052220382,
315511.025614359939937,
315521.444825521553867,
315531.864066536014434,
315542.283337402681354,
315552.702638120506890,
315563.121968688617926,
315573.541329106141347,
315583.960719372145832,
315594.380139485816471,
315604.799589446221944,
315615.219069252547342,
315625.638578903686721,
315636.058118398999795,
315646.477687737438828,
315656.897286918130703,
315667.316915940318722,
315677.736574802838732,
315688.156263505166862,
315698.575982046080753,
315708.995730424881913,
315719.415508640580811,
315729.835316692420747,
315740.255154579412192,
315750.675022300565615,
315761.094919855124317,
315771.514847242273390,
315781.934804460965097,
315792.354791510268115,
315802.774808389542159,
315813.194855097739492,
315823.614931633928791,
315834.035037997236941,
315844.455174186849035,
315854.875340201833751,
315865.295536041259766,
315875.715761704312172,
315886.136017190117855,
315896.556302497687284,
315906.976617626147345,
315917.396962574683130,
315927.817337342363317,
315938.237741928372998,
315948.658176331606228,
315959.078640551422723,
315969.499134586832952,
315979.919658436847385,
315990.340212100767531,
316000.760795577545650,
316011.181408866483252,
316021.602051966416184,
316032.022724876645952,
316042.443427596357651,
316052.864160124387126,
316063.284922460094094,
316073.705714602489024,
316084.126536550698802,
316094.547388303850312,
316104.968269861012232,
316115.389181221427862,
316125.810122383991256,
316136.231093347945716,
316146.652094112359919,
316157.073124676418956,
316167.494185039249714,
316177.915275199746247,
316188.336395157268271,
316198.757544910826255,
316209.178724459547084,
316219.599933802557644,
316230.021172938926611,
316240.442441867780872,
316250.863740588247310,
316261.285069099452812,
316271.706427400466055,
316282.127815490530338,
316292.549233368481509,
316302.970681033679284,
316313.392158485192340,
316323.813665722147562,
316334.235202743439004,
316344.656769548484590,
316355.078366136236582,
316365.499992505763657,
316375.921648656367324,
316386.343334587058052,
316396.765050296846312,
316407.186795784975402,
316417.608571050514001,
316428.030376092588995,
316438.452210910269059,
316448.874075502681080,
316459.295969869010150,
316469.717894008324947,
316480.139847919694148,
316490.561831602302846,
316500.983845055277925,
316511.405888277688064,
316521.827961268485524,
316532.250064027088229,
316542.672196552506648,
316553.094358843751252,
316563.516550899948925,
316573.938772720342968,
316584.361024303943850,
316594.783305649936665,
316605.205616757331882,
316615.627957625256386,
316626.050328252953477,
316636.472728639433626,
316646.895158783881925,
316657.317618685308844,
316667.740108342899475,
316678.162627755722497,
316688.585176922963001,
316699.007755843689665,
316709.430364517087582,
316719.853002942050807,
316730.275671117997263,
316740.698369043821003,
316751.121096718707122,
316761.543854141782504,
316771.966641312115826,
316782.389458228950389,
316792.812304891238455,
316803.235181298223324,
316813.658087448973674,
316824.081023342558183,
316834.503988978045527,
316844.926984354795422,
316855.350009471760131,
316865.773064328008331,
316876.196148922608700,
316886.619263254862744,
316897.042407323839143,
316907.465581128606573,
316917.888784668175504,
316928.312017941847444,
316938.735280948749278,
316949.158573687775061,
316959.581896158226300,
316970.005248359171674,
316980.428630289796274,
316990.852041949052364,
317001.275483336183242,
317011.698954450315796,
317022.122455290460493,
317032.545985855802428,
317042.969546145410277,
317053.393136158527341,
317063.816755894222297,
317074.240405351389199,
317084.664084529445972,
317095.087793427461293,
317105.511532044329215,
317115.935300379467662,
317126.359098431770690,
317136.782926200481597,
317147.206783684552647,
317157.630670883285347,
317168.054587795748375,
317178.478534421068616,
317188.902510758198332,
317199.326516806497239,
317209.750552564975806,
317220.174618032760918,
317230.598713208863046,
317241.022838092525490,
317251.446992682875134,
317261.871176978980657,
317272.295390979910735,
317282.719634684966877,
317293.143908093043137,
317303.568211203324609,
317313.992544015054591,
317324.416906527185347,
317334.841298738960177,
317345.265720649331342,
317355.690172257658560,
317366.114653562952299,
317376.539164564223029,
317386.963705260597635,
317397.388275651377626,
317407.812875735573471,
317418.237505512253847,
317428.662164980545640,
317439.086854139750358,
317449.511572988762055,
317459.936321526824031,
317470.361099752888549,
317480.785907666373532,
317491.210745266056620,
317501.635612551239319,
317512.060509521164931,
317522.485436174727511,
317532.910392511170357,
317543.335378529445734,
317553.760394228913356,
317564.185439608525485,
317574.610514667525422,
317585.035619404865429,
317595.460753819788806,
317605.885917911422439,
317616.311111678835005,
317626.736335121095181,
317637.161588237446267,
317647.586871026956942,
317658.012183488754090,
317668.437525621848181,
317678.862897425540723,
317689.288298898842186,
317699.713730040879454,
317710.139190850779414,
317720.564681327610742,
317730.990201470674947,
317741.415751278924290,
317751.841330751427449,
317762.266939887485933,
317772.692578686110210,
317783.118247146427166,
317793.543945267680101,
317803.969673048763070,
317814.395430488977581,
317824.821217587334104,
317835.247034343017731,
317845.672880755097140,
317856.098756822815631,
317866.524662545125466,
317876.950597921211738,
317887.376562950259540,
317897.802557631395757,
317908.228581963572651,
317918.654635946149938,
317929.080719578021672,
317939.506832858489361,
317949.932975786505267,
317960.359148361312691,
317970.785350581980310,
317981.211582447751425,
317991.637843957461882,
318002.064135110529605,
318012.490455906023271,
318022.916806342953350,
318033.343186420388520,
318043.769596137630288,
318054.196035493805539,
318064.622504487808328,
318075.049003118998371,
318085.475531386444345,
318095.902089289156720,
318106.328676826262381,
318116.755293997004628,
318127.181940800510347,
318137.608617235848214,
318148.035323302028701,
318158.462058998295106,
318168.888824323832523,
318179.315619277651422,
318189.742443858820479,
318200.169298066641204,
318210.596181900124066,
318221.023095358395949,
318231.450038440583739,
318241.877011145872530,
318252.304013473330997,
318262.731045422027819,
318273.158106991089880,
318283.585198179760482,
318294.012318987108301,
318304.439469412202016,
318314.866649454168510,
318325.293859112251084,
318335.721098385460209,
318346.148367272922769,
318356.575665773707442,
318367.002993887115736,
318377.430351612158120,
318387.857738947961479,
318398.285155893652700,
318408.712602448358666,
318419.140078611148056,
318429.567584381264169,
318439.995119757659268,
318450.422684739693068,
318460.850279326317832,
318471.277903516660444,
318481.705557309905998,
318492.133240705181379,
318502.560953701497056,
318512.988696298096329,
318523.416468494106084,
318533.844270288594998,
318544.272101680689957,
318554.699962669517845,
318565.127853254263755,
318575.555773433938157,
318585.983723207784351,
318596.411702574812807,
318606.839711534266826,
318617.267750085215084,
318627.695818226784468,
318638.123915957985446,
318648.552043278119527,
318658.980200186255388,
318669.408386681519914,
318679.836602762923576,
318690.264848429767881,
318700.693123681121506,
318711.121428516111337,
318721.549762933689635,
318731.978126933216117,
318742.406520513759460,
318752.834943674388342,
318763.263396414171439,
318773.691878732468467,
318784.120390628173482,
318794.548932100471575,
318804.977503148489632,
318815.406103771412745,
318825.834733968367800,
318836.263393738365266,
318846.692083080532029,
318857.120801994169597,
318867.549550478288438,
318877.978328532015439,
318888.407136154419277,
318898.835973344743252,
318909.264840102114249,
318919.693736425484531,
318930.122662314155605,
318940.551617767196149,
318950.980602783733048,
318961.409617362893187,
318971.838661503745243,
318982.267735205590725,
318992.696838467381895,
319003.125971288303845,
319013.555133667367045,
319023.984325603931211,
319034.413547096948605,
319044.842798145662528,
319055.272078749083448,
319065.701388906338252,
319076.130728616728447,
319086.560097879148088,
319096.989496692840476,
319107.418925056932494,
319117.848382970609237,
319128.277870432881173,
319138.707387442875188,
319149.136933999834582,
319159.566510102828033,
319169.996115750982426,
319180.425750943308230,
319190.855415679106954,
319201.285109957505483,
319211.714833777514286,
319222.144587138260249,
319232.574370038986672,
319243.004182478704024,
319253.434024456713814,
319263.863895971851889,
319274.293797023477964,
319284.723727610718925,
319295.153687732643448,
319305.583677388262004,
319316.013696576876100,
319326.443745297554415,
319336.873823549482040,
319347.303931331669446,
319357.734068643359933,
319368.164235483563971,
319378.594431851524860,
319389.024657746194862,
319399.454913166933693,
319409.885198112751823,
319420.315512582834344,
319430.745856576191727,
319441.176230092009064,
319451.606633129471447,
319462.037065687705763,
319472.467527765722480,
319482.898019362706691,
319493.328540477901697,
319503.759091110259760,
319514.189671258965973,
319524.620280923321843,
319535.050920102221426,
319545.481588794908021,
319555.912287000392098,
319566.343014718033373,
319576.773771946667694,
319587.204558685712982,
319597.635374934121501,
319608.066220691020135,
319618.497095955652185,
319628.928000727086328,
319639.358935004333034,
319649.789898786693811,
319660.220892073295545,
319670.651914863148704,
319681.082967155438382,
319691.514048949291464,
319701.945160243951250,
319712.376301038311794,
319722.807471331674606,
319733.238671123166569,
319743.669900411856361,
319754.101159196870867,
319764.532447477395181,
319774.963765252439771,
319785.395112521364354,
319795.826489283062983,
319806.257895536778960,
319816.689331281639170,
319827.120796516770497,
319837.552291241299827,
319847.983815454354044,
319858.415369155001827,
319868.846952342428267,
319879.278565015818458,
319889.710207174299285,
319900.141878816823009,
319910.573579942749348,
319921.005310551146977,
319931.437070641026367,
319941.868860211572610,
319952.300679262029007,
319962.732527791406028,
319973.164405798946973,
319983.596313283604104,
319994.028250244678929,
320004.460216681240126,
320014.892212592356373,
320025.324237977270968,
320035.756292835052591,
320046.188377164886333,
320056.620490965782665,
320067.052634236984886,
320077.484806977561675,
320087.917009186698124,
320098.349240863579325,
320108.781502007099334,
320119.213792616617866,
320129.646112691261806,
320140.078462230041623,
320150.510841232084204,
320160.943249696691055,
320171.375687622872647,
320181.808155009755865,
320192.240651856467593,
320202.673178162192926,
320213.105733926058747,
320223.538319147133734,
320233.970933824661188,
320244.403577957593370,
320254.836251545231789,
320265.268954586645123,
320275.701687081018463,
320286.134449027304072,
320296.567240424861666,
320307.000061272759922,
320317.432911570067517,
320327.865791315911338,
320338.298700509534683,
320348.731639149948023,
320359.164607236394659,
320369.597604767826851,
320380.030631743546110,
320390.463688162737526,
320400.896774024353363,
320411.329889327578712,
320421.763034071656875,
320432.196208255598322,
320442.629411878588144,
320453.062644939695019,
320463.495907438162249,
320473.929199373058509,
320484.362520743568894,
320494.795871548762079,
320505.229251787764952,
320515.662661459704395,
320526.096100563823711,
320536.529569099249784,
320546.963067064876668,
320557.396594460122287,
320567.830151283997111,
320578.263737535686232,
320588.697353214200120,
320599.130998318782076,
320609.564672848617192,
320619.998376802715939,
320630.432110180205200,
320640.865872980328277,
320651.299665202212054,
320661.733486844925210,
320672.167337907536421,
320682.601218389288988,
320693.035128289368004,
320703.469067606842145,
320713.903036340780091,
320724.337034490425140,
320734.771062054787762,
320745.205119033169467,
320755.639205424638931,
320766.073321228148416,
320776.507466443057638,
320786.941641068493482,
320797.375845103466418,
320807.810078547219746,
320818.244341398822144,
320828.678633657458704,
320839.112955322256312,
320849.547306392225437,
320859.981686866725795,
320870.416096744709648,
320880.850536025420297,
320891.285004707868211,
320901.719502791354898,
320912.154030274890829,
320922.588587157719303,
320933.023173438792583,
320943.457789117470384,
320953.892434192704968,
320964.327108663797844,
320974.761812529759482,
320985.196545789716765,
320995.631308442912996,
321006.066100488416851,
321016.500921925297007,
321026.935772752854973,
321037.370652970043011,
321047.805562576220836,
321058.240501570282504,
321068.675469951471314,
321079.110467718972359,
321089.545494871912524,
321099.980551409360487,
321110.415637330384925,
321120.850752634345554,
321131.285897320311051,
321141.721071387233678,
321152.156274834473152,
321162.591507660923526,
321173.026769866060931,
321183.462061448721215,
321193.897382408205885,
321204.332732743525412,
321214.768112453981303,
321225.203521538584027,
321235.638959996460471,
321246.074427826795727,
321256.509925028833095,
321266.945451601524837,
321277.381007544056047,
321287.816592855670024,
321298.252207535377238,
321308.687851582420990,
321319.123524995811749,
321329.559227774792816,
321339.994959918491077,
321350.430721425975207,
321360.866512296488509,
321371.302332529041450,
321381.738182122877333,
321392.174061077123042,
321402.609969390847255,
321413.045907063235063,
321423.481874093355145,
321433.917870480567217,
321444.353896223765332,
321454.789951322251000,
321465.226035774918273,
321475.662149581243284,
321486.098292740120087,
321496.534465250791982,
321506.970667112327646,
321517.406898323912174,
321527.843158884730656,
321538.279448793793563,
321548.715768050402403,
321559.152116653509438,
321569.588494602416176,
321580.024901896249503,
321590.461338534019887,
321600.897804514970630,
321611.334299838170409,
321621.770824502862524,
321632.207378508115653,
321642.643961853056680,
321653.080574536870699,
321663.517216558626387,
321673.953887917567044,
321684.390588612761348,
321694.827318643277977,
321705.264078008418437,
321715.700866707251407,
321726.137684738903772,
321736.574532102444209,
321747.011408797232434,
321757.448314822162502,
321767.885250176535919,
321778.322214859363157,
321788.759208869887516,
321799.196232207235880,
321809.633284870476928,
321820.070366858853959,
321830.507478171435650,
321840.944618807407096,
321851.381788765836973,
321861.818988045968581,
321872.256216646754183,
321882.693474567553494,
321893.130761807435192,
321903.568078365526162,
321914.005424240953289,
321924.442799432785250,
321934.880203940381762,
321945.317637762636878,
321955.755100898852106,
321966.192593348037917,
321976.630115109554026,
321987.067666182352696,
321997.505246565560810,
322007.942856258363463,
322018.380495260003954,
322028.818163569492754,
322039.255861186073162,
322049.693588108755648,
322060.131344336783513,
322070.569129869225435,
322081.006944705324713,
322091.444788844150025,
322101.882662284828257,
322112.320565026602708,
322122.758497068542056,
322133.196458409773186,
322143.634449049306568,
322154.072468986618333,
322164.510518220660742,
322174.948596750502475,
322185.386704575328622,
322195.824841694324277,
322206.263008106674533,
322216.701203811448067,
322227.139428807829972,
322237.577683094830718,
322248.015966671810020,
322258.454279537778348,
322268.892621691862587,
322279.330993133189622,
322289.769393861060962,
322300.207823874487076,
322310.646283172594849,
322321.084771754627582,
322331.523289619595744,
322341.961836766742636,
322352.400413195195142,
322362.839018904080149,
322373.277653892466333,
322383.716318159713410,
322394.155011704715434,
322404.593734526773915,
322415.032486624957528,
322425.471267998393159,
322435.910078646324109,
322446.348918567760848,
322456.787787762004882,
322467.226686228008475,
322477.665613965073135,
322488.104570972267538,
322498.543557248834986,
322508.982572793727741,
322519.421617606305517,
322529.860691685520578,
322540.299795030674431,
322550.738927640777547,
322561.178089515015017,
322571.617280652571935,
322582.056501052575186,
322592.495750714209862,
322602.935029636486433,
322613.374337818706408,
322623.813675259880256,
322634.253041959309485,
322644.692437915946357,
322655.131863129092380,
322665.571317597758025,
322676.010801321244799,
322686.450314298563171,
322696.889856528840028,
322707.329428011376876,
322717.769028745184187,
322728.208658729505260,
322738.648317963408772,
322749.088006446021609,
322759.527724176587071,
322769.967471154115628,
322780.407247377908789,
322790.847052846918814,
322801.286887560447212,
322811.726751517620869,
322822.166644717508461,
322832.606567159353290,
322843.046518842107616,
322853.486499765189365,
322863.926509927609004,
322874.366549328493420,
322884.806617966969498,
322895.246715842222329,
322905.686842953437008,
322916.126999299740419,
322926.567184880143031,
322937.007399694004562,
322947.447643740335479,
322957.887917018320877,
322968.328219527087640,
322978.768551265762653,
322989.208912233531009,
322999.649302429519594,
323010.089721852971707,
323020.530170502839610,
323030.970648378424812,
323041.411155478795990,
323051.851691803138237,
323062.292257350636646,
323072.732852120301686,
323083.173476111434866,
323093.614129323046654,
323104.054811754322145,
323114.495523404446431,
323124.936264272604603,
323135.377034357865341,
323145.817833659471944,
323156.258662176434882,
323166.699519907881040,
323177.140406853170134,
323187.581323011254426,
323198.022268381377216,
323208.463242962607183,
323218.904246754187625,
323229.345279755187221,
323239.786341964791063,
323250.227433382184245,
323260.668554006319027,
323271.109703836555127,
323281.550882872019429,
323291.992091111838818,
323302.433328554965556,
323312.874595200875774,
323323.315891048521735,
323333.757216097088531,
323344.198570345703047,
323354.639953793492168,
323365.081366439699195,
323375.522808283392806,
323385.964279323699884,
323396.405779559863731,
323406.847308990953024,
323417.288867616152856,
323427.730455434590112,
323438.172072445449885,
323448.613718647800852,
323459.055394040828105,
323469.497098623774946,
323479.938832395593636,
323490.380595355643891,
323500.822387502936181,
323511.264208836597390,
323521.706059355929028,
323532.147939059999771,
323542.589847947820090,
323553.031786018749699,
323563.473753271857277,
323573.915749706269708,
323584.357775321113877,
323594.799830115574878,
323605.241914088896010,
323615.684027240029536,
323626.126169568276964,
323636.568341072590556,
323647.010541752446443,
323657.452771606738679,
323667.895030634710565,
323678.337318835488986,
323688.779636208142620,
323699.221982752031181,
323709.664358466048725,
323720.106763349554967,
323730.549197401560377,
323740.991660621250048,
323751.434153007809073,
323761.876674560364336,
323772.319225278100930,
323782.761805160087533,
323793.204414205567446,
323803.647052413667552,
323814.089719783456530,
323824.532416314177681,
323834.975142004841473,
323845.417896854807623,
323855.860680863028392,
323866.303494028863497,
323876.746336351265199,
323887.189207829476800,
323897.632108462625183,
323908.075038249895442,
323918.517997190414462,
323928.960985283192713,
323939.404002527648117,
323949.847048922849353,
323960.290124467806891,
323970.733229161764029,
323981.176363003847655,
323991.619525993242860,
324002.062718129134737,
324012.505939410591964,
324022.949189836741425,
324033.392469406884629,
324043.835778119973838,
324054.279115975310560,
324064.722482972079888,
324075.165879109175876,
324085.609304386016447,
324096.052758801728487,
324106.496242355322465,
324116.939755045983475,
324127.383296872954816,
324137.826867835363373,
324148.270467932277825,
324158.714097162941471,
324169.157755526364781,
324179.601443021907471,
324190.045159648638219,
324200.488905405625701,
324210.932680292055011,
324221.376484307111241,
324231.820317449979484,
324242.264179719728418,
324252.708071115543135,
324263.151991636550520,
324273.595941281993873,
324284.039920051000081,
324294.483927942637820,
324304.927964956092183,
324315.372031090490054,
324325.816126345132943,
324336.260250718973111,
324346.704404211312067,
324357.148586821218487,
324367.592798547877464,
324378.037039390415885,
324388.481309348018840,
324398.925608419813216,
324409.369936604867689,
324419.814293902600184,
324430.258680311904754,
324440.703095832024701,
324451.147540462086909,
324461.592014201276470,
324472.036517048720270,
324482.481049003603403,
324492.925610065110959,
324503.370200232311618,
324513.814819504332263,
324524.259467880474404,
324534.704145359690301,
324545.148851941397879,
324555.593587624432985,
324566.038352408213541,
324576.483146291750018,
324586.927969274343923,
324597.372821354889311,
324607.817702532745898,
324618.262612807040568,
324628.707552176900208,
324639.152520641451702,
324649.597518199880142,
324660.042544851312414,
324670.487600594933610,
324680.932685429928824,
324691.377799355424941,
324701.822942370548844,
324712.268114474369213,
324722.713315666187555,
324733.158545945188962,
324743.603805310325697,
324754.049093760957476,
324764.494411296094768,
324774.939757915039081,
324785.385133616800886,
324795.830538400565274,
324806.275972265459131,
324816.721435210783966,
324827.166927235550247,
324837.612448339001276,
324848.057998520263936,
324858.503577778348699,
324868.949186112615280,
324879.394823522190563,
324889.840490006201435,
324900.286185563716572,
324910.731910193921067,
324921.177663896116428,
324931.623446669313125,
324942.069258512638044,
324952.515099425276276,
324962.960969406529330,
324973.406868455407675,
324983.852796571038198,
324994.298753752722405,
325004.744739999412559,
325015.190755310468376,
325025.636799684958532,
325036.082873122009914,
325046.528975620807614,
325056.975107180478517,
325067.421267800207715,
325077.867457479180302,
325088.313676216523163,
325098.759924011363182,
325109.206200862827245,
325119.652506770158652,
325130.098841732542496,
325140.545205748989247,
325150.991598818742204,
325161.438020940928254,
325171.884472114790697,
325182.330952339398209,
325192.777461613935884,
325203.223999937472399,
325213.670567309309263,
325224.117163728573360,
325234.563789194333367,
325245.010443705832586,
325255.457127262139693,
325265.903839862439781,
325276.350581506034359,
325286.797352191875689,
325297.244151919207070,
325307.690980687155388,
325318.137838494963944,
325328.584725341701414,
325339.031641226552892,
325349.478586148645263,
325359.925560107163619,
325370.372563101293053,
325380.819595130160451,
325391.266656192950904,
325401.713746288791299,
325412.160865416750312,
325422.608013576129451,
325433.055190766113810,
325443.502396985713858,
325453.949632234114688,
325464.396896510559600,
325474.844189814117271,
325485.291512144147418,
325495.738863499485888,
325506.186243879434187,
325516.633653283235617,
325527.081091710017063,
325537.528559158847202,
325547.976055628969334,
325558.423581119393930,
325568.871135629480705,
325579.318719158356544,
325589.766331705031916,
325600.213973268808331,
325610.661643848754466,
325621.109343444055412,
325631.557072053896263,
325642.004829677403904,
325652.452616313763428,
325662.900431962159928,
325673.348276621662080,
325683.796150291454978,
325694.244052970781922,
325704.691984658711590,
325715.139945354370866,
325725.587935057061259,
325736.035953765851445,
325746.484001479926519,
325756.932078198296949,
325767.380183920322452,
325777.828318645129912,
325788.276482371788006,
325798.724675099540036,
325809.172896827512886,
325819.621147554833442,
325830.069427280686796,
325840.517736004316248,
325850.966073724732269,
325861.414440441178158,
325871.862836152664386,
325882.311260858608875,
325892.759714558080304,
325903.208197250089142,
325913.656708933995105,
325924.105249608808663,
325934.553819273831323,
325945.002417928131763,
325955.451045570836868,
325965.899702201131731,
325976.348387818259653,
325986.797102421231102,
325997.245846009405795,
326007.694618581735995,
326018.143420137406792,
326028.592250675777905,
326039.041110195859801,
326049.489998696837574,
326059.938916177779902,
326070.387862637930084,
326080.836838076531421,
326091.285842492652591,
326101.734875885420479,
326112.183938254078384,
326122.633029597694986,
326133.082149915513583,
326143.531299206661060,
326153.980477470264304,
326164.429684705624823,
326174.878920911636669,
326185.328186087659560,
326195.777480232878588,
326206.226803346420638,
326216.676155427412596,
326227.125536474864930,
326237.574946488253772,
326248.024385466473177,
326258.473853408882860,
326268.923350314551499,
326279.372876182547770,
326289.822431012231391,
326300.272014802612830,
326310.721627552935388,
326321.171269262267742,
326331.620939929794986,
326342.070639554818626,
326352.520368136349134,
326362.970125673571602,
326373.419912165729329,
326383.869727611832786,
326394.319572011125274,
326404.769445362850092,
326415.219347666075919,
326425.669278920046054,
326436.119239123712759,
326446.569228276493959,
326457.019246377458330,
326467.469293425674550,
326477.919369420444127,
326488.369474360835738,
326498.819608246034477,
326509.269771075225435,
326519.719962847593706,
326530.170183562207967,
326540.620433218311518,
326551.070711815031245,
326561.521019351552241,
326571.971355827059597,
326582.421721240680199,
326592.872115591540933,
326603.322538878826890,
326613.772991101723164,
326624.223472259473056,
326634.673982351087034,
326645.124521375750192,
326655.575089332705829,
326666.025686221080832,
326676.476312040060293,
326686.926966788771097,
326697.377650466340128,
326707.828363072010688,
326718.279104604967870,
326728.729875064280350,
326739.180674449191429,
326749.631502758769784,
326760.082359992258716,
326770.533246148785111,
326780.984161227534059,
326791.435105227690656,
326801.886078148265369,
326812.337079988676123,
326822.788110747875180,
326833.239170425164048,
326843.690259019669611,
326854.141376530402340,
326864.592522956838366,
326875.043698297871742,
326885.494902552745771,
326895.946135720645543,
326906.397397800697945,
326916.848688792088069,
326927.300008694117423,
326937.751357505680062,
326948.202735226135701,
326958.654141854611225,
326969.105577390117105,
326979.557041832129471,
326990.008535179484170,
327000.460057431657333,
327010.911608587543014,
327021.363188646384515,
327031.814797607425135,
327042.266435469849966,
327052.718102232669480,
327063.169797895185184,
327073.621522456407547,
327084.073275915696286,
327094.525058272178285,
327104.976869524864014,
327115.428709673113190,
327125.880578715878073,
327136.332476652460173,
327146.784403482102789,
327157.236359203874599,
327167.688343816902488,
327178.140357320313342,
327188.592399713408668,
327199.044470995373558,
327209.496571165218484,
327219.948700222186744,
327230.400858165463433,
327240.853044994233642,
327251.305260707566049,
327261.757505304703955,
327272.209778784774244,
327282.662081146962009,
327293.114412390452344,
327303.566772514372133,
327314.019161517964676,
327324.471579400356859,
327334.924026160559151,
327345.376501797989476,
327355.829006311658304,
327366.281539700808935,
327376.734101964510046,
327387.186693102063145,
327397.639313112478703,
327408.091961995058227,
327418.544639748928603,
327428.997346373216715,
327439.450081867107656,
327449.902846229728311,
327460.355639460380189,
327470.808461558131967,
327481.261312522168737,
327491.714192351559177,
327502.167101045604795,
327512.620038603432477,
327523.073005024285521,
327533.526000307116192,
327543.979024451342411,
327554.432077455916442,
327564.885159320197999,
327575.338270043197554,
327585.791409624158405,
327596.244578062265646,
327606.697775356529746,
327617.151001506310422,
327627.604256510792766,
327638.057540368987247,
327648.510853080137167,
327658.964194643427618,
327669.417565057927277,
327679.870964322995860,
327690.324392437585630,
327700.777849401056301,
327711.231335212418344,
327721.684849870856851,
327732.138393375673331,
327742.591965725936461,
327753.045566920831334,
327763.499196959543042,
327773.952855841140263,
327784.406543564924505,
327794.860260130022652,
327805.314005535561591,
327815.767779780726414,
327826.221582864702214,
327836.675414786674082,
327847.129275545827113,
327857.583165141229983,
327868.037083572067786,
327878.491030837642029,
327888.945006937021390,
327899.399011869332753,
327909.853045633819420,
327920.307108229724690,
327930.761199656000827,
327941.215319911949337,
327951.669468996697105,
327962.123646909545641,
327972.577853649505414,
327983.032089215819724,
327993.486353607615456,
328003.940646824077703,
328014.394968864391558,
328024.849319727742113,
328035.303699413256254,
328045.758107920119073,
328056.212545247515664,
328066.667011394572910,
328077.121506360534113,
328087.576030144526158,
328098.030582745675929,
328108.485164163168520,
328118.939774396247230,
328129.394413444038946,
328139.849081305670552,
328150.303777980385348,
328160.758503467252012,
328171.213257765572052,
328181.668040874414146,
328192.122852793021593,
328202.577693520463072,
328213.032563056040090,
328223.487461398763116,
328233.942388548050076,
328244.397344502736814,
328254.852329262299463,
328265.307342825690284,
328275.762385192210786,
328286.217456360987853,
328296.672556331206579,
328307.127685101993848,
328317.582842672592960,
328328.038029042072594,
328338.493244209676050,
328348.948488174588419,
328359.403760935994796,
328369.859062492963858,
328380.314392844680697,
328390.769751990388613,
328401.225139929330908,
328411.680556660518050,
328422.136002183193341,
328432.591476496483665,
328443.046979599574115,
328453.502511491707992,
328463.958072172012180,
328474.413661639613565,
328484.869279893755447,
328495.324926933506504,
328505.780602758168243,
328516.236307366809342,
328526.692040758673102,
328537.147802932944614,
328547.603593888634350,
328558.059413625043817,
328568.515262141416315,
328578.971139436820522,
328589.427045510383323,
328599.882980361464433,
328610.338943988899700,
328620.794936392223462,
328631.250957570446189,
328641.707007522752974,
328652.163086248328909,
328662.619193746184465,
328673.075330015795771,
328683.531495056115091,
328693.987688866385724,
328704.443911445734557,
328714.900162793463096,
328725.356442908581812,
328735.812751790334005,
328746.269089437846560,
328756.725455850420985,
328767.181851027067751,
328777.638274967088364,
328788.094727669551503,
328798.551209133700468,
328809.007719358778559,
328819.464258343738038,
328829.920826087880414,
328840.377422590507194,
328850.834047850628849,
328861.290701867430471,
328871.747384640038945,
328882.204096167813987,
328892.660836449649651,
328903.117605485022068,
328913.574403272999916,
328924.031229812593665,
328934.488085103221238,
328944.944969143834896,
328955.401881933736149,
328965.858823472110089,
328976.315793758141808,
328986.772792790841777,
328997.229820569569711,
329007.686877093394287,
329018.143962361558806,
329028.601076373131946,
329039.058219127415214,
329049.515390623593703,
329059.972590860619675,
329070.429819837911054,
329080.887077554536518,
329091.344364009681158,
329101.801679202588275,
329112.259023132326547,
329122.716395797964651,
329133.173797198978718,
329143.631227334437426,
329154.088686203351244,
329164.546173805021681,
329175.003690138633829,
329185.461235203314573,
329195.918808998249006,
329206.376411522680428,
329216.834042775677517,
329227.291702756541781,
329237.749391464283690,
329248.207108898146544,
329258.664855057373643,
329269.122629941091873,
329279.580433548544534,
329290.038265878683887,
329300.496126930927858,
329310.954016704345122,
329321.411935198120773,
329331.869882411439903,
329342.327858343487605,
329352.785862993448973,
329363.243896360450890,
329373.701958443678450,
329384.160049242316745,
329394.618168755550869,
329405.076316982507706,
329415.534493922488764,
329425.992699574562721,
329436.450933937914670,
329446.909197011729702,
329457.367488795251120,
329467.825809287489392,
329478.284158487804234,
329488.742536395322531,
329499.200943009054754,
329509.659378328477032,
329520.117842352425214,
329530.576335080317222,
329541.034856511338148,
329551.493406644556671,
329561.951985479157884,
329572.410593014385086,
329582.869229249248747,
329593.327894183108583,
329603.786587815091480,
329614.245310144382529,
329624.704061170108616,
329635.162840891513042,
329645.621649307606276,
329656.080486417806242,
329666.539352221239824,
329676.998246716917492,
329687.457169904140756,
329697.916121782036498,
329708.375102349789813,
329718.834111606702209,
329729.293149551784154,
329739.752216184278950,
329750.211311503371689,
329760.670435508131050,
329771.129588197916746,
329781.588769571855664,
329792.047979629016481,
329802.507218368700705,
329812.966485789977014,
329823.425781892088708,
329833.885106674220879,
329844.344460135500412,
329854.803842275228817,
329865.263253092474770,
329875.722692586306948,
329886.182160756143276,
329896.641657600994222,
329907.101183120161295,
329917.560737312713172,
329928.020320177834947,
329938.479931714711711,
329948.939571922644973,
329959.399240800703410,
329969.858938348013908,
329980.318664563877974,
329990.778419447306078,
330001.238202997657936,
330011.698015214060433,
330022.157856095640454,
330032.617725641641300,
330043.077623851131648,
330053.537550723354798,
330063.997506257612258,
330074.457490452856291,
330084.917503308388405,
330095.377544823451899,
330105.837614997057244,
330116.297713828447741,
330126.757841316983104,
330137.217997461615596,
330147.678182261530310,
330158.138395716028754,
330168.598637824121397,
330179.058908585226163,
330189.519207998353522,
330199.979536062804982,
330210.439892777591012,
330220.900278142013121,
330231.360692155139986,
330241.821134816331323,
330252.281606124655809,
330262.742106079240330,
330273.202634679328185,
330283.663191924104467,
330294.123777812696062,
330304.584392344346270,
330315.045035518298391,
330325.505707333562896,
330335.966407789441291,
330346.427136885060463,
330356.887894619547296,
330367.348680992203299,
330377.809496002213564,
330388.270339648646768,
330398.731211930746213,
330409.192112847580574,
330419.653042398509569,
330430.114000582660083,
330440.574987399158999,
330451.036002847191412,
330461.497046926058829,
330471.958119634713512,
330482.419220972515177,
330492.880350938590709,
330503.341509532125201,
330513.802696752303746,
330524.263912598311435,
330534.725157069275156,
330545.186430164379999,
330555.647731882927474,
330566.109062224044465,
330576.570421186799649,
330587.031808770552743,
330597.493224974197801,
330607.954669797269162,
330618.416143238777295,
330628.877645297907293,
330639.339175973844249,
330649.800735265656840,
330660.262323172762990,
330670.723939694173168,
330681.185584829130676,
330691.647258576762397,
330702.108960936369840,
330712.570691906963475,
330723.032451487844810,
330733.494239678198937,
330743.956056477094535,
330754.417901883891318,
330764.879775897657964,
330775.341678517521359,
330785.803609742724802,
330796.265569572511595,
330806.727558006008621,
330817.189575042400975,
330827.651620680815540,
330838.113694920495618,
330848.575797760684509,
330859.037929200392682,
330869.500089238979854,
330879.962277875572909,
330890.424495109240524,
330900.886740939284209,
330911.349015364889055,
330921.811318385181949,
330932.273649999406189,
330942.736010206688661,
330953.198399006214458,
330963.660816397168674,
330974.123262378736399,
330984.585736950160936,
330995.048240110569168,
331005.510771859146189,
331015.973332195018884,
331026.435921117488761,
331036.898538625682704,
331047.361184718844015,
331057.823859395983163,
331068.286562656459864,
331078.749294499284588,
331089.212054923875257,
331099.674843929242343,
331110.137661514570937,
331120.600507679162547,
331131.063382422085851,
331141.526285742525943,
331151.989217639726121,
331162.452178112871479,
331172.915167161088903,
331183.378184783621691,
331193.841230979596730,
331204.304305748140905,
331214.767409088672139,
331225.230541000142694,
331235.693701481912285,
331246.156890532991383,
331256.620108152681496,
331267.083354339993093,
331277.546629094402306,
331288.009932414861396,
331298.473264300671872,
331308.936624750960618,
331319.400013764854521,
331329.863431341596879,
331340.326877480489202,
331350.790352180600166,
331361.253855441056658,
331371.717387261218391,
331382.180947640037630,
331392.644536576874088,
331403.108154070854653,
331413.571800121222623,
331424.035474727046676,
331434.499177887628321,
331444.962909602036234,
331455.426669869513717,
331465.890458689245861,
331476.354276060475968,
331486.818121982330922,
331497.281996453995816,
331507.745899474597536,
331518.209831043437589,
331528.673791159584653,
331539.137779822340235,
331549.601797030831221,
331560.065842784242705,
331570.529917081759777,
331580.994019922509324,
331591.458151305792853,
331601.922311230737250,
331612.386499696585815,
331622.850716702465434,
331633.314962247502990,
331643.779236330883577,
331654.243538951908704,
331664.707870109821670,
331675.172229803574737,
331685.636618032527622,
331696.101034795807209,
331706.565480092540383,
331717.029953922028653,
331727.494456283457112,
331737.958987175894435,
331748.423546598642133,
331758.888134550885297,
331769.352751031634398,
331779.817396040249150,
331790.282069575856440,
331800.746771637757774,
331811.211502224963624,
331821.676261336775497,
331832.141048972203862,
331842.605865130724851,
331853.070709811348934,
331863.535583013261203,
331874.000484735704958,
331884.465414977748878,
331894.930373738752678,
331905.395361017785035,
331915.860376814089250,
331926.325421126734000,
331936.790493955079000,
331947.255595298251137,
331957.720725155319087,
331968.185883525642566,
331978.651070408348460,
331989.116285802563652,
331999.581529707531445,
332010.046802122436929,
332020.512103046406992,
332030.977432478743140,
332041.442790418572258,
332051.908176865079440,
332062.373591817449778,
332072.839035274810158,
332083.304507236520294,
332093.770007701474242,
332104.235536669206340,
332114.701094138668850,
332125.166680109221488,
332135.632294579874724,
332146.097937549871858,
332156.563609018397983,
332167.029308984754607,
332177.495037448010407,
332187.960794407408684,
332198.426579862076323,
332208.892393811256625,
332219.358236254076473,
332229.824107189779170,
332240.290006617549807,
332250.755934536573477,
332261.221890946035273,
332271.687875845120288,
332282.153889233013615,
332292.619931108842138,
332303.086001471907366,
332313.552100321394391,
332324.018227656430099,
332334.484383476199582,
332344.950567779887933,
332355.416780566680245,
332365.883021835936233,
332376.349291586549953,
332386.815589817997534,
332397.281916529173031,
332407.748271719494369,
332418.214655388088431,
332428.681067534198519,
332439.147508156893309,
332449.613977255357895,
332460.080474828893784,
332470.547000876686070,
332481.013555397803430,
332491.480138391489163,
332501.946749857044779,
332512.413389793538954,
332522.880058200156782,
332533.346755076199770,
332543.813480420620181,
332554.280234232894145,
332564.747016512032133,
332575.213827257335652,
332585.680666467931587,
332596.147534142946824,
332606.614430281624664,
332617.081354883150198,
332627.548307946824934,
332638.015289471717551,
332648.482299456954934,
332658.949337901955005,
332669.416404805611819,
332679.883500167401507,
332690.350623986276332,
332700.817776261654217,
332711.284956992545631,
332721.752166178193875,
332732.219403817842249,
332742.686669910501223,
332753.153964455646928,
332763.621287452231627,
332774.088638899615034,
332784.556018796865828,
332795.023427143227309,
332805.490863937768154,
332815.958329179906286,
332826.425822868652176,
332836.893345003307331,
332847.360895582998637,
332857.828474606969394,
332868.296082074230071,
332878.763717984256800,
332889.231382336060051,
332899.699075128883123,
332910.166796361911111,
332920.634546034329105,
332931.102324145263992,
332941.570130693959072,
332952.037965679657646,
332962.505829101544805,
332972.973720958747435,
332983.441641250508837,
332993.909589976014104,
333004.377567134273704,
333014.845572724821977,
333025.313606746611185,
333035.781669198942836,
333046.249760080885608,
333056.717879391799215,
333067.186027130694129,
333077.654203296930064,
333088.122407889575697,
333098.590640907874331,
333109.058902350952849,
333119.527192218112759,
333129.995510508539155,
333140.463857221242506,
333150.932232355698943,
333161.400635910918936,
333171.869067886087578,
333182.337528280506376,
333192.806017093185801,
333203.274534323485568,
333213.743079970590770,
333224.211654033628292,
333234.680256511783227,
333245.148887404357083,
333255.617546710418537,
333266.086234429152682,
333276.554950559802819,
333287.023695101554040,
333297.492468053649645,
333307.961269415274728,
333318.430099185614381,
333328.898957363737281,
333339.367843948944937,
333349.836758940422442,
333360.305702337354887,
333370.774674138985574,
333381.243674344441388,
333391.712702952965628,
333402.181759963626973,
333412.650845375785138,
333423.119959188567009,
333433.589101401099470,
333444.058272012742236,
333454.527471022563986,
333464.996698429749813,
333475.465954233426601,
333485.935238433012273,
333496.404551027517300,
333506.873892016184982,
333517.343261398316827,
333527.812659172923304,
333538.282085339247715,
333548.751539896533359,
333559.221022843965329,
333569.690534180786926,
333580.160073906066827,
333590.629642019048333,
333601.099238519032951,
333611.568863405031152,
333622.038516676286235,
333632.508198332157917,
333642.977908371714875,
333653.447646794083994,
333663.917413598624989,
333674.387208784348331,
333684.857032350555528,
333695.326884296489879,
333705.796764621220063,
333716.266673324047588,
333726.736610404099338,
333737.206575860560406,
333747.676569692674093,
333758.146591899683699,
333768.616642480599694,
333779.086721434840001,
333789.556828761473298,
333800.026964459684677,
333810.497128528775647,
333820.967320967756677,
333831.437541775987484,
333841.907790952653158,
333852.378068496880587,
333862.848374407854863,
333873.318708684877492,
333883.789071326958947,
333894.259462333517149,
333904.729881703620777,
333915.200329436454922,
333925.670805531262886,
333936.141309987229761,
333946.611842803540640,
333957.082403979380615,
333967.552993513876572,
333978.023611406446435,
333988.494257656100672,
333998.964932262082584,
334009.435635223577265,
334019.906366539711598,
334030.377126209845301,
334040.847914233105257,
334051.318730608676560,
334061.789575335686095,
334072.260448413435370,
334082.731349841051269,
334093.202279617718887,
334103.673237742739730,
334114.144224215240683,
334124.615239034465048,
334135.086282199539710,
334145.557353709649760,
334156.028453564038500,
334166.499581761832815,
334176.970738302392419,
334187.441923184727784,
334197.913136408198625,
334208.384377971931826,
334218.855647875054274,
334229.326946116751060,
334239.798272696440108,
334250.269627613073681,
334260.741010866011493,
334271.212422454380430,
334281.683862377307378,
334292.155330634152051,
334302.626827223924920,
334313.098352145985700,
334323.569905399519484,
334334.041486983594950,
334344.513096897513606,
334354.984735140402336,
334365.456401711562648,
334375.928096610063221,
334386.399819835205562,
334396.871571386174764,
334407.343351262097713,
334417.815159462275915,
334428.286995985836256,
334438.758860831963830,
334449.230753999843728,
334459.702675488835666,
334470.174625297950115,
334480.646603426430374,
334491.118609873519745,
334501.590644638403319,
334512.062707720149774,
334522.534799118177034,
334533.006918831611983,
334543.479066859639715,
334553.951243201328907,
334564.423447856097482,
334574.895680823014118,
334585.367942101205699,
334595.840231690031942,
334606.312549588619731,
334616.784895796154160,
334627.257270311936736,
334637.729673134977929,
334648.202104264637455,
334658.674563699983992,
334669.147051440377254,
334679.619567484885920,
334690.092111832753289,
334700.564684483106248,
334711.037285435362719,
334721.509914688533172,
334731.982572241686285,
334742.455258094298188,
334752.927972245553974,
334763.400714694464114,
334773.873485440330114,
334784.346284482337069,
334794.819111819670070,
334805.291967451630626,
334815.764851377287414,
334826.237763595883735,
334836.710704106604680,
334847.183672908751760,
334857.656670001393650,
334868.129695383715443,
334878.602749055018649,
334889.075831014546566,
334899.548941261367872,
334910.022079794725869,
334920.495246613805648,
334930.968441717908718,
334941.441665105987340,
334951.914916777575854,
334962.388196731626522,
334972.861504967440851,
334983.334841484145727,
334993.808206281100865,
335004.281599357374944,
335014.755020712036639,
335025.228470344620291,
335035.701948254078161,
335046.175454439711757,
335056.648988900647964,
335067.122551636188291,
335077.596142645459622,
335088.069761927588843,
335098.543409481935669,
335109.017085307626985,
335119.490789403906092,
335129.964521769899875,
335140.438282404909842,
335150.912071307946462,
335161.385888478369452,
335171.859733915422112,
335182.333607618114911,
335192.807509585865773,
335203.281439817743376,
335213.755398312991019,
335224.229385070735589,
335234.703400090278592,
335245.177443370805122,
335255.651514911500271,
335266.125614711549133,
335276.599742770136800,
335287.073899086506572,
335297.548083659843542,
335308.022296489332803,
335318.496537574217655,
335328.970806913683191,
335339.445104506914504,
335349.919430353154894,
335360.393784451531246,
335370.868166801345069,
335381.342577401606832,
335391.817016251734458,
335402.291483350913040,
335412.765978698211256,
335423.240502292872407,
335433.715054134197999,
335444.189634221314918,
335454.664242553350050,
335465.138879129663110,
335475.613543949322775,
335486.088237011630554,
335496.562958315713331,
335507.037707860872615,
335517.512485646177083,
335527.987291670870036,
335538.462125934252981,
335548.936988435394596,
335559.411879173596390,
335569.886798148043454,
335580.361745357862674,
335590.836720802355558,
335601.311724480590783,
335611.786756391928066,
335622.261816535494290,
335632.736904910532758,
335643.212021516170353,
335653.687166351708584,
335664.162339416274335,
335674.637540709052701,
335685.112770229345188,
335695.588027976278681,
335706.063313949096482,
335716.538628146925475,
335727.013970569067169,
335737.489341214648448,
335747.964740082912613,
335758.440167173102964,
335768.915622484346386,
335779.391106015886180,
335789.866617766849231,
335800.342157736595254,
335810.817725924251135,
335821.293322328885552,
335831.768946949974634,
335842.244599786528852,
335852.720280837791506,
335863.195990102889482,
335873.671727581182495,
335884.147493271855637,
335894.623287173977587,
335905.099109286849853,
335915.574959609657526,
335926.050838141643908,
335936.526744881935883,
335947.002679829776753,
335957.478642984409817,
335967.954634345020168,
335978.430653910676483,
335988.906701680796687,
335999.382777654507663,
336009.858881830936298,
336020.335014209384099,
336030.811174789036158,
336041.287363569077570,
336051.763580548635218,
336062.239825727127027,
336072.716099103505258,
336083.192400677187834,
336093.668730447185226,
336104.145088412915356,
336114.621474573388696,
336125.097888927906752,
336135.574331475654617,
336146.050802215875592,
336156.527301147754770,
336167.003828270477243,
336177.480383583228104,
336187.956967085250653,
336198.433578775729984,
336208.910218653909396,
336219.386886718973983,
336229.863582970050629,
336240.340307406499051,
336250.817060027446132,
336261.293840832076967,
336271.770649819634855,
336282.247486989246681,
336292.724352340155747,
336303.201245871663559,
336313.678167582897004,
336324.155117473099381,
336334.632095541339368,
336345.109101786976680,
336355.586136209138203,
336366.063198807125445,
336376.540289580007084,
336387.017408527142834,
336397.494555647659581,
336407.971730940684211,
336418.448934405518230,
336428.926166041346733,
336439.403425847413018,
336449.880713822843973,
336460.358029966999311,
336470.835374278889503,
336481.312746757816058,
336491.790147403022274,
336502.267576213576831,
336512.745033188839443,
336523.222518327995203,
336533.700031630170997,
336544.177573094668332,
336554.655142720555887,
336565.132740507193375,
336575.610366453707684,
336586.088020559283905,
336596.565702823223546,
336607.043413244653493,
336617.521151822817046,
336627.998918556899298,
336638.476713446085341,
336648.954536489734892,
336659.432387686742004,
336669.910267036582809,
336680.388174538442399,
336690.866110191447660,
336701.344073994841892,
336711.822065947810188,
336722.300086049595848,
336732.778134299325757,
336743.256210696301423,
336753.734315239707939,
336764.212447928672191,
336774.690608762553893,
336785.168797740479931,
336795.647014861635398,
336806.125260125205386,
336816.603533530374989,
336827.081835076503921,
336837.560164762777276,
336848.038522588321939,
336858.516908552323002,
336868.995322654023767,
336879.473764892609324,
336889.952235267439391,
336900.430733777408022,
336910.909260422049556,
336921.387815200374462,
336931.866398111800663,
336942.345009155222215,
336952.823648330115248,
336963.302315635548439,
336973.781011070706882,
336984.259734634950291,
336994.738486327463761,
337005.217266147257760,
337015.696074093750212,
337026.174910166068003,
337036.653774363454431,
337047.132666685094591,
337057.611587130115367,
337068.090535697876476,
337078.569512387504801,
337089.048517198243644,
337099.527550129278097,
337110.006611179793254,
337120.485700349090621,
337130.964817636180669,
337141.443963040481322,
337151.923136561119463,
337162.402338197338395,
337172.881567948323209,
337183.360825813258998,
337193.840111791389063,
337204.319425881956704,
337214.798768083972391,
337225.278138396912254,
337235.757536819903180,
337246.236963352130260,
337256.716417992720380,
337267.195900741033256,
337277.675411596137565,
337288.154950557451230,
337298.634517623868305,
337309.114112794864923,
337319.593736069567967,
337330.073387447220739,
337340.553066926950123,
337351.032774508057628,
337361.512510189670138,
337371.992273970914539,
337382.472065851266962,
337392.951885829796083,
337403.431733905686997,
337413.911610078183003,
337424.391514346469194,
337434.871446709788870,
337445.351407167327125,
337455.831395718269050,
337466.311412361916155,
337476.791457097395323,
337487.271529924008064,
337497.751630840823054,
337508.231759847200010,
337518.711916942207608,
337529.192102125147358,
337539.672315395204350,
337550.152556751621887,
337560.632826193585061,
337571.113123720337171,
337581.593449331005104,
337592.073803024890367,
337602.554184801178053,
337613.034594659053255,
337623.515032597759273,
337633.995498616422992,
337644.475992714462336,
337654.956514890829567,
337665.437065144942608,
337675.917643475870136,
337686.398249882855453,
337696.878884365200065,
337707.359546922030859,
337717.840237552649342,
337728.320956256124191,
337738.801703031756915,
337749.282477878790814,
337759.763280796352774,
337770.244111783744302,
337780.724970840034075,
337791.205857964581810,
337801.686773156514391,
337812.167716415191535,
337822.648687739565503,
337833.129687129112426,
337843.610714582900982,
337854.091770100116264,
337864.572853680059779,
337875.053965321916621,
337885.535105024813674,
337896.016272788168862,
337906.497468610992655,
337916.978692492586561,
337927.459944432135671,
337937.941224428883288,
337948.422532481956296,
337958.903868590714410,
337969.385232754226308,
337979.866624971851707,
337990.348045242717490,
338000.829493565950543,
338011.310969940910581,
338021.792474366782699,
338032.274006842635572,
338042.755567367945332,
338053.237155941664241,
338063.718772563268431,
338074.200417231710162,
338084.682089946349151,
338095.163790706312284,
338105.645519510959275,
338116.127276359300595,
338126.609061250754166,
338137.090874184330460,
338147.572715159447398,
338158.054584175231867,
338168.536481230868958,
338179.018406325601973,
338189.500359458616003,
338199.982340629037935,
338210.464349836343899,
338220.946387079544365,
338231.428452357882634,
338241.910545670660213,
338252.392667016945779,
338262.874816396040842,
338273.356993807188701,
338283.839199249458034,
338294.321432722266763,
338304.803694224741776,
338315.285983756068163,
338325.768301315431017,
338336.250646902131848,
338346.733020515355747,
338357.215422154287808,
338367.697851818171330,
338378.180309506191406,
338388.662795217533130,
338399.145308951556217,
338409.627850707387552,
338420.110420484212227,
338430.593018281215336,
338441.075644097698387,
338451.558297932788264,
338462.040979785844684,
338472.523689655994531,
338483.006427542422898,
338493.489193444373086,
338503.971987361030187,
338514.454809291637503,
338524.937659235438332,
338535.420537191559561,
338545.903443159360904,
338556.386377137911040,
338566.869339126511477,
338577.352329124405514,
338587.835347130720038,
338598.318393144698348,
338608.801467165641952,
338619.284569192503113,
338629.767699224816170,
338640.250857261649799,
338650.734043302247301,
338661.217257345851976,
338671.700499391590711,
338682.183769438765012,
338692.667067486443557,
338703.150393534044269,
338713.633747580635827,
338724.117129625577945,
338734.600539667997509,
338745.083977707021404,
338755.567443742067553,
338766.050937772204634,
338776.534459796617739,
338787.018009814608376,
338797.501587825419847,
338807.985193828237243,
338818.468827822245657,
338828.952489806688391,
338839.436179780808743,
338849.919897743733600,
338860.403643694706261,
338870.887417633028235,
338881.371219557826407,
338891.855049468402285,
338902.338907363882754,
338912.822793243569322,
338923.306707106588874,
338933.790648952242918,
338944.274618779716548,
338954.758616588136647,
338965.242642376804724,
338975.726696145022288,
338986.210777891858015,
338996.694887616671622,
339007.179025318473577,
339017.663190996740013,
339028.147384650423191,
339038.631606278999243,
339049.115855881478637,
339059.600133457104675,
339070.084439005295280,
339080.568772525002714,
339091.053134015644901,
339101.537523476348724,
339112.021940906299278,
339122.506386304798070,
339132.990859670971986,
339143.475361004122533,
339153.959890303434804,
339164.444447568152100,
339174.929032797401305,
339185.413645990600344,
339195.898287146701477,
339206.382956265122630,
339216.867653344932478,
339227.352378385432530,
339237.837131385866087,
339248.321912345476449,
339258.806721263390500,
339269.291558138851542,
339279.776422971161082,
339290.261315759387799,
339300.746236502891406,
339311.231185200740583,
339321.716161852295045,
339332.201166456681676,
339342.686199013260193,
339353.171259521041065,
339363.656347979384009,
339374.141464387532324,
339384.626608744612895,
339395.111781049869023,
339405.596981302427594,
339416.082209501764737,
339426.567465646832716,
339437.052749737049453,
339447.538061771483626,
339458.023401749494951,
339468.508769670093898,
339478.994165532698389,
339489.479589336435311,
339499.965041080606170,
339510.450520764337853,
339520.936028386873659,
339531.421563947398681,
339541.907127445330843,
339552.392718879622407,
339562.878338249574881,
339573.363985554547980,
339583.849660793493968,
339594.335363965888973,
339604.821095070859883,
339615.306854107591789,
339625.792641075386200,
339636.278455973370001,
339646.764298800844699,
339657.250169556937180,
339667.736068240890745,
339678.221994852006901,
339688.707949389470741,
339699.193931852467358,
339709.679942240240052,
339720.165980552032124,
339730.652046786970459,
339741.138140944356564,
339751.624263023433741,
339762.110413023270667,
339772.596590943285264,
339783.082796782662626,
339793.569030540529639,
339804.055292216129601,
339814.541581808764022,
339825.027899317501578,
339835.514244741760194,
339846.000618080550339,
339856.487019333231729,
339866.973448498989455,
339877.459905577066820,
339887.946390566707123,
339898.432903467037249,
339908.919444277358707,
339919.406012996856589,
339929.892609624715988,
339940.379234160180204,
339950.865886602608953,
339961.352566950954497,
339971.839275204692967,
339982.326011362951249,
339992.812775424914435,
340003.299567389825825,
340013.786387256928720,
340024.273235025408212,
340034.760110694507603,
340045.247014263411984,
340055.733945731422864,
340066.220905097725336,
340076.707892361504491,
340087.194907522003632,
340097.681950578524265,
340108.169021530193277,
340118.656120376137551,
340129.143247115833219,
340139.630401748232543,
340150.117584272811655,
340160.604794688639231,
340171.092032994958572,
340181.579299191012979,
340192.066593275987543,
340202.553915249183774,
340213.041265109670348,
340223.528642856865190,
340234.016048489836976,
340244.503482007887214,
340254.990943410200998,
340265.478432696079835,
340275.965949864650611,
340286.453494915156625,
340296.941067846782971,
340307.428668658831157,
340317.916297350544482,
340328.403953921049833,
340338.891638369648717,
340349.379350695584435,
340359.867090897983871,
340370.354858976090327,
340380.842654929088894,
340391.330478756339289,
340401.818330456968397,
340412.306210030219518,
340422.794117475335952,
340433.282052791502792,
340443.770015978021547,
340454.258007033960894,
340464.746025958680548,
340475.234072751365602,
340485.722147411259357,
340496.210249937488697,
340506.698380329355132,
340517.186538586160168,
340527.674724707030691,
340538.162938691151794,
340548.651180537824985,
340559.139450246235356,
340569.627747815684415,
340580.116073245182633,
340590.604426534147933,
340601.092807681881823,
340611.581216687394772,
340622.069653549988288,
340632.558118268905673,
340643.046610843448434,
340653.535131272685248,
340664.023679555917624,
340674.512255692330655,
340685.000859681167640,
340695.489491521730088,
340705.978151213203091,
340716.466838754713535,
340726.955554145621136,
340737.444297385052778,
340747.933068472251762,
340758.421867406519596,
340768.910694186983164,
340779.399548812885769,
340789.888431283412501,
340800.377341597981285,
340810.866279755602591,
340821.355245755636133,
340831.844239597208798,
340842.333261279622093,
340852.822310802002903,
340863.311388163710944,
340873.800493363756686,
340884.289626401674468,
340894.778787276474759,
340905.267975987400860,
340915.757192533754278,
340926.246436914720107,
340936.735709129483439,
340947.225009177345783,
340957.714337057375815,
340968.203692768991459,
340978.693076311319601,
340989.182487683661748,
340999.671926885202993,
341010.161393915012013,
341020.650888772623148,
341031.140411457046866,
341041.629961967526469,
341052.119540303305257,
341062.609146463568322,
341073.098780447733589,
341083.588442254811525,
341094.078131884103641,
341104.567849334853236,
341115.057594606303610,
341125.547367697639856,
341136.037168608047068,
341146.526997336826753,
341157.016853883164003,
341167.506738246243913,
341177.996650425484404,
341188.486590419954155,
341198.976558228838257,
341209.466553851438221,
341219.956577286939137,
341230.446628534700722,
341240.936707593733445,
341251.426814463455230,
341261.916949142876547,
341272.407111631473526,
341282.897301928431261,
341293.387520032760222,
341303.877765943936538,
341314.368039661028888,
341324.858341183338780,
341335.348670510051306,
341345.839027640467975,
341356.329412573657464,
341366.819825309037697,
341377.310265845677350,
341387.800734182936139,
341398.291230319999158,
341408.781754256051499,
341419.272305990336463,
341429.762885522155557,
341440.253492850577459,
341450.744127974961884,
341461.234790894435719,
341471.725481608358677,
341482.216200115857646,
341492.706946416175924,
341503.197720508615021,
341513.688522392243613,
341524.179352066537831,
341534.670209530449938,
341545.161094783397857,
341555.652007824450266,
341566.142948652966879,
341576.633917268132791,
341587.124913669249509,
341597.615937855443917,
341608.106989825959317,
341618.598069580097217,
341629.089177116926294,
341639.580312435864471,
341650.071475535980426,
341660.562666416575667,
341671.053885076951701,
341681.545131516235415,
341692.036405733611900,
341702.527707728499081,
341713.019037499907427,
341723.510395047254860,
341734.001780369610060,
341744.493193466332741,
341754.984634336433373,
341765.476102979504503,
341775.967599394440185,
341786.459123580658343,
341796.950675537285861,
341807.442255263624247,
341817.933862758858595,
341828.425498022232205,
341838.917161052930169,
341849.408851850253996,
341859.900570413388778,
341870.392316741636023,
341880.884090834064409,
341891.375892690033652,
341901.867722308845259,
341912.359579689509701,
341922.851464831386693,
341933.343377733777743,
341943.835318395751528,
341954.327286816609558,
341964.819282995595131,
341975.311306931893341,
341985.803358624805696,
341996.295438073517289,
342006.787545277271420,
342017.279680235311389,
342027.771842946822289,
342038.264033410989214,
342048.756251627230085,
342059.248497594497167,
342069.740771312266588,
342080.233072779665235,
342090.725401995994616,
342101.217758960323408,
342111.710143672069535,
342122.202556130359881,
342132.694996334495954,
342143.187464283546433,
342153.679959976929240,
342164.172483413713053,
342174.665034593257587,
342185.157613514806144,
342195.650220177485608,
342206.142854580597486,
342216.635516723268665,
342227.128206604858860,
342237.620924224553164,
342248.113669581536669,
342258.606442675169092,
342269.099243504402693,
342279.592072068771813,
342290.084928367403336,
342300.577812399540562,
342311.070724164368585,
342321.563663661130704,
342332.056630889012013,
342342.549625847372226,
342353.042648535396438,
342363.535698952211533,
342374.028777097060811,
342384.521882969420403,
342395.015016568184365,
342405.508177892887034,
342416.001366942480672,
342426.494583716383204,
342436.987828213779721,
342447.481100433913525,
342457.974400375969708,
342468.467728039191570,
342478.961083422764204,
342489.454466526047327,
342499.947877348226029,
342510.441315888485406,
342520.934782146126963,
342531.428276120277587,
342541.921797810296994,
342552.415347215312067,
342562.908924334566109,
342573.402529167360626,
342583.896161712822504,
342594.389821970311459,
342604.883509938954376,
342615.377225618052762,
342625.870969006791711,
342636.364740104414523,
342646.858538910222705,
342657.352365423284937,
342667.846219643019140,
342678.340101568552200,
342688.834011199127417,
342699.327948533988092,
342709.821913572319318,
342720.315906313480809,
342730.809926756541245,
342741.303974900860339,
342751.798050745681394,
342762.292154290131293,
342772.786285533511546,
342783.280444475007243,
342793.774631113861687,
342804.268845449376386,
342814.763087480678223,
342825.257357207126915,
342835.751654627849348,
342846.245979742147028,
342856.740332549205050,
342867.234713048208505,
342877.729121238575317,
342888.223557119315956,
342898.718020689790137,
342909.212511949182954,
342919.707030896795914,
342930.201577531814110,
342940.696151853480842,
342951.190753860981204,
342961.685383553674910,
342972.180040930630639,
342982.674725991208106,
342993.169438734592404,
343003.664179159968626,
343014.158947266638279,
343024.653743053844664,
343035.148566520772874,
343045.643417666782625,
343056.138296490884386,
343066.633202992496081,
343077.128137170744594,
343087.623099024989642,
343098.118088554299902,
343108.613105758035090,
343119.108150635322090,
343129.603223185520619,
343140.098323407815769,
343150.593451301392633,
343161.088606865552720,
343171.583790099422913,
343182.079001002421137,
343192.574239573674276,
343203.069505812367424,
343213.564799717802089,
343224.060121289105155,
343234.555470525694545,
343245.050847426755354,
343255.546251991414465,
343266.041684219089802,
343276.537144108733628,
343287.032631659880280,
343297.528146871598437,
343308.023689743131399,
343318.519260273722466,
343329.014858462614939,
343339.510484309052117,
343350.006137812277302,
343360.501818971592002,
343370.997527786064893,
343381.493264255113900,
343391.989028377807699,
343402.484820153447799,
343412.980639581335708,
343423.476486660598312,
343433.972361390595324,
343444.468263770395424,
343454.964193799300119,
343465.460151476669125,
343475.956136801629327,
343486.452149773482233,
343496.948190391296521,
343507.444258654490113,
343517.940354562189896,
343528.436478113697376,
343538.932629308255855,
343549.428808144992217,
343559.925014623207971,
343570.421248742204625,
343580.917510501167271,
343591.413799899339210,
343601.910116935905535,
343612.406461610109545,
343622.902833921310958,
343633.399233868636657,
343643.895661451329943,
343654.392116668634117,
343664.888599519734271,
343675.385110003931914,
343685.881648120528553,
343696.378213868651073,
343706.874807247542776,
343717.371428256446961,
343727.868076894723345,
343738.364753161440603,
343748.861457055958454,
343759.358188577403780,
343769.854947725078091,
343780.351734498166479,
343790.848548895912245,
343801.345390917675104,
343811.842260562640149,
343822.339157829934265,
343832.836082718917169,
343843.333035228657536,
343853.830015358631499,
343864.327023107907735,
343874.824058475845959,
343885.321121461573057,
343895.818212064215913,
343906.315330283308867,
343916.812476117920596,
343927.309649567294400,
343937.806850630615372,
343948.304079307301436,
343958.801335596421268,
343969.298619497218169,
343979.795931009051856,
343990.293270131049212,
344000.790636862511747,
344011.288031202566344,
344021.785453150572721,
344032.282902705832385,
344042.780379867297597,
344053.277884634502698,
344063.775417006574571,
344074.272976982756518,
344084.770564562291838,
344095.268179744365625,
344105.765822528221179,
344116.263492913160007,
344126.761190898367204,
344137.258916483144276,
344147.756669666676316,
344158.254450448264834,
344168.752258826978505,
344179.250094802293461,
344189.747958373336587,
344200.245849539234769,
344210.743768299464136,
344221.241714653035160,
344231.739688599307556,
344242.237690137466416,
344252.735719266755041,
344263.233775986533146,
344273.731860295869410,
344284.229972194123548,
344294.728111680480652,
344305.226278754184023,
344315.724473414535169,
344326.222695660602767,
344336.720945491746534,
344347.219222907209769,
344357.717527906293981,
344368.215860488067847,
344378.714220651891083,
344389.212608396948781,
344399.711023722542450,
344410.209466627857182,
344420.707937112194486,
344431.206435174739454,
344441.704960814735387,
344452.203514031483792,
344462.702094824053347,
344473.200703191861976,
344483.699339133978356,
344494.198002649878617,
344504.696693738689646,
344515.195412399596535,
344525.694158631842583,
344536.192932434787508,
344546.691733807499986,
344557.190562749339733,
344567.689419259491842,
344578.188303337316029,
344588.687214981880970,
344599.186154192429967,
344609.685120968322735,
344620.184115308802575,
344630.683137212996371,
344641.182186680322047,
344651.681263709848281,
344662.180368300760165,
344672.679500452592038,
344683.178660164237954,
344693.677847435232252,
344704.177062264585402,
344714.676304651715327,
344725.175574595632497,
344735.674872095871251,
344746.174197151500266,
344756.673549761821050,
344767.172929926018696,
344777.672337643336505,
344788.171772913017776,
344798.671235734422226,
344809.170726106618531,
344819.670244028966408,
344830.169789500592742,
344840.669362520799041,
344851.168963088886812,
344861.668591204041149,
344872.168246865563560,
344882.667930072580930,
344893.167640824336559,
344903.667379120190162,
344914.167144959326833,
344924.666938341048080,
344935.166759264422581,
344945.666607728810050,
344956.166483733511996,
344966.666387277597096,
344977.166318360483274,
344987.666276981297415,
344998.166263139282819,
345008.666276833799202,
345019.166318064031657,
345029.666386829107068,
345040.166483128443360,
345050.666606961109210,
345061.166758326522540,
345071.666937223810237,
345082.167143652215600,
345092.667377611040138,
345103.167639099410735,
345113.667928116803523,
345124.168244662228972,
345134.668588734930381,
345145.168960334383883,
345155.669359459599946,
345166.169786109880079,
345176.670240284525789,
345187.170721982722171,
345197.671231203712523,
345208.171767946681939,
345218.672332211048342,
345229.172923995938618,
345239.673543300596066,
345250.174190124322195,
345260.674864466243889,
345271.175566325662658,
345281.676295701938216,
345292.177052594139241,
345302.677837001567241,
345313.178648923465516,
345323.679488359193783,
345334.180355307704303,
345344.681249768473208,
345355.182171740802005,
345365.683121223759372,
345376.184098216646817,
345386.685102718765847,
345397.186134729243349,
345407.687194247439038,
345418.188281272538006,
345428.689395803783555,
345439.190537840360776,
345449.691707381687593,
345460.192904426774476,
345470.694128975097556,
345481.195381025783718,
345491.696660578018054,
345502.197967631218489,
345512.699302184453700,
345523.200664237025194,
345533.702053788234480,
345544.203470837266650,
345554.704915383365005,
345565.206387425772846,
345575.707886963733472,
345586.209413996490184,
345596.710968523344491,
345607.212550543481484,
345617.714160056202672,
345628.215797060634941,
345638.717461556138005,
345649.219153541896958,
345659.720873017271515,
345670.222619981330354,
345680.724394433374982,
345691.226196372765116,
345701.728025798511226,
345712.229882710089441,
345722.731767106626648,
345733.233678987424355,
345743.735618351667654,
345754.237585198658053,
345764.739579527580645,
345775.241601337736938,
345785.743650628370233,
345796.245727398665622,
345806.747831647924613,
345817.249963375332300,
345827.752122580248397,
345838.254309261799790,
345848.756523419171572,
345859.258765051839873,
345869.761034158989787,
345880.263330739689991,
345890.765654793300200,
345901.268006319121923,
345911.770385316282045,
345922.272791784082074,
345932.775225721765310,
345943.277687128575053,
345953.780176003812812,
345964.282692346663680,
345974.785236156312749,
345985.287807432061527,
345995.790406173269730,
346006.293032379064243,
346016.795686048688367,
346027.298367181385402,
346037.801075776456855,
346048.303811833146028,
346058.806575350638013,
346069.309366328176111,
346079.812184765120037,
346090.315030660538469,
346100.817904013849329,
346111.320804824121296,
346121.823733090830501,
346132.326688813045621,
346142.829671990068164,
346153.332682621083222,
346163.835720705508720,
346174.338786242413335,
346184.841879231098574,
346195.344999670749530,
346205.848147560784128,
346216.351322900329251,
346226.854525688628200,
346237.357755924924277,
346247.861013608518988,
346258.364298738655634,
346268.867611314519309,
346279.370951335353311,
346289.874318800459150,
346300.377713709080126,
346310.881136060517747,
346321.384585853840690,
346331.888063088466879,
346342.391567763581406,
346352.895099878427573,
346363.398659432306886,
346373.902246424229816,
346384.405860853788909,
346394.909502719994634,
346405.413172022264916,
346415.916868759668432,
346426.420592931564897,
346436.924344537255820,
346447.428123575809877,
346457.931930046586785,
346468.435763948771637,
346478.939625281782355,
346489.443514044687618,
346499.947430236788932,
346510.451373857213184,
346520.955344905494712,
346531.459343380702194,
346541.963369282078929,
346552.467422608868219,
346562.971503360313363,
346573.475611535774078,
346583.979747134377249,
346594.483910155366175,
346604.988100598100573,
346615.492318461765535,
346625.996563745546155,
346636.500836448743939,
346647.005136570602190,
346657.509464110422414,
346668.013819067447912,
346678.518201440805569,
346689.022611229855102,
346699.527048433839809,
346710.031513052002992,
346720.536005083529744,
346731.040524527721573,
346741.545071383763570,
346752.049645651073661,
346762.554247328720521,
346773.058876416005660,
346783.563532912114169,
346794.068216816463973,
346804.572928128240164,
346815.077666846627835,
346825.582432970928494,
346836.087226500385441,
346846.592047434183769,
346857.096895771683194,
346867.601771512068808,
346878.106674654525705,
346888.611605198471807,
346899.116563142975792,
346909.621548487397376,
346920.126561230863445,
346930.631601372791920,
346941.136668912367895,
346951.641763848834671,
346962.146886181435548,
346972.652035909355618,
346983.157213032012805,
346993.662417548475787,
347004.167649458162487,
347014.672908760141581,
347025.178195453772787,
347035.683509538299404,
347046.188851012964733,
347056.694219877012074,
347067.199616129626520,
347077.705039770167787,
347088.210490797820967,
347098.715969211887568,
347109.221475011610892,
347119.727008196176030,
347130.232568764826283,
347140.738156716921367,
347151.243772051646374,
347161.749414768244606,
347172.255084866017569,
347182.760782344092149,
347193.266507201828063,
347203.772259438410401,
347214.278039053140674,
347224.783846045262180,
347235.289680414076429,
347245.795542158710305,
347256.301431278523523,
347266.807347772701178,
347277.313291640544776,
347287.819262881239410,
347298.325261494028382,
347308.831287478213198,
347319.337340833153576,
347329.843421557859983,
347340.349529651808552,
347350.855665114126168,
347361.361827943997923,
347371.868018140783533,
347382.374235703784507,
347392.880480632127728,
347403.386752925172914,
347413.893052582046948,
347424.399379602109548,
347434.905733984662220,
347445.412115728715435,
347455.918524833803531,
347466.424961298936978,
347476.931425123591907,
347487.437916306778789,
347497.944434847973753,
347508.450980746187270,
347518.957554000953678,
347529.464154611341655,
347539.970782576652709,
347550.477437896130141,
347560.984120569017250,
347571.490830594615545,
347581.997567972110119,
347592.504332700802479,
347603.011124779877719,
347613.517944208637346,
347624.024790986324660,
347634.531665112241171,
347645.038566585571971,
347655.545495405560359,
347666.052451571449637,
347676.559435082657728,
347687.066445938195102,
347697.573484137537889,
347708.080549679812975,
347718.587642564205453,
347729.094762790075038,
347739.601910356665030,
347750.109085263218731,
347760.616287509037647,
347771.123517093248665,
347781.630774015211500,
347792.138058274111245,
347802.645369869191200,
347813.152708799811080,
347823.660075065155979,
347834.167468664469197,
347844.674889597052243,
347855.182337862032000,
347865.689813458826393,
347876.197316386620514,
347886.704846644657664,
347897.212404232122935,
347907.719989148434252,
347918.227601392660290,
347928.735240964277182,
347939.242907862237189,
347949.750602086016443,
347960.258323634800036,
347970.766072507889476,
347981.273848704528064,
347991.781652223959100,
348002.289483065367676,
348012.797341228055302,
348023.305226711323485,
348033.813139514357317,
348044.321079636458308,
348054.829047076811548,
348065.337041834776755,
348075.845063909480814,
348086.353113300341647,
348096.861190006427933,
348107.369294027041178,
348117.877425361541100,
348128.385584009170998,
348138.893769969057757,
348149.401983240561094,
348159.910223822924308,
348170.418491715274286,
348180.926786917087156,
348191.435109427431598,
348201.943459245725535,
348212.451836371095851,
348222.960240802727640,
348233.468672540038824,
348243.977131582330912,
348254.485617928614374,
348264.994131578248926,
348275.502672530594282,
348286.011240784835536,
348296.519836340157781,
348307.028459195978940,
348317.537109351367690,
348328.045786805741955,
348338.554491558228619,
348349.063223608187400,
348359.571982954745181,
348370.080769597319886,
348380.589583534980193,
348391.098424767144024,
348401.607293293054681,
348412.116189111780841,
348422.625112222740427,
348433.134062625234947,
348443.643040318391286,
348454.152045301510952,
348464.661077573837247,
348475.170137134671677,
348485.679223983315751,
348496.188338118838146,
348506.697479540656786,
348517.206648248014972,
348527.715844240039587,
348538.225067516148556,
348548.734318075468764,
348559.243595917359926,
348569.752901041065343,
348580.262233445711900,
348590.771593130659312,
348601.280980095209088,
348611.790394338546321,
348622.299835859914310,
348632.809304658672772,
348643.318800733948592,
348653.828324085043278,
348664.337874711258337,
348674.847452611778863,
348685.357057785906363,
348695.866690232884139,
348706.376349951955490,
348716.886036942363717,
348727.395751203526743,
348737.905492734396830,
348748.415261534450110,
348758.925057602929883,
348769.434880939021241,
348779.944731542083900,
348790.454609411244746,
348800.964514545805287,
348811.474446945067029,
348821.984406608215068,
348832.494393534609117,
348843.004407723434269,
348853.514449173933826,
348864.024517885467503,
348874.534613857103977,
348885.044737088261172,
348895.554887578124180,
348906.065065325936303,
348916.575270331057254,
348927.085502592613921,
348937.595762109966017,
348948.106048882356845,
348958.616362908971496,
348969.126704189169686,
348979.637072722136509,
348990.147468507057056,
349000.657891543407459,
349011.168341830198187,
349021.678819366847165,
349032.189324152539484,
349042.699856186576653,
349053.210415468260180,
349063.721001996716950,
349074.231615771248471,
349084.742256791156251,
349095.252925055683590,
349105.763620564131998,
349116.274343315686565,
349126.785093309648801,
349137.295870545262005,
349147.806675021711271,
349158.317506738356315,
349168.828365694498643,
349179.339251889206935,
349189.850165321899112,
349200.361105991818476,
349210.872073898091912,
349221.383069040195551,
349231.894091417256277,
349242.405141028459184,
349252.916217873163987,
349263.427321950672194,
349273.938453260110691,
349284.449611800897401,
349294.960797572159208,
349305.472010573139414,
349315.983250803255942,
349326.494518261693884,
349337.005812947580125,
349347.517134860332590,
349358.028483999136370,
349368.539860363292973,
349379.051263951987494,
349389.562694764637854,
349400.074152800312731,
349410.585638058313634,
349421.097150537942071,
349431.608690238557756,
349442.120257159229368,
349452.631851299374830,
349463.143472658179235,
349473.655121234827675,
349484.166797028738074,
349494.678500039037317,
349505.190230265026912,
349515.701987706008367,
349526.213772361224983,
349536.725584229861852,
349547.237423311220482,
349557.749289604602382,
349568.261183109309059,
349578.773103824350983,
349589.285051749320701,
349599.797026883286890,
349610.309029225551058,
349620.821058775356505,
349631.333115531946532,
349641.845199494680855,
349652.357310662686359,
349662.869449035322759,
349673.381614611775149,
349683.893807391345035,
349694.406027373333927,
349704.918274556868710,
349715.430548941367306,
349725.942850526014809,
349736.455179310054518,
349746.967535292787943,
349757.479918473400176,
349767.992328851192724,
349778.504766425525304,
349789.017231195583008,
349799.529723160492722,
349810.042242319788784,
349820.554788672481664,
349831.067362217931077,
349841.579962955438532,
349852.092590884189121,
349862.605246003484353,
349873.117928312625736,
349883.630637810740154,
349894.143374497245532,
349904.656138371326961,
349915.168929432227742,
349925.681747679249384,
349936.194593111518770,
349946.707465728512034,
349957.220365529414266,
349967.733292513410561,
349978.246246679860633,
349988.759228027949575,
349999.272236557037104,
350009.785272266250104,
350020.298335154890083,
350030.811425222316757,
350041.324542467657011,
350051.837686890270561,
350062.350858489342500,
350072.864057264232542,
350083.377283214125782,
350093.890536338207312,
350104.403816635953262,
350114.917124106490519,
350125.430458749004174,
350135.943820562853944,
350146.457209547399543,
350156.970625701709650,
350167.484069025202189,
350177.997539517062251,
350188.511037176474929,
350199.024562002858147,
350209.538113995455205,
350220.051693153334782,
350230.565299476031214,
350241.078932962613180,
350251.592593612440396,
350262.106281424756162,
350272.619996398745570,
350283.133738533710130,
350293.647507829009555,
350304.161304283770733,
350314.675127897295170,
350325.188978668942582,
350335.702856597839855,
350346.216761683346704,
350356.730693924648222,
350367.244653321104124,
350377.758639871899504,
350388.272653576335870,
350398.786694433598313,
350409.300762443046551,
350419.814857603923883,
350430.328979915415403,
350440.843129376939032,
350451.357305987563450,
350461.871509746648371,
350472.385740653553512,
350482.899998707347549,
350493.414283907448407,
350503.928596252982970,
350514.442935743369162,
350524.957302377792075,
350535.471696155495010,
350545.986117075779475,
350556.500565137830563,
350567.015040341066197,
350577.529542684613261,
350588.044072167773265,
350598.558628789847717,
350609.073212550079916,
350619.587823447713163,
350630.102461481990758,
350640.617126652214210,
350651.131818957685027,
350661.646538397588301,
350672.161284971167333,
350682.676058677781839,
350693.190859516616911,
350703.705687487032264,
350714.220542588154785,
350724.735424819402397,
350735.250334179960191,
350745.765270669071469,
350756.280234286037739,
350766.795225030102301,
350777.310242900508456,
350787.825287896615919,
350798.340360017551575,
350808.855459262616932,
350819.370585631171707,
350829.885739122400992,
350840.400919735548086,
350850.916127469972707,
350861.431362324918155,
350871.946624299511313,
350882.461913393170107,
350892.977229605021421,
350903.492572934483178,
350914.007943380798679,
350924.523340943153016,
350935.038765620789491,
350945.554217413009610,
350956.069696319173090,
350966.585202338465024,
350977.100735470128711,
350987.616295713407453,
350998.131883067660965,
351008.647497532132547,
351019.163139106007293,
351029.678807788586710,
351040.194503579230513,
351050.710226477065589,
351061.225976481393445,
351071.741753591573797,
351082.257557806733530,
351092.773389126174152,
351103.289247549255379,
351113.805133075162303,
351124.321045703196432,
351134.836985432601068,
351145.352952262619510,
351155.868946192495059,
351166.384967221645638,
351176.901015349198133,
351187.417090574454051,
351197.933192896598484,
351208.449322315049358,
351218.965478828991763,
351229.481662437727209,
351239.997873140440788,
351250.514110936434008,
351261.030375825066585,
351271.546667805465404,
351282.062986876990180,
351292.579333038767800,
351303.095706290332600,
351313.612106630636845,
351324.128534059273079,
351334.644988575193565,
351345.161470177874435,
351355.677978866500780,
351366.194514640315901,
351376.711077498679515,
351387.227667440776713,
351397.744284465850797,
351408.260928573319688,
351418.777599762252066,
351429.294298032007646,
351439.811023381887935,
351450.327775811078027,
351460.844555318937637,
351471.361361904651858,
351481.878195567580406,
351492.395056306908373,
351502.911944121820852,
351513.428859011852182,
351523.945800976012833,
351534.462770013720728,
351544.979766124160960,
351555.496789306576829,
351566.013839560386259,
351576.530916884657927,
351587.048021278809756,
351597.565152742026839,
351608.082311273610685,
351618.599496872862801,
351629.116709539026488,
351639.633949271228630,
351650.151216068945359,
351660.668509931419976,
351671.185830857779365,
351681.703178847383242,
351692.220553899533115,
351702.737956013355870,
351713.255385188269429,
351723.772841423458885,
351734.290324718225747,
351744.807835071871523,
351755.325372483697720,
351765.842936952714808,
351776.360528478515334,
351786.878147060167976,
351797.395792697032448,
351807.913465388352051,
351818.431165133311879,
351828.948891931329854,
351839.466645781591069,
351849.984426683338825,
351860.502234635932837,
351871.020069638499990,
351881.537931690458208,
351892.055820790934376,
351902.573736939346418,
351913.091680134879425,
351923.609650376776699,
351934.127647664397955,
351944.645671996870078,
351955.163723373552784,
351965.681801793747582,
351976.199907256697770,
351986.718039761588443,
351997.236199307837524,
352007.754385894571897,
352018.272599521209486,
352028.790840186877176,
352039.309107890876476,
352049.827402632567100,
352060.345724411075935,
352070.864073225820903,
352081.382449075928889,
352091.900851960759610,
352102.419281879556365,
352112.937738831562456,
352123.456222816137597,
352133.974733832466882,
352144.493271879793610,
352155.011836957477499,
352165.530429064703640,
352176.049048200773541,
352186.567694364988711,
352197.086367556650657,
352207.605067774944473,
352218.123795019171666,
352228.642549288575538,
352239.161330582515802,
352249.680138900119346,
352260.198974240745883,
352270.717836603696924,
352281.236725988157559,
352291.755642393487506,
352302.274585818813648,
352312.793556263553910,
352323.312553726951592,
352333.831578208249994,
352344.350629706692416,
352354.869708221580368,
352365.388813752098940,
352375.907946297724266,
352386.427105857525021,
352396.946292430802714,
352407.465506016917061,
352417.984746615111362,
352428.504014224628918,
352439.023308844771236,
352449.542630474781618,
352460.061979113903362,
352470.581354761437979,
352481.100757416686974,
352491.620187078893650,
352502.139643747243099,
352512.659127421211451,
352523.178638099867385,
352533.698175782570615,
352544.217740468622651,
352554.737332157208584,
352565.256950847688131,
352575.776596539188176,
352586.296269231184851,
352596.815968922805041,
352607.335695613350254,
352617.855449302121997,
352628.375229988421779,
352638.895037671260070,
352649.414872350345831,
352659.934734024631325,
352670.454622693476267,
352680.974538356182165,
352691.494481011934113,
352702.014450660033617,
352712.534447299840394,
352723.054470930597745,
352733.574521551490761,
352744.094599161937367,
352754.614703761006240,
352765.134835348115303,
352775.654993922507856,
352786.175179483485408,
352796.695392030233052,
352807.215631562052295,
352817.735898078244645,
352828.256191578111611,
352838.776512060896493,
352849.296859525784384,
352859.817233972135000,
352870.337635399249848,
352880.858063806372229,
352891.378519192687236,
352901.899001557554584,
352912.419510900217574,
352922.940047220035922,
352933.460610516252927,
352943.981200787995476,
352954.501818034623284,
352965.022462255496066,
352975.543133449798916,
352986.063831616833340,
352996.584556755784433,
353007.105308866070118,
353017.626087946817279,
353028.146893997443840,
353038.667727017134894,
353049.188587005191948,
353059.709473960800096,
353070.230387883377261,
353080.751328772108536,
353091.272296626295429,
353101.793291445181239,
353112.314313228067476,
353122.835361974197440,
353133.356437682872638,
353143.877540353336371,
353154.398669984831940,
353164.919826576777268,
353175.441010128357448,
353185.962220638815779,
353196.483458107395563,
353207.004722533514723,
353217.526013916300144,
353228.047332255053334,
353238.568677549134009,
353249.090049797785468,
353259.611449000134598,
353270.132875155657530,
353280.654328263481148,
353291.175808323023375,
353301.697315333411098,
353312.218849294004031,
353322.740410204045475,
353333.261998062778730,
353343.783612869621720,
353354.305254623643123,
353364.826923324260861,
353375.348618970718235,
353385.870341562258545,
353396.392091098125093,
353406.913867577619385,
353417.435671000159346,
353427.957501364755444,
353438.479358670883812,
353449.001242917845957,
353459.523154104710557,
353470.045092230895534,
353480.567057295702398,
353491.089049298258033,
353501.611068237980362,
353512.133114114112686,
353522.655186925840098,
353533.177286672580522,
353543.699413353519049,
353554.221566967898980,
353564.743747515080031,
353575.265954994305503,
353585.788189404818695,
353596.310450745921116,
353606.832739016856067,
353617.355054216983262,
353627.877396345546003,
353638.399765401729383,
353648.922161384893116,
353659.444584294280503,
353669.967034129193053,
353680.489510888815857,
353691.012014572625048,
353701.534545179631095,
353712.057102709368337,
353722.579687160963658,
353733.102298533718567,
353743.624936826876365,
353754.147602039796766,
353764.670294171723071,
353775.193013221840374,
353785.715759189508390,
353796.238532074028626,
353806.761331874644384,
353817.284158590540756,
353827.807012221193872,
353838.329892765672412,
353848.852800223336089,
353859.375734593544621,
353869.898695875424892,
353880.421684068394825,
353890.944699171639513,
353901.467741184344050,
353911.990810106042773,
353922.513905935804360,
353933.037028672988527,
353943.560178316780366,
353954.083354866539594,
353964.606558321567718,
353975.129788681049831,
353985.653045944345649,
353996.176330110698473,
354006.699641179351602,
354017.222979149606545,
354027.746344020823017,
354038.269735792127904,
354048.793154462880921,
354059.316600032383576,
354069.840072499879170,
354080.363571864611004,
354090.887098125880584,
354101.410651282989420,
354111.934231335180812,
354122.457838281698059,
354132.981472121959087,
354143.505132855090778,
354154.028820480452850,
354164.552534997230396,
354175.076276404841337,
354185.600044702470768,
354196.123839889361989,
354206.647661964874715,
354217.171510928252246,
354227.695386778737884,
354238.219289515691344,
354248.743219138355926,
354259.267175645916723,
354269.791159037733451,
354280.315169313049410,
354290.839206471224315,
354301.363270511559676,
354311.887361433124170,
354322.411479235393927,
354332.935623917612247,
354343.459795478906017,
354353.983993918809574,
354364.508219236391596,
354375.032471431011800,
354385.556750501855277,
354396.081056448339950,
354406.605389269709121,
354417.129748965147883,
354427.654135534015950,
354438.178548975614831,
354448.702989289187826,
354459.227456473978236,
354469.751950529287569,
354480.276471454417333,
354490.801019248610828,
354501.325593911169562,
354511.850195441395044,
354522.374823838530574,
354532.899479101819452,
354543.424161230621394,
354553.948870224121492,
354564.473606081621256,
354574.998368802480400,
354585.523158385942224,
354596.047974831250031,
354606.572818137647118,
354617.097688304551411,
354627.622585331089795,
354638.147509216621984,
354648.672459960507695,
354659.197437561815605,
354669.722442019963637,
354680.247473334195092,
354690.772531503753271,
354701.297616528056096,
354711.822728406230453,
354722.347867137636058,
354732.873032721516211,
354743.398225157172419,
354753.923444443789776,
354764.448690580844413,
354774.973963567463215,
354785.499263402947690,
354796.024590086599346,
354806.549943617719691,
354817.075323995552026,
354827.600731219339650,
354838.126165288442280,
354848.651626202103216,
354859.177113959507551,
354869.702628560131416,
354880.228170003159903,
354890.753738287719898,
354901.279333413345739,
354911.804955379164312,
354922.330604184593540,
354932.856279828702100,
354943.381982310907915,
354953.907711630512495,
354964.433467786759138,
354974.959250778832939,
354985.485060606151819,
354996.010897267959081,
355006.536760763556231,
355017.062651092070155,
355027.588568252918776,
355038.114512245403603,
355048.640483068826143,
355059.166480722313281,
355069.692505205224734,
355080.218556516862009,
355090.744634656468406,
355101.270739623403642,
355111.796871416852809,
355122.323030036175624,
355132.849215480557177,
355143.375427749415394,
355153.901666841818951,
355164.427932757243980,
355174.954225494933780,
355185.480545054073445,
355196.006891434080899,
355206.533264634199440,
355217.059664653614163,
355227.586091491626576,
355238.112545147538185,
355248.639025620708708,
355259.165532910381444,
355269.692067015799694,
355280.218627936264966,
355290.745215671020560,
355301.271830219426192,
355311.798471580725163,
355322.325139754160773,
355332.851834739034530,
355343.378556534647942,
355353.905305140302517,
355364.432080555241555,
355374.958882778708357,
355385.485711810062639,
355396.012567648547702,
355406.539450293465052,
355417.066359744116198,
355427.593295999628026,
355438.120259059534874,
355448.647248922905419,
355459.174265589157585,
355469.701309057418257,
355480.228379327163566,
355490.755476397578605,
355501.282600267906673,
355511.809750937449280,
355522.336928405507933,
355532.864132671384141,
355543.391363734379411,
355553.918621593678836,
355564.445906248642132,
355574.973217698570807,
355585.500555942649953,
355596.027920980239287,
355606.555312810465693,
355617.082731432921719,
355627.610176846676040,
355638.137649051030166,
355648.665148045285605,
355659.192673828685656,
355669.720226400648244,
355680.247805760300253,
355690.775411907001399,
355701.303044839994982,
355711.830704558582511,
355722.358391062065493,
355732.886104349687230,
355743.413844420749228,
355753.941611274552997,
355764.469404910341837,
355774.997225327475462,
355785.525072525138967,
355796.052946502692066,
355806.580847259319853,
355817.108774794382043,
355827.636729107238352,
355838.164710196957458,
355848.692718063073698,
355859.220752704655752,
355869.748814121063333,
355880.276902311597951,
355890.805017275561113,
355901.333159012254328,
355911.861327520862687,
355922.389522800745908,
355932.917744851147290,
355943.445993671426550,
355953.974269260710571,
355964.502571618417278,
355975.030900743848179,
355985.559256636188366,
355996.087639294739347,
356006.616048718860839,
356017.144484907737933,
356027.672947860730346,
356038.201437577139586,
356048.729954056150746,
356059.258497297065333,
356069.787067299243063,
356080.315664061927237,
356090.844287584419362,
356101.372937865962740,
356111.901614905858878,
356122.430318703409284,
356132.959049257973675,
356143.487806568678934,
356154.016590634826571,
356164.545401455834508,
356175.074239030829631,
356185.603103359229863,
356196.131994440162089,
356206.660912273102440,
356217.189856857294217,
356227.718828191864304,
356238.247826276288833,
356248.776851109694690,
356259.305902691499796,
356269.834981020831037,
356280.364086097164545,
356290.893217919627205,
356301.422376487636939,
356311.951561800378840,
356322.480773857154418,
356333.010012657265179,
356343.539278200012632,
356354.068570484640077,
356364.597889510449022,
356375.127235276682768,
356385.656607782759238,
356396.186007027805317,
356406.715433011238929,
356417.244885732245166,
356427.774365190125536,
356438.303871384181548,
356448.833404313772917,
356459.362963978026528,
356469.892550376418512,
356480.422163508075755,
356490.951803372357972,
356501.481469968508463,
356512.011163295828737,
356522.540883353678510,
356533.070630141242873,
356543.600403657823335,
356554.130203902663197,
356564.660030875238590,
356575.189884574618191,
356585.719765000219923,
356596.249672151287086,
356606.779606027062982,
356617.309566626907326,
356627.839553950005211,
356638.369567995774560,
356648.899608763400465,
356659.429676252300851,
356669.959770461544394,
356680.489891390607227,
356691.020039038674440,
356701.550213405047543,
356712.080414489086252,
356722.610642289975658,
356733.140896807075478,
356743.671178039570805,
356754.201485986879561,
356764.731820648245048,
356775.262182022910565,
356785.792570110235829,
356796.322984909347724,
356806.853426419722382,
356817.383894640544895,
356827.914389571174979,
356838.444911210797727,
356848.975459558772855,
356859.506034614343662,
356870.036636376811657,
356880.567264845478348,
356891.097920019703452,
356901.628601898613852,
356912.159310481511056,
356922.690045767812990,
356933.220807756821159,
356943.751596447720658,
356954.282411839696579,
356964.813253932283260,
356975.344122724607587,
356985.875018215971068,
356996.405940405733418,
357006.936889293079730,
357017.467864877311513,
357027.998867157904897,
357038.529896133870352,
357049.060951804625802,
357059.592034169530962,
357070.123143227712717,
357080.654278978588991,
357091.185441421344876,
357101.716630555398297,
357112.247846379934344,
357122.779088894254528,
357133.310358097718563,
357143.841653989453334,
357154.372976568934973,
357164.904325835290365,
357175.435701787937433,
357185.967104426119477,
357196.498533749079797,
357207.029989756119903,
357217.561472446599510,
357228.092981819820125,
357238.624517874850426,
357249.156080611224752,
357259.687670028069988,
357270.219286124804057,
357280.750928900670260,
357291.282598354911897,
357301.814294486830477,
357312.346017295785714,
357322.877766781020910,
357333.409542941721156,
357343.941345777304377,
357354.473175287013873,
357365.005031470209360,
357375.536914326075930,
357386.068823853973299,
357396.600760053086560,
357407.132722922833636,
357417.664712462457828,
357428.196728671202436,
357438.728771548427176,
357449.260841093258932,
357459.792937305232044,
357470.325060183531605,
357480.857209727342706,
357491.389385936141480,
357501.921588809054811,
357512.453818345500622,
357522.986074544664007,
357533.518357405846473,
357544.050666928349528,
357554.583003111591097,
357565.115365954581648,
357575.647755456913728,
357586.180171617714223,
357596.712614436226431,
357607.245083911868278,
357617.777580043883063,
357628.310102831455879,
357638.842652274004649,
357649.375228370830882,
357659.907831121177878,
357670.440460524230730,
357680.973116579465568,
357691.505799286067486,
357702.038508643337991,
357712.571244650578592,
357723.104007307090797,
357733.636796612176113,
357744.169612565077841,
357754.702455165039282,
357765.235324411420152,
357775.768220303638373,
357786.301142840704415,
357796.834092022152618,
357807.367067847109865,
357817.900070314994082,
357828.433099425048567,
357838.966155176458415,
357849.499237568641547,
357860.032346600841265,
357870.565482272359077,
357881.098644582554698,
357891.631833530555014,
357902.165049115777947,
357912.698291337466799,
357923.231560194981284,
357933.764855687506497,
357944.298177814402152,
357954.831526574911550,
357965.364901968394406,
357975.898303994035814,
357986.431732651253697,
357996.965187939291354,
358007.498669857392088,
358018.032178404857405,
358028.565713580930606,
358039.099275385146029,
358049.632863816455938,
358060.166478874394670,
358070.700120558147319,
358081.233788867073599,
358091.767483800358605,
358102.301205357362051,
358112.834953537385445,
358123.368728339788504,
358133.902529763698112,
358144.436357808473986,
358154.970212473475840,
358165.504093757946976,
358176.038001661072485,
358186.571936182328500,
358197.105897320841905,
358207.639885076088831,
358218.173899447254371,
358228.707940433523618,
358239.242008034372702,
358249.776102248986717,
358260.310223076667171,
358270.844370516773779,
358281.378544568491634,
358291.912745231238659,
358302.446972504199948,
358312.981226386677008,
358323.515506878029555,
358334.049813977559097,
358344.584147684450727,
358355.118507998064160,
358365.652894917642698,
358376.187308442604262,
358386.721748572075740,
358397.256215305475052,
358407.790708641987294,
358418.325228581030387,
358428.859775121847633,
358439.394348263682332,
358449.928948005835991,
358460.463574347668327,
358470.998227288422640,
358481.532906827342231,
358492.067612963845022,
358502.602345697116107,
358513.137105026456993,
358523.671890951285604,
358534.206703470728826,
358544.741542584146373,
358555.276408290839754,
358565.811300590110477,
358576.346219481260050,
358586.881164963531774,
358597.416137036227155,
358607.951135698647704,
358618.486160950094927,
358629.021212789928541,
358639.556291217275430,
358650.091396231553517,
358660.626527832064312,
358671.161686018109322,
358681.696870788873639,
358692.232082143775187,
358702.767320081940852,
358713.302584602846764,
358723.837875705736224,
358734.373193389852531,
358744.908537654497195,
358755.443908498971723,
358765.979305922577623,
358776.514729924674612,
358787.050180504389573,
358797.585657661198638,
358808.121161394286901,
358818.656691703014076,
358829.192248586565256,
358839.727832044358365,
358850.263442075636704,
358860.799078679643571,
358871.334741855738685,
358881.870431603223551,
358892.406147921341471,
358902.941890809452161,
358913.477660266682506,
358924.013456292508636,
358934.549278886232059,
358945.085128047096077,
358955.621003774343990,
358966.156906067335512,
358976.692834925313946,
358987.228790347580798,
358997.764772333437577,
359008.300780882185791,
359018.836815993243363,
359029.372877665620763,
359039.908965898910537,
359050.445080692181364,
359060.981222044851165,
359071.517389956163242,
359082.053584425477311,
359092.589805452036671,
359103.126053035084624,
359113.662327174039092,
359124.198627868026961,
359134.734955116582569,
359145.271308918774594,
359155.807689273962751,
359166.344096181564964,
359176.880529640649911,
359187.416989650810137,
359197.953476210997906,
359208.489989320747554,
359219.026528979360592,
359229.563095185963903,
359240.099687939975411,
359250.636307240638416,
359261.172953087312635,
359271.709625479241367,
359282.246324415726122,
359292.783049896010198,
359303.319801919569727,
359313.856580485473387,
359324.393385593197308,
359334.930217241926584,
359345.467075430962723,
359356.003960159665439,
359366.540871427278034,
359377.077809233102016,
359387.614773576438893,
359398.151764456590172,
359408.688781872915570,
359419.225825824541971,
359429.762896310945507,
359440.299993331369478,
359450.837116885057185,
359461.374266971251927,
359471.911443589429837,
359482.448646738776006,
359492.985876418533735,
359503.523132628120948,
359514.060415366780944,
359524.597724633757025,
359535.135060428525321,
359545.672422750096302,
359556.209811598004308,
359566.747226971376222,
359577.284668869688176,
359587.822137292183470,
359598.359632237988990,
359608.897153706639074,
359619.434701697260607,
359629.972276209213305,
359640.509877241798677,
359651.047504794318229,
359661.585158866073471,
359672.122839456307702,
359682.660546564380638,
359693.198280189593788,
359703.736040331132244,
359714.273826988413930,
359724.811640160682146,
359735.349479847296607,
359745.887346047442406,
359756.425238760537468,
359766.963157985825092,
359777.501103722606786,
359788.039075970125850,
359798.577074727742001,
359809.115099994698539,
359819.653151770413388,
359830.191230054013431,
359840.729334844916593,
359851.267466142424382,
359861.805623945663683,
359872.343808254227042,
359882.882019067183137,
359893.420256383891683,
359903.958520203654189,
359914.496810525772162,
359925.035127349547110,
359935.573470674280543,
359946.111840499215759,
359956.650236823712476,
359967.188659647130407,
359977.727108968538232,
359988.265584787528496,
359998.804087103169877,
360009.342615914880298,
360019.881171221903060,
360030.419753023539670,
360040.958361319091637,
360051.496996107860468,
360062.035657389205880,
360072.574345162371174,
360083.113059426599648,
360093.651800181251019,
360104.190567425685003,
360114.729361159086693,
360125.268181380815804,
360135.807028090173844,
360146.345901286404114,
360156.884800968808122,
360167.423727136745583,
360177.962679789518006,
360188.501658926368691,
360199.040664546657354,
360209.579696649569087,
360220.118755234521814,
360230.657840300817043,
360241.196951847639866,
360251.736089874408208,
360262.275254380307160,
360272.814445364812855,
360283.353662826993968,
360293.892906766326632,
360304.432177182054147,
360314.971474073419813,
360325.510797439841554,
360336.050147280504461,
360346.589523594710045,
360357.128926381876227,
360367.668355641188100,
360378.207811372005381,
360388.747293573571369,
360399.286802245303988,
360409.826337386330124,
360420.365898996125907,
360430.905487073760014,
360441.445101618766785,
360451.984742630389519,
360462.524410107813310,
360473.064104050456081,
360483.603824457619339,
360494.143571328429971,
360504.683344662480522,
360515.223144458723255,
360525.762970716808923,
360536.302823435747996,
360546.842702615074813,
360557.382608253916260,
360567.922540351632051,
360578.462498907465488,
360589.002483920892701,
360599.542495391040575,
360610.082533317268826,
360620.622597698820755,
360631.162688535172492,
360641.702805825392716,
360652.242949568899348,
360662.783119765052106,
360673.323316413094290,
360683.863539512210991,
360694.403789061820135,
360704.944065061281435,
360715.484367509838194,
360726.024696406675503,
360736.565051751269493,
360747.105433542805258,
360757.645841780584306,
360768.186276464024559,
360778.726737592311110,
360789.267225164803676,
360799.807739180803765,
360810.348279639496468,
360820.888846540299710,
360831.429439882514998,
360841.970059665443841,
360852.510705888329539,
360863.051378550531808,
360873.592077651235741,
360884.132803189917468,
360894.673555165762082,
360905.214333578129299,
360915.755138426204212,
360926.295969709462952,
360936.836827427090611,
360947.377711578388698,
360957.918622162716929,
360968.459559179376811,
360979.000522627611645,
360989.541512506722938,
361000.082528816012200,
361010.623571554897353,
361021.164640722563490,
361031.705736318253912,
361042.246858341502957,
361052.788006791321095,
361063.329181667242665,
361073.870382968452759,
361084.411610694252886,
361094.952864844002761,
361105.494145417003892,
361116.035452412441373,
361126.576785829733126,
361137.118145668180659,
361147.659531927085482,
361158.200944605632685,
361168.742383703298401,
361179.283849219267722,
361189.825341152842157,
361200.366859503381420,
361210.908404270128813,
361221.449975452502258,
361231.991573049628641,
361242.533197060925886,
361253.074847485753708,
361263.616524323238991,
361274.158227572857868,
361284.699957233737223,
361295.241713305353187,
361305.783495786890853,
361316.325304677709937,
361326.867139977053739,
361337.409001684281975,
361347.950889798754361,
361358.492804319655988,
361369.034745246346574,
361379.576712578069419,
361390.118706314242445,
361400.660726454050746,
361411.202772996795829,
361421.744845941953827,
361432.286945288593415,
361442.829071036307141,
361453.371223184105475,
361463.913401731348131,
361474.455606677453034,
361484.997838021721691,
361495.540095763280988,
361506.082379901665263,
361516.624690436059609,
361527.167027365765534,
361537.709390690084547,
361548.251780408318155,
361558.794196519767866,
361569.336639023793396,
361579.879107919638045,
361590.421603206661530,
361600.964124884048942,
361611.506672951276414,
361622.049247407470830,
361632.591848252108321,
361643.134475484373979,
361653.677129103569314,
361664.219809109112248,
361674.762515500187874,
361685.305248276097700,
361695.848007436259650,
361706.390792979858816,
361716.933604906254914,
361727.476443214749452,
361738.019307904585730,
361748.562198975239880,
361759.105116425838787,
361769.648060255742166,
361780.191030464193318,
361790.734027050610166,
361801.277050014294218,
361811.820099354488775,
361822.363175070495345,
361832.906277161615435,
361843.449405627150554,
361853.992560466460418,
361864.535741678846534,
361875.078949263552204,
361885.622183219937142,
361896.165443547302857,
361906.708730244892649,
361917.252043312008027,
361927.795382748066913,
361938.338748552312609,
361948.882140723930206,
361959.425559262454044,
361969.969004167011008,
361980.512475437019020,
361991.055973071721382,
362001.599497070419602,
362012.143047432415187,
362022.686624157009646,
362033.230227243620902,
362043.773856691375840,
362054.317512499692384,
362064.861194667813834,
362075.404903195099905,
362085.948638080793899,
362096.492399324313737,
362107.036186924844515,
362117.580000881804153,
362128.123841194319539,
362138.667707861866802,
362149.211600883689243,
362159.755520259204786,
362170.299465987482108,
362180.843438067997340,
362191.387436499993782,
362201.931461282831151,
362212.475512415810954,
362223.019589898118284,
362233.563693729229271,
362244.107823908270802,
362254.651980434777215,
362265.196163307875395,
362275.740372526925057,
362286.284608091227710,
362296.828870000084862,
362307.373158252798021,
362317.917472848668694,
362328.461813787056599,
362339.006181067205034,
362349.550574688415509,
362360.094994650105946,
362370.639440951403230,
362381.183913591841701,
362391.728412570431828,
362402.272937886824366,
362412.817489540029783,
362423.362067529466003,
362433.906671854434535,
362444.451302514353301,
362454.995959508290980,
362465.540642835781910,
362476.085352496011183,
362486.630088488280308,
362497.174850812007207,
362507.719639466376975,
362518.264454450749326,
362528.809295764425769,
362539.354163406707812,
362549.899057376838755,
362560.443977674294729,
362570.988924298260827,
362581.533897248038556,
362592.078896522871219,
362602.623922122293152,
362613.168974045373034,
362623.714052291586995,
362634.259156860120129,
362644.804287750390358,
362655.349444961582776,
362665.894628493057098,
362676.439838344173040,
362686.985074514173903,
362697.530337002419401,
362708.075625808152836,
362718.620940930675715,
362729.166282369464170,
362739.711650123528671,
362750.257044192461763,
362760.802464575448539,
362771.347911271732301,
362781.893384280789178,
362792.438883601746056,
362802.984409234020859,
362813.529961176856887,
362824.075539429672062,
362834.621143991593271,
362845.166774862154853,
362855.712432040483691,
362866.258115525939502,
362876.803825317882001,
362887.349561415554490,
362897.895323818258476,
362908.441112525295466,
362918.986927536083385,
362929.532768849807326,
362940.078636465827003,
362950.624530383502133,
362961.170450602076016,
362971.716397120850161,
362982.262369939126074,
362992.808369056263473,
363003.354394471505657,
363013.900446184212342,
363024.446524193685036,
363034.992628499283455,
363045.538759100192692,
363056.084915995830670,
363066.631099185440689,
363077.177308668266051,
363087.723544443782885,
363098.269806511292700,
363108.816094869922381,
363119.362409519148059,
363129.908750458154827,
363140.455117686418816,
363151.001511203008704,
363161.547931007458828,
363172.094377099012490,
363182.640849476912990,
363193.187348140461836,
363203.733873089076951,
363214.280424322001636,
363224.827001838595606,
363235.373605638102163,
363245.920235719822813,
363256.466892083117273,
363267.013574727228843,
363277.560283651575446,
363288.107018855342176,
363298.653780337946955,
363309.200568098691292,
363319.747382136818487,
363330.294222451630048,
363340.841089042485692,
363351.387981908745132,
363361.934901049535256,
363372.481846464332193,
363383.028818152437452,
363393.575816113094334,
363404.122840345604345,
363414.669890849385411,
363425.216967623564415,
363435.764070667675696,
363446.311199980904348,
363456.858355562493671,
363467.405537411919795,
363477.952745528309606,
363488.499979911081027,
363499.047240559593774,
363509.594527473032940,
363520.141840650816448,
363530.689180092129391,
363541.236545796506107,
363551.783937763015274,
363562.331355991016608,
363572.878800479928032,
363583.426271229051054,
363593.973768237570766,
363604.521291504846886,
363615.068841030239128,
363625.616416813107207,
363636.164018852578010,
363646.711647148127668,
363657.259301698999479,
363667.806982504494954,
363678.354689563973807,
363688.902422876737546,
363699.450182442029472,
363709.997968259267509,
363720.545780327578541,
363731.093618646496907,
363741.641483215265907,
363752.189374033128843,
363762.737291099445429,
363773.285234413458966,
363783.833203974587377,
363794.381199782073963,
363804.929221835220233,
363815.477270133444108,
363826.025344675872475,
363836.573445461981464,
363847.121572491014376,
363857.669725762272719,
363868.217905275116209,
363878.766111028846353,
363889.314343022648245,
363899.862601256056223,
363910.410885728197172,
363920.959196438489016,
363931.507533386175055,
363942.055896570556797,
363952.604285991052166,
363963.152701646962669,
363973.701143537356984,
363984.249611661885865,
363994.798106019676197,
364005.346626610029489,
364015.895173432305455,
364026.443746485805605,
364036.992345769889653,
364047.540971283800900,
364058.089623026899062,
364068.638300998427439,
364079.187005197745748,
364089.735735624213703,
364100.284492277074605,
364110.833275155629963,
364121.382084259239491,
364131.930919587146491,
364142.479781138827093,
364153.028668913408183,
364163.577582910249475,
364174.126523128768895,
364184.675489568151534,
364195.224482227815315,
364205.773501106945332,
364216.322546204901300,
364226.871617521101143,
364237.420715054729953,
364247.969838805147447,
364258.518988771713339,
364269.068164953670930,
364279.617367350263521,
364290.166595960967243,
364300.715850785025395,
364311.265131821739487,
364321.814439070411026,
364332.363772530399729,
364342.913132200948894,
364353.462518081476446,
364364.011930171167478,
364374.561368469498120,
364385.110832975595258,
364395.660323688876815,
364406.209840608702507,
364416.759383734199218,
364427.308953064843081,
364437.858548599993810,
364448.408170338778291,
364458.957818280672655,
364469.507492424920201,
364480.057192770764232,
364490.606919317680877,
364501.156672064913437,
364511.706451011705212,
364522.256256157415919,
364532.806087501463480,
364543.355945042974781,
364553.905828781367745,
364564.455738715943880,
364575.005674846004695,
364585.555637170909904,
364596.105625689961016,
364606.655640402459539,
364617.205681307648774,
364627.755748405004852,
364638.305841693596449,
364648.855961173016112,
364659.406106842332520,
364669.956278700963594,
364680.506476748327259,
364691.056700983608607,
364701.606951406167354,
364712.157228015246801,
364722.707530810323078,
364733.257859790523071,
364743.808214955206495,
364754.358596303791273,
364764.909003835520707,
364775.459437549696304,
364786.009897445735987,
364796.560383522766642,
364807.110895780264400,
364817.661434217414353,
364828.211998833692633,
364838.762589628284331,
364849.313206600607373,
364859.863849749788642,
364870.414519075362477,
364880.965214576455764,
364891.515936252602842,
364902.066684102872387,
364912.617458126740530,
364923.168258323566988,
364933.719084692478646,
364944.269937232893426,
364954.820815944171045,
364965.371720825554803,
364975.922651876346208,
364986.473609095963184,
364997.024592483707238,
365007.575602038763463,
365018.126637760491576,
365028.677699648309499,
365039.228787701460533,
365049.779901919246186,
365060.331042301026173,
365070.882208846102003,
365081.433401553716976,
365091.984620423289016,
365102.535865454061422,
365113.087136645393912,
365123.638433996587992,
365134.189757507003378,
365144.741107175883371,
365155.292483002645895,
365165.843884986476041,
365176.395313126675319,
365186.946767422719859,
365197.498247873794753,
365208.049754479259718,
365218.601287238416262,
365229.152846150682308,
365239.704431215242948,
365250.256042431399692,
365260.807679798570462,
365271.359343315998558,
365281.911032983101904,
365292.462748799065594,
365303.014490763249341,
365313.566258875012863,
365324.118053133599460,
365334.669873538368847,
365345.221720088680740,
365355.773592783836648,
365366.325491623079870,
365376.877416605770122,
365387.429367731208913,
365397.981344998814166,
365408.533348407770973,
365419.085377957439050,
365429.637433647119906,
365440.189515476173256,
365450.741623443842400,
365461.293757549603470,
365471.845917792525142,
365482.398104172141757,
365492.950316687696613,
365503.502555338549428,
365514.054820123827085,
365524.607111043122131,
365535.159428095561452,
365545.711771280562971,
365556.264140597311780,
365566.816536045284010,
365577.368957623664755,
365587.921405331871938,
365598.473879169207066,
365609.026379134913441,
365619.578905228408985,
365630.131457448936999,
365640.684035795857199,
365651.236640268471092,
365661.789270866080187,
365672.341927587985992,
365682.894610433548223,
365693.447319402068388,
365704.000054492847994,
365714.552815705246758,
365725.105603038566187,
365735.658416492107790,
365746.211256065231282,
365756.764121757179964,
365767.317013567313552,
365777.869931494991761,
365788.422875539516099,
365798.975845700071659,
365809.528841976134572,
365820.081864366948139,
365830.634912871872075,
365841.187987490207888,
365851.741088221315295,
365862.294215064437594,
365872.847368018876296,
365883.400547084049322,
365893.953752259199973,
365904.506983543687966,
365915.060240936814807,
365925.613524437940214,
365936.166834046365693,
365946.720169761276338,
365957.273531582090072,
365967.826919508224819,
365978.380333538807463,
365988.933773673314136,
365999.487239911046345,
366010.040732251247391,
366020.594250693218783,
366031.147795236436650,
366041.701365880086087,
366052.254962623468600,
366062.808585466002114,
366073.362234406871721,
366083.915909445553552,
366094.469610581290908,
366105.023337813385297,
366115.577091141138226,
366126.130870563967619,
366136.684676081058569,
366147.238507691829000,
366157.792365395580418,
366168.346249191556126,
366178.900159079232253,
366189.454095057793893,
366200.008057126600761,
366210.562045284954365,
366221.116059532214422,
366231.670099867682438,
366242.224166290718131,
366252.778258800506592,
366263.332377396465745,
366273.886522078013513,
366284.440692844218574,
366294.994889694615267,
366305.549112628505100,
366316.103361645073164,
366326.657636743737385,
366337.211937923799269,
366347.766265184676740,
366358.320618525438476,
366368.874997945677023,
366379.429403444577474,
366389.983835021441337,
366400.538292675570119,
366411.092776406439953,
366421.647286213235930,
366432.201822095317766,
366442.756384051986970,
366453.310972082603257,
366463.865586186409928,
366474.420226362824906,
366484.974892611091491,
366495.529584930627607,
366506.084303320618346,
366516.639047780481633,
366527.193818309460767,
366537.748614906915464,
366548.303437572205439,
366558.858286304573994,
366569.413161103439052,
366579.968061968043912,
366590.522988897690084,
366601.077941891853698,
366611.632920949661639,
366622.187926070590038,
366632.742957253823988,
366643.298014498723205,
366653.853097804647405,
366664.408207170898095,
366674.963342596776783,
366685.518504081701394,
366696.073691624857020,
366706.628905225719791,
366717.184144883416593,
366727.739410597423557,
366738.294702366983984,
366748.850020191399381,
366759.405364070145879,
366769.960734002350364,
366780.516129987372551,
366791.071552024630364,
366801.627000113425311,
366812.182474253000692,
366822.737974442716222,
366833.293500681931619,
366843.849052969948389,
366854.404631306068040,
366864.960235689533874,
366875.515866119938437,
366886.071522596292198,
366896.627205118071288,
366907.182913684519008,
366917.738648295053281,
366928.294408948975615,
366938.850195645529311,
366949.406008384132292,
366959.961847164027859,
366970.517711984517518,
366981.073602845077403,
366991.629519744892605,
367002.185462683322839,
367012.741431659669615,
367023.297426673292648,
367033.853447723493446,
367044.409494809631724,
367054.965567930950783,
367065.521667086810339,
367076.077792276570108,
367086.633943499589805,
367097.190120755054522,
367107.746324042265769,
367118.302553360816091,
367128.858808709715959,
367139.415090088441502,
367149.971397496352438,
367160.527730932692066,
367171.084090396761894,
367181.640475887921639,
367192.196887405531015,
367202.753324948891532,
367213.309788517246488,
367223.866278110130224,
367234.422793726611417,
367244.979335366166197,
367255.535903028037865,
367266.092496711586136,
367276.649116416112520,
367287.205762141034938,
367297.762433885596693,
367308.319131649041083,
367318.875855430844240,
367329.432605230249465,
367339.989381046500057,
367350.546182879188564,
367361.103010727325454,
367371.659864590386860,
367382.216744467732497,
367392.773650358605664,
367403.330582262307871,
367413.887540178257041,
367424.444524105754681,
367435.001534044044092,
367445.558569992543198,
367456.115631950495299,
367466.672719917260110,
367477.229833892197348,
367487.786973874608520,
367498.344139863795135,
367508.901331859116908,
367519.458549859875347,
367530.015793865313753,
367540.573063874850050,
367551.130359887843952,
367561.687681903596967,
367572.245029921410605,
367582.802403940586373,
367593.359803960367572,
367603.917229980288539,
367614.474681999476161,
367625.032160017348360,
367635.589664033264853,
367646.147194046468940,
367656.704750056320336,
367667.262332062120549,
367677.819940063287504,
367688.377574059006292,
367698.935234048694838,
367709.492920031712856,
367720.050632007187232,
367730.608369974652305,
367741.166133933351375,
367751.723923882644158,
367762.281739821832161,
367772.839581750216894,
367783.397449667099863,
367793.955343571898993,
367804.513263463857584,
367815.071209342277143,
367825.629181206692010,
367836.187179056170862,
367846.745202890131623,
367857.303252707875799,
367867.861328508763108,
367878.419430292211473,
367888.977558057405986,
367899.535711803648155,
367910.093891530414112,
367920.652097236830741,
367931.210328922432382,
367941.768586586345918,
367952.326870228105690,
367962.885179846896790,
367973.443515442020725,
367984.001877012895420,
367994.560264558822382,
368005.118678079044912,
368015.677117573039141,
368026.235583040048368,
368036.794074479374103,
368047.352591890317854,
368057.911135272297543,
368068.469704624614678,
368079.028299946512561,
368089.586921237409115,
368100.145568496605847,
368110.704241723404266,
368121.262940917105880,
368131.821666077186819,
368142.380417202773970,
368152.939194293343462,
368163.497997348138597,
368174.056826366460882,
368184.615681347728241,
368195.174562291242182,
368205.733469196304213,
368216.292402062215842,
368226.851360888278577,
368237.410345673968550,
368247.969356418470852,
368258.528393121145200,
368269.087455781351309,
368279.646544398332480,
368290.205658971564844,
368300.764799500175286,
368311.323965983639937,
368321.883158421318512,
368332.442376812396105,
368343.001621156232432,
368353.560891452245414,
368364.120187699678354,
368374.679509897949174,
368385.238858046184760,
368395.798232143977657,
368406.357632190396544,
368416.917058184975758,
368427.476510126958601,
368438.035988015646581,
368448.595491850399412,
368459.155021630518604,
368469.714577355363872,
368480.274159024294931,
368490.833766636613291,
368501.393400191562250,
368511.953059688501526,
368522.512745126790833,
368533.072456505789887,
368543.632193824800197,
368554.191957083123270,
368564.751746280118823,
368575.311561415030155,
368585.871402487333398,
368596.431269496213645,
368606.991162441088818,
368617.551081321202219,
368628.111026136029977,
368638.670996884757187,
368649.230993566685356,
368659.791016181348823,
368670.351064727874473,
368680.911139205622021,
368691.471239614009392,
368702.031365952279884,
368712.591518219793215,
368723.151696415850893,
368733.711900539870840,
368744.272130591096357,
368754.832386568770744,
368765.392668472428340,
368775.952976301312447,
368786.513310054666363,
368797.073669731908012,
368807.634055332338903,
368818.194466855260544,
368828.754904300032649,
368839.315367666014936,
368849.875856952508911,
368860.436372158757877,
368870.996913284179755,
368881.557480328192469,
368892.118073289922904,
368902.678692168788984,
368913.239336964150425,
368923.800007675308734,
368934.360704301623628,
368944.921426842338406,
368955.482175296870992,
368966.042949664522894,
368976.603749944595620,
368987.164576136448886,
368997.725428239384200,
369008.286306252819486,
369018.847210175939836,
369029.408140008163173,
369039.969095748732798,
369050.530077397241257,
369061.091084952699021,
369071.652118414524011,
369082.213177782075945,
369092.774263054714538,
369103.335374231799506,
369113.896511312515941,
369124.457674296281766,
369135.018863182456698,
369145.580077970342245,
369156.141318659239914,
369166.702585248451214,
369177.263877737394068,
369187.825196125369985,
369198.386540411738679,
369208.947910595685244,
369219.509306676685810,
369230.070728654041886,
369240.632176527054980,
369251.193650295026600,
369261.755149957374670,
369272.316675513342489,
369282.878226962289773,
369293.439804303576238,
369304.001407536503393,
369314.563036660489161,
369325.124691674602218,
369335.686372578493319,
369346.248079371289350,
369356.809812052350026,
369367.371570621035062,
369377.933355076704174,
369388.495165418658871,
369399.057001646200661,
369409.618863758689258,
369420.180751755484380,
369430.742665635829326,
369441.304605399142019,
369451.866571044665761,
369462.428562571818475,
369472.990579979901668,
369483.552623268275056,
369494.114692436123732,
369504.676787482982036,
369515.238908408035059,
369525.801055210642517,
369536.363227890222333,
369546.925426445959602,
369557.487650877272245,
369568.049901183578186,
369578.612177364062518,
369589.174479418084957,
369599.736807344947010,
369610.299161144066602,
369620.861540814745240,
369631.423946356284432,
369641.986377768043894,
369652.548835049325135,
369663.111318199487869,
369673.673827217833605,
369684.236362103722058,
369694.798922856571153,
369705.361509475449566,
369715.924121959949844,
369726.486760309315287,
369737.049424522847403,
369747.612114599847700,
369758.174830539734103,
369768.737572341808118,
369779.300340005371254,
369789.863133529783227,
369800.425952914461959,
369810.988798158534337,
369821.551669261476491,
369832.114566222648136,
369842.677489041176159,
369853.240437716653105,
369863.803412248264067,
369874.366412635310553,
369884.929438877268694,
369895.492490973323584,
369906.055568922951352,
369916.618672725337092,
369927.181802379840519,
369937.744957885879558,
369948.308139242697507,
369958.871346449654084,
369969.434579506109003,
369979.997838411421981,
369990.561123164836317,
370001.124433765711728,
370011.687770213407930,
370022.251132507226430,
370032.814520646585152,
370043.377934630669188,
370053.941374458896462,
370064.504840130684897,
370075.068331645161379,
370085.631849001743831,
370096.195392199850176,
370106.758961238781922,
370117.322556117782369,
370127.886176836269442,
370138.449823393486440,
370149.013495788909495,
370159.577194021781906,
370170.140918091346975,
370180.704667997139040,
370191.268443738343194,
370201.832245314319152,
370212.396072724484839,
370222.959925968025345,
370233.523805044416804,
370244.087709952844307,
370254.651640692783985,
370265.215597263537347,
370275.779579664289486,
370286.343587894574739,
370296.907621953578200,
370307.471681840834208,
370318.035767555411439,
370328.599879096844234,
370339.164016464317683,
370349.728179657249711,
370360.292368675000034,
370370.856583516870160,
370381.420824182161596,
370391.985090670175850,
370402.549382980330847,
370413.113701111928094,
370423.678045064327307,
370434.242414836829994,
370444.806810428795870,
370455.371231839584652,
370465.935679068381432,
370476.500152114662342,
370487.064650977728888,
370497.629175656940788,
370508.193726151541341,
370518.758302460948471,
370529.322904584463686,
370539.887532521446701,
370550.452186271199025,
370561.016865833022166,
370571.581571206334047,
370582.146302390436176,
370592.711059384571854,
370603.275842188275419,
370613.840650800731964,
370624.405485221242998,
370634.970345449226443,
370645.535231483983807,
370656.100143324933015,
370666.665080971259158,
370677.230044422380161,
370687.795033677713946,
370698.360048736387398,
370708.925089597876649,
370719.490156261541415,
370730.055248726624995,
370740.620366992428899,
370751.185511058487464,
370761.750680923985783,
370772.315876588225365,
370782.881098050624132,
370793.446345310425386,
370804.011618367105257,
370814.576917219907045,
370825.142241868132260,
370835.707592311198823,
370846.272968548408244,
370856.838370579062030,
370867.403798402578104,
370877.969252018199768,
370888.534731425228529,
370899.100236623140518,
370909.665767611120827,
370920.231324388703797,
370930.796906955016311,
370941.362515309534501,
370951.928149451501667,
370962.493809380277526,
370973.059495095279999,
370983.625206595752388,
370994.190943880996201,
371004.756706950429361,
371015.322495803353377,
371025.888310439127963,
371036.454150857054628,
371047.020017056434881,
371057.585909036744852,
371068.151826797169633,
371078.717770337068941,
371089.283739655860700,
371099.849734752846416,
371110.415755627269391,
371120.981802278605755,
371131.547874706157017,
371142.113972909166478,
371152.680096886993852,
371163.246246639173478,
371173.812422164715827,
371184.378623463213444,
371194.944850533909630,
371205.511103376105893,
371216.077381989220157,
371226.643686372553930,
371237.210016525350511,
371247.776372447027825,
371258.342754136945587,
371268.909161594405305,
371279.475594818824902,
371290.042053809389472,
371300.608538565516938,
371311.175049086567014,
371321.741585371841211,
371332.308147420641035,
371342.874735232384410,
371353.441348806431051,
371364.007988141907845,
371374.574653238407336,
371385.141344095114619,
371395.708060711447615,
371406.274803086649626,
371416.841571220196784,
371427.408365111274179,
371437.975184759299736,
371448.542030163574964,
371459.108901323517784,
371469.675798238313291,
371480.242720907379407,
371490.809669330134057,
371501.376643505762331,
371511.943643433798570,
371522.510669113369659,
371533.077720543893520,
371543.644797724788077,
371554.211900655238423,
371564.779029334662482,
371575.346183762478177,
371585.913363937870599,
371596.480569860315882,
371607.047801529057324,
371617.615058943396434,
371628.182342102867551,
371638.749651006597560,
371649.316985653946176,
371659.884346044331323,
371670.451732177054510,
371681.019144051475450,
371691.586581666895654,
371702.154045022733044,
371712.721534118172713,
371723.289048952690791,
371733.856589525646996,
371744.424155836226419,
371754.991747883846983,
371765.559365667926613,
371776.127009187650401,
371786.694678442378063,
371797.262373431643937,
371807.830094154516701,
371818.397840610530693,
371828.965612798987422,
371839.533410719130188,
371850.101234370376915,
371860.669083752029110,
371871.236958863446489,
371881.804859703988768,
371892.372786272957455,
371902.940738569712266,
371913.508716593554709,
371924.076720343844499,
371934.644749819883145,
371945.212805021146778,
371955.780885946820490,
371966.348992596263997,
371976.917124968837015,
371987.485283064015675,
371998.053466880926862,
372008.621676418988500,
372019.189911677502096,
372029.758172656060196,
372040.326459353615064,
372050.894771769701038,
372061.463109903619625,
372072.031473754730541,
372082.599863322393503,
372093.168278605910018,
372103.736719604639802,
372114.305186317942571,
372124.873678745003417,
372135.442196885414887,
372146.010740738303866,
372156.579310303204693,
372167.147905579244252,
372177.716526565840468,
372188.285173262411263,
372198.853845668199938,
372209.422543782624416,
372219.991267604927998,
372230.560017134586815,
372241.128792370844167,
372251.697593313001562,
372262.266419960476924,
372272.835272312513553,
372283.404150368587580,
372293.973054127942305,
372304.541983589937445,
372315.110938753932714,
372325.679919619229622,
372336.248926185187884,
372346.817958451225422,
372357.387016416527331,
372367.956100080511533,
372378.525209442537744,
372389.094344501907472,
372399.663505257980432,
372410.232691710174549,
372420.801903857733123,
372431.371141699957661,
372441.940405236266088,
372452.509694465959910,
372463.079009388398845,
372473.648350002942607,
372484.217716308892705,
372494.787108305550646,
372505.356525992334355,
372515.925969368603546,
372526.495438433601521,
372537.064933186746202,
372547.634453627339099,
372558.203999754739925,
372568.773571568308398,
372579.343169067346025,
372589.912792251154315,
372600.482441119209398,
372611.052115670754574,
372621.621815905091353,
372632.191541821579449,
372642.761293419694994,
372653.331070698623080,
372663.900873657781631,
372674.470702296413947,
372685.040556613996159,
372695.610436609713361,
372706.180342283099890,
372716.750273633399047,
372727.320230659912340,
372737.890213361999486,
372748.460221738961991,
372759.030255790275987,
372769.600315515184775,
372780.170400913106278,
372790.740511983225588,
372801.310648725018837,
372811.880811137845740,
372822.450999220891390,
372833.021212973631918,
372843.591452395368833,
372854.161717485403642,
372864.732008243212476,
372875.302324667980429,
372885.872666759125423,
372896.443034515948966,
372907.013427937868983,
372917.583847024070565,
372928.154291774088051,
372938.724762187164742,
372949.295258262660354,
372959.865779999876395,
372970.436327398230787,
372981.006900457025040,
372991.577499175560661,
373002.148123553255573,
373012.718773589411285,
373023.289449283329304,
373033.860150634369347,
373044.430877641891129,
373055.001630305312574,
373065.572408623818774,
373076.143212596885860,
373086.714042223757133,
373097.284897503908724,
373107.855778436583932,
373118.426685021084268,
373128.997617256769445,
373139.568575143115595,
373150.139558679307811,
373160.710567864764016,
373171.281602698843926,
373181.852663180849049,
373192.423749310080893,
373202.994861085957382,
373213.565998507780023,
373224.137161574908532,
373234.708350286702625,
373245.279564642405603,
373255.850804641435388,
373266.422070283209905,
373276.993361566972453,
373287.564678492082749,
373298.136021057900507,
373308.707389263785444,
373319.278783108980861,
373329.850202592962887,
373340.421647714916617,
373350.993118474434596,
373361.564614870585501,
373372.136136902845465,
373382.707684570632409,
373393.279257873131428,
373403.850856809702236,
373414.422481379762758,
373424.994131582672708,
373435.565807417733595,
373446.137508884305134,
373456.709235981688835,
373467.280988709244411,
373477.852767066389788,
373488.424571052251849,
373498.996400666481350,
373509.568255908205174,
373520.140136776841246,
373530.712043271749280,
373541.283975392230786,
373551.855933137587272,
373562.427916507294867,
373572.999925500538666,
373583.571960116736591,
373594.144020355248358,
373604.716106215433683,
373615.288217696594074,
373625.860354798089247,
373636.432517519220710,
373647.004705859406386,
373657.576919817947783,
373668.149159394204617,
373678.721424587594811,
373689.293715397303458,
373699.866031822806690,
373710.438373863289598,
373721.010741518228315,
373731.583134786924347,
373742.155553668795619,
373752.727998163085431,
373763.300468269153498,
373773.872963986417744,
373784.445485314121470,
373795.018032251740806,
373805.590604798402637,
373816.163202953757718,
373826.735826716874726,
373837.308476087171584,
373847.881151064066216,
373858.453851646860130,
373869.026577834913041,
373879.599329627584666,
373890.172107024118304,
373900.744910023990087,
373911.317738626443315,
373921.890592830895912,
373932.463472636649385,
373943.036378043005243,
373953.609309049381409,
373964.182265655195806,
373974.755247859575320,
373985.328255661996081,
373995.901289061876014,
374006.474348058400210,
374017.047432651044801,
374027.620542839111295,
374038.193678621901199,
374048.766839998890646,
374059.340026969206519,
374069.913239532441366,
374080.486477687663864,
374091.059741434466559,
374101.633030772034544,
374112.206345699785743,
374122.779686217079870,
374133.353052323276643,
374143.926444017619360,
374154.499861299525946,
374165.073304168356117,
374175.646772623411380,
374186.220266664051451,
374196.793786289636046,
374207.367331499524880,
374217.940902293077670,
374228.514498669537716,
374239.088120628322940,
374249.661768168793060,
374260.235441290191375,
374270.809139992052224,
374281.382864273502491,
374291.956614134076517,
374302.530389573017601,
374313.104190589743666,
374323.678017183498014,
374334.251869353698567,
374344.825747099646833,
374355.399650420760736,
374365.973579316283576,
374376.547533785691485,
374387.121513828285970,
374397.695519443310332,
374408.269550630182493,
374418.843607388203964,
374429.417689716909081,
374439.991797615366522,
374450.565931083110627,
374461.140090119501110,
374471.714274723723065,
374482.288484895252623,
374492.862720633391291,
374503.436981937498786,
374514.011268806934822,
374524.585581241000909,
374535.159919239114970,
374545.734282800636720,
374556.308671924751252,
374566.883086610934697,
374577.457526858546771,
374588.031992666830774,
374598.606484035262838,
374609.181000963144470,
374619.755543449718971,
374630.330111494520679,
374640.904705096676480,
374651.479324255720712,
374662.053968970954884,
374672.628639241622295,
374683.203335067199077,
374693.778056446986739,
374704.352803380286787,
374714.927575866575353,
374725.502373905095737,
374736.077197495149449,
374746.652046636154409,
374757.226921327470336,
374767.801821568456944,
374778.376747358415741,
374788.951698696648236,
374799.526675582572352,
374810.101678015547805,
374820.676705994876102,
374831.251759519975167,
374841.826838590088300,
374852.401943204575218,
374862.977073362912051,
374873.552229064400308,
374884.127410308225080,
374894.702617093920708,
374905.277849420788698,
374915.853107288130559,
374926.428390695306007,
374937.003699641674757,
374947.579034126596525,
374958.154394149372820,
374968.729779709421564,
374979.305190806102473,
374989.880627438658848,
375000.456089606566820,
375011.031577309011482,
375021.607090545468964,
375032.182629315357190,
375042.758193617803045,
375053.333783452340867,
375063.909398818155751,
375074.485039714782033,
375085.060706141463015,
375095.636398097500205,
375106.212115582311526,
375116.787858595314901,
375127.363627135695424,
375137.939421202929225,
375148.515240796259604,
375159.091085915162694,
375169.666956558940001,
375180.242852726893034,
375190.818774418381508,
375201.394721632823348,
375211.970694369461853,
375222.546692627714947,
375233.122716406884138,
375243.698765706329141,
375254.274840525526088,
375264.850940863601863,
375275.427066720090806,
375286.003218094236217,
375296.579394985456020,
375307.155597392993513,
375317.731825316324830,
375328.308078754693270,
375338.884357707516756,
375349.460662174213212,
375360.036992153967731,
375370.613347646198235,
375381.189728650264442,
375391.766135165467858,
375402.342567191226408,
375412.919024726958014,
375423.495507771847770,
375434.072016325255390,
375444.648550386657007,
375455.225109955295920,
375465.801695030590054,
375476.378305611899123,
375486.954941698466428,
375497.531603289768100,
375508.108290385047439,
375518.685002983722370,
375529.261741085094400,
375539.838504688523244,
375550.415293793426827,
375560.992108399106655,
375571.568948504806031,
375582.145814110001083,
375592.722705214109737,
375603.299621816317085,
375613.876563916099258,
375624.453531512641348,
375635.030524605535902,
375645.607543193909805,
375656.184587277355604,
375666.761656854941975,
375677.338751926145051,
375687.915872490324546,
375698.493018546840176,
375709.070190095051657,
375719.647387134260498,
375730.224609663768206,
375740.801857683109120,
375751.379131191468332,
375761.956430188263766,
375772.533754672796931,
375783.111104644427542,
375793.688480102573521,
375804.265881046652794,
375814.843307475792244,
375825.420759389468003,
375835.998236787039787,
375846.575739667809103,
375857.153268031135667,
375867.730821876437403,
375878.308401202957612,
375888.886006010172423,
375899.463636297383346,
375910.041292063891888,
375920.618973308999557,
375931.196680032240693,
375941.774412232800387,
375952.352169910212979,
375962.929953063605353,
375973.507761692395434,
375984.085595796059351,
375994.663455373782199,
376005.241340425098315,
376015.819250949192792,
376026.397186945483554,
376036.975148413272109,
376047.553135352034587,
376058.131147760956082,
376068.709185639512725,
376079.287248987064231,
376089.865337802853901,
376100.443452086357865,
376111.021591836819425,
376121.599757053598296,
376132.177947736112401,
376142.756163883721456,
376153.334405495726969,
376163.912672571372241,
376174.490965110308025,
376185.069283111544792,
376195.647626574675087,
376206.225995499000419,
376216.804389883822296,
376227.382809728442226,
376237.961255032336339,
376248.539725794806145,
376259.118222015211359,
376269.696743692911696,
376280.275290827150457,
376290.853863417461980,
376301.432461463089567,
376312.011084963334724,
376322.589733917731792,
376333.168408325407654,
376343.747108185838442,
376354.325833498383872,
376364.904584262461867,
376375.483360477199312,
376386.062162142130546,
376396.640989256615285,
376407.219841819955036,
376417.798719831393100,
376428.377623290522024,
376438.956552196468692,
376449.535506548709236,
376460.114486346545164,
376470.693491589394398,
376481.272522276500240,
376491.851578407338820,
376502.430659981211647,
376513.009766997362021,
376523.588899455382489,
376534.168057354458142,
376544.747240693948697,
376555.326449473155662,
376565.905683691613376,
376576.484943348506931,
376587.064228443312459,
376597.643538975273259,
376608.222874943807255,
376618.802236348215956,
376629.381623187859077,
376639.961035462154541,
376650.540473170462064,
376661.119936311966740,
376671.699424886202905,
376682.278938892646693,
376692.858478330250364,
376703.438043198664673,
376714.017633497132920,
376724.597249225073028,
376735.176890381786507,
376745.756556966633070,
376756.336248979030643,
376766.915966418222524,
376777.495709283626638,
376788.075477574660908,
376798.655271290510427,
376809.235090430709533,
376819.814934994501527,
376830.394804981304333,
376840.974700390361249,
376851.554621221148409,
376862.134567472967319,
376872.714539145119488,
376883.294536237022839,
376893.874558748153504,
376904.454606677580159,
376915.034680024895351,
376925.614778789284173,
376936.194902970339172,
376946.775052567129023,
376957.355227579188067,
376967.935428005876020,
376978.515653846436180,
376989.095905100286473,
376999.676181766786613,
377010.256483845238108,
377020.836811335000675,
377031.417164235550445,
377041.997542546130717,
377052.577946266101208,
377063.158375394879840,
377073.738829931709915,
377084.319309876067564,
377094.899815227254294,
377105.480345984629821,
377116.060902147495653,
377126.641483715269715,
377137.222090687253512,
377147.802723062923178,
377158.383380841463804,
377168.964064022351522,
377179.544772604887839,
377190.125506588432472,
377200.706265972403344,
377211.287050756043755,
377221.867860938771628,
377232.448696519946679,
377243.029557498986833,
377253.610443875135388,
377264.191355647752061,
377274.772292816254776,
377285.353255379945040,
377295.934243338182569,
377306.515256690385286,
377317.096295435912907,
377327.677359573950525,
377338.258449104032479,
377348.839564025460277,
377359.420704337593634,
377370.001870039792266,
377380.583061131474096,
377391.164277611766011,
377401.745519480260555,
377412.326786736259237,
377422.908079379063565,
377433.489397408033255,
377444.070740822528023,
377454.652109622024000,
377465.233503805648070,
377475.814923372934572,
377486.396368323185015,
377496.977838655759115,
377507.559334370016586,
377518.140855465317145,
377528.722401941020507,
377539.303973796428181,
377549.885571030958090,
377560.467193643969949,
377571.048841634707060,
377581.630515002645552,
377592.212213747086935,
377602.793937867449131,
377613.375687363091856,
377623.957462233258411,
377634.539262477424927,
377645.121088094834704,
377655.702939085022081,
377666.284815447113942,
377676.866717180644628,
377687.448644284799229,
377698.030596759170294,
377708.612574602826498,
377719.194577815360390,
377729.776606396131683,
377740.358660344325472,
377750.940739659417886,
377761.522844340710435,
377772.104974387621041,
377782.687129799451213,
377793.269310575618874,
377803.851516715425532,
377814.433748218172695,
377825.016005083336495,
377835.598287310160231,
377846.180594898120034,
377856.762927846517414,
377867.345286154712085,
377877.927669821947347,
377888.510078847873956,
377899.092513231502380,
377909.674972972425167,
377920.257458070002031,
377930.839968523359857,
377941.422504332091194,
377952.005065495497547,
377962.587652012822218,
377973.170263883599546,
377983.752901107014623,
377994.335563682485372,
378004.918251609487925,
378015.500964887265582,
378026.083703515178058,
378036.666467492585070,
378047.249256818846334,
378057.832071493379772,
378068.414911515428685,
378078.997776884410996,
378089.580667599744629,
378100.163583660731092,
378110.746525066671893,
378121.329491816926748,
378131.912483911029994,
378142.495501348166727,
378153.078544127696659,
378163.661612249095924,
378174.244705711607821,
378184.827824514708482,
378195.410968657641206,
378205.994138139765710,
378216.577332960499916,
378227.160553119145334,
378237.743798615119886,
378248.327069447783288,
378258.910365616437048,
378269.493687120440882,
378280.077033959212713,
378290.660406132112257,
378301.243803638382815,
378311.827226477558725,
378322.410674648883287,
378332.994148151658010,
378343.577646985359024,
378354.161171149287838,
378364.744720642862376,
378375.328295465384144,
378385.911895616154652,
378396.495521094708238,
378407.079171900229994,
378417.662848032137845,
378428.246549489791505,
378438.830276272608899,
378449.414028379949741,
378459.997805810999125,
378470.581608565291390,
378481.165436642128043,
378491.749290040810592,
378502.333168760873377,
378512.917072801443283,
378523.501002162054647,
378534.084956841950770,
378544.668936840549577,
378555.252942157268990,
378565.836972791410517,
378576.421028742333874,
378587.005110009340569,
378597.589216591848526,
378608.173348489217460,
378618.757505700807087,
378629.341688225977123,
378639.925896064029075,
378650.510129214439075,
378661.094387676450424,
378671.678671449481044,
378682.262980532832444,
378692.847314925980754,
378703.431674628227483,
378714.016059638932347,
378724.600469957396854,
378735.184905583038926,
378745.769366515218280,
378756.353852753236424,
378766.938364296511281,
378777.522901144460775,
378788.107463296328206,
378798.692050751415081,
378809.276663509313948,
378819.861301569209900,
378830.445964930579066,
378841.030653592722956,
378851.615367554884870,
378862.200106816599146,
378872.784871377109084,
378883.369661235774402,
378893.954476392071228,
378904.539316845301073,
378915.124182594765443,
378925.709073639940470,
378936.293989980011247,
378946.878931614570320,
378957.463898542860989,
378968.048890764126554,
378978.633908277901355,
378989.218951083486900,
378999.804019180184696,
379010.389112567412667,
379020.974231244588736,
379031.559375210897997,
379042.144544465932995,
379052.729739008820616,
379063.314958839095198,
379073.900203956116457,
379084.485474359127693,
379095.070770047605038,
379105.656091020791791,
379116.241437278105877,
379126.826808818848804,
379137.412205642613117,
379147.997627748467494,
379158.583075135888066,
379169.168547804292757,
379179.754045752983075,
379190.339568981318735,
379200.925117488659453,
379211.510691274423152,
379222.096290337911341,
379232.681914678483736,
379243.267564295558259,
379253.853239188378211,
379264.438939356419723,
379275.024664798984304,
379285.610415515489876,
379296.196191505237948,
379306.781992767588235,
379317.367819301958662,
379327.953671107767150,
379338.539548184198793,
379349.125450530729722,
379359.711378146661445,
379370.297331031470094,
379380.883309184282552,
379391.469312604749575,
379402.055341292056255,
379412.641395245620515,
379423.227474464743864,
379433.813578948844224,
379444.399708697339520,
379454.985863709531259,
379465.572043984720949,
379476.158249522326514,
379486.744480321649462,
379497.330736382165924,
379507.917017703235615,
379518.503324284043629,
379529.089656124124303,
379539.676013222837355,
379550.262395579426084,
379560.848803193424828,
379571.435236063960474,
379582.021694190741982,
379592.608177572779823,
379603.194686209550127,
379613.781220100470819,
379624.367779244843405,
379634.954363642085809,
379645.540973291557748,
379656.127608192560729,
379666.714268344512675,
379677.300953746715095,
379687.887664398644120,
379698.474400299601257,
379709.061161448829807,
379719.647947845922317,
379730.234759490063880,
379740.821596380672418,
379751.408458517107647,
379761.995345898729283,
379772.582258524897043,
379783.169196395028848,
379793.756159508426208,
379804.343147864507046,
379814.930161462572869,
379825.517200301983394,
379836.104264382156543,
379846.691353702393826,
379857.278468262113165,
379867.865608060674276,
379878.452773097320460,
379889.039963371586055,
379899.627178882830776,
379910.214419630181510,
379920.801685613289010,
379931.388976831396576,
379941.976293283922132,
379952.563634970050771,
379963.151001889375038,
379973.738394041080028,
379984.325811424583662,
379994.913254039362073,
380005.500721884600352,
380016.088214959774632,
380026.675733264244627,
380037.263276797311846,
380047.850845558394212,
380058.438439546909649,
380069.026058762101457,
380079.613703203329351,
380090.201372870011255,
380100.789067761565093,
380111.376787877234165,
380121.964533216436394,
380132.552303778589703,
380143.140099563053809,
380153.727920569072012,
380164.315766796178650,
380174.903638243675232,
380185.491534910805058,
380196.079456797044259,
380206.667403901694342,
380217.255376224231441,
380227.843373763957061,
380238.431396520230919,
380249.019444492296316,
380259.607517679745797,
380270.195616081822664,
380280.783739697886631,
380291.371888527355623,
380301.960062569589354,
380312.548261823831126,
380323.136486289557070,
380333.724735966126900,
380344.313010852900334,
380354.901310949178878,
380365.489636254380457,
380376.077986767922994,
380386.666362489049789,
380397.254763417178765,
380407.843189551786054,
380418.431640892056748,
380429.020117437466979,
380439.608619187376462,
380450.197146141028497,
380460.785698297957424,
380471.374275657464750,
380481.962878218793776,
380492.551505981537048,
380503.140158944821451,
380513.728837108239532,
380524.317540470976382,
380534.906269032566343,
380545.495022792194504,
380556.083801749336999,
380566.672605903295334,
380577.261435253545642,
380587.850289799273014,
380598.439169539953582,
380609.028074475005269,
380619.617004603729583,
380630.205959925486241,
380640.794940439576749,
380651.383946145593654,
380661.972977042663842,
380672.562033130205236,
380683.151114407693967,
380693.740220874315128,
380704.329352529603057,
380714.918509372859262,
380725.507691403385252,
380736.096898620715365,
380746.686131023976486,
380757.275388612819370,
380767.864671386370901,
380778.453979343990795,
380789.043312485329807,
380799.632670809398405,
380810.222054315730929,
380820.811463003803510,
380831.400896872743033,
380841.990355922142044,
380852.579840151127428,
380863.169349559349939,
380873.758884145936463,
380884.348443910363130,
380894.938028851989657,
380905.527638970117550,
380916.117274264222942,
380926.706934733607341,
380937.296620377572253,
380947.886331195593812,
380958.476067186973523,
380969.065828351129312,
380979.655614687420893,
380990.245426195149776,
381000.835262873733882,
381011.425124722649343,
381022.015011740964837,
381032.604923928331118,
381043.194861283933278,
381053.784823807247449,
381064.374811497633345,
381074.964824354450684,
381085.554862377000973,
381096.144925564702135,
381106.735013916972093,
381117.325127433054149,
381127.915266112424433,
381138.505429954384454,
381149.095618958352134,
381159.685833123570774,
381170.276072449632920,
381180.866336935665458,
381191.456626581202727,
381202.046941385546234,
381212.637281348055694,
381223.227646468265448,
381233.818036745244171,
381244.408452178467996,
381254.998892767413054,
381265.589358511380851,
381276.179849409672897,
381286.770365461765323,
381297.360906667017844,
381307.951473024673760,
381318.542064534267411,
381329.132681195042096,
381339.723323006415740,
381350.313989967689849,
381360.904682078340556,
381371.495399337669369,
381382.086141745036002,
381392.676909299858380,
381403.267702001438010,
381413.858519849251024,
381424.449362842540722,
381435.040230980725028,
381445.631124263280071,
381456.222042689332739,
381466.812986258475576,
381477.403954969893675,
381487.994948823121376,
381498.585967817401979,
381509.177011952153407,
381519.768081226793583,
381530.359175640682224,
381540.950295193004422,
381551.541439883410931,
381562.132609711086843,
381572.723804675450083,
381583.315024775860365,
381593.906270011619199,
381604.497540382202715,
381615.088835886970628,
381625.680156525224447,
381636.271502296440303,
381646.862873199803289,
381657.454269234905951,
381668.045690400991589,
381678.637136697361711,
381689.228608123492450,
381699.820104678743519,
381710.411626362474635,
381721.003173173987307,
381731.594745112699457,
381742.186342178029008,
381752.777964369335677,
381763.369611685862765,
381773.961284127202816,
381784.552981692482717,
381795.144704381236807,
381805.736452192766592,
381816.328225126431789,
381826.920023181592114,
381837.511846357723698,
381848.103694654069841,
381858.695568070048466,
381869.287466605077498,
381879.879390258516651,
381890.471339029551018,
381901.063312917773146,
381911.655311922484543,
381922.247336042986717,
381932.839385278755799,
381943.431459629093297,
381954.023559093417134,
381964.615683671028819,
381975.207833361346275,
381985.800008163671009,
381996.392208077537362,
382006.984433102130424,
382017.576683236926328,
382028.168958481226582,
382038.761258834449109,
382049.353584295953624,
382059.945934865099844,
382070.538310541247483,
382081.130711323814467,
382091.723137212102301,
382102.315588205645327,
382112.908064303570427,
382123.500565505353734,
382134.093091810413171,
382144.685643218050245,
382155.278219727682881,
382165.870821338670794,
382176.463448050373700,
382187.056099862209521,
382197.648776773363352,
382208.241478783485945,
382218.834205891762394,
382229.426958097610623,
382240.019735400390346,
382250.612537799461279,
382261.205365294183139,
382271.798217883973848,
382282.391095568193123,
382292.983998346200678,
382303.576926217298023,
382314.169879181135911,
382324.762857236724813,
382335.355860383540858,
382345.948888621060178,
382356.541941948584281,
382367.135020365414675,
382377.728123871027492,
382388.321252464840654,
382398.914406146097463,
382409.507584914157633,
382420.100788768613711,
382430.694017708534375,
382441.287271733512171,
382451.880550842790399,
382462.473855035786983,
382473.067184311861638,
382483.660538670374081,
382494.253918110742234,
382504.847322632325813,
382515.440752234484535,
382526.034206916578114,
382536.627686678024475,
382547.221191518125124,
382557.814721436181571,
382568.408276431786362,
382579.001856504124589,
382589.595461652672384,
382600.189091876731254,
382610.782747175719123,
382621.376427549053915,
382631.970132995920721,
382642.563863515912089,
382653.157619108271319,
382663.751399772416335,
382674.345205507590435,
382684.939036313386168,
382695.532892188988626,
382706.126773133873940,
382716.720679147401825,
382727.314610228873789,
382737.908566377824172,
382748.502547593496274,
382759.096553875249811,
382769.690585222444497,
382780.284641634498257,
382790.878723110829014,
382801.472829650738277,
382812.066961253585760,
382822.661117918847594,
382833.255299645767082,
382843.849506433820352,
382854.443738282308914,
382865.037995190592483,
382875.632277158088982,
382886.226584184216335,
382896.820916268217843,
382907.415273409569636,
382918.009655607573222,
382928.604062861762941,
382939.198495171207469,
382949.792952535615768,
382960.387434954114724,
382970.981942426238675,
382981.576474951230921,
382992.171032528509386,
383002.765615157491993,
383013.360222837538458,
383023.954855567892082,
383034.549513348145410,
383045.144196177483536,
383055.738904055440798,
383066.333636981260497,
383076.928394954302348,
383087.523177974042483,
383098.117986039782409,
383108.712819150940049,
383119.307677306816913,
383129.902560506947339,
383140.497468750458211,
383151.092402036883868,
383161.687360365642235,
383172.282343736034818,
383182.877352147363126,
383193.472385599103291,
383204.067444090556819,
383214.662527621199843,
383225.257636190275662,
383235.852769797202200,
383246.447928441455588,
383257.043112122279126,
383267.638320839090738,
383278.233554591308348,
383288.828813378175255,
383299.424097199225798,
383310.019406053761486,
383320.614739941083826,
383331.210098860668950,
383341.805482811876573,
383352.400891794008203,
383362.996325806539971,
383373.591784848831594,
383384.187268920184579,
383394.782778019958641,
383405.378312147571705,
383415.973871302499902,
383426.569455483928323,
383437.165064691391308,
383447.760698924073949,
383458.356358181627002,
383468.952042463177349,
383479.547751768201124,
383490.143486095999833,
383500.739245446107816,
383511.335029817710165,
383521.930839210341219,
383532.526673623244278,
383543.122533055895474,
383553.718417507538106,
383564.314326977706514,
383574.910261465702206,
383585.506220970884897,
383596.102205492672510,
383606.698215030366555,
383617.294249583443161,
383627.890309151087422,
383638.486393732891884,
383649.082503328158055,
383659.678637936187442,
383670.274797556514386,
383680.870982188265771,
383691.467191830975935,
383702.063426484062802,
383712.659686146886088,
383723.255970818689093,
383733.852280498889741,
383744.448615187022369,
383755.044974882272072,
383765.641359584114980,
383776.237769291969016,
383786.834204005077481,
383797.430663722858299,
383808.027148444671184,
383818.623658169934060,
383829.220192898064852,
383839.816752628248651,
383850.413337360136211,
383861.009947092912626,
383871.606581825995818,
383882.203241558803711,
383892.799926290637814,
383903.396636020916048,
383913.993370749056339,
383924.590130474360194,
383935.186915196187329,
383945.783724914013874,
383956.380559627083130,
383966.977419334813021,
383977.574304036737885,
383988.171213731984608,
383998.768148420087527,
384009.365108100406360,
384019.962092772300821,
384030.559102435072418,
384041.156137088139076,
384051.753196730976924,
384062.350281362887472,
384072.947390983172227,
384083.544525591365527,
384094.141685186652467,
384104.738869768567383,
384115.336079336411785,
384125.933313889545389,
384136.530573427444324,
384147.127857949293684,
384157.725167454744224,
384168.322501942981035,
384178.919861413363833,
384189.517245865368750,
384200.114655298355501,
384210.712089711567387,
384221.309549104538746,
384231.907033476571087,
384242.504542827082332,
384253.102077155373991,
384263.699636460980400,
384274.297220743203070,
384284.894830001227092,
384295.492464234645013,
384306.090123442758340,
384316.687807625043206,
384327.285516780684702,
384337.883250909217168,
384348.481010009942111,
384359.078794082335662,
384369.676603125641122,
384380.274437139334623,
384390.872296122775879,
384401.470180075266398,
384412.068088996224105,
384422.666022885008715,
384433.263981741154566,
384443.861965563730337,
384454.459974352386780,
384465.058008106367197,
384475.656066825205926,
384486.254150508088060,
384496.852259154431522,
384507.450392763596028,
384518.048551335115917,
384528.646734868118074,
384539.244943362195045,
384549.843176816706546,
384560.441435230895877,
384571.039718604297377,
384581.638026936096139,
384592.236360225884710,
384602.834718472848181,
384613.433101676462684,
384624.031509836146142,
384634.629942951141857,
384645.228401020925958,
384655.826884044858161,
384666.425392022414599,
384677.023924952780362,
384687.622482835373376,
384698.221065669727977,
384708.819673455087468,
384719.418306190811563,
384730.016963876376394,
384740.615646511141676,
384751.214354094350711,
384761.813086625537835,
384772.411844104004558,
384783.010626529168803,
384793.609433900390286,
384804.208266217028722,
384814.807123478443827,
384825.406005684111733,
384836.004912833275739,
384846.603844925411977,
384857.202801959821954,
384867.801783936098218,
384878.400790853367653,
384888.999822710989974,
384899.598879508557729,
384910.197961245256010,
384920.797067920619156,
384931.396199533890467,
384941.995356084546074,
384952.594537571829278,
384963.193743995390832,
384973.792975354299415,
384984.392231648147572,
384994.991512876120396,
385005.590819037810434,
385016.190150132402778,
385026.789506159489974,
385037.388887118198909,
385047.988293008063920,
385058.587723828561138,
385069.187179578817450,
385079.786660258367192,
385090.386165866511874,
385100.985696402727626,
385111.585251866374165,
385122.184832256694790,
385132.784437573223840,
385143.384067815321032,
385153.983722982346080,
385164.583403073658701,
385175.183108088618610,
385185.782838026585523,
385196.382592887035571,
385206.982372669212054,
385217.582177372649312,
385228.182006996648852,
385238.781861540570389,
385249.381741003890056,
385259.981645385792945,
385270.581574685871601,
385281.181528903369326,
385291.781508037762251,
385302.381512088293675,
385312.981541054439731,
385323.581594935560133,
385334.181673731014598,
385344.781777440337464,
385355.381906062713824,
385365.982059597503394,
385376.582238044240512,
385387.182441402226686,
385397.782669670821633,
385408.382922849443275,
385418.983200937451329,
385429.583503934205510,
385440.183831839181948,
385450.784184651682153,
385461.384562371007632,
385471.984964996692725,
385482.585392528038938,
385493.185844964347780,
385503.786322305211797,
385514.386824549816083,
385524.987351697636768,
385535.587903748091776,
385546.188480700424407,
385556.789082553994376,
385567.389709308336023,
385577.990360962809063,
385588.591037516773213,
385599.191738969529979,
385609.792465320555493,
385620.393216569151264,
385630.993992714735214,
385641.594793756667059,
385652.195619694422930,
385662.796470527246129,
385673.397346254612785,
385683.998246875882614,
385694.599172390357126,
385705.200122797512449,
385715.801098096708301,
385726.402098287362605,
385737.003123368718661,
385747.604173340310808,
385758.205248201440554,
385768.806347951467615,
385779.407472589868121,
385790.008622115885373,
385800.609796528995503,
385811.210995828674641,
385821.812220014107879,
385832.413469084713142,
385843.014743039966561,
385853.616041879169643,
385864.217365601740312,
385874.818714207096491,
385885.420087694539689,
385896.021486063429620,
385906.622909313358832,
385917.224357443396002,
385927.825830453133676,
385938.427328341873363,
385949.028851109032985,
385959.630398753972258,
385970.231971276109107,
385980.833568674803246,
385991.435190949356183,
386002.036838099360466,
386012.638510124001186,
386023.240207022638060,
386033.841928794805426,
386044.443675439746585,
386055.045446956995875,
386065.647243345738389,
386076.249064605566673,
386086.850910735665821,
386097.452781735628378,
386108.054677604581229,
386118.656598342175130,
386129.258543947536964,
386139.860514420201071,
386150.462509759585373,
386161.064529964874964,
386171.666575035720598,
386182.268644971307367,
386192.870739771111403,
386203.472859434434213,
386214.075003960693721,
386224.677173349307850,
386235.279367599578109,
386245.881586710980628,
386256.483830682816915,
386267.086099514504895,
386277.688393205520697,
386288.290711755049415,
386298.893055162625387,
386309.495423427550122,
386320.097816549241543,
386330.700234527117573,
386341.302677360479720,
386351.905145048745908,
386362.507637591334060,
386373.110154987662099,
386383.712697236973327,
386394.315264338743873,
386404.917856292275246,
386415.520473097043578,
386426.123114752408583,
386436.725781257729977,
386447.328472612367477,
386457.931188815855421,
386468.533929867320694,
386479.136695766355842,
386489.739486512262374,
386500.342302104458213,
386510.945142542244866,
386521.548007825098466,
386532.150897952320520,
386542.753812923387159,
386553.356752737658098,
386563.959717394434847,
386574.562706893135328,
386585.165721233235672,
386595.768760413979180,
386606.371824434841983,
386616.974913295183796,
386627.578026994364336,
386638.181165531801526,
386648.784328906913288,
386659.387517118942924,
386669.990730167424772,
386680.593968051602133,
386691.197230771009345,
386701.800518324947916,
386712.403830712777562,
386723.007167933916207,
386733.610529987781774,
386744.213916873733979,
386754.817328591074329,
386765.420765139278956,
386776.024226517765783,
386786.627712725836318,
386797.231223762792069,
386807.834759628225584,
386818.438320321380161,
386829.041905841731932,
386839.645516188582405,
386850.249151361291297,
386860.852811359276529,
386871.456496182014234,
386882.060205828805920,
386892.663940298953094,
386903.267699591990095,
386913.871483707276639,
386924.475292644172441,
386935.079126401979011,
386945.682984980114270,
386956.286868378054351,
386966.890776595158968,
386977.494709630671423,
386988.098667484184261,
386998.702650154882576,
387009.306657642417122,
387019.910689945856575,
387030.514747064793482,
387041.118828998471145,
387051.722935746365692,
387062.327067307953257,
387072.931223682418931,
387083.535404869238846,
387094.139610867947340,
387104.743841677613091,
387115.348097297828645,
387125.952377727953717,
387136.556682967348024,
387147.161013015371282,
387157.765367871499620,
387168.369747535034548,
387178.974152005452197,
387189.578581282054074,
387200.183035364141688,
387210.787514251249377,
387221.392017942736857,
387231.996546437905636,
387242.601099736290053,
387253.205677837133408,
387263.810280739853624,
387274.414908443926834,
387285.019560948654544,
387295.624238253338262,
387306.228940357512329,
387316.833667260478251,
387327.438418961712159,
387338.043195460515562,
387348.647996756248176,
387359.252822848444339,
387369.857673736230936,
387380.462549419258721,
387391.067449896770995,
387401.672375168185681,
387412.277325232862495,
387422.882300090219360,
387433.487299739674199,
387444.092324180470314,
387454.697373412258457,
387465.302447434165515,
387475.907546245725825,
387486.512669846124481,
387497.117818235012237,
387507.722991411632393,
387518.328189375344664,
387528.933412125625182,
387539.538659661891870,
387550.143931983388029,
387560.749229089589790,
387571.354550979798660,
387581.959897653432563,
387592.565269109967630,
387603.170665348763578,
387613.776086369121913,
387624.381532170576975,
387634.987002752313856,
387645.592498113866895,
387656.198018254537601,
387666.803563173743896,
387677.409132870903704,
387688.014727345318533,
387698.620346596464515,
387709.225990623759571,
387719.831659426505212,
387730.437353004119359,
387741.043071355903521,
387751.648814481333829,
387762.254582379770000,
387772.860375050746370,
387783.466192493389826,
387794.072034707234707,
387804.677901691640727,
387815.283793446025811,
387825.889709969749674,
387836.495651262113824,
387847.101617322652601,
387857.707608150609303,
387868.313623745576479,
387878.919664106681012,
387889.525729233573657,
387900.131819125439506,
387910.737933781696483,
387921.344073201820720,
387931.950237385113724,
387942.556426330993418,
387953.162640038877726,
387963.768878508126363,
387974.375141738157254,
387984.981429728271905,
387995.587742477946449,
388006.194079986482393,
388016.800442253297660,
388027.406829277810175,
388038.013241059437860,
388048.619677597482223,
388059.226138891477603,
388069.832624940667301,
388080.439135744411033,
388091.045671302126721,
388101.652231613290496,
388112.258816677262075,
388122.865426493342966,
388133.472061061009299,
388144.078720379737206,
388154.685404448595364,
388165.292113267292734,
388175.898846835072618,
388186.505605151352938,
388197.112388215435203,
388207.719196026853751,
388218.326028584910091,
388228.932885889138561,
388239.539767938666046,
388250.146674733026884,
388260.753606271580793,
388271.360562553745694,
388281.967543578881305,
388292.574549346347339,
388303.181579855678137,
388313.788635106058791,
388324.395715097081847,
388335.002819827990606,
388345.609949298144784,
388356.217103507020511,
388366.824282453977503,
388377.431486138491891,
388388.038714559748769,
388398.645967717282474,
388409.253245610510930,
388419.860548238735646,
388430.467875601374544,
388441.075227697787341,
388451.682604527450167,
388462.290006089664530,
388472.897432383790147,
388483.504883409244940,
388494.112359165679663,
388504.719859652046580,
388515.327384867938235,
388525.934934812772553,
388536.542509485909250,
388547.150108886766247,
388557.757733014586847,
388568.365381868905388,
388578.973055449198000,
388589.580753754707985,
388600.188476784736849,
388610.796224538818933,
388621.403997016313951,
388632.011794216639828,
388642.619616139098071,
388653.227462783164810,
388663.835334148257971,
388674.443230233620852,
388685.051151038671378,
388695.659096562943887,
388706.267066805739887,
388716.875061766360886,
388727.483081444341224,
388738.091125838924199,
388748.699194949760567,
388759.307288775919005,
388769.915407316875644,
388780.523550572223030,
388791.131718541088048,
388801.739911223063245,
388812.348128617333714,
388822.956370723492000,
388833.564637540839612,
388844.172929068736266,
388854.781245306658093,
388865.389586253848393,
388875.997951909783296,
388886.606342273880728,
388897.214757345558610,
388907.823197124060243,
388918.431661608978175,
388929.040150799497496,
388939.648664695094340,
388950.257203295186628,
388960.865766599192284,
388971.474354606354609,
388982.082967316207942,
388992.691604728111997,
389003.300266841484699,
389013.908953655511141,
389024.517665169900283,
389035.126401383779012,
389045.735162296681665,
389056.343947907967959,
389066.952758216939401,
389077.561593223188538,
389088.170452925900463,
389098.779337324551307,
389109.388246418500785,
389119.997180207166821,
389130.606138689967338,
389141.215121866203845,
389151.824129735410679,
389162.433162296831142,
389173.042219549941365,
389183.651301494042855,
389194.260408128669951,
389204.869539453065954,
389215.478695466706995,
389226.087876169010997,
389236.697081559221260,
389247.306311636988539,
389257.915566401439719,
389268.524845851992723,
389279.134149988240097,
389289.743478809308726,
389300.352832314907573,
389310.962210504105315,
389321.571613376436289,
389332.181040931376629,
389342.790493168227840,
389353.399970086291432,
389364.009471685101744,
389374.618997964018490,
389385.228548922343180,
389395.838124559551943,
389406.447724875062704,
389417.057349868176971,
389427.666999538370874,
389438.276673885004129,
389448.886372907436453,
389459.496096605085768,
389470.105844977253582,
389480.715618023532443,
389491.325415743107442,
389501.935238135571126,
389512.545085200108588,
389523.154956936254166,
389533.764853343309369,
389544.374774420692120,
389554.984720167878550,
389565.594690584170166,
389576.204685668984894,
389586.814705421682447,
389597.424749841738958,
389608.034818928397726,
389618.644912681134883,
389629.255031099426560,
389639.865174182516057,
389650.475341929879505,
389661.085534340934828,
389671.695751415041741,
389682.305993151559960,
389692.916259549849201,
389703.526550609385595,
389714.136866329587065,
389724.747206709638704,
389735.357571749249473,
389745.967961447604466,
389756.578375804121606,
389767.188814818102401,
389777.799278489139397,
389788.409766816534102,
389799.020279799646232,
389809.630817437951919,
389820.241379730694462,
389830.851966677466407,
389841.462578277511057,
389852.073214530246332,
389862.683875435090158,
389873.294560991402250,
389883.905271198542323,
389894.516006056044716,
389905.126765563210938,
389915.737549719400704,
389926.348358524031937,
389936.959191976580769,
389947.570050076290499,
389958.180932822637260,
389968.791840214980766,
389979.402772252738941,
389990.013728935387917,
390000.624710262229200,
390011.235716232622508,
390021.846746845985763,
390032.457802101736888,
390043.068881999293808,
390053.679986537899822,
390064.291115717205685,
390074.902269536454696,
390085.513447995006572,
390096.124651092221029,
390106.735878827574197,
390117.347131200484000,
390127.958408210368361,
390138.569709856470581,
390149.181036138383206,
390159.792387055291329,
390170.403762606729288,
390181.015162791998591,
390191.626587610517163,
390202.238037061761133,
390212.849511145032011,
390223.461009859805927,
390234.072533205442596,
390244.684081181243528,
390255.295653786743060,
390265.907251021184493,
390276.518872884102166,
390287.130519374797586,
390297.742190492688678,
390308.353886237251572,
390318.965606607845984,
390329.577351603657007,
390340.189121224335395,
390350.800915469240863,
390361.412734337616712,
390372.024577828997280,
390382.636445942684077,
390393.248338678269647,
390403.860256034880877,
390414.472198012052104,
390425.084164609143045,
390435.696155825513415,
390446.308171660639346,
390456.920212113880552,
390467.532277184654959,
390478.144366872322280,
390488.756481176242232,
390499.368620095890947,
390509.980783630628139,
390520.592971779755317,
390531.205184542865027,
390541.817421919200569,
390552.429683908179868,
390563.041970509220846,
390573.654281721683219,
390584.266617544984911,
390594.878977978602052,
390605.491363021777943,
390616.103772673930507,
390626.716206934535876,
390637.328665803070180,
390647.941149278660305,
390658.553657360957004,
390669.166190049145371,
390679.778747342817951,
390690.391329241218045,
390701.003935743879993,
390711.616566850047093,
390722.229222559253685,
390732.841902870684862,
390743.454607783933170,
390754.067337298416533,
390764.680091413320042,
390775.292870128294453,
390785.905673442524858,
390796.518501355545595,
390807.131353866658174,
390817.744230975338724,
390828.357132680830546,
390838.970058982726187,
390849.583009880268946,
390860.195985372993164,
390870.808985460200347,
390881.422010141250212,
390892.035059415618889,
390902.648133282724302,
390913.261231741809752,
390923.874354792351369,
390934.487502433883492,
390945.100674665591214,
390955.713871487008873,
390966.327092897437979,
390976.940338896354660,
390987.553609483118635,
390998.166904657089617,
391008.780224417743739,
391019.393568764382508,
391030.006937696482055,
391040.620331213402096,
391051.233749314502347,
391061.847191999258939,
391072.460659267031588,
391083.074151117121801,
391093.687667549122125,
391104.301208562334068,
391114.914774156117346,
391125.528364329831675,
391136.141979082894977,
391146.755618414841592,
391157.369282324973028,
391167.982970812532585,
391178.596683877171017,
391189.210421518248040,
391199.824183735006955,
391210.437970526865683,
391221.051781893358566,
391231.665617833728902,
391242.279478347511031,
391252.893363434006460,
391263.507273092691321,
391274.121207322808914,
391284.735166123951785,
391295.349149495363235,
391305.963157436577603,
391316.577189946779981,
391327.191247025621124,
391337.805328672286123,
391348.419434886309318,
391359.033565666992217,
391369.647721013869159,
391380.261900926125236,
391390.876105403352994,
391401.490334444795735,
391412.104588049987797,
391422.718866218347102,
391433.333168949116953,
391443.947496241657063,
391454.561848095618188,
391465.176224510185421,
391475.790625484776683,
391486.405051018868107,
391497.019501111877616,
391507.633975763106719,
391518.248474972031545,
391528.862998737895396,
391539.477547060290817,
391550.092119938519318,
391560.706717371940613,
391571.321339360030834,
391581.935985902266111,
391592.550656997831538,
391603.165352646203246,
391613.780072846915573,
391624.394817599153612,
391635.009586902393494,
391645.624380756111350,
391656.239199159725104,
391666.854042112478055,
391677.468909613788128,
391688.083801663131453,
391698.698718259867746,
391709.313659403531346,
391719.928625093249138,
391730.543615328730084,
391741.158630109042861,
391751.773669433838222,
391762.388733302359469,
391773.003821714140940,
391783.618934668484144,
391794.234072164807003,
391804.849234202527441,
391815.464420781005174,
391826.079631899658125,
391836.694867557904217,
391847.310127755103167,
391857.925412490731105,
391868.540721764031332,
391879.156055574596394,
391889.771413921727799,
391900.386796804785263,
391911.002204223186709,
391921.617636176350061,
391932.233092663693242,
391942.848573684517760,
391953.464079238416161,
391964.079609324631747,
391974.695163942582440,
391985.310743091627955,
391995.926346771302633,
392006.541974980907980,
392017.157627719803713,
392027.773304987524170,
392038.389006783312652,
392049.004733106703497,
392059.620483956998214,
392070.236259333614726,
392080.852059235912748,
392091.467883663426619,
392102.083732615457848,
392112.699606091424357,
392123.315504090685863,
392133.931426612718496,
392144.547373656823765,
392155.163345222477801,
392165.779341308982112,
392176.395361915871035,
392187.011407042620704,
392197.627476688299794,
392208.243570852559060,
392218.859689534700010,
392229.475832734140567,
392240.092000450415071,
392250.708192682650406,
392261.324409430497326,
392271.940650693257339,
392282.556916470348369,
392293.173206761071924,
392303.789521564904135,
392314.405860881262925,
392325.022224709566217,
392335.638613049115520,
392346.255025899445172,
392356.871463259856682,
392367.487925129709765,
392378.104411508538760,
392388.720922395645175,
392399.337457790446933,
392409.954017692361958,
392420.570602100808173,
392431.187211015028879,
392441.803844434674829,
392452.420502358931117,
392463.037184787390288,
392473.653891719179228,
392484.270623154006898,
392494.887379091116600,
392505.504159529926255,
392516.120964469853789,
392526.737793910200708,
392537.354647850443143,
392547.971526289999019,
392558.588429228286259,
392569.205356664664578,
392579.822308598551899,
392590.439285029249731,
392601.056285956350621,
392611.673311379039660,
392622.290361296909396,
392632.907435709261335,
392643.524534615513403,
392654.141658015025314,
392664.758805907214992,
392675.375978291500360,
392685.993175167299341,
392696.610396534029860,
392707.227642391051631,
392717.844912737782579,
392728.462207573524211,
392739.079526897810865,
392749.696870710002258,
392760.314239009458106,
392770.931631795654539,
392781.549049067951273,
392792.166490825708024,
392802.783957068400923,
392813.401447795389686,
392824.018963006034028,
392834.636502699810080,
392845.254066876077559,
392855.871655534254387,
392866.489268673816696,
392877.106906293949578,
392887.724568394303787,
392898.342254974064417,
392908.959966032765806,
392919.577701569767669,
392930.195461584429722,
392940.813246076344512,
392951.431055044638924,
392962.048888488789089,
392972.666746408387553,
392983.284628802619409,
392993.902535670960788,
393004.520467012829613,
393015.138422827702016,
393025.756403114821296,
393036.374407873605378,
393046.992437103588600,
393057.610490804014262,
393068.228568974416703,
393078.846671614155639,
393089.464798722648993,
393100.082950299198274,
393110.701126343279611,
393121.319326854252722,
393131.937551831593737,
393142.555801274662372,
393153.174075182876550,
393163.792373555595987,
393174.410696392238606,
393185.029043692280538,
393195.647415454965085,
393206.265811679884791,
393216.884232366282959,
393227.502677513635717,
393238.121147121419199,
393248.739641188818496,
393259.358159715426154,
393269.976702700543683,
393280.595270143589005,
393291.213862044096459,
393301.832478401251137,
393312.451119214703795,
393323.069784483581316,
393333.688474207476247,
393344.307188385690097,
393354.925927017698996,
393365.544690102862660,
393376.163477640599012,
393386.782289630325977,
393397.401126071403269,
393408.019986963248812,
393418.638872305280529,
393429.257782096916344,
393439.876716337457765,
393450.495675026439130,
393461.114658163220156,
393471.733665747160558,
393482.352697777678259,
393492.971754254191183,
393503.590835176117253,
393514.209940542874392,
393524.829070353764109,
393535.448224608320743,
393546.067403305904008,
393556.686606445815414,
393567.305834027589299,
393577.925086050527170,
393588.544362514046952,
393599.163663417682983,
393609.782988760736771,
393620.402338542509824,
393631.021712762536481,
393641.641111420292873,
393652.260534514964093,
393662.879982046142686,
393673.499454013130162,
393684.118950415402651,
393694.738471252319869,
393705.358016523241531,
393715.977586227585562,
393726.597180364769883,
393737.216798934328835,
393747.836441935389303,
393758.456109367660247,
393769.075801230326761,
393779.695517522923183,
393790.315258244634606,
393800.935023395169992,
393811.554812973714434,
393822.174626979802269,
393832.794465412676800,
393843.414328271930572,
393854.034215556865092,
393864.654127266781870,
393875.274063401331659,
393885.894023959699553,
393896.514008941419888,
393907.134018345852382,
393917.754052172414958,
393928.374110420409124,
393938.994193089369219,
393949.614300178654958,
393960.234431687684264,
393970.854587615816854,
393981.474767962470651,
393992.094972727063578,
394002.715201909071766,
394013.335455507680308,
394023.955733522539958,
394034.576035952952225,
394045.196362798335031,
394055.816714058048092,
394066.437089731451124,
394077.057489818194881,
394087.677914317348041,
394098.298363228561357,
394108.918836551136337,
394119.539334284549113,
394130.159856428042985,
394140.780402981210500,
394151.400973943353165,
394162.021569313830696,
394172.642189092235640,
394183.262833277753089,
394193.883501869917382,
394204.504194868146442,
394215.124912271799985,
394225.745654080237728,
394236.366420292877592,
394246.987210909137502,
394257.608025928493589,
394268.228865350305568,
394278.849729173933156,
394289.470617398852482,
394300.091530024423264,
394310.712467050005216,
394321.333428475074470,
394331.954414299107157,
394342.575424521346577,
394353.196459141210653,
394363.817518158233725,
394374.438601571717300,
394385.059709381079301,
394395.680841585795861,
394406.301998185226694,
394416.923179178731516,
394427.544384565786459,
394438.165614345693029,
394448.786868517985567,
394459.408147081965581,
394470.029450037167408,
394480.650777382834349,
394491.272129118442535,
394501.893505243468098,
394512.514905757212546,
394523.136330659093801,
394533.757779948529787,
394544.379253624996636,
394555.000751687795855,
394565.622274136461783,
394576.243820970237721,
394586.865392188658006,
394597.486987790965941,
394608.108607776754070,
394618.730252145440318,
394629.351920896209776,
394639.973614028654993,
394650.595331542077474,
394661.217073436011560,
394671.838839709700551,
394682.460630362736993,
394693.082445394364186,
394703.704284804058261,
394714.326148591178935,
394724.948036755144130,
394735.569949295429979,
394746.191886211396195,
394756.813847502402496,
394767.435833167983219,
394778.057843207439873,
394788.679877620190382,
394799.301936405536253,
394809.924019563128240,
394820.546127092209645,
394831.168258992198389,
394841.790415262512397,
394852.412595902569592,
394863.034800911787897,
394873.657030289527029,
394884.279284035263117,
394894.901562148297671,
394905.523864628223237,
394916.146191474224906,
394926.768542685837019,
394937.390918262361083,
394948.013318203389645,
394958.635742508107796,
394969.258191176108085,
394979.880664206750225,
394990.503161599393934,
395001.125683353398927,
395011.748229468299542,
395022.370799943397287,
395032.993394778168295,
395043.616013971972279,
395054.238657524285372,
395064.861325434467290,
395075.484017701819539,
395086.106734325876459,
395096.729475306114182,
395107.352240641717799,
395117.975030332396273,
395128.597844377218280,
395139.220682775834575,
395149.843545527488459,
395160.466432631772477,
395171.089344087988138,
395181.712279895436950,
395192.335240053769667,
395202.958224562113173,
395213.581233420118224,
395224.204266627086326,
395234.827324182435405,
395245.450406085525174,
395256.073512335831765,
395266.696642932714894,
395277.319797875650693,
395287.942977163940668,
395298.566180797119159,
395309.189408774429467,
395319.812661095405929,
395330.435937759466469,
395341.059238765970804,
395351.682564114336856,
395362.305913803982548,
395372.929287834209390,
395383.552686204551719,
395394.176108914427459,
395404.799555963254534,
395415.423027350276243,
395426.046523074968718,
395436.670043136866298,
395447.293587535328697,
395457.917156269599218,
395468.540749339328613,
395479.164366743760183,
395489.788008482311852,
395500.411674554401543,
395511.035364959563594,
395521.659079697041307,
395532.282818766310811,
395542.906582166906446,
395553.530369897955097,
395564.154181958991103,
395574.778018349548802,
395585.401879068871494,
395596.025764116435312,
395606.649673491774593,
395617.273607194016222,
395627.897565222752746,
395638.521547577343881,
395649.145554257265758,
395659.769585261819884,
395670.393640590424184,
395681.017720242671203,
395691.641824217746034,
395702.265952515183017,
395712.890105134283658,
395723.514282074524090,
395734.138483335380442,
395744.762708916212432,
395755.386958816263359,
395766.011233035183977,
395776.635531572217587,
395787.259854426956736,
395797.884201598586515,
395808.508573086641263,
395819.132968890597112,
395829.757389009697363,
395840.381833443418145,
395851.006302191177383,
395861.630795252334792,
395872.255312626424711,
395882.879854312865064,
395893.504420310840942,
395904.129010620003100,
395914.753625239594840,
395925.378264169034082,
395936.002927407855168,
395946.627614955417812,
395957.252326811139937,
395967.877062974264845,
395978.501823444443289,
395989.126608220918570,
395999.751417303166818,
396010.376250690605957,
396021.001108382770326,
396031.625990378786810,
396042.250896678189747,
396052.875827280455269,
396063.500782184943091,
396074.125761391012929,
396084.750764898140915,
396095.375792705803178,
396106.000844813243020,
396116.625921219936572,
396127.251021925359964,
396137.876146928872913,
396148.501296229835134,
396159.126469827780966,
396169.751667721953709,
396180.376889911887702,
396191.002136397000868,
396201.627407176652923,
396212.252702250261791,
396222.878021617187187,
396233.503365276963450,
396244.128733228892088,
396254.754125472332817,
396265.379542006819975,
396276.004982831771486,
396286.630447946605273,
396297.255937350622844,
396307.881451043242123,
396318.506989023939241,
396329.132551292073913,
396339.758137847180478,
396350.383748688444030,
396361.009383815457113,
396371.635043227637652,
396382.260726924287155,
396392.886434904881753,
396403.512167168781161,
396414.137923715461511,
396424.763704544282518,
396435.389509654720314,
396446.015339046076406,
396456.641192717826925,
396467.267070669448003,
396477.892972900124732,
396488.518899409507867,
396499.144850197015330,
396509.770825261890423,
396520.396824603609275,
396531.022848221531603,
396541.648896115249954,
396552.274968284007628,
396562.901064727164339,
396573.527185444370843,
396584.153330434870441,
396594.779499698081054,
396605.405693233420607,
396616.031911040248815,
396626.658153118158225,
396637.284419466333929,
396647.910710084368475,
396658.537024971563369,
396669.163364127336536,
396679.789727551164106,
396690.416115242405795,
396701.042527200479526,
396711.668963424861431,
396722.295423914911225,
396732.921908669930417,
396743.548417689453345,
396754.174950972897932,
396764.801508519623894,
396775.428090329107363,
396786.054696400708053,
396796.681326733843889,
396807.307981327874586,
396817.934660182334483,
396828.561363296583295,
396839.188090669980738,
396849.814842302002944,
396860.441618191951420,
396871.068418339360505,
396881.695242743648123,
396892.322091404173989,
396902.948964320239611,
396913.575861491553951,
396924.202782917243894,
396934.829728596843779,
396945.456698529771529,
396956.083692715270445,
396966.710711152991280,
396977.337753842351958,
396987.964820782595780,
396998.591911973140668,
397009.219027413462754,
397019.846167103038169,
397030.473331041110214,
397041.100519227329642,
397051.727731660881545,
397062.354968341242056,
397072.982229267829098,
397083.609514440177009,
397094.236823857529089,
397104.864157519303262,
397115.491515425150283,
397126.118897574197035,
397136.746303965919651,
397147.373734599852469,
397158.001189475238789,
397168.628668591612950,
397179.256171948451083,
397189.883699544996489,
397200.511251380667090,
397211.138827454997227,
397221.766427767346613,
397232.394052317074966,
397243.021701103774831,
397253.649374126689509,
397264.277071385178715,
397274.904792878776789,
397285.532538606901653,
397296.160308568913024,
397306.788102764345240,
397317.415921192441601,
397328.043763852678239,
397338.671630744356662,
397349.299521867069416,
397359.927437220234424,
397370.555376803153194,
397381.183340615360066,
397391.811328656098340,
397402.439340924902353,
397413.067377421073616,
397423.695438144146465,
397434.323523093538824,
397444.951632268668618,
397455.579765668779146,
397466.207923293462954,
397476.836105142021552,
397487.464311213989276,
397498.092541508609429,
397508.720796025532763,
397519.349074763944373,
397529.977377723378595,
397540.605704903195146,
397551.234056302870158,
397561.862431921763346,
397572.490831759234425,
397583.119255814875942,
397593.747704087931197,
397604.376176577818114,
397615.004673284129240,
397625.633194206049666,
397636.261739343113732,
397646.890308694797568,
397657.518902260344476,
397668.147520039288793,
397678.776162031048443,
397689.404828234924935,
397700.033518650452606,
397710.662233277049381,
397721.290972114074975,
397731.919735160889104,
397742.548522416967899,
397753.177333881787490,
397763.806169554591179,
397774.435029435087927,
397785.063913522346411,
397795.692821816017386,
397806.321754315402359,
397816.950711019977462,
397827.579691929102410,
397838.208697042253334,
397848.837726358789951,
397859.466779878130183,
397870.095857599750161,
397880.724959522893187,
397891.354085647210013,
397901.983235971943941,
397912.612410496687517,
397923.241609220625833,
397933.870832143293228,
397944.500079264165834,
397955.129350582545158,
397965.758646097790916,
397976.387965809553862,
397987.017309717077296,
397997.646677819779143,
398008.276070117135532,
398018.905486608506180,
398029.534927293250803,
398040.164392171020154,
398050.793881240941118,
398061.423394502664451,
398072.052931955375243,
398082.682493598724250,
398093.312079431896564,
398103.941689454601146,
398114.571323665906675,
398125.200982065405697,
398135.830664652574342,
398146.460371426714119,
398157.090102387301158,
398167.719857533753384,
398178.349636865430512,
398188.979440381808672,
398199.609268082247581,
398210.239119966165163,
398220.868996033095755,
398231.498896282282658,
398242.128820713318419,
398252.758769325388130,
398263.388742118026130,
398274.018739090824965,
398284.648760242853314,
398295.278805573878344,
398305.908875083085150,
398316.538968769891653,
398327.169086633773986,
398337.799228674208280,
398348.429394890496042,
398359.059585282113403,
398369.689799848478287,
398380.320038589008618,
398390.950301503064111,
398401.580588590062689,
398412.210899849538691,
398422.841235280851834,
398433.471594883361831,
398444.101978656486608,
398454.732386599644087,
398465.362818712368608,
398475.993274993903469,
398486.623755443783011,
398497.254260061425157,
398507.884788846131414,
398518.515341797494330,
398529.145918914757203,
398539.776520197396167,
398550.407145644829143,
398561.037795256532263,
398571.668469031865243,
398582.299166970246006,
398592.929889071034268,
398603.560635333822574,
398614.191405757796019,
398624.822200342547148,
398635.453019087435678,
398646.083861991879530,
398656.714729055238422,
398667.345620276988484,
398677.976535656605847,
398688.607475193392020,
398699.238438886823133,
398709.869426736258902,
398720.500438741175458,
398731.131474900990725,
398741.762535215064418,
398752.393619682989083,
398763.024728303949814,
398773.655861077422742,
398784.287018002883997,
398794.918199079693295,
398805.549404307326768,
398816.180633685260545,
398826.811887212737929,
398837.443164889235049,
398848.074466714228038,
398858.705792687193025,
398869.337142807315104,
398879.968517074303236,
398890.599915487342514,
398901.231338045850862,
398911.862784749420825,
398922.494255597295705,
398933.125750589068048,
398943.757269724039361,
398954.388813001627568,
398965.020380421250593,
398975.651971982326359,
398986.283587684330996,
398996.915227526566014,
399007.546891508682165,
399018.178579629864544,
399028.810291889531072,
399039.442028287216090,
399050.073788822279312,
399060.705573494138662,
399071.337382302270271,
399081.969215246033855,
399092.601072324789129,
399103.232953538070433,
399113.864858885237481,
399124.496788365766406,
399135.128741979016922,
399145.760719724348746,
399156.392721601238009,
399167.024747609219048,
399177.656797747535165,
399188.288872015604284,
399198.920970413077157,
399209.553092939022463,
399220.185239593090955,
399230.817410374700557,
399241.449605283210985,
399252.081824317981955,
399262.714067478664219,
399273.346334764326457,
399283.978626174619421,
399294.610941708902828,
399305.243281366652809,
399315.875645147170871,
399326.508033050049562,
399337.140445074473973,
399347.772881220036652,
399358.405341486039106,
399369.037825872073881,
399379.670334377384279,
399390.302867001504637,
399400.935423743736465,
399411.568004603672307,
399422.200609580555465,
399432.833238673803862,
399443.465891883010045,
399454.098569207475521,
399464.731270646618214,
399475.363996199856047,
399485.996745866606943,
399496.629519646288827,
399507.262317538377829,
399517.895139542291872,
399528.527985657332465,
399539.160855883033946,
399549.793750218697824,
399560.426668663858436,
399571.059611217875499,
399581.692577880283352,
399592.325568650325295,
399602.958583527535666,
399613.591622511215974,
399624.224685600958765,
399634.857772796065547,
399645.490884095954243,
399656.124019500159193,
399666.757179007865489,
399677.390362618665677,
399688.023570332035888,
399698.656802147219423,
399709.290058063808829,
399719.923338081105612,
399730.556642