# All Rights Reserved 2020
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from openstack import resource


class L2gw(resource.Resource):
    """L2 Gateway"""

    resource_key = 'l2_gateway'
    resources_key = 'l2_gateways'
    base_path = '/l2-gateways'

    # capabilities
    allow_create = True
    allow_fetch = True
    allow_commit = True
    allow_delete = True
    allow_list = True

    _allow_unknown_attrs_in_body = True

    _query_mapping = resource.QueryParameters(
        "sort_key", "sort_dir",
        'name', 'project_id'
    )

    # Properties
    #: The ID of the l2 gateway.
    id = resource.Body('id')
    #: The l2 gateway's name.
    name = resource.Body('name')
    #: The ID of the project that owns the l2 gateway.
    project_id = resource.Body('project_id', alias='tenant_id')
    #: Tenant_id (deprecated attribute).
    tenant_id = resource.Body('tenant_id', deprecated=True)
    #: Device name and Interface-names of l2gateway.
    devices = resource.Body('devices')


class L2gwConnection(resource.Resource):
    """L2 Gateway Connection"""

    resource_key = 'l2_gateway_connection'
    resources_key = 'l2_gateway_connections'
    base_path = '/l2-gateway-connections'

    # capabilities
    allow_create = True
    allow_fetch = True
    allow_commit = False
    allow_delete = True
    allow_list = True

    _allow_unknown_attrs_in_body = True

    _query_mapping = resource.QueryParameters(
        "sort_key", "sort_dir",
        'name', 'project_id'
    )

    # Properties
    #: The ID of the l2 gateway connection.
    id = resource.Body('id')
    #: The ID of the project that owns the l2 gateway connection.
    project_id = resource.Body('project_id', alias='tenant_id')
    #: Tenant_id (deprecated attribute).
    tenant_id = resource.Body('tenant_id', deprecated=True)
    #: The id of the Network to which the l2gw-connection is associated.
    network_id = resource.Body('network_id')
    #: The segmentation id that will be applied to the interfaces.
    segmentation_id = resource.Body('segmentation_id')
    #: The id of the L2gw to which the l2gw-connection is associated.
    l2gw_id = resource.Body('l2gw_id')
