/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 */

#if !defined(SIMDE_ARM_NEON_QSUB_H)
#define SIMDE_ARM_NEON_QSUB_H

#include "types.h"

#include "bsl.h"
#include "cgt.h"
#include "dup_n.h"
#include "sub.h"
#include "sub.h"

#include <limits.h>

HEDLEY_DIAGNOSTIC_PUSH
SIMDE_DISABLE_UNWANTED_DIAGNOSTICS
SIMDE_BEGIN_DECLS_

SIMDE_FUNCTION_ATTRIBUTES
int8_t simde_vqsubb_s8(int8_t a, int8_t b) { return simde_math_subs_i8(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubb_s8
#define vqsubb_s8(a, b) simde_vqsubb_s8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
int16_t simde_vqsubh_s16(int16_t a, int16_t b) { return simde_math_subs_i16(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubh_s16
#define vqsubh_s16(a, b) simde_vqsubh_s16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
int32_t simde_vqsubs_s32(int32_t a, int32_t b) { return simde_math_subs_i32(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubs_s32
#define vqsubs_s32(a, b) simde_vqsubs_s32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
int64_t simde_vqsubd_s64(int64_t a, int64_t b) { return simde_math_subs_i64(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubd_s64
#define vqsubd_s64(a, b) simde_vqsubd_s64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint8_t simde_vqsubb_u8(uint8_t a, uint8_t b) { return simde_math_subs_u8(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubb_u8
#define vqsubb_u8(a, b) simde_vqsubb_u8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint16_t simde_vqsubh_u16(uint16_t a, uint16_t b) { return simde_math_subs_u16(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubh_u16
#define vqsubh_u16(a, b) simde_vqsubh_u16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint32_t simde_vqsubs_u32(uint32_t a, uint32_t b) { return simde_math_subs_u32(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubs_u32
#define vqsubs_u32(a, b) simde_vqsubs_u32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint64_t simde_vqsubd_u64(uint64_t a, uint64_t b) { return simde_math_subs_u64(a, b); }
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vqsubd_u64
#define vqsubd_u64(a, b) simde_vqsubd_u64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x8_t simde_vqsub_s8(simde_int8x8_t a, simde_int8x8_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_s8(a, b);
#elif defined(SIMDE_X86_MMX_NATIVE)
    return _mm_subs_pi8(a, b);
#else
    simde_int8x8_private r_, a_ = simde_int8x8_to_private(a), b_ = simde_int8x8_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubb_s8(a_.values[i], b_.values[i]);
    }

    return simde_int8x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_s8
#define vqsub_s8(a, b) simde_vqsub_s8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x4_t simde_vqsub_s16(simde_int16x4_t a, simde_int16x4_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_s16(a, b);
#elif defined(SIMDE_X86_MMX_NATIVE)
    return _mm_subs_pi16(a, b);
#else
    simde_int16x4_private r_, a_ = simde_int16x4_to_private(a), b_ = simde_int16x4_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubh_s16(a_.values[i], b_.values[i]);
    }

    return simde_int16x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_s16
#define vqsub_s16(a, b) simde_vqsub_s16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x2_t simde_vqsub_s32(simde_int32x2_t a, simde_int32x2_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_s32(a, b);
#else
    simde_int32x2_private r_, a_ = simde_int32x2_to_private(a), b_ = simde_int32x2_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubs_s32(a_.values[i], b_.values[i]);
    }

    return simde_int32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_s32
#define vqsub_s32(a, b) simde_vqsub_s32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x1_t simde_vqsub_s64(simde_int64x1_t a, simde_int64x1_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_s64(a, b);
#else
    simde_int64x1_private r_, a_ = simde_int64x1_to_private(a), b_ = simde_int64x1_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubd_s64(a_.values[i], b_.values[i]);
    }

    return simde_int64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_s64
#define vqsub_s64(a, b) simde_vqsub_s64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint8x8_t simde_vqsub_u8(simde_uint8x8_t a, simde_uint8x8_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_u8(a, b);
#elif defined(SIMDE_X86_MMX_NATIVE)
    return _mm_subs_pu8(a, b);
#else
    simde_uint8x8_private r_, a_ = simde_uint8x8_to_private(a), b_ = simde_uint8x8_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubb_u8(a_.values[i], b_.values[i]);
    }

    return simde_uint8x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_u8
#define vqsub_u8(a, b) simde_vqsub_u8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint16x4_t simde_vqsub_u16(simde_uint16x4_t a, simde_uint16x4_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_u16(a, b);
#elif defined(SIMDE_X86_MMX_NATIVE)
    return _mm_subs_pu16(a, b);
#else
    simde_uint16x4_private r_, a_ = simde_uint16x4_to_private(a), b_ = simde_uint16x4_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubh_u16(a_.values[i], b_.values[i]);
    }

    return simde_uint16x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_u16
#define vqsub_u16(a, b) simde_vqsub_u16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x2_t simde_vqsub_u32(simde_uint32x2_t a, simde_uint32x2_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_u32(a, b);
#else
    simde_uint32x2_private r_, a_ = simde_uint32x2_to_private(a), b_ = simde_uint32x2_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubs_u32(a_.values[i], b_.values[i]);
    }

    return simde_uint32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_u32
#define vqsub_u32(a, b) simde_vqsub_u32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint64x1_t simde_vqsub_u64(simde_uint64x1_t a, simde_uint64x1_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsub_u64(a, b);
#else
    simde_uint64x1_private r_, a_ = simde_uint64x1_to_private(a), b_ = simde_uint64x1_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubd_u64(a_.values[i], b_.values[i]);
    }

    return simde_uint64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsub_u64
#define vqsub_u64(a, b) simde_vqsub_u64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int8x16_t simde_vqsubq_s8(simde_int8x16_t a, simde_int8x16_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_s8(a, b);
#elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_i8x16_sub_saturate(a, b);
#elif defined(SIMDE_X86_SSE2_NATIVE)
    return _mm_subs_epi8(a, b);
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_int8x16_private r_, a_ = simde_int8x16_to_private(a), b_ = simde_int8x16_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubb_s8(a_.values[i], b_.values[i]);
    }

    return simde_int8x16_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_s8
#define vqsubq_s8(a, b) simde_vqsubq_s8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int16x8_t simde_vqsubq_s16(simde_int16x8_t a, simde_int16x8_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_s16(a, b);
#elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_i16x8_sub_saturate(a, b);
#elif defined(SIMDE_X86_SSE2_NATIVE)
    return _mm_subs_epi16(a, b);
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_int16x8_private r_, a_ = simde_int16x8_to_private(a), b_ = simde_int16x8_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubh_s16(a_.values[i], b_.values[i]);
    }

    return simde_int16x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_s16
#define vqsubq_s16(a, b) simde_vqsubq_s16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x4_t simde_vqsubq_s32(simde_int32x4_t a, simde_int32x4_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_s32(a, b);
#elif defined(SIMDE_X86_AVX512VL_NATIVE)
    return _mm256_cvtsepi64_epi32(
        _mm256_sub_epi64(_mm256_cvtepi32_epi64(a), _mm256_cvtepi32_epi64(b)));
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_int32x4_private r_, a_ = simde_int32x4_to_private(a), b_ = simde_int32x4_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubs_s32(a_.values[i], b_.values[i]);
    }

    return simde_int32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_s32
#define vqsubq_s32(a, b) simde_vqsubq_s32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x2_t simde_vqsubq_s64(simde_int64x2_t a, simde_int64x2_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_s64(a, b);
#else
    simde_int64x2_private r_, a_ = simde_int64x2_to_private(a), b_ = simde_int64x2_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubd_s64(a_.values[i], b_.values[i]);
    }

    return simde_int64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_s64
#define vqsubq_s64(a, b) simde_vqsubq_s64((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint8x16_t simde_vqsubq_u8(simde_uint8x16_t a, simde_uint8x16_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_u8(a, b);
#elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_u8x16_sub_saturate(a, b);
#elif defined(SIMDE_X86_SSE2_NATIVE)
    return _mm_subs_epu8(a, b);
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_uint8x16_private r_, a_ = simde_uint8x16_to_private(a), b_ = simde_uint8x16_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubb_u8(a_.values[i], b_.values[i]);
    }

    return simde_uint8x16_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_u8
#define vqsubq_u8(a, b) simde_vqsubq_u8((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint16x8_t simde_vqsubq_u16(simde_uint16x8_t a, simde_uint16x8_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_u16(a, b);
#elif defined(SIMDE_WASM_SIMD128_NATIVE)
    return wasm_u16x8_sub_saturate(a, b);
#elif defined(SIMDE_X86_SSE2_NATIVE)
    return _mm_subs_epu16(a, b);
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_uint16x8_private r_, a_ = simde_uint16x8_to_private(a), b_ = simde_uint16x8_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubh_u16(a_.values[i], b_.values[i]);
    }

    return simde_uint16x8_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_u16
#define vqsubq_u16(a, b) simde_vqsubq_u16((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x4_t simde_vqsubq_u32(simde_uint32x4_t a, simde_uint32x4_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_u32(a, b);
#elif defined(SIMDE_POWER_ALTIVEC_P6)
    return vec_subs(a, b);
#else
    simde_uint32x4_private r_, a_ = simde_uint32x4_to_private(a), b_ = simde_uint32x4_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubs_u32(a_.values[i], b_.values[i]);
    }

    return simde_uint32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_u32
#define vqsubq_u32(a, b) simde_vqsubq_u32((a), (b))
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint64x2_t simde_vqsubq_u64(simde_uint64x2_t a, simde_uint64x2_t b)
{
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
    return vqsubq_u64(a, b);
#else
    simde_uint64x2_private r_, a_ = simde_uint64x2_to_private(a), b_ = simde_uint64x2_to_private(b);

    SIMDE_VECTORIZE
    for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
        r_.values[i] = simde_vqsubd_u64(a_.values[i], b_.values[i]);
    }

    return simde_uint64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vqsubq_u64
#define vqsubq_u64(a, b) simde_vqsubq_u64((a), (b))
#endif

SIMDE_END_DECLS_
HEDLEY_DIAGNOSTIC_POP

#endif /* !defined(SIMDE_ARM_NEON_QSUB_H) */
