/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include <QApplication>
#include <QGraphicsItem>
#include <QDebug>
#include <QTime>

#include <cstdlib>
#include <limits>
#include <cmath>

#include <Plasma/Animator>

#include "SmoothTasks/Applet.h"
#include "SmoothTasks/TaskbarLayout.h"
#include "SmoothTasks/TaskItem.h"
#include "SmoothTasks/Global.h"

namespace SmoothTasks {

TaskbarItem::~TaskbarItem() {
	if (item) {
		item->setParentLayoutItem(NULL);
		if (item->ownedByLayout()) {
			delete item;
		}
	}
}

const qreal TaskbarLayout::PIXELS_PER_SECOND = 500;

TaskbarLayout::TaskbarLayout(Qt::Orientation orientation, QGraphicsLayoutItem *parent)
	: QGraphicsLayout(parent),
	  m_draggedItem(NULL),
	  m_currentIndex(-1),
	  m_mouseIn(false),
	  m_orientation(orientation),
	  m_spacing(0.0),
	  m_animationTimer(new QTimer(this)),
	  m_grabPos(),
	  m_fps(35),
	  m_animationsEnabled(true),
	  m_minimumRows(1),
	  m_maximumRows(6),
	  m_expandedWidth(175),
	  m_expandDuration(160),
	  m_preferredSizeChanged(false),
	  m_cellHeight(1.0),
	  m_rows(1)
	  {
	m_animationTimer->setInterval(1000 / m_fps);

	connect(m_animationTimer, SIGNAL(timeout()), this, SLOT(animate()));
}

// more or less copied from QGraphicsLinearLayout
TaskbarLayout::~TaskbarLayout() {
	clear();
}

void TaskbarLayout::setOrientation(Qt::Orientation orientation) {
	if (orientation != m_orientation) {
		m_orientation = orientation;
		foreach (TaskbarItem *item, m_items) {
			item->item->setOrientation(orientation);
		}
		invalidate();
	}
}

void TaskbarLayout::setSpacing(qreal spacing) {
	if (spacing < 0) {
		qWarning("TaskbarLayout::setSpacing: invalid spacing %g", spacing);
		return;
	}

	if (spacing != m_spacing) {
		m_spacing = spacing;
		invalidate();
	}
}

void TaskbarLayout::setFps(int fps) {
	if (fps <= 0) {
		qWarning("TaskbarLayout::setFps: invalid fps %d", fps);
		return;
	}

	if (m_fps != fps) {
		m_fps = fps;
		m_animationTimer->setInterval(1000 / m_fps);
	}
}

void TaskbarLayout::setAnimationSpeed(qreal animationSpeed) {
	if (animationSpeed <= 0) {
		qWarning("TaskbarLayout::setFps: invalid fps %f", animationSpeed);
		return;
	}

	m_animationSpeed = animationSpeed;
}

void TaskbarLayout::setAnimationsEnabled(bool animationsEnabled) {
	m_animationsEnabled = animationsEnabled;

	if (!animationsEnabled) {
		skipAnimation();
	}
}

void TaskbarLayout::setMaximumRows(int maximumRows) {
	if (maximumRows < 1) {
		qWarning("TaskbarLayout::setMaximumRows: invalid maximumRows %d", maximumRows);
		return;
	}

	if (maximumRows != m_maximumRows) {
		m_maximumRows = maximumRows;
		if (m_minimumRows > maximumRows) {
			m_minimumRows = maximumRows;
		}
		invalidate();
	}
}

void TaskbarLayout::setMinimumRows(int minimumRows) {
	if (minimumRows < 1) {
		qWarning("TaskbarLayout::setMinimumRows: invalid minimumRows %d", minimumRows);
		return;
	}

	if (minimumRows != m_minimumRows) {
		m_minimumRows = minimumRows;
		if (m_maximumRows < minimumRows) {
			m_maximumRows = minimumRows;
		}
		invalidate();
	}
}

void TaskbarLayout::setRowBounds(int minimumRows, int maximumRows) {
	if (minimumRows < 1) {
		qWarning("TaskbarLayout::setRowBounds: invalid minimumRows %d", minimumRows);
		return;
	}

	if (minimumRows > maximumRows) {
		qWarning(
			"TaskbarLayout::setRowBounds: invalid row bounds: minimumRows: %d, maximumRows: %d",
			minimumRows, maximumRows);
		return;
	}
	
	if (minimumRows != m_minimumRows || maximumRows != m_maximumRows) {
		m_minimumRows = minimumRows;
		m_maximumRows = maximumRows;
		invalidate();
	}
}

void TaskbarLayout::setExpandedWidth(qreal expandedWidth) {
	if (expandedWidth < 0.0) {
		qWarning("TaskbarLayout::setExpandedSize: invalid expandedWidth %g", expandedWidth);
		return;
	}

	if (expandedWidth != m_expandedWidth) {
		m_expandedWidth = expandedWidth;
		invalidate();
	}
}

void TaskbarLayout::setExpandDuration(int expandDuration) {
	if (expandDuration < 0) {
		qWarning("TaskbarLayout::setExpandDuration: invalid expandDuration %d", expandDuration);
		return;
	}

	if (expandDuration != m_expandDuration) {
		m_expandDuration = expandDuration;
	}
}

QSizeF TaskbarLayout::sizeHint(Qt::SizeHint which, const QSizeF& constraint) const {
	switch (which) {
	case Qt::MinimumSize:
		return QSizeF(0.0, 0.0);
	case Qt::PreferredSize:
		// the value gets cached by effectiveSizeHint()
		{
			QSizeF rdConstraint;
			swapRdAndWorldSize(constraint, rdConstraint, m_orientation == Qt::Vertical);
			qreal maxRdHeigt = rdConstraint.height() > 0 ? rdConstraint.height() : std::numeric_limits<qreal>::infinity();
			qreal maxItemRdHeight;
			qreal rdWithSum = comulativePreferredItemRdWidthStatic(maxRdHeigt, true, 0, &maxItemRdHeight);
			rdWithSum += qMax(0.0, (qreal) (m_items.count() - minimumRows()) * m_spacing);
			QSizeF preferredSize = QSizeF(rdWithSum / (qreal) minimumRows(), (qreal) minimumRows() * maxItemRdHeight);
			swapRdAndWorldSize(preferredSize, preferredSize, m_orientation == Qt::Vertical);
			
			qreal left = 0, top = 0, right = 0, bottom = 0;
			getContentsMargins(&left, &top, &right, &bottom);
			preferredSize.setWidth(preferredSize.width() + left + right);
			preferredSize.setHeight(preferredSize.height() + top + bottom);
			
			qDebug() << "TaskbarLayout::sizeHint(Qt::PreferredSize," << constraint << ")" << preferredSize;
			
			return preferredSize;
		}
	case Qt::MaximumSize:
		return QSizeF(std::numeric_limits<qreal>::max(), std::numeric_limits<qreal>::max());
	case Qt::MinimumDescent:
		return QSizeF(0.0, 0.0);
	default:
		break;
	}

	return QSizeF();
}

void TaskbarLayout::setGeometry(const QRectF& rect) {
	if (rect != geometry())
		qDebug() << "TaskbarLayout::setGeometry" << rect << "(old:" << geometry() << ")";
	else
		qDebug() << "TaskbarLayout::setGeometry" << rect;
	
	static bool skipAni = false;
	skipAni = skipAni || (rect != geometry());
	
	QGraphicsLayout::setGeometry(rect);
	
	static bool geometryIsSet;
	geometryIsSet = false;
	if (m_preferredSizeChanged) {
		m_preferredSizeChanged = false;
		emit sizeHintChanged(Qt::PreferredSize);
		// maybe this emit causes another call of setGeometry
		// but then geometryIsSet will be true such that
		// the layouting is not done twice
	}
	if (geometryIsSet)
		return;
	
	updateLayoutStatic ();
	updateLayoutDynamic();
	if (skipAni) {
		skipAni = false;
		skipAnimation();
	}

	geometryIsSet = true;
}

QRectF TaskbarLayout::effectiveGeometry() const {
	QRectF effectiveRect(geometry());
	qreal left = 0, top = 0, right = 0, bottom = 0;
	getContentsMargins(&left, &top, &right, &bottom);
	
	if (QApplication::isRightToLeft()) {
		if (m_orientation == Qt::Vertical) {
			qSwap(top, bottom);
		}
		else {
			qSwap(left, right);
		}
	}
	effectiveRect.adjust(+left, +top, -right, -bottom);

	return effectiveRect;
}

qreal TaskbarLayout::comulativePreferredItemRdWidthStatic(const qreal maxRdHeight, const bool decomposeGroups, int *const count, qreal* maxItemRdHeight) const {
	int   cnt = 0;
	qreal sum = 0.0;
	qreal maxHeight = 0.0;
	foreach (TaskbarItem* item, m_items) {
		if (decomposeGroups && item->item->abstractGroupableItem()->isGroupItem()) {
			Applet& applet = *item->item->applet();
			foreach(TaskManager::AbstractGroupableItem* subitem, static_cast<const TaskManager::TaskGroup*>(item->item->abstractGroupableItem())->members()) {
				QSizeF size = TaskItem::preferredRdSizeStatic(*subitem, applet, *applet.frame(), m_orientation, maxRdHeight);
				sum += size.width();
				if (size.height() > maxHeight)
					maxHeight = size.height();
				++cnt;
			}
		} else {
			QSizeF size = item->item->preferredRdSizeStatic(maxRdHeight);
			sum += size.width();
			if (size.height() > maxHeight)
				maxHeight = size.height();
			++cnt;
		}
	}
	
	if (count)
		*count = cnt;
	if (maxItemRdHeight)
		*maxItemRdHeight = maxHeight;
	return sum;
}

qreal TaskbarLayout::averagePreferredItemRdWidthStatic(const qreal maxRdHeight, const bool decomposeGroups, int *const count, qreal* maxItemRdHeight) const {
	int cnt;
	qreal averageItemRdWidth = comulativePreferredItemRdWidthStatic(maxRdHeight, decomposeGroups, &cnt, maxItemRdHeight);
	if (cnt > 0)
		averageItemRdWidth /= (qreal) cnt;
	if (averageItemRdWidth < 1.0)
		averageItemRdWidth = 1.0;
	
	if (count)
		*count = cnt;
	return averageItemRdWidth;
}

void TaskbarLayout::buildRows(const int itemsPerRow, QList<RowInfo>& rowInfos, int& rows) const {
	const int N = m_items.size();

	rowInfos.clear();

	int startIndex = 0;
	int endIndex   = 0;

	for (int row = 0; row < rows && endIndex < N; ++ row) {
		startIndex = endIndex;

		if (row + 1 == rows) {
			endIndex = N;
		}
		else {
			endIndex = qMin(startIndex + itemsPerRow, N);
		}

		rowInfos.append(RowInfo(startIndex, endIndex));
	}

	// if we assumed expanded there still might be empty row
	// therefore just scale up the layout (maybe do only this and not the other way of row removal)
	rows = qMax(m_minimumRows, rowInfos.size());
}

void TaskbarLayout::updateItemsRowCache() {
	int rows = m_rowInfos.size();
	for (int row = 0; row < rows; ++row) {
		int start = m_rowInfos[row].startIndex;
		int end   = m_rowInfos[row].endIndex;
		for (int i = start; i < end; ++i)
			m_items[i]->row = row;
	}
}

void TaskbarLayout::rdToWorld(const QRectF& src, QRectF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const {
	if (rtl) {
		if (isVertical) {  // rtl and vertical
			dst.setRect(effectiveRect.left() + src.top(), effectiveRect.top() + src.left(), src.height(), src.width());
		} else {  // rtl
			dst.setRect(effectiveRect.right() - src.right(), effectiveRect.top() + src.top(), src.width(), src.height());
		}
	} else {
		if (isVertical) {  // vertical
			dst.setRect(effectiveRect.left()  + src.top(), effectiveRect.bottom() - src.right(), src.height(), src.width());
		} else {  // normal
			dst.setRect(effectiveRect.left()  + src.left(), effectiveRect.top() + src.top(), src.width(), src.height());
		}
	}
}

void TaskbarLayout::rdToWorld(const QPointF& src, QPointF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const {
	if (rtl) {
		if (isVertical) {  // rtl and vertical
			dst.setX(effectiveRect.left()  + src.y()); dst.setY(effectiveRect.top()    + src.x());
		} else {  // rtl
			dst.setX(effectiveRect.right() - src.x()); dst.setY(effectiveRect.top()    + src.y());
		}
	} else {
		if (isVertical) {  // vertical
			dst.setX(effectiveRect.left()  + src.y()); dst.setY(effectiveRect.bottom() - src.x());
		} else {  // normal
			dst.setX(effectiveRect.left()  + src.x()); dst.setY(effectiveRect.top()    + src.y());
		}
	}
}

void TaskbarLayout::worldToRd(const QRectF& src, QRectF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const {
	if (rtl) {
		if (isVertical) {  // rtl and vertical
			dst.setRect(src.top() - effectiveRect.top(), src.left() - effectiveRect.left(), src.height(), src.width());
		} else {  // rtl
			dst.setRect(effectiveRect.right() - src.right(), src.top() - effectiveRect.top(), src.width(), src.height());
		}
	} else {
		if (isVertical) {  // vertical
			dst.setRect(effectiveRect.bottom() - src.bottom(),  src.left() - effectiveRect.left(), src.height(), src.width());
		} else {  // normal
			dst.setRect(src.left() - effectiveRect.left(), src.top() - effectiveRect.top(), src.width(), src.height());
		}
	}
}

void TaskbarLayout::worldToRd(const QPointF& src, QPointF& dst, const QRectF& effectiveRect, bool rtl, bool isVertical) const {
	if (rtl) {
		if (isVertical) {  // rtl and vertical
			dst.setX(src.y() - effectiveRect.top()); dst.setY(src.x() - effectiveRect.left());
		} else {  // rtl
			dst.setX(effectiveRect.right() - src.x()); dst.setY(src.y() - effectiveRect.top());
		}
	} else {
		if (isVertical) {  // vertical
			dst.setX(effectiveRect.bottom() - src.y()); dst.setY(src.x() - effectiveRect.left());
		} else {  // normal
			dst.setX(src.x() - effectiveRect.left()); dst.setY(src.y() - effectiveRect.top());
		}
	}
}

void TaskbarLayout::swapRdAndWorldSize(const QSizeF& src, QSizeF& dst, bool isVertical) const {
	dst = src;
	if (isVertical)
		dst.transpose();
}

void TaskbarLayout::updateLayoutDynamic() {
	qDebug("TaskbarLayout::updateLayoutDynamic");
	// before updating the geometries of the items set the properties
	// that might get read by the items in their event handlers:
	const bool  isVertical  = m_orientation == Qt::Vertical;
	const qreal spacing     = m_spacing;

	const QRectF& effectiveRect = effectiveGeometry();
	const qreal availableRdWidth = isVertical ? effectiveRect.height() : effectiveRect.width();
	
	const bool rtl = QApplication::isRightToLeft();
	
	// "rd" stands for "reading direction"
	qreal currentRdY = 0.0;
	
	for (int row = 0; row < m_rowInfos.size(); ++ row) {
		const RowInfo& rowInfo = m_rowInfos[row];
		
		int rdWidthsSize = rowInfo.endIndex - rowInfo.startIndex;
		QVector<qreal> rdWidths(rdWidthsSize);
		qreal rdWidthSum = 0;
		for (int index = rowInfo.startIndex; index < rowInfo.endIndex; ++ index) {
			qreal rdWidth;
			if (m_draggedItem)
				rdWidth = m_items[index]->item->preferredRdSizeStatic(m_cellHeight).width();
			else
				rdWidth = m_items[index]->item->preferredRdWidthDynamic(m_cellHeight);
			rdWidths[index - rowInfo.startIndex] = rdWidth;
			rdWidthSum += rdWidth;
		}
		
		qreal currentSpacing = spacing;
		qreal rowSpacings = (rowInfo.endIndex - rowInfo.startIndex - 1) * currentSpacing;
		
		qreal availableRdWidthSum = availableRdWidth - rowSpacings;
		if (availableRdWidthSum <= 0) {
			if (availableRdWidth > 0)
				availableRdWidthSum = availableRdWidth;
			else
				availableRdWidthSum = 0;
			currentSpacing = 0;
			rowSpacings = 0;
		}
		
		// calculate fitting rdWidths
		rdWidthSum = ::SmoothTasks::squeezeWidths(rdWidths, availableRdWidthSum);
		
		// apply rdWidths
		qreal currentRdX = 0; 
		for (int index = rowInfo.startIndex; index < rowInfo.endIndex; ++ index) {
			TaskbarItem *item = m_items[index];
			int i = index - rowInfo.startIndex;
			
			if (!item->item->isVisible()) {
				qDebug() << "new task button:" << item->item->task()->text();
				if (i > 0) {
					QRectF geo(m_items[index - 1]->item->geometry());
					worldToRd(geo, geo, effectiveRect, rtl, isVertical);
					geo.setRect(geo.right() + 0.5 * currentSpacing, geo.top(), 1.0, m_cellHeight);
					rdToWorld(geo, geo, effectiveRect, rtl, isVertical);
					item->item->setGeometry(geo);
				} else {
					QRectF geo(0.0, 0.0, 1.0, m_cellHeight);
					rdToWorld(geo, geo, effectiveRect, rtl, isVertical);
					item->item->setGeometry(geo);
				}
				item->item->setVisible(true);
			}
			
			QRectF targetRect(currentRdX, currentRdY, rdWidths[i], m_cellHeight);
			rdToWorld(targetRect, targetRect, effectiveRect, rtl, isVertical);
			if (m_draggedItem == item) {
				targetRect.moveTo(item->item->pos());
				item->item->setGeometry(targetRect);
			}
			item->item->setAnimationTargetGeometry(targetRect);
			
			currentRdX += rdWidths[i] + currentSpacing;
		}
		
		currentRdY += m_cellHeight + spacing;
	}
	
	startAnimation();
}

void TaskbarLayout::preferredItemSizeStaticChange(TaskItem* item) {
	Q_UNUSED(item);
	qDebug("TaskbarLayout::preferredItemSizeStaticChange");
	m_preferredSizeChanged = true;
	invalidate();
}

void TaskbarLayout::preferredItemSizeDynamicChange(TaskItem* item) {
	Q_UNUSED(item);
	qDebug("TaskbarLayout::preferredItemSizeDynamicChange");
	if (m_draggedItem)
		return;
	updateLayoutDynamic();
}

TaskItem *TaskbarLayout::itemAt(int index) const {
	if (index < 0 || index >= m_items.size()) {
		qWarning("TaskbarLayout::itemAt: invalid index %d", index);
		return NULL;
	}
	return m_items[index]->item;
}

int TaskbarLayout::addItem(TaskItem *item) {
	const int index = count();
	insertItem(index, item);
	return index;
}

void TaskbarLayout::insertItem(int index, TaskItem *item) {
	if (!item) {
		qWarning("TaskbarLayout::insertItem: cannot insert null item");
		return;
	}
	if (indexOf(item) != -1) {
		qWarning("TaskbarLayout::insertItem: cannot instert same item twice");
		return;
	}
	item->setVisible(false);  // make the button invisible until it gets a valid resonable in updateLayoutDynamic()
	item->setParentLayoutItem(this);
	TaskbarItem *titem = new TaskbarItem(item);
	m_items.insert(index, titem);

	item->setOrientation(m_orientation);
	connectItem(item);

	invalidate();
}

void TaskbarLayout::connectItem(TaskItem *item) {
	connect(item, SIGNAL(preferredSizeStaticChange(TaskItem*)), this, SLOT(preferredItemSizeStaticChange(TaskItem*)));
	connect(item, SIGNAL(preferredSizeDynamicChange(TaskItem*)), this, SLOT(preferredItemSizeDynamicChange(TaskItem*)));
}

void TaskbarLayout::move(int fromIndex, int toIndex) {
	if (fromIndex < 0 || fromIndex >= m_items.size()) {
		qWarning("TaskbarLayout::move: invalid fromIndex %d", fromIndex);
		return;
	}

	if (toIndex < 0 || toIndex >= m_items.size()) {
		qWarning("TaskbarLayout::move: invalid toIndex %d", toIndex);
		return;
	}

	m_items.move(fromIndex, toIndex);
	invalidate();
}

void TaskbarLayout::removeAt(int index) {
	if (index < 0 || index >= m_items.size()) {
		qWarning("TaskbarLayout::removeAt: invalid index %d", index);
		return;
	}

	TaskbarItem *item = m_items.takeAt(index);

	if (m_draggedItem == item) {
		m_currentIndex  = -1;
		m_draggedItem   = NULL;
	}

	disconnectItem(item->item);

	delete item;

	invalidate();
}

void TaskbarLayout::removeItem(TaskItem *item) {
	if (item == NULL) {
		qWarning("TaskbarLayout::removeItem: cannot remove null item");
		return;
	}

	removeAt(indexOf(item));
}

TaskItem *TaskbarLayout::itemAt(const QPointF& pos) const {
	const qreal halfSpacing = m_spacing * 0.5;

	foreach (TaskbarItem *item, m_items) {
		QRectF rect = item->item->geometry();
		qreal y = rect.y();
		qreal x = rect.x();
		if (
				pos.y() >= (y - halfSpacing) && pos.y() < (y + rect.height() + halfSpacing) &&
				pos.x() >= (x - halfSpacing) && pos.x() < (x + rect.width()  + halfSpacing)) {
			return item->item;
		}
	}

	return NULL;
}

int TaskbarLayout::rowOf(TaskItem *item) const {
	if (item == NULL) {
		qWarning("TaskbarLayout::rowOf: item cannot be null");
		return -1;
	}

	foreach (TaskbarItem* titem, m_items) {
		if (titem->item == item) {
			return titem->row;
		}
	}

	qWarning("TaskbarLayout::rowOf: not a child item");
	return -1;
}

int TaskbarLayout::rowOf(int index) const {
	if (index < 0 || index >= m_items.size()) {
		qWarning("TaskbarLayout::rowOf: invalid index %d", index);
		return -1;
	}

	return m_items[index]->row;
}

int TaskbarLayout::rowOf(const QPointF& pos) const {
	qDebug() << "TaskbarLayout::rowOf" << pos << " m_rows = " << m_rows;
	QRectF effectiveRect(effectiveGeometry());

	if (m_orientation == Qt::Vertical) {
		qreal x = pos.x();
		if (x <= effectiveRect.left()) {
			return 0;
		}
		else if (x >= effectiveRect.right() || effectiveRect.width() == 0) {
			return m_rows - 1;
		}
		else {
			return (int) ((x - effectiveRect.left()) * m_rows / effectiveRect.width());
		}
	}
	else {
		qreal y = pos.y();
		if (y <= effectiveRect.top()) {
			return 0;
		}
		else if (y >= effectiveRect.bottom() || effectiveRect.height() == 0) {
			return m_rows - 1;
		}
		else {
			return (int) ((y - effectiveRect.top()) * m_rows / effectiveRect.height());
		}
	}
}

int TaskbarLayout::indexOf(const QPointF& pos, int *rowPtr, bool *overLeftPartPtr) const {
	const QRectF effectiveRect(effectiveGeometry());
	const int    row         = rowOf(pos);
	const bool   isVertical  = m_orientation == Qt::Vertical;

	if (row >= m_rowInfos.size()) {
		if (rowPtr         ) *rowPtr          = qMax(0, m_rowInfos.size() - 1);
		if (overLeftPartPtr) *overLeftPartPtr = false;
		return qMax(0, m_items.size() - 1);
	}
	
	const qreal rdPosX = isVertical ? pos.y() : pos.x();
	
	qreal minDistance  = std::numeric_limits<qreal>::infinity();
	int   index        = m_rowInfos[row].startIndex;
	bool  overLeftPart = !QApplication::isRightToLeft();
	
	for (int i = m_rowInfos[row].startIndex; i < m_rowInfos[row].endIndex; ++i) {
		if (m_items[i] == m_draggedItem)
			continue;
		const QRectF animationTargetGeometry = m_items[i]->item->animationTargetGeometry();
		qreal rdItemPosX = isVertical ? animationTargetGeometry.bottom() : animationTargetGeometry.left();
		qreal distance = std::abs(rdItemPosX - rdPosX);
		if (minDistance > distance) {
			minDistance = distance;
			index = i;
			overLeftPart = true;
		}
		rdItemPosX = isVertical ? animationTargetGeometry.top() : animationTargetGeometry.right();
		distance = std::abs(rdItemPosX - rdPosX);
		if (minDistance > distance) {
			minDistance = distance;
			index = i;
			overLeftPart = false;
		}
	}
	
	if (QApplication::isRightToLeft())
		overLeftPart = !overLeftPart;
	
	if (rowPtr         ) *rowPtr          = row;
	if (overLeftPartPtr) *overLeftPartPtr = overLeftPart;
	return index;
}

int TaskbarLayout::indexOf(TaskItem *item) const {
	const int N = m_items.size();
	
	for (int index = 0; index < N; ++ index) {
		if (m_items[index]->item == item) {
			return index;
		}
	}

	return -1;
}

void TaskbarLayout::takeFrom(TaskbarLayout *other) {
	Q_ASSERT(other != NULL);

	if (other == this) {
		return;
	}

	m_currentIndex     = other->m_currentIndex;
	m_draggedItem      = other->m_draggedItem;
	m_mouseIn          = other->m_mouseIn;
	m_grabPos          = other->m_grabPos;
	m_items.append(other->m_items);

	foreach (TaskbarItem *item, other->m_items) {
		item->item->setParentLayoutItem(this);
		other->disconnectItem(item->item);
		connectItem(item->item);
	}

	other->m_draggedItem  = NULL;
	other->m_currentIndex = -1;
	other->m_mouseIn      = false;
	other->m_items.clear();
	other->stopAnimation();

	invalidate();
}

int TaskbarLayout::dragItem(TaskItem *item, QDrag *drag, const QPointF& pos) {
	qDebug("TaskbarLayout::dragItem");
	if (m_draggedItem != NULL) {
		qWarning("TaskbarLayout::dragItem: already dragging");
		return -1;
	}

	int index = indexOf(item);

	if (index == -1) {
		qWarning("TaskbarLayout::dragItem: invalid item");
		return -1;
	}

	m_mouseIn      = true;
	m_draggedItem  = m_items[index];
	m_currentIndex = index;
	m_grabPos      = pos - m_draggedItem->item->geometry().topLeft();

	bool enabled = m_draggedItem->item->graphicsItem()->isEnabled();

	m_draggedItem->item->graphicsItem()->setZValue(1);
	m_draggedItem->item->graphicsItem()->setEnabled(false);

	int dropIndex = index;

	if (drag->exec(Qt::MoveAction) == Qt::IgnoreAction || drag->target() == drag->source()) {
		dropIndex = currentDragIndex();
	}
	
	// free item for layouting
	TaskbarItem* draggedItem = m_draggedItem;
	m_currentIndex = -1;
	m_draggedItem  = NULL;
	
	if (draggedItem == NULL) {
		qDebug("TaskbarLayout::dragItem: item was deleted during dragging");
	}
	else if (draggedItem->item != item) {
		qWarning(
			"TaskbarLayout::dragItem: dragged item changed during dragging!?\n"
			"This _might_ cause a memleak under some circumstances.");
		return -1;
	}
	else {
		draggedItem->item->graphicsItem()->setZValue(0);
		draggedItem->item->graphicsItem()->setEnabled(enabled);
		if (dropIndex >= 0) {
			// move dropped item animated to dest:
			invalidate();
		}
	}

	return dropIndex;
}

void TaskbarLayout::moveDraggedItem(const QPointF& pos) {
	qDebug("TaskbarLayout::moveDraggedItem");
	if (m_draggedItem == NULL) {
		return;
	}

	m_mouseIn = true;
	QRectF effectiveRect(effectiveGeometry());

	QRectF rect(m_draggedItem->item->geometry());
	if (m_grabPos.y() > rect.height()) {
		m_grabPos.setY(rect.height() * 0.5);
	}
	
	if (m_grabPos.x() > rect.width()) {
		m_grabPos.setX(rect.width() * 0.5);
	}

	QPointF newPos(pos - m_grabPos);

	if (newPos.y() < effectiveRect.top()) {
		newPos.setY(effectiveRect.top());
	}
	else if (newPos.y() + rect.height() > effectiveRect.bottom()) {
		newPos.setY(effectiveRect.bottom() - rect.height());
	}
	
	if (newPos.x() < effectiveRect.left()) {
		newPos.setX(effectiveRect.left());
	}
	else if (newPos.x() + rect.width() > effectiveRect.right()) {
		newPos.setX(effectiveRect.right() - rect.width());
	}

	rect.moveTopLeft(newPos);

	m_draggedItem->item->setGeometry(rect);
	m_draggedItem->item->setAnimationTargetGeometry(rect);

	int  row;
	bool overLeftPart;
	int  index = indexOf(pos, &row, &overLeftPart);
	
	if (index == m_currentIndex) {
		return;
	} else if (index < m_currentIndex) {
		if (!overLeftPart)
			index = qMax(0, qMin(m_items.size() - 1, index + 1));
	} else {
		if (overLeftPart)
			index = qMax(0, index - 1);
	}
	
	if (index == m_currentIndex)
		return;
	
	qDebug() << "TaskbarLayout::moveDraggedItem()" << "from" << m_currentIndex << "to" << index;
	m_items.move(m_currentIndex, index);
	m_currentIndex     = index;
	m_draggedItem->row = row;
	invalidate();
}

void TaskbarLayout::dragLeave() {
	qDebug("TaskbarLayout::dragLeave");
	if (m_draggedItem == NULL) {
		return;
	}

	m_mouseIn = false;
	startAnimation();
}

void TaskbarLayout::animate() {
	qreal dt = 0.001 * (qreal) m_animationFrameTimer.restart();
	
	bool stopAni = true;
	
	foreach (TaskbarItem *item, m_items)
		stopAni &= item->item->animateStep(m_animationSpeed, dt);

	if (stopAni)
		stopAnimation();
}

void TaskbarLayout::disconnectItem(TaskItem *item) {
	disconnect(item, SIGNAL(preferredSizeStaticChange(TaskItem*)), this, SLOT(preferredItemSizeStaticChange(TaskItem*)));
	disconnect(item, SIGNAL(preferredSizeDynamicChange(TaskItem*)), this, SLOT(preferredItemSizeDynamicChange(TaskItem*)));
}

void TaskbarLayout::startAnimation() {
	if (m_animationsEnabled && !m_animationTimer->isActive()) {
		m_animationFrameTimer.start();
		m_animationTimer->start();
	}
}

void TaskbarLayout::stopAnimation() {
	m_animationTimer->stop();
}

void TaskbarLayout::skipAnimation() {
	stopAnimation();

	foreach (TaskbarItem *item, m_items)
		if (item != m_draggedItem || !m_mouseIn)
			item->item->skipAnimation();
}

TaskItem *TaskbarLayout::draggedItem() const {
	if (m_draggedItem) {
		return m_draggedItem->item;
	}
	else {
		return NULL;
	}
}

void TaskbarLayout::clear(bool forceDeleteItems) {
	stopAnimation();

	while (!m_items.isEmpty()) {
		TaskbarItem *item = m_items.takeLast();
		TaskItem *titem = item->item;

		if (titem != NULL) {
			disconnectItem(titem);
			if (forceDeleteItems && !titem->ownedByLayout()) {
				delete titem;
				item->item = NULL;
			}
		}
		delete item;
	}

	if (m_draggedItem) {
		m_currentIndex = -1;
		m_draggedItem  = NULL;
	}
}

} // namespace SmoothTasks
