/*
 *  Copyright 2011 Wolfgang Koller - http://www.gofg.at/
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#ifndef FILEAPI_H_SDASDASDAS
#define FILEAPI_H_SDASDASDAS

#include <QNetworkReply>
#include <QtCore>

#include "../cplugin.h"
#include "../cordova.h"

class QNetworkReply;

class FileTransferRequest: public QObject {
    Q_OBJECT

    QNetworkAccessManager &m_manager;
    int m_scId, m_ecId, m_progressId;
    int m_id;
    QSharedPointer<QNetworkReply> m_reply;
    FileTransferRequest* operator=(const FileTransferRequest&) = delete;
    FileTransferRequest(FileTransferRequest&) = delete;
public:
    FileTransferRequest(QNetworkAccessManager &manager, int scId, int ecId, int progressId, int id, CPlugin *plugin):
        m_manager(manager),
        m_scId(scId),
        m_ecId(ecId),
        m_progressId(progressId),
        m_id(id),
        m_plugin(plugin) {
    }

    void download(const QString& url);
    void upload(const QString& _url, const QString& content, QString fileKey, QString fileName, QString mimeType, const QVariantMap &params, const QVariantMap &headers);
    void abort();

signals:
    void done();

private slots:
    void progress(qint64 bytesReceived, qint64 bytesTotal);
    void error(QNetworkReply::NetworkError code);
private:
    CPlugin *m_plugin;
};

class FileAPI : public CPlugin {
    Q_OBJECT
public:
    explicit FileAPI(Cordova *cordova);

    virtual const QString fullName() override {
        return FileAPI::fullID();
    }

    virtual const QString shortName() override {
        return "File";
    }

    static const QString fullID() {
        return "com.cordova.File";
    }

public slots:
    void requestFileSystem(int scId, int ecId, unsigned short p_type, unsigned long long p_size);
    void resolveLocalFileSystemURL(int scId, int ecId, QString p_url);
    void getDirectory(int scId, int ecId, QString p_path, QVariantMap p_options);
    void getFile(int scId, int ecId, QString p_path, QVariantMap p_options);
    void getParent(int scId, int ecId, QString p_path);
    void getMetadata(int scId, int ecId, QString p_path);
    void remove(int scId, int ecId, QString p_path);
    void removeRecursively(int scId, int ecId, QString p_path);
    void file(int scId, int ecId, QString p_path);
    void write(int scId, int ecId, QString p_path, unsigned long long p_position, QString p_data, bool binary);
    void truncate(int scId, int ecId, QString p_path, unsigned long long p_size);
    void readEntries(int scId, int ecId, QString p_path);
    void readAsDataURL(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd);
    void readAsText(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd);
    void readAsBinaryString(int scId, int ecId, QString p_path, bool sliced, int sliceStart, int sliceEnd);
    bool copyFile(int scId, int ecId,const QString& sourceFile, const QString& destinationParentDir, const QString& newName);
    void moveFile(int scId, int ecId,const QString& sourceFile, const QString& destinationParentDir, const QString& newName);
    void copyDir(int scId, int ecId,const QString& sourceFolder, const QString& destFolder, const QString& newName);
    void moveDir(int scId, int ecId,const QString& sourceFolder, const QString& destFolder, const QString& newName);

    void newTransferRequest(int scId, int ecId);
    void transferRequestSetOnProgress(int scId, int ecId, int id);
    void abortRequests(int scId, int ecId, int id);
    void downloadFile(int scId, int ecId, int id, const QString& url);
    void uploadFile(int scId, int ecId, int id, const QString& url, const QString& content, QString fileKey, QString fileName, QString mimeType, const QVariantMap &params, const QVariantMap &headers);

protected:
    bool rmDir(QDir p_dir);
    bool copyFolder(const QString& sourceFolder, const QString& destFolder);

private:
    typedef QMap<QString, QString> MimeTypeMap;
    static MimeTypeMap createMimeTypeMap();
    static MimeTypeMap mimeMap_;

    const QString m_persistentDir;
    QNetworkAccessManager m_manager;
    QMultiMap<int, QSharedPointer<FileTransferRequest> > m_id2request;
    QMap<int, int> m_id2progress;
    int lastRequestId;
};

#endif
