/*
 * IRiver ifp supporting functions, a basic (primative) API.
 * $Id: userfile.c,v 1.4 2005/10/11 01:53:18 jim-campbell Exp $
 *
 * Copyright (C) Geoff Oakham, 2004; <oakhamg@users.sourceforge.net>
 */

static const char rcsid[] = "$Id: ";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <fts.h>

#include "ifp.h"
#include "ifp_os.h"
#include "prim.h"

//semi-constructor
static struct ifp_transfer_status * init_progress(
	struct ifp_transfer_status * p,
	ifp_progress fn, void * fn_context)
{
	if (fn == NULL) {
		return NULL;
	} else {
		p->batch_bytes = 0;
		p->batch_total = 0;
		p->file_bytes = 0;
		p->file_total = 0;
		p->file_name = NULL;
		p->files_count = 0;
		p->files_total = 1;
		p->is_batch = 0;

		p->reserved1 = fn;
		p->reserved2 = fn_context;
		p->reserved3 = NULL;
		return p;
	}
}

/** User callback can return 1 to cancel the transfer.  This get translated
 * into IFP_ERR_USER_CANCEL. */
static inline int update_progress(struct ifp_transfer_status * p, int freshbytes) {
	int i = 0;
	if (p == NULL) {
		return 0;
	} else {
		ifp_progress fn = p->reserved1;
		p->file_bytes += freshbytes;
		p->batch_bytes += freshbytes;
		if (fn == NULL) {
			ifp_err("fn is NULL!");
			i = -1;
			goto out;
		}
		i = fn(p->reserved2, p);
		if (i == 1) {
			i = IFP_ERR_USER_CANCEL;
		}
	    	ifp_err_expect(i, i==IFP_ERR_USER_CANCEL, out, "progress callback error\n");
	}
out:
	return i;
}

/**
 Returns -ENOENT if 'f' doesn't exist; -EACCES if 'f' is read-protected by the device;
 and IFP_ERR_USER_CANCEL is the callback requested a cancel.
 Also expect -ENOSPC, in case the local disc fills.
 */
int _ifp_read_stream_progress(struct ifp_device * dev,
	FILE *dst, const char *f, struct ifp_transfer_status * p)
{

	int i = 0;
	int e = 0;
	unsigned char buf[IFP_BULK_BUFF_SIZE];

	i = ifp_read_open(dev, f);
	ifp_err_expect(i, i==-ENOENT || i==-EACCES, out0,
		"opening file ifp:\\%s\n", f);

	if (p) {
		p->file_total = (int)ifp_read_size(dev);
	}
	while (!ifp_read_eof(dev)) {
		int j;
		i = ifp_read_data(dev, buf, IFP_BULK_BUFF_SIZE);
		if (i < 0){
			e = i;
			//quiet on pipe errors
			if (!(i == -EIO && dev->download_pipe_errors > 0)) {
				ifp_err_i(i, "read data error on file ifp:\\%s\n", f);
			}
			goto out1;
		} else if (i == 0) {
			ifp_wrn("got 0 bytes.. should that happen?  The reported "
				"filesize is %d and current pos is %d\n",
				(int)dev->filesize, (int)dev->current_offset);
			//done
			break;
		}

		j = fwrite(buf,1,i,dst);
		if (i != j) {
			e = errno > 0 ? -errno : -1;
			if (e != -ENOSPC) {
				ifp_wrn("error writing downloaded file: %d bytes written instead of %d. error = %d",
					j, i, e);
			}
			goto out1;
		}

		i = update_progress(p, j);
		ifp_err_expect(i, i==IFP_ERR_USER_CANCEL, out1, "progress callback error\n");
	}

out1:
	e = ifp_read_close(dev);
	if (e)  {
		ifp_err_i(e, "close error on file ifp:\\%s\n", f);
	}

out0:
	return e?e:i;
}

/**
  Returns -EEXIST, -ENOENT, -ENOSPC, IFP_ERR_BAD_FILENAME, IFP_ERR_USER_CANCEL
 */
static int _ifp_write_stream_progress(struct ifp_device * dev,
	FILE *src, int filesize, const char *f,
	struct ifp_transfer_status * p)
{
	int i = 0;
	int e = 0;
	unsigned char buf[IFP_BULK_BUFF_SIZE];

	i = ifp_write_open(dev, f, filesize);
	ifp_err_expect(i, i==-ENOENT || i==-EEXIST || i==-ENOSPC || i==IFP_ERR_BAD_FILENAME,
		out0, "opening new file ifp:\\%s", f);

	while (filesize > 0) {
		int step;
		int j;
		step = min(filesize,IFP_BULK_BUFF_SIZE);
		j = fread(buf,1,step,src);
		if (j <= 0 || j > step) {
			ifp_err("Read error from src.  Got only %d bytes instead of %d as requested.",
				j, step);
			goto out1;
		}
		filesize -= j;

		i = ifp_write_data(dev, buf, j);
		ifp_err_jump(i, out1, "write data error on file ifp:\\%s", f);

                i = update_progress(p, j);
		ifp_err_expect(i, i==IFP_ERR_USER_CANCEL, out1, "progress callback error\n");
	}

out1:
	e = ifp_write_close(dev);
	if (e)  {
		ifp_err_i(e, "close error on file ifp:\\%s", f);
	}
	if (i == IFP_ERR_USER_CANCEL) {
		e = ifp_delete(dev, f);
		if (e) {
			ifp_err_i(e, "error attempting to delete parcially written file ifp:\\%s", f);
		}
	}

out0:
	return e?e:i;
}

static int old_style_progress(void * context, struct ifp_transfer_status * p) {
	int i = 0;

	int(*fn)(void*,int);
	fn = NULL;
	if (p == NULL) {
		ifp_err("p is NULL!");
		i = -1;
		goto err;
	}

	fn = p->reserved3;
	if (fn == NULL) {
		ifp_err("fn is NULL!");
		i = -1;
		goto err;
	}

	i = fn(context, p->file_bytes);
	ifp_err_expect(i, i==1, err, "err from progress callback");

err:
	return i;
}
static struct ifp_transfer_status * init_progress_cludge(
	struct ifp_transfer_status * p, int(*fn)(void*,int), void * context)
{
	if (fn == NULL) {
		return NULL;
	} else {
		p = init_progress(p, old_style_progress, context);
		if (p == NULL) {
			//shouldn't happen
			ifp_err("shouldn't be here");
			return NULL;
		}
		p->reserved3 = fn;
		return p;
	}
}

/** \brief Downloads a file; includes a hook for a progress metre.

  Reads the file 'f' from the device and saves it in 'dst'.
  \param f         name of the remote file we're downloading
  \param dst       where the data will be saved
  \param progress  Optional.  If given, this function will be called
                   occationally so an application can update a progress
		   metre.  (For example.)
  \param context   Context for the progress metre.  (Safe to leave NULL.)

  (Available only in userland.)

  Returns -ENOENT if 'f' doesn't exist; -EACCES if 'f' is read-protected by the device;
  -ENOSPC; and ::IFP_ERR_USER_CANCEL if the callback requested the transfer cancelled.

  (Available only in userland.)
 */
int ifp_read_file_progress(struct ifp_device * dev,
	FILE *dst, const char *f, int(*fn)(void*,int), void * fn_context)
{

	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;

	p = init_progress_cludge(&progress, fn, fn_context);

	i = _ifp_read_stream_progress(dev, dst, f, p);
	ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL,
		err, "error reading into stream");

err:
	return i;
}
IFP_EXPORT(ifp_read_file_progress);

/** \brief Uploads a file; includes a hook for a progress metre.
 Creates a new file 'f' on the device and populates it with data from 'src'.
 Filesize is the number of bytes to be uploaded from 'src'.

  (Note: it appears the device might not need to know the number of bytes
  in a file ahead time.  The current implementation doesn't support this,
  but if you don't have access to the filesize ahead of time, you might
  be able to hack libifp to let you do it anyways.)

  \param src data to be uploaded
  \param filesize  number of bytes to copy from src
  \param f         name of the file to be created
  \param progress  Optional.  If given, this function will be called
                   occationally so an application can update its progress
		   metre.
  \param context   Context for the progress metre.  (Safe to leave NULL.)

  Returns -EEXIST, -ENOENT, -ENOSPC, ::IFP_ERR_BAD_FILENAME, ::IFP_ERR_USER_CANCEL

  (Available only in userland.)
 */
int ifp_write_file_progress(struct ifp_device * dev,
    FILE *src, int filesize, const char *f,
    int(*fn)(void*,int), void * fn_context)
{
	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;

	p = init_progress_cludge(&progress, fn, fn_context);

	i = _ifp_write_stream_progress(dev, src, filesize, f, p);
	ifp_err_expect(i, i==-EEXIST||i==-ENOENT||i==-ENOSPC
		||i==IFP_ERR_BAD_FILENAME||i==IFP_ERR_USER_CANCEL, err,
		"error reading into stream");
err:
    return i;
}
IFP_EXPORT(ifp_write_file_progress);

/** Returns -ENOENT, -EACCES, -ENOSPC, IFP_ERR_USER_CANCEL */
static int _download_file(struct ifp_device * dev,
	const char * remotefile, const char * localfile,
	struct ifp_transfer_status * status)
{
	int i=0;
	FILE * f = NULL;
	int tries = 8;

	if (status) {
		status->file_name = remotefile;
		status->file_bytes = 0;
	}
	if (dev->download_pipe_errors > 0) {
		ifp_dbg("resetting pipe count to 0.  Was %d", dev->download_pipe_errors);
	}
	dev->download_pipe_errors = 0;

	f = fopen(localfile, "wb");
	if (f == NULL) {
		ifp_err("could not open '%s' for writing", localfile);
		return -EIO;
	}

	//Note: this is a work-around for the EPIPE bug.  We can tell a
	//transfer failed because of that bug because i==-EIO and the
	//download_pipe_errors counter is non-zero.
	do {
		if (dev->download_pipe_errors > 0) {
			dev->download_pipe_errors = 0;
			if (i == -EIO) {
				//last loop failed because of a pipe error.
				i = fseek(f, 0, SEEK_SET);
				if (i) {
					i = -errno;
					ifp_err_i(i, "seek failed to rewind file");
					goto err;
				}
				if (status) {
					status->batch_bytes -= status->file_bytes;
					status->file_bytes = 0;
				}
			}
		}
		i = _ifp_read_stream_progress(dev, f, remotefile, status);
		//quiet on pipe errors
		if (!(i == -EIO && dev->download_pipe_errors > 0)) {
			ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL,
				err, "problem reading.. ifp:\\%s", remotefile);
		} else {
			//ifp_dbg("retrying");
		}
		tries--;
	} while (i==-EIO && dev->download_pipe_errors > 0 && tries > 0);

	if (tries == 0 && i==-EIO) {
		ifp_err("Download failed because of the pipe bug.  (I tried several times before giving up.)");
	}

	fclose(f); f = NULL;
	return i;

err:
	fclose(f); f = NULL;
	if (remove(localfile)) {
		ifp_err("couldn't remove %s", localfile);
	}

	return i;
}

/** Returns -EEXIST, -ENOENT, -ENOSPC, IFP_ERR_BAD_FILENAME, or IFP_ERR_USER_CANCEL. */
static int _upload_file(struct ifp_device * dev,
	const char * localfile, const char * remotefile,
	size_t filesize, struct ifp_transfer_status * status)
{
	int i;
	FILE * f = NULL;

	f = fopen(localfile, "rb");
	if (f == NULL) {
		ifp_err("could not open '%s' for reading", localfile);
		return -ENOENT;
	}

	if (status) {
		status->file_name = remotefile;
		status->file_bytes = 0;
		status->file_total = (int)filesize;
	}

	i = _ifp_write_stream_progress(dev, f, (int)filesize, remotefile, status);
	ifp_err_expect(i, i==-ENOENT||i==-EEXIST||i==-ENOSPC
		||i==IFP_ERR_BAD_FILENAME||i==IFP_ERR_USER_CANCEL, err, "problem reading..");

err:
	fclose(f); f = NULL;
	return i;
}

/** \brief Downloads 'remotefile' and saves it directly on the filesystem as
 * 'localfile'.
 *
 * The progress callback function 'fn' and its context pointer are optional.
 * See ::ifp_progress and ::ifp_transfer_status for more information.
 *
 * Returns -ENOENT, -EACCES, -ENOSPC and ::IFP_ERR_USER_CANCEL
 * (Available only in userland.)
 *
 * Note: There is currently a 'EPIPE' bug in the wild that is relatively rare
 * but causes file corruption during download.  ifp_download_file and ifp_download_dir
 * detect and recover from it automatically, but you might see the progress
 * numbers "jump backwards" occasionally.
 */
int ifp_download_file(struct ifp_device * dev,
	const char * remotefile, const char * localfile,
	ifp_progress fn, void * fn_context)
{
	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;

	p = init_progress(&progress, fn, fn_context);
	i = _download_file(dev, remotefile, localfile, p);
	ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL,
		err, "problem reading..");

err:
	return i;
}
IFP_EXPORT(ifp_download_file);

/** \brief Uploads 'localfile' from the filesystem onto the device as
 * 'remotefile'.
 *
 * The progress callback function 'fn' and its context pointer are optional.
 * See ::ifp_progress and ::ifp_transfer_status for more information.
 *
 * Returns -EEXIST, -ENOENT, -ENOSPC, ::IFP_ERR_BAD_FILENAME, ::IFP_ERR_USER_CANCEL
  (Available only in userland.)
 */
int ifp_upload_file(struct ifp_device * dev,
	const char * localfile, const char * remotefile,
	ifp_progress fn, void * fn_context)
{
	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;
	struct stat st;

	p = init_progress(&progress, fn, fn_context);
	i = stat(localfile, &st);
	if (i) {
		i = errno;
		ifp_err_expect(i, i==-ENOENT, err, "couldn't stat file '%s'", localfile);
	}
	i = _upload_file(dev, localfile, remotefile, st.st_size, p);
	ifp_err_expect(i, i==-ENOENT||i==-EEXIST||i==-ENOSPC
		||i==IFP_ERR_BAD_FILENAME||i==IFP_ERR_USER_CANCEL, err,
		"problem writing..");
err:
	return i;
}
IFP_EXPORT(ifp_upload_file);

typedef struct dir_entry {
	struct dir_entry *next;
	char * name;
	int type;
	int filesize;
} dir_entry;

/** inserts 'i' before 'n'. 'p' is the previous field */
static int ll_insert(dir_entry **p, dir_entry *n, dir_entry *i) {
	i->next = n;
	if (p) {
		*p = i;
	}
	return 0;
}

#if 0
static int ll_insert_after(dir_entry *n, dir_entry *i) {
	ll_insert(&(n->next), n, i);

	return 0;
}
#endif

static dir_entry * ll_pop_head(dir_entry ** l) {
	dir_entry * i = *l;
	*l = i->next;
	return i;
}

/** Adds entry to end of linked list, returns new entry.  '*l' will be set to
 * the new entry. */
static dir_entry * queue_dentry(dir_entry ** p, dir_entry * n, const char *name, int ftype, int fsize) {
	int i = 0;
	dir_entry *d;
	if (!p) return NULL;
	if ( (d = malloc(sizeof(dir_entry))) == NULL) {
		return NULL;
	}
	//printf("======================== Pushing dir %s\n",name);
	d->name = strdup(name);
	d->next = NULL;
	d->type = ftype;
	d->filesize = fsize;

	i = ll_insert(p, n, d);
	if (i) {
		printf("[queue_dentry] error doing something %d\n",i);
		return NULL;
	}
	return d;
}
static int dequeue_dentry(dir_entry ** l, char *name, int n, int * ftype, int * fsize)
{
	dir_entry *i;
	if (!l) return -1;
	if (*l) {
		//printf("========================= Poping dir %s\n",name);
		i = ll_pop_head(l);

		strncpy(name,i->name, n);
		*ftype = i->type;
		*fsize = i->filesize;
		free(i->name);
		free(i);
		i = NULL;

		//bak = bak->next;
		//*l = bak;
		return 1;
	} else {
		return 0;
	}
}

struct recursive_context {
	dir_entry **prev;
	dir_entry *next;
};

static int recursive_callback(void * context, int type, const char * f, int filesize)
{
	struct recursive_context * cfc = context;
	int i = 0;
	dir_entry * foo = NULL;

	foo = queue_dentry(cfc->prev, cfc->next, f, type, filesize);
	cfc->prev = &(foo->next);

	return i;
}

struct treewalk_state {
	struct ifp_device * dev;
    	char pathbuff[IFP_MAXPATHLEN];
	dir_entry * q;
	char * remote;
	char * rp;
	int rn;
	struct ifp_treewalk_entry entry;
};

/** \brief Recursively walk a remote directory.  (Interface similar to "fts.h".)
 *
 * Start a treewalk for the 'directory' on the device.  The handle for this
 * session is placed at *handle; this handle is freed by calling ::ifp_treewalk_close.
 *
 * 'dev' won't be left in a "special state" after calling treewalk-family functions.
 * Likewise, please don't leave dev in a "special state" before calling ::ifp_treewalk_open
 * or ::ifp_treewalk_next.
 *
 * Returns -ENOENT if the directory doesn't exist.
 *
 */
int ifp_treewalk_open(struct ifp_device * dev, const char * directory, void ** handle) {
	struct treewalk_state * tws = NULL;
	int len;
	int i = 0;

	if (handle == NULL) {
		i = -1;
		ifp_err("handle shouldn't be null");
		goto err;
	}
	tws = (struct treewalk_state *)malloc(sizeof(struct treewalk_state));
	if (tws == NULL) {
		i = -ENOMEM;
		ifp_err("out of memory");
		goto err;
	}

	i = ifp_is_dir(dev, directory);
	if (i == 0) {
		i = -ENOENT;
	} else if (i == 1) {
		//good
		i = 0;
	}
	ifp_err_expect(i, i==-ENOENT, err, "problem checking ifp:\\%s", directory);

	tws->dev = dev;
	tws->q = NULL;
	tws->entry.path = tws->pathbuff;
	tws->entry.type = IFP_WALK_NONE;

	strncpy(tws->pathbuff, directory, sizeof(tws->pathbuff));
	len = strlen(directory);
	tws->rp = tws->pathbuff + len;
	tws->rn = sizeof(tws->pathbuff) - len;

	queue_dentry(&(tws->q), tws->q, tws->rp, IFP_WALK_DIR_PRE, 0);

err:
	if (i == 0) {
		*handle = tws;
	} else if (tws != NULL) {
		free(tws);
		tws = NULL;
	}

	return i;
}
IFP_EXPORT(ifp_treewalk_open);

/** \brief Returns the next file or directory in a treewalk.
 *
 * The structure returned is valid until the next ::ifp_treewalk_next or
 * ::ifp_treewalk_close function call.  See ::ifp_treewalk_entry for details
 * about the fields.
 *
 * Likewise, please don't leave 'dev' in a "special state" before calling ::ifp_treewalk_next.
 *
 * NULL is returned after the last entry.
 */
struct ifp_treewalk_entry * ifp_treewalk_next(void * tws_p) {
	struct treewalk_state * tws = tws_p;
	struct ifp_treewalk_entry * r = NULL;
	int n;
	int i = 0;

	r = &tws->entry;

	//Check if the last item was a directory.. because we only load those
	//at the last possible minute.
	if (r->type == IFP_WALK_DIR_PRE) {
		struct recursive_context ctx;
		ctx.next = queue_dentry(&tws->q, tws->q, tws->rp, IFP_WALK_DIR_POST, tws->entry.filesize);
		ctx.prev = &tws->q;

		//printf("[_wrt] about to list 'ifp:\\%s'\n", pathbuff);
		i = ifp_list_dirs(tws->dev, tws->pathbuff, recursive_callback, &ctx);
		if (i) {
			ifp_err_i(i, "couldn't get directory list for 'ifp:\\%s'", tws->pathbuff);
			return NULL;
		}

		//append "/" and "\\" to the paths for future use
		tws->rp += r->namelen;
		tws->rn -= r->namelen;
		if (tws->rp - tws->pathbuff > 0 && tws->rp[-1] != '\\') {
			tws->rp[0] = '\\';
			tws->rp++;
			tws->rn--;
		}
		tws->rp[0] = '\0';
	}

	//Read to process the next entry.. but if the queue is empty, there isn't one.
	if (tws->q == NULL) {
		//ifp_err("queue is empty");
		return NULL;
	}

	//copy entry name to paths
	i = dequeue_dentry(&tws->q, tws->rp, tws->rn, &r->type, &r->filesize);
	if (i<=0) {
		ifp_err_i(i, "error dequing");
		return NULL;
	}
	i = 0;
	n = strlen(tws->rp);

	switch(r->type) {
	case IFP_WALK_DIR_POST:
		//printf("[_wrt] dir_post rn=%d n=%d\n", rn, n);
		if ((tws->rp - tws->pathbuff-1) > 0) {
			tws->rp--;
			tws->rn++;
		}
		tws->rp[0] = '\0';
		tws->rp -= n;
		tws->rn += n;
		//fallthrough
	case IFP_WALK_FILE:
	case IFP_WALK_DIR_PRE:
	default:
		r->name = tws->rp;
		r->namelen = n;
		r->pathlen = sizeof(tws->pathbuff) - tws->rn + n;
		break;
	}

	//ifp_err("normal exit, r=%p, r=%p, i=%d", r, i==0 ? r : NULL, i);
	return i==0 ? r : NULL;
}
IFP_EXPORT(ifp_treewalk_next);

/** \brief Releases the resources used in a treewalk.
 *
 * Must be called after each successful call of ::ifp_treewalk_open.
 */
int ifp_treewalk_close(void * tws_p) {
	struct treewalk_state * tws = tws_p;
	int i = 0;

	//empty and free remaining items
	while (tws->q) {
		int a1, a2;
		int e;
		e = dequeue_dentry(&tws->q, tws->rp, tws->rn, &a1, &a2);
		if (e <= 0) {
			ifp_err_i(e, "problem cleaning up");
			if (i == 0) {
				i = e;
			}
		}
	}
	free(tws);
	tws = tws_p = NULL;

	return i;
}
IFP_EXPORT(ifp_treewalk_close);

/** \brief Deletes the directory 'f', its files and subdirectories. (Think of 'rm -Rf'.)
 *
 * Will return -ENOENT if 'f' doesn't exist or isn't a directory.
 *
 * (Available only in userland, at this time.)
 */
int ifp_delete_dir_recursive(struct ifp_device * dev, const char * f)
{
	void * tw = NULL;
	struct ifp_treewalk_entry * r = NULL;
	int i = 0;
	int e;

	i = ifp_treewalk_open(dev, f, &tw);
	ifp_err_expect(i, i==-ENOENT, err_0, "couldn't open directory ifp:\\%s", f);

	//ifp_dbg("[iddr] opened ifp:\\%s", f);
	while(i == 0 && ((r = ifp_treewalk_next(tw)) != NULL)) {
		switch(r->type) {
		case IFP_WALK_FILE:
			//ifp_dbg("about to remove file ifp:\\%s", r->path);
			i = ifp_delete(dev, r->path);
			ifp_err_jump(i, err_1, "couldn't delete file ifp:\\%s", r->path);
			break;
		case IFP_WALK_DIR_POST:
			//ifp_dbg("about to remove dir ifp:\\%s", r->path);
			ifp_rmdir(dev, r->path);
			ifp_err_jump(i, err_1, "couldn't delete dir ifp:\\%s", r->path);
			break;
		default:
			break;
		}
	}
err_1:
	e = ifp_treewalk_close(tw);
	if (e) {
		ifp_err_i(e, "error closing treewalk");
		i = i ? i : e;
	}

err_0:
	return i;
}
IFP_EXPORT(ifp_delete_dir_recursive);

/** Returns the size of all 'f's children.
 *
 * Can return -ENOENT.
 */
static int remote_treesize(struct ifp_device * dev, const char * f, struct ifp_transfer_status * p)
{
	void * tw = NULL;
	struct ifp_treewalk_entry * r = NULL;
	int i = 0;
	int e;
	long bytes = 0;
	int files = 0;

	i = ifp_treewalk_open(dev, f, &tw);
	ifp_err_expect(i, i==-ENOENT, err_0, "couldn't open directory ifp:\\%s", f);

	//ifp_dbg("[iddr] opened ifp:\\%s", f);
	while(i == 0 && ((r = ifp_treewalk_next(tw)) != NULL)) {
		if (r->type == IFP_WALK_FILE) {
			//ifp_dbg("%d bytes used by ifp:\\%s", r->filesize, r->path);
			bytes += r->filesize;
			files++;
		}
	}
	e = ifp_treewalk_close(tw);
	if (e) {
		ifp_err_i(e, "error closing treewalk");
		i = i ? i : e;
	}
	if (i == 0) {
		p->batch_total = bytes;
		p->files_total = files;
	}

err_0:
	return i;
}

/** Returns the size of all 'f's children.
 *
 * Can return -ENOENT.
 */
static int local_treesize(const char * f, struct ifp_transfer_status * p)
{
	FTS * tw = NULL;
	FTSENT * r = NULL;
	char *argv[2] = {(char*)f, NULL};
	int i = 0;
	int e;
	long bytes = 0;
	int files = 0;

	tw = fts_open(argv, FTS_LOGICAL | FTS_NOCHDIR, NULL);
	if (tw == NULL) {
		ifp_err("couldn't open %s", f);
		goto err_0;
	}

	//ifp_dbg("[iddr] opened ifp:\\%s", f);
	while(i == 0 && ((r = fts_read(tw)) != NULL) && r->fts_info != FTS_ERR) {
		if (r->fts_info == FTS_F) {
			//ifp_dbg("%d bytes used by ifp:\\%s", r->filesize, r->path);
			bytes += r->fts_statp->st_size;
			files++;
		}
	}

	if (r && r->fts_info == FTS_ERR) {
		i = r->fts_errno;
		ifp_err_i(i, "error fetching directory entry");
	}
	e = fts_close(tw);
	if (e) {
		ifp_err_i(e, "error closing fts");
		i = i ? i : e;
	}
	if (i == 0) {
		p->batch_total = bytes;
		p->files_total = files;
	}

err_0:
	return i;
}

/**
 *
 * Returns -ENOENT, -EACCES, -ENOSPC IFP_ERR_USER_CANCEL
 * */
static int _ifp_download_dir(struct ifp_device * dev,
	const char * remotedir, const char * localdir,
	struct ifp_transfer_status * p)
{
	char path[256];
	void * tw = NULL;
	struct ifp_treewalk_entry * r = NULL;
	int n,e,i = 0;

	strncpy(path, localdir, sizeof(path));
	n = strlen(path);

	i = ifp_treewalk_open(dev, remotedir, &tw);
	ifp_err_expect(i, i==-ENOENT, err_0, "couldn't open directory ifp:\\%s", remotedir);

	//ifp_dbg("[iddr] opened ifp:\\%s", f);
	while(i == 0 && ((r = ifp_treewalk_next(tw)) != NULL)) {
		//maintain 'path'
		switch(r->type) {
		case IFP_WALK_FILE:
		case IFP_WALK_DIR_PRE:
			strncpy(path + n, r->name, sizeof(path) - n);
			break;

		case IFP_WALK_DIR_POST:
			if (n > 1 && path[n-1] == '/') {
				n--;
				path[n]='\0';
			}
			n -= r->namelen;
			break;
		}

		//do the 'action'
		switch(r->type) {
		case IFP_WALK_DIR_PRE:
			//ifp_dbg("pretending to mkdir  %s", path);
			i = mkdir(path, (S_IRWXU | S_IRWXG | S_IRWXO));
			i = i ? errno : 0;
			ifp_err_jump(i, err_1, "couldn't create %s", path);
			break;
		case IFP_WALK_FILE:
			//ifp_dbg("pretending to create %s", path);
			if (p) {
				p->file_bytes = 0;
				p->file_total = r->filesize;
			}
			i = _download_file(dev, r->path, path, p);
			ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL, err_1,
				"couldn't download file ifp:\\%s", r->path);
			if (p)
				p->files_count++;
			break;
		}

		//maintain 'path'
		if (r->type == IFP_WALK_DIR_PRE) {
			n += r->namelen;
			//append '/' to path if needed
			if (path[n-1] != '/') {
				n++;
				path[n-1]  = '/';
				path[n]  = '\0';
			}
		}
	}
err_1:
	e = ifp_treewalk_close(tw);
	if (e) {
		ifp_err_i(e, "error closing treewalk");
		i = i ? i : e;
	}
err_0:
	return i;
}

//int file_compare_fts(const FTSENT * const*p1, const FTSENT * const*p2)
int file_compare_fts(const FTSENT **p1, const FTSENT **p2)
{
	if (p1 == NULL) { ifp_err("p1 shouldn't be NULL"); return 0; }
	if (p2 == NULL) { ifp_err("p2 shouldn't be NULL"); return 0; }
	if (*p1 == NULL) { ifp_err("*p1 shouldn't be NULL"); return 0; }
	if (*p2 == NULL) { ifp_err("*p2 shouldn't be NULL"); return 0; }
	return strcmp((*p1)->fts_name, (*p2)->fts_name);
}

/**
 *
 * Returns -EEXIST, -ENOENT, -ENOSPC, IFP_ERR_BAD_FILENAME, IFP_ERR_USER_CANCEL
 * FIXME: handle invalid filenames from local filesystem with more grace
 */
static int _ifp_upload_dir(struct ifp_device * dev,
	const char * localdir, const char * remotedir,
	struct ifp_transfer_status * p)
{
	char path[256];
	char *argv[2] = {(char*)localdir, NULL};
	FTS * tw = NULL;
	FTSENT * r = NULL;
	int e, n, i = 0;

	strncpy(path, remotedir, sizeof(path));
	n = strlen(path);

	tw = fts_open(argv, FTS_LOGICAL | FTS_NOCHDIR, file_compare_fts);
	if (tw == NULL) {
		ifp_err("couldn't open %s", localdir);
		goto err_0;
	}

	while(i == 0 && ((r = fts_read(tw)) != NULL) && r->fts_info != FTS_ERR) {
		//maintain 'path'
		switch(r->fts_info) {
		case FTS_F:
		case FTS_D:
			//ignore the root node
			if (r->fts_level != 0) {
				strncpy(path + n, r->fts_name, sizeof(path) - n);
			}
			break;
		case FTS_DP:
			//it doesn't matter if the root node triggers this
			if (n > 1 && path[n-1] == '\\') {
				n--;
				path[n]='\0';
			}
			n -= r->fts_namelen;
			break;
		}

		//do the 'action'
		switch(r->fts_info) {
		case FTS_D:
			//ifp_dbg("mkdir  ifp:\\%s", path);
			i = ifp_mkdir(dev, path);
			ifp_err_jump(i, err_1, "couldn't create ifp:\\%s", path);
			break;
		case FTS_F:
			if (p) {
				p->file_bytes = 0;
				p->file_total = r->fts_statp->st_size;
			}

			i = _upload_file(dev, r->fts_path, path, r->fts_statp->st_size, p);
			ifp_err_expect(i, i==-EEXIST||i==-ENOENT||i==-ENOSPC
				||i==IFP_ERR_BAD_FILENAME||i==IFP_ERR_USER_CANCEL, err_1,
				"couldn't download file ifp:\\%s", r->fts_path);
			if (p)
				p->files_count++;
			break;
		}

		//maintain 'path'
		if (r->fts_info == FTS_D) {
			//matches the strncpy above
			if (r->fts_level != 0) {
				n += r->fts_namelen;
			}

			//Yes, it's ok to do this to the root node
			//append '\\' to path if needed
			if (path[n-1] != '\\') {
				n++;
				path[n-1]  = '\\';
				path[n]  = '\0';
			}
		}
	}
	if (r && r->fts_info == FTS_ERR) {
		i = r->fts_errno;
		ifp_err_i(i, "error fetching directory entry");
	}
err_1:
	e = fts_close(tw);
	if (e) {
		ifp_err_i(e, "error closing fts");
		i = i ? i : e;
	}

err_0:
	return i;
}

/** \brief Downloads the contents of 'remotedir' (including all subdirectories) and saves it
 * as 'localdir'.
 *
 * Note that 'localdir' must not allready exist.  Example:
 * suppose localdir was '/tmp/tunes/tame' and remotedir was '\\classical\\junk'.
 * The directory '/tmp/tunes/tame' will be created and the file
 * '\\classical\\junk\\buz\\fud.ogg' will be copied as '/tmp/tunes/tame/buz/fud.ogg'
 *
 * The progress callback function 'fn' and its context pointer are optional.
 * See ::ifp_progress and ::ifp_transfer_status for more information.
 *
 * Returns -ENOENT, -EACCES, -ENOSPC ::IFP_ERR_USER_CANCEL
 *
 * (Available only in userland.)
 *
 * Note: There is currently a 'EPIPE' bug in the wild that is relatively rare
 * but causes file corruption during download.  ifp_download_file and ifp_download_dir
 * detect and recover from it automatically, but you might see the progress
 * numbers "jump backwards" occasionally.
 */
int ifp_download_dir(struct ifp_device * dev,
	const char * remotedir, const char * localdir,
	ifp_progress fn, void * fn_context)
{
	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;

	p = init_progress(&progress, fn, fn_context);
	if (p) {
		p->is_batch = 1;
		i = remote_treesize(dev, remotedir, p);
		ifp_err_expect(i, i==-ENOENT, err, "couldn't open directory ifp:\\%s", remotedir);
		//ifp_dbg("The total transfer size is expected to be %d bytes",
		//	(int)progress.batch_total);
	}

	i = _ifp_download_dir(dev, remotedir, localdir, p);
	ifp_err_expect(i, i==-ENOENT||i==-EACCES||i==-ENOSPC||i==IFP_ERR_USER_CANCEL, err,
		"couldn't download directory ifp:\\%s", remotedir);

err:
	return i;
}
IFP_EXPORT(ifp_download_dir);

/** \brief Uploads the contents of 'localdir' (including all subdirectories) to the
 * device as 'remotedir'.
 *
 * Note that 'remotedir' must not exist on the remote device.  Example:
 * suppose localdir was '/tmp/tunes/tame' and remotedir was '\\classical\\junk'.
 * The directory '\\classical\\junk' will be created and the file
 * '/tmp/tunes/tame/buz/fud.ogg' will be copied as
 * '\\classical\\junk\\buz\\fud.ogg'.
 *
 * The progress callback function 'fn' and its context pointer are optional.
 * See ::ifp_progress and ::ifp_transfer_status for more information.
 *
 * Returns -EEXIST, -ENOENT, -ENOSPC, ::IFP_ERR_BAD_FILENAME, or ::IFP_ERR_USER_CANCEL.
 * FIXME: handle invalid filenames from local filesystem with more grace
 *
 * (Available only in userland.)
 */
int ifp_upload_dir(struct ifp_device * dev,
	const char * localdir, const char * remotedir,
	ifp_progress fn, void * fn_context)
{
	int i = 0;
	struct ifp_transfer_status progress;
	struct ifp_transfer_status * p = NULL;

	p = init_progress(&progress, fn, fn_context);
	if (p) {
		p->is_batch = 1;
		i = local_treesize(localdir, p);
		ifp_err_expect(i, i==-ENOENT, err, "couldn't open directory %s", localdir);
		//ifp_dbg("The total transfer size is expected to be %d bytes",
		//	(int)progress.batch_total);
	}

	i = _ifp_upload_dir(dev, localdir, remotedir, p);
	ifp_err_expect(i, i==-EEXIST||i==-ENOENT||i==-ENOSPC||i==IFP_ERR_BAD_FILENAME
		||i==IFP_ERR_USER_CANCEL, err, "coudln't upload ifp:\\%s", remotedir);

err:
	return i;
}
IFP_EXPORT(ifp_upload_dir);


#define FIRMWARE_HEADER_SIZE		4

/** \brief Upgrades the firmware.
 *
 * This is much like a file upload, except: the filename on the local disk must
 * be in the format "IFP-?XXT.HEX", "IFP-1XXTC.HEX" or "N10.HEX".  The progress
 * meter only tracks the firmware upload.  The flash-upgrading itself and
 * reboot take extra time we can't predict.
 *
 * Immediately after calling ifp_update_firmware, the caller should release
 * 'dev', and wait a healthy ammount of time (10 or more seconds) before trying
 * to reconnect.  During this time, you'll see the message "upgrading firmware
 * please don't touch" on the device, after which the device will shutdown:
 * user will likely have to turn it back on themselves.
 *
 * I welcome suggestions and/or code on how to help monitor the device status
 * during a firmware upgrade.
 */
int ifp_update_firmware(struct ifp_device *dev, const char * localfile,
	ifp_progress fn, void * fn_context)
{
    char magic_header_original[FIRMWARE_HEADER_SIZE] = {0x39, 0xb0, 0x5d, 0xed};
    char magic_header_n10[FIRMWARE_HEADER_SIZE] = {0x37, 0x13, 0x0d, 0xda};
    FILE *fp;
    int i = 0;
    char buf[FIRMWARE_HEADER_SIZE];
    char * checkdata = NULL;
    char remotename[20];
    const char *basename;
    struct ifp_transfer_status progress;
    struct ifp_transfer_status * p = NULL;
    struct stat st;

    i = stat(localfile, &st);
    if (i) {
	i = errno;
	ifp_err_expect(i, i==-ENOENT, err0, "couldn't stat file '%s'", localfile);
    }

    p = init_progress(&progress, fn, fn_context);

    basename = strrchr(localfile, '/');
    if (basename) {
	    basename++;
    } else {
	    //no '/'
	    basename = localfile;
    }

    if (dev->model == IFP_N10) {
	    //eg "N10.HEX"
	    if (strncmp(basename, "N10", 3) != 0) {
		ifp_err("Firmware filename must be \"N10.HEX\".");
		return -1;
	    }
	    checkdata = magic_header_n10;
    } else {
	    //eg "IFP-5XXT.HEX"
	    if (strncmp(basename, "IFP-", 4) != 0) {
		ifp_err("Firmware filename must be set \"IFP-?XXT.HEX\" or \"IFP-1XXTC.HEX\".");
		return -1;
	    }
	    checkdata = magic_header_original;
    }

    if ( (fp = fopen(localfile, "r")) == NULL) {
	//ifp_dbg("invalid firmware file(too short!).");
	return errno;
    }

    if ( fread(buf, sizeof(unsigned char), sizeof(checkdata), fp)
	    < FIRMWARE_HEADER_SIZE) {
	ifp_err("invalid firmware file(too short!).");
	fclose(fp);
	return -1;
    }
    rewind(fp);	    // DO NOT REMOVE IT!

    if (strncmp(buf, checkdata, FIRMWARE_HEADER_SIZE) != 0) {
	fprintf(stderr, "ifp firmupdate: Invalid format firmware file.\n");
	fclose(fp);
	return -1;
    }

    remotename[0] = '\\';
    strncpy(remotename+1, basename, sizeof(remotename) - 1);
    if ( (i = _ifp_write_stream_progress(dev, fp, (int)st.st_size, remotename, p))) {
	ifp_err_i(i, "Failed firmware upload.");
	fclose(fp);
	return i;
    }
    fclose(fp);

    i = ifp_update_firmware_raw(dev);
    ifp_err_jump(i, err0, "firmware update code failed");

err0:
    return i;
}


